///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/config.h"
#ifdef _RHEOLEF_HAVE_MPI
#include "rheolef/geo.h"
#include "rheolef/par_macros.h"
#include "rheolef/rheostream.h"

namespace rheolef {

extern
array<polymorphic_array<geo_element>::size_type>
geo_partition (const polymorphic_array<geo_element>& elts, size_t map_dim);

// --------------------------------------------------------------------------
// edges & faces renumbering subroutine 
// --------------------------------------------------------------------------
template <class PolymorphicArray>
void
subgeo_renumbering (
    const std::vector<geo_element::size_type>& new_global_idx_vertex_owner,
    const std::vector<geo_element::size_type>& global_new_num_vert,
    const PolymorphicArray&                    old_subgeo,
          PolymorphicArray&                    subgeo,
          array<geo_element::size_type>&       old_isg2par_isg
    )
{
    using namespace std;
    typedef typename geo_element::size_type size_type;

    communicator comm = old_subgeo.ownership().comm();
    size_type par_nsg = old_subgeo.par_size();
    vector<size_type> new_local_subgeo_owner (par_nsg, 0);

    //
    // 1) global all_reduce
    // TODO: not balanced: subgeo nums
    // TODO: also, not optimal: O(N) in communication, memory & CPU, instead of O(N/nproc)
    //
    for (size_type isg = 0, nsg = old_subgeo.size(); isg < nsg; isg++) {
      const geo_element& S = old_subgeo [isg];
      size_type par_old_isg = S.par_old_ie();
      size_type owner = 0;
      for (size_type iloc = 0; iloc < S.size(); iloc++) {
        size_type par_old_iv = S [iloc];
        assert_macro (par_old_iv < par_nv, "vertex index "<<par_old_iv<<" is out of range [0:"<<par_nv<<"[");
        size_type iproc = new_global_idx_vertex_owner [par_old_iv];
        owner = std::max (owner,iproc);
      }
      new_local_subgeo_owner [par_old_isg] = owner;
    }
    vector<size_type> new_global_subgeo_owner (par_nsg, 0);
    mpi::all_reduce (
  	comm,
  	new_local_subgeo_owner.begin().operator->(),
  	par_nsg,
  	new_global_subgeo_owner.begin().operator->(),
  	mpi::maximum<size_type>());
  
    distributor old_subgeo_ownership = old_subgeo.ownership();
    //
    // 2) redistribute the vertex partition
    //
    array<size_type> subgeo_partition (old_subgeo_ownership);
    for (size_type par_old_isg = subgeo_partition.first_index(); par_old_isg < subgeo_partition.last_index(); par_old_isg++) {
      subgeo_partition.set (par_old_isg, new_global_subgeo_owner[par_old_isg]);
    }
    subgeo_partition.assembly();

    array<size_type>  isg2par_old_isg;
    old_subgeo.repartition (
	subgeo_partition,
        subgeo,
  	isg2par_old_isg,
  	old_isg2par_isg);
  
    //
    // 3) vertices S[iloc] of new numbered element table still have old numbering: fix it
    //
    for (size_type isg = 0, nsg = subgeo.size(); isg < nsg; isg++) {
      geo_element& S = subgeo [isg];
      for (size_type iloc = 0, nloc = S.size(); iloc < nloc; iloc++) {
        assert_macro (S[iloc] < par_nv, "vertex index "<<K[iloc]<<" out of range [0:"<<par_nv<<"[");
        assert_macro (global_new_num_vert[S[iloc]] < par_nv, "new vertex index "<<global_new_num_vert[K[iloc]] <<" out of range [0:"<<par_nv<<"[");
        S[iloc] = global_new_num_vert [S[iloc]];
      }
    }
}
// --------------------------------------------------------------------------
// io for geo
// --------------------------------------------------------------------------
template <class T>
iparstream&
geo_mpi_rep<T>::get (iparstream& ips, const geo_basic<T,distributed>& smart_ptr_this) {
  using namespace std;
  check_macro (ips.good(), "bad input stream for geo.");
  communicator comm = polymorphic_array<geo_element,distributed>::ownership().comm();

  size_type io_proc = iparstream::io_proc();
  size_type my_proc = comm.rank();
  size_type nproc   = comm.size();
  if (my_proc == io_proc && !scatch(ips.is(),"\nmesh"))
    error_macro("input stream does not contains a geo.");
  // ------------------------------------------------------------------------
  // 1) read file
  // ------------------------------------------------------------------------
  //
  // 1.1) get header
  //
  size_type par_nv;
  size_type par_ne;
  size_type par_nedg = 0;
  size_type par_n_fac = 0;

  base::_name = "unnamed";

  ips >> base::_version
      >> base::_dimension
      >> par_nv
      >> par_ne;

  warning_macro ("version="<<base::_version);
  warning_macro ("dimension="<<base::_dimension);
  warning_macro ("par_nv="<<par_nv);
  warning_macro ("par_ne="<<par_ne);

  if (base::_version < 2) {
    warning_macro ("mesh version < 2 no more supported in mpi-parallel version");
  } else {
    if (base::_dimension >= 3) {
      ips >> par_n_fac;
      warning_macro ("par_n_fac="<<par_n_fac);
    }
    if (base::_dimension >= 2) {
      ips >> par_nedg;
      warning_macro ("par_nedg="<<par_nedg);
    }
  }
  //
  // 1.2) get coordinates
  //
  array<idx_vertex_type> old_idx_vertex (par_nv);
  old_idx_vertex.get (ips, _point_get<T>(base::_dimension));
  check_macro (ips.good(), "bad input stream for geo.");
  // set old_par_iv index as fisrt field of the idx_vertex pair:
  size_type first_par_iv = old_idx_vertex.ownership().first_index();
  for (size_type iv = 0, nv = old_idx_vertex.size(); iv < nv; iv++) {
    old_idx_vertex [iv].first = first_par_iv + iv;
    trace_macro ("old_idx_vertex ["<<iv<<"].first = "<< old_idx_vertex [iv].first);
  }
  //
  // 1.3) get elements
  //
  polymorphic_array<geo_element> old_elts (par_ne);
  old_elts.get (ips);
  size_type par_old_ie_start = old_elts.ownership().first_index();
  for (size_type ie = 0, ne = old_elts.size(); ie < ne; ie++) {
    geo_element& K = old_elts [ie];
    K.set_par_old_ie (par_old_ie_start + ie);
  }
  //
  // 1.4) get faces & edges
  //
  polymorphic_array<geo_element,distributed> old_faces;
  if   (base::_version  >= 2 && base::_dimension >= 3) {
      old_faces.resize (par_n_fac);
      old_faces.get (ips);
      size_type par_old_ifac_start = old_faces.ownership().first_index();
      for (size_type ifac = 0, nfac = old_faces.size(); ifac < nfac; ifac++) {
        geo_element& F = old_faces [ifac];
        F.set_par_old_ie (par_old_ifac_start + ifac);
      }
  }
  polymorphic_array<geo_element,distributed> old_edges;
  if   (base::_version  >= 2 && base::_dimension >= 2) {
      old_edges.resize (par_nedg);
      old_edges.get (ips);
      size_type par_old_iedg_start = old_edges.ownership().first_index();
      for (size_type iedg = 0, nedg = old_edges.size(); iedg < nedg; iedg++) {
        geo_element& E = old_edges [iedg];
        E.set_par_old_ie (par_old_iedg_start + iedg);
      }
  }
  // ------------------------------------------------------------------------
  // 2) mesh partition & element renumbering
  // ------------------------------------------------------------------------
  array<size_type> partition = geo_partition (old_elts, map_dimension());
  //
  // 2.1) elements renumbering
  //
  array<size_type> ie2par_old_ie; // no more used
  old_elts.repartition (
	partition,
	*this,
  	ie2par_old_ie,
  	_old_ie2par_ie);
  // ------------------------------------------------------------------------
  // 3) vertices renumbering 
  // ------------------------------------------------------------------------
  // 3.1) global all_reduce
  // TODO: not balanced: vertex nums
  // TODO: also, not optimal: O(N) in communication, memory & CPU, instead of O(N/nproc)
  //
  vector<size_type> new_local_idx_vertex_owner (par_nv, 0);
  for (size_type ie = 0, ne = size(); ie < ne; ie++) {
    const geo_element& K = element (ie);
    for (size_type iloc = 0; iloc < K.size(); iloc++) {
      new_local_idx_vertex_owner [K[iloc]] = my_proc;
    }
  }
  vector<size_type> new_global_idx_vertex_owner (par_nv, 0);
  mpi::all_reduce (
	comm,
	new_local_idx_vertex_owner.begin().operator->(),
	par_nv,
	new_global_idx_vertex_owner.begin().operator->(),
	mpi::maximum<size_type>());

  // 3.2) redistribute the vertex partition
  array<size_type> vertex_partition (old_idx_vertex.ownership());
  for (size_type par_old_iv = vertex_partition.first_index(); par_old_iv < vertex_partition.last_index(); par_old_iv++) {
      vertex_partition.set (par_old_iv, new_global_idx_vertex_owner[par_old_iv]);
  }
  vertex_partition.assembly();

  array<size_type>       iv2par_old_iv;
  old_idx_vertex.repartition (
	vertex_partition,
        base::_idx_vertex,
  	iv2par_old_iv,
  	_old_iv2par_iv);
  
  distributor vertex_ownership = base::vertex_ownership();
 
  // TO_CLEAN:
  for (size_type iv = 0, nv = base::_idx_vertex.size(); iv < nv; iv++) {
    trace_macro ("idx_vertex ["<<iv<<"].first = "<< base::_idx_vertex [iv].first);
  }
  //
  // 3.3) vertices K[iloc] of new numbered element table K still have old numbering: fix it
  //
  vector<size_type> local_new_num_vert (par_nv, 0);
  for (size_type par_iv = _old_iv2par_iv.first_index(); par_iv < _old_iv2par_iv.last_index(); par_iv++) {
     size_type iv = par_iv - _old_iv2par_iv.first_index();
     local_new_num_vert [par_iv] = _old_iv2par_iv [iv];
  }
  vector<size_type> global_new_num_vert (par_nv, 0);
  mpi::all_reduce (
	comm,
	local_new_num_vert.begin().operator->(),
	par_nv,
	global_new_num_vert.begin().operator->(),
	mpi::maximum<size_type>());

  for (size_type ie = 0; ie < size(); ie++) {
    geo_element& K = element (ie);
    for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
      assert_macro (K[iloc] < par_nv, "vertex index "<<K[iloc]<<" out of range [0:"<<par_nv<<"[");
      assert_macro (global_new_num_vert[K[iloc]] < par_nv, "new vertex index "<<global_new_num_vert[K[iloc]] <<" out of range [0:"<<par_nv<<"[");
      K[iloc] = global_new_num_vert [K[iloc]];
    }
  }
  // 3.4) loop on elements: identify some vertices, that are referenced
  // by locally-managed geo_elements, but these vertices are managed
  // by another processor: e.g. vertices on a partition boundary.
  std::set<size_type> ext_idx_vertex_set;
  size_type par_iv_first = vertex_ownership.first_index();
  size_type par_iv_last  = vertex_ownership.last_index();
  for (size_type ie = 0, ne = size(); ie < ne; ie++) {
    const geo_element& K = element (ie);
    for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
      assert_macro (K[iloc] < par_nv, "vertex index "<<K[iloc]<<" out of range [0:"<<par_nv<<"[");
      size_type par_iv = K [iloc];
      if (par_iv >= par_iv_first && par_iv < par_iv_last) continue;
      ext_idx_vertex_set.insert (par_iv);
    }
  }
  // 3.5) get external vertices:
#ifdef TO_CLEAN
  mpi_scatter_map (
	vertex_ownership, 
  	base::_idx_vertex.begin(),
	ext_idx_vertex_set,
	_ext_par_iv2par_old_iv_vertex);
#endif // TO_CLEAN
  base::_idx_vertex.scatter(
	ext_idx_vertex_set,
	_ext_par_iv2par_old_iv_vertex);
  // ------------------------------------------------------------------------
  // 4) edge & face renumbering 
  // ------------------------------------------------------------------------
  if (base::_version >= 2 && base::_dimension >= 2) {
    warning_macro ("edge renumbering...");
    subgeo_renumbering (
        new_global_idx_vertex_owner,
        global_new_num_vert,
        old_edges,
        base::_edges,
        _old_iedg2par_iedg);

    warning_macro ("edge renumbering done");
  }
  if (base::_version >= 2 && base::_dimension >= 3) {
    warning_macro ("face renumbering...");
    subgeo_renumbering (
        new_global_idx_vertex_owner,
        global_new_num_vert,
        old_faces,
        base::_faces,
        _old_ifac2par_ifac);

    warning_macro ("face renumbering done");
  }
  // ------------------------------------------------------------------------
  // 5) get domain, until end-of-file
  // ------------------------------------------------------------------------
  // TODO: why ?
  comm.barrier();
  warning_macro ("get domains...");
  do {
    domain_basic<T,distributed> dom;
    bool status = dom.get (ips, smart_ptr_this);
    if (!status) break;
    warning_macro ("get domain: push_back "<<dom.name());
    base::_domains.push_back (dom);
  } while (true);
  // TODO: why ?
  comm.barrier();
  warning_macro ("get domains done");
#ifdef TODO
#endif // TODO
  return ips;
}
// ----------------------------------------------------------------------------
// read from file
// ----------------------------------------------------------------------------
template <class T>
void
geo_mpi_rep<T>::load (
  std::string filename, 
  const communicator& comm, 
  const geo_basic<T,distributed>& smart_ptr_this)
{
  warning_macro ("get geo from file "<<filename<<"...");
  iparstream ips;
  ips.open (filename, "geo", comm);
  warning_macro ("get geo from file "<<filename<<": [1] check...");
  check_macro(ips.good(), "\"" << filename << "[.geo[.gz]]\" not found.");
  warning_macro ("get geo from file "<<filename<<": [1] check done");
  get (ips, smart_ptr_this);
  std::string root_name = delete_suffix (delete_suffix(filename, "gz"), "geo");
  std::string name = get_basename (root_name);
  base::_name = name;
  warning_macro ("get geo from file "<<filename<<" done");
}
// ----------------------------------------------------------------------------
// output
// ----------------------------------------------------------------------------
/// @brief helper permutation class for geo i/o
template <class V = typename polymorphic_traits<geo_element>::derived_type>
struct geo_element_perm {
  typedef geo_element::size_type size_type;
  geo_element_perm (const polymorphic_array<geo_element,distributed,V>& elts) 
    : _elts(elts) {}
  size_type operator[] (size_type ie) const {
    const geo_element& K = _elts [ie];
    return K.par_old_ie();
  }
  const polymorphic_array<geo_element,distributed,V>& _elts;
};
template <class T>
oparstream&
geo_mpi_rep<T>::put (oparstream& ops)  const {
  communicator comm = polymorphic_array<geo_element,distributed>::ownership().comm();

  using namespace std;
  size_type io_proc = oparstream::io_proc();
  size_type my_proc = comm.rank();
  size_type nproc   = comm.size();
  size_type par_nv = base::_idx_vertex.par_size ();
  size_type par_ne = polymorphic_array<geo_element,distributed>::par_size ();
  ops << "#!geo" << endl
      << endl
      << "mesh" << endl
      << base::_version
      << " " << base::_dimension
      << " " << par_nv
      << " " << par_ne;
  if   (base::_version >= 2) {
    if (base::_dimension >= 3) {
      ops << " " << base::_faces.par_size();
    }
    if (base::_dimension >= 2) {
      ops << " " << base::_edges.par_size();
    }
  }
  ops << endl << endl;

  // build a permutationh array (could be avoided, but requires a template Permutation arg in array::permuted_put
  array<size_type> iv2par_old_iv (base::_idx_vertex.ownership());
  for (size_type iv = 0, niv = iv2par_old_iv.size(); iv < niv; iv++) {
    iv2par_old_iv [iv] = base::_idx_vertex [iv].first;
  }
  base::_idx_vertex.permuted_put (ops, iv2par_old_iv, _point_put<T>(base::_dimension));
  ops << endl;

  // elements are permuted back to original order and may
  // refer to original vertices numbering
  std::vector<size_type> vertex_perm ((my_proc == io_proc) ? par_nv : 0);
  size_type tag_gather = distributor::get_new_tag();
  if (my_proc == io_proc) {
    size_type i_start = iv2par_old_iv.ownership().first_index(my_proc);
    size_type i_size  = iv2par_old_iv.ownership().size       (my_proc);
    for (size_type i = 0; i < i_size; i++) {
      vertex_perm [i_start + i] = iv2par_old_iv [i];
    }
    for (size_type iproc = 0; iproc < nproc; iproc++) {
      if (iproc == my_proc) continue;
      size_type i_start = iv2par_old_iv.ownership().first_index(iproc);
      size_type i_size  = iv2par_old_iv.ownership().size       (iproc);
      comm.recv (iproc, tag_gather, vertex_perm.begin().operator->() + i_start, i_size);
    }
  } else {
    comm.send (0, tag_gather, iv2par_old_iv.begin().operator->(), iv2par_old_iv.size());
  }
  geo_element_permuted_put put_element (vertex_perm);

  polymorphic_array<geo_element,distributed>::permuted_put (
	ops, 
	geo_element_perm<> (polymorphic_array<geo_element,distributed>(*this)),
	put_element);
  ops << endl;
  //
  // put faces & edges
  //
  if   (base::_version >= 2 && base::_dimension >= 3) {
      base::_faces.permuted_put (
	ops, 
	geo_element_perm<> (base::_faces),
	put_element);
  }
  if   (base::_version >= 2 && base::_dimension >= 2) {
      base::_edges.permuted_put (
	ops, 
	geo_element_perm<> (base::_edges),
	put_element);
  }
  //
  // put domains
  //
  for (typename std::vector<domain_basic<T,distributed> >::const_iterator
        iter = base::_domains.begin(),
        last = base::_domains.end();
	iter != last; ++iter) {
    ops << endl;
    (*iter).put (ops, *this);
  }
#ifdef TODO
#endif // TODO
  return ops;
}
template <class T>
void
geo_mpi_rep<T>::dump (std::string name)  const {
  base::_idx_vertex.dump        (name + "-vert");
  polymorphic_array<geo_element,distributed>::dump(name + "-elem");
}
// --------------------------------------------------------------------------
// accessors to distributed data
// --------------------------------------------------------------------------
template <class T>
typename geo_mpi_rep<T>::size_type
geo_mpi_rep<T>::par_iv2par_old_iv (size_type par_iv) const
{
    if (par_iv >= vertex_ownership().first_index()
     && par_iv <  vertex_ownership().last_index()) {
        size_type iv = par_iv - vertex_ownership().first_index();
        return base::_idx_vertex [iv].first;
    }
    // here, par_iv is not managed by current proc
    // try on external associative table
    typename idx_vertex_map_type::const_iterator iter = _ext_par_iv2par_old_iv_vertex.find (par_iv);
    check_macro (iter != _ext_par_iv2par_old_iv_vertex.end(), "unexpected vertex index:"<<par_iv);
    return (*iter).second.first;
}
template <class T>
const typename geo_mpi_rep<T>::vertex_type&
geo_mpi_rep<T>::par_vertex (size_type par_iv) const
{
    if (par_iv >= vertex_ownership().first_index()
     && par_iv <  vertex_ownership().last_index()) {
        size_type iv = par_iv - vertex_ownership().first_index();
        return base::_idx_vertex [iv].second;
    }
    // here, par_iv is not managed by current proc
    // try on external associative table
    typename idx_vertex_map_type::const_iterator iter = _ext_par_iv2par_old_iv_vertex.find (par_iv);
    check_macro (iter != _ext_par_iv2par_old_iv_vertex.end(), "unexpected vertex index:"<<par_iv);
    return (*iter).second.second;
}
// --------------------------------------------------------------------------
// access by subgeo(dim,idx)
// --------------------------------------------------------------------------
template <class T>
distributor
geo_mpi_rep<T>::subgeo_old_ownership (size_type dim) const
{
  if (dim == base::_dimension) { return _old_ie2par_ie.ownership(); }
  switch (dim) {
    case 0: return _old_iv2par_iv.ownership();
    case 1: return _old_iedg2par_iedg.ownership();
    case 2: return _old_ifac2par_ifac.ownership();
    default: {
      error_macro ("unexpected subgeo dimension="<<dim);
      return distributor(); // not reached
    }
  }
}
template <class T>
typename geo_mpi_rep<T>::size_type
geo_mpi_rep<T>::old_isubgeo2par_isubgeo (size_type dim, size_type old_isubgeo) const
{
  if (dim == base::_dimension) {
    return _old_ie2par_ie [old_isubgeo];
  }
  switch (dim) {
    case 0: return _old_iv2par_iv     [old_isubgeo];
    case 1: return _old_iedg2par_iedg [old_isubgeo];
    case 2: return _old_ifac2par_ifac [old_isubgeo];
    default: {
      error_macro ("unexpected subgeo dimension="<<dim);
      return 0; // not reached
    }
  }
}
template <class T>
typename geo_mpi_rep<T>::size_type
geo_mpi_rep<T>::isubgeo2par_old_isubgeo (size_type dim, size_type isubgeo) const
{
  if (dim == base::_dimension) {
    return operator[] (isubgeo).par_old_ie();
  }
  switch (dim) {
    case 0: return base::_idx_vertex [isubgeo].par_old_iv();
    case 1: {
	const geo_element& E = base::_edges [isubgeo];
	return E.par_old_ie();
    }
    case 2: {
	const geo_element& F = base::_faces [isubgeo];
	return F.par_old_ie();
    }
    default: {
      error_macro ("unexpected subgeo dimension="<<dim);
      return 0; // not reached
    }
  }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class geo_mpi_rep<Float>;
} // namespace rheolef
#endif // _RHEOLEF_HAVE_MPI
