///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:space_gen
NAME: "space_gen" - output a P1-continuous finite element space 
SYNOPSIS:
   "space_gen" {-I'geodir'} 'mesh'[".geo"] {'domain-name'}*
DESCRIPTION:       
  Generate a 2d or 3d finite element space.
EXAMPLE: 
    enter command as:

    |  space_gen square.geo top left
OPTIONS:
    -I'geodir' add 'geodir' to the GEOPATH search path.
    See also "geo"(3) for GEOPATH mechanism. 

    'filename' specifies the name of the file containing
      the input mesh. The ".geo" extension is assumed.

    "-" read mesh on standard input instead on a file.

    'domain-name' specifies a valid domain name where the
    degrees of freedom are blocked.
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Nicolas.Roquet@imag.fr
    | Pierre.Saramito@imag.fr
SEE ALSO:
    class "space"(3), class "geo"(3), class "base"(3)
DATE:
    16 june 1997
End:
*/
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

void usage()
{
      cerr << "space_gen: usage: space_gen "
	   << "{-Igeodir}*"
	   << "-|mesh[.geo]"
	   << "{domain-name}*"
	   << endl;
      exit (1);
}
int main(int argc, char**argv)
{
    bool verbose = true;
    if (argc <= 1) usage();
    //
    // load geometry
    //
    geo g;  
    int io = 1; 
    while (argv [io][0] == '-' && argv [io][1] == 'I') {
	append_dir_to_rheo_path (argv[io]+2);
	io++;
    }
    if (strcmp (argv[io], "-") == 0) {
	// input geo on standard input
        cerr << "! space: geo on stdin\n";
	cin >> g;
      
    } else {
    
	// input geo on file
  	g = geo(argv[io]);
    }
    //
    // block dof on boundary domains
    //
    space V_0 (g, "P1");
    for (int i = io+1; i < argc; i++) {

      // block a domain
      cerr << "! space: block `" << argv[i] << "'\n";
      V_0.block (argv[i]);
    }
    //
    // freeze V_0 and create the product V_0 x V_0 
    V_0.freeze();
    space V=V_0*V_0; 
    //
    // output space
    //
    cout << V;
    return 0;
}
