# ifndef _RHEO_CG_ABTBC_H
# define _RHEO_CG_ABTBC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:cg_abtbc
NAME: @code{cg_abtbc}, @code{cg_abtbc} -- stabilized mixed linear solver
@findex cg\_abtbc
@cindex mixed linear problem
@cindex Stokes problem
@cindex conjugate gradien algorithm
@cindex finite element method
@cindex stabilized mixed finite element method
@cindex incompresible elasticity
DESCRIPTION: 
  @noindent
  Conjugate gradient algorithm on the pressure p applied to 
  the stabilized stokes problem.
  @example
       [ a  b^T ] [ u ]    [ f ]
       [        ] [   ]  = [   ]
       [ b  -c  ] [ p ]    [ g ]
  @end example
  with the bock diagonal preconditionner : (inv(a),I).
  Such mixed linear problems appears for instance with the discretization
  of Stokes problems with stabilized P1-P1 element, or with nearly
  incompressible elasticity.
  The algorithm requires the resolution of linear
  systems such as: @code{a*x = b}.
  These systems may be solved either by direct or iterative 
  algorithms. Thus, a general matrix solver class is submitted
  to the algorithm.

AUTHOR: 
   | Nicolas.Roquet@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France
DATE:   15 nov 1997
METHODS: @cg_abtbc
End:
*/

#include "rheolef/matrix_solver.h"
namespace rheolef { 

//<cg_abtbc:  
template<
    class MatrixSolver, 
    class MatrixPreconditioner, 
    class Matrix,
    class Vector, 
    class StokesPreconditioner, 
    class Real,
    class Integer>
int
cg_abtbc(
    const MatrixSolver&            m_solver,
    const Matrix&                  A, 
    const MatrixPreconditioner&    M_A,  
    const Matrix&                  B,
    const Matrix&                  C,
    Vector&                        U,
    Vector&                        P,
    const Vector&                  F,
    const Vector&                  G,
    const StokesPreconditioner&    M,
    Integer&                       max_iter, 
    Real&                          tol, 
    std::ostream*                  p_cres = 0)
{
  Real residu ;
  Vector Q, Y, Z, V = U ;  
  Real alpha, beta, rho, rho_1=0 ;
  
  Vector W = U ;

  int status_ms_a1 = m_solver( A, M_A, W, F ) ;
  
  Vector Ftilde   = B*W - G      ;
  Real normFtilde = norm(Ftilde) ;
  
  int status_ms_a2 = m_solver( A, M_A, V, B.trans_mult(P) ) ;
   
  Vector R = Ftilde - C*P - B*V ;
  
  if (normFtilde == Float(0.0)) normFtilde = Float(1.) ;

  residu = norm(R)/normFtilde;
  if (p_cres) *p_cres << "[cg_abtbc] " << 0 << " " << residu << "\n" ;
  if ( residu <= tol ) {
      tol = residu ;
      max_iter = 0 ;
      return 0     ;
  }
  for (Integer k=1 ; k <= max_iter ; k++ ) {
      
      Z = M.solve(R) ;
      rho = dot(R,Z) ;
      
      if (k==1) Q = Z ;
      else
	{
	  beta = rho/rho_1  ;
	  Q    = Z + beta*Q ;  
	}
      int status_ms_a3 = m_solver( A, M_A, V, B.trans_mult(Q) ) ;

      Y = B*V + C*Q ;
      alpha = rho/dot(Q,Y) ;
      
      P += alpha*Q ;
      //      U += alpha*V ;
      R -= alpha*Y ;
      
      residu = norm(R)/normFtilde;
      if (p_cres) *p_cres << "[cg_abtbc] " << k << " " << residu << "\n" ;
      if ( residu <= tol ) {
	  tol = residu ;
	  max_iter = k ;
	  int status_ms_a4 = m_solver( A, M_A, U, F - B.trans_mult(P) ) ;
	  return 0;
      }
      rho_1 = rho ;
  }
  tol = residu ;
  return 1 ;
}
template<
    class MatrixPreconditionner, 
    class Matrix, 
    class Vector, 
    class Real,
    class Integer>
int
cg_abtbc(
    const Matrix&                a,
    const MatrixPreconditionner& ap,
    const Matrix&                b,
    const Matrix&                c,
    Vector&                      u,
    Vector&                      p,
    const Vector&                f,
    const Vector&                g,
    Integer&                     max_iter,
    Real&                        tol,
    std::ostream*                p_cres = &std::cerr)
{
    return cg_abtbc (ldlt_solver<MatrixPreconditionner, Matrix, Vector, Vector>(),
        a, ap, b, c, u, p, f, g, EYE, max_iter, tol, p_cres);
}
//>cg_abtbc:  

}// namespace rheolef

# endif // _RHEO_CG_ABTBC_H 
