# ifndef _RHEO_CSR_H
# define _RHEO_CSR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/vec.h"
#include "rheolef/asr.h"
#include "rheolef/vector_of_iterator.h"
#include "rheolef/scatter_message.h"
namespace rheolef {

// -------------------------------------------------------------
// the sequential representation
// -------------------------------------------------------------
template<class T>
class csr_seq_rep : public vector_of_iterator<std::pair<typename std::vector<T>::size_type,T> > {
public:
    typedef typename std::vector<T>::size_type                            size_type;
    typedef T                                                             element_type;
    typedef sequential                                                    memory_type;
    typedef typename std::pair<size_type,T>                               pair_type;
    typedef typename vector_of_iterator<pair_type>::value_type            data_iterator;
    typedef typename vector_of_iterator<pair_type>::const_value_type      const_data_iterator;
    typedef typename vector_of_iterator<pair_type>::iterator              iterator;
    typedef typename vector_of_iterator<pair_type>::const_iterator        const_iterator;

    csr_seq_rep (size_type loc_nrow1 = 0, size_type loc_ncol1 = 0, size_type loc_nnz1 = 0);
    void resize (size_type loc_nrow1 = 0, size_type loc_ncol1 = 0, size_type loc_nnz1 = 0);
    csr_seq_rep (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1 = 0);
    void resize (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1 = 0);
    csr_seq_rep (const csr_seq_rep<T>& a);
    explicit csr_seq_rep (const asr_seq_rep<T>& a);
    void to_asr (asr_seq_rep<T>& b) const;

    const distributor& row_ownership() const { return _row_ownership; }
    const distributor& col_ownership() const { return _col_ownership; }
    const_iterator begin() const { return vector_of_iterator<pair_type>::begin(); }
    const_iterator end()   const { return vector_of_iterator<pair_type>::end(); }
          iterator begin()       { return vector_of_iterator<pair_type>::begin(); }
          iterator end()         { return vector_of_iterator<pair_type>::end(); }
    size_type nrow() const       { return vector_of_iterator<pair_type>::size()-1; }
    size_type ncol() const { return _col_ownership.size(); }
    size_type nnz() const { return _data.size(); }
    size_type par_nrow() const { return nrow(); }
    size_type par_ncol() const { return ncol(); }
    size_type par_nnz() const { return nnz(); }
    size_type row_first_index () const { return 0; }
    size_type row_last_index () const { return nrow(); }
    size_type col_first_index () const { return 0; }
    size_type col_last_index () const { return ncol(); }
    size_type ext_nnz() const { return 0; }
    iparstream& get (iparstream&);
    oparstream& put (oparstream&, size_type istart = 0) const;
    void dump (const std::string& name, size_type istart = 0) const;
    void mult (const vec<T,sequential>& x, vec<T,sequential>& y) const;
    template <class BinaryOp>
    void assign_add (const csr_seq_rep<T>& a, const csr_seq_rep<T>& b, BinaryOp binop);
//protected:
    distributor          		                          _row_ownership;
    distributor          		                          _col_ownership;
    std::vector<std::pair<typename std::vector<T>::size_type,T> > _data;
};
// -------------------------------------------------------------
// the distributed representation
// -------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template<class T>
class csr_mpi_rep : public csr_seq_rep<T> {
public:
    typedef typename csr_seq_rep<T>::size_type size_type;
    typedef typename csr_seq_rep<T>::element_type element_type;
    typedef distributed memory_type;
    typedef typename csr_seq_rep<T>::iterator iterator;
    typedef typename csr_seq_rep<T>::const_iterator const_iterator;
    typedef typename csr_seq_rep<T>::data_iterator data_iterator;
    typedef typename csr_seq_rep<T>::const_data_iterator const_data_iterator;

    csr_mpi_rep ();
    csr_mpi_rep (const csr_mpi_rep<T>& a);
    explicit csr_mpi_rep (const asr_mpi_rep<T>& a);
    void to_asr (asr_mpi_rep<T>& b) const;

    const distributor& row_ownership() const { return csr_seq_rep<T>::_row_ownership; }
    const distributor& col_ownership() const { return csr_seq_rep<T>::_col_ownership; }
    const communicator& comm() const { return row_ownership().comm(); }
    const_iterator begin() const { return csr_seq_rep<T>::begin(); }
    const_iterator end()   const { return csr_seq_rep<T>::end(); }
          iterator begin()       { return csr_seq_rep<T>::begin(); }
          iterator end()         { return csr_seq_rep<T>::end(); }
    size_type      ext_nnz()  const { return _ext.nnz(); }
    const_iterator ext_begin() const { return _ext.begin(); }
    const_iterator ext_end()   const { return _ext.end(); }
    size_type nrow() const { return csr_seq_rep<T>::nrow(); }
    size_type ncol() const { return csr_seq_rep<T>::ncol(); }
    size_type nnz() const { return csr_seq_rep<T>::nnz(); }
    size_type par_nrow() const { return row_ownership().par_size(); }
    size_type par_ncol() const { return col_ownership().par_size(); }
    size_type par_nnz() const { return _par_nnz; }
    size_type row_first_index () const { return row_ownership().first_index(); }
    size_type row_last_index ()  const { return row_ownership().last_index(); }
    size_type col_first_index () const { return col_ownership().first_index(); }
    size_type col_last_index ()  const { return col_ownership().last_index(); }
    size_type jext2par_j (size_type jext)  const;
    iparstream& get (iparstream&);
    oparstream& put (oparstream&) const;
    void dump (const std::string& name) const;
    void mult(const vec<T,distributed>& x, vec<T,distributed>& y) const;
    template <class BinaryOp>
    void assign_add (const csr_mpi_rep<T>& a, const csr_mpi_rep<T>& b, BinaryOp binop);
protected:
    // diagonal part is the basic csr_seq_rep type
    // extra-diagonal blocs are sequential csr also:
    csr_seq_rep<T>       		_ext;
    std::vector<size_type>    		_jext2par_j;
    size_type            		_par_nnz;
    
    // A*x internal stuff: scatter and buffer:
    mutable scatter_message<std::vector<T> > _from;
    mutable scatter_message<std::vector<T> > _to;
    mutable std::vector<T>    		     _buffer;
};
template<class T>
inline
typename csr_mpi_rep<T>::size_type
csr_mpi_rep<T>::jext2par_j (size_type jext)  const
{
  check_macro (jext < _jext2par_j.size(), "jext2par_j: jext="<<jext<<" is out of range [0:"<<_jext2par_j.size()<<"[");
  return _jext2par_j [jext];
}
#endif // _RHEOLEF_HAVE_MPI
// -------------------------------------------------------------
// the basic class with a smart pointer to representation
// the user-level class with memory-model parameter
// -------------------------------------------------------------
/*Class:csr
NAME:  csr - compressed sparse row matrix (@PACKAGE@-@VERSION@)
SYNOPSYS:       
 Distributed compressed sparse matrix container stored row by row.
TO DO:
   For efficiency purpose, the assembly phase may
   access directly to the csr representation, without
   crossing the reference counting and pointer handler.
   Something like the iterator for dense vectors.
SEE ALSO: "asr"(1), "parstream"(1), "distributor"(1), "parallel"(1)
AUTHORS: Pierre.Saramito@imag.fr
DATE:   10 february 1999
METHODS: @csr
End:
*/
//csr:
template<class R>
class csr_basic : public smart_pointer<R> {
public:

// typedefs:

    typedef typename R::size_type      size_type;
    typedef typename R::element_type   element_type;
    typedef typename R::memory_type    memory_type;
    typedef typename R::iterator       iterator;
    typedef typename R::const_iterator const_iterator;
    typedef typename R::const_data_iterator const_data_iterator;

// allocators/deallocators:

    csr_basic();
    explicit csr_basic(const asr<element_type>&);

// accessors:

    // local sizes
    size_type nrow () const;
    size_type ncol () const;
    size_type nnz () const;
    
    // global sizes
    size_type par_nrow () const;
    size_type par_ncol () const;
    size_type par_nnz () const;
    const distributor& row_ownership() const;
    const distributor& col_ownership() const;

    // range on local memory
    size_type row_first_index () const;
    size_type row_last_index () const;
    size_type col_first_index () const;
    size_type col_last_index () const;

    const_iterator begin() const;
    const_iterator end()   const;
    // accessors, only for R=distributed; TODO: R=sequential fix, e.g. ext_begin==ext_end
    size_type ext_nnz() const;
    const_iterator ext_begin() const;
    const_iterator ext_end()   const;
    size_type jext2par_j (size_type jext) const;

// output:

    void dump (const std::string& name) const;

// algebra:    

    void mult (const vec<element_type,memory_type>& x, vec<element_type,memory_type>& y) const;
    vec<element_type,memory_type> operator* (const vec<element_type,memory_type>& x) const;
};
template <class T, class M = rheo_default_memory_model>
class csr {
public:
    typedef M memory_type;
};
template <class T>
class csr<T,sequential> : public csr_basic<csr_seq_rep<T> > {
public:
    typedef sequential                 memory_type;
    typedef csr_basic<csr_seq_rep<T> > basic_type;
    csr();
    explicit csr(const asr<T,memory_type>&);
    csr<T,memory_type> operator+ (const csr<T,memory_type>& b) const;
    csr<T,memory_type> operator- (const csr<T,memory_type>& b) const;
};
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class csr<T,distributed> : public csr_basic<csr_mpi_rep<T> > {
public:
    typedef distributed                memory_type;
    typedef csr_basic<csr_mpi_rep<T> > basic_type;
    csr();
    explicit csr(const asr<T,memory_type>&);
    csr<T,memory_type> operator+ (const csr<T,memory_type>& b) const;
    csr<T,memory_type> operator- (const csr<T,memory_type>& b) const;
};
#endif // _RHEOLEF_HAVE_MPI

// inputs/outputs:
template <class T, class M>
iparstream& operator >> (iparstream& s, csr<T,M>& x);

template <class T, class M>
oparstream& operator << (oparstream& s, const csr<T,M>& x);
//>csr:
// -------------------------------------------------------------
// inline'd: csr_basic
// -------------------------------------------------------------
template <class R>
inline
csr_basic<R>::csr_basic ()
 : smart_pointer<R> (new_macro(R()))
{
}
template <class R>
inline
csr_basic<R>::csr_basic (const asr<typename R::element_type>& a)
 : smart_pointer<R> (new_macro(R(a.data())))
{
}
template <class R>
inline
const distributor&
csr_basic<R>::row_ownership () const
{
    return smart_pointer<R>::data().row_ownership();
}
template <class R>
inline
const distributor&
csr_basic<R>::col_ownership () const
{
    return smart_pointer<R>::data().col_ownership();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::nrow () const
{
    return smart_pointer<R>::data().nrow();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::ncol () const
{
    return smart_pointer<R>::data().ncol();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::nnz () const
{
    return smart_pointer<R>::data().nnz();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::par_nrow () const
{
    return smart_pointer<R>::data().par_nrow();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::par_ncol () const
{
    return smart_pointer<R>::data().par_ncol();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::par_nnz () const
{
    return smart_pointer<R>::data().par_nnz();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::row_first_index () const
{
    return smart_pointer<R>::data().row_first_index();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::row_last_index () const
{
    return smart_pointer<R>::data().row_last_index();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::col_first_index () const
{
    return smart_pointer<R>::data().col_first_index();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::col_last_index () const
{
    return smart_pointer<R>::data().col_last_index();
}
template <class R>
inline
typename csr_basic<R>::const_iterator
csr_basic<R>::begin () const
{
    return smart_pointer<R>::data().begin();
}
template <class R>
inline
typename csr_basic<R>::const_iterator
csr_basic<R>::end () const
{
    return smart_pointer<R>::data().end();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::ext_nnz () const
{
    return smart_pointer<R>::data().ext_nnz();
}
template <class R>
inline
typename csr_basic<R>::const_iterator
csr_basic<R>::ext_begin () const
{
    return smart_pointer<R>::data().ext_begin();
}
template <class R>
inline
typename csr_basic<R>::const_iterator
csr_basic<R>::ext_end () const
{
    return smart_pointer<R>::data().ext_end();
}
template <class R>
inline
typename csr_basic<R>::size_type
csr_basic<R>::jext2par_j (size_type jext) const
{
    return smart_pointer<R>::data().jext2par_j (jext);
}
template <class R>
inline
void
csr_basic<R>::dump(const std::string& name) const
{
    smart_pointer<R>::data().dump(name);
}
template <class R>
inline
void
csr_basic<R>::mult(
    const vec<typename R::element_type,typename R::memory_type>& x, 
    vec<typename R::element_type,typename R::memory_type>&       y)
    const
{
    smart_pointer<R>::data().mult(x,y);
}
template <class R>
inline
vec<typename csr_basic<R>::element_type, typename csr_basic<R>::memory_type>
csr_basic<R>::operator* (const vec<element_type,memory_type>& x) const
{
    vec<element_type,memory_type> y (row_ownership(), element_type());
    mult (x, y);
    return y;
}
// -------------------------------------------------------------
// inline'd: csr_basic
// -------------------------------------------------------------
template <class T>
inline
csr<T,sequential>::csr ()
 : csr_basic<csr_seq_rep<T> >()
{
}
template <class T>
inline
csr<T,sequential>::csr (const asr<T,sequential>& a)
 : csr_basic<csr_seq_rep<T> >(a)
{
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
inline
csr<T,distributed>::csr ()
 : csr_basic<csr_mpi_rep<T> >()
{
}
template <class T>
inline
csr<T,distributed>::csr (const asr<T,distributed>& a)
 : csr_basic<csr_mpi_rep<T> >(a)
{
}
# endif // _RHEOLEF_HAVE_MPI
template <class T, class M>
inline
iparstream&
operator >> (iparstream& s,  csr<T,M>& x)
{ 
    return x.data().get(s); 
}
template <class T, class M> 
inline
oparstream&
operator << (oparstream& s, const csr<T,M>& x)
{
    return x.data().put(s); 
}
// ------------------------------
// a+b, a-b
// ------------------------------
template <class T> 
inline
csr<T,sequential>
csr<T,sequential>::operator+ (const csr<T,sequential>& b) const {
  csr<T,sequential> c;
  c.data().assign_add (this->data(), b.data(), std::plus<T>());
  return c;
}
template <class T> 
inline
csr<T,sequential>
csr<T,sequential>::operator- (const csr<T,sequential>& b) const {
  csr<T,sequential> c;
  c.data().assign_add (this->data(), b.data(), std::minus<T>());
  return c;
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T> 
inline
csr<T,distributed>
csr<T,distributed>::operator+ (const csr<T,distributed>& b) const {
  csr<T,distributed> c;
  c.data().assign_add (this->data(), b.data(), std::plus<T>());
  return c;
}
template <class T> 
inline
csr<T,distributed>
csr<T,distributed>::operator- (const csr<T,distributed>& b) const {
  csr<T,distributed> c;
  c.data().assign_add (this->data(), b.data(), std::minus<T>());
  return c;
}
#endif // _RHEOLEF_HAVE_MPI
  
} // namespace rheolef
# endif // _RHEO_CSR_H
