#ifndef _RHEOLEF_SORT_WITH_PERMUTATION_H
#define _RHEOLEF_SORT_WITH_PERMUTATION_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*
   sort_with_permutation - Computes the permutation of values that gives 
   a sorted sequence.

   Input Parameters:
       n  - number of values to sort
       i  - values to sort
       vdx - permutation array.  Must be initialized to 0:n-1 on input.

   Notes:
       i is unchanged on output.
       inspirated from petsc-2.0.22/sortip.c
*/

#include "rheolef/compiler.h"
namespace rheolef {

template<class Size, class SizeRandomIterator, class T>
void 
qsort_with_permutation(
    T                  *v,
    SizeRandomIterator vdx,
    Size               right)
{
    if (right <= 1) {
        if (right == 1) {
          if (v[vdx[0]] > v[vdx[1]]) {
	      std::swap (vdx[0], vdx[1]);
	  }
        }
	return;
    }
    std::swap (vdx[0], vdx[right/2]);
    Size vl   = v[vdx[0]];
    Size last = 0;
    for (Size i = 1; i <= right; i++) {
        if (v[vdx[i]] < vl ) {
	    last++; 
	    std::swap (vdx[last], vdx[i]);
	}
    }
    std::swap (vdx[0], vdx[last]);
    qsort_with_permutation (v, vdx,        last-1);
    qsort_with_permutation (v, vdx+last+1, right-(last+1));
}
template<class Size, class SizeRandomIterator, class T>
void
sort_with_permutation(
    Size               n, 
    T                  *v, 
    SizeRandomIterator vdx)
{
    //const size_t n_qsort_min = 8; // buggy qsort
    const size_t n_qsort_min = 80000; // test purpose
    if (n >= n_qsort_min) {
	warning_macro ("qsort...");
        qsort_with_permutation(v,vdx,n-1);
	warning_macro ("qsort done");
	return;
    }
    warning_macro ("bsort...");
    for (Size k = 0; k < n; k++) {
        Size vk = v [vdx [k]];
        for (Size j = k+1; j < n; j++) {
	    if (vk > v [vdx [j]]) {
	        std::swap (vdx[k], vdx[j]);
	        vk = v [vdx [k]];
	    }
        }
    }
    warning_macro ("bsort done");
}

} // namespace rheolef
#endif // _RHEOLEF_SORT_WITH_PERMUTATION_H
