#ifndef BOOST_PP_IS_ITERATING
#ifndef _RHEOLEF_POLYMORPHIC_DATA_MAP_H
#define _RHEOLEF_POLYMORPHIC_DATA_MAP_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/polymorphic_data_vector.h"

#include <map>
#include <typeinfo>
#include <boost/array.hpp>
#include <boost/mpl/vector.hpp>
#include <boost/mpl/size.hpp>
#include <boost/mpl/for_each.hpp>
#include <boost/mpl/at.hpp>

namespace rheolef {
namespace mpl = boost::mpl;

// -----------------------------------------------------------------------
// some auxilliary functions
// -----------------------------------------------------------------------
/// @brief transform a map::iterator into a void*
template <class MapTk>
void*
to_pointer (typename MapTk::iterator iter_tk)
{
  return static_cast<void*>(iter_tk._M_node); // WARNING: depend upon GNU C++ STL (HP & SGI)
}
/// @brief transform a void* into a map::iterator
template <class MapTk>
typename MapTk::iterator
to_iterator (void* ptr)
{
  typedef typename MapTk::iterator iterator;
  typedef typename iterator::_Link_type link_type;
  return iterator (static_cast<link_type>(ptr));
}
// -----------------------------------------------------------------------
// Advantage of this class: allows replacement of a i-th element in the table
//   by another that could have a different derived type.
//   possible application: replace a triangle by a curvilinear triangle
//   on the boundary. Suitable for P2 boundary transformation.
//
/// @brief dynamic memory area handler for polymorphic_array
template <class T, class V, int NV>
struct polymorphic_data_map {};

// --------------------------------------------------------------------------------
// new_entry<Tk>() returns a pointer Tk* newly allocated into the data structure
// via a push_back. We need a specialisation of a template member function
// into a template class. This is not allowed... So, we use the same trick as 
// for the "reserve(n)" member: we use a template class-function, with specialisation.
//
template <class Tk, class T, class V, int NV>
struct my_new_entry_map_t {
  Tk* operator() (polymorphic_data_map<T,V,NV>& x) {
    error_macro ("unexpected type " << typeid_name_macro(Tk));
    return 0;
  }
};
// ----------------------------------------------------------
// run here the recursive inclusion of this file:
// ----------------------------------------------------------
// _RHEOLEF_POLYMORPHIC_MAX_SIZE was defined by "polymorphic_data_vector.h"
#define BOOST_PP_ITERATION_LIMITS (0, _RHEOLEF_POLYMORPHIC_MAX_SIZE)
#define BOOST_PP_FILENAME_1    "rheolef/polymorphic_data_map.h" // this file
#include BOOST_PP_ITERATE()

} // namespace rheolef
#endif // _RHEOLEF_POLYMORPHIC_DATA_MAP_H
#else  // BOOST_PP_IS_ITERATING
// ----------------------------------------------------------
// here, this file is recursively included with growing "N" :
// ----------------------------------------------------------
#define N    BOOST_PP_ITERATION()

template <class T, class V>
struct polymorphic_data_map<T,V,N> {

// typedefs:

    static const size_t _n_variant  = mpl::size<V>::value; // should be = N
    typedef typename std::vector<int>::size_type size_type;

#define _RHEOLEF_typedef(z,k,unused)                    		\
    typedef typename mpl::at_c<V,k>::type T##k;				\
    typedef std::map <size_type, T##k, std::less<size_type>, 		\
	heap_allocator<std::pair<size_type, T##k> > >  map##k##_type;
    BOOST_PP_REPEAT(N, _RHEOLEF_typedef, ~)
#undef _RHEOLEF_typedef

// cstors :

#define _RHEOLEF_stack_initializer(z,k,unused)                  	\
        _stack_##k(), 

    polymorphic_data_map(size_type n=0) :
  	BOOST_PP_REPEAT(N, _RHEOLEF_stack_initializer, ~)
        _counter(),
        _size(0),
        _max_size(n)
    {
        std::fill (_counter.begin(), _counter.end(), 0);
    }
    void resize (size_type n) {
        _max_size = n;
    }

#undef _RHEOLEF_stack_initializer

// accessors & modifiers:


    size_type size () const { return _max_size; }
    size_type size (size_type k)    const { return _counter[k]; }
    const size_type* begin_sizes () const { return _counter.begin(); }

    void erase (size_type index);

#define _RHEOLEF_assign(z,k,unused)  		         				\
    void assign (size_type index, const T##k& value);
    BOOST_PP_REPEAT(N, _RHEOLEF_assign, ~)
#undef _RHEOLEF_assign

    void reset () {

#define _RHEOLEF_stack_clear(z,k,unused)                  	\
	_stack_##k.clear();
  	BOOST_PP_REPEAT(N, _RHEOLEF_stack_clear, ~)
#undef _RHEOLEF_stack_clear

	std::fill (_counter.begin(), _counter.end(), 0);
	_size = 0;
    }
    void update_sizes () { 

#define _RHEOLEF_stack_set_counter(z,k,unused)                 	\
	_counter[k] = _stack_##k.size();
  	BOOST_PP_REPEAT(N, _RHEOLEF_stack_set_counter, ~)
#undef _RHEOLEF_stack_set_counter

	_size = 0;
	for (size_type k = 0; k < _n_variant; k++) {
          _size += _counter[k];
	}
    }

    template <class Tk> Tk* new_entry ();

// data:

#define _RHEOLEF_stack_declare(z,k,unused)                  	\
    map##k##_type                               _stack_##k;
    BOOST_PP_REPEAT(N, _RHEOLEF_stack_declare, ~)
#undef _RHEOLEF_stack_declare

    boost::array<size_type,_n_variant>	      _counter;
    size_type				      _size;
    size_type				      _max_size;

};
// asignment: x[i] = value
// it's more complex: if value is of type Tk, and x[i] points to a Tl != Tk
// then, we have to clear the old x[i] object from the l-eme map structure,
// en ayant avant sauvegarde' son index (qui doit etre egal a i..)
// et inserer valeur dans la Tk-stack avec cet index
template <class T, class V>
void
polymorphic_data_map<T,V,N>::erase (size_type global_index) {
#define _RHEOLEF_erase(z,k,unused)  		                  			\
    typename map##k##_type::iterator iter_##k = _stack_##k.find(global_index); 		\
    if (iter_##k != _stack_##k.end()) {							\
        _stack_##k.erase (iter_##k);							\
        _counter[k]--;									\
        _size--;									\
    }
    BOOST_PP_REPEAT(N, _RHEOLEF_erase, ~)
#undef _RHEOLEF_erase
}

#define _RHEOLEF_assign(z,k,unused)  		         				\
template <class T, class V>								\
void 											\
polymorphic_data_map<T,V,N>::assign (size_type global_index, const T##k& value) {	\
    /* erase any value with the same index */						\
    erase (global_index);								\
    /* here, the location is empty: insert */						\
    typedef typename map##k##_type::iterator iterator;					\
    std::pair<iterator,bool> status							\
         = _stack_##k.insert (std::pair<size_type,T##k>(global_index,value));		\
    check_macro (status.second, "insert failed");					\
    _counter[k]++;									\
    _size++;										\
}
BOOST_PP_REPEAT(N, _RHEOLEF_assign, ~)
#undef _RHEOLEF_assign

// convert a data_map into a data(vector)
template <class T, class V>
void
polymorphic_data_vector<T,V,N>::build (const class polymorphic_data_map<T,V,N>& pool)
{
#define _RHEOLEF_stack_copy(z,k,unused)                  			\
    _stack_##k.resize (pool._stack_##k.size());					\
    copy (pool._stack_##k.begin(), pool._stack_##k.end(), _stack_##k.begin());

    BOOST_PP_REPEAT(N, _RHEOLEF_stack_copy, ~)
    std::copy (pool._counter.begin(), pool._counter.end(), _counter.begin());
    _size = pool._size;

#undef _RHEOLEF_stack_copy
}
// specialization for each Tk:
#define _RHEOLEF_new_entry(z,k,unused)                  			\
template <class T, class V>							\
struct my_new_entry_map_t <typename polymorphic_data_map<T,V,N>::T##k, T,V,N> { \
  typedef typename polymorphic_data_map<T,V,N>::T##k T##k;			\
  typedef typename polymorphic_data_map<T,V,N>::size_type size_type;		\
  T##k* operator()  (polymorphic_data_map<T,V,N>& x) {				\
        typedef typename polymorphic_data_map<T,V,N>::map##k##_type map_type;	\
        typedef typename map_type::iterator                  iterator;		\
	check_macro (x._size < x._max_size, "too many entries");		\
        const size_type variant = k;						\
        std::pair<iterator,bool> status 					\
	 = x._stack_##k.insert (std::pair<size_type,T##k> (x._size, T##k()));	\
	check_macro (status.second, "insert failed");				\
        iterator iter = status.first;						\
	x._counter[variant]++;							\
	x._size++;								\
        T##k* p = &((*iter).second);						\
	return p;								\
  }										\
};
BOOST_PP_REPEAT(N, _RHEOLEF_new_entry, ~)
#undef _RHEOLEF_new_entry

template <class T, class V>
template <class Tk>
inline
Tk*
polymorphic_data_map<T,V,N>::new_entry ()
{
  my_new_entry_map_t<Tk,T,V,N> new_entry_f;
  return new_entry_f (*this);
}

#endif // BOOST_PP_IS_ITERATING
