#ifndef _RHEO_SMART_POINTER_H
#define _RHEO_SMART_POINTER_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// smart pointers
//
// author: Pierre.Saramito@imag.fr
//
// date: 27 january 2000
//
// ref: page 118:
// @book{geron-tawbi-1999,
// 	author = {A. G\'eron and F. Tawbi},
// 	title = {Pour mieux d\'evelopper avec {\tt C++}~:
// 		design pattern, {STL}, {RTTI} et
// 		smart pointers},
// 	publisher = {InterEditions},
// 	year = 1999,
// }
// 
/*Class:occurence
NAME:  @code{occurence}, @code{smart_pointer} - memory management 
@clindex occurence
@clindex smart_pointer
@clindex string
@cindex reference counting
@cindex shallow copy
@cindex smart pointer
DESCRIPTION:       
 @noindent
 Here is a convenient way to implement a true copy semantc,
 by using shallow copies and reference counting, in order to 
 minimise memory copies.
 This concept is generally related to the @dfn{smart pointer}
 method for managing memory.

 @noindent
 The true semantic copy is defined as follows: if an object 
 @code{A} is assigned to
 @code{B}, such as @code{A = B}, every further modification on @code{A} or @code{B}
 does not modify the other.
End:
*/

#include "rheolef/compiler.h"

// -----------------------------------------------------------------------
// smart_pointer
// -----------------------------------------------------------------------
namespace rheolef { 
//<occurence:
template <class T>
class smart_pointer {
public:

// allocators:

	smart_pointer (T* p = 0);
	smart_pointer (const smart_pointer&);
	~smart_pointer ();
	smart_pointer& operator= (const smart_pointer&);

// accessors:

	const T* pointer    () const;
	const T& data       () const;
	const T* operator-> () const;
	const T& operator*  () const;

// modifiers:

	T* pointer ();
	T* operator-> ();
	T& data ();
	T& operator* ();

// implementation:

private:
	struct counter {
		T*  _p;
		int _n;
		counter (T* p = 0);
		~counter ();
		int operator++ ();
		int operator-- ();
	};
	counter *_count;
};
//>occurence:
// -----------------------------------------------------------------------
// inlined
// -----------------------------------------------------------------------

template <class T>
inline
smart_pointer<T>::counter::counter (T* p)
 : _p(p), _n(1)
{
}
template <class T>
inline
smart_pointer<T>::counter::~counter ()
{
    delete_macro(_p);
}
template <class T>
inline
int
smart_pointer<T>::counter::operator++ ()
{
    return ++_n;
}
template <class T>
inline
int
smart_pointer<T>::counter::operator-- ()
{ 
    if (--_n != 0) return _n;
    delete_macro(this);
    return 0;
}
template <class T>
inline
smart_pointer<T>::smart_pointer (T* p)
: _count(new_macro(counter(p)))
{
}
template <class T>
inline
smart_pointer<T>::smart_pointer (const smart_pointer& sp)
 : _count(sp._count)
{
    ++(*_count);
}
template <class T>
inline
smart_pointer<T>::~smart_pointer ()
{
     if (_count != 0) { --(*_count); }
}
template <class T>
inline
smart_pointer<T>&
smart_pointer<T>::operator= (const smart_pointer& sp)
{
    if (_count != sp._count) { 
	--(*_count);
	_count = sp._count;
	++(*_count);
    }
    return *this;
}
template <class T>
inline
const T*
smart_pointer<T>::pointer () const
{
    return _count -> _p;
}
template <class T>
inline
const T&
smart_pointer<T>::data () const
{
    return *pointer();
}
template <class T>
inline
const T*
smart_pointer<T>::operator-> () const
{
    return pointer();
}
template <class T>
inline
const T&
smart_pointer<T>::operator* () const
{
    return data();
}
template <class T>
inline
T*
smart_pointer<T>::pointer ()
{
    if (_count -> _p == 0) return 0;
    if (_count -> _n > 1) {
   	--(_count-> _n);
	T* q = new_macro (T(*(_count -> _p)));
       	_count = new_macro (counter(q));
    }	
    return _count -> _p;
}
template <class T>
inline
T*
smart_pointer<T>::operator-> ()
{
     return pointer();
}
template <class T>
inline
T&
smart_pointer<T>::data ()
{
    return *pointer();
}
template <class T>
inline
T&
smart_pointer<T>::operator* ()
{
    return data();
}
}// namespace rheolef
#endif // _RHEO_SMART_POINTER_H
