///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/field.h"
#include <complex>

namespace rheolef {

template <class T, class M>
field_basic<T,M>::field_basic (
	const space_type& V,
	const T& init_value)
 : _V (V), 
   u (),
   b ()
{
   u.resize (_V.iu_ownership(), init_value);
   b.resize (_V.ib_ownership(), init_value);
}
template <class T, class M>
void 
field_basic<T,M>::resize (
	const space_type& V,
	const T& init_value)
{
   if (_V == V) return;
   _V = V;
   u.resize (_V.iu_ownership(), init_value);
   b.resize (_V.ib_ownership(), init_value);
}
template <class T, class M>
idiststream& 
field_basic<T,M>::get (idiststream& ips)
{
warning_macro ("field::get...");
  communicator comm = ips.comm();
  if ( ! dis_scatch (ips, comm, "\nfield")) {
    error_macro ("read field failed");
    return ips;
  }
  size_type      version,  dis_size1;
  std::string    geo_name, approx;
  ips >> version >> dis_size1
      >> geo_name
      >> approx;
  geo_type omega (geo_name);

  if (!(_V.get_geo().name() == geo_name && _V.get_element().name() == approx)) {
    // cannot re-use _V: build a new space and resize the field
warning_macro ("field::get: resize...");
    resize (space_type (omega,approx));
warning_macro ("field::get: resize done");
  }
  // TODO: read and permut in one pass ? avoid 2 passes of global exchanges
  vec<T,M> u_io  (_V.ios_ownership(), std::numeric_limits<T>::max());
  u_io.get_values (ips);
  vec<T,M> u_new (_V.ownership(),     std::numeric_limits<T>::max());
  for (size_type ios_idof = 0, ios_ndof = _V.ios_ownership().size(); ios_idof < ios_ndof; ios_idof++) {
    T value = u_io [ios_idof];
    size_type dis_idof = _V.ios_idof2dis_idof (ios_idof);
    u_new.dis_entry (dis_idof) = value;
  }
  u_new.dis_entry_assembly();

  for (size_type idof = 0, ndof = _V.ownership().size(); idof < ndof; idof++) {
    dof (idof) = u_new [idof];
  }
warning_macro ("field::get done");
  return ips;
}
template <class T, class M>
odiststream& 
field_basic<T,M>::put (odiststream& ops) const
{
    using namespace std;
    // merge distributed blocked and non-blocked and apply iso_dof permutation:
    vec<T,M> u_io (_V.ios_ownership(), std::numeric_limits<T>::max());
    warning_macro ("V.ios_ownership.size=" << _V.ios_ownership().size());
    warning_macro ("V.ownership.size=" << _V.ownership().size());
    for (size_type idof = 0, ndof = _V.size(); idof < ndof; idof++) {
        T value = dof (idof);
        size_type ios_dis_idof = _V.idof2ios_dis_idof (idof);
        u_io.dis_entry (ios_dis_idof) = value;
    }
    u_io.dis_entry_assembly();  

    // then output
    ops << setprecision(std::numeric_limits<Float>::digits10)
        << "field" << endl
        << "1 " << u_io.dis_size() << endl 
        << _V.get_geo().name() << endl
        << _V.get_element().name() << endl
        << endl
        << u_io;
    
    return ops;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation_base(T,M) 				\
template class field_basic<T,M>;					\
template odiststream& operator << (odiststream&, const field_basic<T,M>&);

#ifdef TODO
#define _RHEOLEF_instanciation(T,M) 					\
_RHEOLEF_instanciation_base(T,M) 					\
_RHEOLEF_instanciation_base(std::complex<T>,M)
#else // TODO
#define _RHEOLEF_instanciation(T,M) 					\
_RHEOLEF_instanciation_base(T,M)
#endif // TODO

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
