///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/space.h"

namespace rheolef {

// ---------------------------------------------------------------
// space_base_rep
// ---------------------------------------------------------------
template <class T, class M>
space_base_rep<T,M>::space_base_rep (
    const geo_basic<T,M>& omega_in,
    std::string approx)
  : _constit(omega_in,approx), 
    _is_freezed(false),
    _idof2blk_iub(),
    _iu_ownership(),
    _ib_ownership()
{
    // omega_in is compressed by space_constitution() allocator
    // when it is a domain: then it becomes a geo_domain, with compressed numbering
    // so, omega_in can be different from omega:
    const geo_basic<T,M>& omega = get_geo();

    if (approx == "") return; // empty element => default cstor
    size_type ndof     = _constit.get_element().get_numbering().ndof (
				get_geo().map_dimension(),
				get_geo().size_by_dimension(),
				get_geo().size_by_variant());
    size_type dis_ndof = _constit.get_element().get_numbering().ndof (
				get_geo().map_dimension(),
				get_geo().dis_size_by_dimension(),
				get_geo().dis_size_by_variant());

    distributor idof_ownership (dis_ndof, comm(), ndof);
    _idof2blk_iub.resize (idof_ownership);
    check_macro (idof_ownership.dis_size() == dis_ndof, "invalid dis_ndof count");
    check_macro (idof_ownership.size()     == ndof,     "invalid ndof count");
}
template <class T, class M>
void
space_base_rep<T,M>::base_freeze_body () const
{
    check_macro (get_element().name() != "", "space with undefined finite element method cannot be used");
    // -----------------------------------------------------------------------
    // loop on domains: mark blocked dofs
    // -----------------------------------------------------------------------
    const geo_basic<T,M>& omega = get_geo();
    size_type first_dis_idof = ownership().first_index();
    size_type last_dis_idof  = ownership().last_index();
    array<size_type,M> blocked_flag (ownership(), 0); // array<bool,M> not supported
    for (typename space_constitution<T,M>::const_iterator
	iter = _constit.begin(),
	last = _constit.end(); iter != last; iter++) {
      const domain_indirect_basic<M>&     dom = (*iter).get_domain_indirect();
      typename space_act<M>::act_type act = (*iter).get_act();
      size_type dom_dim = dom.map_dimension();
      distributor ige_ownership = omega.geo_element_ownership (dom_dim);
      size_type   first_dis_ige = ige_ownership.first_index();
      size_type    last_dis_ige = ige_ownership.last_index();
      bool blk = (act == space_act<M>::block) ? true : false;
      std::vector<size_type> dis_idof;
      geo_element_p P;
      for (size_type ioige = 0, noige = dom.size(); ioige < noige; ioige++) {
        size_type ige     = dom.oige (ioige).index();
        size_type dis_ige = ige + first_dis_ige;
#ifdef TO_CLEAN
        if (dom_dim > 0) {
#endif // TO_CLEAN
          const geo_element& S = omega.get_geo_element (dom_dim, ige);
          get_dis_idof (S, dis_idof);
#ifdef TO_CLEAN
        } else {
          // fix for the d=0 domains of vertices:
          P[0] = dis_ige;
          get_dis_idof (P, dis_idof);
        }
#endif // TO_CLEAN
        for (size_type iloc = 0, nloc = dis_idof.size(); iloc < nloc; iloc++) {
          if (dis_idof[iloc] >= first_dis_idof && dis_idof[iloc] < last_dis_idof) {
            size_type idof = dis_idof [iloc] - first_dis_idof;
            blocked_flag[idof] = blk;
	  } else {
            blocked_flag.dis_entry (dis_idof[iloc]) = blk;
          }
        }
      }
    }
    blocked_flag.dis_entry_assembly();
    // copy the blocked_flag into the iub array, as the "blocked" bit:
    for (size_type idof = 0, ndof = blocked_flag.size(); idof < ndof; idof++) {
      _idof2blk_iub [idof].set_blocked (blocked_flag[idof]);
    }
    // -----------------------------------------------------------------------
    // init numbering
    // -----------------------------------------------------------------------
    size_type n_unknown = 0;
    size_type n_blocked = 0;
    for (size_type idof = 0, ndof = _idof2blk_iub.size(); idof < ndof; idof++) {
     
        bool blk = _idof2blk_iub [idof].is_blocked();
        if (! blk) {
          _idof2blk_iub[idof].set_iub (n_unknown);
          n_unknown++;
        } else {
          _idof2blk_iub[idof].set_iub (n_blocked);
          n_blocked++;
        }
    }
    size_type dis_n_unknown = n_unknown;
    size_type dis_n_blocked = n_blocked;
#ifdef _RHEOLEF_HAVE_MPI
    if (is_distributed<M>::value) {
      dis_n_unknown = mpi::all_reduce (comm(), dis_n_unknown, std::plus<T>());
      dis_n_blocked = mpi::all_reduce (comm(), dis_n_blocked, std::plus<T>());
    }
#endif // // _RHEOLEF_HAVE_MPI
    _iu_ownership = distributor (dis_n_unknown, comm(), n_unknown);
    _ib_ownership = distributor (dis_n_blocked, comm(), n_blocked);
}
// ----------------------------------------------------------------------------
// used by field & form_assembly
// ----------------------------------------------------------------------------
template <class T, class M>
void
space_base_rep<T,M>::get_dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const
{
    freeze_guard();
    size_type nloc = _constit.get_element().get_basis().size(K.variant());
    dis_idof.resize (nloc);
    for (size_type iloc = 0; iloc < nloc; iloc++) {
      dis_idof [iloc] = _constit.get_element().get_numbering().idof (
		get_geo().dis_size_by_dimension(),
		get_geo().dis_size_by_variant(),
		K,
		iloc);
    }
}
// ----------------------------------------------------------------------------
// stamp : e.g. "P1(square)", for field_expr<Expr> checks
// ----------------------------------------------------------------------------
template <class T, class M>
std::string
space_base_rep<T,M>::get_stamp() const
{
    if (get_geo().name() == "*nogeo*") return "";
    return get_element().name() + "(" + get_geo().name() + ")";
}
// ---------------------------------------------------------------
// space_rep<seq>
// ---------------------------------------------------------------
template <class T>
space_rep<T,sequential>::space_rep (
    const geo_basic<T,sequential>& omega,
    std::string approx)
  : space_base_rep<T,sequential>::space_base_rep (omega, approx)
{
}
// ----------------------------------------------------------------------------
// u["left"] 
// => build here the requiresd temporary indirect array
// ----------------------------------------------------------------------------
/**
 Implementation note: there is two numbering styles

   idof     : from the current space
   dom_idof : from a space compacted to the domain "dom", as geo_domain does

  This function returns the renumbering array "dom_idof2dof".
  It is a temporary, used at the fly when using the u[dom] syntax.

  In that case, the geo_domain is not fully builded for efficiency reasons.

  Nevertheless, the renumbering procedure *MAY* be exactly the same as
  those of geo_domain, for compatibility purpose when:

	geo dom = omega["boundary"]);
	space Vh (omega);
	space Wh (dom);
	uh[dom] = gh;

  For a similar renumbering algorithm, see also geo_rep<T,M>::build_from_domain
  when numbering the geo_domain Wh.geo vertices.
 */
template <class T, class M>
array<typename space_base_rep<T,M>::size_type, M>
space_base_rep<T,M>::build_indirect_array (
    const domain_indirect_basic<M>& indirect) const
{
  const geo_basic<T,M>& omega = get_geo();
  size_type first_dis_idof = ownership().first_index();
  size_type  last_dis_idof = ownership().last_index();
  array<size_type> idof_is_on_domain (ownership(), 0); // logical, init to "false"
  size_type map_dim = indirect.map_dimension();
  std::vector<size_type> dis_idof;
  //
  // 1) loop on domain elements and mark used idofs
  //
  for (size_type ioige = 0, noige = indirect.size(); ioige < noige; ioige++) {
    size_type ige = indirect.oige (ioige).index();
    const geo_element& S = omega.get_geo_element (map_dim, ige);
    get_dis_idof (S, dis_idof);
    for (size_type iloc = 0, nloc = dis_idof.size(); iloc < nloc; iloc++) {
      idof_is_on_domain.dis_entry (dis_idof[iloc]) += 1;
    }
  }
  idof_is_on_domain.dis_entry_assembly (set_add_op<size_type,size_type>()); // logical "or"
  //
  // 2) count marked idofs
  //
  size_type dom_ndof = 0;
  for (size_type idof = 0, ndof = idof_is_on_domain.size(); idof < ndof; idof++) {
    if (idof_is_on_domain[idof] != 0) dom_ndof++;
  }
  //
  // 3) numbering dom_idof & permutation: dom_idof --> idof
  //
  communicator comm = ownership().comm();
  distributor dom_idof_ownership (distributor::decide, comm, dom_ndof);
  array<size_type, M> dom_idof2idof (dom_idof_ownership);
  for (size_type dom_idof = 0, idof = 0, ndof = idof_is_on_domain.size(); idof < ndof; idof++) {
    if (idof_is_on_domain[idof] == 0) continue;
    dom_idof2idof [dom_idof] = idof;
    dom_idof++;
  }
  return dom_idof2idof;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class space_base_rep<Float,sequential>;
template class space_rep<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class space_base_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
