#ifndef _RHEO_BRANCH_H
#define _RHEO_BRANCH_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/field.h"
namespace rheolef { 

template <class T, class M> class __obranch;
template <class T, class M> class __iobranch;
template <class T, class M> class __branch_header;
template <class T, class M> class __const_branch_header;
template <class T, class M> class __const_branch_finalize;

/*Class:branch
@cindex animation
@cindex continuation methods
@cindex time-dependent problems
@clindex field

NAME:  @code{branch} - a parameter-dependent sequence of field
DESCRIPTION:       
  Stores a @code{field} sequence together with its associated parameter
  value: a @code{branch} variable represents a pair @code{(t,uh(t))} for
  a specific value of the parameter t.
  Applications concern time-dependent problems and continuation methods.

  This class is convenient for file inputs/outputs
  and building graphical animations.

EXAMPLES:
  Coming soon...

LIMITATIONS:
  This class is under development.

  The @code{branch} class store pointers on @code{field} class
  without reference counting. Thus, @code{branch} automatic 
  variables cannot be returned by functions. @code{branch}
  variable are limited to local variables.

AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France.
    @url{mailto:Pierre.Saramito@imag.fr}
END:
*/

// =============================================================================
// class definition
// =============================================================================

//<branch:  
template <class T, class M = rheo_default_memory_model>
class branch_basic : public std::vector<std::pair<std::string,field_basic<T,M> > > {
public :
// typedefs:

  typedef std::vector<std::pair<std::string,field_basic<T,M> > >     base;
  typedef typename base::size_type                                   size_type;
        
// allocators:

  branch_basic ();
  branch_basic (const std::string& parameter_name, const std::string& u0);
  branch_basic (const std::string& parameter_name, const std::string& u0, const std::string& u1);
  ~branch_basic();

// accessors:

  const T&  parameter () const;
  const std::string& parameter_name () const;
  size_type     n_value () const;
  size_type     n_field () const;

// modifiers:

  void set_parameter (const T& value);
  void set_range (const std::pair<T,T>& u_range);

// input/output:

  // get/set current value
  __obranch<T,M> operator()  (const T& t, const field_basic<T,M>& u0);
  __obranch<T,M> operator()  (const T& t, const field_basic<T,M>& u0, const field_basic<T,M>& u1);
  
  __iobranch<T,M> operator() (T& t, field_basic<T,M>& u0);
  __iobranch<T,M> operator() (T& t, field_basic<T,M>& u0, field_basic<T,M>& u1);

  __branch_header<T,M>         header ();
  __const_branch_header<T,M>   header () const;
  __const_branch_finalize<T,M> finalize () const;

//>branch:

// implementation

public:
  void put_header (odiststream&) const;
  void get_header (idiststream&);
  void put_finalize (odiststream&) const;

  template <class T1, class M1> friend idiststream& operator>> (idiststream&, branch_basic<T1,M1>&);
  template <class T1, class M1> friend odiststream& operator<< (odiststream&, const branch_basic<T1,M1>&);

  template <class T1, class M1> friend class  __obranch;
  template <class T1, class M1> friend class __iobranch;

// file formats
  void put_header_rheolef (odiststream&) const;
  void put_event_rheolef  (odiststream&) const;

  template <class T1> friend void put_header  (odiststream&, const branch_basic<T1,sequential>&);
  template <class T1> friend void put_event   (odiststream&, const branch_basic<T1,sequential>&);
  template <class T1> friend void put_finalize(odiststream&, const branch_basic<T1,sequential>&);

#ifdef _RHEOLEF_HAVE_MPI
  template <class T1> friend void put_header  (odiststream&, const branch_basic<T1,distributed>&);
  template <class T1> friend void put_event   (odiststream&, const branch_basic<T1,distributed>&);
  template <class T1> friend void put_finalize(odiststream&, const branch_basic<T1,distributed>&);
#endif // _RHEOLEF_HAVE_MPI

  template <class T1> friend void put_header_gnuplot  (odiststream&, const branch_basic<T1,sequential>&);
  template <class T1> friend void put_event_gnuplot   (odiststream&, const branch_basic<T1,sequential>&);
  template <class T1> friend void put_finalize_gnuplot(odiststream&, const branch_basic<T1,sequential>&);

  template <class T1> friend void put_header_paraview  (odiststream&, const branch_basic<T1,sequential>&);
  template <class T1> friend void put_event_paraview   (odiststream&, const branch_basic<T1,sequential>&);
  template <class T1> friend void put_finalize_paraview(odiststream&, const branch_basic<T1,sequential>&);

  void put_header_vtk    (odiststream&) const;
  void put_geometry_vtk  (odiststream&) const;
  void put_event_vtk     (odiststream&) const;
  void put_finalize_vtk  (odiststream&) const;

  void put_header_paraview    (odiststream&) const;
  void put_event_paraview     (odiststream&) const;
  void put_finalize_paraview  (odiststream&) const;
// data:
protected:
  std::string           _parameter_name;
  T                     _parameter_value;
  size_type             _n_value;
  mutable size_type     _count_value;
  mutable std::ostream* _p_data_out;
  mutable std::ostream* _p_ctrl_out;
  mutable bool          _header_in_done;
  mutable bool          _header_out_done;
  mutable bool          _finalize_out_done;
  std::pair<T,T>        _u_range;
};
template <class T, class M>
idiststream& operator>> (idiststream&, branch_basic<T,M>&);

template <class T, class M>
odiststream& operator<< (odiststream&, const branch_basic<T,M>&);

typedef branch_basic<Float> branch;

// =============================================================================
// inlined
// =============================================================================
template <class T, class M>
inline
branch_basic<T,M>::branch_basic ()
 : std::vector<std::pair<std::string,field_basic<T,M> > >(0),
   _parameter_name ("*unamed*"),
   _parameter_value(std::numeric_limits<T>::max()),
   _n_value        (std::numeric_limits<size_type>::max()),
   _count_value    (std::numeric_limits<size_type>::max()),
   _p_data_out	   (0),
   _p_ctrl_out	   (0),
   _header_in_done     (false),
   _header_out_done    (false),
   _finalize_out_done  (false),
   _u_range(std::pair<T,T>(std::numeric_limits<T>::min(),
                           std::numeric_limits<T>::max()))
{
}
template <class T, class M>
inline
branch_basic<T,M>::branch_basic (const std::string& parameter_name, const std::string& field_name)
 : std::vector<std::pair<std::string,field_basic<T,M> > >(1),
   _parameter_name (parameter_name),
   _parameter_value(std::numeric_limits<T>::max()),
   _n_value        (std::numeric_limits<size_type>::max()),
   _count_value    (std::numeric_limits<size_type>::max()),
   _p_data_out	   (0),
   _p_ctrl_out	   (0),
   _header_in_done     (false),
   _header_out_done    (false),
   _finalize_out_done  (false),
   _u_range(std::pair<T,T>(std::numeric_limits<T>::min(),
                           std::numeric_limits<T>::max()))
{
    check_macro(parameter_name.length() != 0, "empty parameter name not allowed");
    check_macro(field_name.length() != 0, "empty field name not allowed");
    base::operator[](0).first = field_name;
}
template <class T, class M>
inline
branch_basic<T,M>::branch_basic (const std::string& parameter_name, const std::string& u0, const std::string& u1)
 : std::vector<std::pair<std::string,field_basic<T,M> > >(2),
   _parameter_name (parameter_name),
   _parameter_value(std::numeric_limits<T>::max()),
   _n_value        (std::numeric_limits<size_type>::max()),
   _count_value    (std::numeric_limits<size_type>::max()),
   _p_data_out	   (0),
   _p_ctrl_out	   (0),
   _header_in_done     (false),
   _header_out_done    (false),
   _finalize_out_done  (false),
   _u_range(std::pair<T,T>(std::numeric_limits<T>::min(),
                           std::numeric_limits<T>::max()))
{
    check_macro(parameter_name.length() != 0, "empty parameter name not allowed");
    check_macro(u0.length() != 0, "empty #1 field name not allowed");
    check_macro(u1.length() != 0, "empty #2 field name not allowed");
    base::operator[](0).first = u0;
    base::operator[](1).first = u1;
}
// -----------------------------------------------------------------------------
// accessors
// -----------------------------------------------------------------------------
template <class T, class M>
inline
const T&
branch_basic<T,M>::parameter () const
{
    return _parameter_value;
}
template <class T, class M>
inline
const std::string&
branch_basic<T,M>::parameter_name () const
{
    return _parameter_name;
}
template <class T, class M>
inline
typename branch_basic<T,M>::size_type
branch_basic<T,M>::n_value () const
{
    return _n_value;
}
template <class T, class M>
inline
typename branch_basic<T,M>::size_type
branch_basic<T,M>::n_field () const
{
    return base::size();
}
// -----------------------------------------------------------------------------
// modifiers
// -----------------------------------------------------------------------------
template <class T, class M>
inline
void 
branch_basic<T,M>::set_parameter (const T& x)
{
    _parameter_value = x;
}
template <class T, class M>
inline
void 
branch_basic<T,M>::set_range (const std::pair<T,T>& u_range)
{
    _u_range = u_range;
}
// -----------------------------------------------------------------------------
// io header wrapper
// -----------------------------------------------------------------------------
template <class T, class M>
class __branch_header {
    public:
	__branch_header (branch_basic<T,M>& b) : _b(b) {}
	friend idiststream& operator>> (idiststream& in, __branch_header<T,M> h) { 
		h._b.get_header(in); return in; }
	friend odiststream& operator<< (odiststream& out, __branch_header<T,M> h) { 
		h._b.put_header(out); return out; }
    protected:
    	branch_basic<T,M>& _b;
	template <class T1, class M1> friend class __const_branch_header;
};
template <class T, class M>
class __const_branch_header {
    public:
	__const_branch_header (const branch_basic<T,M>& b) : _b(b) {}
	__const_branch_header (__branch_header<T,M> h) : _b(h._b) {}
	friend odiststream& operator<< (odiststream& out, __const_branch_header<T,M> h) { 
		h._b.put_header(out); return out; }
    protected:
    	const branch_basic<T,M>& _b;
};
template <class T, class M>
inline
__branch_header<T,M>
branch_basic<T,M>::header ()
{
    return *this;
}
template <class T, class M>
inline
__const_branch_header<T,M>
branch_basic<T,M>::header () const
{
    return *this;
}
// -----------------------------------------------------------------------------
// o finalize wrapper
// -----------------------------------------------------------------------------
template <class T, class M>
class __const_branch_finalize {
    public:
	__const_branch_finalize (const branch_basic<T,M>& b) : _b(b) {}
	friend odiststream& operator<< (odiststream& out, __const_branch_finalize<T,M> h) { 
		h._b.put_finalize(out); return out; }
    protected:
    	const branch_basic<T,M>& _b;
};
template <class T, class M>
inline
__const_branch_finalize<T,M>
branch_basic<T,M>::finalize () const
{
    return *this;
}
// -----------------------------------------------------------------------------
// io value wrapper
// -----------------------------------------------------------------------------
template <class T, class M>
class __obranch {
    public:
        __obranch (odiststream& (*put)(odiststream&, const branch_basic<T,M>&), const branch_basic<T,M>& x)
         : _put(put), _x(x) {}
        friend odiststream& operator<< (odiststream& os, __obranch<T,M> m)
         { m._put (os, m._x); return os; }
    private:
	odiststream&    (*_put) (odiststream&, const branch_basic<T,M>&);
        const branch_basic<T,M>& _x;
};
template <class T, class M>
class __iobranch {
    public:
        __iobranch (odiststream& (*put)(odiststream&, const branch_basic<T,M>&), 
	            idiststream& (*get)(idiststream&, branch_basic<T,M>&),
		    branch_basic<T,M>& x,
		    T*                 pt = 0, 
		    field_basic<T,M>*  pu = 0, 
		    field_basic<T,M>*  pp = 0)
         : _put(put), _get(get), _px(&x), _pt(pt), _pu(pu), _pp(pp) {}
        template <class T1, class M1>
	friend odiststream& operator<< (odiststream& os, __iobranch<T1,M1> m);
        template <class T1, class M1>
	friend idiststream& operator>> (idiststream& is, __iobranch<T1,M1> m);
    private:
	odiststream&    (*_put) (odiststream&, const branch_basic<T,M>&);
	idiststream&    (*_get) (idiststream&, branch_basic<T,M>&);
        branch_basic<T,M> *_px;
        T                 *_pt;
        field_basic<T,M>  *_pu;
        field_basic<T,M>  *_pp;
};
template <class T, class M>
inline
odiststream& operator<< (odiststream& os, __iobranch<T,M> m) {
    m._put (os, *m._px);
    return os;
}
template <class T, class M>
inline
idiststream& operator>> (idiststream& is, __iobranch<T,M> m) { 
    m._get (is, *m._px); 
    if (m._pt) { *m._pt = (*m._px).parameter(); }
    if (m._pu) { *m._pu = (*m._px)[0].second; }
    if (m._pp) { *m._pp = (*m._px)[1].second; }
    return is;
}
template <class T, class M>
inline
__obranch<T,M>
branch_basic<T,M>::operator() (const T& t, const field_basic<T,M>& u)
{
    check_macro (base::size() >= 1, "attempt to output a 1-field branch when a " 
	<< base::size() << "-field one was supplied");
    _parameter_value = t;
    base::operator[](0).second = u;
    return __obranch<T,M> (operator<<, *this);
}
template <class T, class M>
inline
__iobranch<T,M>
branch_basic<T,M>::operator() (T& t, field_basic<T,M>& u)
{
    check_macro (base::size() >= 1, "attempt to input/output a 1-field branch when a " 
	<< base::size() << "-field one was supplied");
    _parameter_value = t;
    base::operator[](0).second = u;
    return __iobranch<T,M> (operator<<, operator>>, *this, &t, &u);
}
template <class T, class M>
inline
__obranch<T,M>
branch_basic<T,M>::operator() (const T& t, const field_basic<T,M>& u, const field_basic<T,M>& p)
{
    check_macro (base::size() >= 2, "attempt to output a 2-field branch when a " 
	<< base::size() << "-field one was supplied");
    _parameter_value = t;
    base::operator[](0).second = u;
    base::operator[](1).second = p;
    return __obranch<T,M> (operator<<, *this);
}
template <class T, class M>
inline
__iobranch<T,M>
branch_basic<T,M>::operator() (T& t, field_basic<T,M>& u, field_basic<T,M>& p)
{
    check_macro (base::size() >= 2, "attempt to output a 2-field branch when a " 
	<< base::size() << "-field one was supplied");
    _parameter_value = t;
    base::operator[](0).second = u;
    base::operator[](1).second = p;
    return __iobranch<T,M> (operator<<, operator>>, *this, &t, &u, &p);
}

}// namespace rheolef
#endif // define_RHEO_BRANCH_H
