/****************************************************************************
**
** This file is part of a Qt Solutions component.
** 
** Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
** 
** Contact:  Qt Software Information (qt-info@nokia.com)
** 
** Commercial Usage  
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Solutions Commercial License Agreement provided
** with the Software or, alternatively, in accordance with the terms
** contained in a written agreement between you and Nokia.
** 
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
** 
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
** 
** GNU General Public License Usage 
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
** 
** Please note Third Party Software included with Qt Solutions may impose
** additional restrictions and it is the user's responsibility to ensure
** that they have met the licensing requirements of the GPL, LGPL, or Qt
** Solutions Commercial license and the relevant license of the Third
** Party Software they are using.
** 
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** 
****************************************************************************/

// Implementation of the QWinHost classes

#ifdef QT3_SUPPORT
#undef QT3_SUPPORT
#endif

#include "qwinhost.h"

#include <QtCore/QEvent>
#include <QtCore/QTimer>

//static
QMap<HWND, QWinHost*> QWinHost::winhosts;

/*!
    \class QWinHost qwinhost.h
    \brief The QWinHost class provides an API to use native Win32
    windows in Qt applications.

    QWinHost exists to provide a QWidget that can act as a parent for
    any native Win32 control. Since QWinHost is a proper QWidget, it
    can be used as a toplevel widget (e.g. 0 parent) or as a child of
    any other QWidget.

    QWinHost integrates the native control into the Qt user interface,
    e.g. handles focus switches and laying out.

    Applications moving to Qt may have custom Win32 controls that will
    take time to rewrite with Qt. Such applications can use these
    custom controls as children of QWinHost widgets. This allows the
    application's user interface to be replaced gradually.

    When the QWinHost is destroyed, and the Win32 window hasn't been
    set with setWindow(), the window will also be destroyed.
*/

/*!
    Creates an instance of QWinHost. \a parent and \a f are
    passed on to the QWidget constructor. The widget has by default
    no background.

    \warning You cannot change the parent widget of the QWinHost instance 
    after the native window has been created, i.e. do not call 
    QWidget::setParent or move the QWinHost into a different layout.
*/
QWinHost::QWinHost(QWidget *parent, Qt::WFlags f)
: QWidget(parent, f), wndproc(0),auto_destruct(false), hwnd(0)
{
    setAttribute(Qt::WA_NoBackground);
    setAttribute(Qt::WA_NoSystemBackground);

    connect(this, SIGNAL(clientFocused()), this, SLOT(setFocusSlot()), Qt::QueuedConnection);
}

/*!
    Destroys the QWinHost object. If the hosted Win32 window has not
    been set explicitly using setWindow() the window will be
    destroyed.
*/
QWinHost::~QWinHost()
{
    if (wndproc) {
#if defined(GWLP_WNDPROC)
	QT_WA({
	    SetWindowLongPtr(hwnd, GWLP_WNDPROC, (LONG_PTR)wndproc);
	},{
	    SetWindowLongPtrA(hwnd, GWLP_WNDPROC, (LONG_PTR)wndproc);
	})
#else
	QT_WA({
	    SetWindowLong(hwnd, GWL_WNDPROC, (LONG)wndproc);
	},{
	    SetWindowLongA(hwnd, GWL_WNDPROC, (LONG)wndproc);
	})
#endif
    }

    if (hwnd) {
	winhosts.remove(hwnd);
    }

    if (hwnd && isAutoDestruct()) {
	setAutoDestruct(false);
	SendMessage(hwnd, WM_CLOSE, 0, 0);
	DestroyWindow(hwnd);
    }
}

// static
QWinHost *QWinHost::findHost (HWND client) {
	return (winhosts.value(client));
}

/*!
    Reimplement this virtual function to create and return the native
    Win32 window. \a parent is the handle to this widget, and \a
    instance is the handle to the application instance. The returned HWND
    must be a child of the \a parent HWND.

    The default implementation returns null. The window returned by a
    reimplementation of this function is owned by this QWinHost
    instance and will be destroyed in the destructor.

    This function is called by the implementation of polish() if no
    window has been set explicitly using setWindow(). Call polish() to
    force this function to be called.

    \sa setWindow()
*/
HWND QWinHost::createWindow(HWND parent, HINSTANCE instance)
{
    Q_UNUSED(parent);
    Q_UNUSED(instance);
    return 0;
}


/*!
    Sets the native Win32 window to \a window. If \a window is not a child 
    window of this widget, then it is reparented to become one. If \a window 
    is not a child window (i.e. WS_OVERLAPPED is set), then this function does nothing.

    \sa window(), createWindow(), setAutoDestruct()
*/
void QWinHost::setWindow(HWND window)
{
    if (hwnd && isAutoDestruct())
	DestroyWindow(hwnd);

    hwnd = window;
    fixParent();
}

/*!
    Returns the handle to the native Win32 window, or null if no
    window has been set or created yet.

    \sa setWindow(), createWindow()
*/
HWND QWinHost::window() const
{
    return hwnd;
}

/*!
    \see setAutoDestruct()
*/
bool QWinHost::isAutoDestruct() const
{
    return auto_destruct;
}

/*!
    \a destruct if this is false (the default), the lifetime of the window handle
    will be managed by Windows, QWinHost does not
    call DestroyWindow. To verify that the handle is destroyed when expected, handle
    WM_DESTROY in the window procedure. If \a destruct is true, the client window
    will be destroyed, when this window is destroyed.

    For windows created by createWindow(), this is set to true, else the default is false.

    \sa isAutoDestruct()
 */
void QWinHost::setAutoDestruct(bool destruct)
{
    auto_destruct = destruct;
}

void *getWindowProc(QWinHost *host)
{
    return host ? host->wndproc : 0;
}

/*! just a thin wrapper around QWidget::setFocus(). Needed to handle focus asynchronously. */
void QWinHost::setFocusSlot() {
	if(!hasFocus())	{
		setFocus(Qt::MouseFocusReason);
		activateWindow();
	}
}

bool QWinHost::handleWindowCallback (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg) {
		case WM_SETFOCUS:
		case WM_LBUTTONDOWN:
			if (!hasFocus () && (focusPolicy() & Qt::ClickFocus)) {
				emit (clientFocused());		// handled asynchronously
			}
			break;

		case WM_SYSKEYDOWN:
		case WM_SYSKEYUP:
			QT_WA({
				SendMessage(winId(), msg, wParam, lParam);
			}, {
				SendMessageA(winId(), msg, wParam, lParam);
			})
			break;

		case WM_KEYDOWN:
			if (wParam == VK_TAB) {
			QT_WA({
				SendMessage(winId(), msg, wParam, lParam);
			}, {
				SendMessageA(winId(), msg, wParam, lParam);
			})
			}
			break;
		case WM_SETTEXT:
			clientTitleChanged((char *) lParam);
			break;
		case WM_DESTROY:
			clientDestroyed();
			break;
		default:
			break;
	}
	return true;	// regular handling should happen, too
}


LRESULT CALLBACK WinHostProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    QWinHost *widget = QWinHost::findHost(hwnd);
    WNDPROC oldproc = (WNDPROC)getWindowProc(widget);
    if (widget) {
	if (!widget->handleWindowCallback (hwnd, msg, wParam, lParam)) {
		return 0;
	}
    }

    QT_WA({
	if (oldproc)
	    return CallWindowProc(oldproc, hwnd, msg, wParam, lParam);
	return DefWindowProc(hwnd,msg,wParam,lParam);
    }, {
	if (oldproc)
	    return CallWindowProcA(oldproc, hwnd, msg, wParam, lParam);
	return DefWindowProcA(hwnd,msg,wParam,lParam);
    })
}

QString QWinHost::getClientTitle() const {
	if (!hwnd) return QString();

	char buffer[256];
	::GetWindowText(hwnd, buffer, 255);
	return (QString(buffer));
}

/*!
    Ensures that the window provided a child of this widget, unless
    it is a WS_OVERLAPPED window.
*/
void QWinHost::fixParent()
{
    if (!hwnd)
        return;
    if (!::IsWindow(hwnd)) {
        hwnd = 0;
        return;
    }
    if (::GetParent(hwnd) == winId())
        return;
    long style = GetWindowLong(hwnd, GWL_STYLE);
    if (style & WS_OVERLAPPED)
        return;
    ::SetParent(hwnd, winId());

    winhosts.insert(hwnd, this);

#if defined(GWLP_WNDPROC)
            QT_WA({
                wndproc = (void*)GetWindowLongPtr(hwnd, GWLP_WNDPROC);
                SetWindowLongPtr(hwnd, GWLP_WNDPROC, (LONG_PTR)WinHostProc);
            }, {
                wndproc = (void*)GetWindowLongPtrA(hwnd, GWLP_WNDPROC);
                SetWindowLongPtrA(hwnd, GWLP_WNDPROC, (LONG_PTR)WinHostProc);
            })
#else
                QT_WA({
                wndproc = (void*)GetWindowLong(hwnd, GWL_WNDPROC);
                SetWindowLong(hwnd, GWL_WNDPROC, (LONG)WinHostProc);
            }, {
                wndproc = (void*)GetWindowLongA(hwnd, GWL_WNDPROC);
                SetWindowLongA(hwnd, GWL_WNDPROC, (LONG)WinHostProc);
            })
#endif

            if (style & WS_TABSTOP)
                setFocusPolicy(Qt::FocusPolicy(focusPolicy() | Qt::StrongFocus));
}

/*!
    \reimp
*/
bool QWinHost::event(QEvent *e)
{
    switch(e->type()) {
    case QEvent::Polish:
        if (!hwnd) {
            hwnd = createWindow(winId(), qWinAppInst());
            fixParent();
            setAutoDestruct(hwnd != 0);
        }
        break;
    case QEvent::WindowBlocked:
        if (hwnd)
            EnableWindow(hwnd, false);
        break;
    case QEvent::WindowUnblocked:
        if (hwnd)
            EnableWindow(hwnd, true);
        break;
    }
    return QWidget::event(e);
}

/*!
    \reimp
*/
void QWinHost::showEvent(QShowEvent *e)
{
    QWidget::showEvent(e);

    if (hwnd)
	SetWindowPos(hwnd, HWND_TOP, 0, 0, width(), height(), SWP_SHOWWINDOW);
}

/*!
    \reimp
*/
void QWinHost::focusInEvent(QFocusEvent *e)
{
    QWidget::focusInEvent(e);

    if (hwnd)
	::SetFocus(hwnd);
}

/*!
    \reimp
*/
void QWinHost::resizeEvent(QResizeEvent *e)
{
    QWidget::resizeEvent(e);

    if (hwnd)
	SetWindowPos(hwnd, HWND_TOP, 0, 0, width(), height(), 0);
}

/*!
    \reimp
*/
bool QWinHost::winEvent(MSG *msg, long *result)
{
    switch (msg->message)
    {
    case WM_SETFOCUS:
        if (hwnd) {
            ::SetFocus(hwnd);
            return true;
        }
    default:
        break;
    }

    return QWidget::winEvent(msg, result);
}

