//RLPlot.cpp, Copyright 2000, 2001, 2002, 2003, 2004 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

extern tag_Units Units[];
extern char TmpTxt[];
extern Default defs;

GraphObj *CurrGO, *TrackGO;			//Selected Graphic Objects
Label *CurrLabel = 0L;
Graph *CurrGraph = 0L;
dragHandle *CurrHandle = 0L;
Axis **CurrAxes = 0L;
UndoObj Undo;
int cGraphs = 0;
int cPlots = 0;
int cPages = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// grapic objects
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
GraphObj::GraphObj(GraphObj *par, DataObj *d)
{
	parent = par;
	data = d;
	Id = GO_UNKNOWN;
	type = moveable = 0;
	name = 0L;
	rDims.left = rDims.right = rDims.top = rDims.bottom = 0;
}

GraphObj::~GraphObj()
{
	if(name)free(name);
	name = 0L;
	if(CurrGO == this) CurrGO = 0L;
	if(TrackGO == this) TrackGO = 0L;
}

double
GraphObj::GetSize(int select)
{
	if(parent) return parent->GetSize(select);
	else return defs.GetSize(select);
}

DWORD 
GraphObj::GetColor(int select){
	return defs.Color(select);
}

void *
GraphObj::ObjThere(int x, int y)
{
	if(IsInRect(&rDims, x, y)) return this;
	else return 0L;
}

void
GraphObj::Track(POINT *p, anyOutput *o)
{
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Spread sheet buttons used for rows and columns
ssButton::ssButton(GraphObj *par, int x, int y, int w, int h):GraphObj(par, 0L)
{
	bLBdown = false;
	SetMinMaxRect(&rDims, x, y, x+w, y+h);
	Line.width = 0.0f;				Line.patlength = 1.0f;
	Line.color = 0x00000000L;		Line.pattern = 0x00000000L;
	Fill.type = FILL_NONE;			Fill.color = 0x00d8d8d8L;
	Fill.scale = 1.0;				Fill.hatch = NULL;
	TextDef.ColTxt = 0x00000000L;	TextDef.ColBg = 0x00ffffffL;
	TextDef.fSize = 4.0;			TextDef.RotBL = TextDef.RotCHAR = 0.0;
	TextDef.iSize = 0;				TextDef.Align = TXA_HLEFT | TXA_VTOP;
	TextDef.Mode = TXM_TRANSPARENT;	TextDef.Style = TXS_NORMAL;
	TextDef.Font = FONT_HELVETICA;	TextDef.text = 0L;
}

ssButton::~ssButton()
{
	if(TextDef.text) free(TextDef.text);
}

void
ssButton::DoPlot(anyOutput *o)
{
	POINT pts[3];

	Line.color = 0x00000000L;
	o->SetLine(&Line);
	o->SetFill(&Fill);
	if(bLBdown){
		o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
		}
	else {
		o->oRectangle(rDims.left, rDims.top, rDims.right-1, rDims.bottom-1);
		Line.color = 0x00000000L;
		o->SetLine(&Line);
		pts[0].x = rDims.left;					pts[0].y = pts[1].y = rDims.bottom-1;
		pts[1].x = pts[2].x = rDims.right-1;	pts[2].y = rDims.top-1;
		o->oPolyline(pts, 3);
		Line.color = 0x00ffffffL;
		o->SetLine(&Line);
		pts[0].x = pts[1].x = rDims.left;		pts[0].y = rDims.bottom -3;
		pts[1].y = pts[2].y = rDims.top;		pts[2].x = rDims.right -2;
		o->oPolyline(pts, 3);
		}
	if(TextDef.text) {
		o->SetTextSpec(&TextDef);
#ifdef _WINDOWS
		o->oTextOut((rDims.left + rDims.right)/2-2, (rDims.top + rDims.bottom)/2-1, TextDef.text, 0);
#else
		o->oTextOut((rDims.left + rDims.right)/2-2, (rDims.top + rDims.bottom)/2+1, TextDef.text, 0);
#endif
		}

}
void
ssButton::DoMark(anyOutput *o, bool mark)
{
	bLBdown = mark;
	DoPlot(o);
	o->UpdateRect(&rDims, false);
}

bool
ssButton::Command(int cmd, void *tmpl, anyOutput *o)
{
	char *tmptxt;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_GETTEXT:
		if(TextDef.text && tmpl) {
			strcpy((char*)tmpl, TextDef.text);
			return true;
			}
		return false;
	case CMD_SETTEXT:
		if(TextDef.text) free(TextDef.text);
		if(tmpl) TextDef.text = strdup((char*)tmpl);
		else TextDef.text = 0L;
		return true;
	case CMD_GETTEXTDEF:
		if(!tmpl) return false;
		memcpy(tmpl, &TextDef, sizeof(TextDEF));
		return true;
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		tmptxt = TextDef.text;
		memcpy(&TextDef, tmpl, sizeof(TextDEF));
		TextDef.text = tmptxt;
		return true;
	case CMD_MOUSE_EVENT:
		if(!o || !(mev = (MouseEvent *) tmpl)) break;
		if(IsInRect(&rDims, mev->x, mev->y)) {
			if(bLBdown) {
				if(!(mev->StateFlags & 0x01)) {
					o->HideMark();
					if(bLBdown) {
						bLBdown = false; DoPlot(o);
						}
					}
				}
			else if(mev->StateFlags & 0x01) {
				o->ShowMark(this, MRK_SSB_DRAW);
				}
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// drag handles are graphic objects which allow the user to interactively
//   change the size of an object
dragHandle::dragHandle(GraphObj *par, int which):GraphObj(par, 0L)
{
	type = which;
	Id = GO_DRAGHANDLE;
	LineDef.width = LineDef.patlength = 0.0;
	LineDef.color = LineDef.pattern = 0L;
	FillDef.type = FILL_NONE;
	FillDef.color = 0x00ffffffL;
	FillDef.scale = 1.0;
	FillDef.hatch = 0L;
	minRC = maxRC = 0L;
}

dragHandle::~dragHandle()
{
	if(minRC) free(minRC);
	if(maxRC) free(maxRC);
	if(CurrHandle == this) CurrHandle = 0L;
}

void
dragHandle::DoPlot(anyOutput *o)
{
	double fx, fy, dx, dy;
	int ix, iy;
	fPOINT3D fp3d, ifp3d;
	bool is3D = false;

	if(!o || !parent) return;
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	SetMinMaxRect(&drec, o->co2ix(parent->GetSize(SIZE_XPOS)+dx), 
		o->co2iy(parent->GetSize(SIZE_YPOS)+dy), o->co2ix(parent->GetSize(SIZE_XPOS+1)+dx),
		o->co2iy(parent->GetSize(SIZE_YPOS+1)+dy));
	switch(type) {
	case DH_19:		case DH_12:
		fx = parent->GetSize(SIZE_XPOS);	fy = parent->GetSize(SIZE_YPOS);
		break;
	case DH_99:		case DH_22:
		fx = parent->GetSize(SIZE_XPOS+1);	fy = parent->GetSize(SIZE_YPOS+1);
		break;
	case DH_29:
		fx = (parent->GetSize(SIZE_XPOS) + parent->GetSize(SIZE_XPOS+1))/2.0;
		fy = parent->GetSize(SIZE_YPOS);
		break;
	case DH_39:
		fx = parent->GetSize(SIZE_XPOS+1);	fy = parent->GetSize(SIZE_YPOS);
		break;
	case DH_49:
		fx = parent->GetSize(SIZE_XPOS);
		fy = (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
		break;
	case DH_59:
		fx = (parent->GetSize(SIZE_XPOS) + parent->GetSize(SIZE_XPOS+1))/2.0;
		fy = (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
		break;
	case DH_69:
		fx = parent->GetSize(SIZE_XPOS+1);
		fy = (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
		break;
	case DH_79:
		fx = parent->GetSize(SIZE_XPOS);
		fy = parent->GetSize(SIZE_YPOS+1);
		break;
	case DH_89:
		fx = (parent->GetSize(SIZE_XPOS) + parent->GetSize(SIZE_XPOS+1))/2.0;
		fy = parent->GetSize(SIZE_YPOS+1);
		break;
	case DH_18:	case DH_28:	case DH_38:	case DH_48:
	case DH_58:	case DH_68:	case DH_78:	case DH_88:
		fp3d.fx = parent->GetSize(SIZE_XPOS + type - DH_18);
		fp3d.fy = parent->GetSize(SIZE_YPOS + type - DH_18);
		fp3d.fz = parent->GetSize(SIZE_ZPOS + type - DH_18);
		is3D = true;
		break;
	default:
		if(type >= DH_DATA) {
			fx = parent->GetSize(SIZE_XPOS + type - DH_DATA);
			fy = parent->GetSize(SIZE_YPOS + type - DH_DATA);
			FillDef.color = (this == CurrHandle) ? 0x0L : 0x00ffffffL;
			}
		else return;
		}
	if(is3D) {
		o->cvec2ivec(&fp3d, &ifp3d);
		ix = iround(ifp3d.fx);		iy = iround(ifp3d.fy);
		}
	else {
		ix = o->co2ix(fx+dx);	iy = o->co2iy(fy+dy);
		}
	SetMinMaxRect(&rDims, ix-4, iy-4, ix+4, iy+4);
	memcpy(&upd, &rDims, sizeof(RECT));
	switch(type) {
	case DH_12:	case DH_22:	case DH_19:	case DH_29:	case DH_39:
	case DH_49:	case DH_69:	case DH_79:	case DH_89:	case DH_99:
		o->SetLine(&LineDef);		o->SetFill(&FillDef);
		o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
		o->UpdateRect(&rDims, false);
		break;
	case DH_59:
		o->SetLine(&LineDef);		o->SetFill(&FillDef);
		IncrementMinMaxRect(&rDims, 2);
		o->oCircle(rDims.left, rDims.top, rDims.right, rDims.bottom);
		IncrementMinMaxRect(&rDims, 1);
		o->UpdateRect(&rDims, false);
		break;
	default:
		if(type >= DH_DATA) {
			o->SetLine(&LineDef);		o->SetFill(&FillDef);
			o->oRectangle(rDims.left, rDims.top, rDims.right, rDims.bottom);
			o->UpdateRect(&rDims, false);
			}
		else {
			IncrementMinMaxRect(&rDims, -1);
			o->UpdateRect(&rDims, true);
			}
		break;
		}
}

bool
dragHandle::Command(int cmd, void *tmpl, anyOutput *o)
{
	lfPOINT pos;
	int idx;

	if(!parent) return false;
	switch (cmd) {
	case CMD_MOUSECURSOR:
		if(o) switch(type) {
		case DH_19:	case DH_99:
			o->MouseCursor(MC_SE, false);							break;
		case DH_29:	case DH_89:
			o->MouseCursor(MC_NORTH, false);						break;
		case DH_39:	case DH_79:
			o->MouseCursor(MC_NE, false);							break;
		case DH_49:	case DH_69:
			o->MouseCursor(MC_EAST, false);							break;
		default:
			if(type >= DH_DATA) o->MouseCursor(MC_SALL, false);
			else o->MouseCursor(MC_MOVE, false);					break;
			}
		return true;
	case CMD_MINRC:
		if(!(minRC)) minRC = (RECT*)calloc(1, sizeof(RECT));
		if(tmpl && minRC) SetMinMaxRect(minRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		return true;
	case CMD_MAXRC:
		if(!(maxRC)) maxRC = (RECT*)calloc(1, sizeof(RECT));
		if(tmpl && maxRC) SetMinMaxRect(maxRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		return true;
	case CMD_MOVE:
		idx = type >= DH_DATA ? type - DH_DATA : 0;
		pos.fx = NiceValue(((lfPOINT*)tmpl)[0].fx);
		pos.fy = NiceValue(((lfPOINT*)tmpl)[0].fy);
		if(pos.fx == 0.0 && pos.fy == 0.0) return false;
		parent->Command(CMD_SAVEPOS, &idx, o);
		switch(type) {
		case DH_12:
			parent->SetSize(SIZE_XPOS, pos.fx + parent->GetSize(SIZE_XPOS));
			parent->SetSize(SIZE_YPOS, pos.fy + parent->GetSize(SIZE_YPOS));
			break;
		case DH_22:
			parent->SetSize(SIZE_XPOS+1, pos.fx + parent->GetSize(SIZE_XPOS+1));
			parent->SetSize(SIZE_YPOS+1, pos.fy + parent->GetSize(SIZE_YPOS+1));
			break;
		case DH_19: parent->parent->SetSize(SIZE_XPOS, 
						pos.fx + parent->GetSize(SIZE_XPOS));
		case DH_29: parent->parent->SetSize(SIZE_YPOS,
						pos.fy + parent->GetSize(SIZE_YPOS));
			break;
		case DH_39: parent->parent->SetSize(SIZE_YPOS,
						pos.fy + parent->GetSize(SIZE_YPOS));
		case DH_69: parent->parent->SetSize(SIZE_XPOS+1,
						pos.fx + parent->GetSize(SIZE_XPOS+1));
			break;
		case DH_99: parent->parent->SetSize(SIZE_XPOS+1, 
						pos.fx + parent->GetSize(SIZE_XPOS+1));
		case DH_89: parent->parent->SetSize(SIZE_YPOS+1,
						pos.fy + parent->GetSize(SIZE_YPOS+1));
			break;
		case DH_79: parent->parent->SetSize(SIZE_YPOS+1,
						pos.fy + parent->GetSize(SIZE_YPOS+1));
		case DH_49:
			parent->parent->SetSize(SIZE_XPOS, 
						pos.fx + parent->GetSize(SIZE_XPOS));
			break;
		case DH_18:	case DH_28:	case DH_38:	case DH_48:
		case DH_58:	case DH_68:	case DH_78:	case DH_88:
		case DH_59:
			return parent->parent->Command(cmd, tmpl, o);
		default:
			if (type >= DH_DATA) {
				parent->SetSize(SIZE_XPOS + idx, pos.fx + parent->GetSize(SIZE_XPOS + idx));
				parent->SetSize(SIZE_YPOS + idx, pos.fy + parent->GetSize(SIZE_YPOS + idx));
				CurrGO = parent;
				}
			break;
			}
		return parent->Command(CMD_REDRAW, 0L, o);
		break;
		}
	return false;
}

void *
dragHandle::ObjThere(int x, int y)
{
	if(IsInRect(&rDims, x, y)) return (CurrHandle = this);
	else return 0L;
}


void
dragHandle::Track(POINT *p, anyOutput *o)
{
	POINT p1, p2, pts[5];
	double dx, dy;
	int npts=0, idx;
	DWORD color;

	if(!parent || !o) return;
	Command(CMD_MOUSECURSOR, 0L, o);
	if(upd.right < upd.left) Swap(upd.right, upd.left);
	if(upd.bottom < upd.top) Swap(upd.bottom, upd.top);
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	IncrementMinMaxRect(&upd, 2);
	o->UpdateRect(&upd, false);
	if(type >= DH_19 && type <= DH_99) memcpy(&upd, &drec, sizeof(RECT));
	color = parent->GetColor(COL_DATA_LINE);
	switch(type) {
	case DH_12:
	case DH_22:
		pts[0].x = o->co2ix(parent->GetSize(SIZE_XPOS)+dx);
		pts[0].y = o->co2iy(parent->GetSize(SIZE_YPOS)+dy);
		pts[1].x = o->co2ix(parent->GetSize(SIZE_XPOS+1)+dx);
		pts[1].y = o->co2iy(parent->GetSize(SIZE_YPOS+1)+dy);
		if(type == DH_12) {
			pts[0].x += p->x;			pts[0].y += p->y;
			}
		else if(type == DH_22) {
			pts[1].x += p->x;			pts[1].y += p->y;
			}
		UpdateMinMaxRect(&upd, pts[0].x, pts[0].y);
		UpdateMinMaxRect(&upd, pts[1].x, pts[1].y);
		npts = 2;
		break;
	case DH_19:
		if(minRC && IsInRect(minRC, upd.left+p->x, (upd.top+upd.bottom)>>1))
			upd.left = minRC->left;
		else if(maxRC && !IsInRect(maxRC, upd.left+p->x, (upd.top+upd.bottom)>>1))
			upd.left = upd.left+p->x >= maxRC->right ? maxRC->right : maxRC->left;
		else upd.left += p->x;
	case DH_29:
		if(minRC && IsInRect(minRC, (upd.left + upd.right)>>1, upd.top+p->y))
			upd.top = minRC->top;
		else if(maxRC && !IsInRect(maxRC, (upd.left + upd.right)>>1, upd.top+p->y))
			upd.top = upd.top +p->y >= maxRC->bottom? maxRC->bottom : maxRC->top;
		else upd.top += p->y;
		break;
	case DH_39:
		if(minRC && IsInRect(minRC, (upd.left + upd.right)>>1, upd.top+p->y))
			upd.top = minRC->top;
		else if(maxRC && !IsInRect(maxRC, (upd.left + upd.right)>>1, upd.top+p->y))
			upd.top = upd.top+p->y >= maxRC->bottom ? maxRC->bottom : maxRC->top;
		else upd.top += p->y;
	case DH_69:
		if(minRC && IsInRect(minRC, upd.right+p->x, (upd.top+upd.bottom)>>1))
			upd.right = minRC->right;
		else if(maxRC && !IsInRect(maxRC, upd.right+p->x, (upd.top+upd.bottom)>>1))
			upd.right = upd.right+p->x <= maxRC->left ? maxRC->left : maxRC->right;
		else upd.right += p->x;
		break;
	case DH_99:
		if(minRC && IsInRect(minRC, upd.right+p->x, (upd.top+upd.bottom)>>1))
			upd.right = minRC->right;
		else if(maxRC && !IsInRect(maxRC, upd.right+p->x, (upd.top+upd.bottom)>>1))
			upd.right = upd.right+p->x <= maxRC->left ? maxRC->left : maxRC->right;
		else upd.right += p->x;
	case DH_89:
		if(minRC && IsInRect(minRC, (upd.left + upd.right)>>1, upd.bottom+p->y))
			upd.bottom = minRC->bottom;
		else if(maxRC && !IsInRect(maxRC, (upd.left + upd.right)>>1, upd.bottom+p->y))
			upd.bottom = upd.bottom+p->y <= maxRC->top? maxRC->top : maxRC->bottom;
		else upd.bottom += p->y;
		break;
	case DH_79:
		if(minRC && IsInRect(minRC, (upd.left + upd.right)>>1, upd.bottom+p->y))
			upd.bottom = minRC->bottom;
		else if(maxRC && !IsInRect(maxRC, (upd.left + upd.right)>>1, upd.bottom+p->y))
			upd.bottom = upd.bottom+p->y <= maxRC->top? maxRC->top : maxRC->bottom;
		else upd.bottom += p->y;
	case DH_49:
		if(minRC && IsInRect(minRC, upd.left+p->x, (upd.top+upd.bottom)>>1))
			upd.left = minRC->left;
		else if(maxRC && !IsInRect(maxRC, upd.left+p->x, (upd.top+upd.bottom)>>1))
			upd.left = upd.left+p->x >= maxRC->right ? maxRC->right : maxRC->left;
		else upd.left += p->x;
		break;
	case DH_18:	case DH_28:	case DH_38:	case DH_48:
	case DH_58:	case DH_68:	case DH_78:	case DH_88:
		CurrGO = this;
	case DH_59:
		parent->parent->Track(p, o);
		return;
	default:
		if(type >= DH_DATA) {
			idx = type - DH_DATA;
			pts[1].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx)+dx);
			pts[1].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx)+dy);
			pts[1].x += p->x;				pts[1].y += p->y;
			if(type > DH_DATA) {
				pts[0].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx -1)+dx);
				pts[0].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx -1)+dy);
				}
			else {
				pts[0].x = pts[1].x;		pts[0].y = pts[1].y;
				}
			pts[2].x = o->co2ix(parent->GetSize(SIZE_XPOS + idx +1)+dx);
			pts[2].y = o->co2iy(parent->GetSize(SIZE_YPOS + idx +1)+dy);
			UpdateMinMaxRect(&upd, pts[0].x, pts[0].y);
			UpdateMinMaxRect(&upd, pts[1].x, pts[1].y);
			UpdateMinMaxRect(&upd, pts[2].x, pts[2].y);
			npts = 3;
			if(color == 0x0L || color == 0x00ffffffL) color = 0x00c0c0c0L;
			}
		else return;
		}
	if(type >= DH_19 && type <= DH_99) {
		pts[0].x = pts[4].x = pts[3].x = p1.x = upd.left;	
		pts[0].y = pts[1].y = pts[4].y = p1.y = upd.top;
		pts[1].x = pts[2].x = p2.x = upd.right;	
		pts[2].y = pts[3].y = p2.y = upd.bottom;
		npts = 5;
		if(parent->parent->Id == GO_ELLIPSE) o->ShowEllipse(p1, p2, color);
		}
	o->ShowLine(pts, npts, color);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the dragRect object uses nine dragHandles to create a user interface
//   for modification of rectangular shapes
dragRect::dragRect(GraphObj *par, int which):GraphObj(par, 0L)
{
	int i;

	type = which;
	Id = GO_DRAGRECT;
	if(handles = (dragHandle**)calloc(9, sizeof(dragHandle*)))
		for(i = 0; i < 9; i++){
			if(i == 4 && type == 0) handles[i] = new dragHandle(this, DH_19 + i);
			else if(i != 4) handles[i] = new dragHandle(this, DH_19 + i);
			}
}

dragRect::~dragRect()
{
	int i;

	if(handles) for(i = 0; i < 9; i++) if(handles[i]) DeleteGO(handles[i]);
}

void
dragRect::DoPlot(anyOutput *o)
{
	int i;

	if(handles) for(i = 0; i < 9; i++) if(handles[i]) handles[i]->DoPlot(o);
}

bool
dragRect::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;

	if(!parent) return false;
	switch (cmd) {
	case CMD_MINRC:
	case CMD_MAXRC:
		if(handles) for(i = 0; i < 9; i++) {
			if(handles[i]) handles[i]->Command(cmd, tmpl, o);
			}
		break;
	case CMD_SAVEPOS:
	case CMD_REDRAW:
		return parent->Command(cmd, tmpl, o);
		}
	return false;
}

void *
dragRect::ObjThere(int x, int y)
{
	int i;
	void *go;

	if(handles)	for(i = 0; i < 9; i++) 
		if(handles[i] && (go = (handles[i])->ObjThere(x, y))) return go;
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// implement some kind of virtual trackball for 3D plots
Drag3D::Drag3D(GraphObj *par):GraphObj(par, 0L)
{
	int i;

	Id = GO_DRAG3D;
	if(handles = (dragHandle**)calloc(8, sizeof(dragHandle*)))
		for(i = 0; i < 8; i++){
			handles[i] = new dragHandle(this, DH_18 + i);
			}
}

Drag3D::~Drag3D()
{
	int i;

	if(handles) for(i = 0; i < 8; i++) if(handles[i]) DeleteGO(handles[i]);
}

void
Drag3D::DoPlot(anyOutput *o)
{
	int i;

	if(handles) for(i = 0; i < 8; i++) if(handles[i]) handles[i]->DoPlot(o);
}

void *
Drag3D::ObjThere(int x, int y)
{
	int i;
	void *go;

	if(handles)	for(i = 0; i < 8; i++) 
		if(handles[i] && (go = (handles[i])->ObjThere(x, y))) return go;
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// frame rectangle
FrmRect::FrmRect(GraphObj *par, fRECT *lim, fRECT *c, fRECT *chld):GraphObj(par, 0L)
{
	parent = par;
	limRC = lim;	cRC = c;	chldRC = chld;
	drag = 0L;		mo = 0L;
	Id = GO_FRAMERECT;
	moveable = true;
	Fill.type = FILL_NONE;
	Line.color = FillLine.color = Fill.color = 0x00ffffffL;
	Line.width = FillLine.width = 0.0;
	Line.patlength = FillLine.patlength = Fill.scale = 1.0;
	Line.pattern = FillLine.pattern = 0x0L;
	Fill.hatch = &FillLine;
	minRC = maxRC = 0L;
}

FrmRect::~FrmRect()
{
	if(drag) DeleteGO(drag);		drag = 0L;
	if(minRC) free(minRC);			minRC = 0L;	
	if(maxRC) free(maxRC);			maxRC = 0L;
	if(mo) DelBitmapClass(mo);		mo = 0L;
}

double
FrmRect::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:		return CurrRect.Xmin;
	case SIZE_XPOS+1:	return CurrRect.Xmax;
	case SIZE_YPOS:		return CurrRect.Ymin;
	case SIZE_YPOS+1:	return CurrRect.Ymax;
		}
	return 0.0;
}

bool
FrmRect::SetSize(int select, double value)
{
	double tmp, o_left, o_top;

	o_left = cRC->Xmin;		o_top = cRC->Ymin;
	switch (select & 0xfff) {
	case SIZE_XPOS:
		if(limRC) value -= limRC->Xmin;
		if(swapX) cRC->Xmax = value;
		else cRC->Xmin = value;
		break;
	case SIZE_XPOS+1:
		if(limRC) value -= limRC->Xmin;
		if(swapX) cRC->Xmin = value;
		else cRC->Xmax = value;
		break;
	case SIZE_YPOS:
		if(limRC) value -= limRC->Ymin;
		if(swapY) cRC->Ymin = value;
		else cRC->Ymax = value;
		break;
	case SIZE_YPOS+1:
		if(limRC) value -= limRC->Ymin;
		if(swapY) cRC->Ymax = value;
		else cRC->Ymin = value;
		break;
	default: return false;
		}
	if((swapX && cRC->Xmin < cRC->Xmax) || (!swapX && cRC->Xmin > cRC->Xmax)) {
		tmp = cRC->Xmin;	cRC->Xmin = cRC->Xmax;	cRC->Xmax = tmp;
		}
	if((swapY && cRC->Ymin > cRC->Ymax) || (!swapY && cRC->Ymin < cRC->Ymax)) {
		tmp = cRC->Ymin;	cRC->Ymin = cRC->Ymax;	cRC->Ymax = tmp;
		}
	if(chldRC) {		//check if new rectangle is not inside child rectangle
		if(cRC->Xmin > o_left+ chldRC->Xmin) cRC->Xmin = o_left + chldRC->Xmin;
		if(cRC->Xmax < o_left+ chldRC->Xmax) cRC->Xmax = o_left + chldRC->Xmax;
		if(cRC->Ymin > o_top+ chldRC->Ymin) cRC->Ymin = o_top + chldRC->Ymin;
		if(cRC->Ymax < o_top+ chldRC->Ymax) cRC->Ymax = o_top + chldRC->Ymax;
		}
	if(chldRC && (o_left != cRC->Xmin || o_top != cRC->Ymin)) {
		chldRC->Xmin -= (tmp = cRC->Xmin - o_left);		chldRC->Xmax -= tmp;
		chldRC->Ymin -= (tmp = cRC->Ymin - o_top);		chldRC->Ymax -= tmp;
		}
	return true;
}

bool
FrmRect::SetColor(int select, DWORD col)
{
	switch(select & 0xfff){
	case COL_DRECT:
		Line.color = col;
	case COL_BG:
		Fill.color = col;		return true;
	case COL_GRECT:
		Fill.color = col;		return true;
	case COL_GRECTLINE:
		Line.color = col;		return true;
		}
	return false;
}

void 
FrmRect::DoMark(anyOutput *o, bool mark)
{
	if(!parent || !o) return;
	if(!drag && (drag = new dragRect(this, (!limRC && parent->moveable) ? 0 : 1))){
		if(minRC) drag->Command(CMD_MINRC, minRC, o);
		if(maxRC) drag->Command(CMD_MAXRC, maxRC, o);
		}
	if(mark && drag){
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6);
		mo = GetRectBitmap(&mrc, o);
		drag->DoPlot(o);
		o->UpdateRect(&mrc, false);
		}
	else RestoreRectBitmap(&mo, &mrc, o);
}

void
FrmRect::DoPlot(anyOutput *o)
{
	int x1, y1, x2, y2;
	double tmp;

	if(!(cRC) || !o) return;
	o->dFillCol = Fill.color ^ 0x00ffffff;	//force new brush
	o->dLineCol = Line.color ^ 0x00ffffff;	//force new pen
	o->SetLine(&Line);						o->SetFill(&Fill);
	CurrRect.Xmin = cRC->Xmin;				CurrRect.Xmax = cRC->Xmax;
	CurrRect.Ymin = cRC->Ymax;				CurrRect.Ymax = cRC->Ymin;
	if(limRC) {
		CurrRect.Xmin += limRC->Xmin;		CurrRect.Xmax += limRC->Xmin;
		CurrRect.Ymin += limRC->Ymin;		CurrRect.Ymax += limRC->Ymin;
		}
	if(swapX = (CurrRect.Xmin > CurrRect.Xmax)) {
		tmp = CurrRect.Xmin;	CurrRect.Xmin = CurrRect.Xmax;	CurrRect.Xmax = tmp;
		}
	if(swapY = (CurrRect.Ymin > CurrRect.Ymax)) {
		tmp = CurrRect.Ymin;	CurrRect.Ymin = CurrRect.Ymax;	CurrRect.Ymax = tmp;
		}
	o->oRectangle(x1 = o->co2ix(CurrRect.Xmin), y1 = o->co2iy(CurrRect.Ymin), 
		x2 = o->co2ix(CurrRect.Xmax), y2 = o->co2iy(CurrRect.Ymax));
	SetMinMaxRect(&rDims, x1, y1, x2, y2);
}

bool
FrmRect::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	if(!parent) return false;
	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !(CurrGO) && (o)){
				o->ShowMark(this, MRK_GODRAW);
				if(parent && parent->Id == GO_GRAPH) CurrGraph = (Graph*)parent;
				return true;
				}
			}
		return false;
	case CMD_MINRC:
		if(!(minRC)) minRC = (RECT*)calloc(1, sizeof(RECT));
		if(minRC && tmpl) SetMinMaxRect(minRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		if(drag) drag->Command(cmd, tmpl, o);
		return true;
	case CMD_MAXRC:
		if(!(maxRC)) maxRC = (RECT*)calloc(1, sizeof(RECT));
		if(maxRC && tmpl) SetMinMaxRect(maxRC, ((RECT*)tmpl)->left, ((RECT*)tmpl)->top, 
			((RECT*)tmpl)->right, ((RECT*)tmpl)->bottom);
		if(drag) drag->Command(cmd, tmpl, o);
		return true;
	case CMD_MOVE:
		cRC->Xmin += NiceValue(((lfPOINT*)tmpl)[0].fx);
		cRC->Ymin += NiceValue(((lfPOINT*)tmpl)[0].fy);
		cRC->Xmax += NiceValue(((lfPOINT*)tmpl)[0].fx);
		cRC->Ymax += NiceValue(((lfPOINT*)tmpl)[0].fy);
	case CMD_REDRAW:
		return parent->Command(CMD_REDRAW, 0L, o);
	case CMD_SAVEPOS:
		return parent->Command(cmd, tmpl, o);
	case CMD_SETCHILD:
		chldRC = (fRECT*)tmpl;
		return true;
		}
	return false;
}

void *
FrmRect::ObjThere(int x, int y)
{
	if(drag) return drag->ObjThere(x, y);
	return 0L;
}

void
FrmRect::Track(POINT *p, anyOutput *o)
{
	POINT tpts[5];
	RECT old_rc;

	if(o){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		o->UpdateRect(&rDims, false);
		tpts[0].x = tpts[1].x = tpts[4].x = o->co2ix(cRC->Xmin)+p->x;		
		tpts[0].y = tpts[3].y = tpts[4].y = o->co2iy(cRC->Ymin)+p->y;
		tpts[1].y = tpts[2].y = o->co2iy(cRC->Ymax)+p->y;
		tpts[2].x = tpts[3].x = o->co2ix(cRC->Xmax)+p->x;
		UpdateMinMaxRect(&rDims, tpts[0].x, tpts[0].y);
		UpdateMinMaxRect(&rDims, tpts[2].x, tpts[2].y);	
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		o->ShowLine(tpts, 5, Fill.color ^ 0x00ffffffL);
		}

}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// This is a special object to read certain svg-settings from a *.rlp file
svgOptions::svgOptions(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(defs.svgScript) free(defs.svgScript);
	if(defs.svgAttr) free(defs.svgAttr);
	defs.svgScript = defs.svgAttr = 0L;
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		if(script) defs.svgScript = script;
		if(svgattr) defs.svgAttr = svgattr;
		script = svgattr = 0L;
		}
	Id=GO_SVGOPTIONS;
}

svgOptions::~svgOptions()
{
	if(script)free(script);
	script = 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Symbols are graphic objects
Symbol::Symbol(GraphObj *par, DataObj *d, double x, double y, int which,
		int xc, int xr, int yc, int yr):GraphObj(par, d)
{
	//Symbols with no parent are part of a dialog
	FileIO(INIT_VARS);
	fPos.fx = x;
	fPos.fy = y;
	type = which;
	Id = GO_SYMBOL;
	if(xc >= 0 && xr >= 0 && yc >= 0 && yr >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*2)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			cssRef = 2;
			}
		}
}

Symbol::Symbol(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		SymFill.hatch = (LineDEF *) NULL;
		}
}

Symbol::~Symbol()
{
	Command(CMD_FLUSH, 0L, 0L);
}

double
Symbol::GetSize(int select)
{
	switch(select) {
	case SIZE_MINE:
	case SIZE_SYMBOL:
		return size;
	case SIZE_SYM_LINE:
		return SymLine.width;
	case SIZE_XPOS:
		return fPos.fx;
	case SIZE_YPOS:
		return fPos.fy;
	default:
		return parent ? parent->GetSize(select) : defs.GetSize(select);
		}
}

bool
Symbol::SetSize(int select, double value)
{
	switch(select & 0xfff){
	case SIZE_MINE:
	case SIZE_SYMBOL:
		size = value;
		return true;
	case SIZE_SYM_LINE:
		SymLine.width = value;
		return true;
	case SIZE_XPOS:
		fPos.fx = value;
		return true;
	case SIZE_YPOS:
		fPos.fy = value;
		return true;
	}
	return false;
}

DWORD
Symbol::GetColor(int select)
{
	switch(select) {
	case COL_SYM_LINE:
		return SymLine.color;
	case COL_SYM_FILL:
		return SymFill.color;
	default:
		return parent ? parent->GetColor(select) : defs.Color(select);
		}
}

bool
Symbol::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_SYM_LINE:
		SymLine.color = col;
		if(SymTxt) SymTxt->ColTxt = col;
		return true;
	case COL_SYM_FILL:
		SymFill.color = col;
		return true;
	default:
		return false;
		}
}

void
Symbol::DoPlot(anyOutput *target)
{
	int ix, iy, rx, ry, atype;
	lfPOINT fip;
	POINT pts[5];
	FillDEF cf;

	if(size <= 0.001) return;
	atype = (type  & 0xfff);
	memcpy(&cf, &SymFill, sizeof(FillDEF));
	if(atype == SYM_CIRCLEF || atype == SYM_RECTF || atype == SYM_TRIAUF ||
		atype == SYM_TRIADF || atype == SYM_DIAMONDF) cf.color = SymLine.color;
	if(type & SYM_POS_PARENT) {
		if(!parent) return;
		fip.fx = parent->GetSize(SIZE_XCENTER);
		fip.fy = parent->GetSize(SIZE_YCENTER);
		}
	else if(!target->fp2fip(&fPos, &fip)) return;
	ix = iround(fip.fx);		iy = iround(fip.fy);
	target->SetLine(&SymLine);
	switch(atype){
	default:
	case SYM_CIRCLE:		//circle
	case SYM_CIRCLEF:		//filled circle
		rx = target->un2ix(size/2.0);		ry = target->un2iy(size/2.0); 
		target->SetFill(&cf);
		target->oCircle(ix-rx, iy-ry, ix+rx+1, iy+ry+1, name);
		rx--;ry--;			//smaller marking rectangle
		break;
	case SYM_RECT:			//rectange (square)
	case SYM_RECTF:			//filled rectangle
		rx = target->un2ix(size/2.25676f);
		ry = target->un2iy(size/2.25676f);
		target->SetFill(&cf);
		target->oRectangle(ix-rx, iy-ry, ix+rx+1, iy+ry+1, name);
		break;
	case SYM_TRIAU:			//triangles up and down, open or closed
	case SYM_TRIAUF:
	case SYM_TRIAD:
	case SYM_TRIADF:
		rx = target->un2ix(size/1.48503f);
		ry = target->un2iy(size/1.48503f);
		target->SetFill(&cf);
		pts[0].x = pts[3].x = ix - rx;		pts[1].x = ix;		pts[2].x = ix+rx;
		if(type == SYM_TRIAU || type == SYM_TRIAUF) {
			pts[0].y = pts[2].y = pts[3].y = iy+target->un2iy(size*0.38878f);
			pts[1].y = iy-target->un2iy(size*0.77756f);
			}
		else {
			pts[0].y = pts[2].y = pts[3].y = iy-target->un2iy(size*0.38878f);
			pts[1].y = iy+target->un2iy(size*0.77756f);
			}
		target->oPolygon(pts, 4);
		rx--; ry--;
		break;
	case SYM_DIAMOND:
	case SYM_DIAMONDF:
		rx = target->un2ix(size/1.59588f);
		ry = target->un2iy(size/1.59588f);
		target->SetFill(&cf);
		pts[0].x = pts[2].x = pts[4].x = ix;		
		pts[0].y = pts[4].y = iy -ry;
		pts[1].x = ix +rx;					pts[1].y = pts[3].y = iy;
		pts[2].y = iy +ry;					pts[3].x = ix-rx;
		target->oPolygon(pts, 5);
		rx--;									ry--;
		break;
	case SYM_STAR:			//star is a combination of + and x symbols
	case SYM_PLUS:			//draw a + sign
	case SYM_HLINE:
	case SYM_VLINE:
		rx = target->un2ix(size/2.0f);
		ry = target->un2iy(size/2.0f);
		pts[0].x = pts[1].x = ix;
		pts[0].y = iy - ry;					pts[1].y = iy + ry +1;
		if(type != SYM_HLINE) target->oPolyline(pts, 2);
		pts[0].x = ix -rx;					pts[1].x = ix + rx +1;
		pts[0].y = pts[1].y = iy;
		if(type != SYM_VLINE) target->oPolyline(pts, 2);
		if(type == SYM_VLINE){ rx = 2; break;}
		if(type == SYM_HLINE){ ry = 2; break;}
		if(type == SYM_PLUS) break;		//continue with x symbol for star
	case SYM_CROSS:			//draw a x symbol
		rx = target->un2ix(size/2.5);
		ry = target->un2iy(size/2.5);
		pts[0].x = ix - rx;					pts[1].x = ix + rx;
		pts[0].y = iy - ry;					pts[1].y = iy + ry;
		target->oPolyline(pts, 2);
		Swap(pts[0].y, pts[1].y);
		target->oPolyline(pts, 2);
		break;
	case SYM_TEXT:
		if(!SymTxt) Command(CMD_SETTEXT, (void *)"text", target);
		if(!SymTxt || !SymTxt->text || !SymTxt->text[0])return;
		SymTxt->iSize = target->un2iy(SymTxt->fSize = size *1.5);
		target->SetTextSpec(SymTxt);
		target->oTextOut(ix, iy, SymTxt->text, 0);
		if (target->oGetTextExtent(SymTxt->text, 0, &rx, &ry)){
			rx >>= 1;		ry >>= 1;
			}
		else rx = ry = 10;
		}
	rDims.left = ix-rx-1;				rDims.right = ix+rx+1;
	rDims.top = iy-ry-1;				rDims.bottom = iy+ry+1;
}

bool 
Symbol::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	char *tmptxt;
	AccRange *ac;
	int i, r, c;

	switch (cmd) {
	case CMD_FLUSH:
		if(SymTxt) {
			if(SymTxt->text) free(SymTxt->text);
			free(SymTxt);
			}
		if(ssRef) free(ssRef);	ssRef = 0L;
		if(name)free(name);		name = 0L;
		return true;
	case CMD_REDRAW:
		//if we come here its most likely the result of Undo
		if(parent && parent->Id==GO_REGRESSION)
			return parent->Command(CMD_MRK_DIRTY, 0L, o);
		return false;
	case CMD_GETTEXT:
		if(SymTxt && SymTxt->text && tmpl) {
			strcpy((char*)tmpl, SymTxt->text);
			return true;
			}
		return false;
	case CMD_SYMTEXT_UNDO:
		if(SymTxt && SymTxt->text){
			Undo.String(this, &SymTxt->text, UNDO_CONTINUE);
			if(SymTxt->text) free(SymTxt->text);
			if(tmpl) SymTxt->text = strdup((char*)tmpl);
			else SymTxt->text = 0L;
			return true;
			}
		//fall through if its new
	case CMD_SYMTEXT:		case CMD_SETTEXT:
		if(!SymTxt && (SymTxt = (TextDEF *) calloc(1, sizeof(TextDEF)))) {
			SymTxt->ColTxt = SymLine.color;
			SymTxt->fSize = size*1.5;
			SymTxt->ColBg = parent ? parent->GetColor(COL_BG) : 0x00ffffffL;
			SymTxt->Align = TXA_VCENTER | TXA_HCENTER;
			SymTxt->Style = TXS_NORMAL;
			SymTxt->Mode = TXM_TRANSPARENT;
			SymTxt->Font = FONT_HELVETICA;
			SymTxt->text = 0L;
			}
		if(!SymTxt) return false;
		if(SymTxt->text) free(SymTxt->text);
		if(tmpl) SymTxt->text = strdup((char*)tmpl);
		else SymTxt->text = 0L;
		return true;
	case CMD_SYM_TYPE:
		if(tmpl)type = *((int*)tmpl);
		return true;
	case CMD_GETTEXTDEF:
		if(!SymTxt || !tmpl) return false;
		memcpy(tmpl, SymTxt, sizeof(TextDEF));
		return true;
	case CMD_SYMTEXTDEF:
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		if(SymTxt) tmptxt = SymTxt->text;
		else tmptxt = 0L;
		if(!SymTxt && !(SymTxt = (TextDEF *) calloc(1, sizeof(TextDEF)))) return false;
		memcpy(SymTxt, tmpl, sizeof(TextDEF));
		SymTxt->text = tmptxt;
		return true;
	case CMD_SYM_RANGETEXT:
	case CMD_RANGETEXT:
		if(!data || !tmpl) return false;
		if(!(tmptxt = (char*)malloc(500)))return false;
		if((ac = new AccRange((char*)tmpl)) && ac->GetFirst(&c, &r)) {
			for(i = 0; i <= idx; i++) ac->GetNext(&c, &r);
			data->GetText(r, c, tmptxt, 500);
			delete(ac);
			}
		Command(CMD_SETTEXT, tmptxt, 0L);
		free(tmptxt);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_SYMBOL;
		data = (DataObj *)tmpl;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				o->ShowMark(&rDims, MRK_INVERT);
				CurrGO = this;
				return true;
				}
			break;
			}
		break;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bubbles are graphic objects
Bubble::Bubble(GraphObj *par, DataObj *d, double x, double y, double s, int which, 
	FillDEF *fill, LineDEF *outline, int xc, int xr, int yc, int yr, int sc,
	int sr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;	fPos.fy = y;	fs = s;
	type = which;
	if(fill) {
		memcpy(&BubbleFill,fill, sizeof(FillDEF));
		if(BubbleFill.hatch) memcpy(&BubbleFillLine, BubbleFill.hatch, sizeof(LineDEF));
		}
	BubbleFill.hatch = &BubbleFillLine;
	if(outline)memcpy(&BubbleLine, outline, sizeof(LineDEF));
	Id = GO_BUBBLE;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || sc >= 0 || sr >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*3)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = sc;	ssRef[2].y = sr;
			cssRef = 3;
			}
		}
}

Bubble::Bubble(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Bubble::~Bubble()
{
	Command(CMD_FLUSH, 0L, 0L);
}

void
Bubble::DoPlot(anyOutput *o)
{
	int x1, y1, x2, y2, ix, iy, tmp;
	double fix, fiy;

	o->SetLine(&BubbleLine);
	o->SetFill(&BubbleFill);
	switch(type & 0x0f0) {
	case BUBBLE_UNITS:
		fix = o->un2fix(fs);		fiy = o->un2fiy(fs);
		break;
	case BUBBLE_XAXIS:
		fix = (o->fx2fix(fPos.fx+fs) - o->fx2fix(fPos.fx-fs))/2.0;
		fiy = fix * (o->un2fiy(10.0f)/o->un2fix(10.0f));	//x and y resolution different ?
		break;
	case BUBBLE_YAXIS:
		fix = (o->fy2fiy(fPos.fy-fs) - o->fy2fiy(fPos.fy+fs))/2.0;
		fiy = fix * (o->un2fiy(10.0f)/o->un2fix(10.0f));	//x and y resolution different ?
		break;
		}
	fix = fix < 0.0 ? -fix : fix;							//sign must be positive
	fiy = fiy < 0.0 ? -fiy : fiy;
	rDims.left = rDims.right = iround(o->fx2fix(fPos.fx));
	rDims.top = rDims.bottom = iround(o->fy2fiy(fPos.fy));
	switch(type & 0x00f) {
	case BUBBLE_CIRCLE:
		ix = (int)(fix/2.0);			iy = (int)(fiy/2.0);
		tmp = iround(o->fx2fix(fPos.fx));		x1 = tmp - ix;		x2 = tmp + ix;
		tmp = iround(o->fy2fiy(fPos.fy));		y1 = tmp - iy;		y2 = tmp + iy;
		o->oCircle(x1, y1, x2, y2, name);
		UpdateMinMaxRect(&rDims, x1, y1);	UpdateMinMaxRect(&rDims, x2, y2);
		break;
	case BUBBLE_SQUARE:
		if((type & 0xf00) == BUBBLE_CIRCUM) {
			ix = iround(fix*.392699081);		iy = iround(fiy*.392699081);
			}
		else if((type & 0xf00) == BUBBLE_AREA) {
			ix = iround(fix*.443113462);		iy = iround(fiy*.443113462);
			}
		else {
			ix = iround(fix*.353553391);		iy = iround(fiy*.353553391);
			}
		tmp = iround(o->fx2fix(fPos.fx));		x1 = tmp - ix;		x2 = tmp + ix;
		tmp = iround(o->fy2fiy(fPos.fy));		y1 = tmp - iy;		y2 = tmp + iy;
		o->oRectangle(x1, y1, x2, y2, name);
		UpdateMinMaxRect(&rDims, x1, y1);	UpdateMinMaxRect(&rDims, x2, y2);
		break;
	case BUBBLE_UPTRIA:
	case BUBBLE_DOWNTRIA:
		if((type & 0xf00) == BUBBLE_CIRCUM) {
			fix *= .523598775;		fiy *= .523598775;
			}
		else if((type & 0xf00) == BUBBLE_AREA) {
			fix *= .673386843;		fiy *= .673386843;
			}
		else {
			fix *=.433012702;		fiy *= .433012702;
			}
		ix =  iround(fix);		iy = iround(fiy*.57735);
		tmp = iround(o->fx2fix(fPos.fx));
		pts[0].x = pts[3].x = tmp - ix;		pts[1].x = tmp + ix;		pts[2].x = tmp;
		tmp = iround(o->fy2fiy(fPos.fy));
		if((type & 0x00f) == BUBBLE_UPTRIA) {
			pts[0].y = pts[1].y = pts[3].y = tmp + iy;
			pts[2].y = tmp - iround(fiy*1.1547);
			}
		else {
			pts[0].y = pts[1].y = pts[3].y = tmp - iy;
			pts[2].y = tmp + iround(fiy*1.1547);
			}
		o->oPolygon(pts, 4);
		UpdateMinMaxRect(&rDims, pts[0].x, pts[0].y);
		UpdateMinMaxRect(&rDims, pts[1].x, pts[2].y);
		break;
		}
}

void
Bubble::DoMark(anyOutput *o, bool mark)
{
	if(mark) {
		BubbleFillLine.color ^= 0x00ffffffL;
		BubbleFill.color ^= 0x00ffffffL;
		DoPlot(o);
		BubbleFill.color ^= 0x00ffffffL;
		BubbleFillLine.color ^= 0x00ffffffL;
		}
	else {
		if(parent) parent->DoPlot(o);
		else DoPlot(o);
		}
	o->UpdateRect(&rDims, false);
}

bool 
Bubble::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	bool bSelected = false;
	unsigned long n, s;
	POINT p;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);	ssRef = 0L;
		if(name)free(name);		name = 0L;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, p.x = mev->x, p.y = mev->y) && !CurrGO) {
				switch(type & 0x00f) {
				case BUBBLE_CIRCLE:
					n = s = p.x - ((rDims.right+rDims.left)>>1);
					s *= n;
					n = p.y - ((rDims.bottom+rDims.top)>>1);
					n = isqr(s += n*n) -2;
					bSelected = ((unsigned)((rDims.right-rDims.left)>>1) > n);
					break;
				case BUBBLE_SQUARE:
					bSelected = true;
					break;
				case BUBBLE_UPTRIA:
				case BUBBLE_DOWNTRIA:
					if(!(bSelected = IsInPolygon(&p, pts, 4)))
						bSelected = IsCloseToPL(p, pts, 4);
					break;
					}
				if(bSelected) o->ShowMark(this, MRK_GODRAW);
				return bSelected;
				}
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_BUBBLE;
		data = (DataObj*)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fs);
			return true;
			}
		return false;
	case CMD_BUBBLE_ATTRIB:
		if(tmpl) {
			type &= ~0xff0;
			type |= (*((int*)tmpl) & 0xff0);
			return true;
			}
		return false;
	case CMD_BUBBLE_TYPE:
		if(tmpl) {
			type &= ~0x00f;
			type |= (*((int*)tmpl) & 0x00f);
			return true;
			}
		return false;
	case CMD_BUBBLE_FILL:
		if(tmpl) {
			BubbleFill.type = ((FillDEF*)tmpl)->type;
			BubbleFill.color = ((FillDEF*)tmpl)->color;
			BubbleFill.scale = ((FillDEF*)tmpl)->scale;
			if(((FillDEF*)tmpl)->hatch)
				memcpy(&BubbleFillLine, ((FillDEF*)tmpl)->hatch, sizeof(LineDEF));
			}
		return true;
	case CMD_BUBBLE_LINE:
		if(tmpl) memcpy(&BubbleLine, tmpl, sizeof(LineDEF));
		return true;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Bars are graphic objects
Bar::Bar(GraphObj *par, DataObj *d, double x, double y, int which,int xc, int xr,
		int yc, int yr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	parent = par;
	fPos.fx = x;
	fPos.fy = y;
	type = which;
	if(type & BAR_RELWIDTH) size = 60.0;
	data = d;
	Id = GO_BAR;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*2)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			cssRef = 2;
			}
		}
}

Bar::Bar(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Bar::~Bar()
{
	Command(CMD_FLUSH, 0L, 0L);
}

double
Bar::GetSize(int select)
{
	switch(select){
	case SIZE_XPOS:				return fPos.fx;
	case SIZE_YPOS:				return fPos.fy;
		}
	return 0.0;
}

bool
Bar::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_BAR: 
		size = value;
		return true;
	case SIZE_BAR_LINE:
		BarLine.width = value;
		return true;
	case SIZE_XBASE:
		BarBase.fx = value;
		return true;
	case SIZE_YBASE:
		BarBase.fy = value;
		return true;
		}
	return false;
}

bool
Bar::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_BAR_LINE:
		BarLine.color = col;
		return true;
	case COL_BAR_FILL:
		BarFill.color = col;
		return true;
		}
	return false;
}

void
Bar::DoPlot(anyOutput *target)
{
	int w;
	double fBase, rsize;
	POINT pts[2];

	if(!parent || size <= 0.001) return;
	target->SetLine(&BarLine);
	target->SetFill(&BarFill);
	switch(type & 0xff) {
	case BAR_VERTU:		case BAR_VERTT:		case BAR_VERTB:
		switch(type & 0xff) {
		case BAR_VERTB:
			fBase = parent->GetSize(SIZE_BOUNDS_BOTTOM);
			break;
		case BAR_VERTT:
			fBase = parent->GetSize(SIZE_BOUNDS_TOP);
			break;
		case BAR_VERTU:
			fBase = BarBase.fy;
			break;
			}
		if(type & BAR_RELWIDTH) {
			rsize = size * parent->GetSize(SIZE_BARMINX)/100.0;
			pts[0].x = iround(target->fx2fix(fPos.fx - rsize/2.0));
			pts[1].x = iround(target->fx2fix(fPos.fx + rsize/2.0));
			}
		else {
			w = target->un2ix(size);
			pts[0].x = iround(target->fx2fix(fPos.fx)) - (w>>1);
			pts[1].x = pts[0].x + w;
			}
		if(type & BAR_CENTERED) {
			pts[0].y = iround(target->fy2fiy(fBase - (fPos.fy - fBase)));
			pts[1].y = iround(target->fy2fiy(fBase + (fPos.fy - fBase)));
			}
		else {
			pts[0].y = iround(target->fy2fiy(fBase));
			pts[1].y = iround(target->fy2fiy(fPos.fy));
			}
		break;
	case BAR_HORU:		case BAR_HORR:		case BAR_HORL:
		switch(type & 0xff) {
		case BAR_HORL:
			fBase = parent->GetSize(SIZE_BOUNDS_LEFT);
			break;
		case BAR_HORR:
			fBase = parent->GetSize(SIZE_BOUNDS_RIGHT);
			break;
		case BAR_HORU:
			fBase = BarBase.fx;
			break;
			}
		if(type & BAR_RELWIDTH) {
			rsize = size * parent->GetSize(SIZE_BARMINY)/100.0;
			pts[0].y = iround(target->fy2fiy(fPos.fy - rsize/2.0));
			pts[1].y = iround(target->fy2fiy(fPos.fy + rsize/2.0));
			}
		else {
			w = target->un2iy(size);
			pts[0].y = target->fy2iy(fPos.fy) - w/2;
			pts[1].y = pts[0].y+w;
			}
		if(type & BAR_CENTERED) {
			pts[0].x = target->fx2ix(fBase - (fPos.fx - fBase));
			pts[1].x = target->fx2ix(fBase + (fPos.fx - fBase));
			}
		else {
			pts[0].x = target->fx2ix(fBase);
			pts[1].x = target->fx2ix(fPos.fx);
			}
		break;
	default:
		return;
		}
	if(pts[0].x == pts[1].x || pts[0].y == pts[1].y) {
		target->oSolidLine(pts);
		}
	else target->oRectangle(pts[0].x, pts[0].y, pts[1].x, pts[1].y, name);
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
}

bool 
Bar::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	FillDEF *TmpFill;
	lfPOINT bl;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);		ssRef = 0L;
		if(name)free(name);			name = 0L;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		((Legend*)tmpl)->HasFill(&BarLine, &BarFill);
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				o->ShowMark(&rDims, MRK_INVERT);
				CurrGO = this;
				return true;
				}
			break;
			}
		return false;
	case CMD_BAR_FILL:
		TmpFill = (FillDEF *)tmpl;
		if(TmpFill) {
			BarFill.type = TmpFill->type;
			BarFill.color = TmpFill->color;
			BarFill.scale = TmpFill->scale;
			if(TmpFill->hatch) memcpy(&HatchLine, TmpFill->hatch, sizeof(LineDEF));
			}
		return true;
	case CMD_BAR_TYPE:
		if(tmpl) type = *((int*)tmpl);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_BAR;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			switch(type & 0xff) {
			case BAR_VERTU:
			case BAR_VERTT:
			case BAR_VERTB:
				bl.fx = fPos.fx;
				switch (type & 0xff) {
				case BAR_VERTU:
					bl.fy = BarBase.fy;
					break;
				case BAR_VERTT:
				case BAR_VERTB:
					bl.fy = 0.0f;		//cannot resolve
					break;
					}
				if(type & BAR_CENTERED) bl.fy -= fPos.fy;
				break;
			case BAR_HORU:
			case BAR_HORR:
			case BAR_HORL:
				bl.fy = fPos.fy;
				switch(type & 0xff) {
				case BAR_HORU:
					bl.fx = BarBase.fx;
				case BAR_HORR:
				case BAR_HORL:
					bl.fx = 0.0f;		//cannot resolve
					}
				if(type & BAR_CENTERED) bl.fx -= fPos.fx;
				break;
				}
			((Plot*)parent)->CheckBounds(bl.fx, bl.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Data line is a graphic object
DataLine::DataLine(GraphObj *par, DataObj *d, char *xrange, char *yrange):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_DATALINE;
	if(xrange)ssXref = strdup(xrange);	if(yrange)ssYref = strdup(yrange);
	SetValues();
}
	
DataLine::DataLine(GraphObj *par, DataObj *d, lfPOINT *val, long nval):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Values = val;			nPnt = nval;	nPntSet = nPnt-1;
	Id = GO_DATALINE;
}

DataLine::DataLine(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

DataLine::~DataLine()
{
	if(Values)free(Values);		Values = 0L;
	if(pts) free(pts);			pts = 0L;
	if(ssXref) free(ssXref);	ssXref = 0L;
	if(ssYref) free(ssYref);	ssYref = 0L;
	if(mo) DelBitmapClass(mo);	mo = 0L;
	if(parent)parent->Command(CMD_MRK_DIRTY, 0L, 0L);
}

bool
DataLine::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_DATA_LINE:
		LineDef.color = col;
		return true;
		}
	return false;
}

void
DataLine::DoPlot(anyOutput *target)
{
	int i;
	lfPOINT fip;
	POINT pn, *tmppts;

	if(!Values || nPntSet < 1) return;
	if(mo) DelBitmapClass(mo);		mo = 0L;
	if(pts) free(pts);
	if(type & 0xff) pts = (POINT *)malloc(sizeof(POINT)*(nPntSet+2)*2);
	else pts = (POINT *)malloc(sizeof(POINT)*(nPntSet+2));
	if(!pts)return;
	if(max.fx > min.fx && max.fy > min.fy) dirty = false;
	else if(dirty)Command(CMD_AUTOSCALE, 0L, target);
	cp = 0;
	switch(type & 0xf) {
	case 0:
		for (i = 0; i <= nPntSet; i++){
			target->fp2fip(Values+i, &fip);
			pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 5:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		target->fp2fip(Values+1, &fip);
		pn.y += (pn.y -iround(fip.fy))>>1;
		AddToPolygon(&cp, pts, &pn);
	case 1:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(+fip.fy);
		for (i = 0; i <= nPntSet; i++){
			target->fp2fip(Values+i, &fip);
			pn.x = iround(fip.fx);			AddToPolygon(&cp, pts, &pn);
			pn.y = iround(fip.fy);			AddToPolygon(&cp, pts, &pn);
			}
		if((type &0xf) == 5) {
			target->fp2fip(Values+i-2, &fip);
			pn.x += (pn.x - iround(fip.fx))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 6:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		target->fp2fip(Values+1, &fip);
		pn.x += (pn.x - iround(fip.fx))>>1;
		AddToPolygon(&cp, pts, &pn);
	case 2:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		for (i = 0; i <= nPntSet; i++){
			target->fp2fip(Values+i, &fip);
			pn.y = iround(fip.fy);			AddToPolygon(&cp, pts, &pn);
			pn.x = iround(fip.fx);			AddToPolygon(&cp, pts, &pn);
			}
		if((type &0xf) == 6) {
			target->fp2fip(Values+i-2, &fip);
			pn.y += (pn.y - iround(fip.fy))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 7:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		target->fp2fip(Values+1, &fip);
		pn.x += (pn.x - iround(fip.fx))>>1;
		AddToPolygon(&cp, pts, &pn);
	case 3:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		for (i = 0; i <= nPntSet; i++){
			target->fp2fip(Values+i, &fip);
			pn.x = (pn.x + iround(fip.fx))>>1;	AddToPolygon(&cp, pts, &pn);
			pn.y = iround(fip.fy);				AddToPolygon(&cp, pts, &pn);
			pn.x = iround(fip.fx);
			}
		AddToPolygon(&cp, pts, &pn);
		if((type &0xf) == 7) {
			target->fp2fip(Values+i-2, &fip);
			pn.x += (pn.x - iround(fip.fx))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
	case 8:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		target->fp2fip(Values+1, &fip);
		pn.y += (pn.y - iround(fip.fy))>>1;
		AddToPolygon(&cp, pts, &pn);
	case 4:
		target->fp2fip(Values, &fip);
		pn.x = iround(fip.fx);		pn.y = iround(fip.fy);
		for (i = 0; i <= nPntSet; i++){
			target->fp2fip(Values+i, &fip);
			pn.y = (pn.y + iround(fip.fy))>>1;	AddToPolygon(&cp, pts, &pn);
			pn.x = iround(fip.fx);				AddToPolygon(&cp, pts, &pn);
			pn.y = iround(fip.fy);
			}
		AddToPolygon(&cp, pts, &pn);
		if((type &0xf) == 8) {
			target->fp2fip(Values+i-2, &fip);
			pn.y += (pn.y - iround(fip.fy))>>1;
			AddToPolygon(&cp, pts, &pn);
			}
		break;
		}
	if(cp < 2) return;
	if(isPolygon) {			//for mark polygon only !!
		AddToPolygon(&cp, pts, pts);
		}
	else{
		target->SetLine(&LineDef);
		target->oPolyline(pts, cp);
		}
	if(tmppts = (POINT*)realloc(pts, cp *sizeof(POINT))) pts = tmppts;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < cp; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 2*target->un2ix(LineDef.width);		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i);
}

void
DataLine::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark){
			memcpy(&mrc, &rDims, sizeof(RECT));
			IncrementMinMaxRect(&mrc, 6 + o->un2ix(LineDef.width));
			mo = GetRectBitmap(&mrc, o);
			InvertLine(pts, cp, &LineDef, &mrc, o, mark);
			}
		else if(mo) RestoreRectBitmap(&mo, &mrc, o);
		}
}

bool
DataLine::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	bool bFound = false;
	POINT p1;
	int i;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(&rDims, p1.x= mev->x, p1.y= mev->y) || CurrGO || !o || nPntSet <2)
				return false; 
			if(isPolygon && IsInPolygon(&p1, pts, cp)) bFound = true;
			if(bFound || IsCloseToPL(p1,pts,cp)) 
				return o->ShowMark(this, MRK_GODRAW);
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = isPolygon ? GO_DATAPOLYGON : GO_DATALINE;
		data = (DataObj*)tmpl;
		return true;
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND) {
			if(Id == GO_DATALINE) ((Legend*)tmpl)->HasFill(&LineDef, 0L);
			}
		break;
	case CMD_SET_LINE:
		if(tmpl) memcpy(&LineDef, tmpl, sizeof(LineDEF));
		return true;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		SetValues();
		return true;
	case CMD_AUTOSCALE:
		if(nPntSet < 2 || !Values) return false;
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			if(dirty) {
				min.fx = max.fx = Values[0].fx;	min.fy = max.fy = Values[0].fy;
				for (i = 1; i <= nPntSet; i++){
					min.fx = Values[i].fx < min.fx ? Values[i].fx : min.fx;
					max.fx = Values[i].fx > max.fx ? Values[i].fx : max.fx;
					min.fy = Values[i].fy < min.fy ? Values[i].fy : min.fy;
					max.fy = Values[i].fy > max.fy ? Values[i].fy : max.fy;
					}
				}
			((Plot*)parent)->CheckBounds(min.fx, min.fy);
			((Plot*)parent)->CheckBounds(max.fx, max.fy);
			dirty = false;
			return true;
			}
		return false;
		}
	return false;
}

void
DataLine::SetValues()
{
	AccRange *rX, *rY1=0L, *rY2=0L;
	int i, j, k, l, m, n;
	double x, y;
	char *yref1 = 0L, *yref2 = 0L;
	lfPOINT *tmpValues = 0L;

	if(!ssXref || !ssYref) return;
	if(!(yref1 = strdup(ssYref)))return;
	for(i = 0; yref1[i]; i++) {
		if(yref1[i] == ';') {
			yref1[i++] = 0;
			while(yref1[i] && yref1[i] < 33) i++;
			yref2 = strdup(yref1+i);
			}
		}
	nPnt = nPntSet = 0;
	if(Values) free(Values);		Values = 0L;
	min.fx = min.fy = 3.0e38f;		max.fx = max.fy = -3.0e38f;
	rX = new AccRange(ssXref);		rY1 = new AccRange(yref1);
	if(!rX || !rY1){
		if(yref1) free(yref1);		if(yref2) free(yref2);
		if(rX) delete(rX);	if(rY1) delete(rY1);
		return;
		}
	if(yref2 &&((nPnt = rX->CountItems()) == (rY1->CountItems()))) {
		if(!(Values = (lfPOINT *)calloc(nPnt*2+2, sizeof(lfPOINT)))) return; 
		if(!(rY2 = new AccRange(yref2))) {
			if(yref1) free(yref1);		if(yref2) free(yref2);
			if(rX) delete(rX);	if(rY1) delete(rY1);
			return;
			}
		if(rX->GetFirst(&i, &j) && rY1->GetFirst(&k, &l) && 
			rX->GetNext(&i, &j) && rY1->GetNext(&k, &l) &&
			rY2->GetFirst(&m, &n) && rY2->GetNext(&m, &n)) do {
			if(data->GetValue(j, i, &x)){
				if(data->GetValue(l, k, &y)){
					Values[nPntSet].fx = x;			Values[nPntSet++].fy = y;
					}
				if(data->GetValue(n, m, &y)){
					Values[nPntSet].fx = x;			Values[nPntSet++].fy = y;
					}
				}
			}while(rX->GetNext(&i, &j) && rY1->GetNext(&k, &l) && rY2->GetNext(&m, &n));
		}
	else {
		if((nPnt = rX->CountItems()) != (rY1->CountItems())) return;
		if(!(Values = (lfPOINT *)calloc(nPnt+2, sizeof(lfPOINT)))) return; 
		if(rX->GetFirst(&i, &j) && rY1->GetFirst(&k, &l) && 
			rX->GetNext(&i, &j) && rY1->GetNext(&k, &l)) do {
			if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y)){
				Values[nPntSet].fx = x;				Values[nPntSet++].fy = y;
				}
			}while(rX->GetNext(&i, &j) && rY1->GetNext(&k, &l));
		}
	nPnt = nPntSet;		nPntSet--;	dirty = true;
	Command(CMD_AUTOSCALE, 0L, 0L);
	if(tmpValues = (lfPOINT *)realloc(Values, (nPnt+2)*sizeof(lfPOINT)))
		Values = tmpValues;
	if(rX) delete(rX);	if(rY1) delete(rY1);	if(rY2) delete(rY2);
	if(yref1) free(yref1);	if(yref2) free(yref2);
}

void
DataLine::LineData(lfPOINT *val, long nval)
{
	if(Values)free(Values);		Values = 0L;
	if(pts) free(pts);			pts = 0L;
	Values = val;				dirty = true;			
	nPnt = nval;				nPntSet = nPnt-1;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// DataPolygon is a graphic object based on DataLine
DataPolygon::DataPolygon(GraphObj *par, DataObj *d, char *xrange, char *yrange):
	DataLine(par, d, xrange, yrange)
{
	lfPOINT *fp = Values;
	char *rx = ssXref;
	char *ry = ssYref;

	FileIO(INIT_VARS);
	Values = fp;				//FileIO will just set Values to 0L !
	ssXref = rx;			ssYref = ry;
	Id = GO_DATAPOLYGON;
}

DataPolygon::DataPolygon(int src):DataLine(0L, 0)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

DataPolygon::~DataPolygon()
{
	if(Values)free(Values);		Values =0L;
	if(pts) free (pts);			pts = 0L;
	if(ssXref) free(ssXref);	ssXref = 0L;
	if(ssYref) free(ssYref);	ssYref = 0L;
	if(mo) DelBitmapClass(mo);	mo = 0L;
	if(parent)parent->Command(CMD_MRK_DIRTY, 0L, 0L);
}

void
DataPolygon::DoPlot(anyOutput *target)
{
	if(!Values || nPntSet < 2) return;
	if(mo) DelBitmapClass(mo);	mo = 0L;
	DataLine::DoPlot(target);	//no drawing but fill pts only
	target->SetLine(&LineDef);	target->SetFill(&pgFill);
	target->oPolygon(pts, cp);
}

void
DataPolygon::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark){
			memcpy(&mrc, &rDims, sizeof(RECT));
			IncrementMinMaxRect(&mrc, 6 + o->un2ix(LineDef.width));
			mo = GetRectBitmap(&mrc, o);
			InvertPolygon(pts, cp, &LineDef, &pgFill, &mrc, o, mark);
			}
		else RestoreRectBitmap(&mo, &mrc, o);
		}
}

bool
DataPolygon::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_PG_FILL:
		if(tmpl) {
			memcpy((void*)&pgFill, tmpl, sizeof(FillDEF));
			if(pgFill.hatch) memcpy((void*)&pgFillLine, (void*)pgFill.hatch, sizeof(LineDEF));
			pgFill.hatch = (LineDEF*)&pgFillLine;
			}
		return true;
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND) {
			if(Id == GO_DATAPOLYGON) ((Legend*)tmpl)->HasFill(&LineDef, &pgFill);
			}
		break;
	default:
		return DataLine::Command(cmd, tmpl, o);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a regression line
// Ref.: "Biometry" third edition 1995 (ed. R.R. Sokal and F.J. Rohlf),
// W.H. Freeman and Company, New York; ISBN 0-7167-2411-1; pp. 451ff
RegLine::RegLine(GraphObj *par, DataObj *d, lfPOINT *values, long n, int sel):
	GraphObj(par, d)
{
	FileIO(INIT_VARS);
	type = sel;
	Id = GO_REGLINE;
	uclip.Xmin = uclip.Ymin = lim.Xmin = lim.Ymin = -1.0;
	uclip.Xmax = uclip.Ymax = lim.Xmax = lim.Ymax = 1.0;
	Recalc(values, n);
}

RegLine::RegLine(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) FileIO(FILE_READ);
}

RegLine::~RegLine()
{
	if(pts) free(pts);
	pts = 0L;
	if(parent)parent->Command(CMD_MRK_DIRTY, 0L, 0L);
}

double
RegLine::GetSize(int select)
{
	double a, b;

	switch(select) {
	case SIZE_MX:		return mx;
	case SIZE_MY:		return my;
	case SIZE_A:
	case SIZE_B:
		switch(type & 0x07) {
		case 1:		a = l2.fx;	b = l2.fy;	break;
		case 2:		a = l3.fx;	b = l3.fy;	break;
		case 3:		a = l4.fx;	b = l4.fy;	break;
		case 4:		a = l5.fx;	b = l5.fy;	break;
		default:	a = l1.fx;	b = l1.fy;	break;
			}
		if(select == SIZE_A) return a;
		else return b;
		}
	return 0.0;
}

void
RegLine::DoPlot(anyOutput *o)
{
	int i;
	POINT pn, *tmppts;
	double x, x1, y, d, a, b;
	fRECT cliprc;
	bool dValid;

	switch (type & 0x70) {
	case 0x20:	memcpy(&cliprc, &uclip, sizeof(fRECT));		break;
	case 0x10:
		if(parent) {
			cliprc.Xmin = parent->GetSize(SIZE_BOUNDS_LEFT);
			cliprc.Xmax = parent->GetSize(SIZE_BOUNDS_RIGHT);
			cliprc.Ymin = parent->GetSize(SIZE_BOUNDS_BOTTOM);
			cliprc.Ymax = parent->GetSize(SIZE_BOUNDS_TOP);
			break;
			}
		//no parent: use default
	default:	memcpy(&cliprc, &lim, sizeof(fRECT));		break;
		}
	if(cliprc.Xmax < cliprc.Xmin) {
		x = cliprc.Xmax;	cliprc.Xmax = cliprc.Xmin;	cliprc.Xmin = x;
		}
	if(cliprc.Ymax < cliprc.Ymin) {
		y = cliprc.Ymax;	cliprc.Ymax = cliprc.Ymin;	cliprc.Ymin = y;
		}
	if(cliprc.Xmin == cliprc.Xmax || cliprc.Ymin == cliprc.Ymax) return;
	if(pts) free(pts);
	if(!(pts = (POINT *)malloc(sizeof(POINT)*200)))return;
	switch(type & 0x07) {
	case 1:		a = l2.fx;	b = l2.fy;	break;
	case 2:		a = l3.fx;	b = l3.fy;	break;
	case 3:		a = l4.fx;	b = l4.fy;	break;
	case 4:		a = l5.fx;	b = l5.fy;	break;
	default:	a = l1.fx;	b = l1.fy;	break;
		}
	x = cliprc.Xmin;	d = (cliprc.Xmax - cliprc.Xmin)/200.0;
	for (cp = i = 0; i <= 200; i++){
		dValid = true;
		switch(type & 0x700) {
		case 0x100:					//logarithmic x
			if(dValid = x > defs.min4log) x1 = log10(x);
			break;
		case 0x200:					//reciprocal x
			if(dValid = fabs(x) > defs.min4log) x1 = 1.0/x;
			break;
		case 0x300:					//square root x
			if(dValid = fabs(x) > defs.min4log) x1 = sqrt(x);
			break;
		default:	x1 = x;	break;		//linear x
			}
		y = a + b*x1;
		if(dValid) switch(type & 0x7000) {
		case 0x1000:				//logarithmic y
			y = pow(10.0, y);
			break;
		case 0x2000:				//reciprocal y
			if(dValid = fabs(y) >0.0001) y = 1.0/y;
			break;
		case 0x3000:				//square root y
			if(dValid = fabs(y) >0.0001) y = y*y;
			break;
			}
		if(dValid && y >= cliprc.Ymin && y <= cliprc.Ymax) {
			pn.x = o->fx2ix(x);	pn.y = o->fy2iy(y);
			AddToPolygon(&cp, pts, &pn);
			}
		x += d;
		}
	if(cp < 2) return;
	o->SetLine(&LineDef);
	o->oPolyline(pts, cp);
	if(tmppts = (POINT*)realloc(pts, cp *sizeof(POINT))) pts = tmppts;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < cp; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 2*o->un2ix(LineDef.width);		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i);
}

void
RegLine::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark)InvertLine(pts, cp, &LineDef, &rDims, o, mark);
		else if(parent) parent->Command(CMD_REDRAW, 0L, o);
		}
}

bool
RegLine::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(&rDims, p1.x= mev->x, p1.y= mev->y) || CurrGO || !o || nPoints <2)
				return false; 
			if(IsCloseToPL(p1,pts,cp)) return o->ShowMark(CurrGO= this, MRK_GODRAW);
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_REGLINE;
		return true;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_BOUNDS:
		if(tmpl) {
			memcpy(&lim, tmpl, sizeof(fRECT));
			memcpy(&uclip, tmpl, sizeof(fRECT));
			}
		return true;
	case CMD_AUTOSCALE:
		if(nPoints < 2) return false;
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(lim.Xmin, lim.Ymin);
			((Plot*)parent)->CheckBounds(lim.Xmax, lim.Ymax);
			return true;
			}
		return false;
		}
	return false;
}

void
RegLine::Recalc(lfPOINT *values, long n)
{
	double sx, sy, dx, dy, sxy, sxx, syy;
	double a, b, k;
	long ic;

	sx = sy = 0.0;
	if((nPoints = n)<2) return;
	for(ic = 0; ic < n; ic++) {
		sx += values[ic].fx;		sy += values[ic].fy;
		}
	mx = sx /((double)nPoints);	my = sy/((double)nPoints);
	sxy = sxx = syy = 0.0;
	for(ic = 0; ic < n; ic++) {
		dx = mx - values[ic].fx;	dy = my - values[ic].fy;
		sxx += (dx*dx);	syy += (dy*dy);	sxy += (dx*dy);
		}
	l1.fy = sxy / sxx;			l1.fx = my - (sxy / sxx) * mx;
	b = sxy / syy;				a = mx - (sxy / syy) * my;
	l2.fy = 1.0/b;				l2.fx = -a / b;
	l3.fy = (l1.fy+l2.fy)/2.0;	l3.fx = (l1.fx+l2.fx)/2.0;
	l4.fy = sy/sx;				l4.fx = 0.0;
	if(l5.fx == 0.0 && l5.fx == 0.0){
		l5.fy = l1.fy;				l5.fx = l1.fx;
		}
	//calculate distance point from line algorithm
	//Ref: K. Thompson, 1990: Vertical Distance from a Point to a Line. In:
	//   Graphic Gems (Andrew S. Glassner, ed.), Academic Press,
	//   pp. 47-48; ISBN 0-12-286165-5
	k = (sqrt(1.0/(1.0+l1.fy*l1.fy))+sqrt(1.0/(1.0+l2.fy*l2.fy)))/2.0;
	b = sqrt(1.0/(k*k) -1.0);
	l3.fy = l3.fy > 0.0 ? b : -b;
	l3.fx = my - mx * l3.fy;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a statnard deviation (SD-) ellipse
SDellipse::SDellipse(GraphObj *par, DataObj *d, lfPOINT *values, long n, int sel):
	GraphObj(par, d)
{
	FileIO(INIT_VARS);
	type = sel;
	Id = GO_SDELLIPSE;
	if(val = (lfPOINT*)malloc(n * sizeof(lfPOINT))){
		memcpy(val, values, (nPoints = n)*sizeof(lfPOINT));
		rl = new RegLine(this, data, values, n, type);
		}
}

SDellipse::SDellipse(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) FileIO(FILE_READ);
}

SDellipse::~SDellipse()
{
	if(val) free(val);
	if(pts) free(pts);
	if(!(type & 0x10000) && parent && rl && 
		parent->Command(CMD_DROP_OBJECT, rl, 0L)) return;
	if(rl) DeleteGO(rl);
}

void
SDellipse::DoPlot(anyOutput *o)
{
	int i;
	double a1, b1, a2, b2, fv, k1, k2, ss1, ss2, np, x, dx, si, csi;
	lfPOINT fp, fip;
	POINT p1, *tmppts;

	if(!rl) return;
	if(pts) free(pts);
	if(!(pts = (POINT *)malloc(sizeof(POINT)*420)))return;
	//get line data from regression line object
	mx = rl->GetSize(SIZE_MX);		my = rl->GetSize(SIZE_MY);
	a1 = rl->GetSize(SIZE_A);		b1 = rl->GetSize(SIZE_B);
	b2 = -1.0/b1;	a2 = my - b2 * mx;
	//calculate sine and cosine for back rotation
	fv = sqrt(1.0+b1*b1);			si = b1/fv;			csi = 1.0/fv;
	//calculate distance from line for each point and squared sum of distances
	//Ref: K. Thompson, 1990: Vertical Distance from a Point to a Line. In:
	//   Graphic Gems (Andrew S. Glassner, ed.), Academic Press,
	//   pp. 47-48; ISBN 0-12-286165-5
	k1 = sqrt(1.0/(1.0+b1*b1));			k2 = sqrt(1.0/(1.0+b2*b2));
	// y = a + b*x;
	ss1 = ss2 = 0.0;
	for(i = 0; i < nPoints; i++) {
		fv = (a1 + b1 * val[i].fx - val[i].fy) * k1;	ss1 += (fv*fv);
		fv = (a2 + b2 * val[i].fx - val[i].fy) * k2;	ss2 += (fv*fv);
		}
	np = ((double)(nPoints-1));
	//SD perpendicular and in direction of regression line
	sd1 = sqrt(ss1 /= np);		sd2 = sqrt(ss2 /= np);
	dx = sd2/100.0;
	for(i = 0, cp = 0, x = -sd2; i < 2; i++) {
		do {
			fv = (x*x)/ss2;
			fv = fv < 0.99999 ? sqrt((1.0-fv)*ss1) : 0.0;
			fv = i ? fv : -fv;
			fp.fx = mx + x * csi - fv * si;
			fp.fy = my + x * si + fv * csi;
			switch(type & 0x700) {
			case 0x100:					//logarithmic x
				fp.fx = pow(10.0, fp.fx);
				break;
			case 0x200:					//reciprocal x
				if(fabs(fp.fx) > defs.min4log) fp.fx = 1.0/fp.fx;
				else fp.fx = 0.0;
				break;
			case 0x300:					//square root x
				if(fabs(fp.fx) > defs.min4log) fp.fx = fp.fx*fp.fx;
				else fp.fx = 0.0;
				break;
				}
			switch(type & 0x7000) {
			case 0x1000:				//logarithmic y
				fp.fy = pow(10.0, fp.fy);
				break;
			case 0x2000:				//reciprocal y
				if(fabs(fp.fy) > defs.min4log) fp.fy = 1.0/fp.fy;
				else fp.fy = 0.0;
				break;
			case 0x3000:				//square root y
				if(fabs(fp.fy) > defs.min4log) fp.fy = fp.fy*fp.fy;
				else fp.fy = 0.0;
				break;
				}
			o->fp2fip(&fp, &fip);	p1.x = iround(fip.fx);		p1.y = iround(fip.fy);
			AddToPolygon(&cp, pts, &p1);
			}while((x += dx) < sd2 && x > -sd2);
		x = sd2;
		dx *= -1.0;
		}
	o->SetLine(&LineDef);
	if(cp > 2) {
		AddToPolygon(&cp, pts, pts);		//close polygon
		if(tmppts = (POINT*)realloc(pts, cp *sizeof(POINT))) pts = tmppts;
		SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
		for(i = 2; i < cp; i++) 
			UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
		i = 3*o->un2ix(LineDef.width);		//increase size of rectangle for marks
		IncrementMinMaxRect(&rDims, i);
		o->oPolyline(pts, cp);
		}
	else {
		free(pts);
		cp = 0;
		}
	if(!(type & 0x10000))rl->DoPlot(o);
}

void
SDellipse::DoMark(anyOutput *o, bool mark)
{
	if(pts && cp && o){
		if(mark)InvertLine(pts, cp, &LineDef, &rDims, o, mark);
		else if(parent) parent->Command(CMD_REDRAW, 0L, o);
		}
}

bool
SDellipse::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!(type & 0x10000) && rl && rl->Command(cmd, tmpl, o)) return true;
			if(!IsInRect(&rDims, p1.x= mev->x, p1.y= mev->y) || CurrGO || !o || nPoints <2)
				return false; 
			if(IsCloseToPL(p1,pts,cp)) return o->ShowMark(CurrGO= this, MRK_GODRAW);
			}
		break;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_RMU:
		if(!(type & 0x10000) && parent && rl && parent->Command(CMD_DROP_OBJECT, rl, o)){
			rl = 0L;
			return true;
			}
		return false;
	case CMD_INIT:
		if(rl) return rl->PropertyDlg();
		break;
	case CMD_DROP_OBJECT:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_REGLINE && !rl) {
			rl = (RegLine *)tmpl;
			rl->parent = this;
			return true;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_SDELLIPSE;
		if(rl) rl->Command(cmd, tmpl, o);
		return true;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_BOUNDS:
		if(tmpl) {
			if(rl) rl->Command(cmd, tmpl, o);
			memcpy(&lim, tmpl, sizeof(fRECT));
			}
		return true;
	case CMD_DELOBJ:
		if(tmpl && tmpl == (void*)rl) {
			Undo.ValInt(parent, &type, 0L);
			type |= 0x10000;
			if(parent) parent->Command(CMD_REDRAW, 0L, o);
			return true;
			}
		break;
	case CMD_AUTOSCALE:
		if(nPoints < 2) return false;
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(lim.Xmin, lim.Ymin);
			((Plot*)parent)->CheckBounds(lim.Xmax, lim.Ymax);
			return true;
			}
		break;
		}
	return false;
}

void
SDellipse::Recalc(lfPOINT *values, long n)
{
	if(val) free(val);
	if(val = (lfPOINT*)malloc(n * sizeof(lfPOINT)))
		memcpy(val, values, (nPoints = n)*sizeof(lfPOINT));
	if(rl) rl->Recalc(values, n);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Error bars are simple graphic objects
ErrorBar::ErrorBar(GraphObj *par, DataObj *d, double x, double y, double err, int which,
	int xc, int xr, int yc, int yr, int ec, int er):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;
	ferr = err;
	type = which;
	Id = GO_ERRBAR;
	data = d;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || ec >= 0 || er >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*3)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = ec;	ssRef[2].y = er;
			cssRef = 3;
			}
		}
	Command(CMD_AUTOSCALE, 0L, 0L);
}

ErrorBar::ErrorBar(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	type = ERRBAR_VSYM;
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

ErrorBar::~ErrorBar()
{
	if(ssRef) free(ssRef);
	ssRef = 0L;
}

bool
ErrorBar::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_ERRBAR: 
		SizeBar = value;
		return true;
	case SIZE_ERRBAR_LINE:
		ErrLine.width = value;
		return true;
		}
	return false;
}

bool
ErrorBar::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_ERROR_LINE:
		ErrLine.color = col;
		return true;
		}
	return false;
}

void
ErrorBar::DoPlot(anyOutput *target)
{
	int ie;

	switch (type & 0x0ff) {
	case ERRBAR_VSYM:
	case ERRBAR_VUP:
	case ERRBAR_VDOWN:
		ie = target->un2ix(SizeBar/2.0f);
		break;
	default:
		ie = target->un2iy(SizeBar/2.0f);
		break;
		}
	target->SetLine(&ErrLine);
	switch(type) {
	case ERRBAR_VSYM:
		ebpts[0].x = ebpts[1].x = target->fx2ix(fPos.fx);
		ebpts[0].y = target->fy2iy(fPos.fy-ferr);
		ebpts[4].y = ebpts[5].y = ebpts[1].y = target->fy2iy(fPos.fy+ferr);
		if(ebpts[1].y != ebpts[0].y) target->oSolidLine(ebpts);
		ebpts[4].x = ebpts[2].x = ebpts[0].x - ie;
		ebpts[5].x = ebpts[3].x = ebpts[1].x + ie+1;
		ebpts[2].y = ebpts[3].y = ebpts[0].y;
		if(ebpts[3].x > ebpts[2].x) {
			target->oSolidLine(ebpts+2);
			target->oSolidLine(ebpts+4);
			}
		rDims.left =  ebpts[2].x;		rDims.right = ebpts[3].x;
		rDims.top = ebpts[0].y;			rDims.bottom = ebpts[1].y;
		break;
	case ERRBAR_VUP:
	case ERRBAR_VDOWN:
		ebpts[0].x = ebpts[1].x = target->fx2ix(fPos.fx);
		ebpts[0].y = target->fy2iy(fPos.fy);
		ebpts[2].y = ebpts[3].y = ebpts[1].y = 
			target->fy2iy(type == ERRBAR_VUP ? fPos.fy+ferr : fPos.fy-ferr);
		if(ebpts[1].y != ebpts[0].y) target->oSolidLine(ebpts);
		ebpts[2].x = ebpts[0].x - ie;
		ebpts[3].x = ebpts[1].x + ie+1;
		if(ebpts[3].x > ebpts[2].x) target->oSolidLine(ebpts+2);
		rDims.left =  ebpts[2].x;		rDims.right = ebpts[3].x;
		rDims.top = ebpts[0].y;			rDims.bottom = ebpts[1].y;
		break;
	case ERRBAR_HSYM:
		ebpts[2].x = ebpts[3].x = ebpts[0].x = target->fx2ix(fPos.fx-ferr);
		ebpts[4].x = ebpts[5].x = ebpts[1].x = target->fx2ix(fPos.fx+ferr);
		ebpts[0].y = ebpts[1].y = target->fy2iy(fPos.fy);
		if(ebpts[1].x != ebpts[0].x) target->oSolidLine(ebpts);
		ebpts[2].y = ebpts[4].y = ebpts[0].y - ie;
		ebpts[3].y = ebpts[5].y = ebpts[1].y + ie+1;
		if(ebpts[3].y >ebpts[2].y) {
			target->oSolidLine(ebpts+2);
			target->oSolidLine(ebpts+4);
			}
		rDims.left =  ebpts[0].x;		rDims.right = ebpts[1].x;
		rDims.top = ebpts[2].y;			rDims.bottom = ebpts[3].y;
		break;
	case ERRBAR_HLEFT:
	case ERRBAR_HRIGHT:
		ebpts[0].x = target->fx2ix(fPos.fx);
		ebpts[0].y = ebpts[1].y = target->fy2iy(fPos.fy);
		ebpts[2].x = ebpts[3].x = ebpts[1].x = 
			target->fx2ix(type == ERRBAR_HRIGHT ? fPos.fx+ferr : fPos.fx-ferr);
		if(ebpts[1].x != ebpts[0].x) target->oSolidLine(ebpts);
		ebpts[2].y = ebpts[0].y - ie;
		ebpts[3].y = ebpts[1].y + ie+1;
		if(ebpts[3].y > ebpts[2].y) target->oSolidLine(ebpts+2);
		rDims.left =  ebpts[0].x;		rDims.right = ebpts[1].x;
		rDims.top = ebpts[2].y;			rDims.bottom = ebpts[3].y;
		break;
		}
	if(rDims.left > rDims.right) Swap(rDims.left, rDims.right);
	if(rDims.top > rDims.bottom) Swap(rDims.top, rDims.bottom);
	IncrementMinMaxRect(&rDims, 3);
}

void
ErrorBar::DoMark(anyOutput *o, bool mark)
{
	LineDEF OldLine;

	memcpy(&OldLine, &ErrLine, sizeof(LineDEF));
	if(mark) {
		ErrLine.color = 0x00000000L;
		ErrLine.width = 1.2f;
		DoPlot(o);
		ErrLine.width = 0.4f;
		ErrLine.color = OldLine.color ^ 0x00ffffffL;
		DoPlot(o);
		}
	else {
		ErrLine.color = 0x00ffffffL;		//DEBUG: assume white background
		ErrLine.width = 1.2f;
		DoPlot(o);
		ErrLine.width = OldLine.width;
		ErrLine.color = OldLine.color;
		DoPlot(o);
		}
	memcpy(&ErrLine, &OldLine, sizeof(LineDEF));
	o->UpdateRect(&rDims, false);
}

bool
ErrorBar::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	bool bFound;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		bFound = false;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(&rDims, mev->x, mev->y) || CurrGO) return false; 
			switch (type) {
			case ERRBAR_HSYM:
			case ERRBAR_HLEFT:
			case ERRBAR_HRIGHT:
				if(mev->y >= (ebpts[0].y-2) && mev->y <= (ebpts[1].y+2)) bFound = true;
				else if(mev->x >= (ebpts[2].x-2) && mev->x <= (ebpts[2].x+2)) bFound = true;
				else if(type == ERRBAR_HSYM && mev->x >= (ebpts[4].x-2) &&
					mev->x <= (ebpts[4].x + 2)) bFound = true;
				break;
			case ERRBAR_VSYM:
			case ERRBAR_VUP:
			case ERRBAR_VDOWN:
				if(mev->x >= (ebpts[0].x-2) && mev->x <= (ebpts[1].x+2)) bFound = true;
				else if(mev->y >= (ebpts[2].y-2) && mev->y <= (ebpts[2].y+2)) bFound = true;
				else if(type == ERRBAR_VSYM && mev->y >= (ebpts[4].y-2) &&
					mev->y <= (ebpts[4].y + 2)) bFound = true;
				break;
				}
			if(bFound) o->ShowMark(this, MRK_GODRAW);
			}
	case CMD_SET_DATAOBJ:
		Id = GO_ERRBAR;
		data = (DataObj *) tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &ferr);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			switch(type) {
			case ERRBAR_VSYM:	
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy+ferr);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy-ferr);		break;
			case ERRBAR_VUP:	
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy+ferr);		break;
			case ERRBAR_VDOWN:
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy-ferr);		break;
			case ERRBAR_HSYM:
				((Plot*)parent)->CheckBounds(fPos.fx+ferr, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx-ferr, fPos.fy);		break;
			case ERRBAR_HLEFT:
				((Plot*)parent)->CheckBounds(fPos.fx-ferr, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);				break;
			case ERRBAR_HRIGHT:
				((Plot*)parent)->CheckBounds(fPos.fx+ferr, fPos.fy);
				((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);				break;
				}
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// arrows to data points or with absolute coordinates
Arrow::Arrow(GraphObj * par, DataObj *d, lfPOINT fp1, lfPOINT fp2, int which,
	int xc1, int xr1, int yc1, int yr1, int xc2, int xr2, int yc2, int yr2):
	GraphObj(par, d)
{
	double dx, dy;

	FileIO(INIT_VARS);
	memcpy(&pos1, &fp1, sizeof(lfPOINT));
	memcpy(&pos2, &fp2, sizeof(lfPOINT));
	type = which;
	if(type & ARROW_UNITS) {
		dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
		pos1.fx -= dx;	pos1.fy -= dy;			pos2.fx -= dx;	pos2.fy -= dy;
		}
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || xc2 >= 0 || xr2 >= 0 || 
		yc2 >= 0 || yr2 >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*4)) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = xc2;	ssRef[2].y = xr2;
			ssRef[3].x = yc2;	ssRef[3].y = yr2;
			cssRef = 4;
			}
		}
	bModified = false;
}

Arrow::Arrow(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

Arrow::~Arrow()
{
	Command(CMD_FLUSH, 0L, 0L);
	if(bModified) Undo.InvalidGO(this);
}

double
Arrow::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:		return pos1.fx;
	case SIZE_XPOS+1:	return pos2.fx;
	case SIZE_YPOS:		return pos1.fy;
	case SIZE_YPOS+1:	return pos2.fy;
	case SIZE_GRECT_LEFT:	case SIZE_GRECT_TOP:
	case SIZE_GRECT_RIGHT:	case SIZE_GRECT_BOTTOM:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
Arrow::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_ARROW_LINE:		LineDef.width = value;		return true;
	case SIZE_ARROW_CAPWIDTH:	cw = value;					return true;
	case SIZE_ARROW_CAPLENGTH:	cl = value;					return true;
	case SIZE_XPOS:				pos1.fx = value;			return true;
	case SIZE_XPOS+1:			pos2.fx = value;			return true;
	case SIZE_YPOS:				pos1.fy = value;			return true;
	case SIZE_YPOS+1:			pos2.fy = value;			return true;
		}
	return false;
}

bool
Arrow::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_ARROW:
		LineDef.color = col;
		return true;
		}
	return false;
}

void
Arrow::DoPlot(anyOutput *o)
{
	double si, csi, tmp, fix1, fiy1, fix2, fiy2, dx, dy;

	if(!o || !parent) return;
	if(type & ARROW_UNITS) {
		dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
		fix1 = o->co2fix(pos1.fx+dx);	fix2 = o->co2fix(pos2.fx+dx);
		fiy1 = o->co2fiy(pos1.fy+dy);	fiy2 = o->co2fiy(pos2.fy+dy);
		}
	else {
		fix1 = o->fx2fix(pos1.fx);		fix2 = o->fx2fix(pos2.fx);
		fiy1 = o->fy2fiy(pos1.fy);		fiy2 = o->fy2fiy(pos2.fy);
		}
	if(fix1 == fix2 && fiy1 == fiy2) return;	//zero length
	//draw arrow line
	pts[0].x = iround(fix1);		pts[1].x = iround(fix2);
	pts[0].y = iround(fiy1);		pts[1].y = iround(fiy2);
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	//calculate sine and cosine for cap
	si = fiy1-fiy2;
	tmp = fix2 - fix1;
	si = si/sqrt(si*si + tmp*tmp);
	csi = fix2-fix1;
	tmp = fiy2 - fiy1;
	csi = csi/sqrt(csi*csi + tmp*tmp);
	//draw cap
	pts[2].x = pts[1].x - o->un2ix(csi*cl + si*cw/2.0);
	pts[2].y = pts[1].y + o->un2iy(si*cl - csi*cw/2.0);
	pts[3].x = pts[1].x;		pts[3].y = pts[1].y;
	pts[4].x = pts[1].x - o->un2ix(csi*cl - si*cw/2.0);
	pts[4].y = pts[1].y + o->un2iy(si*cl + csi*cw/2.0);
	switch(type & 0xff) {
	case ARROW_NOCAP:
		pts[2].x = pts[3].x = pts[4].x = pts[1].x;
		pts[2].y = pts[3].y = pts[4].y = pts[1].y;
		break;
		}
	UpdateMinMaxRect(&rDims, pts[2].x, pts[2].y);
	UpdateMinMaxRect(&rDims, pts[4].x, pts[4].y);
	IncrementMinMaxRect(&rDims, 3);
	if(this == CurrGO) o->ShowMark(this, MRK_GODRAW);
	else Redraw(o);
}

void
Arrow::DoMark(anyOutput *o, bool mark)
{
	LineDEF OldLine;

	if(type & ARROW_UNITS) {
		if(!dh1) dh1 = new dragHandle(this, DH_12);
		if(!dh2) dh2 = new dragHandle(this, DH_22);
		}
	else {
		if (dh1) DeleteGO(dh1);		if (dh2) DeleteGO(dh2);
		dh1 = dh2 = 0L;
		}
	memcpy(&OldLine, &LineDef, sizeof(LineDEF));
	if(mark) {
		LineDef.color = 0x00000000L;
		LineDef.width = OldLine.width *3.0;
		Redraw(o);
		LineDef.width = OldLine.width;
		LineDef.color = OldLine.color ^ 0x00ffffffL;
		Redraw(o);
		if(dh1) dh1->DoPlot(o);		if(dh2) dh2->DoPlot(o);
		}
	else if(parent){
		LineDef.color = 0x00ffffffL;
		LineDef.width = OldLine.width *3.0;
		Redraw(o);
		LineDef.width = OldLine.width;
		LineDef.color = OldLine.color;
		parent->DoPlot(o);
		}
	memcpy(&LineDef, &OldLine, sizeof(LineDEF));
	o->UpdateRect(&rDims, false);
}

bool
Arrow::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_SAVEPOS:
		bModified = true;
		Undo.SaveLFP(this, &pos1, 0L);
		Undo.SaveLFP(this, &pos2, UNDO_CONTINUE);
		return true;
	case CMD_FLUSH:
		if (dh1) DeleteGO(dh1);			dh1 = 0L;
		if (dh2) DeleteGO(dh2);			dh2 = 0L;
		if(ssRef) free(ssRef);			ssRef = 0L;
		if(name) free(name);			name = 0L;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
		if(!CurrGO && ObjThere(mev->x, mev->y)){
			o->ShowMark(this, MRK_GODRAW);
			return true;
			}
		}
		break;
	case CMD_ARROW_ORG:
		memcpy(&pos1, tmpl, sizeof(lfPOINT));
		if(ssRef && cssRef >3) 
			ssRef[0].x = ssRef[0].y = ssRef[1].x = ssRef[1].y = -1;
		return true;
	case CMD_ARROW_TYPE:
		if(tmpl) {
			type &= ~0xff;		type |= (*((int*)tmpl));
			return true;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_ARROW;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >3 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &pos1.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &pos1.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &pos2.fx);
			data->GetValue(ssRef[3].y, ssRef[3].x, &pos2.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(pos1.fx, pos1.fy);
			((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy);
			return true;
			}
		break;
	case CMD_MRK_DIRTY:			//from Undo ?
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MOVE:
		bModified = true;
	case CMD_UNDO_MOVE:
		if(type & ARROW_UNITS) {
			if(cmd == CMD_MOVE) Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
			pos1.fx += ((lfPOINT*)tmpl)[0].fx;	pos1.fy += ((lfPOINT*)tmpl)[0].fy;
			pos2.fx += ((lfPOINT*)tmpl)[0].fx;	pos2.fy += ((lfPOINT*)tmpl)[0].fy;
			if(o){
				o->StartPage();		parent->DoPlot(o);		o->EndPage();
				}
			return true;
			}
		break;
		}
	return false;
}

void * 
Arrow::ObjThere(int x, int y)
{
	if(!IsInRect(&rDims, x, y)) return 0L;
	if(IsCloseToLine(&pts[0], &pts[1], x, y) ||
		IsCloseToLine(&pts[2], &pts[3], x, y) ||
		IsCloseToLine(&pts[3], &pts[4], x, y)){
		if(dh1 && dh1->ObjThere(x, y)) return dh1;
		if(dh2 && dh2->ObjThere(x, y)) return dh2;
		return this;
		}
	return 0L;
}

void
Arrow::Track(POINT *p, anyOutput *o)
{
	POINT *tpts;
	RECT old_rc;
	int i;

	if(o && (tpts = (POINT*)malloc(6*sizeof(POINT)))){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		o->UpdateRect(&rDims, false);
		for(i = 0; i < 5; i++) {
			tpts[i].x = pts[i].x+p->x;
			tpts[i].y = pts[i].y+p->y;
			UpdateMinMaxRect(&rDims, tpts[i].x, tpts[i].y);
			}
		switch(type & 0xff) {
		case ARROW_LINE:
			o->ShowLine(tpts+2, 3, LineDef.color);
		case ARROW_NOCAP:
			o->ShowLine(tpts, 2, LineDef.color);
			break;
		case ARROW_TRIANGLE:
			tpts[5].x = tpts[2].x;		tpts[5].y = tpts[2].y;
			o->ShowLine(tpts+2, 4, LineDef.color);
			tpts[1].x = (tpts[2].x + tpts[4].x)>>1;
			tpts[1].y = (tpts[2].y + tpts[4].y)>>1;
			o->ShowLine(tpts, 2, LineDef.color);
			break;
			}
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		free(tpts);
		}
}

void
Arrow::Redraw(anyOutput *o)
{
	FillDEF FillCap;

	o->SetLine(&LineDef);
	o->oSolidLine(pts);
	switch(type & 0xff) {
	case ARROW_NOCAP:
		break;
	case ARROW_LINE:
		o->oSolidLine(pts+2);
		o->oSolidLine(pts+3);
		break;
	case ARROW_TRIANGLE:
		FillCap.type = FILL_NONE;
		FillCap.color = LineDef.color;
		FillCap.scale = 1.0f;
		FillCap.hatch = 0L;
		o->SetFill(&FillCap);
		o->oPolygon(pts+2, 3);
		break;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// universal boxes
Box::Box(GraphObj * par, DataObj *d, lfPOINT fp1, lfPOINT fp2, int which,
	int xc1, int xr1, int yc1, int yr1, int xc2, int xr2,
		int yc2, int yr2):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	memcpy(&pos1, &fp1, sizeof(lfPOINT));
	memcpy(&pos2, &fp2, sizeof(lfPOINT));
	type = which;
	Id = GO_BOX;
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || xc2 >= 0 || xr2 >= 0 || 
		yc2 >= 0 || yr2 >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*4)) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = xc2;	ssRef[2].y = xr2;
			ssRef[3].x = yc2;	ssRef[3].y = yr2;
			cssRef = 4;
			}
		}
}

Box::Box(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Box::~Box()
{
	Command(CMD_FLUSH, 0L, 0L);
}

double
Box::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:
		return (pos1.fx + pos2.fx)/2.0;
	case SIZE_YPOS:
		return (pos1.fy + pos2.fy)/2.0;
		}
	return 1.0;
}

bool
Box::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_BOX_LINE: 
		Outline.width = value;
		return true;
	case SIZE_BOX:
		size = value;
		return true;
		}
	return false;
}

bool
Box::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_BOX_LINE:
		Outline.color = col;
		return true;
		}
	return false;
}

void
Box::DoPlot(anyOutput *o)
{
	double si, csi, tmp, fix1, fiy1, fix2, fiy2, fsize, dx, dy;

	if(!parent || !o || size <= 0.001) return;
	o->SetLine(&Outline);
	o->SetFill(&Fill);
	//calculate coordinates
	fix1 = o->fx2fix(pos1.fx);	fix2 = o->fx2fix(pos2.fx);
	fiy1 = o->fy2fiy(pos1.fy);	fiy2 = o->fy2fiy(pos2.fy);
	//calculate sine and cosine
	si = fiy1-fiy2;
	tmp = fix2 - fix1;
	si = si/sqrt(si*si + tmp*tmp);
	csi = fix2-fix1;
	tmp = fiy2 - fiy1;
	csi = csi/sqrt(csi*csi + tmp*tmp);
	if(type & BAR_WIDTHDATA) {			//use e.g. for density distribution
		dx = si * size;					dy = csi * size;
		pts[0].x = o->fx2ix(pos1.fx + dx);	pts[1].x = o->fx2ix(pos2.fx + dx);
		pts[2].x = o->fx2ix(pos2.fx - dx);	pts[3].x = o->fx2ix(pos1.fx - dx);
		pts[0].y = o->fy2iy(pos1.fy + dy);	pts[1].y = o->fy2iy(pos2.fy + dy);
		pts[2].y = o->fy2iy(pos2.fy - dy);	pts[3].y = o->fy2iy(pos1.fy - dy);
		}
	else if(type & BAR_RELWIDTH) {
		if(!parent) return;
		fsize = parent->GetSize(pos1.fy == pos2.fy ? SIZE_BOXMINY : SIZE_BOXMINX);
		fsize = fsize * size /200.0;
		dx = si * fsize;					dy = csi * fsize;
		pts[0].x = o->fx2ix(pos1.fx + dx);	pts[1].x = o->fx2ix(pos2.fx + dx);
		pts[2].x = o->fx2ix(pos2.fx - dx);	pts[3].x = o->fx2ix(pos1.fx - dx);
		pts[0].y = o->fy2iy(pos1.fy + dy);	pts[1].y = o->fy2iy(pos2.fy + dy);
		pts[2].y = o->fy2iy(pos2.fy - dy);	pts[3].y = o->fy2iy(pos1.fy - dy);
		}
	else {
		dx = o->un2fix(si*size/2.0);		dy = o->un2fiy(csi*size/2.0);
		pts[0].x = iround(fix1 + dx);	pts[1].x = iround(fix2 + dx);
		pts[2].x = iround(fix2 - dx);	pts[3].x = iround(fix1 - dx);
		pts[0].y = iround(fiy1 + dy);	pts[1].y = iround(fiy2 + dy);
		pts[2].y = iround(fiy2 - dy);	pts[3].y = iround(fiy1 - dy);
		}
	pts[4].x = pts[0].x;		pts[4].y = pts[0].y;	//close polygon
	if(pts[0].x == pts[1].x){
		o->oRectangle(pts[3].x, pts[3].y, pts[1].x, pts[1].y, name);
		}
	else {
		o->oPolygon(pts, 5);
		}
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[2].x, pts[2].y);
	UpdateMinMaxRect(&rDims, pts[1].x, pts[1].y);
	UpdateMinMaxRect(&rDims, pts[3].x, pts[3].y);
}

void
Box::DoMark(anyOutput *o, bool mark)
{
	InvertPolygon(pts, 5, &Outline, &Fill, &rDims, o, mark);
}

bool
Box::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);		ssRef = 0L;
		if(name)free(name);			name = 0L;
		return true;
	case CMD_LEGEND:
		if(!tmpl || ((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		((Legend*)tmpl)->HasFill(&Outline, &Fill);
		break;
	case CMD_MRK_DIRTY:				case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				if(pts[0].x == pts[1].x || pts[0].y == pts[1].y) {
					o->ShowMark(&rDims, MRK_INVERT);
					CurrGO = this;
					return true;
					}
				else {
					p.x = mev->x;		p.y = mev->y;
					if(IsInPolygon(&p, pts, 5)) {
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
					}
				}
			break;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_BOX;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >3 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &pos1.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &pos1.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &pos2.fx);
			data->GetValue(ssRef[3].y, ssRef[3].x, &pos2.fy);
			return true;
			}
		return false;
	case CMD_BOX_TYPE:
		if(tmpl)type = *((int*)tmpl);
		return true;
	case CMD_BOX_FILL:
		if(tmpl) {
			memcpy(&Fill, tmpl, sizeof(FillDEF));
			if(Fill.hatch) memcpy(&Hatchline, Fill.hatch, sizeof(LineDEF));
			Fill.hatch = &Hatchline;
			return true;
			}
		break;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(pos1.fx, pos1.fy);
			((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// whisker 
Whisker::Whisker(GraphObj *par, DataObj *d, lfPOINT fp1, lfPOINT fp2, int which,
	int xc1, int xr1, int yc1, int yr1, int xc2, int xr2,
	int yc2, int yr2):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	memcpy(&pos1, &fp1, sizeof(lfPOINT));
	memcpy(&pos2, &fp2, sizeof(lfPOINT));
	type = which;
	Id = GO_WHISKER;
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || xc2 >= 0 || xr2 >= 0 || 
		yc2 >= 0 || yr2 >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*4)) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = xc2;	ssRef[2].y = xr2;
			ssRef[3].x = yc2;	ssRef[3].y = yr2;
			cssRef = 4;
			}
		}
}

Whisker::Whisker(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

bool
Whisker::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_WHISKER: 
		size = value;
		return true;
	case SIZE_WHISKER_LINE:
		LineDef.width = value;
		return true;
		}
	return false;
}

bool
Whisker::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_WHISKER:
		LineDef.color = col;
		return true;
		}
	return false;
}

void
Whisker::DoPlot(anyOutput *o)
{
	double si, csi, tmp, fix1, fiy1, fix2, fiy2, dx, dy;
	int i;

	fix1 = o->fx2fix(pos1.fx);	fix2 = o->fx2fix(pos2.fx);
	fiy1 = o->fy2fiy(pos1.fy);	fiy2 = o->fy2fiy(pos2.fy);
	if(fix1 == fix2 && fiy1 == fiy2) return;	//zero length
	pts[2].x = iround(fix1);		pts[3].x = iround(fix2);
	pts[2].y = iround(fiy1);		pts[3].y = iround(fiy2);
	//calculate sine and cosine
	si = fiy1-fiy2;
	tmp = fix2 - fix1;
	si = si/sqrt(si*si + tmp*tmp);
	csi = fix2-fix1;
	tmp = fiy2 - fiy1;
	csi = csi/sqrt(csi*csi + tmp*tmp);
	dx = o->un2fix(si*size/2.0);
	dy = o->un2fiy(csi*size/2.0);
	//calc cap
	pts[0].x = iround(fix1 - dx);	pts[4].x = iround(fix2 - dx);
	pts[0].y = iround(fiy1 - dy);	pts[4].y = iround(fiy2 - dy);
	pts[1].x = iround(fix1 + dx);	pts[5].x = iround(fix2 + dx);
	pts[1].y = iround(fiy1 + dy);	pts[5].y = iround(fiy2 + dy);
	//draw whisker
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	UpdateMinMaxRect(&rDims, pts[4].x, pts[4].y);
	UpdateMinMaxRect(&rDims, pts[5].x, pts[5].y);
	IncrementMinMaxRect(&rDims, 3+(o->un2ix(LineDef.width)<<1));
	o->SetLine(&LineDef);
	switch(type & 0x0f) {
	case 2:
		pts[4].x = pts[3].x;	pts[4].y = pts[3].y;
		pts[1].x = pts[2].x;	pts[1].y = pts[2].y;
	case 3:
		if((type & 0x0f) == 3){
			pts[5].x = pts[3].x;	pts[5].y = pts[3].y;
			pts[0].x = pts[2].x;	pts[0].y = pts[2].y;
			}
	case 0:
		for(i = 0; i < 5; i+=2) o->oSolidLine(pts+i);
		break;
	case 1:
		pts[4].x = pts[5].x = pts[3].x;		pts[4].y = pts[5].y = pts[3].y;
		pts[1].x = pts[0].x = pts[2].x;		pts[1].y = pts[0].y = pts[2].y;
		o->oSolidLine(pts+2);
		break;
		}
}

void
Whisker::DoMark(anyOutput *o, bool mark)
{
	LineDEF OldLine;
	int i;

	memcpy(&OldLine, &LineDef, sizeof(LineDEF));
	if(mark) {
		LineDef.color = 0x00000000L;
		LineDef.width = 1.2f;
		o->SetLine(&LineDef);
		for(i = 0; i < 5; i+=2) o->oSolidLine(pts+i);
		LineDef.width = 0.4f;
		LineDef.color = OldLine.color ^ 0x00ffffffL;
		o->SetLine(&LineDef);
		for(i = 0; i < 5; i+=2) o->oSolidLine(pts+i);
		}
	else {
		LineDef.color = 0x00ffffffL;		//DEBUG: assume white background
		LineDef.width = 1.2f;
		o->SetLine(&LineDef);
		for(i = 0; i < 5; i+=2) o->oSolidLine(pts+i);
		LineDef.width = OldLine.width;
		LineDef.color = OldLine.color;
		o->SetLine(&LineDef);
		for(i = 0; i < 5; i+=2) o->oSolidLine(pts+i);
		}
	memcpy(&LineDef, &OldLine, sizeof(LineDEF));
	o->UpdateRect(&rDims, false);
}

bool
Whisker::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				if(IsCloseToLine(&pts[2], &pts[3], mev->x, mev->y) ||
					IsCloseToLine(&pts[0], &pts[1], mev->x, mev->y) ||
					IsCloseToLine(&pts[4], &pts[5], mev->x, mev->y)) {
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
				}
			break;
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_WHISKER;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >3 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &pos1.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &pos1.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &pos2.fx);
			data->GetValue(ssRef[3].y, ssRef[3].x, &pos2.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(pos1.fx, pos1.fy);
			((Plot*)parent)->CheckBounds(pos2.fx, pos2.fy);
			return true;
			}
		break;
	case CMD_WHISKER_STYLE:
		if(tmpl) type = *((int*)tmpl);
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// drop line 
DropLine::DropLine(GraphObj *par, DataObj *d, double x, double y, int which, int xc, 
	int xr, int yc, int yr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;
	fPos.fy = y;
	type = which;
	Id = GO_DROPLINE;
	if(xc >= 0 && xr >= 0 && yc >= 0 && yr >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*2)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			cssRef = 2;
			}
		}
	bModified = false;
}

DropLine::DropLine(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

DropLine::~DropLine()
{
	if(bModified) Undo.InvalidGO(this);
	if(ssRef) free(ssRef);
	ssRef = 0L;
}

void
DropLine::DoPlot(anyOutput *o)
{
	int tmp;

	o->RLP.fp = 0.0f;		//reset line pattern start
	if(parent) {
		pts[0].x = pts[1].x = pts[2].x = pts[3].x = o->fx2ix(fPos.fx);
		pts[0].y = pts[1].y = pts[2].y = pts[3].y = o->fy2iy(fPos.fy);
		if(type & DL_LEFT) {
			tmp = o->fx2ix(parent->GetSize(SIZE_BOUNDS_LEFT));
			if(tmp < pts[0].x) pts[0].x = tmp;
			if(tmp > pts[1].x) pts[1].x = tmp;
			}
		if(type & DL_RIGHT) {
			tmp = o->fx2ix(parent->GetSize(SIZE_BOUNDS_RIGHT));
			if(tmp < pts[0].x) pts[0].x = tmp;
			if(tmp > pts[1].x) pts[1].x = tmp;
			}
		if(type & DL_YAXIS) {
			tmp = iround(parent->GetSize(SIZE_YAXISX));
			if(tmp < pts[0].x) pts[0].x = tmp;
			if(tmp > pts[1].x) pts[1].x = tmp;
			}
		if(type & DL_TOP) {
			tmp = o->fy2iy(parent->GetSize(SIZE_BOUNDS_TOP));
			if(tmp < pts[2].y) pts[2].y = tmp;
			if(tmp > pts[3].y) pts[3].y = tmp;
			}
		if(type & DL_BOTTOM) {
			tmp = o->fy2iy(parent->GetSize(SIZE_BOUNDS_BOTTOM));
			if(tmp < pts[2].y) pts[2].y = tmp;
			if(tmp > pts[3].y) pts[3].y = tmp;
			}
		if(type & DL_XAXIS) {
			tmp = iround(parent->GetSize(SIZE_XAXISY));
			if(tmp < pts[2].y) pts[2].y = tmp;
			if(tmp > pts[3].y) pts[3].y = tmp;
			}
		SetMinMaxRect(&rDims, pts[0].x, pts[2].y, pts[1].x, pts[3].y);
		IncrementMinMaxRect(&rDims, 3);
		o->SetLine(&LineDef);
		o->oPolyline(pts, 2);
		o->oPolyline(pts+2, 2);
		}
}

void
DropLine::DoMark(anyOutput *o, bool mark)
{

	InvertLine(pts, 2, &LineDef, 0L, o, mark);
	InvertLine(pts+2, 2, &LineDef, &rDims, o, mark);
}

bool
DropLine::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				if(IsCloseToLine(&pts[0], &pts[1], mev->x, mev->y) ||
					IsCloseToLine(&pts[2], &pts[3], mev->x, mev->y)) {
					o->ShowMark(this, MRK_GODRAW);
					return true;
					}
				}
			break;
			}
		return false;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_DL_LINE:
		if(tmpl) memcpy(&LineDef, tmpl, sizeof(LineDEF));
		return true;
	case CMD_DL_TYPE:
		if(tmpl)type = *((int*)tmpl);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_DROPLINE;
		data = (DataObj *)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define a spherical scanline used for clipping spheres
sph_scanline::sph_scanline(POINT3D *center, int radius, bool bVert):GraphObj(0, 0)
{
	Id = GO_SPHSCANL;
	rad = radius >= 0 ? radius : -radius;
	memcpy(&p1, center, sizeof(POINT3D));	memcpy(&p2, center, sizeof(POINT3D));
	memcpy(&cent, center, sizeof(POINT3D));
	if(vert = bVert) {
		p1.y -= rad;		p2.y += rad;
		}
	else {
		p1.x -= rad;		p2.x += rad;
		}
	if(p1.x < 1) p1.x = 1;			if(p1.y < 1) p1.y = 1;
	if(p2.x < p1.x) p2.x = p1.x;	if(p2.y < p1.y) p2.y = p1.y;
	bValid1 = bValid2 = true;
}

bool
sph_scanline::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch(cmd) {
	case CMD_ADDTOLINE:
		if(bValid1 && tmpl){
			memcpy(&p2, tmpl, sizeof(POINT3D));
			bValid2 = true;
			return true;
			}
		break;
	case CMD_STARTLINE:
		if(!bValid1 && tmpl){
			memcpy(&p1, tmpl, sizeof(POINT3D));
			bValid1 = true;
			}
		break;
		return true;
		}
	return false;
}

void
sph_scanline::DoClip(GraphObj *co)
{
	POINT3D *pla;
	int np, i;

	if(!bValid1 || !bValid2) return;
	switch(co->Id){
	case GO_SPHERE:
		bValid1 = bValid2 = false;
		clip_sphline_sphere(this, &p1, &p2, &cent, rad, iround(co->GetSize(SIZE_RADIUS1)), 
			iround(co->GetSize(SIZE_XPOS)), iround(co->GetSize(SIZE_YPOS)), 
			iround(co->GetSize(SIZE_ZPOS)));
		break;
	case GO_PLANE:
		for(i=0; ((plane*)co)->GetPolygon(&pla, &np, i); i++) {
			bValid1 = bValid2 = false;
			clip_sphline_plane(this, &p1, &p2, &cent, rad, pla, np, ((plane*)co)->GetVec());
			}
		break;
		}
}
	
bool
sph_scanline::GetPoint(POINT *p, int sel)
{
	if(!bValid1 || !bValid2) {
		p->x = p->y = 0;
		return false;
		}
	switch(sel) {
	case 1:
		p->x = p1.x;	p->y = p1.y;
		return true;
	case 2:
		p->x = p2.x;	p->y = p2.y;
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Sphere: a symbol in three dimensional space
Sphere::Sphere(GraphObj *par, DataObj *d, int sel, double x, double y, double z, 
	double r, int xc, int xr, int yc, int yr, int zc, int zr, int rc, int rr)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_SPHERE;
	fPos.fx = x;	fPos.fy = y;	fPos.fz = z;	size = r;
	if(xc >=0 || xr >=0 || yc >=0 || yr >=0 || zc >=0 || zr >=0 || rc >=0 || rr >=0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*4)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			ssRef[3].x = rc;	ssRef[3].y = rr;
			cssRef = 4;
			}
		}
	type = sel;
	bModified = false;
}

Sphere::Sphere(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

Sphere::~Sphere()
{
	if(bModified) Undo.InvalidGO(this);
	Command(CMD_FLUSH, 0L, 0L);
}

double
Sphere::GetSize(int select)
{
	switch(select) {
	case SIZE_MIN_X:		return fip.fx - (double)rx;
	case SIZE_MAX_X:		return fip.fx + (double)rx;
	case SIZE_MIN_Y:		return fip.fy - (double)ry;
	case SIZE_MAX_Y:		return fip.fy + (double)ry;
	case SIZE_MIN_Z:		return fip.fz - (double)rx;
	case SIZE_MAX_Z:		return fip.fz + (double)rx;
	case SIZE_XPOS:			return fip.fx;
	case SIZE_YPOS:			return fip.fy;
	case SIZE_ZPOS:			return fip.fz;
	case SIZE_RADIUS1:	case SIZE_RADIUS2:	return (double)rx;
	case SIZE_XCENT:		return fPos.fx;
	case SIZE_YCENT:		return fPos.fy;
	case SIZE_ZCENT:		return fPos.fz;
	case SIZE_DRADIUS:		return size;
		}
	return 0.0;
}

bool
Sphere::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_SYMBOL:		size = value;			break;
	case SIZE_SYM_LINE:		Line.width = value;		break;
		}
	return true;
}

bool
Sphere::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_SYM_LINE:		Line.color = col;		break;
	case COL_SYM_FILL:		Fill.color = col;		break;
		}
	return true;
}

void
Sphere::DoPlot(anyOutput *o)
{
	int i;

	if(size <= 0.001 || !o) return;
	if(!o->fvec2ivec(&fPos, &fip)) return;
	bDrawDone = false;
	if(scl){
		for(i = 0; i < nscl; i++) if(scl[i]) delete(scl[i]);
		free(scl);
		scl = 0L;	nscl = 0;
		}
	ix = iround(fip.fx);			iy = iround(fip.fy);
	switch (type) {
	case 1:
		rx = ry = iround(size * 0.5 * o->ddx);		break;
	case 2:
		rx = ry = iround(size * 0.5 * o->ddy);		break;
	case 3:
		rx = ry = iround(size * 0.5 * o->ddz);		break;
	default:
		rx = o->un2ix(size/2.0);	ry = o->un2iy(size/2.0);
		type = 0;
		break;
		}
	rDims.left = ix - rx;			rDims.right = ix + rx;
	rDims.top = iy - ry;			rDims.bottom = iy + ry;
	if(o->VPscale > 1.5 && (
		(rDims.right < defs.clipRC.left) || (rDims.left > defs.clipRC.right) ||
		(rDims.bottom < defs.clipRC.top) || (rDims.top > defs.clipRC.bottom))){
		bDrawDone = true;		return;
		}
	if(parent && parent->Command(CMD_SET_GO3D, this, o)) return;
	Command(CMD_REDRAW, 0L, o);
}

bool
Sphere::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);		ssRef = 0L;
		if(name) free(name);		name = 0L;
		if(scl){
			for(i = 0; i < nscl; i++) if(scl[i]) delete(scl[i]);
			free(scl);
			scl = 0L;	nscl = 0;
			}
		return true;
	case CMD_SYM_FILL:
		if(tmpl) memcpy(&Fill, tmpl, sizeof(FillDEF));
		return true;
	case CMD_MRK_DIRTY:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_SPHERE;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		if(bDrawDone) return false;
		bDrawDone = true;
		if(scl) DrawPG(o, 0);
		else {
			o->SetLine(&Line);				o->SetFill(&Fill);
			if(Fill.type & FILL_LIGHT3D) o->oSphere(ix, iy, rx, 0L, 0, 0L);
			else o->oCircle(ix-rx, iy-ry, ix+rx+1, iy+ry+1, name);
			}
		rx--;ry--;			//smaller marking rectangle
		rDims.left = ix-rx-1;				rDims.right = ix+rx+1;
		rDims.top = iy-ry-1;				rDims.bottom = iy+ry+1;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				o->ShowMark(&rDims, MRK_INVERT);
				CurrGO = this;
				return true;
				}
			break;
			}
		break;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >1 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos.fz);
			if(cssRef >3) data->GetValue(ssRef[3].y, ssRef[3].x, &size);
			return true;
			}
		return false;
	case CMD_CLIP:
		if(co = (GraphObj*)tmpl){
			switch(co->Id) {
			case GO_PLANE:
			case GO_SPHERE:
				DoClip(co);
				break;
				}
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz);
			return true;
			}
		break;
		}
	return false;
}

void
Sphere::DoClip(GraphObj *co)
{
	RECT cliprc;
	double d, d1;
	POINT3D cscl;
	int x, y, q, di, de, lim, cx, cy;
	int i;

	if(co && co->Id == GO_SPHERE) {
		if(co->GetSize(SIZE_ZPOS) > fip.fz) return;
		d1 = (d = co->GetSize(SIZE_XPOS) - fip.fx) * d;
		d1 += (d = co->GetSize(SIZE_YPOS) - fip.fy) * d;
		d1 = sqrt(d1 + (d = co->GetSize(SIZE_ZPOS) - fip.fz) * d);
		if(d1 >= (co->GetSize(SIZE_RADIUS1) + rx))return;
		}
	else {
		cliprc.left = iround(co->GetSize(SIZE_MIN_X));
		cliprc.right = iround(co->GetSize(SIZE_MAX_X));
		cliprc.top = iround(co->GetSize(SIZE_MIN_Y));
		cliprc.bottom = iround(co->GetSize(SIZE_MAX_Y));
		if(!OverlapRect(&rDims, &cliprc))return;
		}
	//use a list of horizontal scanlines created by a circular Bresenham's algorithm
	//Ref: C. Montani, R. Scopigno (1990) "Speres-To-Voxel Conversion", in:
	//   Graphic Gems (A.S. Glassner ed.) Academic Press, Inc.; 
	//   ISBN 0-12-288165-5 
	if(!scl && (scl = (sph_scanline**)calloc(rx*7+2, sizeof(sph_scanline*)))) {
		cscl.z = iround(fip.fz);			nscl = 0;
		for(q = 0; q < 2; q++) {
			x = lim = 0;	y = rx;		di = 2*(1-rx);
			while( y >= lim) {
				if(di < 0) {
					de = 2*di + 2*y -1;
					if(de > 0) {
						x++;	y--;	di += (2*x -2*y +2);
						}
					else {
						x++;	di += (2*x +1);
						}
					}
				else {
					de = 2*di -2*x -1;
					if(de > 0) {
						y--;	di += (-2*y +1);
						}
					else {
						x++;	y--;	di += (2*x -2*y +2);
						}
					}
				switch(q) {
				case 0:
					cy = rx - y;			cx = x;
					break;
				case 1:
					cy = rx + x;			cx = y; 
					break;
					}
				cscl.y = iround(fip.fy);		cscl.x = iround(fip.fx);
				cscl.y = cscl.y - rx + cy;
				if(cx > 1) scl[nscl++] = new sph_scanline(&cscl, cx, false);
				}
			}
		}
	if(!scl) return;
	//do clip for every scanline
	for(i = 0; i < nscl; i++) if(scl[i]) scl[i]->DoClip(co);
}

void
Sphere::DrawPG(anyOutput *o, int start)
{
	POINT *pts, np;
	long cp = 0;
	int i = start, step = 1;

	if(!o || !scl ||!nscl) return;
	if((pts = (POINT*)calloc(nscl*2, sizeof(POINT)))) {
		do {
			if(scl[i]) {
				if(step > 0) scl[i]->GetPoint(&np, 1);
				else scl[i]->GetPoint(&np, 2);
				if(np.x && np.y){
					AddToPolygon(&cp, pts, &np);
					}
				else if(cp){
					if(step > 0) DrawPG(o, i);			//split sphere
					step = -1;
					}
				}
			if(i == nscl && step > 0) step = -1;
			else i += step;
			}while(i > start);
		}
	o->SetLine(&Line);				o->SetFill(&Fill);
	if(cp) o->oSphere(ix, iy, rx, pts, cp, name);
	free(pts);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// plane: utility object to draw a flat plane in 3D space
plane::plane(GraphObj *par, DataObj *d, fPOINT3D *pts, int nPts, LineDEF *line, 
	  FillDEF *fill):GraphObj(par, d)
{
	int i;
	long area;
	double vlength, v1[3], v2[3], vp[3], area1;
	bool v_valid = false;

	nli = n_ipts = n_lines = 0;
	nldata = 0L;  ldata = 0L;	co = 0L;	lines = 0L;		PlaneVec = 0L;
	Id = GO_PLANE;	totalArea = 0;
	memcpy(&Line, line, sizeof(LineDEF));
	memcpy(&Fill, fill, sizeof(FillDEF));
	rDims.left = rDims.right = rDims.top = rDims.bottom = 0;
	if(nPts > 2 && (ldata =(POINT3D**)calloc(1, sizeof(POINT3D*))) &&
		(ldata[0] = (POINT3D *)calloc(nPts+1, sizeof(POINT3D))) &&
		(nldata = (int*)calloc(1, sizeof(int)))	&&
		(ipts = (POINT*)calloc(nPts, sizeof(POINT)))){
		for(i = 0; i < nPts; i++) {
			ipts[i].x = ldata[0][i].x = iround(pts[i].fx);
			ipts[i].y = ldata[0][i].y = iround(pts[i].fy);
			ldata[0][i].z = iround(pts[i].fz);
			}
		nldata[0] = nPts;		nli = 1;	n_ipts = nPts;
		xBounds.fx = xBounds.fy = pts[0].fx;	yBounds.fx = yBounds.fy = pts[0].fy;
		zBounds.fx = zBounds.fy = pts[0].fz;
		rDims.left = rDims.right = ipts[0].x;	rDims.top = rDims.bottom = ipts[0].y;
		for(i = 1; i < nPts; i++){
			UpdateMinMaxRect(&rDims, ipts[i].x, ipts[i].y);
			if(pts[i].fx < xBounds.fx) xBounds.fx = pts[i].fx;
			if(pts[i].fx > xBounds.fy) xBounds.fy = pts[i].fx;
			if(pts[i].fy < yBounds.fx) yBounds.fx = pts[i].fy;
			if(pts[i].fy > yBounds.fy) yBounds.fy = pts[i].fy;
			if(pts[i].fz < zBounds.fx) zBounds.fx = pts[i].fz;
			if(pts[i].fz > zBounds.fy) zBounds.fy = pts[i].fz;
			}
		//test if plane vertical
		area1 = (xBounds.fx - xBounds.fy) * (yBounds.fx - yBounds.fy);
		for(area = 0, i = 1; i < nPts; i++) {
			area += (ipts[i].x - ipts[i-1].x) * ((ipts[i].y + ipts[i-1].y)>>1);
			}
		totalArea= area = abs(area);
		area1 = area ? fabs(area1/area) : 101.0;
		if(area < 100 && area1 > 100.0) {			//its small or vertical !
			if(lines=(line_segment**)calloc(nPts, sizeof(line_segment*))){
				for(i = 1; i < nPts; i++) {
					lines[i-1] = new line_segment(par, d, line, &ldata[0][i-1], &ldata[0][i]);
					}
				n_lines = nPts-1;
				}
			}
		else {					//for a visible plane get vector perpendicular to plane
			for (i = 1; i < (nPts-1); i++) {
				v1[0] = pts[i-1].fx - pts[i].fx;	v1[1] = pts[i-1].fy - pts[i].fy;
				v1[2] = pts[i-1].fz - pts[i].fz;	v2[0] = pts[i+1].fx - pts[i].fx;
				v2[1] = pts[i+1].fy - pts[i].fy;	v2[2] = pts[i+1].fz - pts[i].fz;
				vp[0] = v1[1]*v2[2] - v1[2]*v2[1];	vp[1] = v1[2]*v2[0] - v1[0]*v2[2];
				vp[2] = v1[0]*v2[1] - v1[1]*v2[0];
				vlength = sqrt(vp[0]*vp[0]+vp[1]*vp[1]+vp[2]*vp[2]);
				if(v_valid = (vlength > 1.0)) break;
				}
			if(v_valid && (PlaneVec = (double*)calloc(7, sizeof(double)))) {
				PlaneVec[0] =vp[0];		PlaneVec[1] =vp[1];		PlaneVec[2] =vp[2];
				PlaneVec[3] =pts[i].fx;	PlaneVec[4] =pts[i].fy;	PlaneVec[5] =pts[i].fz;
				PlaneVec[6] =vlength;
				}
			}
		}
}

plane::~plane() 
{
	int i;

	if(ldata) {
		for(i = 0; i < nli; i++) if(ldata[i]) free(ldata[i]);
		free(ldata);	ldata = 0L;		nli = 0;
		}
	if(lines) {
		for(i = 0; i < n_lines; i++) if(lines[i]) delete(lines[i]);
		free(lines);	lines = 0L;		n_lines = 0;
		}
	if(nldata) free(nldata);		nldata = 0L;
	if(ipts) free(ipts);			ipts = 0L;
	if(PlaneVec) free(PlaneVec);	PlaneVec = 0L;
}

double
plane::GetSize(int select)
{
	switch(select) {
	case SIZE_MIN_X:		return xBounds.fx;
	case SIZE_MAX_X:		return xBounds.fy;
	case SIZE_MIN_Y:		return yBounds.fx;
	case SIZE_MAX_Y:		return yBounds.fy;
	case SIZE_MIN_Z:		return zBounds.fx;
	case SIZE_MAX_Z:		return zBounds.fy;
		}
	return 0.0;
}

void
plane::DoPlot(anyOutput *o)
{
	int i;

	bDrawDone = false;
	if(Fill.type & FILL_LIGHT3D){
		Fill.color = o->VecColor(PlaneVec, Fill.color2, Fill.color);
		Fill.type &= ~FILL_LIGHT3D;
		}
	if(o->VPscale > 1.5 && (
		//ignore objects outside the display ara
		(rDims.right < defs.clipRC.left) || (rDims.left > defs.clipRC.right) ||
		(rDims.bottom < defs.clipRC.top) || (rDims.top > defs.clipRC.bottom))){
		bDrawDone = true;		return;
		}
	if(lines) {
		//draw line segments for vertical plane
		for(i = 0; i < n_lines; i++) if(lines[i]) lines[i]->DoPlot(o);
		bDrawDone = true;		return;
		}
	if(parent && parent->Command(CMD_SET_GO3D, this, o)) return;
	Command(CMD_REDRAW, 0L, o);
}

void
plane::DoMark(anyOutput *o, bool mark)
{
	FillDEF tmpfill;
	LineDEF tmpline;

	memcpy(&tmpfill, &Fill, sizeof(FillDEF));
	memcpy(&tmpline, &Line, sizeof(LineDEF));
	if(mark){
		tmpfill.color ^= 0x00ffffffL;		tmpline.color ^= 0x00ffffffL;
		}
	o->SetLine(&tmpline);					o->SetFill(&tmpfill);
	o->oPolygon(ipts, n_ipts);
}

bool 
plane::Command(int cmd, void *tmpl, anyOutput *o)
{
	POINT *pt;
	POINT3D *ap;
	int i, j;
	
	switch (cmd) {
	case CMD_REDRAW:
		if(bDrawDone) return false;
		bDrawDone = true;
		if(o && data && nldata){
			o->SetLine(&Line);			o->SetFill(&Fill);
			for(i = 0; i < nli; i++){
				if(nldata[i] > 2 && (pt = (POINT*)malloc(nldata[i]*sizeof(POINT)))){
					for(j = 0; j < nldata[i]; j++) {
						pt[j].x = ldata[i][j].x;	pt[j].y = ldata[i][j].y;
						}
					o->oPolygon(pt, nldata[i]);
					free(pt);
					}
				}
			}
		return true;
	case CMD_STARTLINE:
		if(ap = (POINT3D*)tmpl) {
			if(ldata && nldata && nli) {
				i = nli-1;			j = nldata[i]-1;
				if(ldata[i][j].x == ap->x && ldata[i][j].y == ap->y && ldata[i][j].z == ap->z){
					return true;
					}
				if(IsValidPG(ldata[i], nldata[i])) {
					//close previous polygon first
					if(ldata[i][0].x != ldata[i][j].x || ldata[i][0].y != ldata[i][j].y ||
						ldata[i][0].z != ldata[i][j].z){
						j++;
						ldata[i][j].x = ldata[i][0].x;	ldata[i][j].y = ldata[i][0].y;
						ldata[i][j].z = ldata[i][0].z;	nldata[i]++;
						}
					ldata = (POINT3D**)realloc(ldata, sizeof(POINT3D*) * (nli+1));
					ldata[nli] = (POINT3D*)malloc(sizeof(POINT3D)*2);
					nldata = (int*)realloc(nldata, sizeof(int) * (nli+1));
					}
				else {					//drop incomplete or invalid polygon
					nli--;
					}
				}
			else {
				ldata = (POINT3D**)calloc(1, sizeof(POINT3D*));
				ldata[nli = 0] = (POINT3D*)malloc(sizeof(POINT3D));
				nldata = (int*)calloc(1, sizeof(int));
				}
			if(ldata && nldata) {
				ldata[nli][0].x = ap->x;	ldata[nli][0].y = ap->y;
				ldata[nli][0].z = ap->z;	nldata[nli++] = 1;
				return true;
				}
			}
		break;
	case CMD_ADDTOLINE:
		if((ap = (POINT3D*)tmpl) && ldata && nldata && nli) {
			i= nli-1;	j=nldata[i]-1;
			if(ldata[i][j].x == ap->x && ldata[i][j].y == ap->y && ldata[i][j].z == ap->z){
				//nothing to add
				return j>0 ? true : false;
				}
			ldata[i] = (POINT3D*)realloc(ldata[i], ((j=nldata[i])+2) * sizeof(POINT3D));
			ldata[i][j].x = ap->x;			ldata[i][j].y = ap->y;
			ldata[i][j].z = ap->z;			nldata[i]++;
			return true;
			}
	case CMD_CLIP:
		if(co = (GraphObj*)tmpl){
			switch(co->Id) {
			case GO_PLANE:
				//Clip only planes which are drawn later
				if(GetSize(SIZE_MIN_Z) < co->GetSize(SIZE_MIN_Z)) return false;
				if(nli){
					DoClip(co);
					if(nli && ldata) {
						i = nli-1;			j = nldata[i]-1;
						//is last part valid ?
						if(j < 2) {
							free(ldata[i]);		ldata[i] = 0L;
							nldata[i] = 0;
							nli--;
							}
						//close last polygon
						else if(ldata[i][0].x != ldata[i][j].x ||
							ldata[i][0].y != ldata[i][j].y ||
							ldata[i][0].z != ldata[i][j].z){
							j++;
							ldata[i][j].x = ldata[i][0].x;
							ldata[i][j].y = ldata[i][0].y;
							ldata[i][j].z = ldata[i][0].z;
							nldata[i]++;
							}
						}
					}
				else xBounds.fx=xBounds.fy=yBounds.fx=yBounds.fy=zBounds.fx=zBounds.fy=0.0;
				break;
				}
			}
		break;
		}
	return false;
}

void * 
plane::ObjThere(int x, int y)
{
	POINT p1;

	if(bDrawDone && IsInRect(&rDims, p1.x = x, p1.y = y) &&
		(IsInPolygon(&p1, ipts, n_ipts) || IsCloseToPL(p1, ipts, n_ipts))) return this;
	return 0L;
}

bool
plane::GetPolygon(POINT3D **pla, int *npt, int n)
{
	if(n < nli && ldata && ldata[n]) {
		*pla = ldata[n];	*npt = nldata[n];
		return true;
		}
	return false;
}

void
plane::DoClip(GraphObj *co)
{
	RECT cliprc;
	int o_nli, *o_nldata = 0L;
	POINT3D **o_ldata = 0L;
	POINT3D *tpg;
	int i, j, tnpt;
	bool is_valid = false;

	cliprc.left = iround(co->GetSize(SIZE_MIN_X));
	cliprc.right = iround(co->GetSize(SIZE_MAX_X));
	cliprc.top = iround(co->GetSize(SIZE_MIN_Y));
	cliprc.bottom = iround(co->GetSize(SIZE_MAX_Y));
	if(OverlapRect(&rDims, &cliprc) && co != this) {
		o_nli = nli;		nli = 0;
		o_nldata = nldata;	nldata = 0L;
		o_ldata = ldata;	ldata = 0L;
		switch(co->Id) {
		case GO_PLANE:
			//clip all parts of this plane with all from another plane
			for(i = 0; ((plane*)co)->GetPolygon(&tpg, &tnpt, i); i++){
				for(j = 0; j < o_nli; j++) {
					if(o_nldata[j] >2) clip_plane_plane(this, o_ldata[j], o_nldata[j], PlaneVec, 
						tpg, tnpt, ((plane*)co)->GetVec(), ipts, n_ipts );
					}
				if(nli) is_valid = true;
				if(o_ldata) {
					for(j = 0; j < o_nli; j++) if(o_ldata[j]) free(o_ldata[j]);
					free(o_ldata);
					}
				if(o_nldata) free(o_nldata);
				o_nli = nli;		nli = 0;
				o_nldata = nldata;	nldata = 0L;
				o_ldata = ldata;	ldata = 0L;
				if(!o_nli) return;				//plane is completly hidden
				}
			if(is_valid || i==0){
				nli = o_nli;		o_nli = 0;
				nldata = o_nldata;	o_nldata = 0L;
				ldata = o_ldata;	o_ldata = 0L;
				}
			break;
		default:
			nli = o_nli;		o_nli = 0;
			nldata = o_nldata;	o_nldata = 0L;
			ldata = o_ldata;	o_ldata = 0L;
			break;
			}
		//check shape and recalc some values
		if(is_valid && nli) {
			xBounds.fx = xBounds.fy = ldata[0][0].x;	yBounds.fx = yBounds.fy = ldata[0][0].y;
			zBounds.fx = zBounds.fy = ldata[0][0].z;
			rDims.left = rDims.right = ldata[0][0].x;	rDims.top = rDims.bottom = ldata[0][0].y;
			for(i = 0; i < nli; i++) if(nldata[i] > 2){
				if(ldata[i][0].x != ldata[i][nldata[i]-1].x || ldata[i][0].y != ldata[i][nldata[i]-1].y
					|| ldata[i][0].z != ldata[i][nldata[i]-1].z) {
					ldata[i][nldata[i]].x = ldata[i][0].x;	ldata[i][nldata[i]].y = ldata[i][0].y;
					ldata[i][nldata[i]].z = ldata[i][0].z;	nldata[i]++;
					}
				for(j = 0; j < (nldata[i]-1); j++) {
					UpdateMinMaxRect(&rDims, ldata[i][j].x, ldata[i][j].y);
					if(ldata[i][j].x < xBounds.fx) xBounds.fx = ldata[i][j].x;
					if(ldata[i][j].x > xBounds.fy) xBounds.fy = ldata[i][j].x;
					if(ldata[i][j].y < yBounds.fx) yBounds.fx = ldata[i][j].y;
					if(ldata[i][j].y > yBounds.fy) yBounds.fy = ldata[i][j].y;
					if(ldata[i][j].z < zBounds.fx) zBounds.fx = ldata[i][j].z;
					if(ldata[i][j].z > zBounds.fy) zBounds.fy = ldata[i][j].z;
					}
				}
			}
		}
	if(o_ldata) {
		for(i = 0; i < o_nli; i++) if(o_ldata[i]) free(o_ldata[i]);
		free(o_ldata);
		}
	if(o_nldata) free(o_nldata);
}

bool
plane::IsValidPG(POINT3D *pg, int npg)
{
	int i;
	long area;

	//a polygon must have more than Points
	if(npg < 3) return false;
	//check for a reasonable size
	for(area = 0, i = 1; i < npg; i++) {
		area += (pg[i].x - pg[i-1].x) * ((pg[i].y + pg[i-1].y)>>1);
		}
	area += (pg[0].x - pg[i-1].x) * ((pg[0].y + pg[i-1].y)>>1);
	area = abs(area);
	if(area < 20) return false;
	if(totalArea/area > 100) return false;
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a simple plane in three dimensional space
Plane3D::Plane3D(GraphObj *par, DataObj *da, fPOINT3D *pt, long npt)
	:GraphObj(par, da) 
{
	FileIO(INIT_VARS);
	Id = GO_PLANE3D;
	dt = (fPOINT3D*) memdup(pt, sizeof(fPOINT3D)*npt, 0L);
	ndt = npt;
}

Plane3D::Plane3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Plane3D::~Plane3D()
{
	if(dt) free(dt);	if(pts) free(pts);
	dt = 0L;		ndt = 0L;
	if(ipl) delete (ipl);	ipl = 0L;
}

bool
Plane3D::SetSize(int select, double value)
{
	int i;

	if((select & 0xfff) >= SIZE_XPOS && (select & 0xfff) <=  SIZE_XPOS_LAST){
		if((i = select-SIZE_XPOS) >=0 && i <= 200 && i < (int)ndt){
			dt[i].fx = value;			return true;
			}
		}
	else if((select & 0xfff) >= SIZE_YPOS && (select & 0xfff) <= SIZE_YPOS_LAST){
		if((i = select-SIZE_YPOS) >=0 && i <= 200 && i < (int)ndt){
			dt[i].fy = value;			return true;
			}
		}
	else if((select & 0xfff) >= SIZE_ZPOS && (select & 0xfff) <= SIZE_ZPOS_LAST){
		if((i = select-SIZE_ZPOS) >=0 && i <= 200 && i < (int)ndt){
			dt[i].fz = value;			return true;
			}
		}
	else switch(select) {
	case SIZE_SYM_LINE:
		Line.width = value;
		}
	return false;
}

bool
Plane3D::SetColor(int select, DWORD col)
{
	switch(select) {
	case COL_POLYLINE:		Line.color = col;		return true;
	case COL_POLYGON:		Fill.color = col;		return true;
		}
	return false;
}

void
Plane3D::DoPlot(anyOutput *o)
{
	int i;

	if(ipl) delete ipl;		ipl = 0L;
	if(pts) free(pts);		pts = 0L;
	o->ActualSize(&rDims);
	rDims.left = rDims.right;	rDims.top = rDims.bottom;
	rDims.right = rDims.bottom = 0;
	if((pts = (fPOINT3D*)malloc(sizeof(fPOINT3D)*ndt))){
		for(i = 0; i < ndt; i++) {
			if(!o->fvec2ivec(&dt[i], &pts[i])){
				free(pts);	pts = 0L; return;
				}
			UpdateMinMaxRect(&rDims, iround(pts[i].fx), iround(pts[i].fy));
			}
		if(ipl = new plane(this, data, pts, i, &Line, &Fill))ipl->DoPlot(o);
		}
	IncrementMinMaxRect(&rDims, o->un2ix(Line.width)+1);
}

void
Plane3D::DoMark(anyOutput *o, bool mark)
{
	if(mark){
		if(pts && ipl)ipl->DoMark(o, mark);
		o->UpdateRect(&rDims, false);
		}
	else if(parent) parent->Command(CMD_REDRAW, 0L, o);
}

bool
Plane3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_SET_DATAOBJ:
		Id = GO_PLANE3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_MRK_DIRTY:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SYM_FILL:
		if(tmpl) memcpy(&Fill, tmpl, sizeof(FillDEF));
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				if(ipl && ipl->ObjThere(mev->x, mev->y)){
					o->ShowMark(CurrGO=this, MRK_GODRAW);
					return true;
					}
				}
			break;
			}
		break;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_PG_FILL:
		if(tmpl) {
			memcpy((void*)&Fill, tmpl, sizeof(FillDEF));
			Fill.hatch = 0L;
			}
		break;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH && dt) {
			for(i = 0; i < ndt; i++)
				((Plot*)parent)->CheckBounds3D(dt[i].fx, dt[i].fy, dt[i].fz);
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Brick: a bar in three dimensional space
Brick::Brick(GraphObj *par, DataObj *da, double x, double y, double z, 
		double d, double w, double h, DWORD flg, int xc, int xr, int yc,
		int yr, int zc, int zr, int dc, int dr, int wc, int wr, int hc,
		int hr):GraphObj(par, da)
{
	FileIO(INIT_VARS);
	Id = GO_BRICK;
	fPos.fx = x;	fPos.fy = y;	fPos.fz = z;
	depth = d;		width = w;		height = h;
	flags = flg;
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || zc >= 0 || zr >= 0 ||
		dc >= 0 || dr >= 0 || wc >= 0 || wr >= 0 || hc >= 0 || hr >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*6)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			ssRef[3].x = dc;	ssRef[3].y = dr;
			ssRef[4].x = wc;	ssRef[4].y = wr;
			ssRef[5].x = hc;	ssRef[5].y = hr;
			cssRef = 6;
			}
		}
	bModified = false;
}

Brick::Brick(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}
	
Brick::~Brick()
{
	int i;

	if(faces) {
		for(i = 0; i < 6; i++) if(faces[i]) delete(faces[i]);
		free(faces);
		}
	faces = 0L;
	if(mo) DelBitmapClass(mo);		mo = 0L;
	Command(CMD_FLUSH, 0L, 0L);
	if(bModified) Undo.InvalidGO(this);
}

bool
Brick::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_BAR_LINE:		Line.width = value;		return true;
	case SIZE_BAR_BASE:		fPos.fy = value;		return true;
	case SIZE_BAR:			width = value;			return true;
	case SIZE_BAR_DEPTH:	depth = value;			return true;
		}
	return false;
}

bool
Brick::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_BAR_LINE:		Line.color = col;		return true;
	case COL_BAR_FILL:		Fill.color = col;		return true;
		}
	return false;
}

void
Brick::DoPlot(anyOutput *o)
{
	fPOINT3D cpt[8], fip1, fip2, tmp, itmp, *pg;
	plane *npl;
	double dtmp;
	int i;

	if(mo) DelBitmapClass(mo);		mo = 0L;
	if(!faces || !o || !o->fvec2ivec(&fPos, &fip1)) return;
	if(!(pg = (fPOINT3D *)malloc(5*sizeof(fPOINT3D)))) return;
	for(i = 0; i < 6; i++) {
		if(faces[i]) delete(faces[i]);
		faces[i] = 0L;
		}
	if(flags & 0x800L) {		//height is data
		tmp.fx = fPos.fx;			tmp.fy = height;
		tmp.fz = fPos.fz;			o->fvec2ivec(&tmp, &fip2);
		}
	else {						//height is units
		tmp.fx = tmp.fz = 0.0;		tmp.fy = height;
		o->uvec2ivec(&tmp, &fip2);
		fip2.fx += fip1.fx;			fip2.fy += fip1.fy;
		fip2.fz += fip1.fz;
		}
	//calc output-device coordinates of cubic brick: 8 corners
	tmp.fx = -(width/2.0);			tmp.fz = (depth/2.0);	tmp.fy = 0.0;
	o->uvec2ivec(&tmp, &itmp);
	cpt[0].fx= fip1.fx+itmp.fx;	cpt[0].fy= fip1.fy+itmp.fy;	cpt[0].fz= fip1.fz+itmp.fz;
	cpt[2].fx= fip1.fx-itmp.fx;	cpt[2].fy= fip1.fy-itmp.fy;	cpt[2].fz= fip1.fz-itmp.fz;
	cpt[4].fx= fip2.fx+itmp.fx;	cpt[4].fy= fip2.fy+itmp.fy;	cpt[4].fz= fip2.fz+itmp.fz;
	cpt[6].fx= fip2.fx-itmp.fx;	cpt[6].fy= fip2.fy-itmp.fy;	cpt[6].fz= fip2.fz-itmp.fz;
	tmp.fx = (width/2.0);			tmp.fz = (depth/2.0);	tmp.fy = 0.0;
	o->uvec2ivec(&tmp, &itmp);
	cpt[1].fx= fip1.fx+itmp.fx;	cpt[1].fy= fip1.fy+itmp.fy;	cpt[1].fz= fip1.fz+itmp.fz;
	cpt[3].fx= fip1.fx-itmp.fx;	cpt[3].fy= fip1.fy-itmp.fy;	cpt[3].fz= fip1.fz-itmp.fz;
	cpt[5].fx= fip2.fx+itmp.fx;	cpt[5].fy= fip2.fy+itmp.fy;	cpt[5].fz= fip2.fz+itmp.fz;
	cpt[7].fx= fip2.fx-itmp.fx;	cpt[7].fy= fip2.fy-itmp.fy;	cpt[7].fz= fip2.fz-itmp.fz;
	//set up 6 faces
	pg[0].fx = pg[4].fx = cpt[0].fx;	pg[1].fx = cpt[1].fx;	
	pg[2].fx = cpt[2].fx;				pg[3].fx = cpt[3].fx;
	pg[0].fy = pg[4].fy = cpt[0].fy;	pg[1].fy = cpt[1].fy;	
	pg[2].fy = cpt[2].fy;				pg[3].fy = cpt[3].fy;
	pg[0].fz = pg[4].fz = cpt[0].fz;	pg[1].fz = cpt[1].fz;	
	pg[2].fz = cpt[2].fz;				pg[3].fz = cpt[3].fz;
	faces[0] = new plane(this, data, pg, 5, &Line, &Fill);
	pg[2].fx = cpt[5].fx;				pg[3].fx = cpt[4].fx;
	pg[2].fy = cpt[5].fy;				pg[3].fy = cpt[4].fy;
	pg[2].fz = cpt[5].fz;				pg[3].fz = cpt[4].fz;
	npl = new plane(this, data, pg, 5, &Line, &Fill);
	if(npl->GetSize(SIZE_MAX_Z) > faces[0]->GetSize(SIZE_MAX_Z)) faces[1] = npl;
	else {
		faces[1] = faces[0];		faces[0] = npl;		
		}
	pg[0].fx = pg[4].fx = cpt[2].fx;	pg[1].fx = cpt[6].fx;	
	pg[2].fx = cpt[5].fx;				pg[3].fx = cpt[1].fx;
	pg[0].fy = pg[4].fy = cpt[2].fy;	pg[1].fy = cpt[6].fy;	
	pg[2].fy = cpt[5].fy;				pg[3].fy = cpt[1].fy;
	pg[0].fz = pg[4].fz = cpt[2].fz;	pg[1].fz = cpt[6].fz;	
	pg[2].fz = cpt[5].fz;				pg[3].fz = cpt[1].fz;
	npl = new plane(this, data, pg, 5, &Line, &Fill);
	if((dtmp = npl->GetSize(SIZE_MAX_Z)) > faces[1]->GetSize(SIZE_MAX_Z)) faces[2] = npl;
	else {
		faces[2] = faces[1];
		if(dtmp > faces[0]->GetSize(SIZE_MAX_Z)) faces[1] = npl;
		else {
			faces[1] = faces[0];	faces[0] = npl;
			}
		}
	pg[2].fx = cpt[7].fx;				pg[3].fx = cpt[3].fx;
	pg[2].fy = cpt[7].fy;				pg[3].fy = cpt[3].fy;
	pg[2].fz = cpt[7].fz;				pg[3].fz = cpt[3].fz;
	npl = new plane(this, data, pg, 5, &Line, &Fill);
	dtmp = npl->GetSize(SIZE_MAX_Z);
	for (i = 3; i; i--) {
		if(dtmp >faces[i-1]->GetSize(SIZE_MAX_Z)) {
			faces[i] = npl;			break;
			}
		else faces[i] = faces[i-1];
		}
	if(!i) faces[0] = npl;
	pg[0].fx = pg[4].fx = cpt[4].fx;	pg[1].fx = cpt[7].fx;	
	pg[2].fx = cpt[3].fx;				pg[3].fx = cpt[0].fx;
	pg[0].fy = pg[4].fy = cpt[4].fy;	pg[1].fy = cpt[7].fy;	
	pg[2].fy = cpt[3].fy;				pg[3].fy = cpt[0].fy;
	pg[0].fz = pg[4].fz = cpt[4].fz;	pg[1].fz = cpt[7].fz;	
	pg[2].fz = cpt[3].fz;				pg[3].fz = cpt[0].fz;
	npl = new plane(this, data, pg, 5, &Line, &Fill);
	dtmp = npl->GetSize(SIZE_MAX_Z);
	for (i = 4; i; i--) {
		if(dtmp >faces[i-1]->GetSize(SIZE_MAX_Z)) {
			faces[i] = npl;			break;
			}
		else faces[i] = faces[i-1];
		}
	if(!i) faces[0] = npl;
	pg[2].fx = cpt[6].fx;				pg[3].fx = cpt[5].fx;
	pg[2].fy = cpt[6].fy;				pg[3].fy = cpt[5].fy;
	pg[2].fz = cpt[6].fz;				pg[3].fz = cpt[5].fz;
	npl = new plane(this, data, pg, 5, &Line, &Fill);
	dtmp = npl->GetSize(SIZE_MAX_Z);
	for (i = 5; i; i--) {
		if(dtmp >faces[i-1]->GetSize(SIZE_MAX_Z)) {
			faces[i] = npl;			break;
			}
		else faces[i] = faces[i-1];
		}
	if(!i) faces[0] = npl;
	rDims.left = rDims.right = (int)pg[0].fx;
	rDims.top = rDims.bottom = (int)pg[0].fy;
	for (i= 3; i < 6; i++) if(faces[i]) {
		faces[i]->DoPlot(o);
		UpdateMinMaxRect(&rDims, faces[i]->rDims.left, faces[i]->rDims.top);
		UpdateMinMaxRect(&rDims, faces[i]->rDims.right, faces[i]->rDims.bottom);
		}
	free(pg);
}

void
Brick::DoMark(anyOutput *o, bool mark)
{
	int i;

	if(mark){
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(Line.width));
		mo = GetRectBitmap(&mrc, o);
		if(faces) for(i = 3; i < 6; i++)
			if(faces[i]) faces[i]->DoMark(o, mark);
		o->UpdateRect(&rDims, false);
		}
	else if(mo)	RestoreRectBitmap(&mo, &mrc, o);
}

bool
Brick::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_FLUSH:
		if(ssRef) free(ssRef);		ssRef = 0L;
		if(name) free(name);		name = 0L;
		return true;
	case CMD_BAR_FILL:
		if(tmpl) {
			memcpy(&Fill, tmpl, sizeof(FillDEF));
			Fill.hatch = 0L;
			return true;
			}
		break;
	case CMD_MRK_DIRTY:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_BRICK;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		//Should we ever come here ?
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				if(faces && faces[3] && faces[4] && faces[5] &&
					(faces[3]->ObjThere(mev->x, mev->y) || 
					faces[4]->ObjThere(mev->x, mev->y) ||
					faces[5]->ObjThere(mev->x, mev->y))){
						o->ShowMark(CurrGO=this, MRK_GODRAW);
						return true;
						}
				}
			break;
			}
		break;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef > 5 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos.fz);
			data->GetValue(ssRef[3].y, ssRef[3].x, &depth);
			data->GetValue(ssRef[4].y, ssRef[4].x, &width);
			data->GetValue(ssRef[5].y, ssRef[5].x, &height);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz);
			if(flags & 0x800L) {		//height is data
				((Plot*)parent)->CheckBounds3D(fPos.fx, height, fPos.fz);
				}
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// line_segment: utility object to draw a piece of a polyline in 3D space
line_segment::line_segment(GraphObj *par, DataObj *d, LineDEF *ld, POINT3D *p1, POINT3D *p2)
	:GraphObj(par, d)
{
	double tmp, tmp1, tmp2;

	nli = 0;    nldata = 0L;	  ldata = 0L;	prop = 1.0;		df_go = 0L;
	fmin.fx = fmax.fx = fmin.fy = fmax.fy = fmin.fz = fmax.fz = 0.0;
	ndf_go = 0;
	if(ld) memcpy(&Line, ld, sizeof(LineDEF));
	if(p1 && p2 &&(ldata =(POINT3D**)calloc(1, sizeof(POINT3D*))) &&
		(ldata[0] = (POINT3D*)malloc(2 * sizeof(POINT3D))) &&
		(nldata = (int*)calloc(1, sizeof(int)))){
			if(Line.pattern) {
				tmp1 = (tmp = (p2->x - p1->x)) * tmp;
				tmp2 = (tmp1 += (tmp = (p2->y - p1->y)) * tmp);
				tmp1 += (tmp = (p2->z - p1->z)) * tmp;
				if(tmp1 > 1.0) prop = sqrt(tmp2)/sqrt(tmp1);
				}
			memcpy(&ldata[0][0], p1, sizeof(POINT3D));
			memcpy(&ldata[0][1], p2, sizeof(POINT3D));
			nldata[0] = 2;		nli = 1;
			rDims.left = rDims.right = p1->x;	rDims.top = rDims.bottom = p1->y;
			UpdateMinMaxRect(&rDims, p2->x, p2->y);
			fmin.fx = (double)rDims.left;	fmin.fy = (double)rDims.top;
			fmax.fx = (double)rDims.right;	fmax.fy = (double)rDims.bottom;
			if(p2->z > p1->z) {
				fmin.fz = (double)p1->z;	fmax.fz = (double)p2->z;
				}
			else {
				fmin.fz = (double)p2->z;	fmax.fz = (double)p1->z;
				}
		}
	Id = GO_LINESEG;
}

line_segment::~line_segment()
{
	int i;

	if(ldata) {
		for(i = 0; i < nli; i++) if(ldata[i]) free(ldata[i]);
		free(ldata);
		}
	if(nldata) free(nldata);		nldata = 0L;
}

double
line_segment::GetSize(int select)
{
	switch(select) {
	case SIZE_MIN_Z:
		return fmin.fz;
	case SIZE_MAX_Z:
		return fmax.fz;
		}
	return 0.0;
}

void
line_segment::DoPlot(anyOutput *o)
{
	bDrawDone = false;		co = 0L;
	if(df_go) free(df_go);
	df_go = 0L;				ndf_go = 0;
	if(o->VPscale > 1.5 && (
		(rDims.right < defs.clipRC.left) || (rDims.left > defs.clipRC.right) ||
		(rDims.bottom < defs.clipRC.top) || (rDims.top > defs.clipRC.bottom))){
		bDrawDone = true;		return;
		}
	if(parent && parent->Command(CMD_SET_GO3D, this, o)) return;
	Command(CMD_REDRAW, 0L, o);
}

bool
line_segment::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, j;
	POINT pts[2];
	LineDEF cLine;
	POINT3D *ap;

	switch (cmd) {
	case CMD_DRAW_LATER:
		if(!co) return false;
		if(df_go){
			for(i = 0; i < ndf_go; i++) if(df_go[i] == co) return true;
			if(df_go = (GraphObj**)realloc(df_go, (ndf_go +1) * sizeof(GraphObj*))) {
				df_go[ndf_go++] = co;
				}
			}
		else {
			if(df_go = (GraphObj**)malloc(sizeof(GraphObj*))){
				df_go[0] = co;	ndf_go = 1;
				}
			}
		return true;
	case CMD_REDRAW:
		if(bDrawDone) return false;
		bDrawDone = true;
		if(!nli) return false;
		if(df_go) {
			for(i = 0; i < ndf_go; i++) if(df_go[i]) df_go[i]->Command(cmd, tmpl, o);
			free(df_go);	df_go = 0L;			ndf_go = 0L;
			}
		if(o && ldata && nldata){
			memcpy(&cLine, &Line, sizeof(LineDEF));
			cLine.patlength *= prop;
			o->SetLine(&cLine);
			for(i = 0; i < nli; i++) for(j = 0; j < (nldata[i]-1); j++) {
				pts[0].x = ldata[i][j].x;	pts[0].y = ldata[i][j].y;
				pts[1].x = ldata[i][j+1].x;	pts[1].y = ldata[i][j+1].y;
				if(pts[0].x != pts[1].x || pts[0].y != pts[1].y) o->oPolyline(pts, 2);
				}
			}
		return true;
	case CMD_CLIP:
		if(co = (GraphObj*)tmpl){
			switch(co->Id) {
			case GO_PLANE:
			case GO_SPHERE:
				DoClip(co);
				break;
				}
			}
		return false;
	case CMD_STARTLINE:
		if(tmpl) {
			if(ldata && nldata) {
				ldata = (POINT3D**)realloc(ldata, sizeof(POINT3D*) * (nli+1));
				ldata[nli] = (POINT3D*)malloc(2 * sizeof(POINT3D));
				nldata = (int*)realloc(nldata, sizeof(int)*(nli+1));
				}
			else {
				ldata = (POINT3D**)calloc(1, sizeof(POINT3D*));
				ldata[nli = 0] = (POINT3D*)malloc(2 * sizeof(POINT3D));
				nldata = (int*)calloc(1, sizeof(int));
				}
			if(ldata && nldata) {
				memcpy(&ldata[nli][0], tmpl, sizeof(POINT3D));
				memcpy(&ldata[nli][1], tmpl, sizeof(POINT3D));
				nldata[nli++] = 1;
				return true;
				}
			}
		break;
	case CMD_ADDTOLINE:
		if((ap = (POINT3D*)tmpl) && ldata && nldata && nli && nldata[i =(nli-1)]) {
			j = nldata[i];
			ldata[i] = (POINT3D*)realloc(ldata[i], (nldata[i]+2) * sizeof(POINT3D));
			if(j && ldata[i][j-1].x == ap->x && ldata[i][j-1].y == ap->y &&
				ldata[i][j-1].z == ap->z) return true;
			else {
				j = (nldata[i]++);
				}
			memcpy(&ldata[i][j-1], tmpl, sizeof(POINT3D));
			return true;
			}
		break;
		}
	return false;
}

void *
line_segment::ObjThere(int x, int y)
{
	int i, j;
	POINT pts[2];

	if(ldata && nldata){
		for(i = 0; i < nli; i++) for(j = 0; j < nldata[i]; j +=2) {
			pts[0].x = ldata[i][j].x;	pts[0].y = ldata[i][j].y;
			pts[1].x = ldata[i][j+1].x;	pts[1].y = ldata[i][j+1].y;
			if(IsCloseToLine(&pts[0], &pts[1], x, y))return this;
			}
		}
	return 0L;
}

void
line_segment::DoClip(GraphObj *co)
{
	RECT cliprc;
	int o_nli, *o_nldata = 0L;
	POINT3D **o_ldata = 0L, *pts = 0L, *pla;
	int i, j, k, np, r, cx, cy, cz;
	bool is_valid = false;

	cliprc.left = iround(co->GetSize(SIZE_MIN_X));
	cliprc.right = iround(co->GetSize(SIZE_MAX_X));
	cliprc.top = iround(co->GetSize(SIZE_MIN_Y));
	cliprc.bottom = iround(co->GetSize(SIZE_MAX_Y));
	if(OverlapRect(&rDims, &cliprc)) {
		if(!(pts = (POINT3D*)calloc(2, sizeof(POINT3D))))return;
		o_nli = nli;		nli = 0;
		o_nldata = nldata;	nldata = 0L;
		o_ldata = ldata;	ldata = 0L;
		switch(co->Id) {
		case GO_SPHERE:
			cx = iround(co->GetSize(SIZE_XPOS));
			cy = iround(co->GetSize(SIZE_YPOS));
			cz = iround(co->GetSize(SIZE_ZPOS));
			r = iround(co->GetSize(SIZE_RADIUS1));
			for(i = 0; i < o_nli; i++) for(j = 0; j < (o_nldata[i]-1); j ++) {
				pts[0].x = o_ldata[i][j].x;			pts[0].y = o_ldata[i][j].y;
				pts[0].z = o_ldata[i][j].z;			pts[1].x = o_ldata[i][j+1].x;
				pts[1].y = o_ldata[i][j+1].y;		pts[1].z = o_ldata[i][j+1].z;
				clip_line_sphere(this, &pts, r, cx, cy, cz);
				}
			break;
		case GO_PLANE:
			for(i = 0; ((plane*)co)->GetPolygon(&pla, &np, i); i++){
				for(j = 0; j < o_nli; j++) {
					if(o_nldata[j] >1) for(k = 0; k < (o_nldata[j]-1); k++){
						pts[0].x = o_ldata[j][k].x;			pts[0].y = o_ldata[j][k].y;
						pts[0].z = o_ldata[j][k].z;			pts[1].x = o_ldata[j][k+1].x;
						pts[1].y = o_ldata[j][k+1].y;		pts[1].z = o_ldata[j][k+1].z;
						if(pts[0].x != pts[1].x || pts[0].y != pts[1].y || pts[0].z != pts[1].z)
							clip_line_plane(this, &pts, pla, np, ((plane*)co)->GetVec());
						}
					}
				if(nli) is_valid = true;
				if(o_ldata) {
					for(j = 0; j < o_nli; j++) if(o_ldata[j]) free(o_ldata[j]);
					free(o_ldata);
					}
				if(o_nldata) free(o_nldata);
				o_nli = nli;		nli = 0;
				o_nldata = nldata;	nldata = 0L;
				o_ldata = ldata;	ldata = 0L;
				if(!o_nli) return;				//line is completly hidden
				}
			if(is_valid || i==0){
				nli = o_nli;		o_nli = 0;
				nldata = o_nldata;	o_nldata = 0L;
				ldata = o_ldata;	o_ldata = 0L;
				}
			break;
		default:
			nli = o_nli;		o_nli = 0;
			nldata = o_nldata;	o_nldata = 0L;
			ldata = o_ldata;	o_ldata = 0L;
			break;
			}
		if(pts) free(pts);
		}
	if(o_ldata) {
		for(i = 0; i < o_nli; i++) if(o_ldata[i]) free(o_ldata[i]);
		free(o_ldata);
		}
	if(o_nldata) free(o_nldata);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define a drop line in 3D space
DropLine3D::DropLine3D(GraphObj *par, DataObj *d, fPOINT3D *p1, int xc,
		int xr, int yc, int yr, int zc, int zr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_DROPL3D;
	memcpy(&fPos, p1, sizeof(fPOINT3D));
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || zc >= 0 || zr >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*3)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = zc;	ssRef[2].y = zr;
			cssRef = 3;
			}
		}
	bModified = false;
	type = 0x01;
}

DropLine3D::DropLine3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

DropLine3D::~DropLine3D()
{
	if(bModified) Undo.InvalidGO(this);
	if(mo) DelBitmapClass(mo);		mo = 0L;
	Command(CMD_FLUSH, 0L, 0L);
}

void
DropLine3D::DoPlot(anyOutput *o)
{
	fPOINT3D fip, fp, fp1;
	POINT3D p1, p2;
	int i;

	if(!parent || !o || !o->fvec2ivec(&fPos, &fip)) return;
	if(mo) DelBitmapClass(mo);		mo = 0L;
	for(i = 0; i < 6; i++){
		if(ls[i]) delete(ls[i]);
		ls[i] = 0L;
		}
	p1.x = iround(fip.fx);	p1.y = iround(fip.fy);	p1.z = iround(fip.fz);
	rDims.left = rDims.right = p1.x;	rDims.top = rDims.bottom = p1.y;
	for(i = 0; i < 6; i++) {
		fp.fx = fPos.fx;	fp.fy = fPos.fy;	fp.fz = fPos.fz;
		if(type & (1 << i)){
			switch (i) {
			case 0:	fp.fy = parent->GetSize(SIZE_BOUNDS_YMIN);	break;
			case 1:	fp.fy = parent->GetSize(SIZE_BOUNDS_YMAX);	break;
			case 2:	fp.fz = parent->GetSize(SIZE_BOUNDS_ZMIN);	break;
			case 3:	fp.fz = parent->GetSize(SIZE_BOUNDS_ZMAX);	break;
			case 4:	fp.fx = parent->GetSize(SIZE_BOUNDS_XMIN);	break;
			case 5:	fp.fx = parent->GetSize(SIZE_BOUNDS_XMAX);	break;
				}
			o->fvec2ivec(&fp, &fp1);		p2.x = iround(fp1.fx);
			p2.y = iround(fp1.fy);			p2.z = iround(fp1.fz);
			UpdateMinMaxRect(&rDims, p2.x, p2.y);
			if(ls[i] = new line_segment(this, data, &Line, &p1, &p2)) ls[i]->DoPlot(o);
			mpts[i][0].x = p1.x;	mpts[i][0].y = p1.y;
			mpts[i][1].x = p2.x;	mpts[i][1].y = p2.y;
			}
		}
	IncrementMinMaxRect(&rDims, 5);
}

void
DropLine3D::DoMark(anyOutput *o, bool mark)
{
	int i;

	if(mark) {
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(Line.width));
		mo = GetRectBitmap(&mrc, o);
		for(i = 0; i < 6; i++) {
			if(type & (1 << i)){
				InvertLine(mpts[i], 2, &Line, 0L, o, mark);
				}
			}
		o->UpdateRect(&mrc, false);
		}
	else RestoreRectBitmap(&mo, &mrc, o);
}

bool
DropLine3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		for(i = 0; i < 6; i++){
			if(ls[i]) delete(ls[i]);
			ls[i] = 0L;
			}
		if(ssRef) free(ssRef);		ssRef = 0L;
		if(name) free(name);		name = 0L;
		return true;
	case CMD_DL_TYPE:
		if(tmpl && *((int*)tmpl)) type = *((int*)tmpl);
		return true;
	case CMD_DL_LINE:
		if(tmpl) memcpy(&Line, tmpl, sizeof(LineDEF));
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_DROPL3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued either by Undo (no output given) or
		//  by Plot3D::DoPlot after sorting all objects (output specified)
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				for(i = 0; i < 6; i++) {
					if(ls[i] && ls[i]->ObjThere(mev->x, mev->y)){
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
					}
				}
			break;
			}
		break;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos.fz);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos.fx, fPos.fy, fPos.fz);
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define an arrow in 3D space
Arrow3D::Arrow3D(GraphObj *par, DataObj *d, fPOINT3D *p1, fPOINT3D *p2, int xc1,
		int xr1, int yc1, int yr1, int zc1, int zr1, int xc2, int xr2, int yc2, 
		int yr2, int zc2, int zr2):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_ARROW3D;
	memcpy(&fPos1, p1, sizeof(fPOINT3D));	memcpy(&fPos2, p2, sizeof(fPOINT3D));
	if(xc1 >= 0 || xr1 >= 0 || yc1 >= 0 || yr1 >= 0 || zc1 >= 0 || zr1 >= 0 || 
		xc2 >= 0 || xr2 >= 0 || yc2 >= 0 || yr2 >= 0 || zc2 >= 0 || zr2 >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*6)) {
			ssRef[0].x = xc1;	ssRef[0].y = xr1;
			ssRef[1].x = yc1;	ssRef[1].y = yr1;
			ssRef[2].x = zc1;	ssRef[2].y = zr1;
			ssRef[3].x = xc2;	ssRef[3].y = xr2;
			ssRef[4].x = yc2;	ssRef[4].y = yr2;
			ssRef[5].x = zc2;	ssRef[5].y = zr2;
			cssRef = 6;
			}
		}
	bModified = false;
}

Arrow3D::Arrow3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

Arrow3D::~Arrow3D()
{
	if(bModified) Undo.InvalidGO(this);
	Command(CMD_FLUSH, 0L, 0L);
}

bool
Arrow3D::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_ARROW_LINE:
		Line.width = value;
		return true;
	case SIZE_ARROW_CAPWIDTH:
		cw = value;
		return true;
	case SIZE_ARROW_CAPLENGTH:
		cl = value;
		return true;
		}
	return false;
}

bool
Arrow3D::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_ARROW:
		Line.color = col;
		return true;
		}
	return false;
}

void
Arrow3D::DoPlot(anyOutput *o)
{
	double si, csi, tmp, cwr, clr, d, d1, d2;
	fPOINT3D fip1, fip2, tria[3];
	POINT3D p1, p2, cp1, cp2;
	FillDEF fill;
	int i;

	if(!parent || !o || !o->fvec2ivec(&fPos1, &fip1) ||!o->fvec2ivec(&fPos2, &fip2)) return;
	for(i = 0; i < 3; i++){
		if(ls[i]) delete(ls[i]);
		ls[i] = 0L;
		}
	p1.x = iround(fip1.fx);	p1.y = iround(fip1.fy);	p1.z = iround(fip1.fz);
	p2.x = iround(fip2.fx);	p2.y = iround(fip2.fy);	p2.z = iround(fip2.fz);
	if(p1.x == p2.x && p1.y == p2.y && p1.z == p2.z) return;	//zero length arrow
	rDims.left = rDims.right = p1.x;	rDims.top = rDims.bottom = p1.y;
	UpdateMinMaxRect(&rDims, p2.x, p2.y);
	IncrementMinMaxRect(&rDims, 5);
	if(ls[0] = new line_segment(this, data, &Line, &p1, &p2)) ls[0]->DoPlot(o);
	mpts[0][0].x = p1.x;	mpts[0][0].y = p1.y;	mpts[0][1].x = p2.x;	mpts[0][1].y = p2.y;
	if(p1.x == p2.x && p1.y == p2.y) return;			//zero length in 2D
	if((type & 0xff) == ARROW_NOCAP) return;			//no cap;
	//calculate sine and cosine for cap
	si = fip1.fy-fip2.fy;
	tmp = fip2.fx - fip1.fx;
	si = si/sqrt(si*si + tmp*tmp);
	csi = fip2.fx-fip1.fx;
	tmp = fip2.fy - fip1.fy;
	csi = csi/sqrt(csi*csi + tmp*tmp);
	//cap corners
	d1 = (d = fip2.fx - fip1.fx) * d;
	d1 += ((d = fip2.fy - fip1.fy) * d);
	d2 = d1 + ((d = fip2.fz - fip1.fz) * d);
	d1 = sqrt(d1);	d2 = sqrt(d2);	d = d1/d2;
	cwr = cw;	clr = cl*d;
	cp1.x = p2.x - o->un2ix(csi*clr + si*cwr/2.0);
	cp1.y = p2.y + o->un2iy(si*clr - csi*cwr/2.0);
	cp2.x = p2.x - o->un2ix(csi*clr - si*cwr/2.0);
	cp2.y = p2.y + o->un2iy(si*clr + csi*cwr/2.0);
	cp1.z = cp2.z = p2.z;
	mpts[1][0].x = p2.x;	mpts[1][0].y = p2.y;	mpts[1][1].x = cp1.x;	mpts[1][1].y = cp1.y;
	mpts[2][0].x = p2.x;	mpts[2][0].y = p2.y;	mpts[2][1].x = cp2.x;	mpts[2][1].y = cp2.y;
	if((type & 0xff) == ARROW_LINE) {
		if(ls[1] = new line_segment(this, data, &Line, &p2, &cp1)) ls[1]->DoPlot(o);
		if(ls[2] = new line_segment(this, data, &Line, &p2, &cp2)) ls[2]->DoPlot(o);
		}
	else if((type & 0xff) == ARROW_TRIANGLE) {
		fill.type = FILL_NONE;	fill.color = Line.color;
		fill.scale = 1.0;		fill.hatch = 0L;
		tria[0].fz = tria[1].fz = tria[2].fz = fip2.fz;
		tria[0].fx = cp1.x;		tria[0].fy = cp1.y;
		tria[1].fx = fip2.fx;	tria[1].fy = fip2.fy;
		tria[2].fx = cp2.x;		tria[2].fy = cp2.y;
		if(cap = new plane(this, data, tria, 3, &Line, &fill))cap->DoPlot(o);
	}
}

void
Arrow3D::DoMark(anyOutput *o, bool mark)
{
	int i;

	if(mark) {
		for(i = 0; i < 3; i++) {
			if(ls[i]){
				InvertLine(mpts[i], 2, &Line, 0L, o, mark);
				}
			}
		}
	else if(parent) parent->Command(CMD_REDRAW, 0L, o);
}

bool
Arrow3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		for(i = 0; i < 3; i++){
			if(ls[i]) delete(ls[i]);
			ls[i] = 0L;
			}
		if(cap) delete(cap);	cap = 0L;
		if(ssRef) free(ssRef);		ssRef = 0L;
		if(name) free(name);		name = 0L;
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_ARROW3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_MRK_DIRTY:			//from Undo ?
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !CurrGO) {
				for(i = 0; i < 3; i++) {
					if(ls[i] && ls[i]->ObjThere(mev->x, mev->y)){
						o->ShowMark(this, MRK_GODRAW);
						return true;
						}
					}
				}
			break;
			}
		break;
	case CMD_ARROW_ORG3D:
		if(tmpl) memcpy(&fPos1, tmpl, sizeof(fPOINT3D));
		return true;
	case CMD_ARROW_TYPE:
		if(tmpl) type = *((int*)tmpl);
		return true;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >5 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos2.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos2.fy);
			data->GetValue(ssRef[2].y, ssRef[2].x, &fPos2.fz);
			data->GetValue(ssRef[3].y, ssRef[3].x, &fPos1.fx);
			data->GetValue(ssRef[4].y, ssRef[4].x, &fPos1.fy);
			data->GetValue(ssRef[5].y, ssRef[5].x, &fPos1.fz);
			return true;
			}
		return false;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH) {
			((Plot*)parent)->CheckBounds3D(fPos1.fx, fPos1.fy, fPos1.fz);
			((Plot*)parent)->CheckBounds3D(fPos2.fx, fPos2.fy, fPos2.fz);
			return true;
			}
		break;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a data line in 3D space
Line3D::Line3D(GraphObj *par, DataObj *d, char *rx, char *ry, char *rz)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	if(rx && rx[0]) x_range = strdup(rx);	if(ry && ry[0]) y_range = strdup(ry);
	if(rz && rz[0]) z_range = strdup(rz);
	DoUpdate();
	Id = GO_LINE3D;
	bModified = false;
}

Line3D::Line3D(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

Line3D::~Line3D()
{
	int i;

	if(bModified) Undo.InvalidGO(this);
	if(ls){
		for(i = 0; i < (nPts-1); i++) if(ls[i]) delete(ls[i]);
		free(ls);
		}
	if(pts && npts) free(pts);		pts = 0L;		npts = 0;
	if(x_range) free(x_range);		x_range = 0L;
	if(y_range) free(y_range);		y_range = 0L;
	if(z_range) free(z_range);		z_range = 0L;
	if(values) free(values);		values = 0L;
	if(mo) DelBitmapClass(mo);		mo = 0L;
}

void
Line3D::DoPlot(anyOutput *o)
{
	int i, j;
	fPOINT3D fip;
	POINT3D p1, p2;
	POINT np;


	if(mo) DelBitmapClass(mo);		mo = 0L;
	if(ls) {
		if(pts && npts) free(pts);
		npts = 0;	if(!(pts = (POINT*)calloc(nPts, sizeof(POINT))))return;
		for(i = 0; i< nPts; i++) {
			if(!o->fvec2ivec(&values[i], &fip)) return;
			p2.x = iround(fip.fx);	p2.y = iround(fip.fy);	p2.z = iround(fip.fz);
			np.x = p2.x;			np.y = p2.y;
			AddToPolygon(&npts, pts, &np);
			if(i) {
				UpdateMinMaxRect(&rDims, np.x, np.y);
				j = i-1;
				if(ls[j]) delete(ls[j]);
				if(ls[j] = new line_segment(this, data, &Line, &p1, &p2)) {
					ls[j]->DoPlot(o);
					}
				}
			else {
				rDims.left = rDims.right = p2.x;
				rDims.top = rDims.bottom = p2.y;
				}
			p1.x = p2.x;	p1.y = p2.y;	p1.z = p2.z;
			}
		IncrementMinMaxRect(&rDims, 6);
		}
}

void
Line3D::DoMark(anyOutput *o, bool mark)
{
	if(mark) {
		memcpy(&mrc, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&mrc, 6 + o->un2ix(Line.width));
		mo = GetRectBitmap(&mrc, o);
		InvertLine(pts, npts, &Line, &rDims, o, true);
		}
	else RestoreRectBitmap(&mo, &mrc, o);
}

bool
Line3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;
	int i;

	switch (cmd) {
	case CMD_FLUSH:
		if(name) free(name);		name = 0L;
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_LINE3D;
		data = (DataObj *)tmpl;
		return true;
	case CMD_REDRAW:
		//Note: this command is issued  by Undo (no output given)
		if(!parent) return false;
		if(!o) return parent->Command(cmd, tmpl, o);
		//Should we ever come here ?
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!IsInRect(&rDims, p1.x= mev->x, p1.y=mev->y)|| CurrGO || !o || nPts <2 ||
				!IsCloseToPL(p1, pts, npts))return false;
			o->ShowMark(CurrGO=this, MRK_GODRAW);
			return true;
			}
		return false;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_UPDATE:
		DoUpdate();
	case CMD_AUTOSCALE:
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH){
			if(min.fx == max.fx || min.fy == max.fy){	//z's may be equal !
				min.fx = min.fy = min.fz = HUGE_VAL;
				max.fx = max.fy = max.fz = -HUGE_VAL;
				for(i = 0; i < nPts; i++) {
					if(values[i].fx < min.fx) min.fx = values[i].fx;
					if(values[i].fy < min.fy) min.fy = values[i].fy;
					if(values[i].fz < min.fz) min.fz = values[i].fz;
					if(values[i].fx > max.fx) max.fx = values[i].fx;
					if(values[i].fy > max.fx) max.fy = values[i].fy;
					if(values[i].fz > max.fx) max.fz = values[i].fz;
					}
				}
			((Plot*)parent)->CheckBounds3D(min.fx, min.fy, min.fz);
			((Plot*)parent)->CheckBounds3D(max.fx, max.fy, max.fz);
			return true;
			}
		return false;
	case CMD_SET_GO3D:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
		}
	return false;
}

void
Line3D::DoUpdate()
{
	int n1 = 0, ic = 0, i, j, k, l, m, n;
	double x, y, z;
	AccRange *rX=0L, *rY=0L, *rZ=0L;

	if(!x_range || !y_range || !z_range) return;
	if(values) free(values);	values = 0L;
	if(ls) free(ls);			ls = 0L;
	rX = new AccRange(x_range);
	rY = new AccRange(y_range);
	rZ = new AccRange(z_range);
	min.fx = min.fy = min.fz = HUGE_VAL;	max.fx = max.fy = max.fz = -HUGE_VAL;
	if(rX) n1 = rX->CountItems();
	if(n1 && rY && rZ && (values = (fPOINT3D*)malloc(n1 * sizeof(fPOINT3D)))){
		rX->GetFirst(&i, &j);		rX->GetNext(&i, &j);
		rY->GetFirst(&k, &l);		rY->GetNext(&k, &l);
		rZ->GetFirst(&m, &n);		rZ->GetNext(&m, &n);
		do {
			if(data->GetValue(j, i, &x) && data->GetValue(l, k, &y) && 
				data->GetValue(n, m, &z)){
				values[ic].fx = x;	values[ic].fy = y;	values[ic].fz = z;
				if(x < min.fx) min.fx = x;	if(x > max.fx) max.fx = x;
				if(y < min.fy) min.fy = y;	if(y > max.fy) max.fy = y;
				if(z < min.fz) min.fz = z;	if(z > max.fz) max.fz = z;
				ic++;
				}
			}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l) && rZ->GetNext(&m, &n));
		nPts = ic;
		if(ic > 1) ls = (line_segment **)calloc(ic-1, sizeof(line_segment*));
		}
	if(rX) delete(rX);	if(rY) delete(rY); if(rZ) delete(rZ);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the text label class 
Label::Label(GraphObj *par, DataObj *d, double x, double y, TextDEF *td, DWORD flg,
	 int xc, int xr, int yc, int yr, int tc, int tr):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	fPos.fx = x;		fPos.fy = y;		flags = flg;
	if(parent){
		fDist.fx = parent->GetSize(SIZE_LB_XDIST);
		fDist.fy = parent->GetSize(SIZE_LB_YDIST);
		}
	Id = GO_LABEL;
	if(td){
		memcpy(&TextDef, td, sizeof(TextDEF));
		if(td->text) TextDef.text = strdup(td->text);
		}
	if(xc >= 0 || xr >= 0 || yc >= 0 || yr >= 0 || tc >= 0 || tr >= 0) {
		if(ssRef = (POINT*)malloc(sizeof(POINT)*3)) {
			ssRef[0].x = xc;	ssRef[0].y = xr;
			ssRef[1].x = yc;	ssRef[1].y = yr;
			ssRef[2].x = tc;	ssRef[2].y = tr;
			cssRef = 3;
			}
		}
}

Label::Label(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Label::~Label()
{
	Command(CMD_FLUSH, 0L, 0L);
	if(bModified)Undo.InvalidGO(this);
}

double
Label::GetSize(int select)
{
	switch(select){
	case SIZE_CURSORPOS:
		return (double)CursorPos;
	case SIZE_CURSOR_XMIN:
		return (double) (Cursor.right > Cursor.left ? Cursor.left : Cursor.right);
	case SIZE_CURSOR_XMAX:
		return (double) (Cursor.right > Cursor.left ? Cursor.right : Cursor.left);
	case SIZE_CURSOR_YMIN:
		return (double) (Cursor.bottom > Cursor.top ? Cursor.top : Cursor.bottom);
	case SIZE_CURSOR_YMAX:
		return (double) (Cursor.bottom > Cursor.top ? Cursor.bottom : Cursor.top);
		}
	return 0.0;
}

bool
Label::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_LB_XDIST:			fDist.fx = value;		return true;
	case SIZE_LB_YDIST:			fDist.fy = value;		return true;
	case SIZE_XPOS:				fPos.fx = value;		return true;
	case SIZE_YPOS:				fPos.fy = value;		return true;
		}
	return false;
}

bool
Label::SetColor(int select, DWORD col)
{
	switch(select & 0xfff) {
	case COL_TEXT:
		if(select & UNDO_STORESET) {
			Undo.ValDword(this, &TextDef.ColTxt, UNDO_CONTINUE);
			bModified = true;
			}
		TextDef.ColTxt = col;			bBGvalid = false;
		return true;
	case COL_BG:
		bgcolor = col;	bBGvalid = true;
		return true;
		}
	return false;
}

void
Label::DoPlot(anyOutput *o)
{
	if(!parent || !o) return;
	defDisp = o;
	if(parent && parent->Id == GO_MLABEL) parent->Command(CMD_SETFOCUS, this, o);
	switch(flags & 0x03) {
	case LB_X_DATA:	ix = o->fx2ix(fPos.fx);		break;
	case LB_X_PARENT: 
		ix = iround(parent->GetSize(SIZE_LB_XPOS));
		break;
	default:
		ix = o->co2ix(fPos.fx + parent->GetSize(SIZE_GRECT_LEFT));
		break;
		}
	switch(flags & 0x30) {
	case LB_Y_DATA:	iy = o->fy2iy(fPos.fy);		break;
	case LB_Y_PARENT: 
		iy = iround(parent->GetSize(SIZE_LB_YPOS));
		break;
	default:
		iy = o->co2iy(fPos.fy +parent->GetSize(SIZE_GRECT_TOP));
		break;
		}
	ix += o->un2ix(fDist.fx);		iy += o->un2iy(fDist.fy);
	TextDef.iSize = o->un2iy(TextDef.fSize);
	if(!(CalcRect(o))) return;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	UpdateMinMaxRect(&rDims, pts[2].x, pts[2].y);
	UpdateMinMaxRect(&rDims, pts[3].x, pts[3].y);
	if(TextDef.text && TextDef.text[0]){
		o->oTextOut(ix, iy, TextDef.text, 0);
		}
}

void
Label::DoMark(anyOutput *o, bool mark)
{
	DWORD bgpix[16];
	int i, d, d1, ix, iy, dx, dy, n;
	RECT mrc;
	anyOutput *mo;

	if(mark) {
		//find color with maximum contrast to text
		if(!bBGvalid) {
			mrc.left = rDims.left;		mrc.right = rDims.right;
			mrc.top = rDims.top;		mrc.bottom = rDims.bottom;
			dx = mrc.right - mrc.left;	dy = mrc.bottom - mrc.top;
			if(dx <= 0 || dy <= 0) return;
			if(mo = GetRectBitmap(&mrc, o)) {
				for(i = 0,  ix = 1; ix < dx; ix += dx<<2) {
					for(iy = 1; iy < dy; dy += dy<<2) {
						if(!(mo->oGetPix(pts[ix].x, pts[iy].y, &bgpix[i]))) bgpix[i] = 0x00ffffff;
						i++;
						}
					}
				DelBitmapClass(mo);		n = i;
				}
			bgcolor = bgpix[0];
			d = ColDiff(bgcolor, TextDef.ColTxt);
			for(i = 1; i < n; i++) {
				if(d < (d1 = ColDiff(bgpix[i], TextDef.ColTxt))) {
					d = d1;		bgcolor = bgpix[i];
					}
				}
			if(!d) bgcolor = TextDef.ColTxt ^ 0x00ffffffL;
			bBGvalid = true;
			}
		//in dialogs parent has no parent
		if(parent && parent->parent) o->ShowLine(pts, 5, TextDef.ColTxt);
		ShowCursor(o);	CurrGO = this;		CurrLabel = this;
		}
	else {
		HideTextCursor();
		bgLine.color = bgcolor;			o->SetLine(&bgLine);
		//in dialogs parent has no parent
		if(parent && parent->parent) o->oPolyline(pts, 5);
		IncrementMinMaxRect(&rDims, 3);
		o->UpdateRect(&rDims, false);	IncrementMinMaxRect(&rDims, -3);
		if(CurrLabel == this) CurrLabel = 0L;
		if(parent && parent->Id != GO_MLABEL && (!TextDef.text || !TextDef.text[0]))
			parent->Command(CMD_DELOBJ, (void*)this, o);
		}
}

bool
Label::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	char *tmptxt;

	if(cmd != CMD_SET_DATAOBJ && !parent) return false;
	switch (cmd) {
	case CMD_FLUSH:
		if(CurrLabel == this) CurrLabel = 0L;
		if(TextDef.text) free(TextDef.text);	TextDef.text = 0L;
		if(ssRef) free(ssRef);					ssRef = 0L;
		return true;
	case CMD_POS_FIRST:		case CMD_POS_LAST:
		Undo.ValInt(this, &CursorPos, 0L);
		bModified = true;
		if(o && TextDef.text) {
			CursorPos = (cmd == CMD_POS_LAST) ? strlen(TextDef.text) : 0;
			ShowCursor(o);
			return true;
			}
		return false;
	case CMD_CURRLEFT:
		if(o && CursorPos >0) {
			Undo.ValInt(this, &CursorPos, 0L);
			bModified = true;			CursorPos--;			ShowCursor(o);
			return true;
			}
		return false;
	case CMD_CURRIGHT:
		if(o && TextDef.text && CursorPos < (int)strlen(TextDef.text)) {
			Undo.ValInt(this, &CursorPos, 0L);
			bModified = true;			CursorPos++;			ShowCursor(o);
			return true;
			}
		return false;
	case CMD_ADDCHAR:
		SetModified();
		if(tmpl && 8 != *((int*)tmpl)) return AddChar(*((int*)tmpl), o);
		//value 8 == backspace
	case CMD_BACKSP:
		SetModified();
		if(CursorPos <=0 && o) {
			if(parent->Id == GO_MLABEL) {
				parent->Command(CMD_SETFOCUS, this, o);
				return parent->Command(CMD_BACKSP, tmpl, o);
				}
			RedrawEdit(o);
			return true;
			}
		Undo.ValInt(this, &CursorPos, 0L);
		CursorPos--;							//continue as if delete
	case CMD_DELETE:
		SetModified();
		if(cmd == CMD_DELETE) Undo.ValInt(this, &CursorPos, 0L);
		if(TextDef.text && TextDef.text[CursorPos]) {
			Undo.String(this, &TextDef.text, UNDO_CONTINUE);
			strcpy(TextDef.text + CursorPos, TextDef.text + CursorPos + 1);
			if(o)RedrawEdit(o);
			}
		else if(TextDef.text && parent->Id == GO_MLABEL) {
			parent->Command(CMD_SETFOCUS, this, o);
			return parent->Command(CMD_DELETE, tmpl, o);
			}
		else o->HideMark();
		break;
	case CMD_GETTEXT:
		if(TextDef.text && tmpl) {
			strcpy((char*)tmpl, TextDef.text);
			return true;
			}
		return false;
	case CMD_SETTEXT:
		if(TextDef.text) free(TextDef.text);
		if(tmpl) TextDef.text = strdup((char*)tmpl);
		else TextDef.text = 0L;
		return true;
	case CMD_GETTEXTDEF:
		if(!tmpl) return false;
		memcpy(tmpl, &TextDef, sizeof(TextDEF));
		return true;
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		tmptxt = TextDef.text;
		memcpy(&TextDef, tmpl, sizeof(TextDEF));
		if(!TextDef.text){
			TextDef.text = tmptxt;
			tmptxt = 0L;
			}
		else if(((TextDEF*)tmpl)->text) TextDef.text = strdup(((TextDEF*)tmpl)->text);
		if(tmptxt) free(tmptxt);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_LABEL;
		data = (DataObj*)tmpl;
		return true;
	case CMD_UPDATE:
		if(ssRef && cssRef >2 && data) {
			data->GetValue(ssRef[0].y, ssRef[0].x, &fPos.fx);
			data->GetValue(ssRef[1].y, ssRef[1].x, &fPos.fy);
			if(data->GetText(ssRef[2].y, ssRef[2].x, TmpTxt, TMP_TXT_SIZE)) {
				if(TextDef.text) free(TextDef.text);
				TextDef.text = strdup(TmpTxt);
				}
			return true;
			}
		return false;
	case CMD_SELECT:
		if(!o || !tmpl) return false;
		CalcCursorPos(((POINT*)tmpl)->x, ((POINT*)tmpl)->y, o);
		o->ShowMark(this, MRK_GODRAW);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(ObjThere(mev->x, mev->y)) {
				if(parent && parent->Id == GO_MLABEL) parent->Command(CMD_SETFOCUS, this, o);
				CalcCursorPos(mev->x, mev->y, o);
				o->ShowMark(this, MRK_GODRAW);
				return true;
				}
			break;
			}
		break;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_AUTOSCALE:
		if(parent->Id >= GO_PLOT && parent->Id < GO_GRAPH
			&& (flags & LB_X_DATA) && (flags & LB_Y_DATA)) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
	case CMD_REDRAW:
		if(CurrGO == this) {
			if(parent && parent->parent) RedrawEdit(defDisp);	//not a dialog
			else ShowCursor(defDisp);							//dialog !
			}
		else return parent->Command(cmd, tmpl, o);
		return true;
	case CMD_MOVE:
		if(parent && (parent->Id == GO_MLABEL || parent->Id == GO_LEGITEM))
			return parent->Command(cmd, tmpl, o);
		Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
	case CMD_UNDO_MOVE:
		if(!(flags & 0x03)) fPos.fx += ((lfPOINT*)tmpl)[0].fx;
		if(!(flags & 0x30)) fPos.fy += ((lfPOINT*)tmpl)[0].fy;
		if(o){
			o->StartPage();		parent->DoPlot(o);		o->EndPage();
			}
		return bModified = true;
		}
	return false;
}

void *
Label::ObjThere(int x, int y)
{
	POINT p1;

	if(IsInRect(&rDims, p1.x = x, p1.y = y) && IsInPolygon(&p1, pts, 5))
		return this;
	return 0L;
}

void
Label::Track(POINT *p, anyOutput *o)
{
	POINT *tpts;
	RECT old_rc;
	int i;

	if(!parent) return;
	if(parent->Id == GO_MLABEL || parent->Id == GO_LEGITEM){
		parent->Track(p, o);
		return;
		}
	if(o && (tpts = (POINT*)malloc(5*sizeof(POINT)))){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		//note: mLabel set Id to zero upon trackking
		//   thus rectangle is not updated if parent is a mLabel
		if(parent->Id) o->UpdateRect(&rDims, false);
		for(i = 0; i < 5; i++) {
			tpts[i].x = pts[i].x+p->x;	tpts[i].y = pts[i].y+p->y;
			UpdateMinMaxRect(&rDims, tpts[i].x, tpts[i].y);
			}
		o->ShowLine(tpts, 5, TextDef.ColTxt);
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		free(tpts);
		}
}

bool
Label::CalcRect(anyOutput *o)
{
	int rx1, rx, ry;
	fRECT rc, rcc;

	if(parent && parent->Id != GO_MLABEL) o->SetTextSpec(&TextDef);
	if(TextDef.text && TextDef.text[0]) {
		if(!(o->oGetTextExtent(TextDef.text, strlen(TextDef.text), &rx, &ry))) return false;
		rx++;
		}
	else {
		if(!(o->oGetTextExtent("A", 1, &rx, &ry))) return false;
		rx = 1;
		}
	rx += 4;	rc.Xmin = -2.0f;	rc.Ymin = 0.0f;		rc.Xmax = rx;		rc.Ymax = ry;
	si = sin(TextDef.RotBL *0.01745329252f);	csi = cos(TextDef.RotBL *0.01745329252f);
	if(TextDef.Align & TXA_HCENTER) {
		rc.Xmin -= rx/2.0-1.0;		rc.Xmax -= rx/2.0-1.0;
		}
	else if(TextDef.Align & TXA_HRIGHT) {
		rc.Xmin -= rx-2.0;			rc.Xmax -= rx-2.0;
		}
	if(TextDef.Align & TXA_VCENTER) {
		rc.Ymin -= ry/2.0;			rc.Ymax -= ry/2.0;
		}
	else if(TextDef.Align & TXA_VBOTTOM) {
		rc.Ymin -= ry;				rc.Ymax -= ry;
		}
	if(o->oGetTextExtent(TextDef.text, CursorPos, &rx1, &ry)){
		rx = CursorPos ? (int)rx1 : 0;
		}
	else rx = 0;
	rcc.Xmax = rc.Xmin + (double)rx+2.0;	rcc.Ymin = rc.Ymin+2.0;
	rcc.Xmin = rc.Xmin;						rcc.Ymax = rc.Ymax-2.0;
	pts[0].x = iround(rc.Xmin*csi + rc.Ymin*si)+ix;
	pts[0].y = iround(rc.Ymin*csi - rc.Xmin*si)+iy;
	pts[1].x = iround(rc.Xmax*csi + rc.Ymin*si)+ix;
	pts[1].y = iround(rc.Ymin*csi - rc.Xmax*si)+iy;
	pts[2].x = iround(rc.Xmax*csi + rc.Ymax*si)+ix;
	pts[2].y = iround(rc.Ymax*csi - rc.Xmax*si)+iy;
	pts[3].x = iround(rc.Xmin*csi + rc.Ymax*si)+ix;
	pts[3].y = iround(rc.Ymax*csi - rc.Xmin*si)+iy;
	pts[4].x = pts[0].x;	pts[4].y = pts[0].y;
	Cursor.left = iround(rcc.Xmax*csi + rcc.Ymin*si)+ix;
	Cursor.top = iround(rcc.Ymin*csi - rcc.Xmax*si)+iy;
	Cursor.right = iround(rcc.Xmax*csi + rcc.Ymax*si)+ix;
	Cursor.bottom = iround(rcc.Ymax*csi - rcc.Xmax*si)+iy;
	return true;
}

void 
Label::RedrawEdit(anyOutput *o)
{
	FillDEF bgFill = {FILL_NONE, bgcolor, 1.0, 0L, bgcolor};

	if(!o || !parent) return;
	bgLine.color = bgcolor;		o->SetLine(&bgLine);	o->SetFill(&bgFill);
	o->oPolygon(pts, 5);		IncrementMinMaxRect(&rDims, 3);		
	o->UpdateRect(&rDims, false);
	CalcRect(o);				bgLine.color ^= 0x00ffffffL;
	o->SetLine(&bgLine);		o->oPolygon(pts, 5);
	if(parent->Id == GO_MLABEL) {
		if(parent->parent && parent->parent->Id == GO_LEGITEM && parent->parent->parent)
			parent->parent->parent->DoPlot(o);
		else parent->DoPlot(o);
		}
	else if(parent->Id == GO_LEGITEM && parent->parent) parent->parent->DoPlot(o);
	else DoPlot(o);
	o->UpdateRect(&rDims, false);
	DoMark(o, true);			ShowCursor(o);
}

void
Label::ShowCursor(anyOutput *o)
{
	CalcRect(o);
	ShowTextCursor(o, &Cursor, TextDef.ColTxt);
}

bool
Label::AddChar(int ci, anyOutput *o)
{
	char c, *txt1 = 0L;
	int i, j, k;
	GraphObj *golist[2];

	if(!o) return false;
	if(ci == 13 && parent){		//CR
		if(parent->Id == GO_MLABEL){
			parent->Command(CMD_SETFOCUS, this, o);
			return parent->Command(CMD_ADDCHAR, &ci, o);
			}
		if(golist[1] = new mLabel(parent, data, fPos.fx, fPos.fy, &TextDef, 
			TextDef.text, CursorPos, flags)){
			golist[1]->moveable = moveable;
			golist[1]->SetSize(SIZE_LB_XDIST, fDist.fx);
			golist[1]->SetSize(SIZE_LB_YDIST, fDist.fy);
			}
		golist[0] = this;
		if(!parent->Command(CMD_MUTATE, golist, o)) DeleteGO(golist[1]);
		return false;
		}
	if(ci < 254 && ci > 31) c = (char)ci;
	else return false;
	if(!TextDef.text || CursorPos < 10 || c == ' ') {
		Undo.String(this, &TextDef.text, 0L);
		Undo.ValInt(this, &CursorPos, UNDO_CONTINUE);
		}
	if(TextDef.text) txt1 = (char*)calloc((i = strlen(TextDef.text))+2, sizeof(char));
	else txt1 = (char*)calloc((i = 0)+2, sizeof(char));
	if(txt1) {
		for(j = k = 0; j< i && j < CursorPos; txt1[k++] = TextDef.text[j++]);
		txt1[k++] = c;
		for(; j< i; txt1[k++] = TextDef.text[j++]);
		if(TextDef.text) free(TextDef.text);
		TextDef.text = txt1;
		CursorPos++;
		RedrawEdit(o);
		}
	return true;
}

void
Label::CalcCursorPos(int x, int y, anyOutput *o)
{
	int i, j, ix, x1, y1, x2, h;

	if(!o || !TextDef.text) return;
	TextDef.iSize = o->un2iy(TextDef.fSize);
	if(parent && parent->Id != GO_MLABEL) o->SetTextSpec(&TextDef);
	x1 = ((pts[3].x + pts[4].x)>>1);	y1 = ((pts[3].y + pts[4].y)>>1);
	ix = ((j=(x1-x))*j);	ix += ((j=(y1-y))*j);	ix = isqr(ix);
	for(i = 1,  x1 = 0; TextDef.text[i-1]; i++) {
		o->oGetTextExtent(TextDef.text, i, &x2, &h);
		if(x1 <= ix && x2 >= ix) {
			if((ix-x1) < (x2-ix)) CursorPos = i-1;
			else CursorPos = i;
			return;
			}
		else if(ix < x2){
			CursorPos = i-1;
			return;
			}
		x1 = x2;
		}
	if(pts[3].x < pts[2].x && x < pts[3].x) CursorPos = 0;
	else CursorPos = strlen(TextDef.text);
}

void
Label::SetModified()
{
	AxisDEF *adef;

	bModified = true;
	if(parent && parent->Id==GO_TICK && parent->parent && parent->parent->Id==GO_AXIS){
	adef = ((Axis*)(parent->parent))->GetAxis();
	adef->flags &= ~AXIS_AUTOSCALE;
	}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a multiline label consists of several Label objects
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
mLabel::mLabel(GraphObj *par, DataObj *d, double x, double y, TextDEF *td, char *txt, 
	int cp, DWORD flg):GraphObj(par, d)
{
	int i;

	memcpy(&TextDef, td, sizeof(TextDEF));
	TextDef.text = 0L;		fPos.fx = x;		fPos.fy = y;
	Lines = 0L;				flags = flg;
	fDist.fx = 0.0;			fDist.fy = 0.0;
	CurrGO = CurrLabel = 0L;
	if(txt && (Lines = (Label**)calloc(2, sizeof(Label*)))) {
		for(i = 0; i < cp && txt[i]; i ++) TmpTxt[i] = txt[i];
		TmpTxt[i] = 0;
		if(Lines[0] = new 	Label(this, d, x, y, &TextDef, LB_X_PARENT | LB_Y_PARENT))
			Lines[0]->Command(CMD_SETTEXT, TmpTxt, 0L);
		if(Lines[1] = new 	Label(this, d, x, y, &TextDef, LB_X_PARENT | LB_Y_PARENT)){
			Lines[1]->Command(CMD_SETTEXT, txt+cp, 0L);
			CurrGO = CurrLabel = Lines[1];
			}
		nLines = 2;		cli = 1;
		}
	Id = GO_MLABEL;
}

mLabel::mLabel(GraphObj *par, DataObj *d, double x, double y, TextDEF *td, char *txt)
	:GraphObj(par, d)
{
	int i, nll;
	char **llist;

	memcpy(&TextDef, td, sizeof(TextDEF));
	TextDef.text = 0L;		fPos.fx = x;		fPos.fy = y;
	Lines = 0L;				flags = 0L;			Id = GO_MLABEL;
	fDist.fx = 0.0;			fDist.fy = 0.0;
	CurrGO = CurrLabel = 0L;
	if(txt){
		if((llist=split(txt,'\n',&nll)) && nll && (Lines=(Label**)calloc(nll, sizeof(Label*)))){
			for(i = 0; i < nll; i++) {
				if(llist[i]){
					Lines[i] = new 	Label(this, d, x, y, &TextDef, LB_X_PARENT | LB_Y_PARENT);
					Lines[i]->Command(CMD_SETTEXT, llist[i], 0L);
					free(llist[i]);
					}
				}
			free(llist);	nLines = nll;		cli = 0;
			}
		}
}

mLabel::mLabel(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

mLabel::~mLabel()
{
	int i;
	
	Undo.InvalidGO(this);
	if(Lines){
		for(i = 0; i < nLines; i++) if(Lines[i]) DeleteGO(Lines[i]);
		free(Lines);
		}
}

double
mLabel::GetSize(int select)
{
	switch(select){
	case SIZE_LB_XPOS:	return cPos1.fx;
	case SIZE_LB_YPOS:	return cPos1.fy;
	case SIZE_GRECT_TOP:	
		if (parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
mLabel::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_LB_XDIST:
		undo_flags = CheckNewFloat(&fDist.fx, fDist.fx, value, this, undo_flags);
		return true;
	case SIZE_LB_YDIST:
		undo_flags = CheckNewFloat(&fDist.fy, fDist.fy, value, this, undo_flags);
		return true;
	case SIZE_XPOS:
		undo_flags = CheckNewFloat(&fPos.fx, fPos.fx, value, this, undo_flags);
		return true;
	case SIZE_YPOS:
		undo_flags = CheckNewFloat(&fPos.fy, fPos.fy, value, this, undo_flags);
		return true;
		}
	return false;
}

void
mLabel::DoPlot(anyOutput *o)
{
	int i;
	double dh, dx, dy;

	if(!o || !Lines) return;
	undo_flags = 0L;
	if(parent){		//if this object is part of a dialog we dont have a parent
		dx = parent->GetSize(SIZE_GRECT_LEFT);
		dy = parent->GetSize(SIZE_GRECT_TOP);
		}
	else dx = dy = 0.0;
	cPos.fx = cPos.fy = 0.0;
	TextDef.iSize = o->un2iy(TextDef.fSize);
	switch(flags & 0x03) {
	case LB_X_DATA:		cPos.fx = o->fx2fix(fPos.fx);				break;
	case LB_X_PARENT:	if(parent) cPos.fx = parent->GetSize(SIZE_LB_XPOS);	break;
	default:
		//if no parent its a dialog
		cPos.fx = parent ? o->co2fix(fPos.fx + dx) : fPos.fx;
		break;
		}
	switch(flags & 0x30) {
	case LB_Y_DATA:		cPos.fy = o->fy2fiy(fPos.fy);				break;
	case LB_Y_PARENT:	if(parent) cPos.fy = parent->GetSize(SIZE_LB_YPOS);	break;
	default:	
		//if no parent its a dialog
		cPos.fy = parent ? o->co2fiy(fPos.fy + dy) : fPos.fy;
		break;
		}
	si = sin(TextDef.RotBL *0.01745329252f);	csi = cos(TextDef.RotBL *0.01745329252f);
	if(TextDef.Align & TXA_VBOTTOM) dh = (double)(nLines-1);
	else if(TextDef.Align & TXA_VCENTER) dh = ((double)(nLines-1))/2.0;
	else dh = 0.0;						dh *= TextDef.fSize;
	cPos.fx -= o->un2fix(dh*si);		cPos.fy -= o->un2fiy(dh*csi);
	memcpy(&cPos1, &cPos, sizeof(lfPOINT));
	dist.fx = floor(o->un2fix(TextDef.fSize * si * 1.1));
	dist.fy = floor(o->un2fiy(TextDef.fSize * csi * 1.1));
	o->SetTextSpec(&TextDef);
	rDims.left = rDims.top = rDims.right = rDims.bottom = 0;
	for(i = 0; i < nLines; i++)	if(Lines[i]){
		Lines[i]->Command(CMD_SETTEXTDEF, &TextDef, o);
		Lines[i]->SetSize(SIZE_LB_XDIST, fDist.fx);	Lines[i]->SetSize(SIZE_LB_YDIST, fDist.fy);
		Lines[i]->SetSize(SIZE_XPOS, fPos.fx);		Lines[i]->SetSize(SIZE_YPOS, fPos.fy);
		Lines[i]->moveable = moveable;				Lines[i]->DoPlot(o);
		UpdateMinMaxRect(&rDims, Lines[i]->rDims.left, Lines[i]->rDims.top);
		UpdateMinMaxRect(&rDims, Lines[i]->rDims.right, Lines[i]->rDims.bottom);
		}
	if(CurrLabel && o && Command(CMD_SETFOCUS, CurrLabel, o))
		Lines[cli]->ShowCursor(o);
}

bool
mLabel::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, j, k;
	fRECT t_cur;
	MouseEvent mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(Lines && tmpl && IsInRect(&rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)) 
			for(i = 0; i<nLines; i++) if(Lines[i] && Lines[i]->Command(cmd, tmpl, o)) return true;
		break;
	case CMD_HIDE_MARK:
		if(Lines && o && tmpl) for(i = 0; i< nLines; i++) 
			if(Lines[i] && tmpl == (void*)Lines[i]){
				Lines[i]->DoMark(o, false);
				return true;
				}
		return false;
	case CMD_ADDCHAR:
		if(!tmpl || 13 != *((int*)tmpl)) return false;
		if(!Lines[cli] || !Lines[cli]->Command(CMD_GETTEXT, &TmpTxt, o)) return false;
		k = iround(Lines[cli]->GetSize(SIZE_CURSORPOS));
		if(parent)Undo.ObjConf(this, 0L);
		if(tmpl && Lines && (Lines = (Label**)realloc(Lines, (nLines+1) * sizeof(Label*)))) {
			for(i = nLines-1, j = nLines; i >= cli; i--, j-- ) {
				Lines[j] = Lines[i];
				}
			i++, j++;
			if(Lines[j]) DeleteGO(Lines[j]);	Lines[i] = Lines[j] = 0L;	nLines++;
			if(Lines[j] = new 	Label(this, data, fPos.fx, fPos.fy, &TextDef, LB_X_PARENT | LB_Y_PARENT)){
				Lines[j]->Command(CMD_SETTEXT, TmpTxt+k, o);
				Lines[j]->Command(CMD_POS_FIRST, 0L, o);
				CurrGO = CurrLabel = Lines[j];
				TmpTxt[k] = 0;
				}
			if(Lines[i] = new 	Label(this, data, fPos.fx, fPos.fy, &TextDef, LB_X_PARENT | LB_Y_PARENT))
				Lines[i]->Command(CMD_SETTEXT, TmpTxt, 0L);
			Command(CMD_SETFOCUS, CurrGO, o);
			if(parent) return parent->Command(CMD_REDRAW, 0L, o);
			else if(o) DoPlot(o);
			}
		break;
	case CMD_SETFOCUS:
		for(i = 0; i< nLines; i++) if(Lines[i] == (Label*)tmpl) {
			cli = i;
			cPos1.fx = cPos.fx + dist.fx*i;
			cPos1.fy = cPos.fy + dist.fy*i;
			return true;
			}
		break;
	case CMD_GETTEXT:
		if(tmpl && Lines && nLines && Lines[0]) return Lines[0]->Command(CMD_GETTEXT, tmpl, o);
		break;
	case CMD_ALLTEXT:
		if(tmpl && Lines && nLines){
			for(i = 0, j = 500; i < nLines; i++) {
				if(Lines[i] && Lines[i]->Command(CMD_GETTEXT, TmpTxt, o) && TmpTxt[0]){
					j += sprintf(TmpTxt+j, "%s\n", TmpTxt);
					}
				}
			strcpy((char*)tmpl, TmpTxt+500);
			return true;
			}
		break;
	case CMD_DELETE:
		cli++;
		//fall through
	case CMD_BACKSP:
		if(cli > 0 && cli < nLines && Lines && Lines[cli] && Lines[cli-1]) {
			Lines[cli-1]->Command(CMD_POS_LAST, 0L, o);
			TmpTxt[0] = 0;
			Lines[cli-1]->Command(CMD_GETTEXT, TmpTxt, o);
			Lines[cli]->Command(CMD_GETTEXT, TmpTxt+strlen(TmpTxt), o);
			Lines[cli-1]->Command(CMD_SETTEXT, TmpTxt, o);
			DeleteGO(Lines[cli]);	Lines[cli] = 0L;
			for(i = cli+1; i < nLines; i++){
				Lines[i-1] = Lines[i], Lines[i]= 0L;
				}
			nLines--;
			CurrGO = CurrLabel = Lines[cli-1];
			if(parent) parent->Command(CMD_REDRAW, 0L, o);
			if(CurrLabel) CurrLabel->RedrawEdit(o);
			return true;
			}
		return false;
	case CMD_GETTEXTDEF:
		if(!tmpl) return false;
		memcpy(tmpl, &TextDef, sizeof(TextDEF));
		return true;
	case CMD_SETTEXTDEF:
		if(!tmpl)return false;
		Undo.TextDef(this, &TextDef, undo_flags);
		undo_flags |= UNDO_CONTINUE;
		memcpy(&TextDef, tmpl, sizeof(TextDEF));
		TextDef.text = 0L;
		return true;
	case CMD_SET_DATAOBJ:
		if(Lines) for(i = 0; i< nLines; i++) 
			if(Lines[i]) Lines[i]->Command(cmd, tmpl, o);
		Id = GO_MLABEL;
		data = (DataObj*)tmpl;
	case CMD_REG_GO:
		if(Lines) for(i = 0; i< nLines; i++) 
			if(Lines[i])Lines[i]->Command(cmd, tmpl, o);
		if(cmd == CMD_REG_GO)((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_AUTOSCALE:
		if(parent && parent->Id >= GO_PLOT && parent->Id < GO_GRAPH
			&& (flags & LB_X_DATA) && (flags & LB_Y_DATA)) {
			((Plot*)parent)->CheckBounds(fPos.fx, fPos.fy);
			return true;
			}
		break;
	case CMD_CURRUP:		case CMD_CURRDOWN:
		if(!o) return false;
		o->SetTextSpec(&TextDef);
		Command(CMD_SETFOCUS, CurrGO, o);
		if(cli >= 0 && cli < nLines && Lines && Lines[cli]) {
			t_cur.Xmin = Lines[cli]->GetSize(SIZE_CURSOR_XMIN);
			t_cur.Xmax = Lines[cli]->GetSize(SIZE_CURSOR_XMAX);
			t_cur.Ymin = Lines[cli]->GetSize(SIZE_CURSOR_YMIN);
			t_cur.Ymax = Lines[cli]->GetSize(SIZE_CURSOR_YMAX);
			mev.StateFlags = 0;		mev.Action = MOUSE_LBUP;
			mev.x = iround((t_cur.Xmax+t_cur.Xmin)/2.0);
			mev.y = iround((t_cur.Ymax+t_cur.Ymin)/2.0);
			i = o->un2ix(TextDef.fSize*si);		j = o->un2iy(TextDef.fSize*csi);
			if(cmd == CMD_CURRUP && cli > 0 && Lines[cli-1]) {
				Lines[cli-1]->CalcCursorPos(mev.x -= i, mev.y -= j, o);
				o->ShowMark(CurrGO = CurrLabel = Lines[cli-=1], MRK_GODRAW);
				return Command(CMD_SETFOCUS, Lines[cli], o);
				}
			if(cmd == CMD_CURRDOWN && cli < (nLines-1) && Lines[cli+1]) {
				Lines[cli+1]->CalcCursorPos(mev.x += i, mev.y += j, o);
				o->ShowMark(CurrGO = CurrLabel = Lines[cli+=1], MRK_GODRAW);
				return Command(CMD_SETFOCUS, Lines[cli], o);
				}
			}
		else return false;
		break;
	case CMD_SELECT:
		if(o && tmpl) for(i = 0; i < nLines; i++){
			o->SetTextSpec(&TextDef);
			if(Lines[i] && ((POINT*)tmpl)->y > Lines[i]->rDims.top && ((POINT*)tmpl)->y < 
				Lines[i]->rDims.bottom) return Lines[i]->Command(cmd, tmpl, o);
			}
		break;
	case CMD_DELOBJ:
		if(parent && Lines) for(i = 0; i< nLines; i++) 
			if(Lines[i] && Lines[i] == (Label*)tmpl) return parent->Command(cmd, this, o);
		break;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		else if(o) DoPlot(o);
		break;
	case CMD_MOVE:
		if(parent && parent->Id == GO_LEGITEM) return parent->Command(cmd, tmpl, o);
		Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
	case CMD_UNDO_MOVE:
		if(!(flags & 0x03)) fPos.fx += ((lfPOINT*)tmpl)[0].fx;
		if(!(flags & 0x30)) fPos.fy += ((lfPOINT*)tmpl)[0].fy;
		if(o && parent){
			o->StartPage();		parent->DoPlot(o);		o->EndPage();
			}
		return true;
		}
	return false;
}

void
mLabel::Track(POINT *p, anyOutput *o)
{
	int i;

	if(!parent) return;
	if(parent->Id == GO_LEGITEM){
		parent->Track(p, o);
		return;
		}
	for(i = 0; i < nLines; i++)	if(Lines[i]){
		if(!i) memcpy(&rDims, &Lines[i]->rDims, sizeof(RECT));
		else {
			UpdateMinMaxRect(&rDims, Lines[i]->rDims.left, Lines[i]->rDims.top);
			UpdateMinMaxRect(&rDims, Lines[i]->rDims.right, Lines[i]->rDims.bottom);
			}
		}
	o->UpdateRect(&rDims, false);
	Id = 0L;			//disable reentrance from Label objects
	for(i = 0; i < nLines; i++)	if(Lines[i]) Lines[i]->Track(p, o);
	Id = GO_MLABEL;		//enable entrance from Label objects
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// The following GOs use absolute coordinates
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// The segment object is either a pie slice or a ring segment
//
segment::segment(GraphObj *par, DataObj *d, lfPOINT *c, double r1, double r2,
				 double a1, double a2):GraphObj(par, d)
{
	FileIO(INIT_VARS);
	segFill.hatch = &segFillLine;
	segFill.color = 0x00c0c0c0L;
	fCent.fx = c->fx;		fCent.fy = c->fy;
	radius1 = r1;			radius2 = r2;
	angle1 = a1;			angle2 = a2;
	Id = GO_SEGMENT;
	bModified = false;
}

segment::segment(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

segment::~segment()
{
	if(pts && nPts) free(pts);
	if(bModified) Undo.InvalidGO(this);
}
	
bool 
segment::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_XPOS:		fCent.fx = value;		break;
	case SIZE_YPOS:		fCent.fy = value;		break;
	case SIZE_RADIUS1:	radius1 = value;		break;
	case SIZE_RADIUS2:	radius2 = value;		break;
	case SIZE_ANGLE1:	angle1 = value;			break;
	case SIZE_ANGLE2:	angle2 = value;			break;
	default:			return false;
		}
	return true;
}

void
segment::DoPlot(anyOutput *o)
{
	double dsize, dpt, frx1, frx2, dtmp, da, dda, sia, csia;
	int i, n, npt = 12;
	POINT np, of, cp,  *tmppts;

	if(!o || angle1 == angle2) return;
	dsize = angle1 > angle2 ? angle1 -angle2 : (angle1+360.0)-angle2;
	dpt = dsize*0.01745329252;		//degrees to rad
	frx1 = (double)o->un2fix(radius1);
	frx2 = (double)o->un2fix(radius2);
	dtmp = frx1*dpt;
	npt += (int)(dtmp < dsize ? dtmp : dsize);
	dtmp = frx2*dpt;
	npt += (int)(dtmp < dsize ? dtmp : dsize);
	if(!(pts = (POINT*)malloc(npt*sizeof(POINT))))return;
	nPts = 0;
	n = (dtmp < dsize) ? (int)dtmp : (int)dsize;
	while (n<2) n++;
	da = angle1*0.01745329252;
	dda = (dsize*0.01745329252)/(double)n;
	sia = sin(0.5*((angle2 < angle1 ? angle1 : angle1 +360) + angle2) * 0.01745329252);
	csia = cos(0.5*((angle2 < angle1 ? angle1 : angle1 +360) + angle2) * 0.01745329252);
	of.x = o->un2ix(shift * csia);
	of.y = - (o->un2iy(shift * sia));
	cp.x = o->co2ix(fCent.fx + (parent ? parent->GetSize(SIZE_GRECT_LEFT): 0.0));
	cp.y = o->co2iy(fCent.fy + (parent ? parent->GetSize(SIZE_GRECT_TOP): 0.0));
	for(i = 0; i < n; i++) {
		sia = sin(da);					csia = cos(da);
		np.x = of.x + cp.x;				np.y = of.y + cp.y;
		np.x += o->un2ix(csia*radius2);	np.y -= o->un2iy(sia*radius2);
		AddToPolygon(&nPts, pts, &np);
		da -= dda;
		}
	sia = sin(angle2 *0.01745329252);
	csia = cos(angle2 *0.01745329252);
	np.x = of.x + cp.x;					np.y = of.y + cp.y;
	np.x += o->un2ix(csia*radius2);		np.y -= o->un2iy(sia*radius2);
	AddToPolygon(&nPts, pts, &np);
	dtmp = frx1*dpt;
	n = dtmp < dsize ? (int)dtmp : (int)dsize;
	da = angle2*0.01745329252;
	if(n>1)dda = (dsize*0.01745329252)/(double)n;
	else dda = 0.0;
	for(i = 0; i < n; i++) {
		sia = sin(da);					csia = cos(da);
		np.x = of.x + cp.x;				np.y = of.y + cp.y;
		np.x += o->un2ix(csia*radius1);	np.y -= o->un2iy(sia*radius1);
		AddToPolygon(&nPts, pts, &np);
		da += dda;
		}
	sia = sin(angle1 *0.01745329252);	csia = cos(angle1 *0.01745329252);
	np.x = of.x + cp.x;					np.y = of.y + cp.y;
	np.x += o->un2ix(csia*radius1);		np.y -= o->un2iy(sia*radius1);
	AddToPolygon(&nPts, pts, &np);
	if(nPts <3) return;
	AddToPolygon(&nPts, pts, &pts[0]);	//close polygon
	o->SetLine(&segLine);				o->SetFill(&segFill);
	o->oPolygon(pts, nPts);
	if(tmppts = (POINT*)realloc(pts, nPts *sizeof(POINT))) pts = tmppts;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < nPts; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 3*o->un2ix(segLine.width);		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i+6);
}

void
segment::DoMark(anyOutput *o, bool mark)
{
	if(mark)InvertPolygon(pts, nPts, &segLine, &segFill, &rDims, o, mark);
	else if(parent) {
		parent->DoPlot(o);
		o->UpdateRect(&rDims, false);
		}
}

bool
segment::Command(int cmd, void *tmpl, anyOutput *o)
{
	FillDEF *TmpFill;
	LegItem *leg;

	switch (cmd) {
	case CMD_LEGEND:
		if(tmpl) {
			leg = new LegItem(this, data, 0L, &segLine, &segFill);
			if(!((Legend*)tmpl)->Command(CMD_DROP_OBJECT, leg, o)) DeleteGO(leg);
			}
		return true;
	case CMD_MOUSE_EVENT:
		switch (((MouseEvent*)tmpl)->Action) {
		case MOUSE_LBUP:
			if(ObjThere(((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)) 
				return o->ShowMark(CurrGO=this, MRK_GODRAW);
			}
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_SEGMENT;
		return true;
	case CMD_REDRAW:
		return parent ? parent->Command(cmd, tmpl, o) : false;
	case CMD_SEG_FILL:
		TmpFill = (FillDEF *)tmpl;
		if(TmpFill) {
			segFill.type = TmpFill->type;			segFill.color = TmpFill->color;
			segFill.scale = TmpFill->scale;
			if(TmpFill->hatch) memcpy(&segFillLine, TmpFill->hatch, sizeof(LineDEF));
			}
		return true;
	case CMD_SEG_LINE:
		if(tmpl) memcpy(&segLine, tmpl, sizeof(LineDEF));
		return true;
	case CMD_SEG_MOVEABLE:
		if(tmpl) moveable = *((int*)tmpl);
		return true;
	case CMD_SHIFT_OUT:
		if(tmpl) shift = *((double*)tmpl);
		return true;
	case CMD_MOVE:
		bModified = true;
		Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
	case CMD_UNDO_MOVE:
		fCent.fx += ((lfPOINT*)tmpl)[0].fx;		fCent.fy += ((lfPOINT*)tmpl)[0].fy;
		if(parent)parent->Command(CMD_REDRAW, 0L, o);
		return true;
		}
	return false;
}

void *
segment::ObjThere(int x, int y)
{
	bool bFound = false;
	POINT p1;

	if(IsInRect(&rDims, p1.x = x, p1.y = y)) {
		bFound = IsInPolygon(&p1, pts, nPts);
		if(bFound || IsCloseToPL(p1, pts, nPts)) return this;
		}
	return 0L;
}

void
segment::Track(POINT *p, anyOutput *o)
{
	POINT *tpts;
	RECT old_rc;
	int i;

	if(o && (tpts = (POINT*)malloc(nPts*sizeof(POINT)))){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		o->UpdateRect(&rDims, false);
		for(i = 0; i < nPts; i++) {
			tpts[i].x = pts[i].x+p->x;			tpts[i].y = pts[i].y+p->y;
			UpdateMinMaxRect(&rDims, tpts[i].x, tpts[i].y);
			}
		o->ShowLine(tpts, nPts, segLine.color);
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		free(tpts);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the polyline object
//
polyline::polyline(GraphObj *par, DataObj *d, lfPOINT *fpts, int cpts):
	GraphObj(par, d)
{
	double dx = 0.0, dy = 0.0;
	int i;

	FileIO(INIT_VARS);
	if(parent){
		dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
		}
	if(Values = (lfPOINT*)malloc((nPoints = cpts)* sizeof(lfPOINT))){
		memcpy(Values, fpts, cpts*sizeof(lfPOINT));
		for(i = 0; i < cpts; i++) {
			Values[i].fx -= dx;		Values[i].fy -= dy;
			}
		}
	Id = GO_POLYLINE;
	bModified = false;
}

polyline::polyline(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	bModified = false;
}

polyline::~polyline()
{
	Command(CMD_FLUSH, 0L, 0L);
	if(this == CurrGO) CurrGO = 0L;
	if(bModified) Undo.InvalidGO(this);
}

double
polyline::GetSize(int select)
{
	int i;

	if(select >= SIZE_XPOS && select <=  SIZE_XPOS_LAST){
		if((i = select-SIZE_XPOS) >=0 && i <= 200)
			return i < nPoints ? Values[i].fx : Values[nPoints-2].fx;
		}
	if(select >= SIZE_YPOS && select <= SIZE_YPOS_LAST){
		if((i = select-SIZE_YPOS) >=0 && i <= 200)
			return i < nPoints ? Values[i].fy : Values[nPoints-2].fy;
		}
	return parent ? parent->GetSize(select) : 0.0;
}

DWORD
polyline::GetColor(int select)
{
	return pgLine.color;
}

bool
polyline::SetSize(int select, double value)
{
	int i;

	if((select & 0xfff) >= SIZE_XPOS && (select & 0xfff) <=  SIZE_XPOS_LAST){
		if((i = select-SIZE_XPOS) >=0 && i <= 200 && i < (int)nPoints){
			Values[i].fx = value;
			return true;
			}
		}
	if((select & 0xfff) >= SIZE_YPOS && (select & 0xfff) <= SIZE_YPOS_LAST){
		if((i = select-SIZE_YPOS) >=0 && i <= 200 && i < (int)nPoints){
			Values[i].fy = value;
			return true;
			}
		}
	return false;
}

void
polyline::DoPlot(anyOutput *o)
{
	POINT np, *tmppts;
	double dx, dy;
	int i;

	if(!Values || !nPoints || !o || !parent) return;
	if(pts) free(pts);
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	if(!(pts = (POINT*)malloc((nPoints+2)*sizeof(POINT))))return;
	for(i = nPts = 0; i < nPoints; i++){
		np.x = o->co2ix(Values[i].fx + dx);		np.y = o->co2iy(Values[i].fy + dy);
		AddToPolygon(&nPts, pts, &np);
		}
	if(type == 1) AddToPolygon(&nPts, pts, &pts[0]);	//close polygon
	if(tmppts = (POINT*)realloc(pts, nPts *sizeof(POINT))) pts = tmppts;
	SetMinMaxRect(&rDims, pts[0].x, pts[0].y, pts[1].x, pts[1].y);
	for(i = 2; i < nPts; i++) UpdateMinMaxRect(&rDims, pts[i].x, pts[i].y);
	i = 3*o->un2ix(pgLine.width)+3;		//increase size of rectangle for marks
	IncrementMinMaxRect(&rDims, i);
	o->SetLine(&pgLine);
	if(this == CurrGO) o->ShowMark(this, MRK_GODRAW);
	else switch(type) {
	case 0:				//line
		o->oPolyline(pts, nPts);
		break;
	case 1:				//polygon
		o->SetFill(&pgFill);
		o->oPolygon(pts, nPts);
		break;
		}
}

void
polyline::DoMark(anyOutput *o, bool mark)
{
	RECT upd;

	memcpy(&upd, &rDims, sizeof(RECT));
	IncrementMinMaxRect(&upd, 6 + o->un2ix(pgLine.width)*4);
	if(mark) {
		o->SetLine(&pgLine);
		if(nPoints < 200){
			switch(type) {
			case 0:				//line
				o->oPolyline(pts, nPts);
			break;
			case 1:				//polygon
				o->SetFill(&pgFill);
				o->oPolygon(pts, nPts);
				break;
				}
			ShowPoints(o);
			}
		else InvertLine(pts, nPts, &pgLine, &upd, o, true);;
 		}
	else if(parent)	parent->DoPlot(o);
	o->UpdateRect(&upd, false);
}

bool
polyline::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	POINT p1;
	bool bFound = false;
	int i;

	switch (cmd) {
	case CMD_MRK_DIRTY:			//issued by Undo
		CurrGO = this;
		bModified = true;
	case CMD_FLUSH:
		if(pHandles) {
			for(i = 0; i < nPoints; i++) if(pHandles[i]) delete(pHandles[i]);
			free(pHandles);		pHandles = 0L;
			}
		if(cmd == CMD_FLUSH && Values && nPoints){
			free(Values);
			Values = 0L;	nPoints = 0;
			}
		if(pts && nPts) free(pts);		pts = 0L;		nPts = 0;
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(!ObjThere(p1.x= mev->x, p1.y=mev->y)|| CurrGO || !o || nPoints <2)return false;
			o->ShowMark(CurrGO=this, MRK_GODRAW);
			break;
			}
		return false;
	case CMD_DELOBJ:
		if(pHandles && tmpl && tmpl == (void*)CurrHandle) {
			for(i = 0; i < nPoints; i++) if(pHandles[i] == CurrHandle) {
				Undo.DataMem(this, (void**)&Values, nPoints * sizeof(lfPOINT), &nPoints, 0L);
				for( ; i < nPoints-1; i++) {
					Values[i].fx = Values[i+1].fx;	Values[i].fy = Values[i+1].fy;
					}
				nPoints--;
				if(pHandles[nPoints])delete(pHandles[nPoints]);
				pHandles[nPoints] = 0L;				CurrHandle = 0L;
				CurrGO = this;						bModified = true;
				return true;
				}
			}
		return false;
	case CMD_SAVEPOS:
		if(tmpl && Values) {
			bModified = true;
			i = *(int*)tmpl;
			if(i >= 0 && i < nPoints) Undo.SaveLFP(this, Values + i, 0L);
			}
		return true;
	case CMD_SET_DATAOBJ:
		Id = type == 1 ? GO_POLYGON : GO_POLYLINE;
		return true;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_MOVE:
		bModified = true;
		Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
	case CMD_UNDO_MOVE:
		for(i = 0; i < nPoints; i++) {
			Values[i].fx += ((lfPOINT*)tmpl)[0].fx;
			Values[i].fy += ((lfPOINT*)tmpl)[0].fy;
			}
		if(o) {
			o->StartPage();		parent->DoPlot(o);		o->EndPage();
			}
		return true;
		}
	return false;
}

void * 
polyline::ObjThere(int x, int y)
{
	bool bFound = false;
	POINT p1;
	int i;
	void *ret;

	if(IsInRect(&rDims, p1.x = x, p1.y = y)) {
		if(CurrGO == this && pHandles) 
			for(i = nPoints-1; i >= 0; i--) 
				if((pHandles[i]) && (ret = pHandles[i]->ObjThere(x, y))) return ret;
		if(type == 1) bFound = IsInPolygon(&p1, pts, nPts);
		if(bFound || IsCloseToPL(p1,pts,nPts)) return this;
		}
	return 0L;
}

void
polyline::Track(POINT *p, anyOutput *o)
{
	POINT *tpts;
	RECT old_rc;
	int i;

	if(o && (tpts = (POINT*)malloc(nPts*sizeof(POINT)))){
		memcpy(&old_rc, &rDims, sizeof(rDims));
		o->UpdateRect(&rDims, false);
		for(i = 0; i < nPts; i++) {
			tpts[i].x = pts[i].x+p->x;
			tpts[i].y = pts[i].y+p->y;
			UpdateMinMaxRect(&rDims, tpts[i].x, tpts[i].y);
			}
		o->ShowLine(tpts, nPts, pgLine.color);
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		free(tpts);
		}
}

void
polyline::ShowPoints(anyOutput *o)
{
	int i;

	if(nPoints >= 200) return;
	if(!pHandles) if(pHandles = (dragHandle**)calloc(nPoints+1, sizeof(dragHandle*))){
		for(i = 0; i < nPoints; i++) pHandles[i] = new dragHandle(this, DH_DATA+i);
		}
	else return;
	for(i = 0; i < nPoints; i++) if(pHandles[i]) pHandles[i]->DoPlot(o);
}


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// polygons are based on the polyline object
polygon::polygon(GraphObj *par, DataObj *d, lfPOINT *fpts, int cpts):
	polyline(par, d, fpts, cpts)
{
	Id = GO_POLYGON;
	memcpy(&pgLine, defs.pgLineDEF(0L), sizeof(LineDEF));
	type = 1;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// rectangle with absolute coordinates
rectangle::rectangle(GraphObj *par, DataObj *d, lfPOINT *p1, lfPOINT *p2):
	GraphObj(par, d)
{
	double dx = 0.0, dy = 0.0;

	FileIO(INIT_VARS);
	if(parent){
		dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
		}
	memcpy(&fp1, p1, sizeof(lfPOINT));	memcpy(&fp2, p2, sizeof(lfPOINT));
	fp1.fx -= dx;	fp1.fy -= dy;		fp2.fx -= dx;	fp2.fy -= dy;
	type = 0;
	Id = GO_RECTANGLE;
	bModified = false;
}

rectangle::rectangle(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	type = 0;
	bModified = false;
}

rectangle::~rectangle()
{
	Command(CMD_FLUSH, 0L, 0L);
	if(bModified) Undo.InvalidGO(this);
}

double
rectangle::GetSize(int select)
{
	if(parent && parent->Id== GO_GROUP){
	switch(select) {
		case SIZE_XPOS:		return fp1.fx + parent->GetSize(SIZE_XPOS);
		case SIZE_XPOS+1:	return fp2.fx + parent->GetSize(SIZE_XPOS);
		case SIZE_YPOS:		return fp1.fy + parent->GetSize(SIZE_YPOS);
		case SIZE_YPOS+1:	return fp2.fy + parent->GetSize(SIZE_YPOS);
		case SIZE_GRECT_LEFT:	case SIZE_GRECT_TOP:
		case SIZE_GRECT_RIGHT:	case SIZE_GRECT_BOTTOM:
			return parent->GetSize(select);
			}
		return 0.0;
		}
	switch(select) {
	case SIZE_XPOS:		return fp1.fx;
	case SIZE_XPOS+1:	return fp2.fx;
	case SIZE_YPOS:		return fp1.fy;
	case SIZE_YPOS+1:	return fp2.fy;
	case SIZE_GRECT_LEFT:	case SIZE_GRECT_TOP:
	case SIZE_GRECT_RIGHT:	case SIZE_GRECT_BOTTOM:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
rectangle::SetSize(int select, double value)
{
	switch(select & 0xfff) {
	case SIZE_XPOS:				fp1.fx = value;			return true;
	case SIZE_XPOS+1:			fp2.fx = value;			return true;
	case SIZE_YPOS:				fp1.fy = value;			return true;
	case SIZE_YPOS+1:			fp2.fy = value;			return true;
		}
	return false;
}

void 
rectangle::DoMark(anyOutput *o, bool mark)
{
	RECT upd;

	if(!drc) drc = new dragRect(this, 0);
	memcpy(&upd, &rDims, sizeof(RECT));
	if(mark){
		if(drc) drc->DoPlot(o);
		}
	else if(parent)	parent->DoPlot(o);
	IncrementMinMaxRect(&upd, 6);
	o->UpdateRect(&upd, false);
}

void
rectangle::DoPlot(anyOutput *o)
{
	int x1, y1, x2, y2;
	double tmp, dx, dy;

	if(!parent || !o) return;
	dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
	if(fp1.fx > fp2.fx) {
		tmp = fp2.fx;	fp2.fx = fp1.fx;	fp1.fx = tmp;
		}
	if(fp1.fy > fp2.fy) {
		tmp = fp2.fy;	fp2.fy = fp1.fy;	fp1.fy = tmp;
		}
	if(type == 2) PlotRoundRect(o);
	else {
		x1 = o->co2ix(fp1.fx+dx);			y1 = o->co2iy(fp1.fy+dy);
		x2 = o->co2ix(fp2.fx+dx);			y2 = o->co2iy(fp2.fy+dy);
		o->SetLine(&Line);				o->SetFill(&Fill);
		if(type == 1) o->oCircle(x1, y1, x2, y2, name);
		else o->oRectangle(x1, y1, x2, y2, name);
		SetMinMaxRect(&rDims, x1, y1, x2, y2);
		}
	o->MrkMode = MRK_NONE;
	if(CurrGO == this) o->ShowMark(this, MRK_GODRAW);
}

bool
rectangle::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;

	switch (cmd) {
	case CMD_FLUSH:
		if(pts) free(pts);		pts = 0L;
		if(name) free(name);	name = 0L;
		return true;
	case CMD_SAVEPOS:
		bModified = true;
		Undo.SaveLFP(this, &fp1, 0L);
		Undo.SaveLFP(this, &fp2, UNDO_CONTINUE);
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch (mev->Action) {
		case MOUSE_LBUP:
			if(IsInRect(&rDims, mev->x, mev->y) && !(CurrGO) && (o)){
				o->ShowMark(this, MRK_GODRAW);
				return true;
				}
			}
		return false;
	case CMD_SET_DATAOBJ:
		switch (type) {
		case 1:		Id = GO_ELLIPSE;		break;
		case 2:		Id = GO_ROUNDREC;		break;
		default:	Id = GO_RECTANGLE;		break;
			}
		return true;
	case CMD_MOVE:
		bModified = true;
		Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
	case CMD_UNDO_MOVE:
		fp1.fx += ((lfPOINT*)tmpl)[0].fx;	fp1.fy += ((lfPOINT*)tmpl)[0].fy;
		fp2.fx += ((lfPOINT*)tmpl)[0].fx;	fp2.fy += ((lfPOINT*)tmpl)[0].fy;
		CurrGO = this;
	case CMD_REDRAW:
		if(parent && cmd != CMD_UNDO_MOVE){
			parent->Command(CMD_REDRAW, tmpl, o);
			}
		return true;
	case CMD_SETSCROLL:
		if(parent) return parent->Command(cmd, tmpl, o);
		}
	return false;
}

void
rectangle::Track(POINT *p, anyOutput *o)
{
	POINT tpts[5];
	RECT old_rc;
	double dx, dy;

	if(o && parent){
		dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
		memcpy(&old_rc, &rDims, sizeof(rDims));
		o->UpdateRect(&rDims, false);
		tpts[0].x = tpts[1].x = tpts[4].x = o->co2ix(fp1.fx+dx)+p->x;		
		tpts[0].y = tpts[3].y = tpts[4].y = o->co2iy(fp1.fy+dy)+p->y;
		tpts[1].y = tpts[2].y = o->co2iy(fp2.fy+dy)+p->y;
		tpts[2].x = tpts[3].x = o->co2ix(fp2.fx+dx)+p->x;
		UpdateMinMaxRect(&rDims, tpts[0].x, tpts[0].y);
		UpdateMinMaxRect(&rDims, tpts[2].x, tpts[2].y);	
		if(old_rc.left != rDims.left || old_rc.right != rDims.right || old_rc.top !=
			rDims.top || old_rc.bottom != rDims.bottom)IncrementMinMaxRect(&rDims, 3);
		o->ShowLine(tpts, 5, Line.color);
		if(type == 1) o->ShowEllipse(tpts[0], tpts[2], Line.color);
		}
}

void *
rectangle::ObjThere(int x, int y)
{
	if(drc) return drc->ObjThere(x, y);
	return 0L;
}

//use circular Bresenham's algorithm to draw rounded rectangles
//Ref: C. Montani, R. Scopigno (1990) "Speres-To-Voxel Conversion", in:
//   Graphic Gems (A.S. Glassner ed.) Academic Press, Inc.; 
//   ISBN 0-12-288165-5 
void
rectangle::PlotRoundRect(anyOutput *o)
{
	int i, m, x, y, di, de, lim, ir, x1, x2, y1, y2;
	double dx, dy;
	POINT np;

	dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
	ir = o->un2ix(rad);
	x1 = o->co2ix(fp1.fx+dx);			y1 = o->co2iy(fp1.fy+dy);
	x2 = o->co2ix(fp2.fx+dx);			y2 = o->co2iy(fp2.fy+dy);
	if (x1 > x2) Swap(x1, x2);		if(y1 > y2) Swap(y1, y2);
	if(pts) free(pts);				nPts = 0;
	m = ir*4+10;
	if(!(pts = (POINT*)malloc(m*sizeof(POINT))))return;
	for(i = 0; i < 4; i++) {
		x = lim = 0;	y = ir;	di = 2*(1-ir);
		while (y >= lim){
			if(di < 0) {
				de = 2*di + 2*y -1;
				if(de > 0) {
					x++;	y--;	di += (2*x -2*y +2);
					}
				else {
					x++;	di += (2*x +1);
					}
				}
			else {
				de = 2*di -2*x -1;
				if(de > 0) {
					y--;	di += (-2*y +1);
					}
				else {
					x++;	y--;	di += (2*x -2*y +2);
					}
				}
			switch(i) {
			case 0:
				np.x = x2-ir+x;		np.y = y2-ir+y;
				break;
			case 1:
				np.x = x2-ir+y;		np.y = y1+ir-x;
				break;
			case 2: 
				np.x = x1+ir-x;		np.y = y1+ir-y;
				break;
			case 3:
				np.x = x1+ir-y;		np.y = y2-ir+x;
				break;
				}
			AddToPolygon(&nPts, pts, &np);
			}
		}
	AddToPolygon(&nPts, pts, pts);	//close polygon
	o->SetLine(&Line);				o->SetFill(&Fill);
	o->oPolygon(pts, nPts, name);
	SetMinMaxRect(&rDims, x1, y1, x2, y2);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// ellipse with absolute coordinates
ellipse::ellipse(GraphObj *par, DataObj *d, lfPOINT *p1, lfPOINT *p2)
	:rectangle(par, d, p1, p2)
{
	type = 1;
	Id = GO_ELLIPSE;
}

ellipse::ellipse(int src)
	:rectangle(src)
{
	type = 1;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// rounded rectangle
roundrec::roundrec(GraphObj *par, DataObj *d, lfPOINT *p1, lfPOINT *p2)
	:rectangle(par, d, p1, p2)
{
	type = 2;
	Id = GO_ROUNDREC;
}

roundrec::roundrec(int src)
	:rectangle(src)
{
	type = 2;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Add a legend to the graph
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
LegItem::LegItem(GraphObj *par, DataObj *d, LineDEF *ld, LineDEF *lf, FillDEF *fd)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	if(!ld && !fd && lf) ld = lf;
	if(ld) {
		memcpy(&DataLine, ld, sizeof(LineDEF));
		flags |= 0x01;
		}
	if(lf) memcpy(&OutLine, lf, sizeof(LineDEF));
	if(fd) {
		if(fd->hatch) memcpy(&HatchLine, fd->hatch, sizeof(LineDEF));
		memcpy(&Fill, fd, sizeof(FillDEF));
		Fill.hatch = &HatchLine;
		flags |= 0x02;
		}
	DefDesc(0L);		Id = GO_LEGITEM;		moveable = true;
}

LegItem::LegItem(GraphObj *par, DataObj *d, LineDEF *ld, Symbol *sy)
	:GraphObj(par, d)
{
	FileIO(INIT_VARS);
	if(ld) {
		memcpy(&DataLine, ld, sizeof(LineDEF));		flags |= 0x01;
		}
	if(sy) {
		Sym = sy;		Sym->parent = this;			flags |= 0x04;
		}
	DefDesc(0L);		Id = GO_LEGITEM;		moveable = true;
}

LegItem::LegItem(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	moveable = true;
}

LegItem::~LegItem()
{
	if(Sym) DeleteGO(Sym);		Sym = 0L;
	if(Desc) DeleteGO(Desc);	Desc = 0L;
}

double
LegItem::GetSize(int select)
{
	switch(select) {
	case SIZE_XCENTER:
		return (parent->GetSize((flags & 0x01) ? SIZE_XPOS+2 : SIZE_XPOS) + 
			parent->GetSize((flags & 0x01) ? SIZE_XPOS+3 : SIZE_XPOS+1))/2.0;
	case SIZE_YCENTER:
		return (parent->GetSize(SIZE_YPOS) + parent->GetSize(SIZE_YPOS+1))/2.0;
	case SIZE_LB_XPOS:		case SIZE_LB_YPOS:
	case SIZE_GRECT_TOP:	case SIZE_GRECT_LEFT:
	default:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

void
LegItem::DoPlot(anyOutput *o)
{
	POINT pts[2];

	if(!parent || !o) return;
	hcr.left = iround(parent->GetSize((flags & 0x01) ? SIZE_XPOS+2 : SIZE_XPOS));
	hcr.right = iround(parent->GetSize((flags & 0x01) ? SIZE_XPOS+3 :SIZE_XPOS+1));
	hcr.top = iround(parent->GetSize(SIZE_YPOS));
	hcr.bottom = iround(parent->GetSize(SIZE_YPOS+1));
	SetMinMaxRect(&rDims, hcr.left, hcr.top, hcr.right, hcr.bottom);
	if(flags & 0x02){
		o->SetLine(&OutLine);	o->SetFill(&Fill);
		o->oRectangle(hcr.left, hcr.top, hcr.right, hcr.bottom, name);
		}
	if(flags & 0x01){
		pts[0].x = hcr.left;	pts[1].x = hcr.right;
		pts[0].y = pts[1].y = iround(GetSize(SIZE_YCENTER))+1;
		o->SetLine(&DataLine);	o->oPolyline(pts, 2, 0L);
		}
	if(flags & 0x04){
		if(Sym) Sym->DoPlot(o);
		}
	if(Desc) {
		Desc->moveable = false;	Desc->DoPlot(o);
		if(Desc->rDims.bottom > rDims.bottom){
			parent->SetSize(SIZE_YPOS+1, (double)Desc->rDims.bottom);
			}
		UpdateMinMaxRect(&rDims, Desc->rDims.left, Desc->rDims.top);
		UpdateMinMaxRect(&rDims, Desc->rDims.right, Desc->rDims.bottom);
		}
}

void
LegItem::DoMark(anyOutput *o, bool mark)
{
	RECT cr;
	LineDEF ld = {0.0, 1.0, 0x00000000L, 0x00000000L};
	POINT pts[5];

	if(!parent || !o) return;
	cr.left = hcr.left-5;			cr.right = hcr.right+3;
	cr.top = hcr.top-3;				cr.bottom = hcr.bottom+1;
	ld.color = mark ? 0x00000000L : 0x00ffffffL;	o->SetLine(&ld);
	pts[0].x = pts[3].x = pts[4].x = cr.left;	pts[0].y = pts[1].y = pts[4].y = cr.top;
	pts[1].x = pts[2].x = cr.right;				pts[2].y = pts[3].y = cr.bottom;
	o->oPolyline(pts, 5, name);					IncrementMinMaxRect(&cr, 3);
	o->UpdateRect(&cr, false);
}

bool
LegItem::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmpPlots;

	switch(cmd){
	case CMD_MOUSE_EVENT:
		if(tmpl && IsInRect(&rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y) && o) {
			if(Desc && Desc->Command(cmd, tmpl, o)) return true;
			if(!CurrGO) o->ShowMark(CurrGO=this, MRK_GODRAW);
			}
		break;
	case CMD_HIDE_MARK:
		if(!tmpl || !o) return false;
		if(Desc && Desc == (void*)tmpl) {
			Desc->DoMark(o, false);
			return true;
			}
		return false;
	case CMD_REDRAW:	case CMD_MOVE:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_MUTATE:
		if(!parent || !(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Desc == tmpPlots[0]) {
			Undo.MutateGO((GraphObj**)&Desc, tmpPlots[1], 0L, o);
			return true;
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_LEGITEM;
		data = (DataObj *)tmpl;
		return true;
		}
	return false;
}

void
LegItem::Track(POINT *p, anyOutput *o)
{
	if(parent) parent->Track(p, o);
}

bool
LegItem::HasFill(LineDEF *ld, FillDEF *fd)
{
	if(ld && cmpLineDEF(ld, &OutLine)) return false;
	if(fd && cmpFillDEF(fd, &Fill)) return false;
	if(fd && fd->hatch && cmpLineDEF(fd->hatch, &HatchLine)) return false;
	return true;
}

bool
LegItem::HasSym(LineDEF *ld, Symbol *sy)
{
	if(sy && !Sym) return false;
	if(sy && Sym && sy->Id == GO_SYMBOL) {
		if((Sym->type & 0xfff) != (Sym->type & 0xfff)) return false;
		if(Sym->GetSize(SIZE_SYMBOL) != sy->GetSize(SIZE_SYMBOL)) return false;
		if(Sym->GetSize(SIZE_SYM_LINE) != sy->GetSize(SIZE_SYM_LINE)) return false;
		if(Sym->GetColor(COL_SYM_LINE) != sy->GetColor(COL_SYM_LINE)) return false;
		if(Sym->GetColor(COL_SYM_FILL) != sy->GetColor(COL_SYM_FILL)) return false;
		}
	if(ld && cmpLineDEF(ld, &DataLine)) return false;
	return true;
}

void
LegItem::DefDesc(char *txt)
{
	TextDEF td;

	td.ColTxt = 0x00000000L;		td.ColBg = 0x00ffffffL;
	td.fSize = defs.GetSize(SIZE_TICK_LABELS);
	td.RotBL = td.RotCHAR = 0.0;
	td.iSize = 0;					td.Align = TXA_VTOP | TXA_HLEFT;
	td.Mode = TXM_TRANSPARENT;		td.Style = TXS_NORMAL;
	td.Font = FONT_HELVETICA;		td.text = txt ? strdup(txt) : strdup("text");
	Desc = new Label(this, data, 0, 0, &td, LB_X_PARENT | LB_Y_PARENT);
}

Legend::Legend(GraphObj *par, DataObj *d):GraphObj(par, d)
{
	FileIO(INIT_VARS);		Id = GO_LEGEND;		moveable = true;
}

Legend::Legend(int src):GraphObj(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
	moveable = true;
}

Legend::~Legend()
{
	int i;

	if(Items) {
		for(i = 0; i< nItems; i++) if(Items[i]) DeleteGO(Items[i]);
		free(Items);	Items = 0L;
		}
	if(to) DelBitmapClass(to);		to = 0L;
}
double
Legend::GetSize(int select)
{
	switch(select) {
	case SIZE_XPOS:		return C_Rect.Xmin;
	case SIZE_XPOS+1:	return C_Rect.Xmax;
	case SIZE_XPOS+2:	return E_Rect.Xmin;
	case SIZE_XPOS+3:	return E_Rect.Xmax;
	case SIZE_YPOS:		return C_Rect.Ymin;
	case SIZE_YPOS+1:	return C_Rect.Ymax;
	case SIZE_LB_XPOS:	return lb_pos.fx;
	case SIZE_LB_YPOS:	return lb_pos.fy;
	case SIZE_GRECT_TOP:	case SIZE_GRECT_LEFT:
		if(parent) return parent->GetSize(select);
		break;
		}
	return 0.0;
}

bool
Legend::SetSize(int select, double value)
{
	double tmp;

	switch (select & 0xfff){
	case SIZE_XPOS:		pos.fx = value;		return true;
	case SIZE_YPOS:		pos.fy = value;		return true;
	case SIZE_YPOS+1:
		tmp = value - C_Rect.Ymax;
		C_Rect.Ymin += tmp;		C_Rect.Ymax += tmp;		lb_pos.fy +=tmp;
		}
	return false;
}

void
Legend::DoPlot(anyOutput *o)
{
	int i;
	double y_inc, dx, dy;

	if(!o || !Items) return;
	if(to) DelBitmapClass(to);		to = 0L;
	dx = parent->GetSize(SIZE_GRECT_LEFT);		dy = parent->GetSize(SIZE_GRECT_TOP);
	C_Rect.Xmin = o->co2fix(pos.fx + B_Rect.Xmin + D_Rect.Xmin + dx);
	C_Rect.Ymin = o->co2fiy(pos.fy + B_Rect.Ymin + D_Rect.Ymin + dy);
	C_Rect.Xmax = C_Rect.Xmin + o->un2fix(D_Rect.Xmax - D_Rect.Xmin);
	C_Rect.Ymax = C_Rect.Ymin + o->un2fix(D_Rect.Ymax - D_Rect.Ymin);
	E_Rect.Ymin = C_Rect.Ymin;	E_Rect.Ymax = C_Rect.Ymax;
	E_Rect.Xmin = o->co2fix(pos.fx + B_Rect.Xmin + F_Rect.Xmin + dx);
	E_Rect.Xmax = E_Rect.Xmin + o->un2fix(F_Rect.Xmax - F_Rect.Xmin);
	y_inc = floor(0.5+o->un2fiy(B_Rect.Ymax - B_Rect.Ymin));
	rDims.left = rDims.right = rDims.top = rDims.bottom = 0;
	lb_pos.fx = o->co2fix(pos.fx + B_Rect.Xmax + dx);	lb_pos.fy = C_Rect.Ymin;
	//draw all items
	for(i = 0; i < nItems; i++) {
		if(Items[i]){
			Items[i]->DoPlot(o);
			if(rDims.left == rDims.right || rDims.top == rDims.bottom){
				rDims.left = Items[i]->rDims.left;	rDims.right = Items[i]->rDims.right;
				rDims.top = Items[i]->rDims.top;	rDims.bottom = Items[i]->rDims.bottom;
				}
			else {
				UpdateMinMaxRect(&rDims, Items[i]->rDims.left, Items[i]->rDims.top);
				UpdateMinMaxRect(&rDims, Items[i]->rDims.right, Items[i]->rDims.bottom);
				}
			C_Rect.Ymin += y_inc;		C_Rect.Ymax += y_inc;
			lb_pos.fy += y_inc;
			}
		}
	IncrementMinMaxRect(&rDims, 6);
}

void
Legend::DoMark(anyOutput *o, bool mark)
{
	RECT cr;
	LineDEF ld = {0.0, 1.0, 0x00c0c0c0L, 0x00000000L};
	POINT pts[5];

	if(!parent || !o) return;
	cr.left = rDims.left;			cr.right = rDims.right;
	cr.top = rDims.top;				cr.bottom = rDims.bottom;
	ld.color = mark ? 0x00c0c0c0L : 0x00ffffffL;	o->SetLine(&ld);
	pts[0].x = pts[3].x = pts[4].x = cr.left;	pts[0].y = pts[1].y = pts[4].y = cr.top;
	pts[1].x = pts[2].x = cr.right;				pts[2].y = pts[3].y = cr.bottom;
	o->oPolyline(pts, 5, name);					IncrementMinMaxRect(&cr, 3);
	o->UpdateRect(&cr, false);
}

bool
Legend::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;

	switch(cmd){
	case CMD_MOUSE_EVENT:
		if(o && tmpl && IsInRect(&rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)) {
			if(Items) for(i = 0; i< nItems; i++) 
				if(Items[i] && Items[i]->Command(cmd, tmpl, o)) return true;
			if(!CurrGO) o->ShowMark(CurrGO = this, MRK_GODRAW);
			}
		break;
	case CMD_SET_DATAOBJ:
		if(Items) for(i = 0; i < nItems; i++) if(Items[i]) Items[i]->Command(cmd, tmpl, o);
		Id = GO_LEGEND;
		data = (DataObj *)tmpl;
		return true;
	case CMD_HIDE_MARK:
		if(!tmpl || !o) return false;
		if(Items && nItems) for(i = 0; i < nItems; i++) {
			if(Items[i]) {
				if(tmpl == (void*)Items[i]) {
					Items[i]->DoMark(o, false);
					return true;
					}	
				else if(Items[i]->Command(cmd, tmpl, o)) return true;
				}
			}
		return false;
	case CMD_DELOBJ:
		o->HideMark();
		if(Items && parent) for(i = 0; i < nItems; i++) {
			if(Items[i] && tmpl == (void *)Items[i]) {
				Undo.DeleteGO((GraphObj**)(&Items[i]), 0L, o);
				parent->Command(CMD_REDRAW, NULL, o);
				return true;
				}
			}
		break;
	case CMD_MOVE:
		Undo.MoveObj(this, (lfPOINT*)tmpl, 0L);
	case CMD_UNDO_MOVE:
		pos.fx += ((lfPOINT*)tmpl)[0].fx;	pos.fy += ((lfPOINT*)tmpl)[0].fy;
		CurrGO = this;
	case CMD_REDRAW:
		if(parent && cmd != CMD_UNDO_MOVE){
			parent->Command(CMD_REDRAW, tmpl, o);
			}
		return true;
	case CMD_DROP_OBJECT:
		if(!tmpl) return false;
		if(!(Items = (LegItem**)realloc(Items, (2+nItems)*sizeof(LegItem*))))return false;
		Items[nItems++] = (LegItem*)tmpl;
		Items[nItems-1]->parent = this;
		return true;
		}
	return false;
}

void
Legend::Track(POINT *p, anyOutput *o)
{
	POINT pts[5];
	LineDEF tld = {0.0, 1.0, 0x00c0c0c0, 0x0L};

	if(!p || !o) return;
	if(to) {
		o->CopyBitmap(trc.left, trc.top, to, 0, 0, trc.right - trc.left, 
			trc.bottom - trc.top, false);
		DelBitmapClass(to);		to = 0L;
		o->UpdateRect(&trc, false);
		}
	trc.left = pts[0].x = pts[1].x = pts[4].x = rDims.left + p->x;		
	trc.top = pts[0].y = pts[3].y = pts[4].y = rDims.top + p->y;
	trc.bottom = pts[1].y = pts[2].y = rDims.bottom + p->y;
	trc.right = pts[2].x = pts[3].x = rDims.right + p->x;
	IncrementMinMaxRect(&trc, 3);	to = GetRectBitmap(&trc, o);
	o->SetLine(&tld);				o->oPolyline(pts, 5, 0L);
	o->UpdateRect(&trc, false);
} 

bool
Legend::HasFill(LineDEF *ld, FillDEF *fd)
{
	int i;
	LegItem *li;

	if(Items) for(i = 0; i < nItems; i++) {
		if(Items[i] && Items[i]->HasFill(ld, fd)) return true;
		}
	if(li = new LegItem(this, data, 0L, ld, fd)){
		if(!(Command(CMD_DROP_OBJECT, li, 0L))) DeleteGO(li);
		}
	return false;
}

bool
Legend::HasSym(LineDEF *ld, Symbol *sy)
{
	int i;
	Symbol *ns;
	LegItem *li;

	if(!parent || !sy) return true;
	if(Items) for(i = 0; i < nItems; i++) {
		if(Items[i] && Items[i]->HasSym(ld, sy)) return true;
		}
	if(!(ns = new Symbol(this, data, 0.0, 0.0, sy->type | SYM_POS_PARENT))) return true;
	ns->SetSize(SIZE_SYMBOL, sy->GetSize(SIZE_SYMBOL));
	ns->SetSize(SIZE_SYM_LINE, sy->GetSize(SIZE_SYM_LINE));
	ns->SetColor(COL_SYM_LINE, sy->GetColor(COL_SYM_LINE));
	ns->SetColor(COL_SYM_FILL, sy->GetColor(COL_SYM_FILL));
	if(li = new LegItem(this, data, ld, ns)){
		if(!(Command(CMD_DROP_OBJECT, li, 0L))) DeleteGO(li);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Graphs are graphic objects containing plots, axes, and drawn objects
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Graph::Graph(GraphObj *par, DataObj *d, anyOutput *o):GraphObj(par, d)
{
	Graph::FileIO(INIT_VARS);
	Disp = o;		Id = GO_GRAPH;		cGraphs++;	bModified = true;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

Graph::Graph(int src):GraphObj(0L, 0L)
{
	int i;

	Graph::FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		x_axis.owner = y_axis.owner = (void *)this;
		//do all axes
		for(i = 0; Axes && i< NumAxes; i++) if(Axes[i]) Axes[i]->parent = this;
		//do all plots
		for(i = 0; Plots && i< NumPlots; i++) if(Plots[i]) Plots[i]->parent = this;
		if(x_axis.max > x_axis.min && y_axis.max > y_axis.min &&
			Bounds.Xmax > Bounds.Xmin && Bounds.Ymax > Bounds.Ymin) dirty = false;
		}
	cGraphs++;		bModified = false;
}

Graph::~Graph()
{
	int i;

	Undo.InvalidGO(this);
	DoZoom("reset");
	if(CurrGraph == this) CurrGraph = 0L;
	if(Plots) {
		for(i = 0; i< NumPlots; i++) if(Plots[i]) DeleteGO(Plots[i]);
		free(Plots);
		}
	if(Axes) {
		for(i = 0; i< NumAxes; i++) if(Axes[i]) delete Axes[i];
		free(Axes);
		}
	if(OwnDisp && Disp) DelDispClass(Disp);
	if(frm_g) DeleteGO(frm_g);		if(frm_d) DeleteGO(frm_d);
	if(x_axis.breaks && x_axis.owner == this) free(x_axis.breaks);
	if(y_axis.breaks && y_axis.owner == this) free(y_axis.breaks);
	if(tl_pts) free(tl_pts);
	if(nscp > 0 && nscp <= NumPlots && Sc_Plots) free(Sc_Plots);
	nscp = 0;			Sc_Plots = 0L;
	if(name) free(name);		name = 0L;
	if(filename) free(filename);	filename= 0L;
}

double
Graph::GetSize(int select)
{
	switch(select) {
	case SIZE_LB_XDIST:
	case SIZE_LB_YDIST:			return 0.0f;
	case SIZE_GRECT_TOP:		return GRect.Ymin;
	case SIZE_GRECT_BOTTOM:		return GRect.Ymax;
	case SIZE_GRECT_LEFT:		return GRect.Xmin;
	case SIZE_GRECT_RIGHT:		return GRect.Xmax;
	case SIZE_DRECT_TOP:		return DRect.Ymin;
	case SIZE_DRECT_BOTTOM:		return DRect.Ymax;
	case SIZE_DRECT_LEFT:		return DRect.Xmin;
	case SIZE_DRECT_RIGHT:		return DRect.Xmax;
	case SIZE_BOUNDS_XMIN:		return Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		return Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		return Bounds.Ymin;
	case SIZE_BOUNDS_YMAX:		return Bounds.Ymax;
	case SIZE_BOUNDS_LEFT:		return x_axis.flags & AXIS_INVERT ? x_axis.max : x_axis.min;
	case SIZE_BOUNDS_RIGHT:		return x_axis.flags & AXIS_INVERT ? x_axis.min : x_axis.max;
	case SIZE_BOUNDS_TOP:		return y_axis.flags & AXIS_INVERT ? y_axis.min : y_axis.max;
	case SIZE_BOUNDS_BOTTOM:	return y_axis.flags & AXIS_INVERT ? y_axis.max : y_axis.min;
	case SIZE_YAXISX:
		if(y_axis.flags & AXIS_X_DATA) return CurrDisp->fx2fix(y_axis.loc[0].fx);
		else return CurrDisp->co2fix(y_axis.loc[0].fx);
	case SIZE_XAXISY:
		if(x_axis.flags & AXIS_Y_DATA) return CurrDisp->fy2fiy(x_axis.loc[0].fy);
		else return CurrDisp->co2fiy(x_axis.loc[0].fy);
	default:		return defs.GetSize(select);
		}
}

bool
Graph::SetSize(int select, double val)
{
	switch(select & 0xfff) {
	case SIZE_GRECT_TOP:	GRect.Ymin = val;	return true;
	case SIZE_GRECT_BOTTOM:	GRect.Ymax = val;	return true;
	case SIZE_GRECT_LEFT:	GRect.Xmin = val;	return true;
	case SIZE_GRECT_RIGHT:	GRect.Xmax = val;	return true;
	case SIZE_DRECT_TOP:	DRect.Ymin = val;	return true;
	case SIZE_DRECT_BOTTOM:	DRect.Ymax = val;	return true;
	case SIZE_DRECT_LEFT:	DRect.Xmin = val;	return true;
	case SIZE_DRECT_RIGHT:	DRect.Xmax = val;	return true;
	default: return false;
		}
}

DWORD
Graph::GetColor(int select)
{
	switch(select & 0xfff) {
	case COL_AXIS:		return ColAX;
	case COL_BG:		return ColDR;
		}
	if(parent) return parent->GetColor(select);
	else return defs.Color(select);
}


void
Graph::DoPlot(anyOutput *target)
{
	int i;
	AxisDEF *ax;

	if(nscp > 0 && nscp <= NumPlots && Sc_Plots) free(Sc_Plots);
	nscp = 0;		Sc_Plots = 0L;
	rc_mrk.left = rc_mrk.right = rc_mrk.top = rc_mrk.bottom = -1;
	CurrAxes = Axes;
	if(data) do_formula(data, 0L, 0L);		//init mfcalc
	//verify ownership of axes
	if(Axes) for (i = 0; i < NumAxes; i++){
		if(Axes[i] && (ax = Axes[i]->GetAxis()))
			if(ax == &x_axis || ax == &y_axis) ax->owner = this;
		}
	if(!name){
		sprintf(TmpTxt, "Graph %d", cGraphs);
		name = strdup(TmpTxt);
		}
	if(!target && !Disp) {
		Disp = NewDispClass(this);
		Disp->SetMenu(MENU_GRAPH);
		if(name) Disp->Caption(name);
		Disp->VPorg.fy = (double)Disp->MenuHeight;
		Disp->CheckMenu(ToolMode, true);
		OwnDisp = true;						defs.SetDisp(Disp);
		if(GRect.Xmin > 0.0001 || GRect.Xmin < -0.0001) {
			GRect.Xmax -= GRect.Xmin;	GRect.Xmin = 0.0;
			}
		if(GRect.Ymin > 0.0001 || GRect.Ymin < -0.0001) {
			GRect.Ymax -= GRect.Ymin;	GRect.Ymin = 0.0;
			}
		}
	//the first output class is the display class
	if(!Disp && (!(Disp = target))) return;
	Disp->ActualSize(&defs.clipRC);
	if(OwnDisp) Disp->Erase(Disp->dFillCol = defs.Color(COL_BG));
	CurrDisp = target ? target : Disp;
	CurrDisp->MrkMode = MRK_NONE;
	CurrRect.Xmin=GRect.Xmin + DRect.Xmin;	CurrRect.Xmax=GRect.Xmin + DRect.Xmax;
	CurrRect.Ymin=GRect.Ymin + DRect.Ymax;	CurrRect.Ymax=GRect.Ymin + DRect.Ymin;
	if(dirty) DoAutoscale();
	CurrDisp->SetRect(CurrRect, units, &x_axis, &y_axis);
	if(!frm_g && !(frm_g = new FrmRect(this, 0L, &GRect, 0L))) return;
	frm_g->SetColor(COL_GRECT, ColGR);	frm_g->SetColor(COL_GRECTLINE, ColGRL);
	frm_g->DoPlot(CurrDisp);
	if(type == GT_STANDARD) {
		if(!frm_d && !(frm_d = new FrmRect(this, &GRect, &DRect, 0L))) return;
		SetMinMaxRect(&rDims, CurrDisp->co2ix(CurrRect.Xmin), CurrDisp->co2iy(CurrRect.Ymax), 
			CurrDisp->co2ix(CurrRect.Xmax), CurrDisp->co2iy(CurrRect.Ymin));
		frm_g->Command(CMD_MINRC, &rDims, CurrDisp);
		SetMinMaxRect(&rDims, CurrDisp->co2ix(GRect.Xmin), CurrDisp->co2iy(GRect.Ymax), 
			CurrDisp->co2ix(GRect.Xmax), CurrDisp->co2iy(GRect.Ymin));
		frm_d->Command(CMD_MAXRC, &rDims, CurrDisp);
		frm_d->SetColor(COL_DRECT, ColDR);		frm_d->DoPlot(CurrDisp);
		frm_g->Command(CMD_SETCHILD, &DRect, CurrDisp);
		}
	//do all axes
	if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]){
		Axes[i]->SetColor(COL_BG, ColGR);
		Axes[i]->DoPlot(CurrDisp);
		}
	//do all plots
	if(Plots) for(i = 0; i < NumPlots; i++) if(Plots[i]) {
		if(Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH) {
			if(((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(CurrDisp);
			}
		else {
			Plots[i]->DoPlot(CurrDisp);
			}
		}
	if(target && ToolMode == TM_STANDARD) target->MouseCursor(MC_ARROW, false);
}

bool
Graph::Command(int cmd, void *tmpl, anyOutput *o)
{
	MouseEvent *mev;
	GraphObj **tmpPlots;
	char *f_name;
	RECT rc;
	int i, j;
	DWORD delflg = 0L;

	if(!o) o = CurrDisp;
	switch (cmd){
	case CMD_CAN_DELETE:
		if(bModified) {
			sprintf(TmpTxt, "%s has not been saved.\nDo you want to save it now?", name);
			if(YesNoBox(TmpTxt)) SaveGraphAs(this);
			}
		bModified = false;
		return true;
	case CMD_LAYERS:
		return ShowLayers(this);
	case CMD_SAVEPOS:
		Undo.ValRect(this, &GRect, 0L);
		Undo.ValRect(this, &DRect, UNDO_CONTINUE);
		return true;
	case CMD_LEGEND:
		if(Id == GO_PAGE) {
			if(CurrGraph && CurrGraph->parent == this) return CurrGraph->Command(cmd, tmpl, o);
			if(!CurrGraph && NumPlots == 1 && Plots[0] && Plots[0]->Id == GO_GRAPH){
				CurrGraph = (Graph*)Plots[0];
				return CurrGraph->Command(cmd, tmpl, o);
				}
			InfoBox("No graph selected!\nCreate a new graph first or select\n"
				"a graph before you can add a legend.");
			return false;
			}
		if(Id == GO_GRAPH && !tmpl && Plots) {
			for(i = 0; i< NumPlots; i++){
				if(Plots[i] && Plots[i]->Id == GO_LEGEND) {
					tmpl = (void*)Plots[i];
					Undo.ObjConf(Plots[i], 0L);
					break;
					}
				}
			if(!tmpl) {
				if(!(tmpl = (void*) new Legend(this, data)))return false;
				tmpPlots = (GraphObj**)memdup(Plots, sizeof(GraphObj*) * (NumPlots+2), 0);
				if(!tmpPlots) return false;
				Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
				Undo.SetGO(this, &tmpPlots[NumPlots++], (Plot *)tmpl, 0L);
				free(Plots);		Plots = tmpPlots;	bModified = true;
				}
			if(type == GT_CIRCCHART)((Legend*)tmpl)->SetSize(SIZE_XPOS, DRect.Xmin*3.0);
			for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl,o);
			Command(CMD_REDRAW, 0L, CurrDisp);
			}
		return true;
	case CMD_REPL_GO:
		if(!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Axes) for(i = 0; i < NumAxes; i++) if(Axes[i] && Axes[i] == tmpPlots[0]){
			tmpPlots[1]->parent = this;
			tmpPlots[1]->Command(CMD_SET_DATAOBJ, data, o);
			Axes[i] = (Axis *)tmpPlots[1];
			tmpPlots[0]->parent = 0L;		//disable messaging
			//check for default axes
			if(((Axis*)tmpPlots[0])->GetAxis() == &x_axis) {
				if(x_axis.breaks) free(x_axis.breaks);
				memcpy(&x_axis, Axes[i]->axis, sizeof(AxisDEF));
				if(x_axis.owner == Axes[i]) free(Axes[i]->axis);
				Axes[i]->axis = &x_axis;			x_axis.owner = this;
				}
			else if(((Axis*)tmpPlots[0])->GetAxis() == &y_axis) {
				if(y_axis.breaks) free(y_axis.breaks);
				memcpy(&y_axis, Axes[i]->axis, sizeof(AxisDEF));
				if(y_axis.owner == Axes[i]) free(Axes[i]->axis);
				Axes[i]->axis = &y_axis;			y_axis.owner = this;
				}
			DeleteGO(tmpPlots[0]);
			return bModified = dirty = true;
			}
		if(Plots) for(i = 0; i < NumPlots; i++) if(Plots[i] && Plots[i] == tmpPlots[0]) { 
			return bModified = dirty = ReplaceGO((GraphObj**)&Plots[i], tmpPlots);
			}
		return false;
	case CMD_MUTATE:
		if(!parent || !(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Plots) for (i = 0; i < NumPlots; i++) if(Plots[i] && Plots[i] == tmpPlots[0]) {
			Undo.MutateGO((GraphObj**)&Plots[i], tmpPlots[1], 0L, o);
			if(ToolMode) Command(CMD_TOOLMODE, 0L, o);
			return bModified = true;
			}
		break;
	case CMD_HIDE_MARK:
		if(!tmpl || !o) return false;
		//do frame rectangles
		if(frm_g && tmpl == (void*)frm_g) {
			frm_g->DoMark(o, false);					return true;
			}
		if(frm_d && tmpl == (void*)frm_d) {
			frm_d->DoMark(o, false);					return true;
			}
		//do all axes
		if(Axes)for(i = NumAxes-1; i>=0; i--) {
			if(tmpl == (void*)Axes[i]){
				Axes[i]->DoMark(CurrDisp, false);		return true;
				}
			else if(Axes[i]->Id == GO_AXIS) {
				if(Axes[i]->Command(cmd, tmpl, o))		return true;
				}
			}
		//do all plots
		if(Plots)for(i = NumPlots-1; i>=0; i--) {
			if(tmpl == (void*)Plots[i]){
				Plots[i]->DoMark(CurrDisp, false);		return true;
				}
			else if(Plots[i] && (Plots[i]->Id == GO_MLABEL || Plots[i]->Id == GO_LEGEND || 
				(Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH))) {
				if(Plots[i]->Command(cmd, tmpl, o))		return true;
				}
			}
		return false;
	case CMD_REDRAW:
		if(!CurrDisp) {
			DoPlot(CurrDisp);
			return true;
			}
		if(parent && parent->Id == GO_PAGE) return parent->Command(cmd, tmpl, o);
		if(CurrDisp && CurrDisp->Erase(ColBG)) CurrDisp->StartPage();
		CurrDisp->MrkMode = MRK_NONE;
		DoPlot(CurrDisp);
		if(CurrDisp) CurrDisp->EndPage();
		if(CurrGO && CurrGO == CurrLabel && CurrLabel->Id == GO_LABEL)
			CurrDisp->ShowMark(CurrLabel, MRK_GODRAW);
		return true;
	case CMD_ZOOM:
		return DoZoom((char*)tmpl);
	case CMD_MOUSECURSOR:
		if(Disp)Disp->MouseCursor(MC_ARROW, false);	
		return true;
	case CMD_AXIS:			//one of the plots has changed scaling: reset
		if(o)o->SetRect(CurrRect, units, &x_axis, &y_axis);
		return true;
	case CMD_REG_AXISPLOT:	//notification: plot can hnadle its own axes
		if(nscp > 0 && nscp <= NumPlots && Sc_Plots)  {
			for(i = 0; i < nscp; i++)
				if(Sc_Plots[i] == (GraphObj*)tmpl) return true;
			if(tmpPlots = (GraphObj**)realloc(Sc_Plots, (nscp+1)*sizeof(GraphObj*))){
				tmpPlots[nscp++] = (GraphObj *)tmpl;
				Sc_Plots = tmpPlots;
				}
			else {		//memory allocation error
				nscp = 0;
				Sc_Plots = 0L;
				}
			}
		else {
			if(Sc_Plots = (GraphObj **)calloc(1, sizeof(GraphObj*))){
				Sc_Plots[0] = (GraphObj *)tmpl;
				nscp = 1;
				}
			else nscp = 0;
			}
		return true;
	case CMD_BUSY:
		if(Disp) Disp->MouseCursor(MC_WAIT, true);
		break;
	case CMD_UNDO:
		Command(CMD_TOOLMODE, 0L, o);
		if(CurrDisp) CurrDisp->MouseCursor(MC_WAIT, true);
		Undo.Restore(true, CurrDisp);
		if(CurrDisp) CurrDisp->MouseCursor(MC_ARROW, true);
		return true;
	case CMD_ADDAXIS:
		Command(CMD_TOOLMODE, 0L, o);
		if(Id == GO_PAGE) {
			if(CurrGraph && CurrGraph->parent == this) return CurrGraph->Command(cmd, tmpl, o);
			if(!CurrGraph && NumPlots == 1 && Plots[0] && Plots[0]->Id == GO_GRAPH){
				CurrGraph = (Graph*)Plots[0];
				return CurrGraph->Command(cmd, tmpl, o);
				}
			InfoBox("No graph selected!\nCreate a new graph first or select\n"
				"a graph before you can add an axis.");
			return false;
			}
		else {
			if(type == GT_3D && Plots){
				for(i = 0; i < NumPlots; i++)
					if(Plots[i] && Plots[i]->Id == GO_PLOT3D) return Plots[i]->Command(cmd, tmpl, o); 
				}
			else if(AddAxis()) {
				Command(CMD_REDRAW, tmpl, o);
				return true;
				}
			}
		return false;
	case CMD_CONFIG:
		Command(CMD_TOOLMODE, 0L, o);
		return Configure();
	case CMD_FILENAME:
		if(tmpl) {
			if(filename) free(filename);
			filename=strdup((char*)tmpl);
			}
		break;
	case CMD_SETNAME:
		if(OwnDisp && CurrDisp && tmpl){
			CurrDisp->Caption((char*)tmpl);
			if(name) free(name);	name = strdup((char*)tmpl);
			}
		else return false;
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_GRAPH;
		data = (DataObj *)tmpl;
		//do axes
		if(Axes) for(i = 0; i< NumAxes; i++) if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		//do all plots
		if(Plots) for(i = 0; i< NumPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl,o);
		return true;
	case CMD_OPEN:
		Command(CMD_TOOLMODE, 0L, o);
		if(!parent) return false;
		f_name = OpenGraphName(filename);
		if(f_name && f_name[0]) {
			if(data) data->Command(CMD_UPDHISTORY, 0L, 0L);
			return OpenGraph(Id == GO_PAGE ? this : parent, f_name, 0L);
			}
		return true;
	case CMD_UPDHISTORY:
		if(data) data->Command(CMD_UPDHISTORY, 0L, 0L);
		return true;
	case CMD_UPDATE:
		Command(CMD_TOOLMODE, 0L, o);
		if(parent && parent->Id != GO_PAGE){
			Undo.ValInt(this, &ToolMode, 0L);		//stub, all plots have UNDO_CONTINUE
			}
		if(CurrDisp) CurrDisp->MouseCursor(MC_WAIT, true);
		for(i = 0; i < NumAxes; i++) if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		if(CurrDisp) CurrDisp->MouseCursor(MC_WAIT, false);
		for(i = 0; Plots && i < NumPlots; i++) 
			if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		dirty = bModified = true;		Disp->StartPage();
		if(CurrDisp) CurrDisp->MouseCursor(MC_WAIT, false);
		DoPlot(Disp);		Disp->EndPage();		CurrGO = 0L;
		return true;
	case CMD_DELOBJ_CONT:
		delflg = UNDO_CONTINUE;
	case CMD_DELOBJ:
		Command(CMD_TOOLMODE, 0L, o);
		bModified = true;
		if(!tmpl) return false;
		for(i = 0; i < NumAxes; i++) if(Axes[i] && (void*)Axes[i] == tmpl){
			if(Axes[i]->Command(CMD_CAN_DELETE, 0L, o)) {
				Undo.DeleteGO((GraphObj**)(&Axes[i]), delflg, o);
				return Command(CMD_REDRAW, 0L, o);
				}
			else {
				InfoBox("Axes used for scaling\ncan not be deleted.");
				return false;
				}
			}
		for(i = 0; Plots && i < NumPlots; i++) if(Plots[i] && (void*)Plots[i] == tmpl) {
			Undo.DeleteGO(&Plots[i], delflg, o);
			Undo.StoreListGO(this, &Plots, &NumPlots, UNDO_CONTINUE);
			for(i = j = 0; i < NumPlots; i++) if(Plots[i]) Plots[j++] = Plots[i];
			NumPlots = j;			//compress list of objects
			return Command(CMD_REDRAW, NULL, o);
			}
		if(tmpl == (void*)frm_g && parent && parent->Id == GO_PAGE) 
			return parent->Command(CMD_DELOBJ_CONT, (void*)this, o);
		return false;
	case CMD_TOOLMODE:
		if(o) {
			o->CheckMenu(ToolMode & 0x0f, false);
			o->CheckMenu(ToolMode = tmpl ? (*((int*)tmpl)) & 0x0f : TM_STANDARD, true);
			}
		if(tl_pts && tl_nPts) free(tl_pts);
		tl_pts = 0L;	tl_nPts = 0;
		if(CurrDisp) CurrDisp->MrkMode = MRK_NONE;
		if(o) switch(ToolMode & 0x0f) {
		case TM_TEXT:	o->MouseCursor(MC_TEXT, false);	break;
		case TM_DRAW:		case TM_POLYLINE:		case TM_POLYGON:
		case TM_RECTANGLE:	case TM_ELLIPSE:		case TM_ROUNDREC:
		case TM_ARROW:
			o->MouseCursor(MC_CROSS, false);
			break;
		default:	o->MouseCursor(MC_ARROW, true);	break;
			}
		return Command(CMD_REDRAW, 0L, CurrDisp);
	case CMD_ADDPLOT:
		if(Id == GO_PAGE) {
			if(CurrGraph && CurrGraph->parent == this) return CurrGraph->Command(cmd, tmpl, o);
			if(!CurrGraph && NumPlots == 1 && Plots[0] && Plots[0]->Id == GO_GRAPH){
				CurrGraph = (Graph*)Plots[0];
				return CurrGraph->Command(cmd, tmpl, o);
				}
			InfoBox("No graph selected!\nCreate a new graph first or select\n"
				"a graph before you can add a plot.");
			return false;
			}
		else if(Id == GO_GRAPH) {
			if(type == GT_3D && Plots){
				for(i = 0; i < NumPlots; i++) { 
					if(Plots[i] && Plots[i]->Id == GO_PLOT3D)
						return Plots[i]->Command(cmd, tmpl, CurrDisp);
					}
				}
			else return AddPlot(0x0);
			}
		return false;
	case CMD_MRK_DIRTY:
		return (dirty = true);
	case CMD_CURRLEFT:	case CMD_CURRIGHT:	case CMD_ADDCHAR:
	case CMD_BACKSP:	case CMD_POS_FIRST:	case CMD_POS_LAST:
		defs.SetDisp(o);
		if(CurrLabel) return CurrLabel->Command(cmd, tmpl, o);
	case CMD_CURRUP:	case CMD_CURRDOWN:
		if(CurrLabel && CurrLabel == CurrGO){
			if(CurrLabel->parent && CurrLabel->parent->Id == GO_MLABEL)
				return CurrLabel->parent->Command(cmd, tmpl, o);
			return true;
			}
	case CMD_SHIFTLEFT:	case CMD_SHIFTRIGHT:	case CMD_SHIFTUP:	case CMD_SHIFTDOWN:
		if(Id == GO_PAGE) {
			if(CurrGraph && CurrGraph->parent == this) 
				return CurrGraph->Command(cmd, tmpl, o);
			}
		else {
			if(type == GT_3D && Plots) {
				for(i = 0; i < NumPlots; i++) {
					if(Plots[i] && Plots[i]->Id == GO_PLOT3D)
						return Plots[i]->Command(cmd, tmpl, CurrDisp);
					}
				}
			}
		return false;
	case CMD_MOVE_TOP:	case CMD_MOVE_UP:
	case CMD_MOVE_DOWN:	case CMD_MOVE_BOTTOM:
		Undo.StoreListGO(this, &Plots, &NumPlots, 0L);
		if(MoveObj(cmd, (GraphObj *)tmpl)){
			bModified = true;
			CurrDisp->StartPage();			DoPlot(CurrDisp);
			CurrDisp->EndPage();			return true;
			}
		return false;
	case CMD_DELETE:
		if(!CurrGO) return false;
		bModified = true;
		if(CurrGO == CurrLabel) return CurrLabel->Command(cmd, tmpl, o);
		if(CurrGO->parent)return CurrGO->parent->Command(CMD_DELOBJ, (void*)CurrGO, o);
		return false;
	case CMD_DROP_PLOT:
		if(!tmpl || ((GraphObj*)tmpl)->Id < GO_PLOT) return false;
		if(Id == GO_GRAPH) CurrGraph = this;	bModified =true;
		if(!NumPlots) {
			Plots = (GraphObj**)calloc(2, sizeof(GraphObj*));
			if(Plots) {
				Plots[0] = (Plot *)tmpl;
				switch(Plots[0]->Id) {
				case GO_PIECHART:
				case GO_RINGCHART:
				case GO_STARCHART:
					type = GT_CIRCCHART;
					break;
				case GO_POLARPLOT:
					type = GT_POLARPLOT;
					break;
				case GO_SCATT3D:
				case GO_PLOT3D:
					type = GT_3D;
					break;
				default:
					type = GT_STANDARD;
					break;
					}
				Bounds.Xmin = x_axis.min = ((Plot*)Plots[0])->Bounds.Xmin;
				Bounds.Xmax = x_axis.max = ((Plot*)Plots[0])->Bounds.Xmax;
				Bounds.Ymin = y_axis.min = ((Plot*)Plots[0])->Bounds.Ymin;
				Bounds.Ymax = y_axis.max = ((Plot*)Plots[0])->Bounds.Ymax;
				if(Bounds.Ymax == Bounds.Ymin) {
					if(Bounds.Ymax != 0.0f) {
						Bounds.Ymax = y_axis.max = Bounds.Ymax + (Bounds.Ymax)/10.0f;
						Bounds.Ymin = y_axis.min = Bounds.Ymin - (Bounds.Ymax)/10.0f;
						}
					else {
						Bounds.Ymax = y_axis.max = 1.0f;
						Bounds.Ymin = y_axis.min = -1.0f;
						}
					}
				if(Bounds.Xmax == Bounds.Xmin) {
					if(Bounds.Xmax != 0.0f) {
						Bounds.Xmax = x_axis.max = Bounds.Xmax + (Bounds.Xmax)/10.0f;
						Bounds.Xmin = x_axis.min = Bounds.Xmin - (Bounds.Xmax)/10.0f;
						}
					else {
						Bounds.Xmax = 1.0f;
						Bounds.Xmin = -1.0f;
						}
					}
				NiceAxis(&x_axis, 4);				NiceAxis(&y_axis, 4);
				NumPlots = 1;
				if(type == GT_STANDARD && !NumAxes) CreateAxes(AxisTempl);
				dirty = false;
				return true;
				}
			return false;
			}
		else {
			tmpPlots = (GraphObj**)memdup(Plots, sizeof(GraphObj*) * (NumPlots+2), 0);
			Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
			Undo.SetGO(this, &tmpPlots[NumPlots++], (Plot *)tmpl, 0L);
			free(Plots);			Plots = tmpPlots;
			if(type == GT_STANDARD && ((x_axis.flags & AXIS_AUTOSCALE) || 
				(y_axis.flags & AXIS_AUTOSCALE))) {
				if(x_axis.flags & AXIS_AUTOSCALE) {
					Bounds.Xmin = x_axis.min = ((Plot *)tmpl)->Bounds.Xmin < Bounds.Xmin ?
						((Plot *)tmpl)->Bounds.Xmin : Bounds.Xmin;
					Bounds.Xmax = x_axis.max = ((Plot *)tmpl)->Bounds.Xmax > Bounds.Xmax ?
						((Plot *)tmpl)->Bounds.Xmax : Bounds.Xmax;
					NiceAxis(&x_axis, 4);
					if(Axes)for(i = 0; i < NumAxes; i++) 
						if(Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &x_axis, o);
					}
				if(y_axis.flags & AXIS_AUTOSCALE) {
					Bounds.Ymin = y_axis.min = ((Plot *)tmpl)->Bounds.Ymin < Bounds.Ymin ? 
						((Plot *)tmpl)->Bounds.Ymin : Bounds.Ymin;
					Bounds.Ymax = y_axis.max = ((Plot *)tmpl)->Bounds.Ymax > Bounds.Ymax ?
						((Plot *)tmpl)->Bounds.Ymax : Bounds.Ymax;
					NiceAxis(&y_axis, 4);
					if(Axes)for(i = 0; i < NumAxes; i++) 
						if(Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &y_axis, o);
					}
				}
			dirty = false;
			//Redraw graph to show all plots
			Disp->StartPage();
			DoPlot(Disp);
			Disp->EndPage();
			return true;
			}
		break;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *)tmpl;		defs.SetDisp(o);
		if(CurrGO && CurrGO->moveable && mev->Action == MOUSE_LBDOWN &&
			(TrackGO = (GraphObj*)CurrGO->ObjThere(mev->x, mev->y))){
			ToolMode |= TM_MOVE;
			}
		else if(mev->Action == MOUSE_LBDOWN){
			CurrGO = 0L;
			if((ToolMode & 0xff) == TM_STANDARD || (ToolMode & 0xff) == TM_ZOOMIN){
				rc_mrk.left = mev->x;		rc_mrk.top = mev->y;
				}
			}
		if(ToolMode != TM_STANDARD && ExecTool(mev)) return true;
		switch(mev->Action) {
		case MOUSE_RBUP:
			i = ToolMode;
			ToolMode = TM_STANDARD;
			mev->Action = MOUSE_LBUP;	//fake select
			Command(cmd, tmpl, o);
			ToolMode = i;
			//the default behaviour for right button click is the same as for
			//   double click: execute properties dialog, just continue.
		case MOUSE_LBDOUBLECLICK:
			if(!CurrGO){
				mev->Action = MOUSE_LBUP;
				Command(CMD_MOUSE_EVENT, mev, CurrDisp);
				mev->Action = MOUSE_LBDOUBLECLICK;
				}
			if(CurrGO && CurrGO->PropertyDlg()) {
				bModified = true;
				return Command(CMD_REDRAW, 0L, o);
				}
			else o->HideMark();
			CurrGO = TrackGO = 0L;	CurrLabel = 0L;
			return false;
		case MOUSE_LBUP:
			if(Id == GO_GRAPH){
				CurrGO = TrackGO = 0L;
				CurrGraph = this;
				}
		case MOUSE_MOVE:
			if(mev->Action == MOUSE_MOVE && !(mev->StateFlags & 0x01)) return false;
			//do all axes
			for(i = 0; Axes && i< NumAxes; i++)
				if(Axes[i] && Axes[i]->Command(cmd, tmpl,o)) return true;
			//do all plots
			if(Plots)for(i = NumPlots-1; i>=0; i--)
				if(Plots[i] && Plots[i]->Command(cmd, tmpl,o)) return true;
			if(frm_d && frm_d->Command(cmd, tmpl, o)) return true;
			if(frm_g && frm_g->Command(cmd, tmpl, o)) return true;
			if(mev->Action == MOUSE_MOVE && ToolMode == TM_STANDARD &&
				rc_mrk.left >=0 && rc_mrk.top >=0) ToolMode = TM_MARK;
			if(!CurrGO) CurrGraph = 0L;
			return false;
			}
		break;
	case CMD_SETSCROLL:
		if(o) {
			o->ActualSize(&rc);
			i = o->un2iy(GRect.Ymax);
			o->SetScroll(true, -(i>>3), i+(i>>3), (rc.bottom -rc.top)>>1, - iround(o->VPorg.fy));
			i = o->un2ix(GRect.Xmax);
			o->SetScroll(false, -(i>>3), i+(i>>3), (rc.right -rc.left)>>1, - iround(o->VPorg.fx));
			if(CurrDisp && o->Erase(ColBG)) Command(CMD_REDRAW, 0L, o);
			return true;
			}
		return false;
	case CMD_SETHPOS:
		if(o && tmpl) o->VPorg.fx = - (double)(*((int*)tmpl));
		return Command(CMD_SETSCROLL, tmpl, o);
	case CMD_SETVPOS:
		if(o && tmpl) o->VPorg.fy = - (double)(*((int*)tmpl));
		return Command(CMD_SETSCROLL, tmpl, o);
	case CMD_OBJTREE:
		for(i = 0; Plots && i < NumPlots; i++) if(Plots[i]) {
			((ObjTree*)tmpl)->Command(CMD_REG_GO, Plots[i], 0L);
			if(Plots[i]->Id == GO_STACKBAR || Plots[i]->Id == GO_GRAPH || 
				Plots[i]->Id == GO_PLOT3D || Plots[i]->Id == GO_POLARPLOT) Plots[i]->Command(cmd, tmpl, o);
			}
		return true;
		}
	return false;
}

void
Graph::DoAutoscale()
{
	int i;
	fRECT oB;

	memcpy(&oB, &Bounds, sizeof(fRECT));
	if(type == GT_STANDARD && ((x_axis.flags & AXIS_AUTOSCALE) ||
		(y_axis.flags & AXIS_AUTOSCALE))) {
		for(i = 0; i < NumPlots; i++) {
			if(Plots[i] && (Plots[i]->Id == GO_PLOTSCATT || Plots[i]->Id == GO_BUBBLEPLOT ||
				Plots[i]->Id == GO_BOXPLOT || Plots[i]->Id == GO_STACKBAR ||
				Plots[i]->Id == GO_STACKPG || Plots[i]->Id == GO_DENSDISP ||
				Plots[i]->Id == GO_LIMITS || Plots[i]->Id == GO_FUNCTION ||
				Plots[i]->Id == GO_FITFUNC)) {
				bModified = true;
				Plots[i]->Command(CMD_AUTOSCALE, 0L, CurrDisp);
				if(dirty) {
					if(x_axis.flags & AXIS_AUTOSCALE) {
						Bounds.Xmin = ((Plot*)Plots[i])->Bounds.Xmin;
						Bounds.Xmax = ((Plot*)Plots[i])->Bounds.Xmax;
						}
					if(y_axis.flags & AXIS_AUTOSCALE) {
						Bounds.Ymin = ((Plot*)Plots[i])->Bounds.Ymin;
						Bounds.Ymax = ((Plot*)Plots[i])->Bounds.Ymax;
						}
					dirty = false;
					}
				else {
					if(x_axis.flags & AXIS_AUTOSCALE) {
						Bounds.Xmin = ((Plot*)Plots[i])->Bounds.Xmin < Bounds.Xmin ?
							((Plot*)Plots[i])->Bounds.Xmin : Bounds.Xmin;
						Bounds.Xmax = ((Plot*)Plots[i])->Bounds.Xmax > Bounds.Xmax ?
							((Plot*)Plots[i])->Bounds.Xmax : Bounds.Xmax;
						}
					if(y_axis.flags & AXIS_AUTOSCALE) {
						Bounds.Ymin = ((Plot*)Plots[i])->Bounds.Ymin < Bounds.Ymin ?
							((Plot*)Plots[i])->Bounds.Ymin : Bounds.Ymin;
						Bounds.Ymax = ((Plot*)Plots[i])->Bounds.Ymax > Bounds.Ymax ?
							((Plot*)Plots[i])->Bounds.Ymax : Bounds.Ymax;
						}
					}
				}
			}
		if(Bounds.Xmax == Bounds.Xmin) {
			Bounds.Xmax = oB.Xmax > oB.Xmin ? oB.Xmax : oB.Xmax + 1.0;
			Bounds.Xmin = oB.Xmin < oB.Xmax ? oB.Xmin : oB.Xmin - 1.0;
			}
		if(Bounds.Ymax == Bounds.Ymin) {
			Bounds.Ymax = oB.Ymax > oB.Ymin ? oB.Ymax : oB.Ymax + 1.0;
			Bounds.Ymin = oB.Ymin < oB.Ymax ? oB.Ymin : oB.Ymin - 1.0;
			}
		if(x_axis.flags & AXIS_AUTOSCALE){
			x_axis.min = Bounds.Xmin;	x_axis.max = Bounds.Xmax;
			NiceAxis(&x_axis, 4);
			if(x_axis.min <= 0.0 && ((x_axis.flags & 0xf000) == AXIS_LOG ||
				(x_axis.flags & 0xf000) == AXIS_RECI)) {
				x_axis.min = base4log(&x_axis, 0);
				}
			if(Axes)for(i = 0; i < NumAxes; i++)
				if(Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &x_axis, 0L);
			}
		if(y_axis.flags & AXIS_AUTOSCALE){
			y_axis.min = Bounds.Ymin;	y_axis.max = Bounds.Ymax;
			NiceAxis(&y_axis, 4);
			if(y_axis.min <= 0.0 && ((y_axis.flags & 0xf000) == AXIS_LOG ||
				(y_axis.flags & 0xf000) == AXIS_RECI)) {
				y_axis.min = base4log(&y_axis, 1);
				}
			if(Axes)for(i = 0; i < NumAxes; i++)
				if(Axes[i]) Axes[i]->Command(CMD_AUTOSCALE, &y_axis, 0L);
			}
		}
	dirty = false;
}

void
Graph::CreateAxes(int templ)
{
	AxisDEF tmp_axis;
	TextDEF label_def, tlbdef;
	char label_text[20];
	Label *label;
	double ts, lb_ydist, lb_xdist, tlb_dist;
	DWORD ptick, ntick, utick;

	if(Axes && NumAxes) return;
	label_def.ColTxt = defs.Color(COL_AXIS);
	label_def.ColBg = 0x00ffffffL;
	label_def.fSize = defs.GetSize(SIZE_TICK_LABELS)*1.2;
	label_def.RotBL = label_def.RotCHAR = 0.0;
	label_def.iSize = 0;
	label_def.Align = TXA_VTOP | TXA_HCENTER;
	label_def.Mode = TXM_TRANSPARENT;
	label_def.Style = TXS_NORMAL;
	label_def.Font = FONT_HELVETICA;
	label_def.text = label_text;
	tlbdef.ColTxt = defs.Color(COL_AXIS);
	tlbdef.ColBg = 0x00ffffffL;
	tlbdef.RotBL = tlbdef.RotCHAR = 0.0;
	tlbdef.iSize = 0;
	tlbdef.fSize = defs.GetSize(SIZE_TICK_LABELS);
	tlbdef.Align = TXA_VCENTER | TXA_HCENTER;
	tlbdef.Style = TXS_NORMAL;
	tlbdef.Mode = TXM_TRANSPARENT;
	tlbdef.Font = FONT_HELVETICA;
	tlbdef.text = 0L;
	ts = defs.GetSize(SIZE_AXIS_TICKS);
	switch (tickstyle & 0x07){
	case 1:						//ticks inside
		ntick = AXIS_POSTICKS;		ptick = AXIS_POSTICKS;	utick = AXIS_NEGTICKS;
		ts *= 0.5;
		break;
	case 2:						//centered, symetrical
		ptick = ntick = utick = AXIS_SYMTICKS;
		ts *= 0.75;
		break;
	default:					//ticks outside
		ptick = AXIS_NEGTICKS;		ntick = AXIS_NEGTICKS; utick = AXIS_POSTICKS;
		break;
		}
	tlb_dist = NiceValue(ts * 2.0);
	lb_ydist = NiceValue((ts+defs.GetSize(SIZE_AXIS_TICKS))*2.0);
	lb_xdist = NiceValue((ts+defs.GetSize(SIZE_AXIS_TICKS))*3.0);
	switch(templ) {
	case 0:
		Axes = (Axis**)calloc(5, sizeof(Axis *));
		x_axis.loc[0].fx = GRect.Xmin + DRect.Xmin;
		x_axis.loc[1].fx = GRect.Xmin + DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fy = GRect.Ymin + DRect.Ymin;
		y_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = GRect.Xmin + DRect.Xmin;;
		if((Axes[0] = new Axis(this, data, &x_axis, AXIS_BOTTOM | ptick |	
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			strcpy(label_text, "x-axis");
			label = new Label(Axes[0], data, GRect.Xmin + (DRect.Xmin+DRect.Xmax)/2.0f, 
				GRect.Ymin+DRect.Ymax+defs.GetSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, AXIS_LEFT | ntick | AXIS_AUTOTICK |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			strcpy(label_text, "y-axis");
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, GRect.Xmin + DRect.Xmin - defs.GetSize(SIZE_AXIS_TICKS)*6.0, 
				GRect.Ymin+(DRect.Ymax+DRect.Ymin)/2.0, &label_def, LB_X_PARENT);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		memcpy(&tmp_axis, &x_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fy = tmp_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		if((Axes[2] = new Axis(this, data, &tmp_axis, AXIS_TOP | AXIS_NOTICKS | AXIS_AUTOTICK))){
			Axes[2]->SetSize(SIZE_LB_YDIST, -lb_ydist); 
			Axes[2]->SetSize(SIZE_TLB_YDIST, -tlb_dist); 
			tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			Axes[2]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		memcpy(&tmp_axis, &y_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fx = tmp_axis.loc[1].fx = GRect.Xmin + DRect.Xmax;
		if((Axes[3] = new Axis(this, data, &tmp_axis, AXIS_RIGHT | AXIS_NOTICKS | AXIS_AUTOTICK))){
			Axes[3]->SetSize(SIZE_LB_XDIST, lb_xdist); 
			Axes[3]->SetSize(SIZE_TLB_XDIST, tlb_dist); 
			tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
			Axes[3]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		NumAxes = 4;
		break;
	case 1:
		Axes = (Axis**)calloc(7, sizeof(Axis *));
		if(x_axis.Start >= 0.0f) x_axis.min = x_axis.Start = -x_axis.Step;
		if(y_axis.Start >= 0.0f) y_axis.min = y_axis.Start = -y_axis.Step;
		x_axis.loc[0].fx = GRect.Xmin + DRect.Xmin;
		x_axis.loc[1].fx = GRect.Xmin + DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = 0.0f;
		y_axis.loc[0].fy = GRect.Ymin + DRect.Ymin;
		y_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = 0.0f;
		if((Axes[0] = new Axis(this, data, &x_axis, ptick | AXIS_Y_DATA |	
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			strcpy(label_text, "x-axis");
			label_def.Align = TXA_VTOP | TXA_HRIGHT;
			label = new Label(Axes[0], data, GRect.Xmin + DRect.Xmax - defs.GetSize(SIZE_AXIS_TICKS)*2.0, 
				GRect.Ymin+DRect.Ymax+defs.GetSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, ntick | AXIS_AUTOTICK | AXIS_X_DATA |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist);
			strcpy(label_text, "y-axis");
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HRIGHT;
			label = new Label(Axes[1], data, GRect.Xmin + DRect.Xmin - defs.GetSize(SIZE_AXIS_TICKS)*6.0, 
				GRect.Ymin + DRect.Ymin + defs.GetSize(SIZE_AXIS_TICKS)*2.0, 
				&label_def, LB_X_PARENT);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		memcpy(&tmp_axis, &x_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fy = tmp_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		if(Axes[2] = new Axis(this, data, &tmp_axis, AXIS_TOP | AXIS_NOTICKS | AXIS_AUTOTICK)){
			Axes[2]->SetSize(SIZE_LB_YDIST, -lb_ydist); 
			Axes[2]->SetSize(SIZE_TLB_YDIST, -tlb_dist);
			tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			Axes[2]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		tmp_axis.loc[0].fy = tmp_axis.loc[1].fy = GRect.Ymin + DRect.Ymin;
		if(Axes[3] = new Axis(this, data, &tmp_axis, AXIS_BOTTOM | AXIS_NOTICKS | AXIS_AUTOTICK)){
			Axes[3]->SetSize(SIZE_LB_YDIST, lb_xdist); 
			Axes[3]->SetSize(SIZE_TLB_YDIST, tlb_dist); 
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[3]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		memcpy(&tmp_axis, &y_axis, sizeof(AxisDEF));
		tmp_axis.owner = NULL;
		tmp_axis.loc[0].fx = tmp_axis.loc[1].fx = GRect.Xmin + DRect.Xmin;
		if(Axes[4] = new Axis(this, data, &tmp_axis, AXIS_LEFT | AXIS_NOTICKS | AXIS_AUTOTICK)){
			Axes[4]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[4]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[4]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		tmp_axis.loc[0].fx = tmp_axis.loc[1].fx = GRect.Xmin + DRect.Xmax;
		if(Axes[5] = new Axis(this, data, &tmp_axis, AXIS_RIGHT | AXIS_NOTICKS | AXIS_AUTOTICK)){
			Axes[5]->SetSize(SIZE_LB_XDIST, lb_xdist); 
			Axes[5]->SetSize(SIZE_TLB_XDIST, tlb_dist); 
			tlbdef.Align = TXA_VCENTER | TXA_HLEFT;
			Axes[5]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		NumAxes = 6;
		break;
	case 2:
		Axes = (Axis**)calloc(3, sizeof(Axis *));
		x_axis.loc[0].fx = GRect.Xmin + DRect.Xmin;
		x_axis.loc[1].fx = GRect.Xmin + DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fy = GRect.Ymin + DRect.Ymin;
		y_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = GRect.Xmin + DRect.Xmin;
		if((Axes[0] = new Axis(this, data, &x_axis, AXIS_BOTTOM | ptick |	
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			strcpy(label_text, "x-axis");
			label = new Label(Axes[0], data, GRect.Xmin + (DRect.Xmin+DRect.Xmax)/2.0f, 
				GRect.Ymin+DRect.Ymax+defs.GetSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, AXIS_LEFT | ntick | AXIS_AUTOTICK |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			strcpy(label_text, "y-axis");
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, GRect.Xmin + DRect.Xmin - defs.GetSize(SIZE_AXIS_TICKS)*6.0, 
				GRect.Ymin+(DRect.Ymax+DRect.Ymin)/2.0, &label_def, LB_X_PARENT);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		NumAxes = 2;
		break;
	case 3:
		label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
		Axes = (Axis**)calloc(3, sizeof(Axis *));
		x_axis.loc[0].fx = GRect.Xmin + DRect.Xmin;
		x_axis.loc[1].fx = GRect.Xmin + DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = GRect.Ymin + DRect.Ymin;
		y_axis.loc[0].fy = GRect.Ymin + DRect.Ymin;
		y_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = GRect.Xmin + DRect.Xmin;
		if((Axes[0] = new Axis(this, data, &x_axis, AXIS_TOP | utick |	
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, -lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, -tlb_dist);
			strcpy(label_text, "x-axis");
			label = new Label(Axes[0], data, GRect.Xmin + (DRect.Xmin+DRect.Xmax)/2.0f, 
				GRect.Ymin+DRect.Ymax+defs.GetSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VBOTTOM | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, AXIS_LEFT | ntick | AXIS_AUTOTICK |
			AXIS_AUTOSCALE | AXIS_INVERT | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			strcpy(label_text, "y-axis");
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, GRect.Xmin + DRect.Xmin - defs.GetSize(SIZE_AXIS_TICKS)*6.0, 
				GRect.Ymin+(DRect.Ymax+DRect.Ymin)/2.0, &label_def, LB_X_PARENT);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		NumAxes = 2;
		break;
	case 4:
		Axes = (Axis**)calloc(3, sizeof(Axis *));
		if(x_axis.Start >= 0.0f) x_axis.min = -x_axis.Step;
		x_axis.loc[0].fx = GRect.Xmin + DRect.Xmin;
		x_axis.loc[1].fx = GRect.Xmin + DRect.Xmax;
		x_axis.loc[0].fy = x_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fy = GRect.Ymin + DRect.Ymin;
		y_axis.loc[1].fy = GRect.Ymin + DRect.Ymax;
		y_axis.loc[0].fx = y_axis.loc[1].fx = 0.0f;
		if((Axes[0] = new Axis(this, data, &x_axis, AXIS_BOTTOM | ptick |	
			AXIS_AUTOTICK | AXIS_AUTOSCALE | ((tickstyle & 0x100) ? AXIS_GRIDLINE : 0)))){
			Axes[0]->SetSize(SIZE_LB_YDIST, lb_ydist);
			Axes[0]->SetSize(SIZE_TLB_YDIST, tlb_dist);
			strcpy(label_text, "x-axis");
			label = new Label(Axes[0], data, GRect.Xmin + (DRect.Xmin+DRect.Xmax)/2.0f, 
				GRect.Ymin+DRect.Ymax+defs.GetSize(SIZE_AXIS_TICKS)*4.0f, &label_def, LB_Y_PARENT);
			if(label && Axes[0]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VTOP | TXA_HCENTER;
			Axes[0]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		if((Axes[1] = new Axis(this, data, &y_axis, ntick | AXIS_AUTOTICK | AXIS_X_DATA |
			AXIS_AUTOSCALE | ((tickstyle & 0x200) ? AXIS_GRIDLINE : 0)))){
			Axes[1]->SetSize(SIZE_LB_XDIST, -lb_xdist); 
			Axes[1]->SetSize(SIZE_TLB_XDIST, -tlb_dist); 
			strcpy(label_text, "y-axis");
			label_def.RotBL = 90.0;			label_def.Align = TXA_VBOTTOM | TXA_HCENTER;
			label = new Label(Axes[1], data, GRect.Xmin + DRect.Xmin - defs.GetSize(SIZE_AXIS_TICKS)*6.0, 
				GRect.Ymin+(DRect.Ymax+DRect.Ymin)/2.0, &label_def, LB_X_PARENT);
			if(label && Axes[1]->Command(CMD_DROP_LABEL, (void*)label, 0L)) label = 0L;
			else if(label) DeleteGO(label);
			tlbdef.Align = TXA_VCENTER | TXA_HRIGHT;
			Axes[1]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		label = 0L;
		NumAxes = 2;
		break;
		}
}

bool
Graph::ExecTool(MouseEvent *mev)
{
	static POINT pl = {0, 0}, pc = {0, 0};
	static DWORD color = 0L;
	POINT line[5];
	GraphObj **tmpPlots, *new_go;
	TextDEF *td;
	lfPOINT *lfp, lf;
	int i, j;
	double x, y;

	if(!mev || !CurrDisp) return false;
	td = 0L;
	if(ToolMode & TM_MOVE) switch (mev->Action) {
		case MOUSE_LBDOWN:
			pl.x = pc.x = mev->x;				pl.y = pc.y = mev->y;
			CurrDisp->HideMark();
			if(TrackGO && TrackGO->Command(CMD_MOUSECURSOR, 0L, CurrDisp))
				return true;
			else CurrDisp->MouseCursor(MC_MOVE, false);
			return true;
		case MOUSE_MOVE:
			if(TrackGO) {
				if(TrackGO->Id == GO_DRAGHANDLE && TrackGO->type >= DH_18 &&
					TrackGO->type <= DH_88) {
					lf.fx = CurrDisp->fix2un((double)(mev->x-pl.x));	
					lf.fy = CurrDisp->fiy2un((double)(mev->y-pl.y));
					TrackGO->Track((POINT*)&lf, CurrDisp);
					}
				else {
					pc.x = mev->x-pl.x;				pc.y = mev->y-pl.y;
					TrackGO->Track(&pc, CurrDisp);
					}
				}
			return true;
		case MOUSE_LBUP:
			if(TrackGO) {
				ToolMode &= ~TM_MOVE;
				pc.x = mev->x-pl.x;				pc.y = mev->y-pl.y;
				if(!pc.x && !pc.y){
					Command(CMD_TOOLMODE, (void*)(& ToolMode), CurrDisp);
					return false;
					}
				TrackGO->Track(&pc, CurrDisp);
				lf.fx = CurrDisp->fix2un((double)(mev->x-pl.x));	
				lf.fy = CurrDisp->fiy2un((double)(mev->y-pl.y));
				TrackGO->Command(CMD_MOVE, &lf, CurrDisp);
				bModified = true;
				}
			Command(CMD_TOOLMODE, (void*)(& ToolMode), CurrDisp);
			return true;
		default:
			return true;
		}
	if(ToolMode == TM_MARK || ToolMode == TM_ZOOMIN) {
		switch (mev->Action) {
		case MOUSE_LBDOWN:			//we should not receive that !
			rc_mrk.left = mev->x;	rc_mrk.top = mev->y;
			return false;
		case MOUSE_LBUP:
			j = (i = rc_mrk.left - mev->x)*i;
			j += ((i = rc_mrk.top - mev->y)*i);
			if(j < 20) {						//glitch
				ToolMode = TM_STANDARD;
				return false;
				}
			if(ToolMode == TM_ZOOMIN) {
				rc_mrk.right = mev->x;	rc_mrk.bottom = mev->y;
				ToolMode = TM_STANDARD;
				return Command(CMD_ZOOM, (void*) &"+", 0L);
				}
		default:
			ToolMode = TM_STANDARD;
		case MOUSE_MOVE:
			if((mev->StateFlags &1) && rc_mrk.left >= 0 && rc_mrk.top >= 0) {
				if(rcUpd.left != rcUpd.right && rcUpd.top != rcUpd.bottom)
					CurrDisp->UpdateRect(&rcUpd, false);
				line[0].x = line[3].x = line[4].x = rc_mrk.left;
				line[0].y = line[1].y = line[4].y = rc_mrk.top;
				line[1].x = line[2].x = mev->x;
				line[2].y = line[3].y = mev->y;
				CurrDisp->ShowLine(line, 5, 0x00c0c0c0L);
				rcUpd.left = rcUpd.right = rc_mrk.left;
				rcUpd.top = rcUpd.bottom = rc_mrk.top;
				UpdateMinMaxRect(&rcUpd, rc_mrk.right = mev->x, rc_mrk.bottom = mev->y);
				IncrementMinMaxRect(&rcUpd,2);
				}
			return true;
			}
		}
	if(NumPlots && !Plots[NumPlots-1]) {
		Undo.StoreListGO(this, &Plots, &NumPlots, UNDO_CONTINUE);
		for(i = j = 0; i < NumPlots; i++) if(Plots[i]) Plots[j++] = Plots[i];
		NumPlots = j;
		}
	else if(!Plots && !(Plots = (GraphObj**)calloc(2, sizeof(GraphObj*))))return false;
	else if(Plots[NumPlots]){
		if(tmpPlots = (GraphObj**)memdup(Plots, (NumPlots+2) * sizeof(GraphObj*), 0L)){
			Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
			free(Plots);	Plots = tmpPlots;
			Plots[NumPlots] = Plots[NumPlots+1] = 0L;
			}
		else return false;
		}
	switch(ToolMode & 0x0f) {
	case TM_DRAW:
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			CurrGO = 0L;
			CurrDisp->MrkMode = MRK_NONE;
			Command(CMD_REDRAW, 0L, CurrDisp);
			color = defs.Color(COL_POLYLINE);
			pl.x = pc.x = mev->x;		pl.y = pc.y = mev->y;
			if(tl_pts) free(tl_pts);
			tl_nPts = 0;
			if(tl_pts = (POINT*)malloc(4096 * sizeof(POINT)))
				AddToPolygon(&tl_nPts, tl_pts, &pc);
			return true;
		case MOUSE_LBUP:
			pl.x = mev->x;				pl.y = mev->y;
			if(tl_pts) AddToPolygon(&tl_nPts, tl_pts, &pc);
			// create line object
			if(tl_pts && tl_nPts >1) {			//DEBUG: check for plausibility
				if(lfp = (lfPOINT*)malloc(tl_nPts * sizeof(lfPOINT))){
					for(i = 0; i < tl_nPts; i++) {
						lfp[i].fx = CurrDisp->fix2un(tl_pts[i].x - CurrDisp->VPorg.fx);	
						lfp[i].fy = CurrDisp->fiy2un(tl_pts[i].y - CurrDisp->VPorg.fy);
						}
					if(Plots[NumPlots]) i = NumPlots+1;
					else i = NumPlots;
					Undo.SetGO(this, &Plots[i], new polyline(this, data, lfp, (int)tl_nPts), 0L);
					if(Plots[i]){
						NumPlots = i+1;
						Plots[i]->moveable = 1;
						Plots[i]->DoPlot(CurrDisp);							//init
						CurrDisp->ShowMark(CurrGO = Plots[i], MRK_GODRAW);	//edit
						bModified = true;
						}
					free(lfp);
					}
				if(tl_pts) free(tl_pts);	tl_pts = 0L;	tl_nPts = 0;
				return true;
				}
			if(tl_pts) free(tl_pts);	tl_pts = 0L;	tl_nPts = 0;
			return false;
		case MOUSE_MOVE:
			if((mev->StateFlags &1) && tl_pts && tl_nPts < 4095) {
				memcpy(&pl, &pc, sizeof(POINT));
				line[1].x = pc.x = mev->x;		line[1].y = pc.y = mev->y;
				line[0].x = pl.x;				line[0].y = pl.y;
				CurrDisp->ShowLine(line, 2, color);
				AddToPolygon(&tl_nPts, tl_pts, &pc);
				return true;
				}
			break;
			}
		break;
	case TM_POLYLINE:
	case TM_POLYGON:
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			if(!tl_pts) {
				CurrGO = 0L;
				CurrDisp->MrkMode = MRK_NONE;
				Command(CMD_REDRAW, 0L, CurrDisp);
				color = defs.Color((ToolMode & 0x0f) == TM_POLYLINE ? COL_POLYLINE : COL_POLYGON);
				pl.x = pc.x = mev->x;		pl.y = pc.y = mev->y;
				tl_nPts = 0;
				if(tl_pts = (POINT*)malloc(4096 * sizeof(POINT)))
				AddToPolygon(&tl_nPts, tl_pts, &pc);
				rcDim.left = rcDim.right = rcUpd.left = rcUpd.right = mev->x;
				rcDim.top = rcDim.bottom = rcUpd.top = rcUpd.bottom = mev->y;
				}
			return true;
		case MOUSE_MOVE:
			if(tl_pts && tl_nPts) {
				if(rcUpd.left != rcUpd.right && rcUpd.top != rcUpd.bottom)
					CurrDisp->UpdateRect(&rcUpd, false);
				CurrDisp->ShowLine(tl_pts, tl_nPts, color);
				line[1].x = mev->x;				line[1].y = mev->y;
				line[0].x = pc.x;				line[0].y = pc.y;
				CurrDisp->ShowLine(line, 2, color);
				UpdateMinMaxRect(&rcUpd, mev->x, mev->y);
				memcpy(&rcUpd, &rcDim, sizeof(RECT));
				UpdateMinMaxRect(&rcUpd, mev->x, mev->y);
				IncrementMinMaxRect(&rcUpd, 2);
				return true;
				}
			break;
		case MOUSE_LBUP:
			if(tl_pts && tl_nPts) {
				memcpy(&pl, &pc, sizeof(POINT));
				pc.x = mev->x;				pc.y = mev->y;
				UpdateMinMaxRect(&rcDim, mev->x, mev->y);
				AddToPolygon(&tl_nPts, tl_pts, &pc);
				}
			return true;
		default:			// create line or polygon object
			if(tl_pts && tl_nPts >0) {			//DEBUG: check for plausibility
				pc.x = mev->x;				pc.y = mev->y;
				AddToPolygon(&tl_nPts, tl_pts, &pc);
				if(lfp = (lfPOINT*)malloc(tl_nPts * sizeof(lfPOINT))){
					for(i = 0; i < tl_nPts; i++) {
						lfp[i].fx = CurrDisp->fix2un(tl_pts[i].x-CurrDisp->VPorg.fx);	
						lfp[i].fy = CurrDisp->fiy2un(tl_pts[i].y-CurrDisp->VPorg.fy);
						}
					if(Plots[NumPlots]) i = NumPlots+1;
					else i = NumPlots;
					Undo.SetGO(this, &Plots[i], ToolMode == TM_POLYLINE ?
						new polyline(this, data, lfp, (int)tl_nPts) :
						new polygon(this, data, lfp, (int)tl_nPts), 0L);
					if(Plots[i]){
						NumPlots = i+1;
						Plots[i]->moveable = 1;
						Plots[i]->DoPlot(CurrDisp);							//init
						CurrDisp->ShowMark(CurrGO = Plots[i], MRK_GODRAW);	//edit
						bModified = true;
						}
					free(lfp);
					}
				free(tl_pts);			tl_pts = 0L;		tl_nPts = 0;
				return true;
				}
			}
		if(mev->x == pc.x && mev->y == pc.y) return true;	//rebounce
		break;
	case TM_RECTANGLE:
	case TM_ELLIPSE:
	case TM_ROUNDREC:
	case TM_ARROW:
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			CurrGO = 0L;
			CurrDisp->MrkMode = MRK_NONE;
			Command(CMD_REDRAW, 0L, CurrDisp);
			color = defs.Color((ToolMode & 0x0f) != TM_ARROW ? COL_POLYGON : COL_DATA_LINE);
			pl.x = pc.x = mev->x;		pl.y = pc.y = mev->y;
			rcDim.left = rcDim.right = rcUpd.left = rcUpd.right = mev->x;
			rcDim.top = rcDim.bottom = rcUpd.top = rcUpd.bottom = mev->y;
			return true;
		case MOUSE_MOVE:
			if(mev->StateFlags &1) {
				if(rcUpd.left != rcUpd.right && rcUpd.top != rcUpd.bottom)
					CurrDisp->UpdateRect(&rcUpd, false);
				line[0].x = line[4].x = pl.x;	line[0].y = line[4].y = pl.y;
				if((ToolMode & 0x0f)==TM_ARROW) {
					line[1].x = pc.x = mev->x;		line[1].y = pc.y = mev->y;
					CurrDisp->ShowLine(line, 2, color);
					}
				else {
					line[1].x = pc.x = mev->x;		line[1].y = pl.y;
					line[2].x = mev->x;				line[2].y = pc.y = mev->y;
					line[3].x = pl.x;				line[3].y = mev->y;
					CurrDisp->ShowLine(line, 5, color);
					if((ToolMode & 0x0f) == TM_ELLIPSE) 
						CurrDisp->ShowEllipse(pc, pl, color);
					}
				memcpy(&rcUpd, &rcDim, sizeof(RECT));
				UpdateMinMaxRect(&rcUpd, mev->x, mev->y);
				IncrementMinMaxRect(&rcUpd, 2);
				return true;
				}
			break;
		case MOUSE_LBUP:
			pc.x = mev->x;			pc.y = mev->y;
			if(((ToolMode & 0x0f)==TM_ARROW || (abs(pc.x-pl.x) >5 && abs(pc.y-pl.y) >5)) && 
				(lfp = (lfPOINT*)malloc(2 * sizeof(lfPOINT)))){
				lfp[0].fx = CurrDisp->fix2un(pl.x - CurrDisp->VPorg.fx);
				lfp[0].fy = CurrDisp->fiy2un(pl.y - CurrDisp->VPorg.fy);
				lfp[1].fx = CurrDisp->fix2un(pc.x - CurrDisp->VPorg.fx);
				lfp[1].fy = CurrDisp->fiy2un(pc.y - CurrDisp->VPorg.fy);
				if(Plots[NumPlots]) i = NumPlots+1;
				else i = NumPlots;
				if(ToolMode == TM_RECTANGLE) new_go = new rectangle(this, data,
					&lfp[0], &lfp[1]);
				else if(ToolMode == TM_ELLIPSE) new_go = new ellipse(this, data,
					&lfp[0], &lfp[1]);
				else if(ToolMode == TM_ROUNDREC) new_go = new roundrec(this, data,
					&lfp[0], &lfp[1]);
				else if(ToolMode == TM_ARROW) new_go = new Arrow(this, data,
					lfp[0], lfp[1], ARROW_UNITS | ARROW_LINE);
				else new_go = 0L;
				if(new_go) Undo.SetGO(this, &Plots[i], new_go, 0L);
				if(Plots[i]){
					NumPlots = i+1;
					Plots[i]->DoPlot(CurrDisp);							//init
					CurrDisp->ShowMark(CurrGO = Plots[i], MRK_GODRAW);				//edit
					Plots[i]->moveable = 1;
					bModified = true;
					}
				free(lfp);
				}
			else if(rcUpd.left != rcUpd.right && rcUpd.top != rcUpd.bottom) {
				CurrDisp->UpdateRect(&rcUpd, false);
				}
			}
		break;
	case TM_TEXT:
		if(Plots[NumPlots]) i = NumPlots+1;
		else i = NumPlots;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(!(td = (TextDEF *)calloc(1, sizeof(TextDEF))))return false;
			x = CurrDisp->fix2un(mev->x-CurrDisp->VPorg.fx);
			y = CurrDisp->fiy2un(mev->y-CurrDisp->VPorg.fy);
			td->ColTxt = defs.Color(COL_TEXT);		td->ColBg = defs.Color(COL_BG);
			td->RotBL = td->RotCHAR = 0.0f;			td->fSize = defs.GetSize(SIZE_TEXT);
			td->Align = TXA_VTOP | TXA_HLEFT;		td->Style = TXS_NORMAL;
			td->Mode = TXM_TRANSPARENT;				td->Font = FONT_HELVETICA;
			td->text = 0L;
			CurrGO = 0L;
			CurrDisp->MrkMode = MRK_NONE;
			Command(CMD_REDRAW, 0L, CurrDisp);
			y -= td->fSize/2.0;
			Undo.SetGO(this, &Plots[i], new Label(this, data, x, y, td, 0L), 0L);
			if(Plots[i]){
				NumPlots = i+1;
				Plots[i]->DoPlot(CurrDisp);							//init
				CurrDisp->ShowMark(CurrGO = Plots[i], MRK_GODRAW);	//edit
				Plots[i]->moveable = 1;
				}
			free(td);
			return true;
			}
		break;
		}
	return false;
}

bool
Graph::MoveObj(int cmd, GraphObj *g)
{
	int i, j;
	GraphObj *g1 = 0;

	if(!g || NumPlots <2 || g->parent != this) return false;
	switch(cmd) {
	case CMD_MOVE_TOP:
		for(i = j = 0; i <NumPlots; i++){
			if(g == Plots[i]) g1 = Plots[i];
			else Plots[j++] = Plots[i];
			}
		if(g1) {
			Plots[j++] = g1;
			return true;
			}
		break;
	case CMD_MOVE_UP:
		for(i = 0; i<NumPlots-1; i++){
			if(g == Plots[i]){
				g1 = Plots[i];	Plots[i] = Plots[i+1];	Plots[i+1] = g1;
				return true;
				}
			}
		break;
	case CMD_MOVE_DOWN:
		for(i = 1; i<NumPlots; i++){
			if(g == Plots[i]){
				g1 = Plots[i];	Plots[i] = Plots[i-1];	Plots[i-1] = g1;
				return true;
				}
			}
		break;
	case CMD_MOVE_BOTTOM:
		if(Plots[0] == g) return false;
		for(i =  j = NumPlots-1; i >= 0; i--) {
			if(g == Plots[i]) g1 = Plots[i];
			else Plots[j--] = Plots[i];
			}
		if(g1) {
			Plots[j--] = g1;
			return true;
			}
		break;
		}
	return false;
}

bool
Graph::DoZoom(char *z)
{
	RECT cw;
	double fac, f1, f2;
	ZoomDEF *tz;
	Graph *cg;

	if(!z) return false;
	if(0==strcmp("fit", z)) {
		if(CurrGraph) cg = CurrGraph;
		else cg = this;
		rc_mrk.left = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_LEFT))-4;
		rc_mrk.right = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_RIGHT))+4
			+ iround(CurrDisp->MenuHeight);
		rc_mrk.top = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_TOP))-4 
			- iround(CurrDisp->MenuHeight);
		rc_mrk.bottom = CurrDisp->un2ix(cg->GetSize(SIZE_GRECT_BOTTOM))+4;
		CurrDisp->ActualSize(&cw);
		f1 = (double)(cw.bottom - cw.top)/(double)(rc_mrk.bottom - rc_mrk.top);
		f2 = ((double)(cw.right - cw.left)/(double)(rc_mrk.right - rc_mrk.left));
		fac = f1 < f2 ? f1 : f2;
		if((CurrDisp->VPscale * fac) > 100.0) fac = 100.0/CurrDisp->VPscale;
		if((CurrDisp->VPscale * fac) < 0.05) fac = 0.05/CurrDisp->VPscale;
		if(fac == 1.0) return false;
		if(tz = (ZoomDEF*)memdup(zoom_def, sizeof(ZoomDEF)*(zoom_level+1), 0)){
			if(zoom_def) free(zoom_def);
			zoom_def = tz;
			zoom_def[zoom_level].org.fx = CurrDisp->VPorg.fx;
			zoom_def[zoom_level].org.fy = CurrDisp->VPorg.fy;
			zoom_def[zoom_level].scale = CurrDisp->VPscale;
			zoom_level++;
			}
		CurrDisp->VPscale *= fac;
		if(CurrDisp->VPscale < 0.05) CurrDisp->VPscale = 0.05; 
		if(CurrDisp->VPscale > 100.0) CurrDisp->VPscale = 100.0; 
		CurrDisp->VPorg.fx = -rc_mrk.left * fac;
		CurrDisp->VPorg.fy = -rc_mrk.top * fac;
		HideTextCursor();
		Command(CMD_SETSCROLL, 0L, CurrDisp);
		return true;
		}
	else if(0==strcmp("+", z)) {
		if(rc_mrk.left >= 0 && rc_mrk.right >= 0 && rc_mrk.top >= 0 && rc_mrk.bottom >= 0) {
			if(rc_mrk.left > rc_mrk.right) Swap(rc_mrk.left, rc_mrk.right);
			if(rc_mrk.top > rc_mrk.bottom) Swap(rc_mrk.top, rc_mrk.bottom);
			if(5 > (rc_mrk.right - rc_mrk.left) || 5 > (rc_mrk.bottom - rc_mrk.top)) {
				rc_mrk.left = rc_mrk.left = rc_mrk.left = rc_mrk.left = -1;
				ToolMode = TM_STANDARD;		Command(CMD_TOOLMODE, &ToolMode, CurrDisp);
				return false;
				}
			CurrDisp->ActualSize(&cw);
			fac = (double)(cw.bottom - cw.top)/(double)(rc_mrk.bottom - rc_mrk.top);
			fac += ((double)(cw.right - cw.left)/(double)(rc_mrk.right - rc_mrk.left));
			fac /= 2.0;
			if((CurrDisp->VPscale * fac) > 100.0) fac = 100.0/CurrDisp->VPscale;
			if((CurrDisp->VPscale * fac) < 0.05) fac = 0.05/CurrDisp->VPscale;
			if(fac == 1.0) return false;
			if(tz = (ZoomDEF*)memdup(zoom_def, sizeof(ZoomDEF)*(zoom_level+1), 0)){
				if(zoom_def) free(zoom_def);
				zoom_def = tz;
				zoom_def[zoom_level].org.fx = CurrDisp->VPorg.fx;
				zoom_def[zoom_level].org.fy = CurrDisp->VPorg.fy;
				zoom_def[zoom_level].scale = CurrDisp->VPscale;
				zoom_level++;
				}
			CurrDisp->VPscale *= fac;
			if(CurrDisp->VPscale < 0.05) CurrDisp->VPscale = 0.05; 
			if(CurrDisp->VPscale > 100.0) CurrDisp->VPscale = 100.0; 
			CurrDisp->VPorg.fx = CurrDisp->VPorg.fx * fac - rc_mrk.left * fac;
			CurrDisp->VPorg.fy = CurrDisp->VPorg.fy * fac - rc_mrk.top * fac;
			HideTextCursor();
			Command(CMD_SETSCROLL, 0L, CurrDisp);
			CurrDisp->MouseCursor(MC_ARROW, false);
			rc_mrk.left = rc_mrk.left = rc_mrk.left = rc_mrk.left = -1;
			ToolMode = TM_STANDARD;		Command(CMD_TOOLMODE, &ToolMode, CurrDisp);
			return true;
			}
		else {
			ToolMode = TM_ZOOMIN;			CurrDisp->MouseCursor(MC_ZOOM, true);
			}
		}
	else if(0==strcmp("-", z)) {
		HideTextCursor();
		if(zoom_def && zoom_level > 0) {
			zoom_level--;
			if(CurrDisp->VPscale == zoom_def[zoom_level].scale &&
				CurrDisp->VPorg.fx == zoom_def[zoom_level].org.fx &&
				CurrDisp->VPorg.fy == zoom_def[zoom_level].org.fy) {
				DoZoom(z);
				}
			else {
				CurrDisp->VPscale = zoom_def[zoom_level].scale;
				CurrDisp->VPorg.fx = zoom_def[zoom_level].org.fx;
				CurrDisp->VPorg.fy = zoom_def[zoom_level].org.fy;
				}
			}
		else {
			CurrDisp->VPorg.fx = CurrDisp->VPorg.fy = 0.0;
			CurrDisp->VPscale = Id == GO_PAGE ? 0.5 : 1.0;
			}
		Command(CMD_SETSCROLL, 0L, CurrDisp);
		return true;
		}
	else if(0==strcmp("25", z)){
		CurrDisp->VPscale = 0.25;		return DoZoom("org");
		}
	else if(0==strcmp("50", z)){
		CurrDisp->VPscale = 0.50;		return DoZoom("org");
		}
	else if(0==strcmp("100", z)){
		CurrDisp->VPscale = 1.00;		return DoZoom("org");
		}
	else if(0==strcmp("200", z)){
		CurrDisp->VPscale = 2.00;		return DoZoom("org");
		}
	else if(0==strcmp("400", z)){
		CurrDisp->VPscale = 4.00;		return DoZoom("org");
		}
	else if(0==strcmp("org", z)){
		CurrDisp->VPorg.fx = 0.0;		CurrDisp->VPorg.fy = iround(CurrDisp->MenuHeight);
		HideTextCursor();
		Command(CMD_SETSCROLL, 0L, CurrDisp);
		return DoZoom("reset");
		}
	else if(0==strcmp("reset", z)){
		if(zoom_def && zoom_level > 0) free(zoom_def);
		zoom_def = 0L;			zoom_level = 0;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Pages are graphic objects containing graphs and drawn objects
Page::Page(GraphObj *par, DataObj *d):Graph(par, d, 0L)
{
	FileIO(INIT_VARS);
	cGraphs--;		cPages++;		Id = GO_PAGE;	bModified = true;
}

Page::Page(int src):Graph(src)
{
	int i;

	//most of the object is read by Graph::FileIO()
	ColBG = 0x00e8e8e8L;
	LineDef.width = 0.0;	LineDef.patlength = 1.0;
	LineDef.color = LineDef.pattern = 0x0L;
	FillDef.type = FILL_NONE;
	FillDef.color = 0x00ffffffL;	//use white paper
	FillDef.scale = 1.0;
	FillDef.hatch = 0L;
	cGraphs--;		cPages++;	bModified = false;
	if(Plots) for(i = 0; i < NumPlots; i++) if(Plots[i]) Plots[i]->moveable = 1;
}

void
Page::DoPlot(anyOutput *o)
{
	int i;
	POINT pts[3];

	if(!o && !Disp) {
		Disp = NewDispClass(this);
		Disp->SetMenu(MENU_PAGE);
		sprintf(TmpTxt, "Page %d", cPages);
		if(!name) name = strdup(TmpTxt);			Disp->Caption(TmpTxt);
		Disp->VPorg.fy = iround(Disp->MenuHeight);
		Disp->VPscale = 0.5;
		Disp->CheckMenu(ToolMode, true);
		OwnDisp = true;
		}
	//the first output class is the display class
	if(!Disp && (!(Disp = o))) return;
	CurrDisp = o ? o : Disp;
	CurrDisp->Erase(CurrDisp->dFillCol = ColBG);
	if(OwnDisp && CurrDisp == Disp)LineDef.color = 0x0L;
	else LineDef.color = FillDef.color;
	CurrDisp->SetLine(&LineDef);
	CurrDisp->SetFill(&FillDef);
	CurrDisp->oRectangle(rDims.left = CurrDisp->co2ix(GRect.Xmin), 
		rDims.top = CurrDisp->co2iy(GRect.Ymin), rDims.right = CurrDisp->co2ix(GRect.Xmax),
		rDims.bottom = CurrDisp->co2iy(GRect.Ymax));
	pts[0].x = rDims.left+7;				pts[0].y = pts[1].y = rDims.bottom;
	pts[1].x = pts[2].x = rDims.right;		pts[2].y = rDims.top +3;
	CurrDisp->oPolyline(pts, 3);
	//do all plots
	if(Plots) for(i = 0; i < NumPlots; i++) if(Plots[i]) Plots[i]->DoPlot(CurrDisp);

}

bool
Page::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmpPlots;
	Graph *g;

	switch(cmd) {
	case CMD_MOUSE_EVENT:
		return Graph::Command(cmd, tmpl, o);
	case CMD_REDRAW:
		Disp->StartPage();
		DoPlot(Disp);
		Disp->EndPage();
		return true;
	case CMD_CONFIG:
		return Configure();
	case CMD_DROP_GRAPH:
		if(!(tmpPlots = (GraphObj**)memdup(Plots, (NumPlots+2)*sizeof(GraphObj*), 0))) 
			return false;
		Undo.ListGOmoved(Plots, tmpPlots, NumPlots);
		free(Plots);			Plots = tmpPlots;	Plots[NumPlots] = Plots[NumPlots+1] = 0L;
		Undo.SetGO(this, &Plots[NumPlots], (GraphObj*)tmpl, 0L);
		if(Plots[NumPlots]){
			Plots[NumPlots]->parent = this;
			Plots[NumPlots]->Command(CMD_SET_DATAOBJ, data, 0L);
			if(Plots[NumPlots]->Id == GO_GRAPH) CurrGraph = (Graph*)Plots[NumPlots];
			Plots[NumPlots]->moveable = 1;		//all page items should be freely
			}									//   moveable by user
		NumPlots++;
		if(CurrDisp) {
			CurrDisp->StartPage();			DoPlot(CurrDisp);
			CurrDisp->EndPage();
			}
		return true;
	case CMD_NEWGRAPH:
		if((g = new Graph(this, data, Disp)) && g->PropertyDlg() && 
			Command(CMD_DROP_GRAPH, g, o))return true;
		else if(g) DeleteGO(g);
		return false;
	case CMD_SET_DATAOBJ:
		Graph::Command(cmd, tmpl, o);
		Id = GO_PAGE;
		return true;
	default:
		return Graph::Command(cmd, tmpl, o);
	}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Create a tree structure of all interesting objects
//   This object is used e.g. for layer control
ObjTree::ObjTree(GraphObj *par, DataObj *d, GraphObj *root):GraphObj(par, d)
{
	Id = GO_OBJTREE;
	TextDef.ColTxt = 0x00000000L;		TextDef.ColBg = 0x00ffffffL;
	TextDef.fSize = 4.0;				TextDef.RotBL = TextDef.RotCHAR = 0.0;
	TextDef.iSize = 0;					TextDef.Align = TXA_HLEFT | TXA_VTOP;
	TextDef.Mode = TXM_TRANSPARENT;		TextDef.Style = TXS_NORMAL;
	TextDef.Font = FONT_HELVETICA;		TextDef.text = 0L;
	count = 0;		maxcount = 100;
	if(list = (GraphObj**) malloc(100 * sizeof(GraphObj*))) {
		list[count++] = root;
		}
	root->Command(CMD_OBJTREE, this, 0L);
}

ObjTree::~ObjTree()
{
	if(list) free(list);		list = 0L;
}

void
ObjTree::DoPlot(anyOutput *o)
{
	int i, n, ix, iy;
	GraphObj *curr_obj;

	if(!o || !list) return;
	o->Erase(0x00ffffffL);
	ix = 10;	iy = 0;
	for(i = 0; i < count; i++, iy += TextDef.iSize) {
		if(list[i]) {
			curr_obj = list[i];			n = 0;
			if(i) while(curr_obj && curr_obj != list[0]) {
				if(curr_obj != list[0]) n += sprintf(TmpTxt + n, " -");
				if(curr_obj) curr_obj = curr_obj->parent; 
				}
			sprintf(TmpTxt + n, "%s%s", n ? " " : "", get_name(i));
			if(list[i]->Id >= GO_PLOT && list[i]->Id < GO_GRAPH) {
				TextDef.ColTxt = ((Plot*)list[i])->hidden ? 0x00000080L : 0x00008000L;
				}
			else TextDef.ColTxt = 0x00000000L;
			o->SetTextSpec(&TextDef);
			o->oTextOut(ix, iy, TmpTxt, 0);
			}
		}
}

bool
ObjTree::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch(cmd){
	case CMD_SET_DATAOBJ:
		Id = GO_OBJTREE;
		return true;
	case CMD_REG_GO:
		if(tmpl && ((GraphObj*)tmpl)->Id >= GO_PLOT && ((GraphObj*)tmpl)->Id < GO_SPREADDATA) {
			if(count >= maxcount) {
				maxcount += 100;
				list = (GraphObj**) realloc(list, maxcount);
				}
			if(list) list[count++] = (GraphObj*)tmpl;
			}
		return true;
	case CMD_TEXTDEF:
		if(tmpl) memcpy(&TextDef, tmpl, sizeof(TextDEF));
		TextDef.text = 0L;
		return true;
		}
	return false;
}

anyOutput *
ObjTree::CreateBitmap(int *bw, int *bh, anyOutput *tmpl)
{
	anyOutput *bmp = 0L;
	int h;
	
	h = tmpl->un2iy(TextDef.fSize) * count;
	if(h > *bh) *bh = h;
	if(bmp = NewBitmapClass(*bw, *bh, tmpl->hres, tmpl->vres)) DoPlot(bmp);
	return bmp;
}

char *
ObjTree::get_name(int li)
{
	if(li < count && list[li] && list[li]->name) return list[li]->name;
	else return "(unknown)";
}

int
ObjTree::get_vis(int li)
{
	if(li < count && list[li] && list[li]->Id >= GO_PLOT && list[li]->Id < GO_GRAPH)
		return ((Plot*)list[li])->hidden ? 0 : 1;
	return 2;
}

bool
ObjTree::set_vis(int li, bool vis)
{
	if(li < count && list[li] && list[li]->Id >= GO_PLOT && list[li]->Id < GO_GRAPH) {
		((Plot*)list[li])->hidden = vis ? 0 : 1;
		list[li]->Command(CMD_MRK_DIRTY, 0L, 0L);
		list[li]->Command(CMD_REDRAW, 0L, 0L);
		}
	return false;
}
