//spreadwin.cpp, (c)2000, 2001, 2002, 2003, 2004 by R. Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags

#ifdef _WINDOWS
	#include <io.h>					//for read/write
#else
	#define O_BINARY 0x0
	#include <unistd.h>
#endif

extern const LineDEF BlackLine;
extern EditText *CurrText;
extern char *LoadFile;
extern char TmpTxt[];
extern Default defs;
extern UndoObj Undo;
static ReadCache *Cache = 0L;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Get item from *.csv file
bool GetItemCSV(char *Text, int cbText)
{
	char c;
	int i;

	for (i = 0,	*Text = 0; i < cbText; ) {
		c = Cache->Getc();
		switch(c) {
		case ',':			//column separator
			Text[i] = 0;
			return true;
		case 0x0a:			//end of line: mark by false return but text o.k.
			Text[i] = 0;
			return false;
		default:
			if(c > 0x20) Text[i++] = c;	//printable character
			else if(i >0 && c == 0x20) Text[i++] = c;
			else if(!c && Cache->IsEOF()) {
				Text[i] = 0;
				return false;
				}
			else Text[i] = 0;	//ignore non printing characters
			}
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// process a memory block (i.e. clipboard data) as if file input
int ProcMemData(GraphObj *g, unsigned char *ptr, bool dispatch)
{
	int i, RetVal = FF_UNKNOWN, nt, nl, nc, cc;

	if(ptr) {
		for(i = nt = nl = nc = 0; ptr[i]; i++) {
			switch(ptr[i]) {
			case 0x09:				//tab
				nt++;
				break;
			case 0x0a:				//LF
				nl++;
				break;
			case ',':
				nc++;
				break;
				}
			}
		if(dispatch && i && !nt && !nl)
			for(i = 0; ptr[i]; i++){
				cc = ptr[i];
				g->Command(CMD_ADDCHAR, (void*)(& cc), 0L);
				}
		else if(nt) RetVal = FF_TSV;
		else if(nl && ptr[0] == '<') RetVal = FF_XML;
		else if(nc == nl && defs.DecPoint[0] == ',') RetVal = FF_TSV;
		else if(nl && nc && 0 == (nc % nl)) RetVal = FF_CSV;
		else if(nl) RetVal = FF_TSV;
		if(dispatch) switch(RetVal) {
		case FF_CSV:	g->Command(CMD_PASTE_CSV, ptr, 0L);	break;
		case FF_TSV:	g->Command(CMD_PASTE_TSV, ptr, 0L);	break;
		case FF_XML:	g->Command(CMD_PASTE_XML, ptr, 0L); break;
			}
		}
	return RetVal;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// This graphic object displays a spreadsheet
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class SpreadWin:public GraphObj{
public:
	anyOutput *w;
	POINT ssOrg;
	RECT currRC;

	SpreadWin(DataObj *Data);
	~SpreadWin();
	void DoPlot(anyOutput *target);
	bool Command(int cmd, void *tmpl, anyOutput *o);

	bool ShowGrid(int CellWidth, int CellHeight, int FirstWidth);
	bool PrintData(anyOutput *o);

private:
	int ch, cw, fw;				//cell height and width, row button width
	char *filename;
	TextDEF ssText;
	ssButton **cButtons, **rButtons;
	ssButton *aButton;
	DataObj *d;
	int NumGraphs;
	Graph **g;
};

SpreadWin::SpreadWin(DataObj *Data):GraphObj(0L, Data)
{
	d = Data;	g = 0L;		ssOrg.x =  ssOrg.y = 0;		NumGraphs = 0;
	ch = 19;	cw = 76;	fw = 32;	filename=0L;	aButton = 0L;
	if(w = NewDispClass(this)){
		w->hasHistMenu = true;
		ssText.RotBL = ssText.RotCHAR = 0.0;
		ssText.fSize = 0.0f;
#ifdef _WINDOWS
		ssText.iSize = w->un2iy(defs.GetSize(SIZE_CELLTEXT));
#else
		ssText.iSize = w->un2iy(defs.GetSize(SIZE_CELLTEXT)*.8f);
#endif
		ssText.Align = TXA_VTOP | TXA_HLEFT;		ssText.Mode = TXM_TRANSPARENT;
		ssText.Style = TXS_NORMAL;					ssText.ColBg = 0x00d8d8d8L;
		ssText.ColTxt = 0x00000000L;				ssText.text = 0L;
		ssText.Font = FONT_HELVETICA;				w->SetTextSpec(&ssText);
		w->SetMenu(MENU_SPREAD);					w->FileHistory();
		w->Erase(0x00d8d8d8L);						w->Caption("RLPlot data");
		cw = w->un2ix(defs.GetSize(SIZE_CELLWIDTH));
		ch = w->un2iy(defs.GetSize(SIZE_CELLTEXT)) + 2;
		fw = 32;
		}
	cButtons = rButtons = 0L;
	Id = GO_SPREADDATA;
}

SpreadWin::~SpreadWin()
{
	int i;

	if(cButtons) {
		for(i = 0; cButtons[i]; i++) if(cButtons[i]) delete(cButtons[i]);
		free(cButtons);
		}
	if(rButtons) {
		for(i = 0; rButtons[i]; i++) if(rButtons[i]) delete(rButtons[i]);
		free(rButtons);
		}
	if (aButton) delete(aButton);
	if (w) delete w;
	if (g && NumGraphs) {
		for(i = 0; i < NumGraphs; i++) if(g[i]) {
			g[i]->Command(CMD_CAN_DELETE, 0L, 0L);
			delete(g[i]);
			}
		free (g);
		}
	if(filename) free(filename);	filename=0L;
}

void
SpreadWin::DoPlot(anyOutput *o)
{
	o->ActualSize(&currRC);
	o->StartPage();
	d->Command(CMD_DOPLOT, (void*)this, o);
	o->EndPage();
}

bool
SpreadWin::Command(int cmd, void *tmpl, anyOutput *o)
{
	char *Name;
	Graph **g1, *g2;
	int i, j, k;
	MouseEvent *mev;

	if(d) {
		switch(cmd) {
		case CMD_DROP_GRAPH:
			if(o) o->FileHistory();
			if(!g) g = (Graph **)calloc(2, sizeof(Graph*));
			else {
				g1 = (Graph **)calloc(NumGraphs+2, sizeof(Graph*));
				if(!g1) return false;
				for(i = 0; i < NumGraphs; i++) g1[i] = g[i];
				free(g);
				g = g1; 
				}
			if(!g) return false;
			g[NumGraphs] = (Graph *)tmpl;
			if(g[NumGraphs]){
				g[NumGraphs]->parent = this;
				NumGraphs++;
				g[NumGraphs-1]->Command(CMD_SET_DATAOBJ, (void *)d, 0L);
				g[NumGraphs-1]->DoPlot(NULL);
				}
			return true;
		case CMD_NEWGRAPH:
			if((g2 = new Graph(this, d, 0L)) && g2->PropertyDlg() && 
				Command(CMD_DROP_GRAPH, g2, o))return true;
			else if(g2) DeleteGO(g2);
			return false;
		case CMD_NEWPAGE:
			if((g2 = new Page(this, d)) && 
				Command(CMD_DROP_GRAPH, g2, o))return true;
			else if(g2) DeleteGO(g2);
			return false;
		case CMD_DELGRAPH:
			if (g && NumGraphs) {
				for(i = 0; i < NumGraphs; i++) if(g[i]){
					g[i]->Command(CMD_CAN_DELETE, 0L, 0L);
					DeleteGO(g[i]);
					}
				free (g);
				}
			g = 0L;
			NumGraphs = 0;
			Undo.Flush();
			return true;
		case CMD_DELOBJ:
			if (g) {
				for(i = 0; i <= NumGraphs; i++) {
					if(g[i] == (Graph *)tmpl) {
						delete (g[i]);
						g[i] = 0L;
						return true;
						}
					}
				}
			return false;
		case CMD_SAVEDATAAS:
			if((Name = SaveDataAsName(filename)) && Name[0]){
				if(o) o->FileHistory();
				if(Name && d->WriteData(Name)) {
					if(filename) free(filename);
					filename = strdup(Name);
					}
				}
			return true;
		case CMD_DROPFILE:
			if(IsRlpFile((char*)tmpl)) return OpenGraph(this, (char*)tmpl, 0L);
			else if(d->ReadData((char*)tmpl, 0L, FF_UNKNOWN)){
				if(filename) free(filename);
				filename = strdup((char*)tmpl);
				return Command(CMD_SETSCROLL, 0L, w);
				}
			return false;
		case CMD_OPEN:
			if((Name = OpenDataName(filename)) && Name[0]){
				if(o) o->FileHistory();
				if(IsRlpFile(Name)) return OpenGraph(this, Name, 0L);
				else if(d->ReadData(Name, 0L, FF_UNKNOWN)){
					if(filename) free(filename);
					filename = strdup(Name);
					return Command(CMD_SETSCROLL, 0L, w);
					}
				}
			return false;
		case CMD_ADDROWCOL:
			DoSpShSize(d);
			return true;
		case CMD_MOUSE_EVENT:
			if(o && (mev = (MouseEvent*)tmpl) && mev->y > o->MenuHeight) {
				if(mev->x < fw && mev->y < (o->MenuHeight + ch) && aButton) {
					aButton->Command(cmd, tmpl, o);
					}
				else if(mev->x < fw && rButtons) {
					i = (mev->y - o->MenuHeight - ch)/ch;
					if(rButtons[i]) rButtons[i]->Command(cmd, tmpl, o);
					}
				else if(mev->y < (o->MenuHeight + ch) && cButtons) {
					i = (mev->x - fw)/cw;
					if(cButtons[i]) cButtons[i]->Command(cmd, tmpl, o);
					}
				else if(o->MrkMode == MRK_SSB_DRAW) o->HideMark();
				}
		case CMD_COPY_SYLK:		case CMD_ADDCHAR:
		case CMD_COPY_TSV:		case CMD_COPY_XML:		case CMD_QUERY_COPY:
		case CMD_TAB:			case CMD_SHTAB:
		case CMD_CURRLEFT:		case CMD_CURRIGHT:		case CMD_CURRUP:
		case CMD_CURRDOWN:		case CMD_SHIFTRIGHT:	case CMD_POS_FIRST:
		case CMD_POS_LAST:		case CMD_SHIFTLEFT:		case CMD_DELETE:
		case CMD_TOOLMODE:		case CMD_FILLRANGE:		case CMD_PASTE_TSV:
		case CMD_PASTE_XML:		case CMD_PASTE_CSV:		case CMD_CUT:
			return d->Command(cmd, tmpl, o); 
		case CMD_SETSCROLL:
			HideTextCursor();
			o->ActualSize(&currRC);
			k = (currRC.bottom-currRC.top)/ch;
			d->GetSize(&i, &j);
			o->SetScroll(true, 0, j, k, ssOrg.y);
			k = (currRC.right-currRC.left)/cw;
			o->SetScroll(false, 0, i, k, ssOrg.x);
			DoPlot(o);
			return true;
		case CMD_PAGEUP:
			k = (currRC.bottom-currRC.top)/ch;
			k = k > 3 ? k-2 : 1;
			ssOrg.y = ssOrg.y > k ? ssOrg.y-k : 0;
			return Command(CMD_SETSCROLL, tmpl, o);
		case CMD_PAGEDOWN:
			k = (currRC.bottom-currRC.top)/ch;
			k = k > 3 ? k-2 : 1;
			d->GetSize(&i, &j);
			ssOrg.y = ssOrg.y < j-k*2 ? ssOrg.y+k : j > k ? j-k : 0;
			return Command(CMD_SETSCROLL, tmpl, o);
		case CMD_SETHPOS:
			ssOrg.x = *((int*)tmpl) >= 0 ? *((long*)tmpl) : 0L;
			return Command(CMD_SETSCROLL, tmpl, o);
		case CMD_SETVPOS:
			ssOrg.y = *((int*)tmpl) >= 0 ? *((long*)tmpl) : 0L;
			return Command(CMD_SETSCROLL, tmpl, o);
		case CMD_SETFOCUS:
			return true;
		case CMD_KILLFOCUS:
			return true;
		case CMD_TEXTSIZE:
			if(tmpl)ssText.iSize = *((int*)tmpl);
			break;
		case CMD_CONFIG:
			return defs.PropertyDlg();
		case CMD_NONE:
			return true;
		case CMD_PRINT:
			return PrintData(o);
			}
		}
	return false;
}

bool
SpreadWin::ShowGrid(int CellWidth, int CellHeight, int FirstWidth)
{
	int i, c, nr, nc;
	RECT rc;
	char text[20];
	POINT grid[2];
	TextDEF ButtText;
	bool redim = false;

	//DEBUG: Make sure not to draw the grid much larger than the dektop
	if(ch != CellHeight || cw != CellWidth || fw != FirstWidth) redim = true;
	ch = CellHeight;	cw = CellWidth;		fw = FirstWidth;
	if(redim){
		if(cButtons) {
			for(i = 0; cButtons[i]; i++) if(cButtons[i]) delete(cButtons[i]);
			free(cButtons);
			}
		if(rButtons) {
			for(i = 0; rButtons[i]; i++) if(rButtons[i]) delete(rButtons[i]);
			free(rButtons);
			}
		if(aButton) delete(aButton);		aButton = 0L;
		cButtons = rButtons = 0L;
		}
	if(!aButton) aButton = new ssButton(this, 0, w->MenuHeight, FirstWidth, CellHeight);
	memcpy(&ButtText, &ssText, sizeof(TextDEF));
	ButtText.Align = TXA_HCENTER | TXA_VCENTER;
	w->GetSize(&rc);
	if(!cButtons) {
		c = (rc.right/CellWidth)+1;
		cButtons = (ssButton **)calloc(c, sizeof(ssButton*));
		for(i = 0; i < (c-1); i++) cButtons[i] = 
			new ssButton(this, i*CellWidth+FirstWidth, w->MenuHeight, 
			CellWidth+1, CellHeight);
		}
	if(!rButtons) {
 		c = (rc.bottom/CellHeight)+1;
		rButtons = (ssButton**)calloc(c, sizeof(ssButton*));
		for(i = 0; i < (c-1); i++) rButtons[i] = 
			new ssButton(this, 0, i*CellHeight+CellHeight+w->MenuHeight, 
			FirstWidth, CellHeight);
		}
	w->SetLine((LineDEF *)&BlackLine);
	d->GetSize(&nc, &nr);
	grid[0].x = rc.left+FirstWidth;
	i = (nc-ssOrg.x)*CellWidth+FirstWidth;
	grid[1].x = i < rc.right ? i : rc.right;
	if(rButtons) for(i = 0; rButtons[i]; i++) {
		sprintf(text, "%d", i+1+ssOrg.y);
		if(rButtons[i]) {
			rButtons[i]->Command(CMD_SETTEXTDEF, &ButtText, w);
			rButtons[i]->Command(CMD_SETTEXT, text, w);
			rButtons[i]->DoPlot(w);
			w->SetLine((LineDEF *)&BlackLine);
			}
		grid[0].y = grid[1].y = w->MenuHeight + i*CellHeight - 1;
		if(i < (2+nr-ssOrg.y)) w->oSolidLine(grid);
		}
	grid[0].y = rc.top+CellHeight+w->MenuHeight;
	i = (1+nr-ssOrg.y)*CellHeight;
	grid[1].y = (i+w->MenuHeight)< rc.bottom ? i+w->MenuHeight : rc.bottom;
	if(cButtons) for(i = 0; cButtons[i]; i++) {
		if(cButtons[i]) {
			cButtons[i]->Command(CMD_SETTEXTDEF, &ButtText, w);
			cButtons[i]->Command(CMD_SETTEXT, Int2ColLabel(i+ssOrg.x, true), w);
			cButtons[i]->DoPlot(w);
			w->SetLine((LineDEF *)&BlackLine);
			}
		grid[0].x = grid[1].x = i*CellWidth+FirstWidth-1;
		if(i <= (nc-ssOrg.x)) w->oSolidLine(grid);
		}
	w->SetTextSpec(&ssText);
	if(aButton) aButton->DoPlot(w);
	return true;
}

bool
SpreadWin::PrintData(anyOutput *o)
{
	int i, j, k, l, pfw, pcw, pch, rpp, cpp, nc, nr, ix, iy, cpages;
	RECT rc, margin, margin_first;
	POINT pp_pos, ss_pos, grid[2];
	LineDEF Line1, Line2;
	TextDEF td, tdp;
	bool bContinue;
	time_t ti = time(0L);
	double val;

	Line1.patlength = Line2.patlength = 1.0;
	Line1.color = Line2.color = 0x0;			//black gridlines
	Line1.pattern = Line2.pattern = 0x0;		//solid lines
	switch(defs.cUnits) {
	case 1:		Line1.width = 0.01;			break;
	case 2:		Line1.width = 0.003937;		break;
	default:	Line1.width = 0.1;			break;
		}
	Line2.width = Line1.width * 3.0;
	d->GetSize(&nc, &nr);
	if(!(o->StartPage())) return false;
	pfw = iround(o->hres * ((double)fw)/w->hres);
	pcw = iround(o->hres * ((double)cw)/w->hres);
	pch = iround(o->vres * ((double)ch)/w->vres + o->vres/20.0);
	o->ActualSize(&rc);
	tdp.ColTxt = 0x0;	tdp.ColBg = 0x00ffffffL; 
	tdp.fSize = tdp.RotBL = tdp.RotCHAR = 0.0;	tdp.Align = TXA_HRIGHT | TXA_VCENTER;
#ifdef _WINDOWS
	tdp.iSize = iround(o->hres/6.0);
#else
	tdp.iSize = iround(o->hres/7.5);
#endif
	tdp.Mode = TXM_TRANSPARENT;
	tdp.Style = TXS_NORMAL;				tdp.Font = FONT_HELVETICA;	tdp.text = 0L;
	memcpy(&td, &ssText, sizeof(TextDEF));	
	td.Align = TXA_HCENTER | TXA_VCENTER;	
	td.Style = TXS_NORMAL;	td.iSize = 0;
#ifdef _WINDOWS
	td.fSize = defs.GetSize(SIZE_CELLTEXT);
#else
	td.fSize = defs.GetSize(SIZE_CELLTEXT)*.8f;
#endif
	margin.left = iround(o->hres);	margin.right = iround(o->hres/2.0);
	margin.top = margin.bottom = iround(o->hres);
	memcpy(&margin_first, &margin, sizeof(RECT));
	cpp = (rc.right - margin.left - margin.right - pfw)/pcw;
	rpp = (rc.bottom - margin.top - margin.bottom - pch)/pch;
	pp_pos.x = margin.left;		pp_pos.y = margin.top;
	ss_pos.x = 0;				ss_pos.y = 0;		cpages = 1;
	do {
		pp_pos.x = margin.left;		pp_pos.y = margin.top;
		k = (ss_pos.x + cpp) > nc ? nc-ss_pos.x : cpp;
		l = (ss_pos.y + rpp) > nr ? nr-ss_pos.y : rpp;
		grid[0].y = margin.top +pch; grid[1].y = grid[0].y + l * pch;
		o->SetLine(&Line2);
		grid[0].x = grid[1].x = pp_pos.x;		o->oSolidLine(grid);
		grid[0].x = grid[1].x = pp_pos.x+pfw;	o->oSolidLine(grid);
		o->SetLine(&Line1);
		for(i = 1; i <= k; i++) {			//vertical grid
			grid[0].x = grid[1].x = pp_pos.x + pfw + i * pcw;
			o->oSolidLine(grid);
			}
		grid[0].x = margin.left+pfw;	grid[1].x = grid[0].x + k * pcw;
		o->SetLine(&Line2);
		grid[0].y = grid[1].y = pp_pos.y;		o->oSolidLine(grid);
		grid[0].y = grid[1].y = pp_pos.y+pch;	o->oSolidLine(grid);
		o->SetLine(&Line1);
		for(i = 1; i <= l; i++) {			//horizontal grid
			grid[0].y = grid[1].y = pp_pos.y + pch + i * pch;
			o->oSolidLine(grid);
			}
		o->SetLine(&Line2);
		td.Align = TXA_HCENTER | TXA_VCENTER;	o->SetTextSpec(&td);
		grid[0].y = margin.top;	 grid[1].y = grid[0].y + pch;
		iy = margin.top + (pch >>1);
		for(i = 0; i <= k; i++) {			//column headers
			grid[0].x = grid[1].x = pp_pos.x + pfw + i * pcw;
			o->oSolidLine(grid);			ix = grid[0].x + (pcw >>1);
			if(i < k) o->oTextOut(ix, iy, Int2ColLabel(i+ss_pos.x, true), 0);
			}
		td.Align = TXA_HRIGHT | TXA_VCENTER;	
		o->SetTextSpec(&td);	ix = margin.left + pfw - iround(o->hres/20.0);
		grid[0].x = margin.left;	grid[1].x = grid[0].x + pfw;
		for(i = 0; i <= l; i++) {			//row labels
			grid[0].y = grid[1].y = pp_pos.y + pch + i * pch;
			o->oSolidLine(grid);	iy = grid[0].y + (pch >>1);
			sprintf(TmpTxt, "%d", i+1+ss_pos.y);
			if(i < l) o->oTextOut(ix, iy, TmpTxt, 0);
			}
		for(i = 0; i < k; i++) {			//spreadsheet data
			for (j = 0; j < l; j++) {
				if(d->GetText(j+ss_pos.y, i+ss_pos.x, TmpTxt, TMP_TXT_SIZE)){
					if(d->etRows[j+ss_pos.y][i+ss_pos.x]->isFormula()){
						td.Align = TXA_HRIGHT | TXA_VCENTER;
						ix = margin.left+pfw+pcw + i*pcw - iround(o->hres/20.0);
						d->etRows[j+ss_pos.y][i+ss_pos.x]->GetValue(&val);
						sprintf(TmpTxt,"%g", val);
						}
					else if(d->etRows[j+ss_pos.y][i+ss_pos.x]->isValue()){
						td.Align = TXA_HRIGHT | TXA_VCENTER;
						ix = margin.left+pfw+pcw + i*pcw - iround(o->hres/20.0);
						}
					else {
						td.Align = TXA_HLEFT | TXA_VCENTER;
						ix = margin.left+pfw + i*pcw + iround(o->hres/20.0);
						}
					iy = pp_pos.y + pch + (pch>>1) + j * pch;
					o->SetTextSpec(&td);
					o->oTextOut(ix, iy, TmpTxt, 0);
					}
				}
			}
		//prepare for next table
		ss_pos.x += k;			bContinue = false;
		if(ss_pos.x >= nc) {ss_pos.x = 0; ss_pos.y += l; }
		if(ss_pos.y < nr) {
			ix = pfw + (cpp % nc)*pcw + iround(o->hres/3.5);
			iy = (l+2)*pch;
			if((margin.left + ix + pfw + k*pcw) < (rc.right-margin.right)) {
				margin.left += pfw + k*pcw + iround(o->hres/3.5);
				bContinue = true;
				}
			else if((margin.top + iy + pch + l*pch) < (rc.bottom - margin.bottom)) {
				margin.top += iy;	margin.left = margin_first.left;
				bContinue = true;
				}
			else {
				tdp.Align = TXA_HRIGHT | TXA_VCENTER; o->SetTextSpec(&tdp);
				sprintf(TmpTxt, "page %d", cpages++);
				o->oTextOut(rc.right-margin.right, rc.bottom-(margin.bottom>>1), TmpTxt, 0);
				tdp.Align = TXA_HCENTER | TXA_VCENTER; o->SetTextSpec(&tdp);
				sprintf(TmpTxt, "%s", ctime(&ti));	TmpTxt[24] = 0;
				o->oTextOut((rc.right-rc.left)>>1, rc.bottom-(margin.bottom>>1), TmpTxt, 0);
				tdp.Align = TXA_HLEFT | TXA_VCENTER; o->SetTextSpec(&tdp);
				sprintf(TmpTxt, "RLPlot %s", SZ_VERSION);
				o->oTextOut(margin_first.left, rc.bottom-(margin.bottom>>1), TmpTxt, 0);
				memcpy(&margin, &margin_first, sizeof(RECT));
				bContinue = true;
				o->Eject();
				}
			}
		}while(bContinue);
	tdp.Align = TXA_HRIGHT | TXA_VCENTER; o->SetTextSpec(&tdp);
	sprintf(TmpTxt, "page %d", cpages++);
	o->oTextOut(rc.right-margin.right, rc.bottom-(margin.bottom>>1), TmpTxt, 0);
	tdp.Align = TXA_HCENTER | TXA_VCENTER; o->SetTextSpec(&tdp);
	sprintf(TmpTxt, "%s", ctime(&ti));	TmpTxt[24] = 0;
	o->oTextOut((rc.right-rc.left)>>1, rc.bottom-(margin.bottom>>1), TmpTxt, 0);
	tdp.Align = TXA_HLEFT | TXA_VCENTER; o->SetTextSpec(&tdp);
	sprintf(TmpTxt, "RLPlot %s", SZ_VERSION);
	o->oTextOut(margin_first.left, rc.bottom-(margin.bottom>>1), TmpTxt, 0);
	o->EndPage();
	return true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// This data object is a spreadsheet
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class SpreadData:public DataObj{
public:
	SpreadData();
	~SpreadData();
	bool Init(int nRows, int nCols);
	bool mpos2dpos(POINT *mp, POINT *dp);
	bool Select(POINT *p);
	void MarkRange(char *range);
	void HideMark(bool cclp);
	bool WriteData(char *FileName);
	bool AddCols(int nCols);
	bool AddRows(int nRows);
	bool ChangeSize(int nCols, int nRows);
	void DoPlot(anyOutput *o);
	bool DelRange();
	bool PasteRange(int cmd, char *txt);
	bool InitCopy(int cmd, void *tmpl, anyOutput *o);
	bool Command(int cmd, void *tmpl, anyOutput *o);
	bool ReadData(char *FileName, unsigned char *buffer, int type);

	bool ReadXML(char *file, unsigned char *buffer, int type);
	bool ReadTSV(char *file, unsigned char *buffer, int type);
	bool MemList(unsigned char **ptr, int type);

private:
	int CellHeight, CellWidth, FirstWidth, r_disp, c_disp;
	RECT rcCopy, cp_src_rec;				//bounding rectangle for copy range
	bool bActive, new_mark, bCopyCut;
	POINT currpos;
	anyOutput *w;
	SpreadWin *Disp;
	char *m_range, *c_range;	//mark and copy ranges
	char *err_msg;				//error message
};

SpreadData::SpreadData()
{
	Disp = 0L;	m_range = 0L;	c_range = 0L;	w = 0L;	err_msg=0;
	CellWidth = CellHeight = FirstWidth = 0;
	currpos.x = currpos.y = r_disp = c_disp = 0;
	bActive = bCopyCut = false;
	rcCopy.left = rcCopy.right = rcCopy.top = rcCopy.bottom = 0;
	cp_src_rec.left = cp_src_rec.right = cp_src_rec.top = cp_src_rec.bottom = 0;
	if(defs.IniFile && FileExist(defs.IniFile)) {
		OpenGraph(0L, defs.IniFile, 0L);
		}
}

SpreadData::~SpreadData()
{
	FlushData();
	if(Disp) delete Disp;			Disp = 0L;
	if(m_range) free(m_range);		m_range = 0L;
	if(c_range) free(c_range);		c_range = 0L;
}

bool
SpreadData::Init(int nRows, int nCols)
{
	int i, j;
	POINT p1, p2;
	RECT rc;

	rcCopy.left = rcCopy.top = 0;	rcCopy.bottom = cRows = nRows;
	rcCopy.right = cCols = nCols;	currpos.x = currpos.y = 0;
	new_mark = bCopyCut = false;
	if(!Disp) {
		Disp = new SpreadWin(this);
		w = Disp->w;
		if(w) {
			CellWidth = w->un2ix(defs.GetSize(SIZE_CELLWIDTH));
			CellHeight = w->un2iy(defs.GetSize(SIZE_CELLTEXT)) + 2;
			FirstWidth = 32;
			w->ActualSize(&rc);
			r_disp = (rc.bottom-rc.top)/CellHeight+1;
			c_disp = (rc.right-rc.left)/CellWidth+1;
			}
		else return false;
		Disp->ShowGrid(CellWidth, CellHeight, FirstWidth);
		}
	if(etRows)FlushData();
	etRows = (EditText ***)calloc (cRows, sizeof(EditText **));
	if(etRows) for(i = 0, p1.x = FirstWidth, p1.y = CellHeight + w->MenuHeight; i < cRows; i++) {
		etRows[i] = (EditText **)calloc(cCols, sizeof(EditText *));
		p2.y = p1.y + CellHeight;
		if(etRows[i]) for(j = 0; j < cCols; j++) {
			p2.x = p1.x + CellWidth;
#ifdef _DEBUG
			char text[20];
			sprintf (text, "%.2f", i*10.0 + j);
			etRows[i][j] = new EditText(this, p1, p2, text);
#else
			etRows[i][j] = new EditText(this, p1, p2, 0L);
#endif
			if(etRows[i][j]) etRows[i][j]->Redraw(w, false);
			p1.x += CellWidth;
			}
		p1.y += CellHeight;
		p1.x = FirstWidth;
		}
	if (LoadFile) {
		strcpy(TmpTxt, LoadFile);	//we will reenter by recursion !
		free(LoadFile);
		LoadFile = 0L;
		Disp->Command(CMD_DROPFILE, TmpTxt, w);
		}
	return true;
}

bool
SpreadData::mpos2dpos(POINT *mp, POINT *dp)
{
	if(mp->x < (FirstWidth+10) && mp->x > FirstWidth && Disp->ssOrg.x >0) {
		Disp->ssOrg.x -= 1;
		if(CurrText) CurrText->Update(2, w, mp);		CurrText = 0L;
		Disp->Command(CMD_SETSCROLL, 0L, w);
		mp->x += CellWidth;
		}
	if(mp->y < (w->MenuHeight + CellHeight+10) && mp->y > (w->MenuHeight + CellHeight) && Disp->ssOrg.y >0) {
		Disp->ssOrg.y -= 1;
		if(CurrText) CurrText->Update(2, w, mp);		CurrText = 0L;
		Disp->Command(CMD_SETSCROLL, 0L, w);
		mp->y += CellHeight;
		}
	if(mp->x > (Disp->currRC.right-w->MenuHeight-10) && Disp->ssOrg.x < (cCols-2)) {
		Disp->ssOrg.x += 1;
		if(CurrText) CurrText->Update(2, w, mp);		CurrText = 0L;
		Disp->Command(CMD_SETSCROLL, 0L, w);
		mp->x -= CellWidth;
		}
	if(mp->y > (Disp->currRC.bottom-10) && Disp->ssOrg.y < (cRows-5)) {
		Disp->ssOrg.y += 1;
		do {
			mp->y -= CellHeight;
		} while(mp->y > (Disp->currRC.bottom-CellHeight));
		if(CurrText) CurrText->Update(2, w, mp);		CurrText = 0L;
		Disp->Command(CMD_SETSCROLL, 0L, w);
		}
	dp->y = (mp->y - w->MenuHeight - CellHeight)/CellHeight + Disp->ssOrg.y;
	dp->x = (mp->x - FirstWidth)/CellWidth + Disp->ssOrg.x;
	if(dp->y >= cRows) dp->y = cRows-1;	
	if(dp->x >= cCols) dp->x = cCols-1;
	return true;
}

bool
SpreadData::Select(POINT *p)
{
	if(CurrText && CurrText->isInRect(p)){
		CurrText->Update(1, w, p);
		return true;
		}
	mpos2dpos(p, &currpos);
	if(currpos.y < cRows && currpos.x < cCols && currpos.y >= 0 && currpos.x >= 0) {
		if(etRows[currpos.y][currpos.x]) {
			if(etRows[currpos.y][currpos.x] == CurrText) CurrText->Update(1, w, p);
			else {
				if(CurrText) CurrText->Update(2, w, p);
				CurrText = etRows[currpos.y][currpos.x];
				DoPlot(w);
				CurrText->Update(1, w, p);
				}
			return true;
			}
		}
	if(CurrText) CurrText->Update(2, w, p);		CurrText = 0L;
	return false;
}

void
SpreadData::MarkRange(char *range)
{
	AccRange *nr, *oldr;
	int r, c;

	oldr = nr = 0L;
	if(m_range && range && !strcmp(m_range, range)) return;		//no change
	if(m_range) oldr = new AccRange(m_range);
	if(range) nr = new AccRange(range);
	if(oldr && nr && oldr->GetFirst(&c, &r)) {
		for( ; oldr->GetNext(&c, &r); ) {
			if(r >= Disp->ssOrg.y && r < (r_disp +Disp->ssOrg.y) && c >= Disp->ssOrg.x 
				&& c < (c_disp + Disp->ssOrg.x) && !nr->IsInRange(c, r)){
				etRows[r][c]->Mark(w, etRows[r][c] != CurrText ? 0 : 1);
				}
			}
		}
	if(nr && nr->GetFirst(&c, &r)) {
		for( ; nr->GetNext(&c, &r); ) {
			if(r >= Disp->ssOrg.y && r < (r_disp +Disp->ssOrg.y) && c >= Disp->ssOrg.x 
				&& c < (c_disp + Disp->ssOrg.x))
				etRows[r][c]->Mark(w, etRows[r][c] != CurrText ? 2 : 3);
			}
		}
	if (m_range) free(m_range);	m_range = 0L;
	if(range) m_range = strdup(range);
	if(oldr) delete(oldr);	if(nr) delete(nr);
	new_mark = true;
}

void
SpreadData::HideMark(bool cclp)
{
	if(cclp && c_range && c_range != m_range){
		free(c_range);	c_range = 0L;
		rcCopy.left = rcCopy.top = 0;
		rcCopy.bottom = cRows;		rcCopy.right = cCols;
		}
	if(m_range){
		free(m_range);	m_range = 0L;
		DoPlot(w);
		}
	if(cclp) EmptyClip();
	new_mark = bCopyCut = false;
}

bool
SpreadData::WriteData(char *FileName)
{
	FILE *File;
	int i, j;
	char tmp[800];
	unsigned char *buff = 0L;

	if(!cRows || !cCols || !etRows) return false;
	BackupFile(FileName);
	if(!(File = fopen(FileName, "w")))return false;
	HideMark(true);
	rcCopy.left = rcCopy.top = 0;	rcCopy.bottom = cRows;		rcCopy.right = cCols;
	i = strlen(FileName);
	//test for xml extension
	if(!strcmp(".xml", FileName+i-4) || !strcmp(".XML", FileName+i-4)) {
		MemList(&buff, FF_XML);
		if(buff){
			fprintf(File, "%s", buff);
			free(buff);					fclose(File);
			return true;
			}
		}
	//test for tsv extension
	if(!strcmp(".tsv", FileName+i-4) || !strcmp(".TSV", FileName+i-4)) {
		MemList(&buff, FF_TSV);
		if(buff){
			fprintf(File, "%s", buff);
			free(buff);					fclose(File);
			return true;
			}
		}
	//else write csv
	for(i = 0; i < cRows; i++) {
		for(j = 0; j < cCols; j++) {
			if(etRows[i][j] && etRows[i][j]->GetItem(tmp, sizeof(tmp)))
				fprintf(File, "%s", tmp);
			if(j < (cCols-1)) fprintf(File, ",");
			}
		fprintf(File, "\n");
		}
	fclose(File);
	return true;
}

bool
SpreadData::ReadData(char *FileName, unsigned char *buffer, int type)
{
	int i, j, l;
	char ItemText[20];
	bool success;
	POINT pt;

	if(FileName) {		//read disk file
		if(0 == strcmp(".xml", FileName+strlen(FileName)-4) ||
			0 == strcmp(".XML", FileName+strlen(FileName)-4)){
			if(ReadXML(FileName, buffer, type)){
				rcCopy.left = rcCopy.top = 0;
				rcCopy.right = cCols;
				rcCopy.bottom = cRows;
				return true;
				}
			return false;
			}
		if(0 == strcmp(".tsv", FileName+strlen(FileName)-4) ||
			0 == strcmp(".TSV", FileName+strlen(FileName)-4)){
			if(ReadTSV(FileName, buffer, type)){
				rcCopy.left = rcCopy.top = 0;
				rcCopy.right = cCols;
				rcCopy.bottom = cRows;
				return true;
				}
			return false;
			}
		if(!(Cache = new ReadCache())) return false;
		if(! Cache->Open(FileName)) {
			delete Cache;
			sprintf(TmpTxt, "Error open data file\n\"%s\"", FileName);
			ErrorBox(TmpTxt);
			return false;
			}
		if(!Init(1, 1)) goto ReadError;
		}
	else if(buffer) {	//read memory buffer
		switch(type) {
		case FF_TSV:
			return ReadTSV(FileName, buffer, type);
		case FF_XML:
			return ReadXML(FileName, buffer, type);
		case FF_CSV:
			if(!(Cache = new MemCache(buffer))) return false;
			break;
		default:
			ErrorBox("Read from buffer with\nunknown format failed.");
			return false;
			}
		}
	else return false;
	i = j = 0;
	pt.x = pt.y = 0;		//pt is a dummy argument only
	do {
		if((success = GetItemCSV(ItemText, sizeof(ItemText)-1)) || ItemText[0]){
			if(j >= cCols && !AddCols(j+1)) goto ReadError;
			if(i >= cRows && !AddRows(i+1)) goto ReadError;
			if(etRows[i][j]) etRows[i][j]->SetText("");
			l = strlen(ItemText);
			if(l>1 && ItemText[0] == '"') {
				ItemText[l-1] = 0;
				etRows[i][j]->SetText(ItemText+1);
				etRows[i][j]->Update(20, 0L, &pt);
				}
			else if(l){
				etRows[i][j]->SetText(ItemText);
				etRows[i][j]->Update(10, 0L, &pt);
				}
			j++;
			}
		if(!success && !Cache->IsEOF()) {i++; j = 0;}	//eol
		}while (ItemText[0] || !Cache->IsEOF());		//eof

	Cache->Close();
	delete Cache;
	Cache = 0L;
	if(FileName) {
		rcCopy.left = rcCopy.top = 0;
		rcCopy.right = cCols;
		rcCopy.bottom = cRows;
		}
	Disp->ssOrg.x = Disp->ssOrg.y = 0;
	return true;

ReadError:
	Cache->Close();
	delete Cache;
	Cache = 0L;
	return false;

}

bool
SpreadData::AddCols(int nCols)
{
	EditText **NewRow;
	int i, j;
	POINT p1, p2;

	if (nCols <= cCols) return false;
	if(etRows && etRows[0] && etRows[0][cCols-1]) {
		p1.x = p2.x = (CellWidth + etRows[0][cCols-1]->GetX());		
		p1.y = p2.y = etRows[0][cCols-1]->GetY();
		p2.x += CellWidth;						p2.y += CellHeight;
		}
	else return false;
	for(i = 0; i < cRows; i++) {
		NewRow = (EditText **)realloc(etRows[i], nCols * sizeof(EditText *));
		if(NewRow) {
			for(j = cCols; j < nCols; j++) {
				NewRow[j] = new EditText(this, p1, p2, NULL);
				p1.x += CellWidth;				p2.x += CellWidth;
				}
			etRows[i] = NewRow;
			}
		else {							//memory allocation error
			cCols = nCols;
			//DEBUG: we should warn the user that not all cells are available
			//   or even better we remove some columns up to i rows
			return false;
			}
		p1.x = p2.x = (CellWidth + etRows[0][cCols-1]->GetX());
		p2.x += CellWidth;						p2.y += CellHeight;
		p1.y += CellHeight;
		}
	cCols = nCols;
	return true;
}

bool
SpreadData::AddRows(int nRows)
{
	int i, j, x;
	POINT p1, p2;
	EditText ***NewRows;

	if (nRows <= cRows) return false;
	i = cRows-1;
	if(etRows && etRows[i] && etRows[i][0]) {
		x = p1.x = p2.x = etRows[i][0]->GetX();	p1.y = etRows[i][0]->GetY();
		p1.y += CellHeight;						p2.y = p1.y + CellHeight;
		p2.x += CellWidth;
		}
	else return false;
	NewRows = (EditText ***)realloc(etRows, nRows * sizeof(EditText **));
	if(NewRows) etRows = NewRows;
	else return false;					//memory allocation error
	for(i = cRows; i < nRows; i++){
		etRows[i] = (EditText **)calloc(cCols, sizeof(EditText *));
		if(etRows[i]) {
			for(j = 0; j < cCols; j++) {
				etRows[i][j] = new EditText(this, p1, p2, NULL);
				p1.x += CellWidth;				p2.x += CellWidth;
				}
			}
		else {							//memory allocation error
			cRows = i-1;
			return false;
			}
		p1.y += CellHeight;						p2.y += CellHeight;
		p1.x = p2.x = x;						p2.x += CellWidth;
		}
	cRows = nRows;
	return true;
}

bool
SpreadData::ChangeSize(int nCols, int nRows)
{
	int i, j;
	bool RetVal;
	
	RetVal = true;
	if(nRows && nRows < cRows) {
		for (i = nRows; i < cRows; i++) {
			if(etRows[i]) for (j = 0; j < cCols; j++) {
				if(etRows[i][j]) delete etRows[i][j];
				etRows[i][j] = NULL;
				}
			free(etRows[i]);
			etRows[i] = NULL;
			}
		cRows = nRows;
		}
	if(nCols && nCols < cCols) {
		for (i = 0; i < cRows; i++) {
			for (j = nCols; j < cCols; j++) {
				if(etRows[i][j]) delete etRows[i][j];
				etRows[i][j] = NULL;
				}
			}
		cCols = nCols;
		}
	if(nCols > cCols) if(!AddCols(nCols))RetVal = false;
	if(RetVal && nRows > cRows) if(!AddRows(nRows))RetVal = false;
	if(w) {
		sprintf(TmpTxt, "%d00", nRows);
		w->oGetTextExtent(TmpTxt, 0, &i, &j);
		if(i > FirstWidth) FirstWidth = i;
		Disp->Command(CMD_SETSCROLL, 0L, w);
		}
	rcCopy.left = rcCopy.right = 0;
	rcCopy.right = cCols;
	rcCopy.bottom = cRows;
	return RetVal;
}

void
SpreadData::DoPlot(anyOutput *o)
{
	RECT rc;
	int i, j, r, c;
	AccRange *ar;

	if(!w || !Disp) return;
	w->Erase(0x00d8d8d8L);					w->ActualSize(&rc);
	r_disp = (rc.bottom-rc.top)/CellHeight+1;
	c_disp = (rc.right-rc.left)/CellWidth+1;
	Disp->ShowGrid(CellWidth, CellHeight, FirstWidth);
	rc.top = w->MenuHeight;					rc.bottom = rc.top + CellHeight;
	for(i = 0; i < (cRows - Disp->ssOrg.y) && i < r_disp; i++) {
		rc.left = FirstWidth;				rc.right = rc.left + CellWidth;
		rc.top += CellHeight;				rc.bottom += CellHeight;
		for(j = 0; j< (cCols - Disp->ssOrg.x) && j < cCols; j++) {
			r = i + Disp->ssOrg.y;	c = j + Disp->ssOrg.x; 
			if(r < cRows && r >= 0 && c < cCols && c >=0 && etRows[r][c]){
				etRows[r][c]->SetRec(&rc);	etRows[r][c]->Redraw(w, false);
				}
			rc.left += CellWidth;			rc.right += CellWidth;
			}
		}
	if(m_range && (ar = new AccRange(m_range)) && ar->GetFirst(&c, &r)) {
		for( ; ar->GetNext(&c, &r); ) {
			if(r >= Disp->ssOrg.y && r < (r_disp + Disp->ssOrg.y) && c >= Disp->ssOrg.x 
				&& c < (c_disp + Disp->ssOrg.x))
				etRows[r][c]->Mark(w, etRows[r][c] != CurrText ? 2 : 3);
			}
		delete (ar);
		}
	if(c_range) InitCopy(0, 0L, w);			//move animated rectangle
	w->ActualSize(&rc);				w->UpdateRect(&rc, false);
	if(err_msg) ErrorBox(err_msg);	err_msg = 0L;
}

bool
SpreadData::DelRange()
{
	AccRange *ar;
	int r, c;

	if(m_range && (ar = new AccRange(m_range)) && ar->GetFirst(&c, &r)) {
		for( ; ar->GetNext(&c, &r); ) {
			if(r < cRows && c < cCols){
				if(CurrText && r == currpos.y && c == currpos.x) {
					CurrText->Update(2, 0L, 0L);
					CurrText = 0L;
					}
				if(etRows[r][c] && etRows[r][c]->text){
					etRows[r][c]->SetText("");
					}
				}
			}
		delete (ar);	HideTextCursor();
		}
	HideMark(false);
	return true;
}

bool
SpreadData::PasteRange(int cmd, char *txt)
{
	AccRange *cr;
	int r, c;
	RECT mrk_range;

	if(new_mark && m_range && (cr = new AccRange(m_range)) && cr->GetFirst(&c, &r)) {
		cr->BoundRec(&mrk_range);
		for( ; cr->GetNext(&c, &r); ) if(c >= 0 && c < cCols && r >= 0 && r < cRows){
			currpos.x = c;	currpos.y = r;
			switch(cmd){
			case CMD_PASTE_TSV: ReadTSV(0L, (unsigned char*)txt, FF_TSV);	break;
			case CMD_PASTE_XML: ReadXML(0L, (unsigned char*)txt, FF_XML);	break;
			case CMD_PASTE_CSV: ReadData(0L, (unsigned char*)txt, FF_CSV);	break;
				}
			if((mrk_range.right - mrk_range.left) == (cp_src_rec.right - cp_src_rec.left) &&
				(mrk_range.bottom - mrk_range.top) == (cp_src_rec.bottom - cp_src_rec.top)) break;
			if((mrk_range.right - mrk_range.left) == 1 && (cp_src_rec.right - cp_src_rec.left) > 1) break;
			if((mrk_range.bottom - mrk_range.top) == 1 && (cp_src_rec.bottom - cp_src_rec.top) > 1) break;
			}
		delete cr;		return true;
		}
	else switch(cmd){
		case CMD_PASTE_TSV:
			return ReadTSV(0L, (unsigned char*)txt, FF_TSV);
		case CMD_PASTE_XML:
			return ReadXML(0L, (unsigned char*)txt, FF_XML);
		case CMD_PASTE_CSV:
			return ReadData(0L, (unsigned char*)txt, FF_CSV);
		}
	return false;
}

bool
SpreadData::InitCopy(int cmd, void *tmpl, anyOutput *o)
{
	int r, c;
	AccRange *ar;
	RECT rc_band;
	bool bRet = false;

	if(cmd) {
		bCopyCut = (cmd == CMD_CUT);
		if(rcCopy.right > cCols) rcCopy.right = cCols;
		if(rcCopy.bottom > cRows) rcCopy.bottom = cRows;
		new_mark = false;
		if(m_range && m_range[0]) {
			if(c_range) free(c_range);		c_range = strdup(m_range);
			if(c_range && (ar = new AccRange(c_range))) {
				ar->BoundRec(&rcCopy);
				delete ar;			bRet = true;;
				}
			}
		else if(GetCopyRange(&rcCopy, this)){
			//The range is stored in TmpTxt
			if(!TmpTxt[0]) return false;
			if(m_range) free(m_range);		if(c_range) free(c_range);
			c_range = strdup(TmpTxt);		m_range = strdup(TmpTxt);
			DoPlot(o);		bRet = true;
			}
		}
	if(bRet || !cmd) {		//calculate animated mark
		if(rcCopy.right < Disp->ssOrg.x || rcCopy.bottom < Disp->ssOrg.y) {
			HideCopyMark();
			return bRet;
			}
		c = rcCopy.left >= Disp->ssOrg.x ? rcCopy.left : Disp->ssOrg.x;
		r = rcCopy.top >= Disp->ssOrg.y ? rcCopy.top : Disp->ssOrg.y;
		if(etRows[r][c]){
			rc_band.left = etRows[r][c]->GetX()-1;	rc_band.top = etRows[r][c]->GetY()-1;
			}
		else return bRet;
		c = rcCopy.right < (Disp->ssOrg.x + c_disp) ? rcCopy.right : Disp->ssOrg.x + c_disp;
		r = rcCopy.bottom < (Disp->ssOrg.y + r_disp)? rcCopy.bottom : Disp->ssOrg.y + r_disp;
		if(etRows[r][c]){
			rc_band.right = etRows[r][c]->GetX()+CellWidth;
			rc_band.bottom = etRows[r][c]->GetY()+CellHeight;
			ShowCopyMark(o, &rc_band, 1);
			}
		}
	return bRet;
}

bool
SpreadData::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	static int move_cr = CMD_CURRDOWN;
	MouseEvent *mev;
	POINT p, cp;
	
	if(!o) o = w;
	switch(cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		p.x = mev->x;				p.y = mev->y;
		if((mev->StateFlags & 1) || mev->Action == MOUSE_LBUP) {
			mpos2dpos(&p, &cp);
			if(cp.y >= cRows || cp.x >= cCols || cp.y < 0 || cp.x < 0) return false;
			}
		switch (mev->Action) {
		case MOUSE_LBDOWN:
			bActive = true;
			if(CurrText && !CurrText->isInRect(&p)){
				CurrText->Update(2, w, &p);			 CurrText = 0L;	
				DoPlot(w);
				}
			if(p.x < FirstWidth) cp.x = 0;
			if(p.y < (w->MenuHeight+CellHeight)) cp.y = 0;
			currpos.y = cp.y;					currpos.x = cp.x;
		case MOUSE_LBDOUBLECLICK:		case MOUSE_MOVE:
			if(!bActive) return false;
			if(!m_range && !CurrText && cp.y < cRows && cp.x < cCols && cp.y >= 0 && cp.x >= 0)
				CurrText = etRows[cp.y][cp.x];
			if(mev->Action == MOUSE_MOVE && (mev->StateFlags & 1) 
				&& !(CurrText && CurrText->isInRect(&p))) {
				//mark rectangular range
				if(!m_range && CurrText) {
					CurrText->Update(2, w, &p);		 CurrText = 0L;	
					}
				if(p.x < FirstWidth || p.y < (w->MenuHeight+CellHeight)) {
					i = sprintf(TmpTxt, "%s%d:", Int2ColLabel(p.x < FirstWidth ? 0 : currpos.x, false),
						p.y < (w->MenuHeight+CellHeight) ? 0 : currpos.y+1);
					sprintf(TmpTxt+i, "%s%d", Int2ColLabel(p.x < FirstWidth ? cCols-1 : cp.x, false), 
						p.y < (w->MenuHeight+CellHeight) ? cRows : cp.y+1);
					}
				else {
					i = sprintf(TmpTxt, "%s%d:", Int2ColLabel(currpos.x, false), currpos.y+1);
					sprintf(TmpTxt+i, "%s%d", Int2ColLabel(cp.x, false), cp.y+1);
					}
				if(!CurrText)MarkRange(TmpTxt);
				return true;
				}
			if(mev->Action == MOUSE_LBDOUBLECLICK) bActive = false;
			if(!(mev->StateFlags & 1)) return false;
			if(CurrText && CurrText->isInRect(&p)) {
				return CurrText->Command(CMD_MOUSE_EVENT, o, (DataObj *)mev);
				}
			if(etRows[cp.y][cp.x]) 
				return etRows[cp.y][cp.x]->Command(CMD_MOUSE_EVENT, o, (DataObj *)mev);
			return false;
		case MOUSE_LBUP:
			if(bActive){
				if(p.x < FirstWidth || p.y < (w->MenuHeight+CellHeight)) {
					if(p.x < FirstWidth) {
						currpos.x = 0;	cp.x = cCols-1;
						}
					if(p.y < (w->MenuHeight+CellHeight)) {
						currpos.y = 0;	cp.y = cRows-1;
						}
					i = sprintf(TmpTxt, "%s%d:", Int2ColLabel(currpos.x, false), currpos.y+1);
					sprintf(TmpTxt+i, "%s%d", Int2ColLabel(cp.x, false), cp.y+1);
					MarkRange(TmpTxt);
					}
				if(m_range) {
					CurrText = etRows[currpos.y][currpos.x];	CurrText->Update(1, w, &p);
					DoPlot(w);
					}
				else return Select(&p);
				}
			}
		break;
	case CMD_PASTE_TSV:		case CMD_PASTE_XML:	case CMD_PASTE_CSV:
		if(PasteRange(cmd, (char*)tmpl)) return Disp->Command(CMD_SETSCROLL, 0L, w);
		return false;
	case CMD_UNLOCK:
		HideMark(false);
		break;
	case CMD_ERROR:
		err_msg = (char*)tmpl;
		break;
	case CMD_CLEAR_ERROR:
		err_msg = 0L;
		break;
	case CMD_TOOLMODE:
		HideMark(true);
		if(CurrText) CurrText->Update(2, w, 0L);
		if(CurrText) CurrText->Update(1, w, 0L);
		break;
	case CMD_UPDHISTORY:
		if(w) w->FileHistory();
		break;
	case CMD_ADDCHAR:
		if(CurrText){
			i = *((int*)tmpl);
			if(i == 27) return Command(CMD_TOOLMODE, tmpl, o);
			if(i !=3 && i != 22 && i != 24) HideMark(true);	//Do not hide upon ^C, ^V, ^X
			if(i == 13) return CurrText->Command(move_cr, w, this);
			switch(i) {
			case 8: return CurrText->Command(CMD_BACKSP, w, this);	//Backspace
			default: return CurrText->AddChar(*((int*)tmpl), w);
				}
			}
		break;
	case CMD_CURRIGHT:		case CMD_CURRDOWN:
		move_cr = cmd;
	case CMD_CURRLEFT:		case CMD_CURRUP:		case CMD_SHIFTRIGHT:
	case CMD_POS_FIRST:		case CMD_POS_LAST:		case CMD_SHIFTLEFT:
		if(err_msg) ErrorBox(err_msg);	err_msg = 0L;
		HideMark(false);
		if(CurrText)return CurrText->Command(cmd, w, this);
		return false;
	case CMD_SHTAB:
		if(currpos.y >= cRows) currpos.y = cRows-1;
		if(currpos.x == Disp->ssOrg.x && Disp->ssOrg.x > 0) {
			Disp->ssOrg.x -= 1;
			Disp->Command(CMD_SETSCROLL, 0L, w);
			}
		if(currpos.x > Disp->ssOrg.x && etRows[currpos.y][currpos.x]) {
			currpos.x -=1;
			}
	case CMD_TAB:
		if(currpos.y >= cRows) currpos.y = cRows-1;
		if(cmd == CMD_TAB && currpos.x < (cCols-1) && etRows[currpos.y][currpos.x]) {
			if((FirstWidth+(currpos.x - Disp->ssOrg.x +2)*CellWidth) > Disp->currRC.right){
				Disp->ssOrg.x += 1;
				Disp->Command(CMD_SETSCROLL, 0L, w);
				}
			currpos.x +=1;
			}
		if(CurrText) CurrText->Update(2, w, &p);
		CurrText = etRows[currpos.y][currpos.x];
		if(CurrText){
			CurrText->Update(1, w, &p);
			CurrText->Command(cmd == CMD_TAB ? CMD_POS_FIRST : CMD_POS_LAST, o, this);
			}
		return true;
	case CMD_DELETE:
		if(m_range) DelRange();
		else if(CurrText) return CurrText->Command(cmd, o, this);
		return true;
	case CMD_QUERY_COPY:		case CMD_CUT:
		return InitCopy(cmd, tmpl, w);
	case CMD_GET_CELLDIMS:
		if(tmpl) {
			((int*)tmpl)[0] = FirstWidth;	((int*)tmpl)[1] = CellWidth;
			((int*)tmpl)[2] = CellHeight;
			}
		break;
	case CMD_SET_CELLDIMS:
		if(tmpl) {
			FirstWidth = ((int*)tmpl)[0];	CellWidth = ((int*)tmpl)[1];
			CellHeight = ((int*)tmpl)[2];
			}
		break;
	case CMD_TEXTSIZE:
		if(tmpl){
			CellHeight = *((int*)tmpl);
			return Disp->Command(cmd, tmpl, o);
			}
		return false;
	case CMD_COPY_TSV:
		return MemList((unsigned char**)tmpl, FF_TSV);
	case CMD_COPY_SYLK:
		return MemList((unsigned char**)tmpl, FF_SYLK);
	case CMD_COPY_XML:
		return MemList((unsigned char**)tmpl, FF_XML);
	case CMD_REDRAW:	case CMD_DOPLOT:
		DoPlot(o);
		break;
	case CMD_FILLRANGE:
		FillSsRange(this, &m_range);
		DoPlot(o);
		break;
	}
	return true;
}

bool
SpreadData::ReadXML(char *file, unsigned char *buffer, int type)
{
	int i, row, col, tag;
	bool bContinue, bRet = false;
	ReadCache *XMLcache;
	POINT pt, mov;
	char TmpTxt[1024], *tmp_range;

	if(file) {
		if(!(XMLcache = new ReadCache())) return false;
		if(! XMLcache->Open(file)) {
			delete XMLcache;
			sprintf(TmpTxt, "Error open file\n\"%s\"", file);
			ErrorBox(TmpTxt);
			return false;
			}
		if(!Init(1, 1)) goto XMLError;
		}
	else if(buffer && type == FF_XML){
		if(!(XMLcache = new MemCache(buffer))) return false;
		if(buffer && bCopyCut) {
			tmp_range = m_range;	m_range = c_range;
			c_range = 0L;			DelRange();
			m_range = tmp_range;
			}
		}
	else return false;
	pt.x = pt.y = mov.x = mov.y = 0;
	cp_src_rec.left = cp_src_rec.right = cp_src_rec.top = cp_src_rec.bottom = 0;
	do {
		row = col = 0;
		do {
			TmpTxt[0] = XMLcache->Getc();
			}while(TmpTxt[0] && TmpTxt[0] != '<');
		for(i = 1; i < 5; TmpTxt[i++] = XMLcache->Getc());
		TmpTxt[i] = 0;
		if(!strcmp("<cell", TmpTxt)) tag = 1;
		else if(!strcmp("<pos1", TmpTxt)) tag = 2;
		else if(!strcmp("<pos2", TmpTxt)) tag = 3;
		else tag = 0;
		if(tag) {
			do {
				TmpTxt[0] = XMLcache->Getc();
				}while(TmpTxt[0] && TmpTxt[0] != '"');
			if (TmpTxt[0]) for (i =0; i <10 && ('"' != (TmpTxt[i] =XMLcache->Getc())); i++){
				TmpTxt[i+1] = 0;
				row = (int)atoi(TmpTxt);
				}
			do {
				TmpTxt[0] = XMLcache->Getc();
				}while(TmpTxt[0] && TmpTxt[0] != '"');
			if (TmpTxt[0]) for (i =0; i <10 && ('"' != (TmpTxt[i] =XMLcache->Getc())); i++){
				TmpTxt[i+1] = 0;
				col = (int)atoi(TmpTxt);
				}
			if(tag ==2) {
				mov.x = col;				mov.y = row;
				cp_src_rec.left = col;		cp_src_rec.top = row;
				}
			else if(tag ==3) {
				cp_src_rec.right = col;		cp_src_rec.bottom = row;
				}
			else if(row && col) do {
				do {
					TmpTxt[0] = XMLcache->Getc();
					}while(TmpTxt[0] && TmpTxt[0] != '<');
				for(i = 1; i < 6; TmpTxt[i++] = XMLcache->Getc());
				TmpTxt[i] = 0;
				if(bContinue =(0 == strcmp("<text>", TmpTxt))) {
					for(i = 0; i < 1023 && ('<' != (TmpTxt[i] =XMLcache->Getc())); i++);
					TmpTxt[i] = 0;
					//xml indices start at 1:1 !
					row += currpos.y-1;	col += currpos.x-1;
					if(row >= cRows)AddRows(row+1);
					if(col >= cCols)AddCols(col+1);
					if(i && etRows[row] && etRows[row][col]) {
						if(TmpTxt[0] == '=') {
							MoveFormula(this, TmpTxt, TmpTxt, currpos.x-mov.x, currpos.y-mov.y);
							}
						etRows[row][col]->SetText(TmpTxt);
						etRows[row][col]->Update(20, 0L, &pt);
						}
					}
				}while(!bContinue);
			}
		}while(!XMLcache->IsEOF());
	bRet = true;
XMLError:
	XMLcache->Close();
	delete XMLcache;
	bCopyCut = false;
	return bRet;
}

bool
SpreadData::ReadTSV(char *file, unsigned char *buffer, int type)
{
	int i, row, col;
	char c;
	bool bRet = false;
	ReadCache *TSVcache;
	POINT pt;

	if(file) {
		if(!(TSVcache = new ReadCache())) return false;
		if(! TSVcache->Open(file)) {
			delete TSVcache;
			sprintf(TmpTxt, "Error open file\n\"%s\"", file);
			ErrorBox(TmpTxt);
			return false;
			}
		if(!Init(1, 1)) goto TSVError;
		}
	else if(buffer && type == FF_TSV) {
		if(!(TSVcache = new MemCache(buffer))) return false;
		}
	else return false;
	row = currpos.y;	col = currpos.x;
	pt.x = pt.y = 0;
	do {
		do {
			TmpTxt[0] = TSVcache->Getc();
			switch(TmpTxt[0]) {
			case 0x0d:					//CR
			case 0x0a:					//LF
				if(col == currpos.x) break;
				row ++;			col = currpos.x;		break;
			case 0x09:					//tab
				col ++;			break;
				}
			}while(TmpTxt[0] && TmpTxt[0] < 33);
		for(i = 1; (TmpTxt[i] = c = TSVcache->Getc())>=32; i++)
			if(i >= 4094) i = 4094;
		if(TmpTxt[0] && row >= cRows)AddRows(row+1);
		if(TmpTxt[0] && col >= cCols)AddCols(col+1);
		TmpTxt[i] = 0;
		if(TmpTxt[0] && etRows[row] && etRows[row][col]) {
			etRows[row][col]->SetText(TmpTxt);
			etRows[row][col]->Update(20, 0L, &pt);
			switch(c) {
			case 0x0d:					//CR
			case 0x0a:					//LF
				row ++;			col = currpos.x;		break;
			case 0x09:					//tab
				col ++;			break;
				}
			}
		}while(!TSVcache->IsEOF());
	bRet = true;
TSVError:
	TSVcache->Close();
	delete TSVcache;
	return bRet;
}

bool 
SpreadData::MemList(unsigned char **ptr, int type)
{
	int i, j, k, nc, nl, cb = 0, cbd = 0, size;
	char tmptxt[8000];
	double val;
	*ptr = (unsigned char *)malloc(size = 10000);
	unsigned char *tmpptr;
	bool bLimit = true;

	if(!(*ptr))return false;
	if(rcCopy.left < 0) rcCopy.left = 0;
	if(rcCopy.right >= cCols) rcCopy.right = cCols-1;
	if(rcCopy.top < 0) rcCopy.top = 0;
	if(rcCopy.bottom >= cRows) rcCopy.bottom = cRows-1;
	if(type == FF_SYLK) cbd = sprintf((char*)*ptr, "ID;\n");
	else if(type == FF_XML) {
		cbd = sprintf((char*)*ptr, "<?xml version=\"1.0\" "
		"encoding=\"UTF-8\"?><!DOCTYPE spreadsheet-snippet >"
		"<spreadsheet-snippet rows=\"%d\" columns=\"%d\" >\n", cRows, cCols);
		if(rcCopy.left || rcCopy.top || rcCopy.bottom || rcCopy.right){
			cbd += sprintf((char*)*ptr+cbd, " <pos1 row=\"%d\" "
				"column=\"%d\"></pos1>\n", rcCopy.top, rcCopy.left);
			cbd += sprintf((char*)*ptr+cbd, " <pos2 row=\"%d\" "
				"column=\"%d\"></pos2>\n", rcCopy.bottom, rcCopy.right);
			}
		}
	for(nl =0, i = rcCopy.top; i <= rcCopy.bottom; i++, nl++) {
		for(nc = 0, j = rcCopy.left; j <= rcCopy.right; cb = 0, j++, nc++) {
			switch (type) {
			case FF_TSV:
				if(nl || nc) cb = sprintf(tmptxt,"%s", nc ? "\t" : "\n");
				if(etRows[i] && etRows[i][j] && etRows[i][j]->text){
					for(k = 0; k < 7990 && (etRows[i][j]->text[k]); k++) 
						tmptxt[cb++] = etRows[i][j]->text[k];
					tmptxt[cb] = 0;
					}
				break;
			case FF_SYLK:
				if(etRows[i] && etRows[i][j] && etRows[i][j]->text){
					cb = sprintf(tmptxt, "C;Y%d;X%d;K", nl+1, nc+1);
					if(etRows[i][j]->GetValue(&val)){
						cb += sprintf(tmptxt+cb, "%lf", val);
						while(tmptxt[cb-1] == '0') tmptxt[--cb] = 0;
						if(tmptxt[cb-1] == '.') tmptxt[--cb] = 0;
						cb += sprintf(tmptxt+cb, "\n");
						}
					else {
						etRows[i][j]->GetItem(TmpTxt, 258);
						cb += sprintf(tmptxt+cb, "%s\n", TmpTxt);
						if(cb >= 267) bLimit = false;
						}
					}
				break;
			case FF_XML:
				if(etRows[i] && etRows[i][j] && etRows[i][j]->text){
					cb = sprintf(tmptxt, " <cell row=\"%d\" column=\"%d\" >\n", nl+1, nc+1);
					cb += sprintf(tmptxt+cb, "  <text>");
					for(k = 0; k < 7880 && (etRows[i][j]->text[k]); k++) 
						tmptxt[cb++] = etRows[i][j]->text[k];
					cb += sprintf(tmptxt+cb, "</text>\n </cell>\n");
					}				 
				}
			if((cbd+cb+100) > size){
				if(tmpptr = (unsigned char*)realloc(*ptr, size+10000)) {
					*ptr = tmpptr;					size += 10000;
					}
				else return true;	//not all but something on clipboard
				}
			memcpy(*ptr+cbd, tmptxt, cb+1);
			cbd += cb;
			}
		if(type == FF_SYLK) {
			if(!bLimit) {
				cbd += sprintf((char*)*ptr+cbd, "C;Y%d;X1;K\"long strings were"
				" truncated to 256 characters due to limitations of the SYLK format!\"\n", i+2);
				}
			sprintf((char*)*ptr+cbd, "E\n");
			}
		else if(type == FF_TSV) sprintf((char*)*ptr+cbd,"\n");
		else if(type == FF_XML) sprintf((char*)*ptr+cbd,"</spreadsheet-snippet>\n");
		}
	return true;
}

void SpreadMain(bool show)
{
	static SpreadData *w = 0L;

	if(show) {
		w = new SpreadData();
		if(!w ||!(w->Init(10, 10))){
			delete w;
			w = 0L;
			}
		do_formula(w, 0L, 0L);		//init mfcalc
		}
	else if (w) {
		delete(w);
		}
}


