//reports.cpp, Copyright (c) 2006 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Create statistical reports
//

#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include "TheDialog.h"

extern char TmpTxt[];
extern Default defs;

//prototypes: WinSpec.cpp
void *CreateDlgWnd(char *title, int x, int y, int width, int height, tag_DlgObj *d, DWORD flags);

static int curr_id;
static fRECT dBounds;
static TextDEF txtdef1, txtdef2;
static double linsp1, linsp2;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// init report variables
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static void rep_init()
{
	curr_id = 1;
	txtdef1.ColTxt = txtdef2.ColTxt = 0x0L;
	txtdef1.ColBg = txtdef2.ColBg = 0x00ffffffL;
	txtdef1.fSize = defs.GetSize(SIZE_TEXT);
	txtdef2.fSize = txtdef1.fSize *1.2;
	txtdef1.RotBL = txtdef2.RotBL = 0.0;
	txtdef1.RotCHAR = txtdef2.RotCHAR = 0.0;
	txtdef1.iSize = txtdef2.iSize = 0;
	txtdef1.Align = txtdef2.Align = TXA_HLEFT | TXA_VTOP;
	txtdef1.Mode = txtdef2.Mode = TXM_TRANSPARENT;
	txtdef1.Style = txtdef2.Style = TXS_NORMAL;
	txtdef1.Font = txtdef2.Font = FONT_HELVETICA;
	txtdef1.text = txtdef2.text = 0L;
#ifdef _WINDOWS
	linsp1 = txtdef1.fSize*1.2;	linsp2 = txtdef1.fSize*1.5;
#else
	linsp1 = txtdef1.fSize*1.5;	linsp2 = txtdef1.fSize*2.0;
#endif

}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// utility to add a line to a text buffer
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static void add_to_buff(char** dest, int *pos, int *csize, char *txt)
{
	int len;

	len = strlen(txt);
	if((*pos+len+1)>= *csize) {
		*dest = (char*)realloc(*dest, *csize += 1000);
		}
	if(*dest) {
		*pos += sprintf(*dest+*pos, "%s", txt);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create a text label for a report
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* mk_label(double x, double y, bool moveable, int align, TextDEF *td, char*text)
{
	int csize, pos = 0;
	char *res, line[120];

	if(!(res = (char*)malloc(csize = 1000)))return 0L;
	sprintf(line, "\n[%d=Label]\nPos= %g %g\n", curr_id++, x, y);
	add_to_buff(&res, &pos, &csize, line);
	if(moveable) {
		sprintf(line, "moveable= 1\n");
		add_to_buff(&res, &pos, &csize, line);
		}
	sprintf(line, "TxtDef= 0x00000000 0x00ffffff %g %g %g %d 1 0 0 \"%s\"\n",
		td->fSize, td->RotBL, td->RotCHAR, align, text);
	add_to_buff(&res, &pos, &csize, line);
	return res;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create general information on report page
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static void mk_header(Page *page, char* desc)
{
	time_t ti = time(0L);
	char *txt_obj, label[80];
	double rpos;

	if(!page) return;
	rpos = page->GetSize(SIZE_GRECT_RIGHT) - txtdef1.fSize*5.0;
	if(txt_obj = mk_label(txtdef1.fSize*5.0, page->GetSize(SIZE_GRECT_TOP)+txtdef2.fSize*6.0,
		false, TXA_HLEFT, &txtdef2, desc)) {
		OpenGraph(page, 0L, (unsigned char*)txt_obj, false);
		free(txt_obj);
		}
	if(txt_obj = mk_label(rpos, page->GetSize(SIZE_GRECT_TOP)+txtdef1.fSize*5.0,
		false, TXA_HRIGHT, &txtdef1, ctime(&ti))) {
		OpenGraph(page, 0L, (unsigned char*)txt_obj, false);
		free(txt_obj);
		}
	sprintf(label, "RLPlot %s", SZ_VERSION);
	if(txt_obj = mk_label(rpos, page->GetSize(SIZE_GRECT_BOTTOM)-txtdef1.fSize*6.0,
		false, TXA_HRIGHT, &txtdef1, label)) {
		OpenGraph(page, 0L, (unsigned char*)txt_obj, false);
		free(txt_obj);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create horizontal ruler
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static void mk_hr(GraphObj *parent, double x1, double x2, double y)
{
	int csize, pos = 0;
	char *res, line[120];

	if(!(res = (char*)malloc(csize = 1000)))return;
	sprintf(line, "\n[%d=polyline]\nData=(2){ %g %g %g %g}\n", curr_id++, x1, y, x2, y);
	add_to_buff(&res, &pos, &csize, line);
	sprintf(line, "Line= %g %g 0x0 0x0\n", txtdef1.fSize/20.0, txtdef1.fSize);
	add_to_buff(&res, &pos, &csize, line);
	OpenGraph(parent, 0L, (unsigned char*)res, false);
	free(res);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create report table for anova ...
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static void mk_table(GraphObj *parent, double x, double y, int type, double **dda)
{
	char *cheaders[] = {"<i>df</i>", "<i>SS</i>", "<i>MS</i>", "<i>F</i>", "<i>P</i>"};
	char *rheaders[] = {"Source of variation", type == 2 ? (char*)"Explained by regression":
		(char*)"Among groups", type == 2 ? (char*)"Unexplained":(char*)"Within groups", "Total"};
	char *cfmt[8];
	int i, j, nl, nc[8];
	double posc[8];
	char *txt_obj;

	switch(type) {
	case 1:	case 2:
		nl = 3;	nc[0] = 5;	nc[1] = 3;	nc[2] = 2;
		posc[0] = x + txtdef1.fSize*14.0;		posc[1] = posc[0] + txtdef1.fSize*5.0;
		posc[2] = posc[1] + txtdef1.fSize*6.0;	posc[3] = posc[2] + txtdef1.fSize*6.0;
		posc[4] = posc[3] + txtdef1.fSize*6.0;	cfmt[0] = "%.0lf";
		cfmt[1] = GetNumFormat(floor(log10(dda[2][1])-3.0));	
		cfmt[2] = GetNumFormat(floor(log10(dda[0][2]+dda[0][1])-3.0));
		cfmt[3] = "%0.3lf";						cfmt[4] = "%0.4lf";
		break;
	default: return;
		}
	if(type == 1 || type == 2) {
		if(txt_obj = mk_label(x, y, false, TXA_HLEFT, &txtdef1, rheaders[0])) {
			OpenGraph(parent, 0L, (unsigned char*)txt_obj, false);
			free(txt_obj);
			}
		for(i = 0; i < 5; i++) {
			if(txt_obj = mk_label(posc[i], y, false, TXA_HRIGHT, &txtdef1, cheaders[i])) {
				OpenGraph(parent, 0L, (unsigned char*)txt_obj, false);
				free(txt_obj);
				}
			if(i) posc[i] += linsp1;
			}
		mk_hr(parent, x, posc[4], y + linsp1);
		y += (txtdef1.fSize *1.5);
		}
	for(i = 0; i < nl; i++) {
		if(txt_obj = mk_label(x, y, false, TXA_HLEFT, &txtdef1, rheaders[i+1])) {
			OpenGraph(parent, 0L, (unsigned char*)txt_obj, false);
			free(txt_obj);
			}
		for(j = 0; j < nc[i]; j++) {
			if(j == 4 && dda[i][j] > 0.0 && dda[i][j] < 0.0001)
				strcpy(TmpTxt, "< 0.0001");
			else sprintf(TmpTxt, cfmt[j], dda[i][j]);
			if(txt_obj = mk_label(posc[j], y, false, TXA_HRIGHT, &txtdef1, TmpTxt)) {
				OpenGraph(parent, 0L, (unsigned char*)txt_obj, false);
				free(txt_obj);
				}
			}
		if(i < (nl-2)) y += linsp1;
		else {
			mk_hr(parent, x, posc[4], y + linsp1);
			y += linsp2;
			}
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// create a scatterplot for a report
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char* mk_scatt(double *x, double *y, double *ss, int *ny, int n)
{
	int i, csize, pos, first;
	char *res, line[80];
	double size, linew, tmp;

	if(!(res = (char*)malloc(csize = 1000)))return 0L;
	if(n < 20) size = defs.GetSize(SIZE_SYMBOL);
	else size = defs.GetSize(SIZE_SYMBOL)/2.0 + 20.0 * defs.GetSize(SIZE_SYMBOL)/(2.0 * n);
	linew = defs.GetSize(SIZE_SYM_LINE);
	first = curr_id;
	for(i = pos = 0; i < n && res; i++) {
		sprintf(line, "\n[%d=Symbol]\nPos= %g %g\n", curr_id++, x ? x[i] : (double)(i+1), y[i]);
		add_to_buff(&res, &pos, &csize, line);
		sprintf(line, "Size= %g\n", size);
		add_to_buff(&res, &pos, &csize, line);
		sprintf(line, "Line= %g 1 0x0 0x0\nFillCol= 0x00ffffff\n", linew);
		add_to_buff(&res, &pos, &csize, line);
		}
	if(ss && ny) {
		for(i = 0; i < n && res; i++) {
			if(ny[i] > 1) tmp = sqrt(ss[i]/(ny[i]-1));
			else tmp = 0.0;
			sprintf(line, "\n[%d=ErrorBar]\nPos= %g %g\n", curr_id++, x ? x[i] : (double)(i+1), y[i]);
			add_to_buff(&res, &pos, &csize, line);
			sprintf(line, "Err= %g\nDesc= \"Std. Dev.\"\n", tmp);
			add_to_buff(&res, &pos, &csize, line);
			}
		for(i = 0; i < n && res; i++) {
			if(ny[i] > 1) tmp = sqrt(ss[i]/(ny[i]-1));
			else tmp = 0.0;
			sprintf(line, "\n[%d=Label]\nPos= %g %g\n", curr_id++, x ? x[i]:(double)(i+1), y[i] +tmp);
			add_to_buff(&res, &pos, &csize, line);
			sprintf(line, "Dist= 0 %g\nFlags= 0x00000011\n", -txtdef1.fSize/4.0);
			add_to_buff(&res, &pos, &csize, line);
			sprintf(line, "TxtDef= 0x00000000 0x00ffffff %g %g %g %d 1 0 0 \"%s%d\"\n",
				txtdef1.fSize, txtdef1.RotBL, txtdef1.RotCHAR, TXA_HCENTER | TXA_VBOTTOM, 
				n > 6 ? "" : "n = ", ny[i]);
			add_to_buff(&res, &pos, &csize, line);
			}
		}
	sprintf(line, "\n[%d=PlotScatt]\n", curr_id++);
	add_to_buff(&res, &pos, &csize, line);
	sprintf(line, "Bounds= %g %g %g %g\n", dBounds.Xmin, dBounds.Ymax, dBounds.Xmax, dBounds.Ymin);
	add_to_buff(&res, &pos, &csize, line);
	sprintf(line, "Symbols=(%d){", n);		add_to_buff(&res, &pos, &csize, line);
	for(i = 0; i < n; i++, first++) {
		sprintf(line, "%s%d", i ? "," : "", first);
		add_to_buff(&res, &pos, &csize, line);
		if(i && (i%16) && first < (curr_id-2)) {
			sprintf(line, "\n   ");			add_to_buff(&res, &pos, &csize, line);
			}
		}
	sprintf(line, "}\n");					add_to_buff(&res, &pos, &csize, line);
	if(ss && ny) {
		sprintf(line, "ErrBars=(%d){", n);	add_to_buff(&res, &pos, &csize, line);
		for(i = 0; i < n; i++, first++) {
			sprintf(line, "%s%d", i ? "," : "", first);
			add_to_buff(&res, &pos, &csize, line);
			if(i && (i%16) && first < (curr_id-2)) {
				sprintf(line, "\n   ");		add_to_buff(&res, &pos, &csize, line);
				}
			}
		sprintf(line, "}\n");				add_to_buff(&res, &pos, &csize, line);
		sprintf(line, "Labels=(%d){", n);	add_to_buff(&res, &pos, &csize, line);
		for(i = 0; i < n; i++, first++) {
			sprintf(line, "%s%d", i ? "," : "", first);
			add_to_buff(&res, &pos, &csize, line);
			if(i && (i%16) && first < (curr_id-2)) {
				sprintf(line, "\n   ");		add_to_buff(&res, &pos, &csize, line);
				}
			}
		sprintf(line, "}\n");				add_to_buff(&res, &pos, &csize, line);
		}
	return res;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// one way anova
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *AnovaDlg_Tmpl = 
	"1,2,,DEFAULT,PUSHBUTTON,-1,148,10,45,12\n"
	"2,3,,,PUSHBUTTON,-2,148,25,45,12\n"
	"3,,4,ISPARENT | CHECKED,GROUP,0,0,0,0,0\n"
	"4,10,100,ISPARENT | CHECKED,SHEET,1,5,10,130,80\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,101,,,LTEXT,2,10,30,60,8\n"
	"101,102,,,RANGEINPUT,-15,20,40,100,10\n"
	"102,103,,CHECKED,RADIO1,3,10,55,60,9\n"
	"103,,,LASTOBJ,RADIO1,4,10,65,60,9";
void
rep_anova(GraphObj *parent, DataObj *data)
{
	TabSHEET tab1 = {0, 45, 10, "Anova Input"};
	DlgInfo *AnovaDlg;
	void *dyndata[] = {(void*)&tab1, (void*)"rectangular range for variables",
		(void*)"variables arranged in columns", (void*)"    -\"-           -\"-      in rows"};
	DlgRoot *Dlg;
	void *hDlg;
	double **cols = 0L, tmp, *csums=0L, mtot, *css=0L, ssa, ssw, sst;
	double **res_tab = 0L;
	int i, j, res, nr, nc, ntot, *ncols = 0L;;
	bool bContinue = false;
	AccRange *rD =0L;
	char *mrk, *txt_obj;
	RECT rec;
	Graph *graph;
	Page *page;

	if(!parent || !data) return;
	if(!(AnovaDlg = CompileDialog(AnovaDlg_Tmpl, dyndata))) return;
	if(data->Command(CMD_GETMARK, &mrk, 0L)) {
		strcpy(TmpTxt, mrk);
		}
	else {
		data->ValueRec(&rec);
		i = sprintf(TmpTxt,"%s%d", Int2ColLabel(rec.left,false), rec.top + 1);
		sprintf(TmpTxt+i, ":%s%d", Int2ColLabel(rec.right, false), rec.bottom+1);
		}
	if(!(Dlg = new DlgRoot(AnovaDlg)))return;
	Dlg->ItemCmd(101, CMD_SET_DATAOBJ, data);
	hDlg = CreateDlgWnd("One Way Anova", 50, 50, 420, 220, Dlg, 0x0L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
			}
		}while (res < 0);
	if(res == 1 && Dlg->GetText(101, TmpTxt) &&(rD = new AccRange(TmpTxt))
		&& rD->BoundRec(&rec) && (res_tab = (double**)calloc(3, sizeof(double*)))
		&& (res_tab[0] = (double*) malloc(5*sizeof(double)))
		&& (res_tab[1] = (double*) malloc(5*sizeof(double)))
		&& (res_tab[2] = (double*) malloc(5*sizeof(double)))) {
		rep_init();
		if(Dlg->GetCheck(102)) {
			nr = rec.bottom - rec.top +1;		nc = rec.right - rec.left +1;
			if((cols = (double**)malloc(nc * sizeof(double*))) && (ncols = (int*)malloc(nc * sizeof(int)))) {
				for(i = rec.left; i <= rec.right; i++) {
					ncols[i-rec.left] = 0;	cols[i-rec.left] = (double*)malloc(nr * sizeof(double));
					if(cols[i-rec.left]) for(j = rec.top; j <= rec.bottom; j++) {
						if(data->GetValue(j, i, &tmp)) cols[i-rec.left][ncols[i-rec.left]++] = tmp;
						}
					}
				}
			}
		else {
			nr = rec.right - rec.left +1;		nc = rec.bottom - rec.top +1;
			if((cols = (double**)malloc(nc * sizeof(double*))) && (ncols = (int*)malloc(nc * sizeof(int)))) {
				for(i = rec.top; i <= rec.bottom; i++) {
					ncols[i-rec.top] = 0;	cols[i-rec.top] = (double*)malloc(nr * sizeof(double));
					if(cols[i-rec.top]) for(j = rec.left; j <= rec.right; j++) {
						if(data->GetValue(j, i, &tmp)) cols[i-rec.top][ncols[i-rec.top]++] = tmp;
						}
					}
				}
			}
		if(cols && nr >1 && nc >1 && (csums = (double*)malloc(nc *sizeof(double)))
			&& (css = (double*)malloc(nc *sizeof(double)))) {
			dBounds.Ymin = HUGE_VAL;		dBounds.Ymax = -HUGE_VAL;
			for(i = ntot = 0, mtot = 0.0; i < nc; i++) {
				for(j = 0, csums[i] = 0.0; j < ncols[i]; j++) {
					csums[i] += cols[i][j];
					if(dBounds.Ymin > cols[i][j]) dBounds.Ymin = cols[i][j];
					if(dBounds.Ymax < cols[i][j]) dBounds.Ymax = cols[i][j];
					}
				mtot += csums[i];			ntot += ncols[i];
				if(ncols[i]) csums[i] /= ((double)ncols[i]);
				}
			dBounds.Xmin = 0.0;				dBounds.Xmax = nc;
			if(ntot) mtot /= ((double)ntot);
			for(i = 0; i < nc; i++) {
				for(j = 0, css[i] = 0.0; j < ncols[i]; j++) {
					tmp = cols[i][j] - csums[i];	css[i] += (tmp*tmp);
					}
				}
			for(i = 0, ssa = ssw = sst = 0.0;  i < nc; i++) {
				tmp =(csums[i] - mtot);		ssa += (tmp*tmp) * ((double)ncols[i]);
				ssw += css[i];
				}
			sst = ssa + ssw;
			res_tab[0][0] = nc - 1;				res_tab[1][0] = ntot - nc;
			res_tab[2][0] = ntot -1;			res_tab[0][1] = ssa;
			res_tab[1][1] = ssw;				res_tab[2][1] = sst;
			res_tab[0][2] = ssa/res_tab[0][0];	res_tab[1][2] = ssw/res_tab[1][0];
			res_tab[0][3] = res_tab[0][2]/res_tab[1][2];
			res_tab[0][4] = f_dist(res_tab[0][3], res_tab[0][0], res_tab[1][0]);
			page = new Page(parent, data);
			mk_header(page, "<b>One Way ANOVA</b>");
			if((graph = new Graph(parent, data, 0L)) && (txt_obj = mk_scatt(0L, csums, css, ncols, nc))){
				OpenGraph(graph, 0L, (unsigned char*)txt_obj, false);
				free(txt_obj);								graph->moveable = 0;
				graph->GRect.Xmin += (txtdef1.fSize*5.0);	graph->GRect.Xmax += (txtdef1.fSize*5.0);
				graph->GRect.Ymin += (txtdef1.fSize*10.0);	graph->GRect.Ymax += (txtdef1.fSize*10.0);
				page->Command(CMD_DROP_GRAPH, graph, 0L);
				}
			mk_table(page, graph->GRect.Xmin, graph->GetSize(SIZE_GRECT_BOTTOM)+txtdef2.fSize*3.0,
				1, res_tab);
			parent->Command(CMD_DROP_GRAPH, page, 0L);
			}
		}
	if(cols) {
		for(i = 0; i < nc; i++) if(cols[i]) free(cols[i]);
		free(cols);
		}
	if(res_tab) {
		for(i = 0; i < 3; i++) if(res_tab[i]) free(res_tab[i]);
		free(res_tab);
		}
	if(rD) delete rD;	if(ncols) free(ncols);	if(csums) free(csums);
	if(css) free(css);	CloseDlgWnd(hDlg);	delete Dlg;	free(AnovaDlg);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// linear regression analysis
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
static char *RegrDlg_Tmpl =
	"1,2,,DEFAULT,PUSHBUTTON,-1,148,10,45,12\n"
	"2,3,,,PUSHBUTTON,-2,148,25,45,12\n"
	"3,,4,ISPARENT | CHECKED,GROUP,0,0,0,0,0\n"
	"4,10,100,ISPARENT | CHECKED,SHEET,1,5,10,130,100\n"
	"10,,,CHECKED,CHECKPIN,0,5,0,12,8\n"
	"100,101,,,LTEXT,2,10,30,60,8\n"
	"101,102,,,RANGEINPUT,-15,20,40,100,10\n"
	"102,103,,,LTEXT,3,10,55,60,8\n"
	"103,104,,,RANGEINPUT,-16,20,65,100,10\n"
	"104,105,,,LTEXT,4,10,80,60,8\n"
	"105,106,,,EDVAL1,5,70,80,25,10\n"
	"106,107,,,LTEXT,-10,97,80,60,8\n"
	"107,,,LASTOBJ,CHECKBOX,6,10,95,100,8";

void
rep_regression(GraphObj *parent, DataObj *data)
{
	TabSHEET tab1 = {0, 60, 10, "Regression Input"};
	double ci = 95.0;
	DlgInfo *RegrDlg;
	void *dyndata[] = {(void*)&tab1, (void*)"range for independent variable (x)",
		(void*)"range for dependent variable (y)", (void*)"confidence interval:",
		(void*)&ci, (void*)" include origin"};
	DlgRoot *Dlg;
	void *hDlg;
	int i, n, n1, rx, cx, ry, cy, res, align = 0;
	bool bContinue = false, bParZ;
	AccRange *rX = 0L, *rY = 0L;
	double *x = 0L, *y = 0L, **res_tab = 0L;
	double sx, sy, dx, dy, sxy, sxx, syy, sdy, df, t, ts, ty;
	double dres[10], ly[4], regr_sum[5][3];
	char *txt_obj;
	Graph *graph;
	Page *page;

	if(!parent || !data) return;
	if(!(RegrDlg = CompileDialog(RegrDlg_Tmpl, dyndata))) return;
	UseRangeMark(data, 1, TmpTxt, TmpTxt+100);
	if(!(Dlg = new DlgRoot(RegrDlg)))return;
	Dlg->ItemCmd(101, CMD_SET_DATAOBJ, data);	Dlg->ItemCmd(103, CMD_SET_DATAOBJ, data);
	hDlg = CreateDlgWnd("Linear Regression", 50, 50, 420, 260, Dlg, 0x0L);
	do {
		LoopDlgWnd();
		res = Dlg->GetResult();
		switch (res) {
		case 0:
			if(bContinue) res = -1;
			else if(Dlg->GetCheck(10)) res = -1;
			break;
		case -1:
			bContinue = false;
			break;
		case 1:
			Dlg->GetValue(105, &ci);			bParZ = Dlg->GetCheck(107);
			if(rX) delete rX;	if(rY) delete rY;
			rX = rY = 0L;
			if(Dlg->GetText(101, TmpTxt)) rX = new AccRange(TmpTxt);
			n = rX ? rX->CountItems() : 0;
			if(!n) {
				ErrorBox("range not specified\nor not valid.");
				bContinue = true;
				res = -1;
				}
			if(n && Dlg->GetText(103, TmpTxt) && (rY = new AccRange(TmpTxt))){
				if(n != rY->CountItems()) {
					ErrorBox("both ranges must be given\nand must have the same size");
					bContinue = true;
					res = -1;
					}
				}
			}
		}while (res < 0);
	n1 = n;
	if(res == 1 && n >1 && rX && rY && (x = (double*)malloc(n*sizeof(double))) 
		&& (y = (double*)malloc(n*sizeof(double)))
		&& (res_tab = (double**)calloc(3, sizeof(double*)))
		&& (res_tab[0] = (double*) malloc(5*sizeof(double)))
		&& (res_tab[1] = (double*) malloc(5*sizeof(double)))
		&& (res_tab[2] = (double*) malloc(5*sizeof(double)))) {
		rX->GetFirst(&cx, &rx);				rY->GetFirst(&cy, &ry);
		rep_init();
		dBounds.Xmin = dBounds.Ymin = HUGE_VAL;		dBounds.Xmax = dBounds.Ymax = -HUGE_VAL;
		for(i = n = 0; i < n1; i++) {
			if(rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) 
				&& data->GetValue(rx, cx, &x[n]) && data->GetValue(ry, cy, &y[n])){
				if(dBounds.Xmin > x[n]) dBounds.Xmin = x[n];
				if(dBounds.Xmax < x[n]) dBounds.Xmax = x[n];
				if(dBounds.Ymin > y[n]) dBounds.Ymin = y[n];
				if(dBounds.Ymax < y[n]) dBounds.Ymax = y[n];
				n++;
				}
			}
		if(!bParZ) {
			for(i = 0, 	sx = sy = 0.0; i < n; i++) {
				sx += x[i];			sy += y[i];
				}
			dres[2] = sx /n;		dres[3] = sy/n;
			}
		else {
			dres[2] = sx = dres[3] = sy = 0.0;
			}
		sxy = sxx = syy = 0.0;
		for(i = 0; i < n; i++) {
			dx = x[i]-dres[2];	dy = y[i]-dres[3];
			sxx += (dx*dx);		syy += (dy*dy);		sxy += (dx*dy);
			}
		dres[0] = sxy / sxx;	dres[1] = dres[3] - dres[0] * dres[2];
		for(i = 0, sdy = 0.0; i < n; i++) {
			dy = y[i] - (dres[1] + x[i] *dres[0]);
			sdy += (dy * dy);
			}
		df = bParZ ? (n-1) : (n-2);		sdy = sdy/df;					dres[4] = sqrt(sdy/sxx);
		dres[5] = sxx/(n-1);			dres[6] = syy/(n-1);			dres[7] = sdy;
		dres[8] = sxy/sdy*sxy/sxx;		dres[9] = f_dist(dres[8], 1.0, df);
		t = distinv(t_dist, df, 1.0, (100.0-ci)/100.0, 2.0); 
		if (n && (graph = new Graph(parent, data, 0L))) {
			i = sprintf(TmpTxt, "[1=Function]\nx1= %g\nx2= %g\nxstep= %g\nLine= %g 1 0x000000ff 0x0\n", 
				dBounds.Xmin, dBounds.Xmax, (dBounds.Xmax -dBounds.Xmin)/100.0, defs.GetSize(SIZE_DATA_LINE));
			i += sprintf(TmpTxt+i, "f_xy=\"%g+x*%g\\n\"", dres[1], dres[0]);
			OpenGraph(graph, 0L, (unsigned char*)TmpTxt, false);
			i = sprintf(TmpTxt, "[1=Function]\nx1= %g\nx2= %g\nxstep= %g\nLine= %g 1 0x000000ff 0x0\n", 
				dBounds.Xmin, dBounds.Xmax, (dBounds.Xmax -dBounds.Xmin)/100.0, defs.GetSize(SIZE_DATA_LINE)*0.5);
			i += sprintf(TmpTxt+i, "f_xy=\"y=%g+x*%g;\\nts=sqrt((((x-%g)^2)/%g+%g)*%g);\\ny=y+ts*%g\\n\"", 
				dres[1], dres[0], dres[2], sxx, (1.0/(double)n), dres[7], t);
			OpenGraph(graph, 0L, (unsigned char*)TmpTxt, false);
			i = sprintf(TmpTxt, "[1=Function]\nx1= %g\nx2= %g\nxstep= %g\nLine= %g 1 0x000000ff 0x0\n", 
				dBounds.Xmin, dBounds.Xmax, (dBounds.Xmax -dBounds.Xmin)/100.0, defs.GetSize(SIZE_DATA_LINE)*0.5);
			i += sprintf(TmpTxt+i, "f_xy=\"y=%g+x*%g;\\nts=sqrt((((x-%g)^2)/%g+%g)*%g);\\ny=y-ts*%g\\n\"", 
				dres[1], dres[0], dres[2], sxx, (1.0/(double)n), dres[7], t);
			OpenGraph(graph, 0L, (unsigned char*)TmpTxt, false);
			ts = t * sqrt(dres[7]*((dBounds.Xmax-dres[2])*(dBounds.Xmax-dres[2])/sxx +1.0/(double)n));
			ty = dBounds.Xmax * dres[0] +dres[1];
			ly[0] = ty +ts;		ly[1] = ty -ts;
			ts = t * sqrt(dres[7]*((dBounds.Xmin-dres[2])*(dBounds.Xmin-dres[2])/sxx +1.0/(double)n));
			ty = dBounds.Xmin * dres[0] +dres[1];
			ly[2] = ty +ts;		ly[3] = ty -ts;
			for(i = 0; i < 4; i++) if(ly[i] > -HUGE_VAL && ly[i] < HUGE_VAL) {
				if(ly[i] < dBounds.Ymin) dBounds.Ymin = ly[i];
				if(ly[i] > dBounds.Ymax) dBounds.Ymax = ly[i];
				}
			if(txt_obj = mk_scatt(x, y, 0L, 0L, n)){
				OpenGraph(graph, 0L, (unsigned char*)txt_obj, false);
				free(txt_obj);
				}
			if(!bParZ) sprintf(TmpTxt, "y = %g %c %g * x", dres[1],
				(dres[0] < 0.0 ? '-' : '+'), fabs(dres[0]));
			else sprintf(TmpTxt, "y = %g * x", fabs(dres[0]));
			if(txt_obj = mk_label((graph->GetSize(SIZE_DRECT_LEFT) + graph->GetSize(SIZE_DRECT_RIGHT))/2.0,
				graph->GetSize(SIZE_DRECT_TOP)+txtdef1.fSize/2.0, true, TXA_HCENTER, &txtdef1, TmpTxt)) {
				OpenGraph(graph, 0L, (unsigned char*)txt_obj, false);
				free(txt_obj);
				}
			page = new Page(parent, data);					page->Command(CMD_DROP_GRAPH, graph, 0L);
			graph->moveable = 0;
			graph->GRect.Xmin += (txtdef1.fSize*5.0);		graph->GRect.Xmax += (txtdef1.fSize*5.0);
			graph->GRect.Ymin += (txtdef1.fSize*10.0);		graph->GRect.Ymax += (txtdef1.fSize*10.0);
			mk_header(page, "<b>Linear Regression Analysis</b>");
			res_tab[0][0] = 1;						res_tab[1][0] = df;
			res_tab[2][0] = df+1.0;					res_tab[0][1] = sxy*sxy/sxx;
			res_tab[1][1] = syy-res_tab[0][1];		res_tab[2][1] = syy;
			res_tab[0][2] = res_tab[0][1];			res_tab[1][2] = res_tab[1][1]/df;
			res_tab[0][3] = dres[8];				res_tab[0][4] = dres[9];
			regr_sum[1][0] = res_tab[0][1]/syy;		regr_sum[0][0] = sqrt(regr_sum[1][0]);
			mk_table(page, graph->GRect.Xmin, graph->GetSize(SIZE_GRECT_BOTTOM)+txtdef2.fSize*3.0,
				2, res_tab);
			parent->Command(CMD_DROP_GRAPH, page, 0L);
			}
		}
	CloseDlgWnd(hDlg);
	delete Dlg;				if(res_tab) {
		for(i = 0; i < 3; i++) if(res_tab[i]) free(res_tab[i]);
		free(res_tab);
		}
	if(x) free(x);			if(y) free(y);
	if(rX) delete rX;		if(rY) delete rY;		free(RegrDlg);
}
