//basic.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2013
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

enum mode {
 NORMAL = 0,
 SYSTEM = 1
};

static int _connect_server(struct roar_connection * con, const char * server, int type, int flags, uint_least32_t timeout);

int __get_daemonimage(const char ** daemonimage, enum mode * mode, const char * server) {
 *daemonimage = NULL;
 *mode = NORMAL;

 if ( !!strncmp(server, "+fork", 5) ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 server += 5;

 if ( server[0] == '=' ) {
  server++;

  if ( server[0] == 0 ) {
   // no special case, we just ignore it.
  } else if ( server[0] == 'd' && server[1] == ':' ) {
   server += 2;
   *daemonimage = server;
   *mode = NORMAL;
  } else if ( server[0] == '!' ) {
   server += 1;
   *daemonimage = server;
   *mode = SYSTEM;
  } else {
   roar_err_set(ROAR_ERROR_ILLSEQ);
   return -1;
  }
 }

 if ( *daemonimage == NULL )
  *daemonimage = roar_libroar_get_config()->daemonimage;

 // we keep this step to be compatible with older versions.
 if ( *daemonimage == NULL || **daemonimage == 0 ) {
  *daemonimage = roar_env_get("ROAR_DAEMONIMAGE");
  if ( *daemonimage != NULL ) {
   ROAR_WARN("__get_daemonimage(*): Usage of $ROAR_DAEMONIMAGE is obsolete. Use ROAR_OPTIONS=daemonimage:...");
  }
 }

 if ( *daemonimage == NULL || **daemonimage == 0 )
  *daemonimage = roar_libroar_get_path_static("bin-default-daemonimage");

 return 0;
}

#if defined(ROAR_TARGET_WIN32)
#define NUM_TRIES 16
static int _start_server_win32(struct roar_connection * con, const char * server, int type, int flags, uint_least32_t timeout) {
 enum mode mode = NORMAL;
 const char * daemonimage = NULL;
 char buf[64];
 int port;
 size_t i;
 int fh;
 int ret;

 if ( __get_daemonimage(&daemonimage, &mode, server) == -1 )
  return -1;

 if ( mode != NORMAL ) {
  roar_err_set(ROAR_ERROR_NOSYS);
  return -1;
 }

 for (i = 0; i < NUM_TRIES; i++) {
  port = roar_random_uint16();
  if ( port < 1025 ) /* it seems on some strange systems the magic border is at 1025 not 1024. */
   continue;

  fh = roar_socket_connect(ROAR_SOCKET_TYPE_TCP, "localhost", port);
  if ( fh == -1 )
   break;

  closesocket(fh);
 }

 if ( i == NUM_TRIES ) {
  roar_err_set(ROAR_ERROR_LOOP);
  return -1;
 }

 // port is the port to use. It seems to be 'free'.

#if defined(_P_DETACH)
#define MODE _P_DETACH
#elif defined(_P_NOWAIT)
#define MODE _P_NOWAIT
#else
#define MODE _P_NOWAITO
#endif

 snprintf(buf, sizeof(buf), "%i", port);
 roar_err_clear_all();
 if ( _spawnlp(MODE, daemonimage, daemonimage, "--tcp", "--bind", "localhost", "--port", buf, (const char*)NULL) < 0 ) {
  roar_err_update();
  ROAR_ERR("_start_server_win32(server='%s', ...): Can not start server: %s: %s", server, daemonimage, roar_errorstring);
  return -1;
 }

 snprintf(buf, sizeof(buf), "localhost:%i", port);

 // give the daemon some time to come up:
 roar_usleep(25000);

 ret = -1;
 for (i = 0; ret == -1 && i < NUM_TRIES; i++) {
  ret = _connect_server(con, buf, type, flags, timeout);
  if ( ret == -1 && i < NUM_TRIES )
   roar_sleep(1);
 }

 if ( ret == -1 )
  return -1;

 con->flags |= ROAR_CON_FLAGS_TERMINATE;

 return 0;
}
#elif !defined(ROAR_TARGET_MICROCONTROLLER)
static int _start_server_posix(struct roar_connection * con, const char * server, int type, int flags, uint_least32_t timeout) {
 enum mode mode = NORMAL;
 const char * daemonimage = NULL;
 int socks[2];
 int r;
 char fhstr[12];
 size_t len;

 if ( __get_daemonimage(&daemonimage, &mode, server) == -1 )
  return -1;

 len = roar_mm_strlen(daemonimage) + 9; // 9 = '+fork=' + 'X:' + '\0'
 con->server_name = roar_mm_malloc(len);
 if ( con->server_name != NULL ) {
  snprintf(con->server_name, len, "+fork=%s%s", mode == NORMAL ? "d:" : "!", daemonimage);
 }

 if ( socketpair(AF_UNIX, SOCK_STREAM, 0, socks) == -1 ) {
  roar_err_from_errno();
  return -1;
 }

 r = roar_fork(NULL);

 if ( r == -1 ) { // error!
  ROAR_ERR("_start_server(*): Can not fork: %s", roar_error2str(roar_error));
  close(socks[0]);
  close(socks[1]);
  return -1;
 } else if ( r == 0 ) { // we are the child
  roar_watchdog_stop();

  close(socks[0]);

  close(ROAR_STDIN ); // we do not want roard to have any standard input
  close(ROAR_STDOUT); // STDOUT is also not needed, so we close it,
                      // but STDERR we keep open for error messages.

  snprintf(fhstr, sizeof(fhstr), "%i", socks[1]);

  switch (mode) {
   case NORMAL:
     execlp(daemonimage, daemonimage, "--no-listen", "--client-fh", fhstr, (_LIBROAR_GOOD_CAST char*)NULL);
    break;
   case SYSTEM:
     dup2(socks[1], ROAR_STDIN );
     dup2(socks[1], ROAR_STDOUT);
     execl(roar_libroar_get_path_static("bin-sh"), roar_libroar_get_path_static("bin-sh"), "-c", daemonimage, (_LIBROAR_GOOD_CAST char*)NULL);
     execlp("sh", "sh", "-c", daemonimage, (_LIBROAR_GOOD_CAST char*)NULL);
    break;
  }

  // we are still alive?
  ROAR_ERR("_start_server(*): alive after exec(), that's bad!");
  ROAR_U_EXIT(1);
 } else { // we are the parent
  close(socks[1]);
  if ( roar_vio_open_fh_socket(con->viocon, socks[0]) == -1 ) {
   close(socks[0]);
   return -1;
  } else {
   con->flags |= ROAR_CON_FLAGS_VIO;
  }
  return 0;
 }

 return -1;
}
#endif

static int _start_server(struct roar_connection * con, const char * server, int type, int flags, uint_least32_t timeout) {
#if defined(ROAR_TARGET_WIN32)
 return _start_server_win32(con, server, type, flags, timeout);
#elif !defined(ROAR_TARGET_MICROCONTROLLER)
 return _start_server_posix(con, server, type, flags, timeout);
#else
 ROAR_ERR("_start_server(*): There is no UNIX Domain Socket, download a real OS.");
 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
#endif
}

static int _connect_internal(struct roar_connection * con, const char * server, int type, int flags, uint_least32_t timeout) {
 struct roar_libroar_config * config = roar_libroar_get_config();
 struct roar_vio_calls * vio;

 if ( config->connect_internal != NULL ) {
  vio = config->connect_internal(con, server, type, flags, timeout);
  if ( vio == NULL )
   return -1;

  con->viocon = vio;
  con->flags |= ROAR_CON_FLAGS_VIO;

  return 0;
 }

 roar_err_set(ROAR_ERROR_BADHOST);
 return -1;
}

static int _connect_server(struct roar_connection * con, const char * server, int type, int flags, uint_least32_t timeout) {
#if defined(ROAR_HAVE_STAT) && defined(ROAR_HAVE_H_SYS_STAT)
 struct stat sockstat;
#endif
 const char * obj = NULL;
 char user_sock[128];
 int is_decnet = 0;
 int port = 0;
 int i = 0;
 int fh = -1;
 int err;
 int ret;

 if ( con == NULL || server == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( !strcmp(server, "+invalid") ) {
  roar_err_set(ROAR_ERROR_CANCELED);
  return -1;
 } else if ( !strncmp(server, "+dstr=", 6) ) {
  if ( roar_vio_open_dstr_simple(con->viocon, server+6, ROAR_VIOF_READWRITE) == -1 )
   return -1;
  con->flags |= ROAR_CON_FLAGS_VIO;
  con->server_name = roar_mm_strdup(server);
  return 0;
 } else if ( !strcmp(server, "+fork") || !strncmp(server, "+fork=", 6) ) {
  return _start_server(con, server, type, flags, timeout);
 } else if ( !strcmp(server, "+internal") || !strncmp(server, "+internal=", 10) ) {
  ret = _connect_internal(con, server, type, flags, timeout);
  if ( ret == 0 )
   con->server_name = roar_mm_strdup(server);
  return ret;
 }

 strncpy(user_sock, server, sizeof(user_sock)-1);
 user_sock[sizeof(user_sock)-1] = 0;


 if ( *user_sock != '/' ) { // don't test AF_UNIX sockets for ports
  for (i = 0; user_sock[i] != 0; i++) {
   if ( user_sock[i] == ':' ) {
    if ( user_sock[i+1] == ':' ) { // DECnet, leave unchanged
     is_decnet = 1;
     obj = &user_sock[i+2];
     break;
    }

    port = atoi(&(user_sock[i+1]));
    user_sock[i] = 0;
    break;
   }
  }
 }

 if ( is_decnet ) {
  if ( *user_sock == ':' ) {
   if ( roar_socket_get_local_nodename() != NULL ) {
    strncpy(user_sock, roar_socket_get_local_nodename(), sizeof(user_sock)-1);
    user_sock[sizeof(user_sock)-1] = 0;
    roar_mm_strlcat(user_sock, server, sizeof(user_sock)-1);
    user_sock[sizeof(user_sock)-1] = 0;
    obj  = strstr(user_sock, "::");
    obj += 2;
   }
  }

  if ( *obj == 0 ) {
   roar_mm_strlcat(user_sock, ROAR_DEFAULT_OBJECT, sizeof(user_sock)-1);
   user_sock[sizeof(user_sock)-1] = 0;
  }

  ROAR_DBG("roar_connect_raw(*): user_sock='%s'", user_sock);
 }

 if ( port || is_decnet ) {
  fh = roar_socket_connect(type, user_sock, port);
  // restore the original string
  user_sock[i] = ':';
 } else {
#if defined(ROAR_HAVE_STAT) && defined(ROAR_HAVE_H_SYS_STAT)
  if ( user_sock[0] == '/' ) {
   if ( stat(user_sock, &sockstat) == 0 ) {
    if ( S_ISCHR(sockstat.st_mode) ) {
#ifdef O_NOCTTY
     fh = open(user_sock, O_RDWR|O_NOCTTY, 0666);
#else
     fh = open(user_sock, O_RDWR, 0666);
#endif
    }
   }
  }
#endif
  if ( fh == -1 )
   fh = roar_socket_connect(type, user_sock, ROAR_DEFAULT_PORT);
 }

 if ( fh == -1 )
  return -1;

 if ( roar_vio_open_fh_socket(con->viocon, fh) == -1 ) {
  err = roar_error;
#ifdef ROAR_TARGET_WIN32
  closesocket(fh);
#else
  close(fh);
#endif
  roar_error = err;
  return -1;
 } else {
  con->flags |= ROAR_CON_FLAGS_VIO;
 }

 con->server_name = roar_mm_strdup(server);

 roar_err_set(ROAR_ERROR_NONE);
 return 0;
}

static int roar_connect_raw (struct roar_connection * con, const char * server, int flags, uint_least32_t timeout) {
#ifdef ROAR_HAVE_LIBSLP
 struct roar_libroar_config * config = roar_libroar_get_config();
#endif
#if defined(ROAR_HAVE_UNIX)
 char user_sock[128];
 const char * roar_server;
#endif
#if defined(ROAR_HAVE_LIBSLP) || !defined(ROAR_TARGET_MICROCONTROLLER)
 int i = 0;
#endif
#if !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER)
 struct passwd * pwd;
#endif
#ifdef ROAR_HAVE_LIBDNET
 struct stat decnet_stat;
#endif
#ifdef ROAR_HAVE_LIBX11
 struct roar_x11_connection * x11con;
#endif
#ifdef ROAR_HAVE_LIBSLP
 struct roar_server * list;
 int workarounds_store;
#endif

 roar_err_set(ROAR_ERROR_UNKNOWN);

 if ( timeout != 0 ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 if ( flags & ROAR_ENUM_FLAG_HARDNONBLOCK )
  flags |= ROAR_ENUM_FLAG_NONBLOCK;


 if ( server == NULL || *server == 0 )
  server = roar_libroar_get_server();

 if ( server == NULL || *server == 0 )
  server = roar_env_get("ROAR_SERVER");

#ifdef ROAR_HAVE_LIBX11
 if ( server == NULL || *server == 0 ) {
  if ( (x11con = roar_x11_connect(NULL)) != NULL ) {
   server = roar_x11_get_prop(x11con, "ROAR_SERVER");
   roar_x11_disconnect(x11con);
  }
 }
#endif

#if !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER)
 if ( (server == NULL || *server == 0) && (i = readlink(roar_libroar_get_path_static("sysconf-roarserver"), user_sock, sizeof(user_sock)-1)) != -1 ) {
   user_sock[i] = 0;
   server = user_sock;
 }
#endif

 if ( server != NULL && !strcasecmp(server, "+slp") ) {
  server = roar_slp_find_roard(0);
  if ( server == NULL ) {
   return -1;
  }
 }

 // "+default" is an alias for NULL.
 if ( server != NULL && !strcasecmp(server, "+default") ) {
  server = NULL;
 }

 if ( server == NULL || *server == 0 ) {
  /* connect via defaults */

#ifdef ROAR_HAVE_UNIX
#ifndef ROAR_TARGET_MICROCONTROLLER
  roar_server = roar_env_get("HOME");
#else
  roar_server = NULL;
#endif

  if ( roar_server == NULL ) {
#if !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER)
   if ( (pwd = getpwuid(getuid())) == NULL ) {
    roar_server = roar_libroar_get_path_static("dir-nx-home");
   } else {
    roar_server = pwd->pw_dir;
   }
#else
   roar_server = "/WIN32-SUCKS";
#endif
  }

  snprintf(user_sock, sizeof(user_sock)-1, "%s/%s", roar_server, ROAR_DEFAULT_SOCK_USER);
  user_sock[sizeof(user_sock)-1] = 0;

  if ( _connect_server(con, user_sock, ROAR_SOCKET_TYPE_UNIX, flags, timeout) == 0 )
   return 0;

  if ( _connect_server(con, ROAR_DEFAULT_SOCK_GLOBAL, ROAR_SOCKET_TYPE_UNIX, flags, timeout) == 0 )
   return 0;
#endif

  if ( _connect_server(con, ROAR_DEFAULT_HOSTPORT, ROAR_SOCKET_TYPE_TCP, flags, timeout) == 0 )
   return 0;

#ifdef ROAR_HAVE_LIBDNET
  if ( stat(ROAR_PROC_NET_DECNET, &decnet_stat) == 0 ) {
   if ( roar_socket_get_local_nodename() ) {
    snprintf(user_sock, 79, "%s::%s", roar_socket_get_local_nodename(), ROAR_DEFAULT_OBJECT);
    if ( _connect_server(con, user_sock, ROAR_SOCKET_TYPE_DECNET, flags, timeout) == 0 )
     return 0;
   }
  }
#endif

  if ( _connect_server(con, "+abstract", ROAR_SOCKET_TYPE_UNKNOWN, flags, timeout) == 0 )
   return 0;

#ifdef ROAR_HAVE_LIBSLP
 if ( !(config->workaround.workarounds & ROAR_LIBROAR_CONFIG_WAS_NO_SLP) &&
      !(flags & ROAR_ENUM_FLAG_NONBLOCK) &&
      !(flags & ROAR_ENUM_FLAG_LOCALONLY)
    ) {
  if ( (server = roar_slp_find_roard(0)) != NULL ) {
   if ( _connect_server(con, server, ROAR_SOCKET_TYPE_UNKNOWN, 0, 0) == 0 )
    return 0;

   /* in case we can not connect to the server given this may be a cache problem,
      we do a new lookup with the cache disabled in this case                     */
   ROAR_WARN("roar_connect_raw(*): Can not connect to SLP located server, disabling cache");
   if ( (server = roar_slp_find_roard(1)) != NULL )
    if ( _connect_server(con, server, ROAR_SOCKET_TYPE_UNKNOWN, 0, 0) == 0 )
     return 0;
  }
 }

 workarounds_store = config->workaround.workarounds;
 config->workaround.workarounds |= ROAR_LIBROAR_CONFIG_WAS_NO_SLP;
 list = roar_enum_servers(flags, -1, -1);
 config->workaround.workarounds = workarounds_store;
 if ( list != NULL ) {
  for (i = 0; list[i].server != NULL; i++) {
   if ( _connect_server(con, list[i].server, ROAR_SOCKET_TYPE_UNKNOWN, 0, 0) == 0 ) {
    roar_enum_servers_free(list);
    return 0;
   }
  }
  roar_enum_servers_free(list);
 }
#endif

 return -1;

 } else {
  /* connect via (char*)server */
  if ( _connect_server(con, server, ROAR_SOCKET_TYPE_UNKNOWN, flags, timeout) != 0 )
   return -1;
  return 0;
 }

 roar_err_set(ROAR_ERROR_NODEV);
 ROAR_DBG("roar_connect_raw(*) = -1 // error=NODEV");
 return -1;
}

int roar_connect     (struct roar_connection * con, const char * server, int flags, uint_least32_t timeout) {
 struct roar_libroar_config * config = roar_libroar_get_config();

 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( roar_connect_none(con) == -1 )
  return -1;

 if ( roar_connect_raw(con, server, flags | config->serverflags, timeout) == -1 )
  return -1;

 return 0;
}

int roar_connect_none (struct roar_connection * con) {
 struct roar_libroar_config * config = roar_libroar_get_config();

 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 memset(con, 0, sizeof(struct roar_connection));
 con->refc        = 1;
 con->flags       = ROAR_CON_FLAGS_ISCLIENT;
 con->version     = _ROAR_MESSAGE_VERSION;
 con->cb_userdata = NULL;
 con->cb          = NULL;
 con->server_stds = NULL;
 con->server_name = NULL;

 roar_err_init(&(con->errorframe));

 con->viocon = &(con->viocon_store);

// con->flags |= ROAR_CON_FLAGS_VIO;

 if ( config->protocolversion != -1 )
  roar_set_connection_version(con, config->protocolversion);

 roar_err_set(ROAR_ERROR_NONE);
 return 0;
}

int roar_connect_vio (struct roar_connection * con, struct roar_vio_calls * vio) {
 if ( con == NULL || vio == NULL ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 if ( roar_connect_none(con) == -1 )
  return -1;

 con->viocon = vio;
 con->flags |= ROAR_CON_FLAGS_VIO;

 return -1;
}

int roar_connect_fh (struct roar_connection * con, int fh) {

 if ( con == NULL || fh == -1 ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 if ( roar_connect_none(con) == -1 )
  return -1;

 // special hack to set an illegal value used internally in libroar:
 if ( fh == -2 )
  fh = -1;

 if ( roar_vio_open_fh_socket(con->viocon, fh) != -1 ) {
  con->flags |= ROAR_CON_FLAGS_VIO;
 } else {
  return -1;
 }

 roar_err_set(ROAR_ERROR_NONE);
 return 0;
}

int roar_get_connection_fh (struct roar_connection * con) {
 int fh;

 ROAR_DBG("roar_get_connection_fh(con=%p) = ?", con);

 roar_debug_warn_sysio("roar_get_connection_fh", "roar_get_connection_vio2", NULL);

 if ( con == NULL )
  return -1;

 ROAR_DBG("roar_get_connection_fh(con=%p) = ?", con);

 if ( roar_vio_ctl(con->viocon, ROAR_VIO_CTL_GET_FH, &fh) == -1 ) {
#ifdef ROAR_TARGET_WIN32
  if ( roar_vio_ctl(con->viocon, ROAR_VIO_CTL_GET_SELECT_FH, &fh) == -1 )
   return -1;
#else
  return -1;
#endif
 }

 ROAR_DBG("roar_get_connection_fh(con=%p) = %i", con, fh);

 return fh;
}

struct roar_vio_calls * roar_get_connection_vio2 (struct roar_connection * con) {
 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 if ( con->flags & ROAR_CON_FLAGS_VIO )
  return con->viocon;

// TODO: try to open the VIO.

 roar_err_set(ROAR_ERROR_BADFH);
 return NULL;
}

const char * roar_get_connection_server(struct roar_connection * con) {
 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return NULL;
 }

 return con->server_name;
}

int roar_connectionref(struct roar_connection * con) {
 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 con->refc++;

 return 0;
}

int roar_connectionunref(struct roar_connection * con) {
 struct roar_vio_calls * vio;
 struct roar_message m;

 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 con->refc--;

 if ( con->refc )
  return 0;

 if ( con->flags & ROAR_CON_FLAGS_TERMINATE ) {
  if ( roar_terminate(con, 1) == -1 ) {
   ROAR_WARN("roar_connectionunref(con=%p{.server_name='%s'}): Can not terminate server as requested.", con, con->server_name);
   if ( roar_terminate(con, 0) == -1 ) {
    ROAR_ERR("roar_connectionunref(con=%p{.server_name='%s'}): Can not exit server. BAD.", con, con->server_name);
   }
  }
 }

 memset(&m, 0, sizeof(m));

 m.datalen =  0;
 m.stream  = -1;
 m.pos     =  0;
 m.cmd     = ROAR_CMD_QUIT;

 roar_req(con, &m, NULL);

 if ( (vio = roar_get_connection_vio2(con)) != NULL ) {
  roar_vio_close(vio);
 }

 if ( con->server_stds != NULL ) {
  roar_stds_free(con->server_stds);
  con->server_stds = NULL;
 }

 if ( con->server_name != NULL ) {
  roar_mm_free(con->server_name);
  con->server_name = NULL;
 }

 if ( con->flags & ROAR_CON_FLAGS_FREESELF ) {
  roar_mm_free(con);
 } else {
  roar_connect_fh(con, -2);
 }

 return 0;
}

int roar_set_connection_callback(struct roar_connection * con,
                                 void (*cb)(struct roar_connection * con,
                                            struct roar_message    * mes,
                                            void                   * data,
                                            void                   * userdata),
                                 void * userdata) {
 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 con->cb       = cb;
 con->cb_userdata = userdata;

 return 0;
}

int roar_set_connection_version(struct roar_connection * con, int version) {
 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( version == -1 )
  version = _ROAR_MESSAGE_VERSION;

 if ( version < 0 ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 con->version = version;

 if ( version == 2 ) {
  con->flags |= ROAR_CON_FLAGS_SUPPORT_V2;
 }

 return 0;
}

int16_t roar_message_genseq(struct roar_connection * con, int is_client) {
 static int16_t store = 0;
 int16_t ret = 0;

 ret = store++;

 if ( ret == 0 )
  ret = store++;

 if ( is_client == -1 ) {
  if ( con != NULL ) {
   is_client = con->flags & ROAR_CON_FLAGS_ISCLIENT ? 1 : 0;
  }
 }

 if ( is_client == -1 )
  is_client = 1;

 ret |= 0x8000U;
 if ( !is_client )
  ret -= 0x8000U;

 return ret;
}

int roar_sync         (struct roar_connection * con) {
 // wait for any non-client reqs
 return roar_wait_msg(con, 0x0000, 0x8000);
}

int roar_wait_msg     (struct roar_connection * con, int16_t seq, int16_t seqmask) {
 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
}

int roar_noop         (struct roar_connection * con) {
 struct roar_message mes;

 if ( con == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 memset(&mes, 0, sizeof(mes));

 mes.cmd = ROAR_CMD_NOOP;
 mes.stream = -1;

 return roar_req3(con, &mes, NULL);
}

int roar_identify   (struct roar_connection * con, const char * name) {
 struct roar_message mes;
 uint32_t pid;
 int max_len;

 roar_err_set(ROAR_ERROR_UNKNOWN);

 ROAR_DBG("roar_identify(*): try to identify myself...");

 memset(&mes, 0, sizeof(mes));

 mes.cmd    = ROAR_CMD_IDENTIFY;
 mes.stream = -1;
 mes.pos    =  0;

 ROAR_DBG("roar_identify(*): name=%p", name);

 if ( name == NULL )
  name = "libroar client";

 ROAR_DBG("roar_identify(*): name=%p", name);

 max_len = roar_mm_strlen(name);
 ROAR_DBG("roar_identify(*): strlen(name) = %i", max_len);

 if ( max_len > (LIBROAR_BUFFER_MSGDATA - 5) )
  max_len = LIBROAR_BUFFER_MSGDATA - 5;

 mes.datalen = 5 + max_len;
 mes.data[0] = 1;

 pid = getpid();
 mes.data[1] = (pid & 0xFF000000UL) >> 24;
 mes.data[2] = (pid & 0x00FF0000UL) >> 16;
 mes.data[3] = (pid & 0x0000FF00UL) >>  8;
 mes.data[4] = (pid & 0x000000FFUL) >>  0;
 ROAR_DBG("roar_identify(*): pid = %i", (int)pid);

 strncpy(mes.data+5, name, max_len);

 return roar_req3(con, &mes, NULL);
}

//ll
