//vio_stream.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2013
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

static ssize_t _vio_stream_read    (struct roar_vio_calls * vio, void *buf, size_t count) {
 return roar_vio_read(roar_get_connection_vio2(vio->inst), buf, count);
}

static ssize_t _vio_stream_write   (struct roar_vio_calls * vio, void *buf, size_t count) {
 return roar_vio_write(roar_get_connection_vio2(vio->inst), buf, count);
}

static roar_off_t   _vio_stream_lseek   (struct roar_vio_calls * vio, roar_off_t offset, int whence) {
 return roar_vio_lseek(roar_get_connection_vio2(vio->inst), offset, whence);
}
static int     _vio_stream_sync    (struct roar_vio_calls * vio) {
 return roar_vio_sync(roar_get_connection_vio2(vio->inst));
}
static int     _vio_stream_ctl     (struct roar_vio_calls * vio, roar_vio_ctl_t cmd, void * data) {
 if (vio == NULL) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if (cmd == -1) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 switch (cmd) {
  case ROAR_VIO_CTL_GET_NAME:
    if ( data == NULL ) {
     roar_err_set(ROAR_ERROR_FAULT);
     return -1;
    }

    *(char**)data = "stream";
    return 0;
   break;
  case ROAR_VIO_CTL_GET_NEXT:
    *(struct roar_vio_calls **)data = roar_get_connection_vio2(vio->inst);
    return 0;
   break;
  case ROAR_VIO_CTL_SET_NEXT:
    roar_err_set(ROAR_ERROR_NOTSUP);
    return -1;
   break;
  case ROAR_VIO_CTL_NONBLOCK:
    return roar_vio_ctl(roar_get_connection_vio2(vio->inst), ROAR_VIO_CTL_NONBLOCK, data);
   break;
 }

 return roar_vio_ctl(roar_get_connection_vio2(vio->inst), cmd, data);
}

static int     _vio_stream_close   (struct roar_vio_calls * vio) {
 roar_vio_close(roar_get_connection_vio2(vio->inst));
 roar_mm_free(vio->inst);

 return 0;
}

int     roar_vio_simple_stream (struct roar_vio_calls * calls,
                                uint32_t rate, uint32_t channels, uint32_t bits, uint32_t codec,
                                const char * server, int dir, const char * name, int mixer) {
 struct roar_connection * con = NULL;
 struct roar_stream       stream;
 int err;

 if ( calls == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( roar_stream_new(&stream, rate, channels, bits, codec) == -1 )
  return -1;

 con = roar_mm_malloc(sizeof(struct roar_connection));
 if ( con == NULL )
  return -1;

 memset(con, 0, sizeof(struct roar_connection));

 if ( roar_simple_connect(con, server, name) == -1 ) {
  err = roar_error;
  roar_mm_free(con);
  roar_error = err;
  return -1;
 }

 if ( roar_stream_connect(con, &stream, dir, mixer) == -1 ) {
  err = roar_error;
  roar_disconnect(con);
  roar_mm_free(con);
  roar_error = err;
  return -1;
 }

 if ( roar_stream_exec(con, &stream) == -1 ) {
  err = roar_error;
  roar_disconnect(con);
  roar_mm_free(con);
  roar_error = err;
  return -1;
 }

 roar_vio_clear_calls(calls);

 calls->inst       = con;
 calls->read       = _vio_stream_read;
 calls->write      = _vio_stream_write;
 calls->lseek      = _vio_stream_lseek;
 calls->sync       = _vio_stream_sync;
 calls->ctl        = _vio_stream_ctl;
 calls->close      = _vio_stream_close;

 if ( dir == ROAR_DIR_PLAY ) {
  roar_vio_shutdown(calls, SHUT_RD);
 } else if ( dir == ROAR_DIR_MONITOR || dir == ROAR_DIR_RECORD ) {
  roar_vio_shutdown(calls, SHUT_WR);
 }

 return 0;
}

static int _roar_simple_new_stream_obj (struct roar_connection * con, struct roar_stream * s, uint32_t rate, uint32_t channels, uint32_t bits, uint32_t codec, int dir, int mixer) {
 struct roar_libroar_config * config = roar_libroar_get_config();
 char file[80] = {0};
 int fh = -1, listen = -1;
 static int count = 0;
 int    port = 0;
#if defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_LIBDNET)
 int    opt  = 1;
#endif
#ifdef ROAR_HAVE_IPV4
 struct sockaddr_in   socket_addr;
 socklen_t            len            = sizeof(struct sockaddr_in);
#else
 struct sockaddr      socket_addr;
 socklen_t            len            = sizeof(struct sockaddr);
#endif
#ifdef ROAR_HAVE_SELECT
 int confh;
 fd_set fds;
 struct timeval timeout = {10, 0};
 struct roar_message    mes;
#endif
#ifdef ROAR_HAVE_UNIX
 int socks[2]; // for socketpair()
#endif
 struct roar_sockname sockname;

 // make valgrind happy
 memset(&socket_addr, 0, sizeof(socket_addr));
#ifdef ROAR_HAVE_SELECT
 memset(&mes,         0, sizeof(mes));
#endif

 ROAR_DBG("_roar_simple_new_stream_obj(con=%p, s=%p, rate=%i, channels=%i, bits=%i, codec=%i, dir=%i, mixer=%i) = ?", con, s, (int)rate, (int)channels, (int)bits, (int)codec, dir, mixer);

 if ( config != NULL ) {
  if ( config->workaround.workarounds & ROAR_LIBROAR_CONFIG_WAS_USE_EXECED ) {
   return roar_simple_new_stream_attachexeced_obj(con, s, rate, channels, bits, codec, dir, mixer);
  }
 }

 ROAR_DBG("_roar_simple_new_stream_obj(con=%p, s=%p, rate=%i, channels=%i, bits=%i, codec=%i, dir=%i, mixer=%i) = ?", con, s, (int)rate, (int)channels, (int)bits, (int)codec, dir, mixer);

 roar_libroar_nowarn();
 if ( roar_vio_ctl(roar_get_connection_vio2(con), ROAR_VIO_CTL_GET_SOCKNAME, &sockname) == -1 ) {
  roar_libroar_warn();
#ifdef ROAR_OS_OPENBSD
  sockname.type = ROAR_SOCKET_TYPE_UNIX;
#else
  ROAR_DBG("_roar_simple_new_stream_obj(con=%p, s=%p, rate=%i, channels=%i, bits=%i, codec=%i, dir=%i, mixer=%i) = -1", con, s, (int)rate, (int)channels, (int)bits, (int)codec, dir, mixer);

  return -1;
#endif
 }
 roar_libroar_warn();

 ROAR_DBG("_roar_simple_new_stream_obj(con=%p, s=%p, rate=%i, channels=%i, bits=%i, codec=%i, dir=%i, mixer=%i) = ?", con, s, (int)rate, (int)channels, (int)bits, (int)codec, dir, mixer);

 if ( sockname.type == ROAR_SOCKET_TYPE_DECNET ) {
  if ( roar_socket_get_local_nodename() ) {
   snprintf(file, 24, "%s::roar$TMP%04x%02x", roar_socket_get_local_nodename(), getpid(), count++);
  } else {
   return -1;
  }
#ifdef ROAR_HAVE_IPV4
 } else {
  strncpy(file, inet_ntoa(socket_addr.sin_addr), sizeof(file) - 1);
#endif
 }

 if ( sockname.type != ROAR_SOCKET_TYPE_UNIX ) {
  if ( (listen = roar_socket_listen(sockname.type, file, port)) == -1 ) {
   return -1;
  }
 }

 if ( sockname.type == ROAR_SOCKET_TYPE_INET ) {
#ifdef ROAR_HAVE_IPV4
  setsockopt(listen, SOL_SOCKET, SO_REUSEADDR, (void*)&opt, sizeof(int));

  len = sizeof(struct sockaddr_in);
  if ( getsockname(listen, (struct sockaddr *)&socket_addr, &len) == -1 ) {
   return -1;
  }
  port = ROAR_NET2HOST16(socket_addr.sin_port);
  ROAR_DBG("roar_simple_new_stream_obj(*): port=%i", port);
#else
  return -1;
#endif
 } else if ( sockname.type == ROAR_SOCKET_TYPE_DECNET ) {
#ifdef ROAR_HAVE_LIBDNET
  setsockopt(listen, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(int));
#else
  return -1;
#endif
 }

 if ( roar_stream_new(s, rate, channels, bits, codec) == -1 ) {
  return -1;
 }

 if ( roar_stream_connect(con, s, dir, mixer) == -1 ) {
  return -1;
 }

 if ( sockname.type != ROAR_SOCKET_TYPE_UNIX ) {
#ifdef ROAR_HAVE_SELECT
  if ( roar_stream_connect_to_ask(con, s, sockname.type, file, port) != -1 ) {

   FD_ZERO(&fds);
   FD_SET(listen, &fds);

   confh = roar_get_connection_fh(con);

   if ( confh != -1 ) {
    FD_SET(confh, &fds);
   }

   if ( select((confh > listen ? confh : listen) + 1, &fds, NULL, NULL, &timeout) < 1 ) {
    close(listen);

    // we don't need to check the content as we know it failed...
    if ( roar_recv_message(con, &mes, NULL) == -1 )
     return -1;

    if ( roar_kick(con, ROAR_OT_STREAM, s->id) == -1 )
     return -1;

    return roar_simple_new_stream_attachexeced_obj(con, s, rate, channels, bits, codec, dir, mixer);
   }

   if ( FD_ISSET(listen, &fds) ) {
    if ( (fh = accept(listen, NULL, NULL)) != -1 ) {
     /* errr, do we need we any error handling here? */
    }

    if ( roar_recv_message(con, &mes, NULL) == -1 ) {
     if ( fh != -1 )
      close(fh);
     fh = -1;
    } else if ( mes.cmd != ROAR_CMD_OK ) {
     if ( fh != -1 )
      close(fh);
     fh = -1;
    }
   } else {
    // we don't need to check the content as we know it failed...
    if ( roar_recv_message(con, &mes, NULL) == -1 ) {
     close(listen);
     return -1;
    }

    if ( mes.cmd != ROAR_CMD_OK ) {
     close(listen);
     if ( roar_kick(con, ROAR_OT_STREAM, s->id) == -1 )
      return -1;

     return roar_simple_new_stream_attachexeced_obj(con, s, rate, channels, bits, codec, dir, mixer);
    } else { // seems like we have a positive reply. So we retry the listen socket:
     FD_ZERO(&fds);
     FD_SET(listen, &fds);
     timeout.tv_sec = 0;
     timeout.tv_usec = 128000L;
     fh = -1;
     if ( select(listen + 1, &fds, NULL, NULL, &timeout) > 0 ) {
      if ( (fh = accept(listen, NULL, NULL)) == -1 ) {
       close(listen);
       if ( roar_kick(con, ROAR_OT_STREAM, s->id) == -1 )
        return -1;

       return roar_simple_new_stream_attachexeced_obj(con, s, rate, channels, bits, codec, dir, mixer);
      }
     }
    }
   }
  }

  close(listen);
#else
  return -1;
#endif
 } else { // this is sockname.type == ROAR_SOCKET_TYPE_UNIX
#ifdef ROAR_HAVE_UNIX
  if ( socketpair(AF_UNIX, SOCK_STREAM, 0, socks) == -1 ) {
   roar_kick(con, ROAR_OT_STREAM, s->id); // we do not need to check for errors
                                          // as we return -1 in both whys
   return -1;
  }

  if ( roar_stream_passfh(con, s, socks[0]) == -1 ) {
   roar_kick(con, ROAR_OT_STREAM, s->id); // we do not need to check for errors
                                          // as we return -1 anyway.
   close(socks[0]);
   close(socks[1]);

   return roar_simple_new_stream_attachexeced_obj(con, s, rate, channels, bits, codec, dir, mixer);
  }

  close(socks[0]);
  fh = socks[1];
#else
  roar_kick(con, ROAR_OT_STREAM, s->id);
  return -1;
#endif
 }

 if ( fh != -1 ) {
  if ( dir == ROAR_DIR_PLAY ) {
   (void)ROAR_SHUTDOWN(fh, SHUT_RD);
  } else if ( dir == ROAR_DIR_MONITOR || dir == ROAR_DIR_RECORD ) {
   (void)ROAR_SHUTDOWN(fh, SHUT_WR);
  }
 }

 s->fh = fh;

 ROAR_DBG("_roar_simple_new_stream_obj(con=%p, s=%p, rate=%i, channels=%i, bits=%i, codec=%i, dir=%i, mixer=%i) = %i", con, s, (int)rate, (int)channels, (int)bits, (int)codec, dir, mixer, fh);
 return fh;
}

int     roar_vio_simple_new_stream_obj (struct roar_vio_calls * calls,
                                        struct roar_connection * con,
                                        struct roar_stream * s,
                                        uint32_t rate, uint32_t channels, uint32_t bits, uint32_t codec,
                                        int dir, int mixer) {
 struct roar_stream stream;
 int fh;

 ROAR_DBG("roar_vio_simple_new_stream_obj(*) = ?");

 if ( calls == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( s == NULL )
  s = &stream;

 if ( (fh = _roar_simple_new_stream_obj(con, s, rate, channels, bits, codec, dir, mixer)) == -1 ) {
  ROAR_DBG("roar_vio_simple_new_stream_obj(*) = -1");
  return -1;
 }

 ROAR_DBG("roar_vio_simple_new_stream_obj(*): fh=%i", fh);

 return roar_vio_open_fh_socket(calls, fh);
}

//ll
