//roarctl.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2013
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

/* ckport options:
 * ckport: ignore-symbol: roar_req of target libroar0 -- Used within ping command.
 */

#define _UNITS_T_BASE_USEC

#include <roaraudio.h>
#include <roaraudio/units.h>
#include <libroardsp/libroardsp.h>

#if defined(ROAR_HAVE_SETGID) && defined(ROAR_HAVE_SETUID)
#define _POSIX_USERS
#endif

#ifdef _POSIX_USERS
#include <pwd.h>
#include <grp.h>
#endif

#include <sys/time.h>
#include <time.h>

#ifdef ROAR_HAVE_LIBM
#include <math.h>
#endif

int g_verbose = 0;

void display_client (struct roar_connection * con, int id);
void display_stream (struct roar_connection * con, int id);
int display_mixer (struct roar_connection * con, int stream);
int show_meta_all (struct roar_connection * con, int id);

void usage (void) {
 printf("roarctl [OPTIONS]... COMMAND [OPTS] [COMMAND [OPTS] [COMMAND [OPTS] [...]]]\n");

 printf("\nOptions:\n\n");

 printf("  --server SERVER         - Set server hostname\n"
        "  --help                  - Show this help\n"
        "  --verbose   -v          - Show verbose output\n"
        "  --list-aiprofiles       - Show audio info profiles and exit\n"
        "  --list-libstandards     - Gets list of library (libroar) supported standards\n"
        "  --enum-servers          - Show a list of possible servers\n"
        "  --hash-password         - Hash a password and exit\n"
       );

 printf("\nCommands:\n\n");
 printf(
        "  help                    - Show this help\n"
        "  sleep TIME              - Sleeps for TIME seconds\n"
#ifdef ROAR_HAVE_GETTIMEOFDAY
        "  ping  NUM               - Do NUM pings using NOOP commands\n"
#endif
        "  servername              - Show name of server currently connected to\n"
        "  whoami                  - Get own client ID\n"
        "\n"
        "  beep                    - Ask the server to beep.\n"
        "\n"
        "  listaiprofiles          - list audio info profiles\n"
        "  aiprofileget PROFILE    - show audio info profile PROFILE\n"
        "\n"
        "  standby, off            - Go into standby mode\n"
        "  resume, on              - Go into active mode\n"
        "  standbymode             - Show current standby mode\n"
        "  exit                    - Quits the roard (must be used as last command)\n"
        "  terminate               - Like exit but let the server up to serve still connected clients,\n"
        "                            new clients cann't connect and the server terminates after the last\n"
        "                            client disconnected\n"
        "\n"
        "  volume ID CHAN [scale S] V0 V1...\n"
        "                          - Sets volume for stream ID\n"
        "                            CHAN is the number of channels or 'mono' or 'stereo'\n"
        "                            if mono or stereo is chosen roarctl trys to set\n"
        "                            sensfull values for all channels even if the output\n"
        "                            is has more channels.\n"
        "                            An optional scale can be given using S.\n"
        "                            all other args are the volumes of the channels\n"
        "                            you may use integer or percent values.\n"
        "                            percent values can floating points.\n"
        "\n"
        "  flag   ID FLAGS         - Sets flags FLAGS on stream ID. FLAGS may be a comma\n"
        "                            separated list of flags.\n"
        "  unflag ID FLAGS         - Unsets flags on a stream. See flag.\n"
        "  toogleflag ID FLAGS     - Toggels flags on a stream. See flag.\n"
        "  protectflag ID FLAGS    - Protects flags on a stream. Those flags can not be\n"
        "                            changed anymore after the they got protected. See flag.\n"
        "\n"
        "  role ID ROLE            - Set role ROLE on stream ID.\n"
        "\n"
        "  kick TYPE ID            - Kicks object of TYPE with id ID\n"
        "                            Types: client stream sample source\n"
        "  newvirtual P D E R B C  - Adds a new virtual (child) stream\n"
        "                            Parameters:\n"
        "                             P: Parent stream ID, D: Direction,\n"
        "                             E: codEc, R: sample Rate,\n"
        "                             B: bits per sample, C: nummer of channels\n"
        "\n"
        "  metaget  ID TYPE        - Read meta date of type TYPE from stream ID\n"
// TODO: document metaset here.
        "  metasave ID FILE        - Saves meta data of stream ID to file FILE\n"
        "  metaload ID FILE        - Loads meta data from file FILE and set it on stream ID\n"
        "\n"
        "  serverinfo              - Gets general information about the server\n"
        "  libinfo                 - Gets general information about the library (libroar)\n"
        "  servertime              - Gets server's time\n"
        "  serveroinfo             - Gets Information about server output\n"
        "  serveroinfo2 DIR        - Gets Information about server output for stream direction dir\n"
        "\n"
        "  serverstandards         - Gets list of server supported standards\n"
        "  libstandards            - Gets list of library (libroar) supported standards\n"
        "\n"
        "  listclients             - Gets Information about clients\n"
        "  clientinfo ID           - Gets Information about client ID\n"
        "  liststreams             - Gets Information about streams\n"
        "  streaminfo ID           - Gets Information about stream ID\n"
        "  allinfo                 - Get all infos\n"
       );
}

static void _bin2hex(char * out, char * in, size_t inlen, int uppercase) {
 const char * tab = uppercase ? "0123456789ABCDEF" : "0123456789abcdef";
 unsigned char c;
 int nib0, nib1;

 for (; inlen; inlen--) {
  c = *(in++);
  nib0 = (c & 0x0F) >> 0;
  nib1 = (c & 0xF0) >> 4;
//  printf("inlen=%zu, c=%u, nibs={%i, %i}\n", inlen, (unsigned)c, nib1, nib0);
  *(out++) = tab[nib1];
  *(out++) = tab[nib0];
 }
 *out = 0;
}

int hash_password (void) {
 char * pw = NULL;
 char salt[8];
 char digest[24];
 char out[2*(8+24)+3+5+1] = "tiger:";

 if ( roar_passwd_simple_ask_pw(&pw, "Password to hash?", NULL) == -1 )
  return -1;

 roar_random_gen_nonce(salt, sizeof(salt));

 roar_hash_salted_buffer(digest, pw, roar_mm_strlen(pw), ROAR_HT_TIGER, salt, sizeof(salt));

 roar_mm_free(pw);

 _bin2hex(out+6, salt, sizeof(salt), 0);
 out[6+16] = ':';
 _bin2hex(out+6+16+1, digest, sizeof(digest), 0);

 out[sizeof(out)-1] = 0;

 printf("Password: %s\n", out);

 return 0;
}

int enum_servers (void) {
 struct roar_server * list;
 struct roar_server * c;
 int flags    = ROAR_ENUM_FLAG_DESC|ROAR_ENUM_FLAG_LOCATION;
 int dir      = -1;
 int socktype = -1;
 int i;

 if ( (list = roar_enum_servers(flags, dir, socktype)) == NULL )
  return -1;

 printf("Server           Location         Description\n");
 printf("----------------------------------------------------------------------\n");

 for (i = 0; ; i++) {
  c = &(list[i]);
  printf("%-16s %-16s %s\n",
             c->server      == NULL ? "(default)" : c->server,
             c->location    == NULL ? ""          : c->location,
             c->description == NULL ? ""          : c->description
        );
  if ( c->server == NULL )
   break;
 }

 roar_enum_servers_free(list);

 return 0;
}

#ifdef ROAR_HAVE_GETTIMEOFDAY
int ping (struct roar_connection * con, int num) {
 struct timeval         try, ans;
 struct roar_message    m;
 register int           ret;
 int                    i;
 double                 cur, min = 3600*1000, max = 0, sum = 0;

 if ( num == 0 )
  return 0;

 for (i = 0; i < num; i++) {
  memset(&m, 0, sizeof(m));

  m.cmd = ROAR_CMD_NOOP;
  m.stream  = -1;
  m.datalen =  0;

  // we use roar_req() directly here because of speed.
  // roar_noop() does basicly the same but is a bit slower.

  gettimeofday(&try, NULL);
  ret = roar_req(con, &m, NULL);
  gettimeofday(&ans, NULL);

  if ( ret == -1 )
   return -1;

  while (ans.tv_sec > try.tv_sec) {
   ans.tv_sec--;
   ans.tv_usec += 1000000;
  }
  ans.tv_usec -= try.tv_usec;

  cur = ans.tv_usec/1000.0;

  printf("Pong from server: seq=%i time=%.3f%cs\n", i, cur < 0.5 ? cur*1000.0 : cur, cur < 0.5 ? 'u' : 'm');

  sum += cur;
  if ( min > cur )
   min = cur;
  if ( cur > max )
   max = cur;

  if ( i != (num - 1) )
   roar_sleep(1);
 }

 printf("\n--- ping statistics ---\n");
 printf("%i packets transmitted\n", i);
 printf("rtt min/avg/max = %.3f/%.3f/%.3f ms\n", min, sum/(double)i, max);

 return 0;
}
#endif

#define _pm(m,n) if ( info->m != NULL ) printf(con != NULL ? "Server %-15s: %s\n" : "Library %-14s: %s\n", (n), (info->m));
void server_info (struct roar_connection * con) {
 struct roar_server_info * info = con != NULL ? roar_server_info(con) : roar_library_info();
 long hostid;

 if ( info == NULL ) {
  fprintf(stderr, "Error: can not get server info\n");
  return;
 }

 _pm(version, "Version");
 _pm(location, "Location");
 _pm(description, "Description");
 _pm(contact, "Contact");
 _pm(serial, "Serial");
 _pm(address, "Address");
 _pm(uiurl, "UI URL");
 _pm(license, "License");
// _pm(hostid, "HostID");
 if ( info->hostid != NULL ) {
  if ( sscanf(info->hostid, "0x%lx", &hostid) != 1 )
   hostid = 0;

  if ( hostid ) {
   printf(con != NULL ? "Server %-15s: %s (%li.%li.%li.%li)\n"
                      : "Library %-14s: %s (%li.%li.%li.%li)\n", "HostID", info->hostid,
           ((hostid & 0x00FF0000UL) >> 16UL),
           ((hostid & 0xFF000000UL) >> 24UL),
           ((hostid & 0x000000FFUL) >>  0UL),
           ((hostid & 0x0000FF00UL) >>  8UL)
         );
  } else {
   _pm(hostid, "HostID");
  }
 }
 _pm(build, "Build");
 _pm(un.sysname, "System sysname");
 _pm(un.release, "System release");
 _pm(un.nodename, "System nodename");
 _pm(un.machine, "System machine");

 roar_server_info_free(info);
}
#undef _pm

void server_time (struct roar_connection * con) {
 struct roar_time time;
#ifdef ROAR_HAVE_CTIME_R
 time_t server_time;
 char server_time_buf[32];
#endif


 if ( roar_get_time(con, &time) == -1 ) {
  fprintf(stderr, "Error: can not get server time: %s\n", roar_vs_strerr(roar_error));
  return;
 }

#ifdef ROAR_HAVE_CTIME_R
 server_time = time.t_sec;
 if ( ctime_r(&server_time, server_time_buf) == NULL )
  server_time_buf[0] = 0;

 if ( server_time_buf[0] )
  if ( server_time_buf[strlen(server_time_buf)-1] == '\n' )
   server_time_buf[strlen(server_time_buf)-1] = 0;
#endif

 if ( g_verbose ) {
  printf("Server time           : %s%s%s%" LIBROAR__ll "u.%06" LIBROAR__ll "u [+%" LIBROAR__ll "u/2^64] sec\n",
#ifdef ROAR_HAVE_CTIME_R
         "\"",
         server_time_buf,
         "\" ",
#else
         "", "", "",
#endif
         (LIBROAR__longlong unsigned int)time.t_sec,
         (LIBROAR__longlong unsigned int) time.t_ssec / (LIBROAR__longlong unsigned int)18446744073709LLU,
         (LIBROAR__longlong unsigned int) time.t_ssec);
 } else {
  if ( time.c_freq == 1000000000LL && time.t_ssec == 0 ) {
   printf("Server time           : %s%s%s%" LIBROAR__ll "u sec\n",
#ifdef ROAR_HAVE_CTIME_R
         "\"",
         server_time_buf,
         "\" ",
#else
         "", "", "",
#endif
          (LIBROAR__longlong unsigned int)time.t_sec);
  } else {
   printf("Server time           : %s%s%s%" LIBROAR__ll "u.%06" LIBROAR__ll"u sec\n",
#ifdef ROAR_HAVE_CTIME_R
         "\"",
         server_time_buf,
         "\" ",
#else
         "", "", "",
#endif
          (LIBROAR__longlong unsigned int)time.t_sec,
          (LIBROAR__longlong unsigned int) time.t_ssec / (LIBROAR__longlong unsigned int)18446744073709LLU);
  }
 }
 if ( time.c_freq )
  printf("Server clock frequency: %f Hz\n", (float)((double)time.c_freq / (double)1000000000.));
 if ( time.c_drift ) {
#ifdef ROAR_HAVE_LIBM
  printf("Server clock drift    : %" LIBROAR__ll "u:2^64 (~10^%f)\n",
         (LIBROAR__longlong unsigned int)time.c_drift, logf((float)time.c_drift)/2.302585f);
#else
  printf("Server clock drift    : %" LIBROAR__ll "u:2^64\n",
         (LIBROAR__longlong unsigned int)time.c_drift);
#endif
 }
}

void server_oinfo (struct roar_connection * con, int dir) {
 struct roar_stream s;

 if ( roar_server_oinfo(con, &s, dir) == -1 ) {
  fprintf(stderr, "Error: can not get server output info\n");
  return;
 }

 printf("Stream direction      : %s\n", roar_dir2str(s.dir));
 printf("Server Output rate    : %i Hz\n", s.info.rate);
 printf("Server Output bits    : %i\n", s.info.bits);
 printf("Server Output channels: %i\n", s.info.channels);
 printf("Server Output codec   : %i (%s%s)\n", s.info.codec, roar_codec2str(s.info.codec),
                                     s.info.codec == ROAR_CODEC_DEFAULT ? " native" : "");
// printf("Server Output rate: %i", s.info.rate);
  if ( g_verbose > 1 && s.pos != (uint32_t)-1 ) {
   if ( s.info.rate && s.info.channels ) {
    printf("Server Position       : %lu S (%.3fs)\n", (unsigned long int) s.pos, (float)s.pos/(s.info.rate*s.info.channels));
   } else {
    printf("Server Position       : %lu S\n", (unsigned long int) s.pos);
   }
  }
}

static void print_stds (const struct roar_stds * stds) {
 size_t i;
 int vendor, standard, version;
 char numbuf[2][8];
 const char * vendor_name;

 for (i = 0; i < stds->stds_len; i++) {
  vendor   = ROAR_STD_VENDOR(stds->stds[i]);
  standard = ROAR_STD_STD(stds->stds[i]);
  version  = ROAR_STD_VERSION(stds->stds[i]);

  if ( (vendor_name = roar_stds_vendor2str(vendor)) == NULL ) {
   snprintf(numbuf[0], sizeof(numbuf[0]), "%i", vendor);
   numbuf[0][sizeof(numbuf[0])-1] = 0;
   vendor_name = numbuf[0];
  }

  if ( version == 0 ) {
   numbuf[1][0] = 0;
  } else {
   snprintf(numbuf[1], sizeof(numbuf[1]), "-%i", version);
   numbuf[1][sizeof(numbuf[1])-1] = 0;
  }

  printf("Server standard       : %s-%i%s\n", vendor_name, standard, numbuf[1]);
 }
}

void server_standards (struct roar_connection * con) {
 struct roar_stds * stds;

 if ( roar_caps_stds(con, &stds, NULL, -1) == -1 ) {
  fprintf(stderr, "Error: can not get server standards\n");
  return;
 }

 print_stds(stds);

 roar_stds_free(stds);
}

void lib_standards (void) {
 print_stds(libroar_libstds());
}

const char * proc_name (pid_t pid) {
 static char ret[80] = "?";
#ifdef __linux__
 char file[80], buf[80], *r;
 int  i;

 snprintf(file, sizeof(file)-1, "/proc/%i/exe", pid);
 file[sizeof(file)-1] = 0;

 ret[0] = '?';
 ret[1] = 0;

 if ( (i = readlink(file, buf, sizeof(buf)-1)) != -1 ) {
  buf[i] = 0;
  if ( (r = strrchr(buf, '/')) != NULL ) {
   r++;
   if ( *r != 0 ) {
    strncpy(ret, r, sizeof(ret)-1);
    ret[sizeof(ret)-1] = 0;
   }
  }
 }
#else
 (void)pid;
#endif

 return ret;
}

void list_clients (struct roar_connection * con) {
 int i;
 int num;
 int id[ROAR_CLIENTS_MAX];

 if ( (num = roar_list_clients(con, id, ROAR_CLIENTS_MAX)) == -1 ) {
  fprintf(stderr, "Error: can not get client list\n");
  return;
 }

 for (i = 0; i < num; i++) {
  display_client(con, id[i]);
 }
}

void display_client (struct roar_connection * con, int id) {
 static int self_id = -1;
 static struct roar_client   self_client_store;
 static struct roar_client * self_client = NULL;
 struct roar_client c;
 char tmp[80];
 int h;
#ifdef _POSIX_USERS
 struct group  * grp = NULL;
 struct passwd * pwd = NULL;
#endif

 if ( self_id == -1 ) {
  if ( (self_id = roar_get_clientid(con)) != -1 ) {
   if ( roar_get_client(con, &self_client_store, self_id) == -1 ) {
    self_id = -1;
   } else {
    self_client = &self_client_store;
   }
  }
 }

 printf("client %i:\n", id);
 if ( roar_get_client(con, &c, id) == -1 ) {
  fprintf(stderr, "Error: can not get client info\n");
  return;
 }

 if ( c.name[0] != '\0' )
  printf("Client name           : %s\n", c.name);
  if ( roar_nnode_get_socktype(&(c.nnode)) != ROAR_SOCKET_TYPE_UNKNOWN ) {
  if ( roar_nnode_to_str(&(c.nnode), tmp, sizeof(tmp)) == 0 ) {
   printf("Client network node   : %s\n", tmp);
  }
 }

 if ( c.pid != -1 ) {
  if ( self_id != -1 && roar_nnode_cmp(&(self_client->nnode), &(c.nnode)) == 0 ) {
   printf("Client PID            : %i(%s)\n", c.pid, proc_name(c.pid));
  } else { 
   printf("Client PID            : %i\n", c.pid);
  }
 }
 if ( c.uid != -1 ) {
#ifdef _POSIX_USERS
  if ( self_id != -1 && roar_nnode_cmp(&(self_client->nnode), &(c.nnode)) == 0 ) {
   pwd = getpwuid(c.uid);
   grp = getgrgid(c.gid);
   printf("Client UID/GID        : %i(%s)/%i(%s)\n", c.uid, pwd ? pwd->pw_name : "?", c.gid, grp ? grp->gr_name : "?");
  } else {
#else
  if ( 1 ) {
#endif
   printf("Client UID/GID        : %i/%i\n", c.uid, c.gid);
  }
 }

 if ( g_verbose && c.proto != ROAR_PROTO_NONE ) {
  printf("Client Protocol       : %s\n", roar_proto2str(c.proto));
 }

 if ( g_verbose && c.byteorder != ROAR_BYTEORDER_UNKNOWN ) {
  if ( c.byteorder == ROAR_BYTEORDER_NETWORK ) {
   roar_mm_strscpy(tmp, " (network byteorder");
  } else {
   *tmp = 0;
  }

  if ( c.byteorder == ROAR_BYTEORDER_NATIVE ) {
   if ( *tmp ) {
    roar_mm_strscat(tmp, ", native");
   } else {
    roar_mm_strscpy(tmp, " (native");
   }
  }

  if ( *tmp )
   roar_mm_strscat(tmp, ")");

  printf("Client Byteorder      : %s%s\n", roar_byteorder2str(c.byteorder), tmp);
 }

 if ( c.execed != -1 )
  printf("Execed stream         : %i\n", c.execed);

 for (h = 0; h < ROAR_CLIENTS_MAX_STREAMS_PER_CLIENT; h++)
  if ( c.streams[h] != -1 )
   printf("stream                : %i\n", c.streams[h]);
}

void list_streams (struct roar_connection * con) {
 int i;
 int num;
 int id[ROAR_STREAMS_MAX];

 if ( (num = roar_list_streams(con, id, ROAR_STREAMS_MAX)) == -1 ) {
  fprintf(stderr, "Error: can not get stream list\n");
  return;
 }

 for (i = 0; i < num; i++) {
  display_stream(con, id[i]);
 }
}

void display_stream (struct roar_connection * con, int id) {
 char chanmap[ROAR_MAX_CHANNELS];
 struct roar_stream s;
 struct roar_stream_info info;
 char buffer[1024];
 char * flags = buffer;
 char * name  = buffer;
 char * infotext;
 const char * mime;
 size_t len;

 printf("stream %i:\n", id);
 if ( roar_get_stream(con, &s, id) == -1 ) {
  fprintf(stderr, "Error: can not get stream info\n");
  return;
 }
 printf("Stream direction      : %s\n", roar_dir2str(s.dir));

 if ( roar_stream_get_name(con, &s, name, sizeof(buffer)) == 0 )
  printf("Stream name           : %s\n", name);

 if ( (int)s.pos_rel_id == -1 ) {
  printf("Relativ position id   : none (stream not synchronized)\n");
 } else if ( (int)s.pos_rel_id == id ) {
  printf("Relativ position id   : %i (self synchronized)\n", s.pos_rel_id);
 } else {
  printf("Relativ position id   : %i (synchronized)\n", s.pos_rel_id);
 }
 if ( g_verbose > 1 && s.pos != (uint32_t)-1 ) {
  if ( s.info.rate && s.info.channels ) {
   printf("Position              : %lu S (%.3fs)\n", (unsigned long int) s.pos,
                                   (float)s.pos/(s.info.rate*s.info.channels));
  } else {
   printf("Position              : %lu S\n", (unsigned long int) s.pos);
  }
 }

 switch (s.dir) {
  case ROAR_DIR_MIDI_IN:
  case ROAR_DIR_MIDI_OUT:
    infotext = " ticks/s";
   break;
  case ROAR_DIR_LIGHT_IN:
  case ROAR_DIR_LIGHT_OUT:
    infotext = " updates/s";
   break;
  default:
    infotext = " Hz";
 }
 if ( s.info.rate )
  printf("Stream sample rate    : %i%s\n", s.info.rate, infotext);
 if ( s.info.bits )
  printf("Stream bits           : %i\n", s.info.bits);
 if ( s.info.channels )
  printf("Stream channels       : %i\n", s.info.channels);

 mime = roar_codec2mime(s.info.codec);
 printf("Stream codec          : %2i (%s%s%s%s)\n", s.info.codec, roar_codec2str(s.info.codec),
                                      s.info.codec == ROAR_CODEC_DEFAULT ? " native" : "",
                                      mime == NULL ? "" : " mimetype:",
                                      mime == NULL ? "" : mime
                                     );
 if ( roar_stream_get_info(con, &s, &info) != -1 ) {
  if ( info.codec != s.info.codec ) {
   mime = roar_codec2mime(info.codec);
   printf("Streamed codec        : %2i (%s%s%s%s)\n", info.codec, roar_codec2str(info.codec),
                                      info.codec == ROAR_CODEC_DEFAULT ? " native" : "",
                                      mime == NULL ? "" : " mimetype:",
                                      mime == NULL ? "" : mime
                                     );
  }

  if ( g_verbose ) {
   if ( info.block_size )
    printf("Stream block size     : %i Byte\n", info.block_size);

   printf("Underruns pre/post    : %i/%i\n",   info.pre_underruns, info.post_underruns);
   if ( g_verbose > 1 )
    printf("Stream delay          : %ims (%.2fm)\n",   (int)info.delay/1000, (info.delay*(float)_SPEED_OF_SOUND));

   if ( g_verbose > 1 )
    printf("Stream mixer          : %i\n",   info.mixer);

   if ( g_verbose > 1 )
    printf("Stream state          : %s\n",   roar_streamstate2str(info.state));

   if ( g_verbose > 1 )
    printf("Stream role           : %s\n",   roar_role2str(info.role));

   *flags = 0;
#define _strcat(dst,src) roar_mm_strlcat((dst),(src), sizeof(buffer))
   if ( info.flags & ROAR_FLAG_PRIMARY )
    _strcat(flags, "primary ");
   if ( info.flags & ROAR_FLAG_SYNC )
    _strcat(flags, "sync ");
   if ( info.flags & ROAR_FLAG_OUTPUT )
    _strcat(flags, "output ");
   if ( info.flags & ROAR_FLAG_SOURCE )
    _strcat(flags, "source ");
   if ( info.flags & ROAR_FLAG_META )
    _strcat(flags, "meta ");
   if ( info.flags & ROAR_FLAG_AUTOCONF )
    _strcat(flags, "autoconf ");
   if ( info.flags & ROAR_FLAG_CLEANMETA )
    _strcat(flags, "cleanmeta ");
   if ( info.flags & ROAR_FLAG_HWMIXER )
    _strcat(flags, "hwmixer ");
   if ( info.flags & ROAR_FLAG_PAUSE )
    _strcat(flags, "pause ");
   if ( info.flags & ROAR_FLAG_MUTE )
    _strcat(flags, "mute ");
   if ( info.flags & ROAR_FLAG_MMAP )
    _strcat(flags, "mmap ");
   if ( info.flags & ROAR_FLAG_ANTIECHO )
    _strcat(flags, "antiecho ");
   if ( info.flags & ROAR_FLAG_VIRTUAL )
    _strcat(flags, "virtual ");
   if ( info.flags & ROAR_FLAG_RECSOURCE )
    _strcat(flags, "recsource ");
   if ( info.flags & ROAR_FLAG_PASSMIXER )
    _strcat(flags, "passmixer ");
   if ( info.flags & ROAR_FLAG_PRETHRU )
    _strcat(flags, "prethru ");
   if ( info.flags & ROAR_FLAG_IMMUTABLE )
    _strcat(flags, "immutable ");
   if ( info.flags & ROAR_FLAG_ENHANCE )
    _strcat(flags, "enhance ");
   if ( info.flags & ROAR_FLAG_SINGLESINK )
    _strcat(flags, "singlesink ");
#undef _strcat

   printf("Flags                 : %s\n", flags);
  }
 }

 if ( g_verbose && (s.info.channels != 0 && s.info.channels <= ROAR_MAX_CHANNELS) ) {
  len = ROAR_MAX_CHANNELS;
  if ( roar_stream_get_chanmap(con, &s, chanmap, &len) == -1 ) {
   fprintf(stderr, "Error: can not get stream channel map\n");
  } else {
   if ( roardsp_chanlist2str(chanmap, len, buffer, 1024) == -1 ) {
    fprintf(stderr, "Error: can not convert channel map into string\n");
   } else {
    printf("Channel Map           : %s\n", buffer);
   }
  }
 }

 if ( s.dir != ROAR_DIR_THRU ) {
  if ( s.info.channels != 0 && s.info.channels <= ROAR_MAX_CHANNELS ) {
   display_mixer(con, id);
  }
  show_meta_all(con, id);
 }
}

int display_mixer (struct roar_connection * con, int stream) {
 struct roar_stream_rpg rpg;
 struct roar_mixer_settings mixer;
 struct roar_stream s;
 int channels;
 int i;
 float fs;

 if ( roar_get_vol(con, stream, &mixer, &channels) == -1 ) {
  fprintf(stderr, "Error: can not get stream mixer info for stream %i\n", stream);
  return -1;
 }

 roar_stream_new_by_id(&s, stream);
 if ( roar_stream_get_rpg(con, &s, &rpg) != -1 ) {
  printf("ReplayGain Mode       : %s\n", roar_rpgmode2str(rpg.mode));
 }

#ifdef ROAR_HAVE_LIBM
#define _DB ", %.2fdB"
#else
#define _DB ""
#endif

 fs = (float)mixer.scale / 100.;

 if ( channels ) { // we hide RPG info for zero-channel streams
  printf("Mixer ReplayGain      : %i/%i (%.2f%%" _DB ")\n", mixer.rpg_mul, mixer.rpg_div,
                                                          100.*(float)mixer.rpg_mul/((float)mixer.rpg_div)
#ifdef ROAR_HAVE_LIBM
                           , 20*log10f((float)mixer.rpg_mul/(float)mixer.rpg_div)
#endif
        );
 }

 for (i = 0; i < channels; i++)
  printf("Mixer volume chan %2i  : %i/%i (%.2f%%" _DB ")\n", i, mixer.mixer[i], mixer.scale,
                           (float)mixer.mixer[i]/fs
#ifdef ROAR_HAVE_LIBM
                          , 20*log10f((float)mixer.mixer[i]/(float)mixer.scale)
#endif
        );

 return 0;
}

static unsigned int set_mixer_parse_volume (char * k, int len, uint16_t scale) {
 float fs = scale;

 switch (k[len - 1]) {
  case '%':
    k[len - 1] = 0;
    return (atof(k)*fs)/100.;
   break;
#ifdef ROAR_HAVE_LIBM
  case 'b':
  case 'B':
    // TODO: can we hanle the error better?
    if ( len < 2 )
     return 0;

    k[len - 2] = 0;
    return powf(10, atof(k)/20.f)*fs;
   break;
#endif
 }

 return atoi(k);
}

int set_mixer (struct roar_connection * con, int * cur, int max, char * arg[]) {
 uint16_t scale = 65535;
 int chans = 0;
 int id;
 int i;
 int len;
 int old_chans;
 char * k;
 struct roar_mixer_settings mixer;
 struct roar_mixer_settings old_mixer;
 int mode = ROAR_SET_VOL_ALL;

 if (*cur + 2 > max)
  return -1;

 id = atoi(arg[++(*cur)]);

 k = arg[++(*cur)];

 if ( !strcmp(arg[*cur + 1], "scale") ) {
  (*cur)++; // 'scale'
  (*cur)++;
  scale = set_mixer_parse_volume(arg[*cur], strlen(arg[*cur]), 65535);
 }

 mixer.scale = scale;

 if ( roar_get_vol(con, id, &old_mixer, &old_chans) == -1 ) {
  fprintf(stderr, "Error: can not get stream mixer info for stream %i\n", id);
  return -1;
 }

 if ( strcmp(k, "mono") == 0 ) {
  chans = 1;
  mode  = ROAR_SET_VOL_UNMAPPED;
 } else if ( strcmp(k, "stereo") == 0 ) {
  chans = 2;
  mode  = ROAR_SET_VOL_UNMAPPED;
 } else {
  chans = atoi(k);
 }

 // ensure most simple mode:
 if ( old_chans == chans && mode == ROAR_SET_VOL_UNMAPPED )
  mode = ROAR_SET_VOL_ALL;

 if ( *cur + chans > max )
  return -1;

 for (i = 0; i < chans; i++) {
  k   = arg[++(*cur)];
  len = strlen(k);

  mixer.mixer[i] = set_mixer_parse_volume(k, len, scale);
 }

 if ( roar_set_vol(con, id, &mixer, chans, mode) == 0 )
  return 0;

 // no fallback if already using ROAR_SET_VOL_ALL:
 if ( mode == ROAR_SET_VOL_ALL )
  return -1;

 if ( roar_conv_volume(&mixer, &mixer, old_chans, chans) == -1 )
  return -1;

 chans = old_chans;
 mode  = ROAR_SET_VOL_ALL;

 if ( roar_set_vol(con, id, &mixer, chans, mode) == 0 )
  return 0;

 return -1;
}


int newvirtual (struct roar_connection * con, char *p_s, char *d_s, char *e_s, char *r_s, char *b_s, char *c_s) {
 struct roar_stream s;
 int dir    = roar_str2dir(d_s);
 int parent = atoi(p_s);

 ROAR_DBG("newvirtual(*): dir=%i, parent=%i", dir, parent);

 if ( roar_stream_new(&s, roar_str2rate(r_s), roar_str2channels(c_s), roar_str2bits(b_s), roar_str2codec(e_s)) == -1 )
  return -1;

 return roar_simple_connect_virtual(con, &s, parent, dir);
}

int set_meta (struct roar_connection * con, int id, char * mode, char * type, char * val) {
 struct roar_meta   meta;
 struct roar_stream s;
 int mode_i = ROAR_META_MODE_SET;

 if ( roar_stream_new_by_id(&s, id) == -1 )
  return -1;

// printf("set_meta(*): mode='%s', type='%s', val='%s'\n", mode, type, val);

 if ( strcmp(mode, "add") == 0 ) {
  mode_i = ROAR_META_MODE_ADD;
 }

 meta.type   = roar_meta_inttype(type);
 meta.value  = val;
 meta.key[0] = 0;

 if ( meta.type == -1 ) {
  fprintf(stderr, "Error: unknown type: %s\n", type);
  return -1;
 }

// printf("D: type=%i, mode=%i\n", meta.type, mode_i);

 if ( roar_stream_meta_set(con, &s, mode_i, &meta) == -1 )
  return -1;

 meta.type  = ROAR_META_TYPE_NONE;
 meta.value = NULL;

 return roar_stream_meta_set(con, &s, ROAR_META_MODE_FINALIZE, &meta);
}

int load_meta (struct roar_connection * con, int id, char * file) {
 struct roar_meta   meta;
 struct roar_stream s;
 int mode_i = ROAR_META_MODE_SET;
 FILE * in;
 char lion[1024];
 char * v;

 if ( roar_stream_new_by_id(&s, id) == -1 )
  return -1;

 if ( (in = fopen(file, "r")) == NULL )
  return -1;

 while (fgets(lion, 1024, in) != NULL) {
  if ( (v = strtok(lion, "\r\n")) != NULL )
   if ( (v = strtok(NULL, "\r\n")) != NULL )
    *(v-1) = 0;

  if ( (v = strstr(lion, "=")) == NULL ) {
   fprintf(stderr, "Error: can not parse meta data lion: %s\n", lion);
   continue;
  }

  *v++ = 0;

  meta.type   = roar_meta_inttype(lion);
  meta.value  = v;
  meta.key[0] = 0;

  if ( meta.type == -1 ) {
   fprintf(stderr, "Error: unknown type: %s\n", lion);
   continue;
  }

  if ( roar_stream_meta_set(con, &s, mode_i, &meta) == -1 ) {
   fclose(in);
   return -1;
  }
 }

 fclose(in);

 meta.type  = ROAR_META_TYPE_NONE;
 meta.value = NULL;

 return roar_stream_meta_set(con, &s, ROAR_META_MODE_FINALIZE, &meta);
}

int show_meta_type (struct roar_connection * con, int id, const char * type) {
 struct roar_meta   meta;
 struct roar_stream s;

 if ( roar_stream_new_by_id(&s, id) == -1 )
  return -1;

 meta.type  = roar_meta_inttype(type);

 if ( meta.type == -1 ) {
  fprintf(stderr, "Error: unknown type: %s\n", type);
  return -1;
 }

 if ( roar_stream_meta_get(con, &s, &meta) == -1 )
  return -1;

 printf("Meta %-17s: %s\n", roar_meta_strtype(meta.type), meta.value);

 roar_meta_free(&meta);

 return 0;
}

int show_meta_all (struct roar_connection * con, int id) {
 struct roar_stream s;
 int types[ROAR_META_MAX_PER_STREAM];
 int i;
 int len;

 if ( roar_stream_new_by_id(&s, id) == -1 )
  return -1;

 if ( (len = roar_stream_meta_list(con, &s, types, ROAR_META_MAX_PER_STREAM)) == -1 )
  return -1;

 for (i = 0; i < len; i++)
  show_meta_type(con, id, roar_meta_strtype(types[i]));

 return 0;
}

int save_meta (struct roar_connection * con, int id, char * file) {
 struct roar_stream s;
 struct roar_meta   meta;
 int types[ROAR_META_MAX_PER_STREAM];
 int i;
 int len;
 FILE * out;

 if ( roar_stream_new_by_id(&s, id) == -1 )
  return -1;

 if ( (len = roar_stream_meta_list(con, &s, types, ROAR_META_MAX_PER_STREAM)) == -1 )
  return -1;

 if ( (out = fopen(file, "w")) == NULL )
  return -1;

 for (i = 0; i < len; i++) {
/*
  show_meta_type(con, id, roar_meta_strtype(types[i]));
*/
  meta.type  = types[i];

  if ( roar_stream_meta_get(con, &s, &meta) == -1 )
   continue;

//  printf("Meta %-17s: %s\n", roar_meta_strtype(meta.type), meta.value);

  fprintf(out, "%s=%s\n", roar_meta_strtype(meta.type), meta.value);

  roar_meta_free(&meta);
 }

 fclose(out);

 return 0;
}

int set_flags (struct roar_connection * con, int id, int action, char * flags) {
 uint32_t f = ROAR_FLAG_NONE;
 char * c;
 struct roar_stream s[1];

 if ( roar_stream_new_by_id(s, id) == -1 )
  return -1;

 c = strtok(flags, ",");
 while (c != NULL) {
  if ( !strcmp(c, "meta") ) {
   f |= ROAR_FLAG_META;
  } else if ( !strcmp(c, "primary") ) {
   f |= ROAR_FLAG_PRIMARY;
  } else if ( !strcmp(c, "sync") ) {
   f |= ROAR_FLAG_SYNC;
  } else if ( !strcmp(c, "cleanmeta") ) {
   f |= ROAR_FLAG_CLEANMETA;
  } else if ( !strcmp(c, "hwmixer") ) {
   f |= ROAR_FLAG_HWMIXER;
  } else if ( !strcmp(c, "pause") ) {
   f |= ROAR_FLAG_PAUSE;
  } else if ( !strcmp(c, "mute") ) {
   f |= ROAR_FLAG_MUTE;
  } else if ( !strcmp(c, "mmap") ) {
   f |= ROAR_FLAG_MMAP;
  } else if ( !strcmp(c, "antiecho") ) {
   f |= ROAR_FLAG_ANTIECHO;
  } else if ( !strcmp(c, "recsource") ) {
   f |= ROAR_FLAG_RECSOURCE;
  } else if ( !strcmp(c, "passmixer") ) {
   f |= ROAR_FLAG_PASSMIXER;
  } else if ( !strcmp(c, "virtual") ) {
   f |= ROAR_FLAG_VIRTUAL;
  } else if ( !strcmp(c, "prethru") ) {
   f |= ROAR_FLAG_PRETHRU;
  } else if ( !strcmp(c, "immutable") ) {
   f |= ROAR_FLAG_IMMUTABLE;
  } else if ( !strcmp(c, "enhance") ) {
   f |= ROAR_FLAG_ENHANCE;
  } else if ( !strcmp(c, "singlesink") ) {
   f |= ROAR_FLAG_SINGLESINK;
  } else {
   fprintf(stderr, "Error: unknown flag: %s\n", c);
   return -1;
  }

  c = strtok(NULL, ",");
 }

 return roar_stream_set_flags(con, s, f, action);
}

int set_role (struct roar_connection * con, int id, int role) {
 struct roar_stream s;

 if ( id == -1 )
  return -1;

 if ( roar_stream_new_by_id(&s, id) == -1 )
  return -1;

 if ( roar_stream_set_role(con, &s, role) == -1 )
  return -1;

 return 0;
}

int show_aiprofile (const char * profile) {
 struct roar_audio_info info;
 const char * mime;

 if ( roar_profile2info(&info, profile) == -1 ) {
  fprintf(stderr, "Error: unknown profile: %s\n", profile);
  return -1;
 }

 mime = roar_codec2mime(info.codec);

 printf("Profile Name          : %s\n", profile);

 if ( info.rate )
  printf("Profile sample rate   : %i\n", info.rate);
 if ( info.bits )
  printf("Profile bits          : %i\n", info.bits);
 if ( info.channels )
  printf("Profile channels      : %i\n", info.channels);

 printf("Profile codec         : %2i (%s%s%s%s)\n", info.codec, roar_codec2str(info.codec),
                                       info.codec == ROAR_CODEC_DEFAULT ? " native" : "",
                                       mime == NULL ? "" : " mimetype:",
                                       mime == NULL ? "" : mime);

 return 0;
}

int list_aiprofiles (void) {
 const char * list[1024];
 ssize_t ret;
 ssize_t i;

 ret = roar_profiles_list(list, 1024, 0);

 if ( ret == -1 ) {
  fprintf(stderr, "Error: can not read list of profiles\n");
  return -1;
 }

 for (i = 0; i < ret; i++) {
  printf("profile %" LIBROAR__ll "i:\n", (LIBROAR__longlong signed int)i);
  show_aiprofile(list[i]);
 }

 return 0;
}

int main (int argc, char * argv[]) {
 struct roar_connection con;
 const char * server   = NULL;
 const char * k = NULL;
 int    i;
 int    t = 0;

 for (i = 1; i < argc; i++) {
  k = argv[i];

  if ( strcmp(k, "--server") == 0 ) {
   server = argv[++i];
  } else if ( strcmp(k, "-v") == 0 || strcmp(k, "--verbose") == 0 ) {
   g_verbose++;
  } else if ( strcmp(k, "--help") == 0 ) {
   usage();
   return 0;
  } else if ( strcmp(k, "--list-aiprofiles") == 0 ) {
   list_aiprofiles();
   return 0;
  } else if ( strcmp(k, "--list-libstandards") == 0 ) {
   lib_standards();
   return 0;
  } else if ( strcmp(k, "--enum-servers") == 0 ) {
   enum_servers();
   return 0;
  } else if ( strcmp(k, "--hash-password") == 0 ) {
   hash_password();
   return 0;
  } else if ( *k == '-' ) {
   fprintf(stderr, "Error: unknown argument: %s\n", k);
   usage();
   return 1;
  } else {
   break;
  }
 }

 // connect

 if ( roar_simple_connect(&con, server, "roarctl") == -1 ) {
  fprintf(stderr, "Error: Can not connect to server: %s: %s(%i)\n",
          server == NULL ? "(default)" : server, roar_vs_strerr(roar_error), roar_error);
  return 1;
 }

 if ( i == argc ) {
  fprintf(stderr, "Error: No Commands given\n");
  return 0; // this is not a fatal error...
 }

 for (; i < argc; i++) {
  k = argv[i];
  // cmd is in k

  printf("--- [ %s ] ---\n", k);

  if ( !strcmp(k, "help") ) {
   usage();

  } else if ( !strcmp(k, "sleep") ) {
   roar_sleep(atoi(argv[++i]));

  } else if ( !strcmp(k, "ping") ) {
#ifdef ROAR_HAVE_GETTIMEOFDAY
   if ( ping(&con, atoi(argv[++i])) == -1 ) {
    fprintf(stderr, "Error: can not ping\n");
   }
#else
    fprintf(stderr, "Error: ping not supported.\n");
    i++;
#endif
  } else if ( !strcmp(k, "servername") ) {
   printf("Server Name           : %s\n", roar_get_connection_server(&con));

  } else if ( !strcmp(k, "standby") || !strcmp(k, "off") ) {
   if ( roar_set_standby(&con, ROAR_STANDBY_ACTIVE) == -1 ) {
    fprintf(stderr, "Error: can not set mode to standby\n");
   } else {
    printf("going into standby\n");
   }
  } else if ( !strcmp(k, "resume") || !strcmp(k, "on") ) {
   if ( roar_set_standby(&con, ROAR_STANDBY_INACTIVE) == -1 ) {
    fprintf(stderr, "Error: can not set mode to active\n");
   } else {
    printf("going into active mode\n");
   }

  } else if ( !strcmp(k, "exit") ) {
   if ( roar_terminate(&con, 0) == -1 ) {
    fprintf(stderr, "Error: can not quit server\n");
   } else {
    printf("Server quited\n");
    break;
   }
  } else if ( !strcmp(k, "terminate") ) {
   if ( roar_terminate(&con, 1) == -1 ) {
    fprintf(stderr, "Error: can not terminate server\n");
   } else {
    printf("Server got asked to quited\n");
    break;
   }

  } else if ( !strcmp(k, "standbymode") ) {
   t = roar_get_standby(&con);
   if ( t == -1 ) {
    fprintf(stderr, "Error: can not get stanby mode\n");
   } else if ( t == ROAR_STANDBY_ACTIVE ) {
    printf("Server is in standby\n");
   } else if ( t == ROAR_STANDBY_INACTIVE ) {
    printf("Server is active\n");
   } else {
    fprintf(stderr, "Error: unknown standby mode: %i\n", t);
   }

  } else if ( !strcmp(k, "whoami") ) {
   printf("My client ID is: %i\n", roar_get_clientid(&con));
  } else if ( !strcmp(k, "beep") ) {
   roar_beep(&con, NULL);
  } else if ( !strcmp(k, "serverinfo") ) {
   server_info(&con);
  } else if ( !strcmp(k, "libinfo") ) {
   server_info(NULL);
  } else if ( !strcmp(k, "servertime") ) {
   server_time(&con);
  } else if ( !strcmp(k, "serveroinfo") ) {
   server_oinfo(&con, -1);
  } else if ( !strcmp(k, "serveroinfo2") ) {
   t = roar_str2dir(argv[++i]);
   if ( t == -1 ) {
    fprintf(stderr, "Error: unknown stream direction: %s\n", argv[i]);
   } else {
    server_oinfo(&con, t);
   }
  } else if ( !strcmp(k, "serverstandards") ) {
   server_standards(&con);
  } else if ( !strcmp(k, "libstandards") ) {
   lib_standards();
  } else if ( !strcmp(k, "listclients") ) {
   list_clients(&con);
  } else if ( !strcmp(k, "clientinfo") ) {
   display_client(&con, atoi(argv[++i]));
  } else if ( !strcmp(k, "liststreams") ) {
   list_streams(&con);
  } else if ( !strcmp(k, "streaminfo") ) {
   display_stream(&con, atoi(argv[++i]));
  } else if ( !strcmp(k, "allinfo") ) {
   server_info(&con);
   printf("\n");
   server_time(&con);
   printf("\n");
   server_standards(&con);
   printf("\n");
   server_oinfo(&con, -1);
   printf("\n");
   list_clients(&con);
   printf("\n");
   list_streams(&con);

  } else if ( !strcmp(k, "kick") ) {
   t = roar_str2ot((k = argv[++i]));
   if ( t == -1 ) {
    fprintf(stderr, "Error: unknown type: %s\n", k);
    continue;
   }
   //t = atoi(argv[i++]);
   if ( roar_kick(&con, t, atoi(argv[++i])) == -1 ) {
    fprintf(stderr, "Error: can not kick %s\n", k);
   } else {
    printf("%s kicked\n", k);
   }

  } else if ( !strcmp(k, "newvirtual") ) {
   if ( newvirtual(&con, argv[i+1], argv[i+2], argv[i+3], argv[i+4], argv[i+5], argv[i+6]) == -1 ) {
    fprintf(stderr, "Error: can not create new virtual stream\n");
   } else {
    printf("virtual stream created\n");
   }
   i += 6;

  } else if ( !strcmp(k, "volume") ) {
   if ( set_mixer(&con, &i, argc, argv) == -1 ) {
    fprintf(stderr, "Error: can not set volume\n");
   } else {
    printf("volume changed\n");
   }

  } else if ( !strcmp(k, "flag") ) {
   i++;
   if ( set_flags(&con, atoi(argv[i]), ROAR_SET_FLAG, argv[i+1]) == -1 ) {
    fprintf(stderr, "Error: can not set flags\n");
   } else {
    printf("flags changed\n");
   }
   i++;
  } else if ( !strcmp(k, "unflag") ) {
   i++;
   if ( set_flags(&con, atoi(argv[i]), ROAR_RESET_FLAG, argv[i+1]) == -1 ) {
    fprintf(stderr, "Error: can not reset flags\n");
   } else {
    printf("flags changed\n");
   }
   i++;
  } else if ( !strcmp(k, "toggleflag") ) {
   i++;
   if ( set_flags(&con, atoi(argv[i]), ROAR_TOGGLE_FLAG, argv[i+1]) == -1 ) {
    fprintf(stderr, "Error: can not toggle flags\n");
   } else {
    printf("flags changed\n");
   }
   i++;
  } else if ( !strcmp(k, "protectflag") ) {
   i++;
   if ( set_flags(&con, atoi(argv[i]), ROAR_NOOP_FLAG|ROAR_PROTECT_FLAG, argv[i+1]) == -1 ) {
    fprintf(stderr, "Error: can not protect flags\n");
   } else {
    printf("flags protected\n");
   }
   i++;

  } else if ( !strcmp(k, "role") ) {
   i++;
   if ( set_role(&con, atoi(argv[i]), roar_str2role(argv[i+1])) == -1 ) {
    fprintf(stderr, "Error: can not set stream role\n");
   } else {
    printf("stream role set\n");
   }
   i++;

  } else if ( !strcmp(k, "metaset") ) {
   i++;
   if ( set_meta(&con, atoi(argv[i]), argv[i+1], argv[i+2], argv[i+3]) == -1 ) {
    fprintf(stderr, "Error: can not set meta data\n");
   } else {
    printf("meta data changed\n");
   }
   i += 3;
  } else if ( !strcmp(k, "metaget") ) {
   i++;
   if ( show_meta_type(&con, atoi(argv[i]), argv[i+1]) == -1 ) {
    fprintf(stderr, "Error: can not get meta data\n");
   }
   i++;
  } else if ( !strcmp(k, "metasave") ) {
   i++;
   if ( save_meta(&con, atoi(argv[i]), argv[i+1]) == -1 ) {
    fprintf(stderr, "Error: can not get meta data\n");
   } else {
    printf("meta data saved\n");
   }
   i++;
  } else if ( !strcmp(k, "metaload") ) {
   i++;
   if ( load_meta(&con, atoi(argv[i]), argv[i+1]) == -1 ) {
    fprintf(stderr, "Error: can not set meta data\n");
   } else {
    printf("meta data saved\n");
   }
   i++;


  } else if ( !strcmp(k, "listaiprofiles") || !strcmp(k, "listprofiles") ) {
   if ( list_aiprofiles() == -1 ) {
    fprintf(stderr, "Error: can not list profiles\n");
   }
  } else if ( !strcmp(k, "aiprofileget") || !strcmp(k, "profileget") ) {
   i++;
   if ( show_aiprofile(argv[i]) == -1 ) {
    fprintf(stderr, "Error: can not get profile data\n");
   }
  } else {
   fprintf(stderr, "Error: invalid command: %s\n", k);
  }

 }

 roar_disconnect(&con);

 return 0;
}

//ll
