//roardtmf.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2013
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include <libroardsp/libroardsp.h>

#ifdef ROAR_HAVE_LIBM
#include <math.h>

// in ms:
#if 0
#define ON_TIME  (180)
#define OFF_TIME ( 80)
#else
#define ON_TIME  (180)
#define OFF_TIME ( 80)
#endif
#define SUM_TIME (ON_TIME+OFF_TIME)

void usage (void) {
 printf("roardtmf [OPTIONS]... PHONE NUMBER\n");

 printf("\nOptions:\n\n");

 printf("  --server SERVER    - Set server hostname\n"
        "  --rate  -R RATE    - Set sample rate\n"
        "  --help             - Show this help\n"
        "  --mode-dtmf        - Use standard DTMF (default)\n"
        "  --mode-roar        - Use RoarAudio extended DTMF\n"
       );

}

int main (int argc, char * argv[]) {
 enum {
  MODE_DTMF,
  MODE_ROAR
 } mode = MODE_DTMF;
 int    rate     = ROAR_RATE_DEFAULT;
 const char * server   = NULL;
 const char * k;
 int    i;
 const char * name = "roardtmf";
 roar_vs_t * vss = NULL;
 const char * tones = NULL;
 void * buf;
 size_t len_break;
 size_t len_tone;
 size_t samples;
 size_t len;
 uint16_t c;
 int err;

 for (i = 1; i < argc; i++) {
  k = argv[i];

  if ( !strcmp(k, "--server") || !strcmp(k, "-s") ) {
   server = argv[++i];
  } else if ( !strcmp(k, "--rate") || !strcmp(k, "-r") || !strcmp(k, "-R") ) {
   rate = roar_str2rate(argv[++i]);
  } else if ( !strcmp(k, "--mode-dtmf") ) {
   mode = MODE_DTMF;
  } else if ( !strcmp(k, "--mode-roar") ) {
   mode = MODE_ROAR;
  } else if ( !strcmp(k, "--help") ) {
   usage();
   return 0;
  } else if ( tones == NULL ) {
   tones = argv[i++];
  } else {
   fprintf(stderr, "Error: unknown argument: %s\n", k);
   usage();
   return 1;
  }
 }

 if ( tones == NULL ) {
  return 1;
 }

 len_break = roar_dtmf_mus2samples(OFF_TIME * 1000, rate);
 len_tone  = roar_dtmf_mus2samples(ON_TIME  * 1000, rate);
 samples = len_break + len_tone;

/*
 printf("samples=%llu\n", (long long unsigned int)samples);
 return 0;
*/

 if ( (buf = roar_mm_malloc((len = 16*samples/8))) == NULL )
  return 4;

 if ( (vss = roar_vs_new_playback(server, name, rate, 1, ROAR_CODEC_DEFAULT, 16, &err)) == NULL ) {
  fprintf(stderr, "Error: can not start playback: %s\n", roar_vs_strerr(err));
  roar_mm_free(buf);
  return 1;
 }

 while ( (c = *tones) != 0 ) {
  tones++;

  if ( mode == MODE_DTMF ) {
   if ( c == '+' || c == '.' || c == '_' )
    continue;
  } else if ( mode == MODE_ROAR ) {
   c = ROAR_DTMF_CHAR_ROAR(c);
  }

  if ( mode == MODE_DTMF && c == ' ' ) {
   roar_dtmf_break(buf, len_tone, rate, ROAR_DTMF_OPTIONS_NONE);
  } else {
   if ( roar_dtmf_tone(buf, len_tone, rate, ROAR_DTMF_OPTIONS_NONE, c) == -1 ) {
    roar_mm_free(buf);
    return 5;
   }
  }
  roar_dtmf_tone(buf+len_tone*sizeof(int16_t), len_break, rate, ROAR_DTMF_OPTIONS_NONE,
                 mode == MODE_DTMF ? ROAR_DTMF_CHAR_BREAK : ROAR_DTMF_CHAR_NOOP);
  if ( roar_vs_write(vss, buf, len, &err) != (ssize_t)len ) {
   fprintf(stderr, "Error: can not write data: %s\n", roar_vs_strerr(err));
   roar_mm_free(buf);
   return 5;
  }
 }

 roar_vs_sync(vss, ROAR_VS_WAIT, NULL);

 roar_vs_close(vss, ROAR_VS_FALSE, 0);

 roar_mm_free(buf);

 return 0;
}

#else
int main (void) {
 fprintf(stderr, "Error: No Math library support compiled in.\n");
 return 1;
}
#endif

//ll
