/* 
 * ------------------------------------------------------------------
 * Role PlayingDB V2.0 by Deepwoods Software
 * ------------------------------------------------------------------
 * rpg.i - RPG SWIG Interface file
 * Created by Robert Heller on Tue Jul 14 12:24:16 1998
 * ------------------------------------------------------------------
 * Modification History: 
 * $Log: rpg.i,v $
 * Revision 1.15  2000/02/11 00:32:01  heller
 * MS-Windows fixes...
 *
 * Revision 1.14  1999/07/13 20:33:14  heller
 * Minor Doc fixes.
 *
 * Revision 1.13  1999/07/13 00:57:13  heller
 * Documentation updates: spelling, grammer, punctuation, etc.
 *
 * Revision 1.12  1999/01/09 03:35:01  heller
 * Fix TOC problems...
 *
 * Revision 1.11  1999/01/03 18:59:29  heller
 * Toc & Index
 *
 * Revision 1.10  1999/01/03 06:35:15  heller
 * Fix minor probs with Bibliography
 *
 * Revision 1.9  1999/01/02 20:42:32  heller
 * Add in Bibliography hacks
 *
 * Revision 1.8  1998/12/27 22:07:06  heller
 * Fix spelling errors
 *
 * Revision 1.7  1998/12/27 04:11:49  heller
 * Updated documentation: added intro and adjusted things to fit.
 *
 * Revision 1.6  1998/12/21 14:58:04  heller
 * Added image field to "Item"
 *
 * Revision 1.5  1998/12/18 21:07:11  heller
 * Changed "NextSpaceFilename" to NextSpaceIndexString"
 *
 * Revision 1.4  1998/11/27 20:45:12  heller
 * Fix small spelling error
 *
 * Revision 1.3  1998/08/23 18:36:44  heller
 * Added scale factor argument to MakeGraphicCommannd member function.
 *
 * Revision 1.2  1998/08/23 17:04:43  heller
 * Added in a psuedo-member function to the Space class to generate Tcl/Tk
 * graphic commands to draw a basic space.
 *
 * Revision 1.1  1998/08/23 14:47:38  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Role Playing DB -- A database package that creates and maintains
 * 		       a database of RPG characters, monsters, treasures,
 * 		       spells, and playing environments.
 * 
 *     Copyright (C) 1995,1998-1999  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */
%title "Role Playing DB V2.0 -- Tcl Interface"
%module Rpg
%{
#include <iostream.h>
#include <string.h>
#include <Record.h>
#include <Dice.h>
#include <Character.h>
#include <Monster.h>
#include <Spells.h>
#include <Dressings.h>
#include <Space.h>
#include <math.h>

static char rcsid[] = "$Id: rpg.i,v 1.15 2000/02/11 00:32:01 heller Rel $";
#define TclRecord Record
%}

%text %{\tableofcontents%  %}

%section "Copyright Info"

%text %{
\index{Copyright Info}\typeout {Generated from "$Id: rpg.i,v 1.15 2000/02/11 00:32:01 heller Rel $".}       Role Playing DB -- 
		\parbox[t]{3.5in}{A database package that creates and maintains
 		       a database of RPG characters, monsters, treasures,
 		       spells, and playing environments.}
 
     Copyright (C) 1995,1998-1999  \parbox[t]{3.5in}{Robert Heller D/B/A Deepwoods Software \\
 			51 Locke Hill Road \\
 			Wendell, MA 01379-9728 }
 
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
 
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
 
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
%}

%section "Introduction"

%text %{
This manual describes the Tcl interface to the C++ class library used
by the Role Playing DB.  This consists of a collection of classes for
the various base data structures used to represent the various
informational objects used in Role Playing Games.  These informational
objects consist of {\it Characters}, {\it Monsters}, {\it ``Magic''
Spells}, {\it (Dungeon) Dressings}, and {\it ``Spaces''}.  All of these
data objects are built upon a common low-level data object, a {\tt
Record}.  Also included is a specialized random number generator that
emulates one or more dice, dice being commonly used to implement random
{\tt chance} or ``fate'' in most Role Playing Games\footnote{Some use
shuffled decks of cards and others use a mix.}.

Although TSR's {\it Advanced Dungeons \& Dragons} \cite{Gygax77},
\cite{Gygax78}, and \cite{Gygax79} were used as a reference in the
design of these data structures, they should be generic enough to be
usable for any other Role Playing Game system.
%}

%include typemaps.i

%init %{
	Tcl_PkgProvide(interp,"Rpg","2.0");
%}

%section "Dice Class"

%apply unsigned int *OUTPUT {unsigned int& ns,unsigned int& nd};
%text %{
  \index{Dice Class}Dice Class -- Just a random number generator using a dice model.
%}

class Dice {
public:
	Dice(unsigned int ns = 0,unsigned int nd = 1);//Constructor, create 1 or more dice.
	~Dice();//Destructor
	unsigned int Roll();//Roll them bones...
	bool TypeOfDice(unsigned int& ns,unsigned int& nd);//Return the type of dice.
};


%section "Record Class"

typedef int size_t
%text %{
 \index{Record Class}Record structure.

  Core resident record - it has a size and a chunk of memory (the
  record itself). This structure is lifted from the Home Librarian
  package.  A very re-usable piece of code.

   Lots of fun.  I wish C++ had a garbage collector...
%}

struct Record {
	Record ();// Constructor: preallocated buffer.
	~Record (); // Destructor: free up allocated memory.
};

%typemap(tcl,out) TclRecord * {
	char *p = $source->buffer, *p1;
	int bytesleft = $source->size;
	Tcl_ResetResult(interp);
	while (bytesleft > 2)
	{
		Tcl_AppendElement(interp,p);
		p1 = p + strlen(p) + 1;
		bytesleft -= p1 - p;
		p = p1;
		if (*p == '\n') bytesleft = 0;
	}
}

%typemap(tcl,in) TclRecord * {
	int tempc,i, len;
	char **tempv;
	if (Tcl_SplitList(interp,$source,&tempc,&tempv) == TCL_ERROR) return TCL_ERROR;
	len = 2;
	for (i = 0; i < tempc; i++) len += strlen(tempv[i]) + 1;
	Record *tr = new Record(len);
	char *p = tr->buffer;
	for (i = 0; i < tempc; i++)
	{
		strcpy(p,tempv[i]);
		p += strlen(tempv[i]) + 1;
	}
	*p++ = '\n';
	*p++ = '\0';
	Tcl_Free((char*)tempv);
	$target = tr;
}

%typemap(tcl,freearg) TclRecord * {delete $source;}

%typemap(tcl,out) int MyTcl_Result {
 return $source;
}

%apply int MyTcl_Result { int Record_ReadRecord };
%apply int MyTcl_Result { int Record_WriteRecord };


%addmethods Record {
	TclRecord * ReturnRecord () {return self;} // Return the Record in printable form.
	void SetRecord(TclRecord * source) { *self = *source;} // Set the Record from its printable form.
	int ReadRecord(Tcl_Interp *interp,char *filename)
	{
  	Tcl_Channel inch = Tcl_OpenFileChannel(interp,filename,"r",0);
  	if (inch == NULL) return TCL_ERROR;
  	int s = Tcl_SetChannelOption(interp,inch,"-translation","binary");
  	if (s != TCL_OK)
	{
		Tcl_Close(interp,inch);
		return TCL_ERROR;
	}
	int eof = Tcl_Seek(inch,0,SEEK_END);
	if (eof < 0)
	{
		int err = Tcl_GetErrno();
		Tcl_Close(interp,inch);
		Tcl_AppendResult(interp,filename,": Could not get EOF pos: ",
				strerror(err),NULL);
		return TCL_ERROR;
	}
	self->NewBuffer(eof);
	Tcl_Seek(inch,0,SEEK_SET);
	int Nread = Tcl_Read(inch,self->buffer,eof);
	if (Nread < 0)
	{
		int err = Tcl_GetErrno();
		Tcl_Close(interp,inch);
		Tcl_AppendResult(interp,filename,": Could not read: ",
				strerror(err),NULL);
		return TCL_ERROR;
	}
	Tcl_Close(interp,inch);
  	return TCL_OK;
  } // Load Record from a file.
  int WriteRecord (Tcl_Interp *interp,char *filename)
  {
  	Tcl_Channel ouch = Tcl_OpenFileChannel(interp,filename,"w",0640);
  	if (ouch == NULL) return TCL_ERROR;
  	int s = Tcl_SetChannelOption(interp,ouch,"-translation","binary");
  	if (s != TCL_OK)
	{
		Tcl_Close(interp,ouch);
		return TCL_ERROR;
	}
	int Nwrite = Tcl_Write(ouch,self->buffer,self->size);
	if (Nwrite < (int)(self->size))
	{
		int err = Tcl_GetErrno();
		Tcl_Close(interp,ouch);
		Tcl_AppendResult(interp,filename,": Could not write: ",
				strerror(err),NULL);
		return TCL_ERROR;
	}
	Tcl_Close(interp,ouch);
	return TCL_OK;
  } // Write Record to file.
}


%section "Character Class"

%text %{
  \index{Character Class}Basic character class.  Contains all of the information needed to describe
  a player or non player character.
%}

%native(NewCharacter) int NewCharacter(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /* 
This is the interface to the constructor for {\tt Character} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt Character} function with a {\tt -this} 
argument to do that: \verb+[Character -this [NewCharacter options...]]+.

\end{minipage}

\makebox[0.5in]{}\begin{minipage}[t]{6in}

The defined options for this function are:

\begin{tabular}{lllp{2in}}
flag & type & default & explanation \\
-strength & integer & 0 & Set the initial character strength value \\
-intelligence & integer & 0 & Set the initial character intelligence value \\
-wisdom & integer & 0 & Set the initial character wisdom value \\
-dexterity & integer & 0 &Set the initial character dexterity value \\
-constitution & integer & 0 &Set the initial character constitution value \\
-charisma & integer & 0 &Set the initial character charisma value \\
-exceptionalstrength & integer & 0 &Set the initial character exceptional strength value \\
-level & integer & 1 &Set the initial character level \\
-hitdice & integer & 6 & Set the initial character hit die type (number of sides) \\
-numhitdice & integer & 1 &Set the initial character number of hit dice \\
-maxhitdice & integer & 1 &Set the initial character maximum number of hit dice \\
-name & string & ``'' &Set the initial character name \\
-player & string & ``'' &Set the initial character player name \\
-race & string & ``Human'' &Set the initial character race \\
-characterclass & string & ``npc'' &Set the initial character character class \\
-alignment & string & ``'' &Set the initial character alignment \\
-sex & string & ``'' &Set the initial character sex \\
-age & integer & 0 &Set the initial character age \\
-commentary & string & ``'' &Set the initial character commentary \\
-image & string & ``'' &Set the initial character image (GIF file) \\
\end{tabular} */


class Character {
public:
	Character(); // Base Constructor.
	~Character(); // Destructor: free up allocated memory.
	int HitPoints() const;// Return hit points.
	int Age() const;// Return age.
	int Strength() const;// Return strength.
	int Intelligence() const;// Return intelligence.
	int Wisdom() const;// Return Wisdom.
	int Dexterity() const;// Return dexterity.
	int Constitution() const;// Return constitution
	int Charisma() const;// Return charisma.
	int ExceptionalStrength() const;// Return exceptional strength.
	int Level() const;// Return level.
	int ExperiencePoints () const; // Return experience points.
	int Gold () const; // Return gold.
	const char *Name() const;// Return name.
	const char *Player() const;// Return player.
	const char *Race() const;// Return race.
	const char *CharacterClass() const;// Return character class.
	const char *Alignment() const;// Return alignment.
	const char *Sex() const;// Return sex.
	const char *Comments() const;// Return comments.
	const char *Image() const; // Return image.
	int SetAge(int newage);// Set age.
	int SetStrength(int newS);// Set strength.
	int SetIntelligence(int newI);// Set intelligence.
	int SetWisdom(int newW);// Set wisdom.
	int SetDexterity(int newD);// Set dexterity.
	int SetConstitution(int newC);// Set constitution.
	int SetCharisma(int newCh);// Set charisma.
	int SetExceptionalStrength(int newES);// Set exceptional strength.
	int SetExperiencePoints(int newEP); // Set experience points.
	int SetGold(int newGP); // Set gold.
	const char *SetName(const char * newN);// Set name.
	const char *SetPlayer(const char * newP);// Set player.
	const char *SetRace(const char *newR);// Set race.
	const char *SetCharacterClass(const char *newCh);// Set character class.
	const char *SetAlignment(const char *newA);// Set alignment.
	const char *SetSex(const char *newS);// Set sex.
	const char *SetComments(const char *newC);// Set comments.
	const char *SetImage(const char *newI);//Set image.
	void AdvanceLevel();// Advance character's level.
	void UpdateFromRecord (const Record &rec); // Update character from Record.
};

%addmethods Character {
	Record * RawData() {return new Record((Record)(*self));}	// Type conversion: convert to a Record.
}

%{

static int NewCharacter(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	Character * _result;
	int s = 0,i = 0,w = 0,d = 0,c = 0,ch = 0,es = 0,l = 1,hd = 6,nd = 1,
	    md = 1, ag = 0;
	char *n = "", *p = "", *r = "Human", *chc = "npc", *align = "",
	     *sex = "", *com = "", *im = "";
        int iac = 1;

        Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-strength") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&s) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-intelligence") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&i) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-wisdom") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&w) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-dexterity") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&d) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-constitution") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&c) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-charisma") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&ch) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-exceptionalstrength") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&es) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-level") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&l) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-hitdice") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&hd) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-numhitdice") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&nd) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-maxhitdice") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&md) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-name") == 0)
		{
			n = argv[iac+1];
		} else if (strcmp(argv[iac],"-player") == 0)
		{
			p = argv[iac+1];
		} else if (strcmp(argv[iac],"-race") == 0)
		{
			r = argv[iac+1];
		} else if (strcmp(argv[iac],"-characterclass") == 0)
		{
			chc = argv[iac+1];
		} else if (strcmp(argv[iac],"-alignment") == 0)
		{
			align = argv[iac+1];
		} else if (strcmp(argv[iac],"-sex") == 0)
		{
			sex = argv[iac+1];
		} else if (strcmp(argv[iac],"-age") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&ag) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-commentary") == 0)
		{
			com = argv[iac+1];
		} else if (strcmp(argv[iac],"-image") == 0)
		{
			im = argv[iac+1];
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -strength ",
					 "-intelligence -wisdom -dexterity ",
					 "-constitution -charisma ",
					 "-exceptionalstrength -level ",
					 "-hitdice -numhitdice -maxhitdice ",
					 "-name -player -race -characterclass ",
					 "-alignment -sex -age -commentary -image",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	_result = new Character(s,i,w,d,c,ch,es,l,hd,nd,md,n,p,r,chc,align,sex,
				ag,com,im);
	SWIG_MakePtr(interp->result, (void *) _result,"_Character_p");
	return TCL_OK;
}


%}

%section "Monster Class"

%text %{
\index{Monster Class}  Basic monster class. Contains all of the information needed to describe a
  monster.
%}

%apply int *OUTPUT {int& L,int& H};

%native(NewMonster) int NewMonster(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /* 
This is the interface to the constructor for {\tt Monster} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt Monster} function with a {\tt -this} 
argument to do that: \verb+[Monster -this [NewMonster options...]]+.

The defined options for this function are:

\begin{tabular}{lllp{2in}} 
flag & type & default & explanation \\
-hitpoints & integer & 0 & The monster's hitpoints (only if -hittype is
Monster::Points). \\
-hitdie & integer & 8 & The monster's hit die (only if -hittype is Monster::Dice). \\ 
-numhitdice & integer & 1 & The number of hit dice (only if -hittype is Monster::Dice). \\ 
-hitadjust & integer & 0 & The hit adjustment (only if -hittype is Monster::Dice). \\ 
-armorclass & integer & 0 & The monster's armor class \\ 
-landspeed & integer & 0 & The monster's land speed \\ 
-flyingspeed & integer & 0 & The monster's flying speed \\ 
-swimmingspeed & integer & 0 & The monster's swimming speed \\ 
-burrowingspeed & integer & 0 & The monster's burrowing speed \\ 
-webspeed & integer & 0 & The monster's web speed \\ 
-percentlair & integer & 0 & The monster's percent in lair \\ 
-numattacks & integer & 0 & The monster's number of attacks \\
-mindamage & integer & 0 & The monster's minimum damage per attack \\
-maxdamage & integer & 0 & The monster's maximum damage per attack \\
-magicres & integer & 0 & The monster's magical resistance \\
-minnumappear & integer & 0 & The minimum number appearing \\
-maxnumappear & integer & 0 & The maximum number appearing \\
-intelligence & enum & - & The monster's intelligence rating \\
-frequency & enum & - & The monster's frequency of occurrence \\
\end{tabular}

\end{minipage}

\makebox[0.5in]{}\begin{minipage}[t]{6in}

\begin{tabular}{lllp{2in}}
flag & type & default & explanation \\
-hittype & enum & - & The monster's hit type \\
-size & double & 0.0 & The monster's size \\
-name & string & ``'' & The name or type of monster \\
-alignment & string & ``'' & The monster's alignment \\
-treasure & string & ``'' & The monster's treasure type \\
-specialatt & string & ``'' & The monster's special attack modes \\
-specialdef & string & ``'' & The monster's special defensive modes \\
-psionics & string & ``'' & The monster's psionic abilities \\
-commentary & string & ``'' & Additional commentary \\
-image & string & ``'' & Name of image file \\
\end{tabular} */

class Monster {
public:
	Monster(); // Base Constructor.
	~Monster(); // Destructor.
	const char *Name() const; // Return Name/Type.
	const char *Alignment() const; // Return the alignment.
	const char *TreasureType() const; // Return the treasure type.
	const char *SpecialAttacks() const; // Return the special attacks.
	const char *SpecialDefences() const; // Return the special defenses.
	const char *Psionics() const; // Return the psionics.
	const char *Comments() const; // Return the commentary.
	const char *Image() const; // Return the image.
	int HitDieSides() const; // Return the hit die sides.
	int NumHitDice() const; // Return the number of hit dice.
	int HitAdjustment() const; // Return the hit dice adjustment.
	int ArmorClass() const; // Return the armor class.
	int LandSpeed() const; // Return the speed on land.
	int FlyingSpeed() const; // Return the speed in the air (flying).
	int SwimmingSpeed() const; // Return the speed in the water (swimming).
	int BurrowingSpeed() const; // Return the speed in the earth (burrowing).
	int WebSpeed() const; // Return the speed in web.
	int PercentInLair() const; // Return the percent in lair.
	int NumberOfAttacks() const; // Return the number of attacks.
	int MagicalResistance() const; // Return the magical resistance.
	int HitPoints() const; // Return the number of hit points.
	void DamagePerAttack(int& L,int& H) const; // Return the damage per attack.
	void NumberAppearing(int& L,int& H) const; // Return the number appearing.
	double Size() const; // Return the size.
	void UpdateFromRecord (const Record &rec);// Update monster from Record.
};

%addmethods Monster {
	Record * RawData() {return new Record((Record)(*self));}	// Type conversion: convert to a Record.
	const char * Intelligence()  
	{
		switch (self->Intelligence())
		{
			case Monster::Non: return "Monster::Non";break;
			case Monster::Animal: return "Monster::Animal";break;
			case Monster::Semi: return "Monster::Semi";break;
			case Monster::Low: return "Monster::Low";break;
			case Monster::Average: return "Monster::Average";break;
			case Monster::Very: return "Monster::Very";break;
			case Monster::Highly: return "Monster::Highly";break;
			case Monster::Exceptionally: return "Monster::Exceptionally";break;
			case Monster::Genius: return "Monster::Genius";break;
			case Monster::SupraGenius: return "Monster::SupraGenius";break;
			case Monster::Godlike: return "Monster::Godlike";break;
			case Monster::BogusIntelligence: return "Monster::BogusIntelligence";break;
		}
		return NULL;
	}// Return the intelligence rating.
	const char * Frequency()
	{
		switch (self->Frequency())
		{
			case Monster::Unique: return "Monster::Unique";break;
			case Monster::VeryRare: return "Monster::VeryRare";break;
			case Monster::Rare: return "Monster::Rare";break;
			case Monster::Uncommon: return "Monster::Uncommon";break;
			case Monster::Common: return "Monster::Common";break;
			case Monster::BogusFrequency: return "Monster::BogusFrequency";break;
		}
		return NULL;
	}  // Return the frequency.
	const char * Hittype()
	{
		switch (self->Hittype())
		{
			case Monster::Points: return "Monster::Points";break;
			case Monster::Dice: return "Monster::Dice";break;
			case Monster::BogusHitType: return "Monster::BogusHitType";break;
		}
		return NULL;
	}  // Return the hit type.
}

%{

static int NewMonster(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	Monster * _result = NULL;
	int hp = 0, hd = 8, nd = 1, had = 0, ac = 0, m = 0, mf = 0, ms = 0,
	    mb = 0, mw = 0, pl = 0, na = 0, daL = 0, daH = 0, mr = 0, noaL = 0,
	    noaH = 0;
	Monster::IntelligenceRating i = Monster::BogusIntelligence;
	Monster::FreqType f = Monster::BogusFrequency;
	double s = 0;
	char *nm = "", *al = "", *tt = "", *sa = "", *sd = "", *ps = "", 
	     *com = "", *im = "";
	Monster::HitType h = Monster::Dice;
        int iac = 1;

	Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-hitpoints") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&hp) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-hitdie") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&hd) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-numhitdice") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&nd) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-hitadjust") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&had) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-armorclass") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&ac) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-landspeed") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&m) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-flyingspeed") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&mf) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-swimmingspeed") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&ms) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-burrowingspeed") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&mb) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-webspeed") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&mw) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-percentlair") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&pl) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-numattacks") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&na) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-mindamage") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&daL) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-maxdamage") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&daH) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-magicres") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&mr) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-minnumappear") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&noaL) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-maxnumappear") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&noaH) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-intelligence") == 0)
		{
			if (strcmp(argv[iac+1],"Monster::Non") == 0)
			{
				i = Monster::Non;
			} else if (strcmp(argv[iac+1],"Monster::Animal") == 0) 
			{
				i = Monster::Animal;
			} else if (strcmp(argv[iac+1],"Monster::Semi") == 0) 
			{
				i = Monster::Semi;
			} else if (strcmp(argv[iac+1],"Monster::Low") == 0) 
			{
				i = Monster::Low;
			} else if (strcmp(argv[iac+1],"Monster::Average") == 0) 
			{
				i = Monster::Average;
			} else if (strcmp(argv[iac+1],"Monster::Very") == 0) 
			{
				i = Monster::Very;
			} else if (strcmp(argv[iac+1],"Monster::Highly") == 0) 
			{
				i = Monster::Highly;
			} else if (strcmp(argv[iac+1],"Monster::Exceptionally") == 0) 
			{
				i = Monster::Exceptionally;
			} else if (strcmp(argv[iac+1],"Monster::Genius") == 0) 
			{
				i = Monster::Genius;
			} else if (strcmp(argv[iac+1],"Monster::SupraGenius") == 0) 
			{
				i = Monster::SupraGenius;
			} else if (strcmp(argv[iac+1],"Monster::Godlike") == 0) 
			{
				i = Monster::Godlike;
			} else
			{
				Tcl_AppendResult(interp,
						 "Not a valid intelligence rating: ",
						 argv[iac+1],", valid values are: ",
						 "Monster::Non, Monster::Animal, ",
						 "Monster::Semi, Monster::Low, ",
						 "Monster::Average, Monster::Very, ",
						 "Monster::Highly, Monster::Exceptionally, ",
						 "Monster::Genius, Monster::SupraGenius, ",
						 "Monster::Godlike",NULL);
				return TCL_ERROR;
			}
		} else if (strcmp(argv[iac],"-frequency") == 0)
		{
			if (strcmp(argv[iac+1],"Monster::Unique") == 0)
			{
				f = Monster::Unique;
			} else if (strcmp(argv[iac+1],"Monster::VeryRare") == 0)
			{
				f = Monster::VeryRare;
			} else if (strcmp(argv[iac+1],"Monster::Rare") == 0)
			{
				f = Monster::Rare;
			} else if (strcmp(argv[iac+1],"Monster::Uncommon") == 0)
			{
				f = Monster::Uncommon;
			} else if (strcmp(argv[iac+1],"Monster::Common") == 0)
			{
				f = Monster::Common;
			} else
			{
				Tcl_AppendResult(interp,
						 "Not a valid frequency type: ",
						 argv[iac+1],", valid values are: ",
						 "Monster::Unique, Monster::VeryRare, ",
						 "Monster::Rare, Monster::Uncommon, ",
						 "Monster::Common",NULL);
				return TCL_ERROR;
			}	
		} else if (strcmp(argv[iac],"-hittype") == 0)
		{
			if (strcmp(argv[iac+1],"Monster::Points") == 0)
			{
				h = Monster::Points;
			} else if (strcmp(argv[iac+1],"Monster::Dice") == 0)
			{
				h = Monster::Dice;
			} else
			{
				Tcl_AppendResult(interp,
						 "Not a valid hit type: ",
						 argv[iac+1],", valid values are: ",
						 "Monster::Points or Monster::Dice",
						 NULL);
				return TCL_ERROR;
			}
		} else if (strcmp(argv[iac],"-size") == 0)
		{
			if (Tcl_GetDouble(interp,argv[iac+1],&s) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-name") == 0)
		{
			nm = argv[iac+1];
		} else if (strcmp(argv[iac],"-alignment") == 0)
		{
			al = argv[iac+1];
		} else if (strcmp(argv[iac],"-treasure") == 0)
		{
			tt = argv[iac+1];
		} else if (strcmp(argv[iac],"-specialatt") == 0)
		{
			sa = argv[iac+1];
		} else if (strcmp(argv[iac],"-specialdef") == 0)
		{
			sd = argv[iac+1];
		} else if (strcmp(argv[iac],"-psionics") == 0)
		{
			ps = argv[iac+1];
		} else if (strcmp(argv[iac],"-commentary") == 0)
		{
			com = argv[iac+1];
		} else if (strcmp(argv[iac],"-image") == 0)
		{
			im = argv[iac+1];
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -hitpoints ",
					 "-hitdie -numhitdice -hitadjust ",
					 "-armorclass -landspeed -flyingspeed", 
					 "-swimmingspeed -burrowingspeed ",
					 "-webspeed -percentlair -numattacks ",
					 "-mindamage -maxdamage -magicres ",
					 "-minnumappear -maxnumappear ",
					 "-intelligence -frequency -hittype ",
					 "-size -name -alignment -treasure ",
					 "-specialatt -specialdef -psionics ",
					 "-commentary -image",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	if (i == Monster::BogusIntelligence)
	{
		Tcl_AppendResult(interp,"Missing (required) option: -intelligence",NULL);
		return TCL_ERROR;
	}
	if (f == Monster::BogusFrequency)
	{
		Tcl_AppendResult(interp,"Missing (required) option: -frequency",NULL);
		return TCL_ERROR;
	}
	if (h == Monster::BogusHitType)
	{
		Tcl_AppendResult(interp,"Missing (required) option: -hittype",NULL);
		return TCL_ERROR;
	} else if (h == Monster::Points)
	{
		_result = new Monster(hp,ac,m,mf,ms,mb,mw,pl,na,daL,daH,mr,
				      noaL,noaH,i,f,s,nm,al,tt,sa,sd,ps,com,im);
	} else if (h == Monster::Dice)
	{
		_result = new Monster(hd,nd,had,ac,m,mf,ms,mb,mw,pl,na,daL,daH,
				      mr,noaL,noaH,i,f,s,nm,al,tt,sa,sd,ps,com,im);
	}
	SWIG_MakePtr(interp->result, (void *) _result,"_Monster_p");
	return TCL_OK;
}


%}

%text %{
 Monster ``instance''.  Used to implement a working instance of a monster. 
%}

class MonsterInstance {
public:
	MonsterInstance(const Monster *iOf, const char *iN = ""); // Constructor.
	~MonsterInstance(); // Destructor.
	const char *InstanceName() const; // Return the name.
	int InstanceHitPoints() const; // Return the hit points.
	int UpdateInstanceHitPoints(int adj); // Adjust the hit points.
	const Monster *InstanceOf () const; // Return base monster object.
};


%section "Spell Classes"

%text %{
  \index{Spell Class}Basic spell class.  Contains all of the information needed to describe a
  spell.
%}

%native(NewSpell) int NewSpell(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /* 
This is the interface to the constructor for {\tt Spell} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt Spell} function with a {\tt -this} 
argument to do that: \verb+[Spell -this [NewSpell options...]]+.

\end{minipage}

\makebox[0.5in]{}\begin{minipage}[t]{6in}

The defined options for this function are:

\begin{tabular}{lllp{2in}} 
flag & type & default & explanation \\
-class & string & ``'' & Character class that can use the spell. \\
-name & string & ``'' & Name of the spell. \\
-type & string & ``'' & Type of spell. \\
-description & string & ``'' & Description of spell. \\
-area & string & ``'' & Area effected by spell. \\
-casttime & string & ``'' & Casting time. \\
-duration & string & ``'' & Duration of spell. \\
-savethrow & string & ``'' & Saving throw (if any). \\
-level & integer & 0 & Spell level. \\
-range & integer & 0 & Spell range. \\
-reversible & boolean & false & Spell reversibility. \\
-verbal & boolean & false & Spell has verbal component. \\
-somatic & boolean & false & Spell has somatic component. \\
-material & boolean & false & Spell has material component. \\
\end{tabular} */

class Spell {
public:	
	Spell(); // Base constructor.
	~Spell(); // Destructor.
	void UpdateFromRecord (const Record &rec);// Update Spell from Record.
	const char *SpellClass() const;// Return the spell's class.
	const char *Name() const;// Return the spell's name.
	const char *SpellType() const;// Return the spell's type.
	const char *Description() const;// Return the spell's description.
	const char *AreaOfEffect() const;// Return the spell's area of effect.
	const char *CastingTime() const;// Return the spell's casting time.
	const char *Duration() const;// Return the spell's Duration.
	const char *SavingThrow() const;// Return the spell's saving Throw.
	int Level() const;// Return the spell's level.
	int Range() const;// Return the spell's range.
	bool ReversibleP() const;// Return spell's reversibility flag.
	bool VerbalP() const;// Return spell's verbal component flag.
	bool SomaticP() const;// Return spell's somatic component flag.
	bool MaterialP() const;// Return spell's material component flag.
};

%addmethods Spell {
	Record * RawData() {return new Record((Record)(*self));}	// Type conversion: convert to a Record.
}

%{

static int NewSpell(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	Spell * _result = NULL;
	int l = 0, r = 0;
	int rev = 0, verb = 0, som = 0, mat = 0;
	char *spc = "", *nm = "", *spt = "", *desc = "", *a = "", *ct = "", 
	     *dt = "", *st = "";
        int iac = 1;

	Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-level") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&l) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-range") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&r) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-reversible") == 0)
		{
			if (Tcl_GetBoolean(interp,argv[iac+1],&rev) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-verbal") == 0)
		{
			if (Tcl_GetBoolean(interp,argv[iac+1],&verb) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-somatic") == 0)
		{
			if (Tcl_GetBoolean(interp,argv[iac+1],&som) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-material") == 0)
		{
			if (Tcl_GetBoolean(interp,argv[iac+1],&mat) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-name") == 0)
		{
			nm = argv[iac+1];
		} else if (strcmp(argv[iac],"-class") == 0)
		{
			spc = argv[iac+1];
		} else if (strcmp(argv[iac],"-type") == 0)
		{
			spt = argv[iac+1];
		} else if (strcmp(argv[iac],"-description") == 0)
		{
			desc = argv[iac+1];
		} else if (strcmp(argv[iac],"-area") == 0)
		{
			a = argv[iac+1];
		} else if (strcmp(argv[iac],"-casttime") == 0)
		{
			ct = argv[iac+1];
		} else if (strcmp(argv[iac],"-duration") == 0)
		{
			dt = argv[iac+1];
		} else if (strcmp(argv[iac],"-savethrow") == 0)
		{
			st = argv[iac+1];
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -class ",
					 "-name -type -description ",
					 "-area -casttime -duration ", 
					 "-savethrow -level ",
					 "-range -reversible -verbal ",
					 "-somatic -material ",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	_result = new Spell(spc,nm,spt,desc,a,ct,dt,st,l,r,(bool)(rev==1),
			    (bool)(verb==1),(bool)(som==1),(bool)(mat==1));
	SWIG_MakePtr(interp->result, (void *) _result,"_Spell_p");
	return TCL_OK;
}

%}

%section "Dressings Classes"

%text %{
  Various dressing classes.
%}

%subsection "Treasure Class"

%text %{
\index{Treasure Class}Treasures are the things the characters are trying to get and the
 monsters are guarding.
%}

%native(NewTreasure) int NewTreasure(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /*
This is the interface to the constructor for {\tt Treasure} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt Treasure} function with a {\tt -this} 
argument to do that: \verb+[Treasure -this [NewTreasure options...]]+.

\end{minipage}

\makebox[0.5in]{}\begin{minipage}[t]{6in}

The defined options for this function are:

\begin{tabular}{lllp{2in}} 
flag & type & default & explanation \\
-name & string & ``'' & Name of the treasure. \\
-description & string & ``'' & Description of treasure. \\
-image & string & ``'' & Image file (GIF) of treasure. \\
-weight & integer & 0 & Weight of treasure. \\
-armorclassadj& integer & 0 &  Armor class adjustment. \\
-tohitadj& integer & 0 &  To hit adjustment. \\
-damageadj& integer & 0 &  Damage adjustment. \\
-magicalresistanceadj& integer & 0 &  Magical resistance adjustment. \\
-damageprotectionadj& integer & 0 &  Damage protection adjustment. \\
-strengthadj& integer & 0 &  Strength adjustment. \\
-intelligenceadj& integer & 0 &  Intelligence adjustment. \\
-wisdomadj& integer & 0 &  Wisdom adjustment. \\
-dexterityadj& integer & 0 &  Dexterity adjustment. \\
-constitutionadj& integer & 0 &  Constitution adjustment. \\
-charismaadj& integer & 0 &  Charisma adjustment. \\
-groundmovementadj& integer & 0 &  Ground Movement adjustment. \\
-flyingadj& integer & 0 &  Flying adjustment. \\
-swimmingadj& integer & 0 &  Swimming adjustment. \\
-value& integer & 0 &  Value. \\
\end{tabular} */

class Treasure {
public:
	Treasure(); // Base constructor.
	~Treasure(); // Destructor.
	void UpdateFromRecord (const Record &rec); // Update Treasure from Record.
	const char *Name() const; // Return name.
	const char *Description() const; // Return description.
	const char *Image() const; // Return image.
	int Weight() const; // Return weight.     
	int ArmorClassAdj() const; // Return armor class adjustment.
	int ToHitAdj() const; // Return to hit adjustment.
	int DamageAdj() const; // Return damage adjustment.
	int MagicalResistanceAdj() const; // Return magical resistance adjustment.
	int DamageProtectionAdj() const; // Return damage protection adjustment.
	int StrengthAdj() const; // Return strength adjustment.
	int IntelligenceAdj() const; // Return intelligence adjustment.
	int WisdomAdj() const; // Return wisdom adjustment.
	int DexterityAdj() const; // Return dexterity adjustment.
	int ConstitutionAdj() const; // Return constitution adjustment.
	int CharismaAdj() const; // Return charisma adjustment.
	int GroundMovementAdj() const; // Return ground movement adjustment.
	int FlyingAdj() const; // Return flying adjustment.
	int SwimmingAdj() const; // Return swimming adjustment.
	int Value() const; // Return value.
};

%addmethods Treasure {
	Record * RawData() {return new Record((Record)(*self));}	// Type conversion: convert to a Record.
}

%{

static int NewTreasure(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	Treasure * _result = NULL;
	char *nm = "", *d = "", *i = "";
	int w = 0, aCA = 0, toHA = 0, damA = 0, magResA = 0, damProtA = 0,
	    sA = 0, iA = 0, wA = 0, dA = 0, cA = 0, chA = 0, gA = 0,
	    fA = 0, swA = 0, v = 0;
        int iac = 1;

	Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-name") == 0)
		{
			nm = argv[iac+1];
		} else if (strcmp(argv[iac],"-description") == 0)
		{
			d = argv[iac+1];
		} else if (strcmp(argv[iac],"-image") == 0)
		{
			i = argv[iac+1];
		} else if (strcmp(argv[iac],"-weight") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&w) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-armorclassadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&aCA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-tohitadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&toHA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-damageadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&damA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-magicalresistanceadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&magResA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-damageprotectionadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&damProtA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-strengthadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&sA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-intelligenceadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&iA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-wisdomadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&wA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-dexterityadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&dA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-constitutionadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&cA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-charismaadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&chA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-groundmovementadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&gA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-flyingadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&fA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-swimmingadj") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&swA) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-value") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&v) == TCL_ERROR) return TCL_ERROR;
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -name ",
					 "-description -image -weight ",
					 "-armorclassadj -tohitadj -damageadj ", 
					 "-magicalresistanceadj -damageprotectionadj ",
					 "-strengthadj -intelligenceadj -wisdomadj ",
					 "-dexterityadj -constitutionadj ",
					 "-charismaadj -groundmovementadj ",
					 "-flyingadj -swimmingadj -value",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	_result = new Treasure(nm,d,i,w,aCA,toHA,damA,magResA,damProtA,sA,iA,
			       wA,dA,cA,chA,gA,fA,swA,v);
	SWIG_MakePtr(interp->result, (void *) _result,"_Treasure_p");
	return TCL_OK;
}

%}

%subsection "TrickTrap Class"

%text %{\index{TrickTrap Class}Tricks and Traps are used to protect treasure and to generally
  keep the players on their toes.
%}

%native(NewTrickTrap) int NewTrickTrap(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /*
This is the interface to the constructor for {\tt TrickTrap} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt TrickTrap} function with a {\tt -this}
argument to do that: \verb+[TrickTrap -this [NewTrickTrap options...]]+.

The defined options for this function are:

\begin{tabular}{lllp{2in}}
flag & type & default & explanation \\
-name & string & ``'' & Name of the trick or trap. \\
-type & string & ``'' & Type of the trick or trap. \\
-description & string & ``'' & Description of trick or trap. \\
-image & string & ``'' & Image file (GIF) of trick or trap. \\
\end{tabular} */

class TrickTrap {
public:

	TrickTrap(); // Constructor.
	void UpdateFromRecord (const Record &rec);// Update Treasure from Record.
	~TrickTrap(); // Destructor.
	const char *Name() const; // Return name.
	const char *Type() const; // Return trick or trap type.
	const char *Description() const; // Return description.
	const char *Image() const; // Return image.
};


%addmethods TrickTrap {
	Record * RawData() {return new Record((Record)(*self));}	// Type conversion: convert to a Record.
}

%{

static int NewTrickTrap(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	TrickTrap * _result = NULL;
	char *nm = "", *tt = "", *d = "", *i = "";
        int iac = 1;

	Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-name") == 0)
		{
			nm = argv[iac+1];
		} else if (strcmp(argv[iac],"-type") == 0)
		{
			tt = argv[iac+1];
		} else if (strcmp(argv[iac],"-description") == 0)
		{
			d = argv[iac+1];
		} else if (strcmp(argv[iac],"-image") == 0)
		{
			i = argv[iac+1];
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -name ",
					 "-type -description -image",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	_result = new TrickTrap(nm,tt,d,i);
	SWIG_MakePtr(interp->result, (void *) _result,"_TrickTrap_p");
	return TCL_OK;
}

%}

%subsection "Dressing Class"

%text %{\index{Dressing Class}Random dungeon dressings -- random odds and ends.  Sometimes these
  things have value, but real treasures use the Treasure class.
%}

%native(NewDressing) int NewDressing(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /*
This is the interface to the constructor for {\tt Dressing} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt Dressing} function with a {\tt -this}
argument to do that: \verb+[Dressing -this [NewDressing options...]]+.

The defined options for this function are:

\begin{tabular}{lllp{2in}}
flag & type & default & explanation \\
-name & string & ``'' & Name of the dressing. \\
-value & integer & 0 & Value of the dressing. \\
-description & string & ``'' & Description of dressing. \\
-image & string & ``'' & Image file (GIF) of dressing. \\
\end{tabular} */

class Dressing {
public:
	void UpdateFromRecord (const Record &rec); // Update Dressing from Record.
	Dressing(); // Constructor.
	~Dressing(); // Destructor.
	const char *Name() const; // Return name.
	const char *Description() const; // Return description.
	const char *Image() const; // Return image.
        int Value() const; // Return value.
};
		
%addmethods Dressing {
	Record * RawData() {return new Record((Record)(*self));} // Type conversion: convert to a Record.
}

%{

static int NewDressing(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	Dressing * _result = NULL;
	char *nm = "", *d = "", *i = "";
	int v;
        int iac = 1;

	Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-name") == 0)
		{
			nm = argv[iac+1];
		} else if (strcmp(argv[iac],"-value") == 0)
		{
			if (Tcl_GetInt(interp,argv[iac+1],&v) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-description") == 0)
		{
			d = argv[iac+1];
		} else if (strcmp(argv[iac],"-image") == 0)
		{
			i = argv[iac+1];
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -name ",
					 "-value -description -image",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	_result = new Dressing(nm,d,i,v);
	SWIG_MakePtr(interp->result, (void *) _result,"_Dressing_p");
	return TCL_OK;
}

%}


%section "Space classes"

%text %{\index{Space Class}Spaces -- Squares and Hexes, where things happen. %}

%subsection "Geometric constants."

class GeoConstants {
public:
  const float Width = 100.0; // Space ``width''.
  const float HexSideLength = 57.735; // Side of a hex (computed to give a width of 100).
  const float HexPeakHeight = 28.8675; // Height of peak above vertical sides.
};

%subsection "class Exit"

%text %{\index{Exit Class} Exit points and other inter-spatial interconnection points like
  windows and staircases.
 %}

#ifdef TESTING
%native(NewExit) int NewExit(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /*
This is the interface to the constructor for {\tt Exit} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt Exit} function with a {\tt -this}
argument to do that: \verb+[Exit -this [NewExit options...]]+.

The defined options for this function are:

\begin{tabular}{lllp{2in}}
flag & type & default & explanation \\
-type & Exit::ExitType & Exit::Unspecified & Type of exit. \\
-xcenter & double & 0.0 & X coordinate of center. \\
-ycenter & double & 0.0 & Y coordinate of center. \\ 
-wallaligned & boolean & true & Wall alignment flag. \\
-description & string & ``'' & Description of exit. \\
-image  & string & ``'' & Picture of exit. \\
-next & string & ``'' & Next space filename. \\
\end{tabular} */
#endif

class Exit {
public:
	double XCenter() const; // Return center x coordinate.
	double YCenter() const; // Return center y coordinate.
	bool WallAligned() const; // Return wall alignment flag.
	const char *Description() const; // Return description.
	const char *Image() const; // Return picture of exit.
	const char *NextSpaceIndexString() const; // Return next space filename.
	Exit (); // Constructor.
	~Exit(); // Destructor.
};

%addmethods Exit {
   const char *Type() {
	Exit::ExitType t = self->Type();
	switch (t)
	{
		case Exit::Doorway: return "Exit::Doorway"; break;
		case Exit::Door: return "Exit::Door"; break;
		case Exit::LockedDoor: return "Exit::LockedDoor"; break;
		case Exit::SecretDoor: return "Exit::SecretDoor"; break;
		case Exit::OnewayDoor: return "Exit::OnewayDoor"; break;
		case Exit::TrapDoorUp: return "Exit::TrapDoorUp"; break;
		case Exit::TrapDoorDown: return "Exit::TrapDoorDown"; break;
		case Exit::StairsUp: return "Exit::StairsUp"; break;
		case Exit::StairsDown: return "Exit::StairsDown"; break;
		case Exit::WindowUnglazed: return "Exit::WindowUnglazed"; break;
		case Exit::WindowGlazed: return "Exit::WindowGlazed"; break;
		case Exit::Chimney: return "Exit::Chimney"; break;
		case Exit::Pit: return "Exit::Pit"; break;
		case Exit::Unspecified: return "Exit::Unspecified"; break;
	}
	return "Exit::Unspecified";
   } // Return type of exit.
}
		
#ifdef TESTING
%{

static int NewExit(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	Exit * _result = NULL;
	char *d = "", *i = "", *ns = "";
	int wa = 1;
	Exit::ExitType t = Exit::Unspecified;
	double x = 0.0, y = 0.0;
        int iac = 1;

	Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-type") == 0)
		{
			if (strcmp(argv[iac+1],"Exit::Doorway") == 0) {t = Exit::Doorway;}
			else if (strcmp(argv[iac+1],"Exit::Door") == 0) {t = Exit::Door;}
			else if (strcmp(argv[iac+1],"Exit::LockedDoor") == 0) {t = Exit::LockedDoor;}
			else if (strcmp(argv[iac+1],"Exit::SecretDoor") == 0) {t = Exit::SecretDoor;}
			else if (strcmp(argv[iac+1],"Exit::OnewayDoor") == 0) {t = Exit::OnewayDoor;}
			else if (strcmp(argv[iac+1],"Exit::TrapDoorUp") == 0) {t = Exit::TrapDoorUp;}
			else if (strcmp(argv[iac+1],"Exit::TrapDoorDown") == 0) {t = Exit::TrapDoorDown;}
			else if (strcmp(argv[iac+1],"Exit::StairsUp") == 0) {t = Exit::StairsUp;}
			else if (strcmp(argv[iac+1],"Exit::StairsDown") == 0) {t = Exit::StairsDown;}
			else if (strcmp(argv[iac+1],"Exit::WindowUnglazed") == 0) {t = Exit::WindowUnglazed;}
			else if (strcmp(argv[iac+1],"Exit::WindowGlazed") == 0) {t = Exit::WindowGlazed;}
			else if (strcmp(argv[iac+1],"Exit::Chimney") == 0) {t = Exit::Chimney;}
			else if (strcmp(argv[iac+1],"Exit::Pit") == 0) {t = Exit::Pit;}
			else if (strcmp(argv[iac+1],"Exit::Unspecified") == 0) {t = Exit::Unspecified;}
			else {
				Tcl_AppendResult(interp,"Bad ExitType: ",
						argv[iac+1],
						", allowed ExitTypes are: Exit::Doorway, ",
						"Exit::Door, Exit::LockedDoor, ",
						"Exit::SecretDoor, Exit::OnewayDoor, ",
						"Exit::TrapDoorUp, Exit::TrapDoorDown, ",
						"Exit::StairsUp, Exit::StairsDown, ",
						"Exit::WindowUnglazed, Exit::WindowGlazed, ",
						"Exit::Chimney, Exit::Pit, or ",
						"Exit::Unspecified",NULL);
				return TCL_ERROR;
			}
			
		} else if (strcmp(argv[iac],"-xcenter") == 0)
		{
			if (Tcl_GetDouble(interp,argv[iac+1],&x) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-ycenter") == 0)
		{
			if (Tcl_GetDouble(interp,argv[iac+1],&y) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-wallaligned") == 0)
		{
			if (Tcl_GetBoolean(interp,argv[iac+1],&wa) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-description") == 0)
		{
			d = argv[iac+1];
		} else if (strcmp(argv[iac],"-image") == 0)
		{
			i = argv[iac+1];
		} else if (strcmp(argv[iac],"-next") == 0)
		{
			ns = argv[iac+1];
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -type ",
					 "-xcenter -ycenter -wallaligned ",
					 "-description -image -next",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	_result = new Exit(t,x,y,(bool)(wa == 1),d,i,ns);
	SWIG_MakePtr(interp->result, (void *) _result,"_Exit_p");
	return TCL_OK;
}

%}

#endif

%subsection "ExitVector Class"

class ExitVector {
public:
	ExitVector(); // Constructor.
	~ExitVector(); // Destructor.
	const Exit* Index(
		int index // Index.
	) const; // Index function.
	const Exit* Nearest (
		double x, // X coordinate.
		double y // Y coordinate.
	) const; // Select nearest to (x,y) function.
	int ElementCount () const; // Return element count.
#ifdef TESTING
	void DeleteAtIndex(
		int index // Element to delete.
	); // Delete element by index.
	void DeleateNear(
		double x, // X coordinate.
		double y // Y coordinate.
	); // Delete element near coordinate.
	void InsertExit(
		const Exit& source // Exit object to insert.
	); // Insertion function
#endif
}; // Vector of Exits.

%subsection "Item Class"

%text %{\index{Item Class}% %}
class Item {
public:
	double XCenter() const; // Return center x coordinate.
	double YCenter() const; // Return center y coordinate.
	const char *Image() const; // Return image.
	const char *Filename() const; // Return file name.
	Item (); // Constructor.
	~Item(); // Destructor.
};// Item in space.

%addmethods Item {
  const char *Type() {
	Item::ItemType t = self->Type();
	switch (t)
	{
		case Item::Character: return "Item::Character"; break;
		case Item::Monster: return "Item::Monster"; break;
		case Item::Treasure: return "Item::Treasure"; break;
		case Item::TrickTrap: return "Item::TrickTrap"; break;
		case Item::Dressing: return "Item::Dressing"; break;
		case Item::Unspecified: return "Item::Unspecified"; break;
	}
	return "Item::Unspecified";
  }// Return type of item.
}


%subsection "ItemVector Class"

class ItemVector {
public:
	ItemVector(); // Constructor.
	~ItemVector(); // Destructor.
	const Item* Index(
		int index // Index.
	) const; // Index function.
	const Item* Nearest (
		double x, // X coordinate.
		double y // Y coordinate.
	) const; // Select nearest to (x,y) function.
	int ElementCount () const; // Return element count.
#ifdef TESTING
	void InsertItem(
		const Item& source // Item object to insert.
	); // Insertion function
	void DeleteAtIndex(
		int index // Element to delete.
	); // Delete element by index.
	void DeleateNear(
		double x, // X coordinate.
		double y // Y coordinate.
	); // Delete element near coordinate.
#endif
}; // Vector of Items.


%subsection "Space Class"

%native(NewSpace) int NewSpace(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]); /*
This is the interface to the constructor for {\tt Space} objects.  The argument 
list is an option-value list.  This function does not make a first class Tcl
object.  You need to use the {\tt Space} function with a {\tt -this}
argument to do that: \verb+[Space -this [NewSpace options...]]+.

The defined options for this function are:

\begin{tabular}{lllp{2in}}
flag & type & default & explanation \\
-shape & enum SpaceShape & Space::Undefined & Space shape. \\
-xcenter & double & 0.0 & X coordinate of the center. \\
-ycenter & double & 0.0 & Y coordinate of the center. \\
-name & string & ``'' & Name of space. \\
-description & string & ``'' & Description of space.\\
-backgroundcolor & string & ``white'' & Background color of space. \\
\end{tabular} */

class Space {
public:
	void UpdateFromRecord (const Record &rec); // Update Space from Record.
	Space(); // Constructor.
	~Space(); // Destructor.
	double CenterX() const; // Return this space's center point X coordinate.
	double SetCenterX(double newX); // Set the space's center point X coordinate.
	double CenterY() const; // Return this space's center point Y coordinate.
	double SetCenterY(double newY); // Set the space's center point Y coordinate.
	const Exit* NearestExit(
		double x, // X coordinate.
		double y // Y coordinate.
	) const; // Return the nearest exit.
	const Exit* IndexedExit(
		int index // Index.
	) const; // Return the ith exit.
	int NumberOfExits() const; // Return the count of exits.
	void InsertExit (const Exit& source); // Insert an exit.
	void DeleteExitNear(
		double x, // X coordinate.
		double y // Y coordinate.
	); // Delete exit nearest to x,y.
	void DeleteExitAtIndex(
		int index // Element to delete.
	); // Delete exit by index.
	const Item* Nearestitem(
		double x, // X coordinate.
		double y // Y coordinate.
	) const; // Return the nearest item.
	const Item* IndexedItem(
		int index // Index.
	) const; // Return the ith item.
	int NumberOfItems() const; // Return the count of items.
	void InsertItem (const Item& source); // Insert an item.
	void DeleteItemNear(
		double x, // X coordinate.
		double y // Y coordinate.
	); // Delete item nearest to x,y.
	void DeleteItemAtIndex(
		int index // Element to delete.
	); // Delete item by index.
	const char *Name() const; // Return name of the space.
	const char *SetName(const char *newN); // Set the name of the space.
	const char *Description() const; // Return description of the space.
	const char *SetDescription(const char *newD); // Set the description of the space. 
	const char *BackgroundColor() const; // Return background color of the space.
	const char *SetBackgroundColor(const char *newBG); // Set the background color of the space.
};// Basic Space class.	

%apply int MyTcl_Result { int Space_InsertNewExit };
%apply int MyTcl_Result { int Space_InsertNewItem };
%apply int MyTcl_Result { int Space_SetShape };
%apply int MyTcl_Result { int Space_MakeGraphicCommannd };

%addmethods Space {
  const char * Shape() {
  	Space::SpaceShape s = self->Shape();
  	switch (s)
  	{
		case Space::Square: return "Space::Square"; break;
		case Space::Hexagon: return "Space::Hexagon"; break;
		case Space::Undefined: return "Space::Undefined"; break;
	}
	return "Space::Undefined";
  } // Return this space's shape.
  int SetShape(Tcl_Interp *interp,const char *newS)
  {
	Space::SpaceShape s = Space::Undefined;
	if (strcmp(newS,"Space::Square") == 0) {s = Space::Square;}
	else if (strcmp(newS,"Space::Hexagon") == 0) {s = Space::Hexagon;}
	else if (strcmp(newS,"Space::Undefined") == 0) {s = Space::Undefined;}
	else {
		Tcl_AppendResult(interp,"Bad SpaceShape: ",newS,
				", allowed SpaceShapes are: Space::Square, ",
				"Space::Hexagon, ",
				"Space::Undefined",NULL);
		return TCL_ERROR;
	}
	s = self->SetShape(s);
	switch (s)
  	{
		case Space::Square: Tcl_AppendResult(interp,"Space::Square",NULL); break;
		case Space::Hexagon: Tcl_AppendResult(interp,"Space::Hexagon",NULL); break;
		case Space::Undefined: Tcl_AppendResult(interp,"Space::Undefined",NULL); break;
	}
	return TCL_OK;
  }// Set this space's shape.
  Record * RawData() {return new Record((Record)(*self));} // Type conversion: convert to a Record.
  int InsertNewExit(Tcl_Interp *interp,const char *type="Exit::Unspecified",double x=0.0, double y=0.0,bool wa=true,const char *d="",const char *im="",const char *ns="")
  {
    Exit::ExitType t = Exit::Unspecified;
    if (strcmp(type,"Exit::Doorway") == 0) {t = Exit::Doorway;}
    else if (strcmp(type,"Exit::Door") == 0) {t = Exit::Door;}
    else if (strcmp(type,"Exit::LockedDoor") == 0) {t = Exit::LockedDoor;}
    else if (strcmp(type,"Exit::SecretDoor") == 0) {t = Exit::SecretDoor;}
    else if (strcmp(type,"Exit::OnewayDoor") == 0) {t = Exit::OnewayDoor;}
    else if (strcmp(type,"Exit::TrapDoorUp") == 0) {t = Exit::TrapDoorUp;}
    else if (strcmp(type,"Exit::TrapDoorDown") == 0) {t = Exit::TrapDoorDown;}
    else if (strcmp(type,"Exit::StairsUp") == 0) {t = Exit::StairsUp;}
    else if (strcmp(type,"Exit::StairsDown") == 0) {t = Exit::StairsDown;}
    else if (strcmp(type,"Exit::WindowUnglazed") == 0) {t = Exit::WindowUnglazed;}
    else if (strcmp(type,"Exit::WindowGlazed") == 0) {t = Exit::WindowGlazed;}
    else if (strcmp(type,"Exit::Chimney") == 0) {t = Exit::Chimney;}
    else if (strcmp(type,"Exit::Pit") == 0) {t = Exit::Pit;}
    else if (strcmp(type,"Exit::Unspecified") == 0) {t = Exit::Unspecified;}
    else {
	Tcl_AppendResult(interp,"Bad ExitType: ",
			type,
			", allowed ExitTypes are: Exit::Doorway, ",
			"Exit::Door, Exit::LockedDoor, ",
			"Exit::SecretDoor, Exit::OnewayDoor, ",
			"Exit::TrapDoorUp, Exit::TrapDoorDown, ",
			"Exit::StairsUp, Exit::StairsDown, ",
			"Exit::WindowUnglazed, Exit::WindowGlazed, ",
			"Exit::Chimney, Exit::Pit, or ",
			"Exit::Unspecified",NULL);
	return TCL_ERROR;
    }
    Exit *e = new Exit(t,x,y,wa,d,im,ns);
    self->InsertExit(*e);
    delete e;
    return TCL_OK;
  } //Insert a new exit.
  int InsertNewItem(Tcl_Interp *interp,const char *type="Item::Unspecified",double x=0.0, double y=0.0,const char *im="",const char *fn="")
  {
    Item::ItemType t = Item::Unspecified;
    if (strcmp(type,"Item::Character") == 0) {t = Item::Character;}
    else if (strcmp(type,"Item::Monster") == 0) {t = Item::Monster;}
    else if (strcmp(type,"Item::Treasure") == 0) {t = Item::Treasure;}
    else if (strcmp(type,"Item::TrickTrap") == 0) {t = Item::TrickTrap;}
    else if (strcmp(type,"Item::Dressing") == 0) {t = Item::Dressing;}
    else if (strcmp(type,"Item::Unspecified") == 0) {t = Item::Unspecified;}
    else {
	Tcl_AppendResult(interp,"Bad ItemType: ",
			type,
			", allowed ItemTypes are: Item::Character, ",
			"Item::Monster, Item::Treasure, ",
			"Item::TrickTrap, Item::Dressing, ",
			"Item::Unspecified",NULL);
	return TCL_ERROR;
    }
    Item *itm = new Item(t,x,y,im,fn);
    self->InsertItem(*itm);
    delete itm;
    return TCL_OK;
  } //Insert a new item.
  int MakeGraphicCommannd(Tcl_Interp *interp,double scaleFactor = 1.0)
  {
    double hWidth = GeoConstants::Width / 2.0,
           halfsl = GeoConstants::HexSideLength / 2.0;
    Space::SpaceShape s = self->Shape();
    double cx = self->CenterX(), cy = self->CenterY();
    static char xxS[12], yyS[12];
    switch (s)
    {
	case Space::Undefined: 
	  Tcl_AppendResult(interp,"Shape undefined, cannot draw.",NULL);
	  return TCL_ERROR;
	  break;
	case Space::Square:
          Tcl_AppendElement(interp,"rectangle");
	  sprintf(xxS,"%f",(cx - hWidth)*scaleFactor); 
	  sprintf(yyS,"%f",(cy - hWidth)*scaleFactor); 
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  sprintf(xxS,"%f",(cx + hWidth)*scaleFactor); 
	  sprintf(yyS,"%f",(cy + hWidth)*scaleFactor); 
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  Tcl_AppendElement(interp,"-outline");
	  Tcl_AppendElement(interp,"black");
	  Tcl_AppendElement(interp,"-fill");
	  Tcl_AppendElement(interp,(char*)self->BackgroundColor());
	  Tcl_AppendElement(interp,"-tag");
	  Tcl_AppendElement(interp,(char*)self->Name());
	  return TCL_OK;
	case Space::Hexagon:
	  Tcl_AppendElement(interp,"polygon");
	  sprintf(xxS,"%f",(cx-hWidth)*scaleFactor); 
	  sprintf(yyS,"%f",(cy+halfsl)*scaleFactor);
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  sprintf(yyS,"%f",(cy-halfsl)*scaleFactor);
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  sprintf(xxS,"%f",cx*scaleFactor); 
	  sprintf(yyS,"%f",(cy-(halfsl+GeoConstants::HexPeakHeight))*scaleFactor);
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  sprintf(xxS,"%f",(cx+hWidth)*scaleFactor); 
	  sprintf(yyS,"%f",(cy-halfsl)*scaleFactor);
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  sprintf(yyS,"%f",(cy+halfsl)*scaleFactor);
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  sprintf(xxS,"%f",cx*scaleFactor); 
	  sprintf(yyS,"%f",(cy+(halfsl+GeoConstants::HexPeakHeight))*scaleFactor);
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  sprintf(xxS,"%f",(cx-hWidth)*scaleFactor); 
	  sprintf(yyS,"%f",(cy+halfsl)*scaleFactor);
	  Tcl_AppendElement(interp,xxS);
	  Tcl_AppendElement(interp,yyS);
	  Tcl_AppendElement(interp,"-outline");
	  Tcl_AppendElement(interp,"black");
	  Tcl_AppendElement(interp,"-fill");
	  Tcl_AppendElement(interp,(char*)self->BackgroundColor());
	  Tcl_AppendElement(interp,"-tag");
	  Tcl_AppendElement(interp,(char*)self->Name());
	  return TCL_OK;
    }
    Tcl_AppendResult(interp,"Should not get here!",NULL);
    return TCL_ERROR;
  }/* Generate the command body to draw the base space:
      \verb+eval [concat .canvas create [space MakeGraphicCommannd ?scale?]]+. */
}

%{

static int NewSpace(ClientData /*clientData*/, Tcl_Interp *interp, int argc, char *argv[])
{
	Space * _result = NULL;
	char *d = "", *n = "", *bg = "white";
	Space::SpaceShape s = Space::Undefined;
	double x = 0.0, y = 0.0;
        int iac = 1;

	Tcl_ResetResult(interp);
	while (iac < argc-1)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		}
		if (strcmp(argv[iac],"-shape") == 0)
		{
			if (strcmp(argv[iac+1],"Space::Square") == 0) {s = Space::Square;}
			else if (strcmp(argv[iac+1],"Space::Hexagon") == 0) {s = Space::Hexagon;}
			else if (strcmp(argv[iac+1],"Space::Undefined") == 0) {s = Space::Undefined;}
			else {
				Tcl_AppendResult(interp,"Bad SpaceShape: ",
						argv[iac+1],
						", allowed SpaceShapes are: Space::Square, ",
						"Space::Hexagon, ",
						"Space::Undefined",NULL);
				return TCL_ERROR;
			}
			
		} else if (strcmp(argv[iac],"-xcenter") == 0)
		{
			if (Tcl_GetDouble(interp,argv[iac+1],&x) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-ycenter") == 0)
		{
			if (Tcl_GetDouble(interp,argv[iac+1],&y) == TCL_ERROR) return TCL_ERROR;
		} else if (strcmp(argv[iac],"-name") == 0)
		{
			n = argv[iac+1];
		} else if (strcmp(argv[iac],"-description") == 0)
		{
			d = argv[iac+1];
		} else if (strcmp(argv[iac],"-backgroundcolor") == 0)
		{
			bg = argv[iac+1];
		} else
		{
			Tcl_AppendResult(interp,"Undefined option:",argv[iac],
					 " allowed options are: -shape ",
					 "-xcenter -ycenter -name ",
					 "-description -backgroundcolor",
					 NULL);
			return TCL_ERROR;
		}
		iac += 2;
	}
	if (iac < argc)
	{
		if (*argv[iac] != '-')
		{
			Tcl_AppendResult(interp,"Not an option: ",argv[iac],NULL);
			return TCL_ERROR;
 		} else
		{
			Tcl_AppendResult(interp,"Extra option without a value: ",argv[iac],NULL);
			return TCL_ERROR;
		}
	}
	_result = new Space(s,x,y,n,d,bg);
	SWIG_MakePtr(interp->result, (void *) _result,"_Space_p");
	return TCL_OK;
}


%}

%text %{
\clearpage
\addcontentsline{toc}{section}{References}
\bibliography{../RPG}
\bibliographystyle{plain} 
\clearpage
\addcontentsline{toc}{section}{Index}
\printindex%  %}
