/* -*- Mode: C++; tab-width: 8; c-basic-offset: 2; indent-tabs-mode: nil; -*- */

//#define DEBUGTAG "ReplaySession"
#define USE_BREAKPOINT_TARGET 1
#define USE_TIMESLICE_COALESCING 1

#include "ReplaySession.h"

#include <syscall.h>
#include <sys/prctl.h>

#include <algorithm>

#include "AutoRemoteSyscalls.h"
#include "kernel_metadata.h"
#include "log.h"
#include "replay_syscall.h"
#include "task.h"
#include "util.h"

using namespace rr;
using namespace std;

/* Why a skid region?  Interrupts generated by perf counters don't
 * fire at exactly the programmed point (as of 2013 kernel/HW);
 * there's a variable slack region, which is technically unbounded.
 * This means that an interrupt programmed for retired branch k might
 * fire at |k + 50|, for example.  To counteract the slack, we program
 * interrupts just short of our target, by the |SKID_SIZE| region
 * below, and then more slowly advance to the real target.
 *
 * How was this magic number determined?  Trial and error: we want it
 * to be as small as possible for efficiency, but not so small that
 * overshoots are observed.  If all other possible causes of overshoot
 * have been ruled out, like memory divergence, then you'll know that
 * this magic number needs to be increased if the following symptom is
 * observed during replay.  Running with DEBUGLOG enabled (see above),
 * a sequence of log messages like the following will appear
 *
 * 1. programming interrupt for [target - SKID_SIZE] ticks
 * 2. Error: Replay diverged.  Dumping register comparison.
 * 3. Error: [list of divergent registers; arbitrary]
 * 4. Error: overshot target ticks=[target] by [i]
 *
 * The key is that no other replayer log messages occur between (1)
 * and (2).  This spew means that the replayer programmed an interrupt
 * for ticks=[target-SKID_SIZE], but the tracee was actually interrupted
 * at ticks=[target+i].  And that in turn means that the kernel/HW
 * skidded too far past the programmed target for rr to handle it.
 *
 * If that occurs, the SKID_SIZE needs to be increased by at least
 * [i].
 *
 * NB: there are probably deeper reasons for the target slack that
 * could perhaps let it be deduced instead of arrived at empirically;
 * perhaps pipeline depth and things of that nature are involved.  But
 * those reasons if they exit are currently not understood.
 */
static const int SKID_SIZE = 70;

static void debug_memory(Task* t) {
  if (should_dump_memory(t, t->current_trace_frame())) {
    dump_process_memory(t, t->current_trace_frame().time(), "rep");
  }
  if (t->session().can_validate() &&
      should_checksum(t, t->current_trace_frame())) {
    /* Validate the checksum we computed during the
     * recording phase. */
    validate_process_memory(t, t->current_trace_frame().time());
  }
}

static void remap_shared_mmap(AutoRemoteSyscalls& remote, EmuFs& dest_emu_fs,
                              const Mapping& m, const MappableResource& r) {
  LOG(debug) << "    remapping shared region at " << m.start << "-" << m.end;
  remote.syscall(syscall_number_for_munmap(remote.arch()), m.start,
                 m.num_bytes());
  // NB: we don't have to unmap then re-map |t->vm()|'s idea of
  // the emulated file mapping.  Though we'll be remapping the
  // *real* OS mapping in |t| to a different file, that new
  // mapping still refers to the same *emulated* file, with the
  // same emulated metadata.

  auto emufile = dest_emu_fs.at(r.id);
  // TODO: this duplicates some code in replay_syscall.cc, but
  // it's somewhat nontrivial to factor that code out.
  int remote_fd;
  {
    string path = emufile->proc_path();
    AutoRestoreMem child_path(remote, path.c_str());
    int oflags =
        (MAP_SHARED & m.flags) && (PROT_WRITE & m.prot) ? O_RDWR : O_RDONLY;
    remote_fd = remote.syscall(syscall_number_for_open(remote.arch()),
                               child_path.get().as_int(), oflags);
    if (0 > remote_fd) {
      FATAL() << "Couldn't open " << path << " in tracee";
    }
  }
  // XXX this condition is x86/x64-specific, I imagine.
  remote_ptr<void> addr = remote.mmap_syscall(m.start, m.num_bytes(), m.prot,
                                              // The remapped segment *must* be
                                              // remapped at the same address,
                                              // or else many things will go
                                              // haywire.
                                              m.flags | MAP_FIXED, remote_fd,
                                              m.offset / page_size());
  ASSERT(remote.task(), addr == m.start);

  remote.syscall(syscall_number_for_close(remote.arch()), remote_fd);
}

ReplaySession::~ReplaySession() {
  // We won't permanently leak any OS resources by not ensuring
  // we've cleaned up here, but sessions can be created and
  // destroyed many times, and we don't want to temporarily hog
  // resources.
  kill_all_tasks();
  assert(tasks().size() == 0 && vms().size() == 0);
  gc_emufs();
  assert(emufs().size() == 0);
}

void ReplaySession::copy_state_to(Session& dest, EmuFs& dest_emu_fs) {
  for (auto vm : sas) {
    // Creating a checkpoint of a session with active breakpoints
    // or watchpoints is not supported.
    assert(!vm->has_breakpoints());
    assert(!vm->has_watchpoints());

    Task* some_task = *vm->task_set().begin();
    pid_t tgid = some_task->tgid();
    Task* group_leader = find_task(tgid);
    LOG(debug) << "  forking tg " << tgid
               << " (real: " << group_leader->real_tgid() << ")";

    if (group_leader->is_probably_replaying_syscall()) {
      group_leader->finish_emulated_syscall();
    }

    Task* clone_leader = group_leader->os_fork_into(&dest);
    dest.on_create(clone_leader);
    LOG(debug) << "  forked new group leader " << clone_leader->tid;

    {
      AutoRemoteSyscalls remote(clone_leader);
      for (auto& kv : clone_leader->vm()->memmap()) {
        const Mapping& m = kv.first;
        const MappableResource& r = kv.second;
        if (!r.is_shared_mmap_file()) {
          continue;
        }
        remap_shared_mmap(remote, dest_emu_fs, m, r);
      }

      for (auto t : group_leader->task_group()->task_set()) {
        if (group_leader == t) {
          continue;
        }
        LOG(debug) << "    cloning " << t->rec_tid;

        if (t->is_probably_replaying_syscall()) {
          t->finish_emulated_syscall();
        }
        Task* t_clone = t->os_clone_into(clone_leader, remote);
        dest.on_create(t_clone);
        t_clone->copy_state(t);
      }
    }
    LOG(debug) << "  restoring group-leader state ...";
    clone_leader->copy_state(group_leader);
  }
  assert(dest.vms().size() > 0);
}

ReplaySession::shr_ptr ReplaySession::clone() {
  LOG(debug) << "Deepforking ReplaySession " << this << " ...";

  shr_ptr session(new ReplaySession(*this));
  LOG(debug) << "  deepfork session is " << session.get();
  session->tracees_consistent = tracees_consistent;
  memcpy(session->syscallbuf_flush_buffer_array, syscallbuf_flush_buffer_array,
         sizeof(syscallbuf_flush_buffer_array));

  copy_state_to(*session, session->emufs());

  return session;
}

DiversionSession::shr_ptr ReplaySession::clone_diversion() {
  LOG(debug) << "Deepforking ReplaySession " << this
             << " to DiversionSession...";

  DiversionSession::shr_ptr session(new DiversionSession(*this));
  LOG(debug) << "  deepfork session is " << session.get();

  copy_state_to(*session, session->emufs());

  return session;
}

void ReplaySession::gc_emufs() { emu_fs->gc(*this); }

/*static*/ ReplaySession::shr_ptr ReplaySession::create(const string& dir) {
  shr_ptr session(new ReplaySession(dir));

  // Because we execvpe() the tracee, we must ensure that $PATH
  // is the same as in recording so that libc searches paths in
  // the same order.  So copy that over now.
  //
  // And because we use execvpe(), the exec'd tracee will start
  // with a fresh environment guaranteed to be the same as in
  // replay, so we don't have to worry about any mutation here
  // affecting post-exec execution.
  for (auto& e : session->trace_in.initial_envp()) {
    if (e.find("PATH=") == 0) {
      // NB: intentionally leaking this string.
      putenv(strdup(e.c_str()));
    }
  }

  Task* t = Task::spawn(*session, session->trace_in,
                        session->trace_reader().peek_frame().tid());
  session->on_create(t);

  return session;
}

void ReplaySession::advance_to_next_trace_frame(TraceFrame::Time stop_at_time) {
  if (last_task()) {
    // Already finished replay; no need to advance (and in fact, we can't).
    return;
  }

  trace_frame = trace_in.read_frame();

  // Subsequent reschedule-events of the same thread can be
  // combined to a single event.  This meliorization is a
  // tremendous win.
  if (USE_TIMESLICE_COALESCING && trace_frame.event().type == EV_SCHED) {
    TraceFrame next_frame = trace_in.peek_frame();
    while (EV_SCHED == next_frame.event().type &&
           next_frame.tid() == trace_frame.tid() &&
           stop_at_time != next_frame.time() &&
           !trace_instructions_up_to_event(next_frame.time())) {
      trace_frame = trace_in.read_frame();
      next_frame = trace_in.peek_frame();
    }
  }
}

bool ReplaySession::is_ignored_signal(int sig) {
  switch (sig) {
    // SIGCHLD can arrive after tasks die during replay.  We don't
    // care about SIGCHLD unless it was recorded, in which case
    // we'll emulate its delivery.
    case SIGCHLD:
    // SIGWINCH arrives when the user resizes the terminal window.
    // Not relevant to replay.
    case SIGWINCH:
      return true;
    default:
      return false;
  }
}

/**
 * Continue until reaching either the "entry" of an emulated syscall,
 * or the entry or exit of an executed syscall.  |emu| is nonzero when
 * we're emulating the syscall.  Return COMPLETE when the next syscall
 * boundary is reached, or INCOMPLETE if advancing to the boundary was
 * interrupted by an unknown trap.
 */
Completion ReplaySession::cont_syscall_boundary(Task* t, ExecOrEmulate emu,
                                                RunCommand stepi) {
  bool is_syscall_entry = SYSCALL_ENTRY == trace_frame.event().state;
  if (is_syscall_entry) {
    t->stepped_into_syscall = false;
  }

  ResumeRequest resume_how;
  if (emu == EMULATE && stepi == RUN_SINGLESTEP) {
    resume_how = RESUME_SYSEMU_SINGLESTEP;
  } else if (emu == EMULATE) {
    resume_how = RESUME_SYSEMU;
  } else if (stepi == RUN_SINGLESTEP) {
    resume_how = RESUME_SINGLESTEP;
    // Annoyingly, PTRACE_SINGLESTEP doesn't raise
    // PTRACE_O_SYSGOOD traps.  (Unlike
    // PTRACE_SINGLESTEP_SYSEMU, which does.)  That means
    // if we just blindly stepi'd the tracee to a
    // non-emulated syscall, we'd shoot right past it
    // without knowing.
    //
    // The correct solution to this problem is to emulate
    // all syscalls during replay and then inject the
    // executed ones (as we do for mmap).  But for now,
    // work around this problem by recognizing syscall
    // insns and issuing PTRACE_SYSCALL to enter them
    // instead of PTRACE_SINGLESTEP.
    if (is_at_syscall_instruction(t, t->ip()) || t->stepped_into_syscall) {
      resume_how = RESUME_SYSCALL;
      // Leave this breadcrumb on syscall entry so
      // that we know to issue PTRACE_SYSCALL to
      // exit the syscall the next time we stepi.
      t->stepped_into_syscall = is_syscall_entry;
    }
  } else {
    resume_how = RESUME_SYSCALL;
  }
  t->resume_execution(resume_how, RESUME_WAIT);

  t->child_sig = t->pending_sig();
  if (is_ignored_signal(t->child_sig)) {
    return cont_syscall_boundary(t, emu, stepi);
  }

  if (t->ptrace_event() == PTRACE_EVENT_EXEC) {
    t->post_exec(&t->current_trace_frame().regs());
  }

  if (SIGTRAP == t->child_sig) {
    return INCOMPLETE;
  } else if (t->child_sig) {
    ASSERT(t, false) << "Replay got unrecorded signal " << t->child_sig;
  }

  assert(t->child_sig == 0);

  return COMPLETE;
}

/**
 * Advance to the next syscall entry (or virtual entry) according to
 * |step|.  Return COMPLETE if successful, or INCOMPLETE if an unhandled trap
 * occurred.
 */
Completion ReplaySession::enter_syscall(Task* t, RunCommand stepi) {
  if (cont_syscall_boundary(t, current_step.syscall.emu, stepi) == INCOMPLETE) {
    return INCOMPLETE;
  }
  t->validate_regs();
  return COMPLETE;
}

/**
 * Advance past the reti (or virtual reti) according to |step|.
 * Return COMPLETE if successful, or INCOMPLETE if an unhandled trap occurred.
 */
Completion ReplaySession::exit_syscall(Task* t, RunCommand stepi) {
  ExecOrEmulate emu = current_step.syscall.emu;

  if (emu == EXEC) {
    if (cont_syscall_boundary(t, emu, stepi) == INCOMPLETE) {
      return INCOMPLETE;
    }
  }

  for (int i = 0; i < current_step.syscall.num_emu_args; ++i) {
    t->set_data_from_trace();
  }
  if (current_step.syscall.emu_ret) {
    t->set_return_value_from_trace();
  }
  uint32_t flags = 0;
  if (t->arch() == SupportedArch::x86 &&
      (X86Arch::pwrite64 == current_step.syscall.number ||
       X86Arch::pread64 == current_step.syscall.number)) {
    flags |= Task::IGNORE_ESI;
  }
  t->validate_regs(flags);

  if (emu == EMULATE) {
    t->finish_emulated_syscall();
  }
  return COMPLETE;
}

/**
 * Advance |t| to the next signal or trap.  If |stepi| is |SINGLESTEP|,
 * then execution resumes by single-stepping.  Otherwise it continues
 * normally.  The delivered signal is recorded in |t->child_sig|.
 */
void ReplaySession::continue_or_step(Task* t, RunCommand stepi,
                                     int64_t tick_period) {
  int child_sig_gt_zero;

  ResumeRequest resume_how;
  if (stepi == RUN_SINGLESTEP) {
    resume_how = RESUME_SINGLESTEP;
  } else {
    /* We continue with RESUME_SYSCALL for error checking:
     * since the next event is supposed to be a signal,
     * entering a syscall here means divergence.  There
     * shouldn't be any straight-line execution overhead
     * for SYSCALL vs. CONT, so the difference in cost
     * should be neglible. */
    resume_how = RESUME_SYSCALL;
  }
  t->resume_execution(resume_how, RESUME_WAIT, 0, tick_period);

  t->child_sig = t->pending_sig();
  child_sig_gt_zero = (0 < t->child_sig);
  if (child_sig_gt_zero) {
    return;
  }
  ASSERT(t, child_sig_gt_zero)
      << "Replaying `" << Event(trace_frame.event())
      << "': expecting tracee signal or trap, but instead at `"
      << t->syscall_name(t->regs().original_syscallno())
      << "' (ticks: " << t->tick_count() << ")";
}

/**
 * Return nonzero if |t| was stopped for a breakpoint trap (int3),
 * as opposed to a trace trap.  Return zero in the latter case.
 */
static bool is_breakpoint_trap(Task* t) {
  assert(SIGTRAP == t->child_sig);

  const siginfo_t& si = t->get_siginfo();
  assert(SIGTRAP == si.si_signo);

  /* XXX unable to find docs on which of these "should" be
   * right.  The SI_KERNEL code is seen in the int3 test, so we
   * at least need to handle that. */
  return SI_KERNEL == si.si_code || TRAP_BRKPT == si.si_code;
}

/**
 * Return one of the (non-zero) enumerated TRAP_* debugger-trap types
 * above if the SIGTRAP generated by the child is intended for the
 * debugger, or zero if it's meant for rr internally.
 *
 * NB: calling this function while advancing the ticks counter through hpc
 * interrupts when emulating asynchronous signal delivery *will*
 * result in bad results.  Don't call this function from there; it's
 * not necessary.
 */
enum ExecStateType {
  UNKNOWN,
  NOT_AT_TARGET,
  AT_TARGET
};
TrapType ReplaySession::compute_trap_type(Task* t, int target_sig,
                                          SignalDeterministic deterministic,
                                          ExecStateType exec_state,
                                          RunCommand stepi) {
  TrapType trap_type;

  assert(SIGTRAP == t->child_sig);

  /* We're not replaying a trap, and it was clearly raised on
   * behalf of the debugger.  (The debugger will verify
   * that.) */
  if (SIGTRAP != target_sig
          /* Replay of deterministic signals never internally
           * single-steps or sets internal breakpoints. */
      &&
      (DETERMINISTIC_SIG == deterministic
           /* Replay of async signals will sometimes internally
            * single-step when advancing to an execution target,
            * so the trap was only clearly for the debugger if
            * the debugger was requesting single-stepping. */
       ||
       (stepi == RUN_SINGLESTEP && NOT_AT_TARGET == exec_state))) {
    return stepi == RUN_SINGLESTEP ? TRAP_STEPI : TRAP_BKPT_USER;
  }

  /* We're trying to replay a deterministic SIGTRAP, or we're
   * replaying an async signal. */

  trap_type = t->vm()->get_breakpoint_type_for_retired_insn(t->ip());
  if (TRAP_BKPT_USER == trap_type || TRAP_BKPT_INTERNAL == trap_type) {
    assert(is_breakpoint_trap(t));
    return trap_type;
  }

  if (is_breakpoint_trap(t)) {
    /* We successfully replayed a recorded deterministic
     * SIGTRAP.  (Because it must have been raised by an
     * |int3|, but not one we injected.)  Not for the
     * debugger, although we'll end up notifying it
     * anyway. */
    assert(DETERMINISTIC_SIG == deterministic);
    return TRAP_NONE;
  }

  if (DETERMINISTIC_SIG == deterministic) {
    /* If the delivery of SIGTRAP is supposed to be
     * deterministic and we didn't just retire an |int 3|
     * and this wasn't a breakpoint, we must have been
     * single stepping.  So definitely for the
     * debugger. */
    assert(stepi == RUN_SINGLESTEP);
    return TRAP_STEPI;
  }

  /* We're replaying an async signal. */

  if (AT_TARGET == exec_state) {
    /* If we're at the target of the async signal
     * delivery, prefer delivering the signal to retiring
     * a possible debugger single-step; we'll notify the
     * debugger anyway. */
    return TRAP_NONE;
  }

  /* Otherwise, we're not at the execution target, so may have
   * been internally single-stepping.  We'll notify the debugger
   * if it was also requesting single-stepping.  The debugger
   * won't care about the rr-internal trap if it wasn't
   * requesting single-stepping. */
  return stepi == RUN_SINGLESTEP ? TRAP_STEPI : TRAP_NONE;
}

/**
 * Shortcut for callers that don't care about internal breakpoints.
 * Return nonzero if |t|'s trap is for the debugger, zero otherwise.
 */
bool ReplaySession::is_debugger_trap(Task* t, int target_sig,
                                     SignalDeterministic deterministic,
                                     ExecStateType exec_state,
                                     RunCommand stepi) {
  TrapType type =
      compute_trap_type(t, target_sig, deterministic, exec_state, stepi);
  assert(TRAP_BKPT_INTERNAL != type);
  return TRAP_NONE != type;
}

static void guard_overshoot(Task* t, const Registers& target_regs,
                            Ticks target_ticks, Ticks remaining_ticks,
                            Ticks ticks_slack, bool ignored_early_match,
                            Ticks ticks_left_at_ignored_early_match) {
  if (remaining_ticks < -ticks_slack) {
    remote_ptr<uint8_t> target_ip = target_regs.ip();

    LOG(error) << "Replay diverged.  Dumping register comparison.";
    /* Cover up the internal breakpoint that we may have
     * set, and restore the tracee's $ip to what it would
     * have been had it not hit the breakpoint (if it did
     * hit the breakpoint).*/
    t->vm()->remove_breakpoint(target_ip, TRAP_BKPT_INTERNAL);
    if (remote_ptr<uint8_t>(t->regs().ip()) ==
        target_ip + sizeof(AddressSpace::breakpoint_insn)) {
      t->move_ip_before_breakpoint();
    }
    Registers::compare_register_files(t, "rep overshoot", t->regs(), "rec",
                                      target_regs, LOG_MISMATCHES);
    if (ignored_early_match) {
      ASSERT(t, false) << "overshot target ticks=" << target_ticks << " by "
                       << -remaining_ticks << "; ignored early match with "
                       << ticks_left_at_ignored_early_match << " ticks left";
    } else {
      ASSERT(t, false) << "overshot target ticks=" << target_ticks << " by "
                       << -remaining_ticks;
    }
  }
}

static void guard_unexpected_signal(Task* t) {
  Event ev;
  int child_sig_is_zero_or_sigtrap =
      (0 == t->child_sig || SIGTRAP == t->child_sig);
  /* "0" normally means "syscall", but continue_or_step() guards
   * against unexpected syscalls.  So the caller must have set
   * "0" intentionally. */
  if (child_sig_is_zero_or_sigtrap) {
    return;
  }
  if (t->child_sig) {
    ev = SignalEvent(t->child_sig, NONDETERMINISTIC_SIG, t->arch());
  } else {
    ev = SyscallEvent(max(0L, (long)t->regs().original_syscallno()), t->arch());
  }
  ASSERT(t, child_sig_is_zero_or_sigtrap) << "Replay got unrecorded event "
                                          << ev << " while awaiting signal";
}

static bool is_same_execution_point(Task* t, const Registers& rec_regs,
                                    Ticks ticks_left, Ticks ticks_slack,
                                    bool* ignoring_early_match,
                                    Ticks* ticks_left_at_ignored_early_match) {
  MismatchBehavior behavior =
#ifdef DEBUGTAG
      LOG_MISMATCHES
#else
      EXPECT_MISMATCHES
#endif
      ;
  if (ticks_left > 0) {
    if (ticks_left <= ticks_slack &&
        Registers::compare_register_files(t, "(rep)", t->regs(), "(rec)",
                                          rec_regs, EXPECT_MISMATCHES)) {
      *ignoring_early_match = true;
      *ticks_left_at_ignored_early_match = ticks_left;
    }
    LOG(debug) << "  not same execution point: " << ticks_left
               << " ticks left (@" << HEX(rec_regs.ip()) << ")";
#ifdef DEBUGTAG
    Registers::compare_register_files(t, "(rep)", t->regs(), "(rec)", rec_regs,
                                      LOG_MISMATCHES);
#endif
    return false;
  }
  if (ticks_left < -ticks_slack) {
    LOG(debug) << "  not same execution point: " << ticks_left
               << " ticks left (@" << HEX(rec_regs.ip()) << ")";
#ifdef DEBUGTAG
    Registers::compare_register_files(t, "(rep)", t->regs(), "(rec)", rec_regs,
                                      LOG_MISMATCHES);
#endif
    return false;
  }
  if (!Registers::compare_register_files(t, "rep", t->regs(), "rec", rec_regs,
                                         behavior)) {
    LOG(debug) << "  not same execution point: regs differ (@"
               << HEX(rec_regs.ip()) << ")";
    return false;
  }
  LOG(debug) << "  same execution point";
  return true;
}

Ticks ReplaySession::get_ticks_slack(Task* t) {
  if (cpuid_bug_detector.is_cpuid_bug_detected()) {
    // Somewhat arbitrary guess
    return 6;
  }
  return 0;
}

/**
 * Run execution forwards for |t| until |ticks| is reached, and the $ip
 * reaches the recorded $ip.  Return COMPLETE if successful or INCOMPLETE if an
 * unhandled interrupt occurred.  |sig| is the pending signal to be
 * delivered; it's only used to distinguish debugger-related traps
 * from traps related to replaying execution.  |ticks| is an inout param
 * that will be decremented by branches retired during this attempted
 * step.
 */
Completion ReplaySession::advance_to(Task* t, const Registers& regs, int sig,
                                     RunCommand stepi, Ticks ticks) {
  pid_t tid = t->tid;
  remote_ptr<uint8_t> ip = regs.ip();
  Ticks ticks_left;
  Ticks ticks_slack = get_ticks_slack(t);
  bool did_set_internal_breakpoint = false;
  bool ignored_early_match = false;
  Ticks ticks_left_at_ignored_early_match = 0;

  assert(t->hpc.ticks_fd() > 0);
  assert(t->child_sig == 0);

  /* Step 1: advance to the target ticks (minus a slack region) as
   * quickly as possible by programming the hpc. */
  ticks_left = ticks - t->tick_count();

  LOG(debug) << "advancing " << ticks_left << " ticks to reach " << ticks << "/"
             << ip;

  /* XXX should we only do this if (ticks > 10000)? */
  while (ticks_left - SKID_SIZE > SKID_SIZE) {
    if (SIGTRAP == t->child_sig) {
      /* We proved we're not at the execution
       * target, and we haven't set any internal
       * breakpoints, and we're not temporarily
       * internally single-stepping, so we must have
       * hit a debugger breakpoint or the debugger
       * was single-stepping the tracee.  (The
       * debugging code will verify that.) */
      return INCOMPLETE;
    }
    t->child_sig = 0;

    LOG(debug) << "  programming interrupt for " << (ticks_left - SKID_SIZE)
               << " ticks";

    continue_or_step(t, stepi, ticks_left - SKID_SIZE);
    if (PerfCounters::TIME_SLICE_SIGNAL == t->child_sig ||
        is_ignored_signal(t->child_sig)) {
      t->child_sig = 0;
    }
    guard_unexpected_signal(t);

    /* TODO this assertion won't catch many spurious
     * signals; should assert that the siginfo says the
     * source is io-ready and the fd is the child's fd. */
    if (fcntl(t->hpc.ticks_fd(), F_GETOWN) != tid) {
      FATAL() << "Scheduled task " << tid
              << " doesn't own hpc; replay divergence";
    }

    ticks_left = ticks - t->tick_count();
  }
  guard_overshoot(t, regs, ticks, ticks_left, ticks_slack, ignored_early_match,
                  ticks_left_at_ignored_early_match);

  /* Step 2: more slowly, find our way to the target ticks and
   * execution point.  We set an internal breakpoint on the
   * target $ip and then resume execution.  When that *internal*
   * breakpoint is hit (i.e., not one incidentally also set on
   * that $ip by the debugger), we check again if we're at the
   * target ticks and execution point.  If not, we temporarily
   * remove the breakpoint, single-step over the insn, and
   * repeat.
   *
   * What we really want to do is set a (precise)
   * retired-instruction interrupt and do away with all this
   * cruft. */
  while (true) {
    /* Invariants here are
     *  o ticks_left is up-to-date
     *  o ticks_left >= -ticks_slack
     *
     * Possible state of the execution of |t|
     *  0. at a debugger trap (breakpoint or stepi)
     *  1. at an internal breakpoint
     *  2. at the execution target
     *  3. not at the execution target, but incidentally
     *     at the target $ip
     *  4. otherwise not at the execution target
     *
     * Determining whether we're at a debugger trap is
     * surprisingly complicated, but we delegate the work
     * to |compute_debugger_trap()|.  The rest can be
     * straightforwardly computed with ticks value and
     * registers. */
    bool at_target = is_same_execution_point(
        t, regs, ticks_left, ticks_slack, &ignored_early_match,
        &ticks_left_at_ignored_early_match);
    if (SIGTRAP == t->child_sig) {
      TrapType trap_type =
          compute_trap_type(t, sig, NONDETERMINISTIC_SIG,
                            at_target ? AT_TARGET : NOT_AT_TARGET, stepi);
      switch (trap_type) {
        case TRAP_BKPT_USER:
        case TRAP_STEPI:
          /* Case (0) above: interrupt for the
           * debugger. */
          LOG(debug) << "    trap was debugger interrupt " << trap_type;
          if (did_set_internal_breakpoint) {
            t->vm()->remove_breakpoint(ip, TRAP_BKPT_INTERNAL);
            did_set_internal_breakpoint = false;
          }
          return INCOMPLETE;
        case TRAP_BKPT_INTERNAL: {
          /* Case (1) above: cover the tracks of
           * our internal breakpoint, and go
           * check again if we're at the
           * target. */
          LOG(debug) << "    trap was for target $ip";
          /* (The breakpoint would have trapped
           * at the $ip one byte beyond the
           * target.) */
          assert(!at_target);

          t->child_sig = 0;
          t->move_ip_before_breakpoint();
          /* We just backed up the $ip, but
           * rewound it over an |int $3|
           * instruction, which couldn't have
           * retired a branch.  So we don't need
           * to adjust |ticks_count()|. */
          continue;
        }
        case TRAP_NONE:
          /* Otherwise, we must have been forced
           * to single-step because the tracee's
           * $ip was incidentally the same as
           * the target.  Unfortunately, it's
           * awkward to assert that here, so we
           * don't yet.  TODO. */
          LOG(debug) << "    (SIGTRAP; stepi'd target $ip)";
          assert(stepi == RUN_CONTINUE);
          t->child_sig = 0;
          break;
      }
    }
    /* We had to keep the internal breakpoint set (if it
     * was when we entered the loop) for the checks above.
     * But now we're either done (at the target) or about
     * to resume execution in one of a variety of ways,
     * and it's simpler to start out knowing that the
     * breakpoint isn't set. */
    if (did_set_internal_breakpoint) {
      t->vm()->remove_breakpoint(ip, TRAP_BKPT_INTERNAL);
      did_set_internal_breakpoint = false;
    }

    if (at_target) {
      // Adjust dynamic ticks count to match trace, in case
      // there was slack.
      t->set_tick_count(ticks);
      /* Case (2) above: done. */
      return COMPLETE;
    }

    /* At this point, we've proven that we're not at the
     * target execution point, and we've ensured the
     * internal breakpoint is unset. */
    if (USE_BREAKPOINT_TARGET && regs.ip() != t->regs().ip()) {
      /* Case (4) above: set a breakpoint on the
       * target $ip and PTRACE_CONT in an attempt to
       * execute as many non-trapped insns as we
       * can.  (Unless the debugger is stepping, of
       * course.)  Trapping and checking
       * are-we-at-target is slow.  It bears
       * repeating that the ideal implementation
       * would be programming a precise counter
       * interrupt (insns-retired best of all), but
       * we're forced to be conservative by observed
       * imprecise counters.  This should still be
       * no slower than single-stepping our way to
       * the target execution point. */
      LOG(debug) << "    breaking on target $ip";
      t->vm()->set_breakpoint(ip, TRAP_BKPT_INTERNAL);
      did_set_internal_breakpoint = true;
      continue_or_step(t, stepi);
    } else {
      /* Case (3) above: we can't put a breakpoint
       * on the $ip, because resuming execution
       * would just trap and we'd be back where we
       * started.  Single-step past it. */
      LOG(debug) << "    (single-stepping over target $ip)";
      continue_or_step(t, RUN_SINGLESTEP);
    }

    if (PerfCounters::TIME_SLICE_SIGNAL == t->child_sig ||
        is_ignored_signal(t->child_sig)) {
      /* We don't usually expect a time-slice signal
       * during this phase, but it's possible for an
       * ignored signal to interrupt the previous
       * step just as the tracee enters the slack
       * region, i.e., where a ticks signal was just
       * about to fire.  (There's not really a
       * non-racy way to disable the ticks interrupt,
       * and we need to keep the counter running for
       * overshoot checking anyway.)  So this is the
       * most convenient way to squelch that
       * "spurious" signal. */
      t->child_sig = 0;
    }
    guard_unexpected_signal(t);

    /* Maintain the "'ticks_left'-is-up-to-date"
     * invariant. */
    ticks_left = ticks - t->tick_count();
    guard_overshoot(t, regs, ticks, ticks_left, ticks_slack,
                    ignored_early_match, ticks_left_at_ignored_early_match);
  }
}

/**
 * Emulates delivery of |sig| to |oldtask|.  Returns INCOMPLETE if
 * emulation was interrupted, COMPLETE if completed.
 */
Completion ReplaySession::emulate_signal_delivery(
    Task* oldtask, int sig, TraceFrame::Time stop_at_time) {
  // We are now at the exact point in the child where the signal
  // was recorded, emulate it using the next trace frame (records
  // the state at sighandler entry).
  advance_to_next_trace_frame(stop_at_time);
  Task* t = current_task();
  if (!t) {
    // Trace terminated abnormally.  We'll pop out to code
    // that knows what to do.
    return INCOMPLETE;
  }
  ASSERT(oldtask, t == oldtask) << "emulate_signal_delivery changed task";

  ASSERT(t, trace_frame.event().type == EV_SIGNAL_DELIVERY ||
                trace_frame.event().type == EV_SIGNAL_HANDLER)
      << "Unexpected signal disposition";
  // Entering a signal handler seems to clear FP/SSE registers for some
  // reason. So we saved those cleared values, and now we restore that
  // state so they're cleared during replay.
  if (trace_frame.event().type == EV_SIGNAL_HANDLER) {
    t->set_extra_regs(trace_frame.extra_regs());
  }

  /* Restore the signal-hander frame data, if there was one. */
  SignalDeterministic deterministic =
      Event(trace_frame.event()).Signal().deterministic;
  bool restored_sighandler_frame = 0 < t->set_data_from_trace();
  if (restored_sighandler_frame) {
    t->push_event(SignalEvent(sig, deterministic, t->arch()));
    t->ev().transform(EV_SIGNAL_DELIVERY);
    LOG(debug) << "--> restoring sighandler frame for " << signal_name(sig);
    t->ev().transform(EV_SIGNAL_HANDLER);
  } else if (possibly_destabilizing_signal(t, sig, deterministic)) {
    t->push_event(SignalEvent(sig, deterministic, t->arch()));
    t->ev().transform(EV_SIGNAL_DELIVERY);
    t->destabilize_task_group();
    t->pop_signal_delivery();
  }
  /* If this signal had a user handler, and we just set up the
   * callframe, and we need to restore the $sp for continued
   * execution. */
  t->set_regs(trace_frame.regs());
  /* Delivered the signal. */
  t->child_sig = 0;

  t->validate_regs();
  return COMPLETE;
}

void ReplaySession::check_ticks_consistency(Task* t, const Event& ev) {
  if (!can_validate() || trace_frame.event().has_exec_info == NO_EXEC_INFO) {
    return;
  }

  Ticks ticks_slack = get_ticks_slack(t);
  Ticks ticks_now = t->tick_count();
  Ticks trace_ticks = trace_frame.ticks();

  ASSERT(t, llabs(ticks_now - trace_ticks) <= ticks_slack)
      << "ticks mismatch for '" << ev << "'; expected " << trace_ticks
      << ", got " << ticks_now << "";
  // Sync task ticks with trace ticks so we don't keep accumulating errors
  t->set_tick_count(trace_ticks);
}

/**
 * Advance to the delivery of the deterministic signal |sig| and
 * update registers to what was recorded.  Return COMPLETE if successful or
 * INCOMPLETE  if an unhandled interrupt occurred.
 */
Completion ReplaySession::emulate_deterministic_signal(Task* t, int sig,
                                                       RunCommand stepi) {
  Event ev(trace_frame.event());

  continue_or_step(t, stepi);
  if (is_ignored_signal(t->child_sig)) {
    t->child_sig = 0;
    return emulate_deterministic_signal(t, sig, stepi);
  } else if (SIGTRAP == t->child_sig &&
             is_debugger_trap(t, sig, DETERMINISTIC_SIG, UNKNOWN, stepi)) {
    return INCOMPLETE;
  }
  ASSERT(t, t->child_sig == sig) << "Replay got unrecorded signal "
                                 << t->child_sig << " (expecting " << sig
                                 << ")";
  check_ticks_consistency(t, ev);

  if (EV_SEGV_RDTSC == ev.type()) {
    t->set_regs(trace_frame.regs());
    /* We just "delivered" this pseudosignal. */
    t->child_sig = 0;
    return COMPLETE;
  }

  current_step.action = TSTEP_DELIVER_SIGNAL;
  return INCOMPLETE;
}

/**
 * Run execution forwards for |t| until |t->trace.ticks| is reached,
 * and the $ip reaches the recorded $ip.  After that, deliver |sig| if
 * nonzero.  Return COMPLETE if successful or INCOMPLETE if an unhandled
 * interrupt occurred.
 */
Completion ReplaySession::emulate_async_signal(Task* t, int sig,
                                               RunCommand stepi, Ticks ticks) {
  if (advance_to(t, trace_frame.regs(), 0, stepi, ticks) == INCOMPLETE) {
    return INCOMPLETE;
  }
  if (sig) {
    current_step.action = TSTEP_DELIVER_SIGNAL;
    current_step.signo = sig;
    return INCOMPLETE;
  }
  return COMPLETE;
}

/**
 * Skip over the entry/exit of either an arm-desched-event or
 * disarm-desched-event ioctl(), as described by |ds|.  Return INCOMPLETE
 * if an unhandled interrupt occurred, COMPLETE if the ioctl() was
 * successfully skipped over.
 */
Completion ReplaySession::skip_desched_ioctl(Task* t, ReplayDeschedState* ds,
                                             RunCommand stepi) {
  /* Skip ahead to the syscall entry. */
  if (DESCHED_ENTER == ds->state &&
      cont_syscall_boundary(t, EMULATE, stepi) == INCOMPLETE) {
    return INCOMPLETE;
  }
  ds->state = DESCHED_EXIT;

  bool is_desched_syscall =
      (DESCHED_ARM == ds->type ? t->is_arm_desched_event_syscall()
                               : t->is_disarm_desched_event_syscall());
  ASSERT(t, is_desched_syscall) << "Failed to reach desched ioctl; at "
                                << t->syscall_name(
                                       t->regs().original_syscallno()) << "("
                                << t->regs().arg1() << ", " << t->regs().arg2()
                                << ") instead";
  /* Emulate a return value of "0".  It's OK for us to hard-code
   * that value here, because the syscallbuf lib aborts if a
   * desched ioctl returns non-zero (it doesn't know how to
   * handle that). */
  Registers r = t->regs();
  r.set_syscall_result(0);
  t->set_regs(r);
  t->finish_emulated_syscall();
  return COMPLETE;
}

/**
 * Restore the recorded syscallbuf data to the tracee, preparing the
 * tracee for replaying the records.  Return the number of record
 * bytes and a pointer to the first record through outparams.
 */
void ReplaySession::prepare_syscallbuf_records(Task* t) {
  if (!current_step.flush.need_buffer_restore) {
    return;
  }
  current_step.flush.need_buffer_restore = false;

  // Read the recorded syscall buffer back into the buffer
  // region.
  auto buf = t->trace_reader().read_raw_data();
  current_step.flush.num_rec_bytes_remaining = buf.data.size();

  assert(current_step.flush.num_rec_bytes_remaining <= SYSCALLBUF_BUFFER_SIZE);
  memcpy(syscallbuf_flush_buffer_array, buf.data.data(),
         current_step.flush.num_rec_bytes_remaining);

  // The stored num_rec_bytes in the header doesn't include the
  // header bytes, but the stored trace data does.
  current_step.flush.num_rec_bytes_remaining -= sizeof(struct syscallbuf_hdr);
  assert(buf.addr == t->syscallbuf_child);
  const syscallbuf_hdr* flush_hdr = syscallbuf_flush_buffer_hdr();
  assert(flush_hdr->num_rec_bytes ==
         current_step.flush.num_rec_bytes_remaining);

  current_step.flush.syscall_record_offset = 0;

  LOG(debug) << "Prepared " << current_step.flush.num_rec_bytes_remaining
             << " bytes of syscall records";
}

/**
 * Bail if |t| isn't at the buffered syscall |syscallno|.
 */
static void assert_at_buffered_syscall(Task* t, int syscallno) {
  ASSERT(t, t->is_in_untraced_syscall())
      << "Bad ip " << t->ip() << ": should have been buffered-syscall ip";
  ASSERT(t, t->regs().original_syscallno() == syscallno)
      << "At " << t->syscall_name(t->regs().original_syscallno())
      << "; should have been at " << t->syscall_name(syscallno) << "("
      << syscallno << ")";
}

/**
 * Bail if |rec_rec| and |rep_rec| haven't been prepared for the same
 * syscall (including desched'd-ness and reserved extra space).
 */
static void assert_same_rec(Task* t, const struct syscallbuf_record* rec_rec,
                            struct syscallbuf_record* rep_rec) {
  ASSERT(t, (rec_rec->syscallno == rep_rec->syscallno &&
             rec_rec->desched == rep_rec->desched &&
             rec_rec->size == rep_rec->size))
      << "Recorded rec { no=" << rec_rec->syscallno
      << ", desched:" << rec_rec->desched << ", size: " << rec_rec->size
      << " } "
      << "being replayed as { no=" << rep_rec->syscallno
      << ", desched:" << rep_rec->desched << ", size: " << rep_rec->size
      << " }";
}

/**
 * Directly restore the uaddr/uaddr2 outparams that were saved to
 * buffer.  Because the syscallbuf can't use scratch values for the
 * futexes, it can't restore the record data itself.
 */
static void restore_futex_words(Task* t, const struct syscallbuf_record* rec) {
  ssize_t extra_data_size = rec->size - sizeof(*rec);
  bool saved_uaddr2 = (2 * sizeof(uint32_t) == extra_data_size);
  ASSERT(t, sizeof(uint32_t) == extra_data_size || saved_uaddr2)
      << "Futex should have saved 4 or 8 bytes, but instead saved "
      << extra_data_size;

  remote_ptr<int> child_uaddr = t->regs().arg1();
  auto rec_uaddr = *reinterpret_cast<const int*>(rec->extra_data);
  t->write_mem(child_uaddr, rec_uaddr);

  if (saved_uaddr2) {
    remote_ptr<int> child_uaddr2 = t->regs().arg5();
    auto rec_uaddr2 =
        *reinterpret_cast<const int*>(rec->extra_data + sizeof(int));
    t->write_mem(child_uaddr2, rec_uaddr2);
  }
}

/**
 * Try to flush one buffered syscall as described by |flush|.  Return
 * INCOMPLETE if an unhandled interrupt occurred, and COMPLETE if the syscall
 * was flushed (in which case |flush->state == DONE|).
 */
Completion ReplaySession::flush_one_syscall(Task* t, RunCommand stepi) {
  const syscallbuf_hdr* flush_hdr = syscallbuf_flush_buffer_hdr();
  const struct syscallbuf_record* rec_rec =
      (const struct syscallbuf_record*)((uint8_t*)flush_hdr->recs +
                                        current_step.flush
                                            .syscall_record_offset);
  struct syscallbuf_record* child_rec =
      (struct syscallbuf_record*)((uint8_t*)t->syscallbuf_hdr->recs +
                                  current_step.flush.syscall_record_offset);
  int call = rec_rec->syscallno;
  // TODO: use syscall_defs table information to determine this.
  ExecOrEmulate emu = is_madvise_syscall(call, t->arch()) ? EXEC : EMULATE;

  switch (current_step.flush.state) {
    case FLUSH_START:
      ASSERT(t, 0 == ((uintptr_t)rec_rec & (sizeof(int) - 1)))
          << "Recorded record must be int-aligned, but instead is " << rec_rec;
      ASSERT(t, 0 == ((uintptr_t)child_rec & (sizeof(int) - 1)))
          << "Replaying record must be int-aligned, but instead is %p"
          << child_rec;
      ASSERT(t, 0 == rec_rec->desched || 1 == rec_rec->desched)
          << "Recorded record is corrupted: rec->desched is "
          << rec_rec->desched;
      // We'll check at syscall entry that the recorded and
      // replayed record values match.

      LOG(debug) << "Replaying buffered `" << t->syscall_name(call)
                 << "' (ret:" << rec_rec->ret << ") which does"
                 << (!rec_rec->desched ? " not" : "") << " use desched event";

      if (!rec_rec->desched) {
        current_step.flush.state = FLUSH_ENTER;
      } else {
        current_step.flush.state = FLUSH_ARM;
        current_step.flush.desched.type = DESCHED_ARM;
        current_step.flush.desched.state = DESCHED_ENTER;
      }
      return flush_one_syscall(t, stepi);

    case FLUSH_ARM:
      /* Skip past the ioctl that armed the desched
       * notification. */
      LOG(debug) << "  skipping over arm-desched ioctl";
      if (skip_desched_ioctl(t, &current_step.flush.desched, stepi) ==
          INCOMPLETE) {
        return INCOMPLETE;
      }
      current_step.flush.state = FLUSH_ENTER;
      return flush_one_syscall(t, stepi);

    case FLUSH_ENTER:
      LOG(debug) << "  advancing to buffered syscall entry";
      if (cont_syscall_boundary(t, emu, stepi) == INCOMPLETE) {
        return INCOMPLETE;
      }
      assert_at_buffered_syscall(t, call);
      assert_same_rec(t, rec_rec, child_rec);
      current_step.flush.state = FLUSH_EXIT;
      return flush_one_syscall(t, stepi);

    case FLUSH_EXIT: {
      LOG(debug) << "  advancing to buffered syscall exit";

      EmuFs::AutoGc gc(*this, t->arch(), call, SYSCALL_EXIT);

      assert_at_buffered_syscall(t, call);

      // Restore saved trace data.
      memcpy(child_rec->extra_data, rec_rec->extra_data, rec_rec->size);

      // Restore return value.
      // TODO: try to share more code with cont_syscall_boundary()
      if (emu == EXEC) {
        if (cont_syscall_boundary(t, emu, stepi) == INCOMPLETE) {
          return INCOMPLETE;
        }
        assert_at_buffered_syscall(t, call);
      }
      Registers r = t->regs();
      r.set_syscall_result(rec_rec->ret);
      t->set_regs(r);
      if (emu == EMULATE) {
        t->finish_emulated_syscall();
      }

      if (is_futex_syscall(call, t->arch())) {
        restore_futex_words(t, rec_rec);
      } else if (is_write_syscall(call, t->arch())) {
        rep_maybe_replay_stdio_write(t);
      }

      if (!rec_rec->desched) {
        current_step.flush.state = FLUSH_DONE;
        return COMPLETE;
      }
      current_step.flush.state = FLUSH_DISARM;
      current_step.flush.desched.type = DESCHED_DISARM;
      current_step.flush.desched.state = DESCHED_ENTER;
      return flush_one_syscall(t, stepi);
    }
    case FLUSH_DISARM:
      /* And skip past the ioctl that disarmed the desched
       * notification. */
      LOG(debug) << "  skipping over disarm-desched ioctl";
      if (skip_desched_ioctl(t, &current_step.flush.desched, stepi) ==
          INCOMPLETE) {
        return INCOMPLETE;
      }
      current_step.flush.state = FLUSH_DONE;
      return COMPLETE;

    default:
      FATAL() << "Unknown buffer-flush state " << current_step.flush.state;
      return COMPLETE; /* unreached */
  }
}

/**
 * Replay all the syscalls recorded in the interval between |t|'s
 * current execution point and the next non-syscallbuf event (the one
 * that flushed the buffer).  Return COMPLETE if successful or INCOMPLETE if an
 * unhandled interrupt occurred.
 */
Completion ReplaySession::flush_syscallbuf(Task* t, RunCommand stepi) {
  prepare_syscallbuf_records(t);

  const syscallbuf_hdr* flush_hdr =
      t->replay_session().syscallbuf_flush_buffer_hdr();

  while (current_step.flush.num_rec_bytes_remaining > 0) {
    if (flush_one_syscall(t, stepi) == INCOMPLETE) {
      return INCOMPLETE;
    }

    assert(FLUSH_DONE == current_step.flush.state);

    const struct syscallbuf_record* record =
        (const struct syscallbuf_record*)((uint8_t*)flush_hdr->recs +
                                          current_step.flush
                                              .syscall_record_offset);
    size_t stored_rec_size = stored_record_size(record->size);
    current_step.flush.syscall_record_offset += stored_rec_size;
    current_step.flush.num_rec_bytes_remaining -= stored_rec_size;
    current_step.flush.state = FLUSH_START;

    LOG(debug) << "  " << current_step.flush.num_rec_bytes_remaining
               << " bytes remain to flush";
  }
  return COMPLETE;
}

Completion ReplaySession::patch_next_syscall(Task* t, RunCommand stepi) {
  if (cont_syscall_boundary(t, EMULATE, stepi) == INCOMPLETE) {
    return INCOMPLETE;
  }
  bool did_patch = t->vm()->monkeypatcher().try_patch_syscall(t);
  ASSERT(t, did_patch) << "Should have patched the syscall, but did not!";
  return COMPLETE;
}

/** Return true when replaying/debugging should cease after |t| exits. */
bool ReplaySession::is_last_interesting_task(Task* t) {
  return (0 == debugged_tgid() && tasks().size() == 1) ||
         (t->tgid() == debugged_tgid() &&
          t->task_group()->task_set().size() == 1);
}

/**
 * Return true if replaying |ev| by running |step| should result in
 * the target task having the same ticks value as it did during
 * recording.
 */
static bool has_deterministic_ticks(const Event& ev,
                                    const ReplayTraceStep& step) {
  if (ev.has_ticks_slop()) {
    return false;
  }
  // We won't necessarily reach the same ticks when replaying an
  // async signal, due to debugger interrupts and other
  // implementation details.  This is checked in |advance_to()|
  // anyway.
  return TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT != step.action;
}

/**
 * Try to execute |step|, adjusting for |req| if needed.  Return COMPLETE if
 * |step| was made, or INCOMPLETE if there was a trap or |step| needs
 * more work.
 */
Completion ReplaySession::try_one_trace_step(Task* t, RunCommand stepi,
                                             TraceFrame::Time stop_at_time) {
  switch (current_step.action) {
    case TSTEP_RETIRE:
      return COMPLETE;
    case TSTEP_ENTER_SYSCALL:
      return enter_syscall(t, stepi);
    case TSTEP_EXIT_SYSCALL:
      return exit_syscall(t, stepi);
    case TSTEP_DETERMINISTIC_SIGNAL:
      return emulate_deterministic_signal(t, current_step.signo, stepi);
    case TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT:
      return emulate_async_signal(t, current_step.target.signo, stepi,
                                  current_step.target.ticks);
    case TSTEP_DELIVER_SIGNAL:
      return emulate_signal_delivery(t, current_step.signo, stop_at_time);
    case TSTEP_FLUSH_SYSCALLBUF:
      return flush_syscallbuf(t, stepi);
    case TSTEP_PATCH_SYSCALL:
      return patch_next_syscall(t, stepi);
    case TSTEP_DESCHED:
      return skip_desched_ioctl(t, &current_step.desched, stepi);
    default:
      FATAL() << "Unhandled step type " << current_step.action;
      return COMPLETE;
  }
}

/**
 * Set up rep_trace_step state in t's Session to start replaying towards
 * the event given by the session's current_trace_frame --- but only if
 * it's not already set up.
 * Return true if we should continue replaying, false if the debugger
 * requested a restart. If this returns false, t's Session state was not
 * modified.
 */
void ReplaySession::setup_replay_one_trace_frame(Task* t) {
  Event ev(trace_frame.event());

  LOG(debug) << "[line " << trace_frame.time() << "] " << t->rec_tid
             << ": replaying " << Event(ev) << "; state "
             << state_name(trace_frame.event().state);
  if (t->syscallbuf_hdr) {
    LOG(debug) << "    (syscllbufsz:" << t->syscallbuf_hdr->num_rec_bytes
               << ", abrtcmt:" << t->syscallbuf_hdr->abort_commit
               << ", locked:" << t->syscallbuf_hdr->locked << ")";
  }

  if (t->child_sig != 0) {
    assert(EV_SIGNAL == ev.type() && t->child_sig == ev.Signal().number);
    t->child_sig = 0;
  }

  /* Ask the trace-interpretation code what to do next in order
   * to retire the current frame. */
  memset(&current_step, 0, sizeof(current_step));

  switch (ev.type()) {
    case EV_UNSTABLE_EXIT:
      t->unstable = true;
    /* fall through */
    case EV_EXIT: {
      if (is_last_interesting_task(t)) {
        LOG(debug) << "last interesting task in " << debugged_tgid() << " is "
                   << t->rec_tid << " (" << t->tid << ")";
        set_last_task(t);
        return;
      }

      /* If the task was killed by a terminating signal,
       * then it may have ended abruptly in a syscall or at
       * some other random execution point.  That's bad for
       * replay, because we detach from the task after we
       * replay its "exit".  Since we emulate signal
       * delivery, the task may happily carry on with
       * (non-emulated!) execution after we detach.  That
       * execution might include things like |rm -rf ~|.
       *
       * To ensure that the task really dies, we send it a
       * terminating signal here.  One would like to use
       * SIGKILL, but for not-understood reasons that causes
       * shutdown hangs when joining the exited tracee.
       * Other terminating signals have not been observed to
       * hang, so that's what's used here.. */
      syscall(SYS_tkill, t->tid, SIGABRT);
      // TODO dissociate address space from file table
      bool file_table_dying = (1 == t->vm()->task_set().size());
      delete t;
      /* Early-return because |t| is gone now. */
      if (file_table_dying) {
        gc_emufs();
      }
      return;
    }
    case EV_DESCHED:
      current_step.action = TSTEP_DESCHED;
      current_step.desched.type = ARMING_DESCHED_EVENT == ev.Desched().state
                                      ? DESCHED_ARM
                                      : DESCHED_DISARM;
      current_step.desched.state = DESCHED_ENTER;
      break;
    case EV_SYSCALLBUF_ABORT_COMMIT:
      t->syscallbuf_hdr->abort_commit = 1;
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_SYSCALLBUF_FLUSH:
      current_step.action = TSTEP_FLUSH_SYSCALLBUF;
      current_step.flush.need_buffer_restore = true;
      current_step.flush.num_rec_bytes_remaining = 0;
      break;
    case EV_SYSCALLBUF_RESET:
      t->syscallbuf_hdr->num_rec_bytes = 0;
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_PATCH_SYSCALL:
      current_step.action = TSTEP_PATCH_SYSCALL;
      break;
    case EV_SCHED:
      current_step.action = TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT;
      current_step.target.ticks = trace_frame.ticks();
      current_step.target.signo = 0;
      break;
    case EV_SEGV_RDTSC:
      current_step.action = TSTEP_DETERMINISTIC_SIGNAL;
      current_step.signo = SIGSEGV;
      break;
    case EV_INTERRUPTED_SYSCALL_NOT_RESTARTED:
      LOG(debug) << "  popping interrupted but not restarted " << t->ev();
      t->pop_syscall_interruption();
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_EXIT_SIGHANDLER:
      LOG(debug) << "<-- sigreturn from " << t->ev();
      t->pop_signal_handler();
      current_step.action = TSTEP_RETIRE;
      break;
    case EV_SIGNAL:
      current_step.signo = ev.Signal().number;
      current_step.action = ev.Signal().deterministic == DETERMINISTIC_SIG
                                ? TSTEP_DETERMINISTIC_SIGNAL
                                : TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT;
      if (TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT == current_step.action) {
        current_step.target.signo = current_step.signo;
        current_step.target.ticks = trace_frame.ticks();
      }
      break;
    case EV_SYSCALL:
      rep_process_syscall(t, &current_step);
      break;
    default:
      FATAL() << "Unexpected event " << ev;
  }
}

ReplaySession::ReplayResult ReplaySession::replay_step(
    RunCommand command, TraceFrame::Time stop_at_time) {
  ReplayResult result;

  Task* t = current_task();

  if (EV_TRACE_TERMINATION == trace_frame.event().type) {
    set_last_task(t);
    result.status = REPLAY_EXITED;
    return result;
  }

  result.status = REPLAY_CONTINUE;
  result.break_status.reason = BREAK_NONE;

  /* If we restored from a checkpoint, the steps might have been
   * computed already in which case step.action will not be TSTEP_NONE.
   */
  if (current_step.action == TSTEP_NONE) {
    setup_replay_one_trace_frame(t);
    if (current_step.action == TSTEP_NONE) {
      // Already at the destination event.
      if (last_task()) {
        result.status = REPLAY_EXITED;
      } else {
        advance_to_next_trace_frame(stop_at_time);
      }
      return result;
    }
  }

  /* Advance towards fulfilling |current_step|. */
  ReplayTraceStepType current_action = current_step.action;
  if (try_one_trace_step(t, command, stop_at_time) == INCOMPLETE) {
    if (EV_TRACE_TERMINATION == trace_frame.event().type) {
      // An irregular trace step had to read the
      // next trace frame, and that frame was an
      // early-termination marker.  Otherwise we
      // would have seen the marker at
      // |schedule_task()|.
      set_last_task(t);
      result.status = REPLAY_EXITED;
      return result;
    }

    // We got INCOMPLETE because there was some kind of debugger trap.
    if (current_step.action != current_action &&
        current_step.action == TSTEP_DELIVER_SIGNAL) {
      result.break_status.reason = BREAK_SIGNAL;
      result.break_status.task = t;
      result.break_status.signal = current_step.signo;
    } else {
      result.break_status = diagnose_debugger_trap(t, t->child_sig);
      ASSERT(t, result.break_status.reason != BREAK_SIGNAL)
          << "Expected either SIGTRAP at $ip " << t->ip()
          << " or USER breakpoint just after it";
      ASSERT(t, result.break_status.reason != BREAK_SINGLESTEP ||
                    command == RUN_SINGLESTEP);
    }

    /* Don't restart with SIGTRAP anywhere. */
    t->child_sig = 0;
    return result;
  }

  if (TSTEP_ENTER_SYSCALL == current_step.action) {
    rep_after_enter_syscall(t, current_step.syscall.number);
    cpuid_bug_detector.notify_reached_syscall_during_replay(t);
  }
  if (can_validate() && SYSCALL_EXIT == trace_frame.event().state &&
      Flags::get().check_cached_mmaps) {
    t->vm()->verify(t);
  }

  Event ev(trace_frame.event());
  if (has_deterministic_ticks(ev, current_step)) {
    check_ticks_consistency(t, ev);
  }

  debug_memory(t);

  if (command == RUN_SINGLESTEP && EV_SEGV_RDTSC == ev.type()) {
    // We completed this RDTSC event, and that counts as a completed singlestep.
    result.break_status.reason = BREAK_SINGLESTEP;
  }

  // Record that this step completed successfully.
  current_step.action = TSTEP_NONE;
  advance_to_next_trace_frame(stop_at_time);

  return result;
}
