/* This file is part of the KDE project
   Copyright (C) 2006 Tom Albers <tomalbers@kde.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kglobalsettings.h>
#include <klocale.h>
#include <dcopclient.h>

#include <math.h>

#include "rsistats.h"
#include "rsiglobals.h"

RSIGlobals *RSIGlobals::m_instance = 0;
RSIStats *RSIGlobals::m_stats = 0;

RSIGlobals::RSIGlobals( QObject *parent, const char *name )
: QObject( parent, name )
{
    resetUsage();
    slotReadConfig();
}

RSIGlobals::~RSIGlobals()
{
    delete m_stats;
    m_stats = 0L;
}

RSIGlobals *RSIGlobals::instance()
{
    if ( !m_instance )
    {
      m_instance = new RSIGlobals();
      m_stats = new RSIStats();
    }

    return m_instance;
}

QString RSIGlobals::formatSeconds( const int seconds )
{
    int mins, secs, hours, remaining;
    QString hString, mString1, mString2, sString1, sString2;

    remaining = seconds;

    hours = (int)floor(remaining/3600);
    remaining = remaining-(hours*3600);

    mins = (int)floor(remaining/60);
    secs = remaining-(mins*60);

    hString = i18n("One hour","%n hours", hours);
    mString1 = i18n("One minute","%n minutes", mins);
    mString2 = i18n("one minute","%n minutes", mins);
    sString1 = i18n("One second","%n seconds", secs);
    sString2 = i18n("one second","%n seconds", secs);

    if ( hours > 0 && mins >0 )
        return(i18n("Arguments: hours, minutes "
                    "both as you defined earlier",
                    "%1 and %2").arg(hString, mString2) );
    else if ( hours > 0 && mins == 0 )
        return( hString );

    else if ( hours == 0 )
    {
        if (mins > 0 && secs > 0)
            return(i18n("Arguments: minutes, seconds "
                        "both as you defined earlier",
                        "%1 and %2").arg(mString1, sString2) );

        else if ( mins == 0 && secs > 0 )
            return( sString1 );

        else if ( mins >0 && secs == 0 )
            return( mString1 );

        else
            return( i18n("0 seconds") );
    }

    return( QString::null ); //should not get here
}

void RSIGlobals::slotReadConfig()
{
    KConfig* config = kapp->config();

    config->setGroup("General Settings");

    m_intervals["tiny_minimized"] = config->readNumEntry("TinyInterval", 10)*60;
    m_intervals["tiny_maximized"] = config->readNumEntry("TinyDuration", 20);
    m_intervals["big_minimized"] = config->readNumEntry("BigInterval", 60)*60;
    m_intervals["big_maximized"] = config->readNumEntry("BigDuration", 1)*60;
    m_intervals["big_maximized"] = config->readNumEntry("BigDuration", 1)*60;

    if (config->readBoolEntry("DEBUG", false))
    {
        kdDebug() << "Debug mode activated" << endl;
        m_intervals["tiny_minimized"] = m_intervals["tiny_minimized"]/60;
        m_intervals["big_minimized"] = m_intervals["big_minimized"]/60;
        m_intervals["big_maximized"] = m_intervals["big_maximized"]/60;
    }

    //--------------- read the DCOP settings
    m_dcopstartbig.clear();
    m_dcopstarttiny.clear();
    m_dcopendbig.clear();
    m_dcopendtiny.clear();

    QMap<QString,QString> map;
    map=config->entryMap("DCOP");

    QMap<QString, QString>::const_iterator i;
    for (i = map.constBegin(); i != map.constEnd(); ++i)
    {
        QStringList list = QStringList::split(",",i.data());

        if (list[2] == "On")    // only active ones
            if (list[3] != "true") // tiny breaks as well?
            {
                m_dcopstarttiny.append(list[0]);
                m_dcopendtiny.append(list[1]);
            }
            m_dcopstartbig.append(list[0]);
            m_dcopendbig.append(list[1]);
    }
}

QColor RSIGlobals::getTinyBreakColor( int secsToBreak ) const
{
    int minimized = m_intervals["tiny_minimized"];
    double v = 100 * secsToBreak / (double)minimized;

    v = v > 100 ? 100 : v;
    v = v < 0 ? 0 : v;

    return QColor ( (int)(255 - 2.55 * v), (int)(1.60 * v), 0 );
}

QColor RSIGlobals::getBigBreakColor( int secsToBreak ) const
{
    int minimized = m_intervals["big_minimized"];
    double v = 100 * secsToBreak / (double)minimized;

    v = v > 100 ? 100 : v;
    v = v < 0 ? 0 : v;

    return QColor ( (int)(255 - 2.55 * v), (int)(1.60 * v), 0 );
}

void RSIGlobals::resetUsage()
{
    m_usageArray.fill( false, 60 * 60 * 24 );
}

void RSIGlobals::executeDCOP(const QString &command)
{
    /*
            Syntax without data: "kopete" "KopeteIface" "setAway()"
            Syntax with data: "kopete" "KopeteIface" "setAway(QString)" "someStringContainingData"
    */

    QString _command = command;

    if ( command.startsWith( "dcop " ) )
        _command = _command.remove( 0, 5 );

    kdDebug() << "Execute " << _command << endl;
    QCString app =  _command.section(' ',0,0).utf8();
    QCString obj =  _command.section(' ',1,1).utf8();
    QCString fun =  _command.section(' ',2,2).utf8();
    QCString data = _command.section(' ',3,3).utf8();

    if ( app != "rsibreak" )
    {
        if ( data.isEmpty() && fun.right(2) != "()" )
            fun += "()";

        if (!kapp->dcopClient()->send(app,obj,fun, data))
            kdDebug() << "Command exectution failed" << endl;
    }
}

void RSIGlobals::DCOPBreak(bool start, bool big)
{
    QStringList commands;
    if (start)
        big ? commands=m_dcopstartbig : commands=m_dcopstarttiny;
    else
        big ? commands=m_dcopendbig : commands=m_dcopendtiny;

    for( QStringList::Iterator it = commands.begin(); it != commands.end() ;
         ++it)
    {
        executeDCOP(*it);
    }
}

#include "rsiglobals.moc"
