/*************************************************************************** 
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              * 
 *                                                                         * 
 *   This program is free software; you can redistribute it and/or modify  * 
 *   it under the terms of the GNU General Public License as published by  * 
 *   the Free Software Foundation; either version 2 of the License, or     * 
 *   (at your option) any later version.                                   * 
 *                                                                         * 
 *   This program is distributed in the hope that it will be useful,       * 
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        * 
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         * 
 *   GNU General Public License for more details.                          * 
 *                                                                         * 
 *   You should have received a copy of the GNU General Public License     * 
 *   along with this program; if not, write to the                         * 
 *   Free Software Foundation, Inc.,                                       * 
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             * 
 *                                                                         * 
 *   Licensed under the GNU GPL                                            * 
 *   Original code supplied under license from RaLink Inc, 2003.           * 
 ***************************************************************************/ 

 /*************************************************************************** 
 *      Module Name: rtmp_main.c 
 *              
 *      Abstract:  
 *              
 *      Revision History: 
 *      Who             When            What 
 *      --------        -----------     ----------------------------- 
 *      PaulL           25th Nov 02     Created
 *      MarkW           9th  Feb 04     Baseline of code
 *      defekt          13th Feb 04     ATOMIC kmalloc fix
 *      MarkW           13th Mar 04     EXPORT_NO_SYMBOLS 2.6 fix
 *      JerzyK          1st  May 04     Fixes to interrupt handler
 *      fstanchina      30th May 04     Made interrupt fix 2.4 ok
 *      fstanchina      07th Jun 04     Remove local alloc_netdev()
 *      fstanchina      07th Jun 04     Print some info at init time
 *      MarkW           14th Aug 04     Promisc enable ioctl code 
 *      hpernu		16th Aug 04     AMD64 Support
 *	MarkW		19th Sep 04	Removed annoying DEBUG output 
 *	RobinC(RT2500)  16th Dec 04     support ifpreup scripts
 *      Bruno           31st Jan 04     Network device name module param
 *      MarkW           21st Feb 05     Changes to MLME init timing
 *      MeelisR         2nd  Mar 04     PCI management fixes
 *      MichalL         5th  Mar 04     BitKeeper slot_name fix
 ***************************************************************************/ 

 
#include "rt_config.h"

//	Global static variable, Debug level flag
// Don't hide this behind debug define. There should be as little difference between debug and no-debug as possible.
int	debug = 0;	/* Default is off. */
MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug, "Enable level: accepted values: 1 to switch debug on, 0 to switch debug off.");

static char *ifname = NULL;
MODULE_PARM(ifname, "s");
MODULE_PARM_DESC(ifname, "Network device name (default ra%d)");

// The driver version information
static char version[] __devinitdata =
KERN_INFO DRV_NAME " " DRV_VERSION " " DRV_RELDATE "  http://rt2400.sourceforge.net\n";

// Following information will be show when you run 'modinfo'
MODULE_AUTHOR("http://rt2400.sourceforge.net");
MODULE_DESCRIPTION("Ralink RT2400 802.11b WLAN driver " DRV_VERSION " " DRV_RELDATE);
MODULE_LICENSE("GPL");

static INT __devinit RT2400_init_one (
	IN	struct pci_dev				*pPci_Dev,
	IN	const struct pci_device_id	*ent)
{
	INT rc;

	// wake up and enable device
	if (pci_enable_device (pPci_Dev))
	{
		rc = -EIO;
	}
	else
	{
		rc = RT2400_probe(pPci_Dev, ent);		
                if (rc)
                    pci_disable_device(pPci_Dev);
	}
	return rc;
}

//
// PCI device probe & initialization function
//
INT __devinit   RT2400_probe(
	IN	struct pci_dev			*pPci_Dev, 
	IN	const struct pci_device_id	*ent)
{
	struct	net_device		*net_dev;
	RTMP_ADAPTER			*pAd;
	CHAR					*print_name;
	INT						chip_id = (int) ent->driver_data;
	/*	ULONG*/void *					csr_addr;
	CSR3_STRUC				StaMacReg0;
	CSR4_STRUC				StaMacReg1;
	INT						Status = -ENODEV;
	
	printk(version);

        print_name = pPci_Dev ? pci_name(pPci_Dev) : "rt2400";

	// alloc_etherdev() will set net_dev->name to "eth%d"
	net_dev = alloc_etherdev(sizeof(RTMP_ADAPTER));
	if (net_dev == NULL) 
	{
		DBGPRINT(RT_DEBUG_TRACE, "init_ethernet failed\n");
		goto err_out;
	}
	
	SET_MODULE_OWNER(net_dev);

        #if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0))
           SET_NETDEV_DEV(net_dev, &(pPci_Dev->dev));
        #endif
		
	if (pci_request_regions(pPci_Dev, print_name))
		goto err_out_free_netdev;
	
	// Interrupt IRQ number
	net_dev->irq = pPci_Dev->irq;
	
	// map physical address to virtual address for accessing register
	csr_addr =  ioremap(pci_resource_start(pPci_Dev, 0), pci_resource_len(pPci_Dev, 0));
	if (!csr_addr) 
	{
		DBGPRINT(RT_DEBUG_TRACE, "ioremap failed for device %s, region 0x%X @ 0x%lX\n",
			print_name, (ULONG)pci_resource_len(pPci_Dev, 0), pci_resource_start(pPci_Dev, 0));
		goto err_out_free_mmio_region;
	}

	// Save CSR virtual address and irq to device structure
	net_dev->base_addr = (unsigned long) csr_addr;
	pAd = netdev_priv(net_dev);
	pAd->CSRBaseAddress = net_dev->base_addr;
	pAd->net_dev = net_dev;

	// Set DMA master
	pci_set_master(pPci_Dev);

	// Read MAC address
	// 3. Read MAC address
	RTMP_IO_READ32(pAd, CSR3, &StaMacReg0.word);
	RTMP_IO_READ32(pAd, CSR4, &StaMacReg1.word);
	net_dev->dev_addr[0] = StaMacReg0.field.Byte0;
	net_dev->dev_addr[1] = StaMacReg0.field.Byte1;
	net_dev->dev_addr[2] = StaMacReg0.field.Byte2;
	net_dev->dev_addr[3] = StaMacReg0.field.Byte3;
	net_dev->dev_addr[4] = StaMacReg1.field.Byte4;
	net_dev->dev_addr[5] = StaMacReg1.field.Byte5;
	
	pAd->chip_id = chip_id;
	pAd->pPci_Dev = pPci_Dev;

	// The chip-specific entries in the device structure.
	net_dev->open = RT2400_open;
	net_dev->hard_start_xmit = RTMPSendPackets;
	net_dev->stop = RT2400_close;
	net_dev->get_stats = RT2400_get_ether_stats;

#if WIRELESS_EXT >= 12
	net_dev->get_wireless_stats = RT2400_get_wireless_stats;
#endif

	net_dev->set_multicast_list = RT2400_set_rx_mode;
	net_dev->do_ioctl = RT2400_ioctl;
	net_dev->set_mac_address = rt2400_set_mac_address;

	// register_netdev() will call dev_alloc_name() for us
	// TODO: remove the following line to keep the default "eth%d" (see above)
        if (ifname == NULL)
             strcpy(net_dev->name, "ra%d");
        else 
             strncpy(net_dev->name, ifname, IFNAMSIZ);

	// Register this device
	Status = register_netdev(net_dev);
	if (Status)
		goto err_out_unmap;

	printk(KERN_INFO DRV_NAME ": %s at 0x%lx, VA 0x%1lx, IRQ %d.\n",
		net_dev->name, pci_resource_start(pPci_Dev, 0), (unsigned long)csr_addr, pPci_Dev->irq);

	// Set driver data
	pci_set_drvdata(pPci_Dev, net_dev);

	// moved to here by RobinC so if-preup can work
	// When driver now loads it is loaded up with "factory" defaults
	// All this occurs while the net iface is down
	// iwconfig can then be used to configure card BEFORE
	// ifconfig ra0 up is applied.
	// Note the RT2500STA.dat file will still overwrite settings 
	// but it is useful for the settings iwconfig doesn't let you at
	PortCfgInit(pAd);

        // initialize MLME
        Status = MlmeInit(pAd);
        if (Status)
              goto err_out;

	return 0;

err_out_unmap:
	iounmap((void *)csr_addr);
err_out_free_mmio_region:
	pci_release_regions(pPci_Dev);	
err_out_free_netdev:
	kfree (net_dev);
err_out:
	return Status;
}


INT RT2400_open(
	IN	struct net_device *net_dev)
{
	PRTMP_ADAPTER	pAd = netdev_priv(net_dev);
	INT		status;

	// 1. allocate DMA buffers
	status = RTMPAllocDMAMemory(pAd);
	if (status)
		return status;
	
	// 2. request interrupt
	//
	// Disable interrupts here which is as soon as possible
	// This statement should never be true. We might consider to remove it later
	//
	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_ACTIVE))
	{
		NICDisableInterrupt(pAd);
	}

	status = request_irq(pAd->pPci_Dev->irq, RTMPIsr, SA_SHIRQ, net_dev->name, net_dev);
	if (status)
		return status;
	RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_IN_USE);

	// 3. Read MAC address
	NICReadAdapterInfo(pAd);

	DBGPRINT(RT_DEBUG_TRACE, "%s: RT2400_open() irq %d. MAC = %02x:%02x:%02x:%02x:%02x:%02x \n",
		net_dev->name, pAd->pPci_Dev->irq, pAd->CurrentAddress[0], pAd->CurrentAddress[1], pAd->CurrentAddress[2],
		pAd->CurrentAddress[3], pAd->CurrentAddress[4], pAd->CurrentAddress[5]);

	NICInitTransmit(pAd);

	// load in data from RT2500STA.dat file
	// note this will TRASH any preup settings applied
	// if the parameter exists in the file
	RTMPReadParametersFromFile(pAd);

	if(status != NDIS_STATUS_SUCCESS) {
	    return status;
	}

	pAd->MediaState = NdisMediaStateDisconnected;
	RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_MEDIA_STATE_CHANGE);
	
	pAd->NumberOfMcAddresses = 0;
	memset(&pAd->McastTable, 0x00, sizeof(pAd->McastTable));
	
	
	// Initialize Asics
	NICInitializeAdapter(pAd);

	NICReadEEPROMParameters(pAd);

	NICInitAsicFromEEPROM(pAd);
		
	// Start net interface tx /rx
	netif_start_queue(net_dev);

	// Set the timer to check for link beat.
	init_timer(&pAd->timer);
	pAd->timer.expires = jiffies + DEBUG_TASK_DELAY;
	pAd->timer.data = (unsigned long)net_dev;
	pAd->timer.function = &RT2400_timer;				// timer handler
	add_timer(&pAd->timer);

	netif_carrier_on(net_dev);

	// Enable interrupt
	RTMP_IO_WRITE32(pAd, CSR7, 0xffffffff); // clear garbage interrupts
	NICEnableInterrupt(pAd);

	return 0;
}

VOID RT2400_timer(
	IN	unsigned long data)
{
	struct net_device	*net_dev = (struct net_device *)data;
	RTMP_ADAPTER		*pAd = netdev_priv(net_dev);

//	NICCheckForHang(pAd);
	
	pAd->timer.expires = jiffies + DEBUG_TASK_DELAY;
	add_timer(&pAd->timer);
}

/*
	========================================================================
	
	Routine Description:
		hard_start_xmit handler

	Arguments:
		skb				point to sk_buf which upper layer transmit
		net_dev			point to net_dev
	Return Value:
		None

	Note:
	
	========================================================================
*/
INT	RTMPSendPackets(
	IN	struct sk_buff *skb,
	IN	struct net_device *net_dev)
{
	RTMP_ADAPTER	*pAd = netdev_priv(net_dev);
	NDIS_STATUS		Status = NDIS_STATUS_SUCCESS;
	
	if (!INFRA_ON(pAd) && !ADHOC_ON(pAd))
	{
		// Drop send request since there are no physical connection yet
		// Check the association status for infrastructure mode
		// And Mibss for Ad-hoc mode setup
		dev_kfree_skb_irq(skb);
	}
	else
	{
		if (skb->len > TX_BUFFER_SIZE)
		{
			// packet too long, drop it
			dev_kfree_skb_irq(skb);
			pAd->stats.tx_dropped++;
		}

		net_dev->trans_start = jiffies;
		Status = RTMPSendPacket(net_dev, skb);
		
		if (Status != NDIS_STATUS_SUCCESS)
		{
			//
			// Errors before enqueue stage
			//
			dev_kfree_skb_irq(skb);			
		}
	}

	// Dequeue one frame from SendTxWait queue and process it
	// There are two place calling dequeue for TX ring.
	// 1. Here, right after queueing the frame.
	// 2. At the end of TxRingTxDone service routine.
	if ((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BSS_SCAN_IN_PROGRESS)) && 
		(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)) &&
		(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS)))
	{
		RTMPDeQueuePacket(pAd);
	}

	return 0;
}

/*
	========================================================================
	
	Routine Description:
		Interrupt handler
		
	Arguments:
		irq							interrupt line
		dev_instance				Pointer to net_device
		rgs							store process's context before entering ISR, 
									this parameter is just for debug purpose.

	Return Value:
		irqreturn_t

	Note:

		
	========================================================================
*/
irqreturn_t RTMPIsr(
       int             irq,
       void            *dev_instance,
       struct pt_regs  *rgs)
{
	struct net_device	*net_dev = (struct net_device*) dev_instance;
	PRTMP_ADAPTER		pAd = netdev_priv(net_dev);
	INTSRC_STRUC		IntSource;
	int handled = 0;

	// 1. Disable interrupt
	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_IN_USE) && RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_ACTIVE))
	{
	        NICDisableInterrupt(pAd);
	}
	
	// 2. Get the interrupt sources & saved to local variable
	RTMP_IO_READ32(pAd, CSR7, &IntSource.word);
	RTMP_IO_WRITE32(pAd, CSR7, IntSource.word);

	//
	// Handle interrupt, walk through all bits
	// Should start from highest priority interrupt
	// The priority can be adjust by altering processing if statement
	//
	// If required spinlock, each interrupt service routine has to acquire
	// and release itself.
	//
	if (IntSource.field.TbcnExpire)
	{
		DBGPRINT(RT_DEBUG_INFO, "====> RTMPHandleTbcnInterrupt\n");
		RTMPHandleTbcnInterrupt(pAd);
		handled = 1;
	}

	if (IntSource.field.TwakeExpire)
	{
		DBGPRINT(RT_DEBUG_TRACE, "====> RTMPHandleTwakeupInterrupt\n");
		RTMPHandleTwakeupInterrupt(pAd);
		handled = 1;
	}

	if (IntSource.field.TatimwExpire)
	{
		DBGPRINT(RT_DEBUG_TRACE, "====> RTMPHandleTatimInterrupt\n");
		// RTMPHandleTatimInterrupt(pAd);
		handled = 1; // not sure about this...
	}

	if (IntSource.field.TxRingTxDone)
	{
		DBGPRINT(RT_DEBUG_INFO, "====> RTMPHandleTxRingTxDoneInterrupt\n");
		RTMPHandleTxRingTxDoneInterrupt(pAd);
		handled = 1;
	}

	if (IntSource.field.AtimRingTxDone)
	{
		DBGPRINT(RT_DEBUG_INFO, "====> RTMPHandleAtimRingTxDoneInterrupt\n");
		RTMPHandleAtimRingTxDoneInterrupt(pAd);
		handled = 1;
	}

	if (IntSource.field.PrioRingTxDone)
	{
		DBGPRINT(RT_DEBUG_INFO, "====> RTMPHandlePrioRingTxDoneInterrupt\n");
		RTMPHandlePrioRingTxDoneInterrupt(pAd);
		handled = 1;
	}

	if (IntSource.field.RxDone)
	{
		DBGPRINT(RT_DEBUG_INFO, "====> RTMPHandleRxDoneInterrupt\n");
		RTMPHandleRxDoneInterrupt(net_dev);
		handled = 1;
	}

	// Do nothing if Reset in progress
	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS))
	{
		if(handled)
			return IRQ_RETVAL(IRQ_HANDLED);
		else
			return IRQ_RETVAL(IRQ_NONE);
	}

	// 3. Re-enable the interrupt, disabled earlier
	NICEnableInterrupt(pAd);

	if(handled)
		return IRQ_RETVAL(IRQ_HANDLED);
	else
		return IRQ_RETVAL(IRQ_NONE);
}

#if WIRELESS_EXT >= 12
/*
	========================================================================
	
	Routine Description:
		get wireless statistics

	Arguments:
		net_dev						Pointer to net_device

	Return Value:
		struct iw_statistics

	Note:
		This function will be called when query /proc
		
	========================================================================
*/
struct iw_statistics *RT2400_get_wireless_stats(
	IN	struct net_device *net_dev)
{
	RTMP_ADAPTER *pAd = netdev_priv(net_dev);

        // MW: 19/9 Removed as it always floods the syslog
	// DBGPRINT(RT_DEBUG_TRACE, "RT2400_get_wireless_stats --->\n");
	
	// TODO: All elements are zero before be implemented
	
	pAd->iw_stats.status = 0;						// Status - device dependent for now
	
	pAd->iw_stats.qual.qual = pAd->Mlme.RoamCqi;			// link quality (%retries, SNR, %missed beacons or better...)
	pAd->iw_stats.qual.level = pAd->PortCfg.LastRssi;	// signal level (dBm)
		
	pAd->iw_stats.qual.noise = 0;			// noise level (dBm)
	pAd->iw_stats.qual.updated = 1;		// Flags to know if updated

	pAd->iw_stats.discard.nwid = 0;		// Rx : Wrong nwid/essid
	pAd->iw_stats.miss.beacon = 0;		// Missed beacons/superframe

	// pAd->iw_stats.discard.code, discard.fragment, discard.retries, discard.misc has counted in other place
	
	return &pAd->iw_stats;
}
#endif

/*
	========================================================================
	
	Routine Description:
		return ethernet statistics counter

	Arguments:
		net_dev						Pointer to net_device

	Return Value:
		net_device_stats*

	Note:
		
		
	========================================================================
*/
struct net_device_stats *RT2400_get_ether_stats(
	IN	struct net_device *net_dev)
{
	RTMP_ADAPTER *pAd = netdev_priv(net_dev);

	DBGPRINT(RT_DEBUG_INFO, "RT2400_get_ether_stats --->\n");

	pAd->stats.rx_packets = pAd->WlanCounters.ReceivedFragmentCount;		// total packets received
	pAd->stats.tx_packets = pAd->WlanCounters.TransmittedFragmentCount;		// total packets transmitted
	
	pAd->stats.rx_bytes= pAd->RalinkCounters.ReceivedByteCount;				// total bytes received
	pAd->stats.tx_bytes = pAd->RalinkCounters.TransmittedByteCount;			// total bytes transmitted
	
	pAd->stats.rx_errors = pAd->Counters.RxErrors;							// bad packets received
	pAd->stats.tx_errors = pAd->Counters.TxErrors;							// packet transmit problems

	pAd->stats.rx_dropped = pAd->Counters.RxNoBuffer;						// no space in linux buffers
	pAd->stats.tx_dropped = pAd->WlanCounters.FailedCount;					// no space available in linux

	pAd->stats.multicast = pAd->WlanCounters.MulticastReceivedFrameCount;	// multicast packets received
	pAd->stats.collisions = pAd->Counters.OneCollision + pAd->Counters.MoreCollisions;	// Collision packets

	pAd->stats.rx_length_errors = 0;
	pAd->stats.rx_over_errors = pAd->Counters.RxNoBuffer;					// receiver ring buff overflow
	pAd->stats.rx_crc_errors = 0;//pAd->WlanCounters.FCSErrorCount;		// recved pkt with crc error
	pAd->stats.rx_frame_errors = pAd->Counters.RcvAlignmentErrors;			// recv'd frame alignment error
	pAd->stats.rx_fifo_errors = pAd->Counters.RxNoBuffer;					// recv'r fifo overrun
	pAd->stats.rx_missed_errors = 0;											// receiver missed packet
	
	// detailed tx_errors
	pAd->stats.tx_aborted_errors = 0;
	pAd->stats.tx_carrier_errors = 0;
	pAd->stats.tx_fifo_errors = 0;
	pAd->stats.tx_heartbeat_errors = 0;
	pAd->stats.tx_window_errors = 0;
	
	// for cslip etc
	pAd->stats.rx_compressed = 0;
	pAd->stats.tx_compressed = 0;
	
	return &pAd->stats;
}

/*
	========================================================================
	
	Routine Description:
		Set to filter multicast list

	Arguments:
		net_dev						Pointer to net_device

	Return Value:
		VOID

	Note:
		
		
	========================================================================
*/
VOID RT2400_set_rx_mode(
	IN	struct net_device *net_dev)
{
	// TODO: set_multicast_list
	RTMP_ADAPTER *pAd;
        pAd = netdev_priv(net_dev);
        if (net_dev->flags&IFF_PROMISC)
        {
              pAd->bAcceptPromiscuous = TRUE;
              RTMP_IO_WRITE32(pAd, RXCSR0, 0x6e);
              DBGPRINT(RT_DEBUG_TRACE,"rt2400 acknowledge PROMISC on\n");
        }
        else
        {
              pAd->bAcceptPromiscuous = FALSE;
              RTMP_IO_WRITE32(pAd, RXCSR0, 0x7e);
              DBGPRINT(RT_DEBUG_TRACE, "rt2400 acknowledge PROMISC off\n");
        }

}

int rt2400_set_mac_address(struct net_device *net_dev, void *addr)
{
	RTMP_ADAPTER		*pAd = netdev_priv(net_dev);
	struct sockaddr		*mac = (struct sockaddr*) addr;
	u32			set_mac;

	if (netif_running(net_dev))
		return -EBUSY;

	BUG_ON(net_dev->addr_len != ETH_ALEN);

	memcpy(net_dev->dev_addr, mac->sa_data, ETH_ALEN);
	memcpy(pAd->CurrentAddress, mac->sa_data, ETH_ALEN);

	memset(&set_mac, 0x00, sizeof(u32));
	set_mac = (net_dev->dev_addr[0]) |
			(net_dev->dev_addr[1] << 8) |
			(net_dev->dev_addr[2] << 16) |
			(net_dev->dev_addr[3] << 24);

	RTMP_IO_WRITE32(pAd, CSR3, set_mac);

	memset(&set_mac, 0x00, sizeof(u32));
	set_mac = (net_dev->dev_addr[4]) |
			(net_dev->dev_addr[5] << 8);

	RTMP_IO_WRITE32(pAd, CSR4, set_mac);

	printk(KERN_INFO "***rt2x00***: Info - Mac address changed to: %02x:%02x:%02x:%02x:%02x:%02x.\n", net_dev->dev_addr[0], net_dev->dev_addr[1], net_dev->dev_addr[2], net_dev->dev_addr[3], net_dev->dev_addr[4], net_dev->dev_addr[5]);

	return 0;
}

//
// Close driver function
//
INT	RT2400_close(
	IN	struct net_device *net_dev)
{
	RTMP_ADAPTER	*pAd = netdev_priv(net_dev);

	DBGPRINT(RT_DEBUG_TRACE, "%s: ===> RT2400_close\n", net_dev->name);

	// Disable interrupt first
	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_ACTIVE))
	{
		NICDisableInterrupt(pAd);
	}

	// Stop Mlme state machine
	MlmeHalt(pAd);
	del_timer_sync(&pAd->PortCfg.RfTuningTimer);

	netif_stop_queue(net_dev);
	netif_carrier_off(net_dev);
	
	// Shut down monitor timer task
	del_timer_sync(&pAd->timer);

	// Disable Rx, register value supposed will remain after reset
	NICIssueReset(pAd);
	
	// Free IRQ
	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_IN_USE))
	{
		// Deregister interrupt function
		free_irq(net_dev->irq, net_dev);
		RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_IN_USE);
	}

	// Free Ring buffers
	RTMPFreeDMAMemory(pAd);

	return 0;
}

//
// Remove driver function
//
static VOID __devexit RT2400_remove_one(
	IN	struct pci_dev	*pPci_Dev)
{
	struct net_device	*net_dev = pci_get_drvdata(pPci_Dev);

	// Unregister network device
	unregister_netdev(net_dev);

	// Unmap CSR base address
	iounmap((void *)(net_dev->base_addr));
	
	// release memory regions
        pci_release_regions(pPci_Dev);

        // disable the device
        pci_disable_device(pPci_Dev);

	// Free pre-allocated net_device memory
	kfree(net_dev);
}

//
// Ralink PCI device table, include all supported chipsets
//
static struct pci_device_id rt2400_pci_tbl[] __devinitdata =
{
	{0x1814, 0x0101, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RT2460A},
	{0,}			                    /* terminate list */
};
MODULE_DEVICE_TABLE(pci, rt2400_pci_tbl);

//
// Our PCI driver structure
//
static struct pci_driver rt2400_driver =
{
	name:		"rt2400",
	id_table:	rt2400_pci_tbl,
	probe:		RT2400_init_one,
#if LINUX_VERSION_CODE >= 0x20412		// Red Hat 7.3
	remove:		__devexit_p(RT2400_remove_one),
#else
	remove:		__devexit(RT2400_remove_one),
#endif
};

// =======================================================================
// LOAD / UNLOAD sections
// =======================================================================
//
// Driver module load function
//
static INT __init rt2400_init_module(VOID)
{
#ifdef EXPORT_NO_SYMBOLS
	EXPORT_NO_SYMBOLS;		// must use in init_module to avoid output symbol to /proc/ksyms
#endif
	return pci_module_init(&rt2400_driver);
}

//
// Driver module unload function
//
static VOID __exit rt2400_cleanup_module(VOID)
{
	pci_unregister_driver(&rt2400_driver);
}

module_init(rt2400_init_module);
module_exit(rt2400_cleanup_module);
