/*
*  Rubrica
*  file: dialoghi.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gnome.h>

#include "dialog.h"
#include "log.h"
#include "ui.h"
#include "rubrica_type.h"
#include "main.h"


GtkWidget* rubrica_dialog_create(GtkWidget *parent, gchar *title, 
				 gchar *str, gchar *pixmap)
{
  GtkWidget *dialog;
  GtkWidget *hbox;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *image;
  GtkWidget *sep;

  hbox = gtk_hbox_new(FALSE, 3);
  gtk_container_set_border_width (GTK_CONTAINER(hbox), 2);

  if (pixmap)
    {
      sep = gtk_vseparator_new();
      image = rubrica_ui_image_new(pixmap);
      
      gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 3);
      gtk_box_pack_start(GTK_BOX(hbox), sep, TRUE, FALSE, 3);
      gtk_widget_show(sep);
    }

  frame = rubrica_ui_frame_new(title);
  gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 3);
  
  label = gtk_label_new(str);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_misc_set_padding (GTK_MISC (label), 10, 0);
  gtk_container_add(GTK_CONTAINER(frame), label);
  gtk_widget_show(label);
  gtk_widget_show(hbox);

  dialog = gtk_dialog_new_with_buttons(title, GTK_WINDOW(parent), 
				       GTK_DIALOG_MODAL, 
				       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, 
				       NULL);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_dialog_set_has_separator(GTK_DIALOG(dialog), TRUE);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 3);

  return dialog;
}


GtkWidget* rubrica_dialog_with_href_create(GtkWidget *parent, gchar *title, 
					   gchar *str, gchar *pixmap, 
					   gchar *uri, gchar *uri_str)
{
  GtkWidget *dialog;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *image;
  GtkWidget *sep;
  GtkWidget *hsep;
  GtkWidget *href1;

  hbox = gtk_hbox_new(FALSE, 5);

  if (pixmap)
    {
      sep = gtk_vseparator_new();
      image = rubrica_ui_image_new(pixmap);
      
      gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 3);
      gtk_box_pack_start(GTK_BOX(hbox), sep, TRUE, FALSE, 3);
      gtk_widget_show(sep);
    }

  vbox = gtk_vbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 3);
  gtk_widget_show(vbox);

  frame = rubrica_ui_frame_new(title);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 3);

  label = gtk_label_new(str);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_misc_set_padding (GTK_MISC (label), 10, 0);
  gtk_container_add(GTK_CONTAINER(frame), label);
  gtk_widget_show(label);
  gtk_widget_show(hbox);

  hsep = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), hsep, FALSE, TRUE, 3);
  gtk_widget_show (hsep);  

  href1 = gnome_href_new (uri, uri_str);
  gtk_box_pack_start(GTK_BOX(vbox), href1, FALSE, TRUE, 3);
  gtk_widget_show (href1);  

  dialog = gtk_dialog_new_with_buttons(title, GTK_WINDOW(parent), 
				       GTK_DIALOG_MODAL, 
				       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, 
				       NULL);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_dialog_set_has_separator(GTK_DIALOG(dialog), TRUE);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 3);

  return dialog;
}


void rubrica_dialog_message(gchar *str, GtkMessageType message, 
			    GtkButtonsType button)
{
  GtkWidget *app;
  GtkWidget *dialog;

  app = rubrica_get_app();  
  if (!app) return;

  dialog = gtk_message_dialog_new (GTK_WINDOW(app), GTK_DIALOG_MODAL, 
				   message, button, str);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);

  gtk_widget_show(dialog);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);  
}

void rubrica_dialog_simple_message(gchar *str)
{
  rubrica_dialog_message(str, GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE);
}

void rubrica_dialog_error(gchar *str)
{
  GtkWidget *app;
  GtkWidget *dialog;

  app = rubrica_get_app();  
  if (!app) return;

  dialog = gtk_message_dialog_new (GTK_WINDOW(app), GTK_DIALOG_MODAL, 
				   GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
				   str);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);

  gtk_widget_show(dialog);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);  
}


GtkWidget* rubrica_dialog_query(gchar *str, GtkButtonsType buttons)
{
  GtkWidget *app;
  GtkWidget *dialog;

  app = rubrica_get_app();
  
  dialog = gtk_message_dialog_new(GTK_WINDOW(app), GTK_DIALOG_MODAL, 
				  GTK_MESSAGE_QUESTION, buttons, str);

  return dialog;
}


void
rubrica_dialog_info (void)
{
  const gchar *authors[] = {
    "Nicola Fragale <nicolafragale@libero.it>",
    NULL
  };

  const gchar *documenters[] = {
    N_("No documenters yet, sorry"),
    NULL
  };

  const gchar *translators = 
    "English and Italian by Nicola Fragale <nicolafragale@libero.it>\n"
    "French by Jean-pierre Pedron <jppedron@club-internet.fr>\n"
    "Brazilian by Ricardo Sandrin <rsandrin@cipsga.org.br>\n"
    "Russian by Sergey Volozhanin <svo@asplinux.udm.net>\n"
    "Russian by Denis Rtveliashvili <rtvd@8ka.mipt.ru>\n"    
    "Japanese by Takeshi Aihana <aihana@gnome.gr.jp>\n"
    "Spanish by Carlos Cortes <carcoco@grupobbva.net>\n"
    "Deutsch by Norman Jonas <normanjonas@web.de>\n"
    "Deutsch by Marc Reibel <marc.reibel@reibel-web.de>\n"
    "Bulgarian by Fgswre Dwer <vonodna@yahoo.com>\n"
    "Slovak by Marin Tomko <macros@zoznam.sk>\n"; 

  GtkWidget *about1;
  GdkPixbuf *about1_icon_pixbuf;

  about1 = gnome_about_new ("Rubrica", VERSION,
			    "Nicola Fragale (c) 2000-2002",
			    _("Rubrica, an addressbook written "
			      "for Gnome Environment."),
			    authors,
			    documenters,
			    translators,
			    NULL);

  about1_icon_pixbuf = rubrica_ui_pixbuf_new("rub.png");
  if (about1_icon_pixbuf)
    {
      gtk_window_set_icon (GTK_WINDOW (about1), about1_icon_pixbuf);
    }

  gtk_widget_show(about1);
}


void
rubrica_dialog_notify_bugs(void)
{
  GtkWidget *app;
  GtkWidget *dialog;
  gchar *str;

  app = rubrica_get_app();

  str = g_strdup(_("If you found a bug in rubrica\n"
		   "please, send me an email with\n"
		   "a bug description.\n\n"
		   "Or, better, if rubrica crashes,\n"
		   "a bug-buddy report.\n\nThanks"));
 
  dialog = rubrica_dialog_create(app, _("Bugs"), str, "bug.png");
  gtk_widget_show(dialog);

  gtk_dialog_run(GTK_DIALOG(dialog));

  g_free(str);
  gtk_widget_destroy(dialog);
}


void
rubrica_dialog_will_do (void)
{
  GtkWidget *app;
  GtkWidget *dialog;

  app = rubrica_get_app();  

  dialog = gtk_message_dialog_new (GTK_WINDOW(app), GTK_DIALOG_MODAL,
                                   GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
                                   _("This function isn't yet implemented"));

  gtk_widget_show(dialog);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


void
rubrica_dialog_developing(void)
{
  GtkWidget *app;
  GtkWidget *dialog;

  app = rubrica_get_app();  

  dialog = gtk_message_dialog_new (GTK_WINDOW(app), GTK_DIALOG_MODAL,
                                   GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
                                   _("Warning, this feature is in a\n"
				     "development status, and could crash.\n"
				     "In this case, please, tell me."));

  gtk_widget_show(dialog);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


void
rubrica_dialog_help_me (void)
{
  GtkWidget *app;
  GtkWidget *dialog;

  app = rubrica_get_app();  

  dialog = gtk_message_dialog_new (GTK_WINDOW(app),
				   GTK_DIALOG_MODAL,
                                   GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
				   _("Rubrica needs localization.\n\n"
				     "Please, if your language isn't "
				     "supported,\n translate rubrica."));
  
  gtk_widget_show(dialog);
  g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
                            G_CALLBACK (gtk_widget_destroy),
                            GTK_OBJECT (dialog));
}



void
rubrica_dialog_goto_homepage (void)
{
  GtkWidget *app;
  GtkWidget *dialog;
  gchar *str;

  app = rubrica_get_app();

  str = g_strdup(_("Click button to go \nto rubrica home page"));
 
  dialog = rubrica_dialog_with_href_create(app, _("Home page"), str, 
					   "casa.png", 
					   "http://digilander.iol.it/nfragale",
					   _("Rubrica home page"));
  gtk_widget_show(dialog);

  gtk_dialog_run(GTK_DIALOG(dialog));

  g_free(str);
  gtk_widget_destroy(dialog);
}

void
rubrica_dialog_translation_rules (void)
{
  GtkWidget *app;
  GtkWidget *dialog;
  gchar *str;

  app = rubrica_get_app();  

  str = g_strdup(_("Rubrica can use more languages, in address book preview,"
		   "\nto show its fields. User can choose to use one of the"
		   "\ndefault rubrica languages (Italian or English) or can"
		   "customize\nthe dictionary.\n"));

  dialog = gtk_message_dialog_new (GTK_WINDOW(app), GTK_DIALOG_MODAL,
                                   GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
                                   str);

  gtk_widget_show(dialog);
  gtk_dialog_run (GTK_DIALOG (dialog));
  g_free(str);
  gtk_widget_destroy (dialog);  
}


void rubrica_dialog_error_loading_file(gchar *file, RubricaErrorType err)
{
  gchar *message;

  switch (err)
    {
    case RE_IMPORT_FILE:
      message = g_strdup_printf(_("Import file not performed.\n"
				  "I can't read the file: %s"), file);
      
      rubrica_dialog_message(message, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);
      g_free(message);
      break;

    case RE_FILE_NOT_EXIST:
      message = g_strdup_printf(_("I can't read %s.\nFile doesn't exist."), 
				file);

      rubrica_dialog_message(message, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);  
      
      g_free(message);
      break;
      
    case RE_PARSE_XML_DOC:
      message = g_strdup_printf(_("I can't parse xml structure of file: %s."),
				file);

      rubrica_dialog_message(message, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);  

      g_free(message);
      break;
      
    case RE_NO_RUBRICA_HEADER:
      message = g_strdup_printf(_("Header error.\n%s is not a rubrica file\n"
				  "Or this file is corrupted"), file);

      rubrica_dialog_message(message, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);  
      
      g_free(message);
      break;

    case RE_SELECT_FILE:
      rubrica_dialog_message(_("You must select a file before.\n"),
			     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);
      break;

    default:
      rubrica_dialog_message(_("I can't load default addressbook."),
			     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);
      break;
    }
}


void rubrica_dialog_error_in_preferences_key(gchar *str)
{
  gchar *buffer;

  buffer = g_strdup_printf(_("\nCan't read configuration values from "
			     "gconf for %s key"),str);

  g_print(buffer);
  rubrica_log_write_log(buffer);
  
  g_free(buffer);
}


void rubrica_dialog_error_loading_preferences_keys(void)
{
  gchar *buffer;

  buffer = g_strdup(_("Can't read configuration values from gconf.\n"
		      "May be rubrica.schemas isn't installed right way\n\n"
		      "Exec gconftool-2 --makefile-install-rule=rubrica.schemas\n"
		      "or gconftool-2 --install-schema-file=rubrica.schemas\n"
		      "You can find rubrica.schemas in the /etc/gconf/schema "
		      "directory.\n\nrubrica.log file contains unread keys."));

  rubrica_dialog_message(buffer, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);
  rubrica_log_write_log(buffer);

  g_free(buffer);
}
