/*
*  Rubrica
*  file: import.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <gnome.h>

#include "rubrica.h"
#include "log.h"
#include "file.h"
#include "main.h"
#include "dialog.h"
#include "import.h"
#include "import_int.h"
#include "vcard.h"
#include "csv.h"
#include "view.h"
#include "card.h"

static gchar *gnomecard = ".gnome/GnomeCard.gcrd";
static gchar *kdecard   = ".kde/share/apps/kabc/std.vcf";
/* static gchar *pine      = ".addressbook"; */ // will be used

RubricaImportType importfrom;

void rubrica_import_init(GtkWidget *fentry, RubricaImportType import)
{  
  importfrom = import; 
}

void rubrica_import_set(RubricaImportType import)
{
  importfrom = (RubricaImportType) import;  
}

RubricaImportType rubrica_import_get(void)
{
  return (RubricaImportType) importfrom;
}


gboolean rubrica_import_data(GtkWidget *app)
{
  Rubrica *rubrica;
  GtkWidget *fentry;
  gchar *filename = NULL;
  gchar *msg;
  gboolean imported = FALSE;
  
  fentry  = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY
				       (rubrica_import_get_file_entry()));
  filename = g_strdup(gtk_entry_get_text(GTK_ENTRY(fentry)));
 
  switch (importfrom)
    {
    case IMPORT_GNOMECARD:
      /* 	 if no name is set in gnome_entry, then 
		 load default gnomecard addressbook 
      */
      if (strcmp(filename, "") == 0)
	{
	  /* load the default GnomeCard file: GnoneCard.gcrd*/
	  filename = g_strdup_printf("%s%s%s", g_get_home_dir(), 
				     G_DIR_SEPARATOR_S, 
				     gnomecard);

	  if (!g_file_exists(filename))
	    {
	      rubrica_dialog_error_loading_file(filename, RE_FILE_NOT_EXIST);

	      g_free(filename);
	      return FALSE;
	    }

	  msg = g_strdup_printf("\nimporting gnomecard file: %s", filename);
	  rubrica_log_write_log(msg);
	  g_free(msg);

	  imported = rubrica_import_open_vcard_file(filename);
	}
      else	/* import a gnomecard file */
	{
	  msg = g_strdup_printf("\nimporting gnomecard file: %s", filename); 
	  rubrica_log_write_log(msg);
	  g_free(msg);

	  imported = rubrica_import_open_vcard_file(filename);
	}
      break;
      
    case IMPORT_KAB:
      filename = g_strdup_printf("%s%s%s", g_get_home_dir(),
				 G_DIR_SEPARATOR_S, kdecard);

      if (!g_file_exists(filename))
	{
	  rubrica_dialog_error_loading_file(filename, RE_FILE_NOT_EXIST);

	  g_free(filename); 
	  return FALSE;
	}
      
      msg = g_strdup_printf("\nimporting Kabc file: %s", filename);      
      rubrica_log_write_log(msg);
      g_free(msg);

      imported = rubrica_import_open_vcard_file(filename);
      break;

    case IMPORT_RUBRICA:
      if (strcmp(filename, "") == 0)
	{
	  rubrica_dialog_error_loading_file(NULL, RE_SELECT_FILE);
	  
	  return FALSE;
	}
      else 	/* apri il file e leggilo */
	{
	  msg = g_strdup_printf("\nimporting rubrica's file: %s", filename);
	  rubrica_log_write_log(msg);
	  g_free(msg);

	  imported = rubrica_import_open_old_rubrica_file(filename);
	}
      break;

    case IMPORT_CSV_MOZILLA:
    case IMPORT_CSV_NETSCAPE:
      if (strcmp(filename, "") == 0)
	{
	  rubrica_dialog_error_loading_file(NULL, RE_SELECT_FILE);

	  return FALSE;
	}
      else 
	{
	  msg = g_strdup_printf("\nimporting csv file: %s", filename);
	  rubrica_log_write_log(msg);
	  g_free(msg);

	  imported = rubrica_import_open_csv_file(filename);  	  
	}
      break;      

    case IMPORT_VCARD:
      if (strcmp(filename,"") == 0)
	{
	  rubrica_dialog_error_loading_file(NULL, RE_SELECT_FILE);

	  return FALSE;
	}
      else 
	{
	  msg = g_strdup_printf("\nimporting vcard file: %s", filename);
	  rubrica_log_write_log(msg);
	  g_free(msg);

	  imported = rubrica_import_open_vcard_file(filename);     
	}
      break;      
      
    case IMPORT_PINE:     
      rubrica_dialog_will_do();
      break;

    default:
      return FALSE;
      break;

    }

  if (imported)
    {
      rubrica = rubrica_get_current_rubrica();
      
      rubrica_set_path(rubrica, g_dirname(filename));
      rubrica_set_name(rubrica, g_strdup(g_basename(filename)));
      rubrica_set_new(rubrica, FALSE);
      rubrica_set_empty(rubrica, FALSE);
      rubrica_set_modified(rubrica, TRUE);
      
      rubrica_view_set_items(rubrica_get_items(rubrica));
    }
    
  if (filename)
    g_free(filename);

  return TRUE;
}


gboolean 
rubrica_import_open_vcard_file(gchar *file)
{
  Rubrica *rubrica;
  RubricaVCard* vcard;

  if (!g_file_test(file,G_FILE_TEST_EXISTS))
  {
    rubrica_dialog_error_loading_file (file,RE_FILE_NOT_EXIST);

    return FALSE;
  }
      
  rubrica_card_add_page_view(file);
  rubrica = rubrica_get_current_rubrica();
  vcard   = rubrica_vcard_new();

  g_signal_connect(G_OBJECT (vcard), "vcard_decode_card",
		   (GCallback) rubrica_import_decoded_vcard_item,
		   rubrica);

  if (!rubrica_vcard_open_file(vcard, file))
    {
      rubrica_dialog_error_loading_file(file, RE_IMPORT_FILE);

      rubrica_vcard_free_vcard(vcard);
      return FALSE;
    }

  rubrica_vcard_free_vcard(vcard);

  return TRUE;
}


gboolean 
rubrica_import_open_csv_file(gchar *file)
{
  FILE *fp;

  if ((fp=fopen(file, "r")) == NULL)
    {
      rubrica_dialog_error_loading_file (file, RE_IMPORT_FILE);

      return FALSE;
    }

  if (rubrica_csv_import_file(fp))
    return TRUE;

  return FALSE;
}


gboolean 
rubrica_import_open_old_rubrica_file(gchar *fname)
{
  RubricaErrorType err;
  gboolean loaded;
  
  loaded = rubrica_file_load_xmlfile(fname, &err);

  if (!loaded)
    {
      rubrica_dialog_error_loading_file(fname, err);      

      return FALSE;
    }
  
  return TRUE;    
}



void 
rubrica_import_decoded_vcard_item(RubricaVCard* vcard, gpointer data)
{
  Rubrica* rubrica = (Rubrica *) data;
  RubricaItem* item;
  TelNum* tt;
  gint bday = -1;
  
  item = rubrica_item_new();
  if (!item)
    return;

  item->delete      = TRUE;
  item->group.label = "";
  item->group.type  = NO_GROUP;

  item->card                = g_strdup(rubrica_vcard_get_fn(vcard));
  item->personal.first      = g_strdup(rubrica_vcard_get_gn(vcard));
  item->personal.middle     = g_strdup(rubrica_vcard_get_an(vcard));
  item->personal.last       = g_strdup(rubrica_vcard_get_n(vcard));
  item->personal.profession = "";
  item->personal.prefix     = g_strdup(rubrica_vcard_get_hp(vcard));
  item->personal.title      = g_strdup(rubrica_vcard_get_hs(vcard));

  if (rubrica_vcard_get_bday(vcard) != NULL)
    bday = atoi(rubrica_vcard_get_bday(vcard));
		 
  if (bday != -1)
    {      
      item->personal.birthknow  = TRUE;
      item->personal.birthdate  = bday;
    }
  else
    {
      item->personal.birthknow  = FALSE;
      item->personal.birthdate  = time(NULL);
    }

  item->address.street   = g_strdup(rubrica_vcard_get_str(vcard));
  item->address.number   = "";
  item->address.zip      = g_strdup(rubrica_vcard_get_code(vcard));
  item->address.city     = g_strdup(rubrica_vcard_get_loc(vcard));
  item->address.province = g_strdup(rubrica_vcard_get_reg(vcard));
  item->address.state    = "";
  item->address.country  = g_strdup(rubrica_vcard_get_count(vcard));

  item->net.web = NULL;
  do
    {
      item->net.web = g_list_append(item->net.web, 
				    g_strdup(rubrica_vcard_get_url(vcard)));
    } while (rubrica_vcard_next_url(vcard));
    
  
  item->net.email = NULL;
  do
    {    
      item->net.email = g_list_append(item->net.email,
				      g_strdup(rubrica_vcard_get_email(vcard)));
    } while (rubrica_vcard_next_email(vcard));


  item->phone.telephone = NULL;
  do
    {
      tt = g_malloc(sizeof(TelNum));
      if (tt)
	{
	  tt->number = g_strdup(rubrica_vcard_get_tel_num(vcard));
	  tt->type   = g_strdup(rubrica_vcard_get_tel_type(vcard));
	  item->phone.telephone = g_list_append(item->phone.telephone, tt);
	}
    } while (rubrica_vcard_next_tel(vcard));

  item->company.name       = "";
  item->company.street     = "";
  item->company.number     = "";
  item->company.zip        = "";
  item->company.city       = "";
  item->company.province   = "";
  item->company.country    = "";
  item->company.web        = "";
  item->company.email      = "";
  item->company.operator   = "";
  item->company.fax        = "";
  item->company.green      = "";
  item->company.customer_care = "";
  item->company.notes      = "";

  item->work.assigment     = g_strdup(rubrica_vcard_get_role(vcard));
  item->work.organization  = g_strdup(rubrica_vcard_get_org(vcard));
  item->work.department    = "";
  item->work.subdep        = "";
  item->work.secretary     = "";
  item->work.telephone     = "";

  item->notes.is_married   = FALSE;
  item->notes.has_children = FALSE;
  item->notes.birthknow    = FALSE;
  item->notes.anniverknow  = FALSE;
  item->notes.birthdate    = time(NULL);
  item->notes.anniverdate  = time(NULL);
  item->notes.spouse       = "";
  item->notes.children     = "";
  item->notes.hobbies      = "";
  item->notes.notes        = g_strdup(rubrica_vcard_get_note(vcard));
  item->notes.pubkey       = g_strdup(rubrica_vcard_get_key(vcard));

  item->created     = time(NULL);
  item->last_change = time(NULL);
    
  rubrica_append_item(rubrica, item); 
}
