#!/bin/env ruby
#
#----------------------------------------------------------------------
# 
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version. This program is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
# Public License for more details; it is available at
# <http://www.fsf.org/copyleft/gpl.html>, or by writing to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
# Written by Gordon Miller <gmiller@bittwiddlers.com>. Inspired by and partly
# derived from the Python version by Michael Haggerty. If you find a problem
# or have a suggestion, please let me know at <gmiller@bittwiddlers.com>.
# Other feedback would also be appreciated.
#
#----------------------------------------------------------------------
#
# This file contains the implementations of the Plot object (PlotBase, Plot,
# Splot, and Contour). This file should not be included directly by an
# application, the Gnuplot.rb file should be included instead.
#
#----------------------------------------------------------------------

class PlotBase

  # Interface to a gnuplot program.
  #
  # A Gnuplot represents a higher-level interface to a gnuplot
  # program.  It can plot 'PlotItem's, which represent each thing to
  # be plotted on the current graph.  It keeps a reference to each of
  # the PlotItems used in the current plot, so that they (and their
  # associated temporary files) are not deleted prematurely.
  #
  # Members:
  #
  # 'itemlist' -- a list of the PlotItems that are associated with the
  #               current plot.  These are deleted whenever a new plot
  #               command is issued via the `plot' method.
  # 'plotcmd' -- 'plot' or 'splot', depending on what was the last
  #              plot command.
  #
  # Methods:
  #
  # '__init__' -- if a filename argument is specified, the commands
  #               will be written to that file instead of being piped
  #               to gnuplot.
  # 'plot' -- clear the old plot and old PlotItems, then plot the
  #           arguments in a fresh plot command.  Arguments can be: a
  #           PlotItem, which is plotted along with its internal
  #           options; a string, which is plotted as a Func; or
  #           anything else, which is plotted as a Data.
  # 'splot' -- like 'plot', except for 3-d plots.
  # 'hardcopy' -- replot the plot to a postscript file (if filename
  #               argument is specified) or pipe it to the printer as
  #               postscript othewise.  If the option `color' is set
  #               to true, then output color postscript.
  # 'replot' -- replot the old items, adding any arguments as
  #             additional items as in the plot method.
  # 'refresh' -- issue (or reissue) the plot command using the current
  #              PlotItems.
  # '__call__' -- pass an arbitrary string to the gnuplot process,
  #               followed by a newline.
  # 'xlabel', 'ylabel', 'title' -- set corresponding plot attribute.
  # 'interact' -- read lines from stdin and send them, one by one, to
  #               the gnuplot interpreter.  Basically you can type
  #               commands directly to the gnuplot command processor.
  # 'load' -- load a file (using the gnuplot `load' command).
  # 'save' -- save gnuplot commands to a file (using gnuplot `save'
  #           command) If any of the PlotItems is a temporary file, it
  #           will be deleted at the usual time and the save file will
  #           be pretty useless :-).
  # 'clear' -- clear the plot window (but not the itemlist).
  # 'reset' -- reset all gnuplot settings to their defaults and clear
  #            the current itemlist.
  # 'set_string' -- set or unset a gnuplot option whose value is a
  #                 string.
  # '_clear_queue' -- clear the current PlotItem list.
  # '_add_to_queue' -- add the specified items to the current
  #                    PlotItem list.
  attr_reader :plotcmd, :itemList
  attr :debug

  def initialize (plotcmd, filename=nil, persist=0)
    # Create a Gnuplot object.
    # 
    # Create a 'Gnuplot' object.  By default, this starts a gnuplot
    # process and prepares to write commands to it.
    # 
    # Keyword arguments:
    # 
    #   'filename=<string>' -- if a filename is specified, the
    #                          commands are instead written to that
    #                          file (e.g., for later use using
    #                          'load').
    #
    #   'persist=1' -- start gnuplot with the '-persist' option
    #                  (which creates a new plot window for each
    #                  plot command).  (This option is not available
    #                  on older versions of gnuplot.)
    #
    #   'debug=1' -- echo the gnuplot commands to stderr as well as
    #                sending them to gnuplot.

    @gnuplot = if filename == nil then
		 GnuplotProcess.new (persist=persist)
	       else
		 GnuplotFile.new (filename)
	       end

    @itemList = Array.new()
    @plotcmd = plotcmd
  end

  def draw (*items)
    # Draw a new plot.
    # 
    # Clear the current plot and create a new 2-d plot containing
    # the specified items.  Each arguments should be of the
    # following types:
    # 
    # 'PlotItem' (e.g., 'Data', 'File', 'Func') -- This is the most
    # flexible way to call plot because the PlotItems can
    # contain suboptions.  Moreover, PlotItems can be
    # saved to variables so that their lifetime is longer
    # than one plot command; thus they can be replotted
    # with minimal overhead.
    # 
    # 'string' (e.g., 'sin(x)') -- The string is interpreted as
    # 'Func(string)' (a function that is computed by
    # gnuplot).
    # 
    # Anything else -- The object, which should be convertible to an
    # array, is converted to a 'Data' item, and
    # thus plotted as data.  If the conversion
    # fails, an exception is raised.
    
    clearQueue()
    addToQueue(items)
    refresh()
  end

  def insert(s)

    # Send a command string to gnuplot.
    #
    # Send the string s as a command to gnuplot, followed by a
    # newline.  All communication with the gnuplot process (except
    # for inline data) is through this method.
    
    @gnuplot.insert(s)
    puts "gnuplot #{s}" if @debug
  end
  
  def configurePlot
    # Inserts the 'set' commands to gnuplot for setting up the plot.
  end

  def refresh
    # Refresh the plot, using the current PlotItems.
    # 
    # Refresh the current plot by reissuing the gnuplot plot command
    # corresponding to the current itemList.

    configurePlot ()

    plotcmds = Array.new()

    @itemList.each { |item| plotcmds.push ( item.command() ) }
    fullcmd = @plotcmd + " " + plotcmds.join (", ")
    insert ( fullcmd )

    @itemList.each { |item| item.pipein ( @gnuplot ) }
    @gnuplot.flush()
  end

  def clearQueue
    @itemList.clear()
  end
  
  def addToQueue (items)
    # Add a list of items to the itemlist (but don't plot them).
    # 
    # 'items' is a sequence of items, each of which should be a
    # 'PlotItem' of some kind, a string (interpreted as a function
    # string for gnuplot to evaluate), or a Numeric array (or
    # something that can be converted to a Numeric array).

    items.each { |item|
      @itemList.push (item) if item.is_a? PlotItem
      @itemList.push ( Func.new(item) ) if item.is_a? String
      @itemList.push ( DataSet.new(item) ) if item.is_a? Array
    }
  end

  def redraw (*items)
    # Replot the data, possibly adding new PlotItems.
    # 
    # Replot the existing graph, using the items in the current
    # itemlist.  If arguments are specified, they are interpreted as
    # additional items to be plotted alongside the existing items on
    # the same graph.  See 'plot' for details.

    addToQueue(items)
    refresh()
  end

  def clear
    # Clear the plot window (without affecting the current itemlist).
    insert ("clear")
  end

  def reset
    # Reset all gnuplot settings to their defaults and clear itemlist.
    insert ('reset')
    clearQueue()
  end

  def load(filename)
    # Load a file using gnuplot's 'load' command.
    insert ("load '#{filename}'")
  end

  def save(filename)
    # Save the current plot commands using gnuplot's 'save' command.
    insert ("save '#{filename}'")
  end

  def setString(option, s=nil)
    # Set a string option, or if s is omitted, unset the option.
    if s == nil then
      insert ("set #{option}")
    else
      insert ("set #{option} #{s}")
    end
  end

  def setXlabel(s=nil)
    # Set the plot's xlabel.
    setString("xlabel", "'#{s}'")
  end

  def setYlabel(s=nil)
    # Set the plot's ylabel.
    setString("ylabel", "'#{s}'")
  end

  def setTitle (s=nil)
    setString('title', "'#{s}'")
  end
end

class Plot < PlotBase
  
  def initialize (filename=nil, persist=0)
    super('plot', filename, persist)
  end

end

class Splot < PlotBase
  
  def initialize (filename=nil, persist=0)
    super ('splot', filename, persist)
  end

  def setZlabel (s=nil)
    setString("zlabel", "'#{s}'")
  end

end

class ContourPlot < PlotBase

  def initialize (filename=nil, persist=0)
    super ('splot', filename, persist)
  end

  private
  def configurePlot
    super
    setString ("view", "0, 0")
    setString ("nosurface")
    setString ("contour", "base")
    setString ("cntrparam", "levels incremental 0, 5, 40")
  end
end

# $Id: Plot.rb,v 1.2 2001/01/25 06:49:08 gmiller Exp $
