/*-GNU-GPL-BEGIN-*
RULI - Resolver User Layer Interface - Querying DNS SRV records
Copyright (C) 2003 Everton da Silva Marques

RULI is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

RULI is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with RULI; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*-GNU-GPL-END-*/

/*
  $Id: ruli_sock.c,v 1.6 2003/03/15 05:39:08 evertonm Exp $
  */


#include <stdio.h>      /* FIXME: remove me [used for fprintf() debug] */
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <assert.h>
#include <string.h>

#include <ruli_sock.h>


#ifdef RULI_SOCK_DUMP_DEBUG
static void dump_buf(FILE *out, const ruli_uint8_t *buf, int len)
{
    int i;
    fprintf(out, " dump=%d @%u", len, (unsigned int) buf);
    for(i = 0; i < len; ++i)
      fprintf(out, " %02x", (unsigned char) buf[i]);
}
#endif

static int solve_protocol(const char *proto_name)
{
  struct protoent *pe;

  pe = getprotobyname(proto_name);
  if (!pe)
    return -1;

  return pe->p_proto;
}

int ruli_sock_create(int domain, int type, const char *proto_name)
{
  int                proto;

  int                sd;
  long               flags;

  struct sockaddr_in sa;
  unsigned int       sa_len;


  /*
   * Solve protocol
   */
  proto = solve_protocol(proto_name);
  if (proto == -1)
    return -1;

  /*
   * Create socket descriptor
   */
  sd = socket(domain, type, proto);
  if (sd == -1)
    return -1;

  /*
   * Specify non-blocking behavior
   */
  flags = fcntl(sd, F_GETFL, 0);
  if (flags == -1) {
    close(sd);
    return -1;
  }

  if (fcntl(sd, F_SETFL, flags | O_NONBLOCK)) {
    close(sd);
    return -1;
  }

  /*
   * Bind the socket to local addresses
   */
  sa_len = sizeof(sa);
  sa.sin_family       = AF_INET;
  sa.sin_port         = htons(0);   /* request a local port */
  sa.sin_addr.s_addr  = INADDR_ANY; /* all local addresses */
  memset((char *) sa.sin_zero, 0, sizeof(sa.sin_zero));

  if (bind(sd, (struct sockaddr *) &sa, sa_len)) {
    close(sd);
    return -1;
  }

  /*
   * Return the socket descriptor
   */

  return sd;
}

int ruli_sock_create_udp()
{
  int sd = ruli_sock_create(PF_INET, SOCK_DGRAM, "udp");

#ifdef SO_BSDCOMPAT
  if (sd == -1)
    return -1;

  /*
   * We don't want Linux ECONNREFUSED on UDP sockets
   */
  {
    int one = 1;

    if (setsockopt(sd, SOL_SOCKET, SO_BSDCOMPAT, &one, sizeof(one)))
      return -1;
  }
#endif /* Linux SO_BSDCOMPAT */

  return sd;
}

int ruli_sock_create_tcp()
{
  return ruli_sock_create(PF_INET, SOCK_STREAM, "tcp");
}

/*
  Return true if socket has successfully connected
 */
int ruli_sock_has_connected(int tcp_sd)
{
  int       optval;
  socklen_t optlen = sizeof(optval);
  int       result = getsockopt(tcp_sd, SOL_SOCKET, SO_ERROR, 
				&optval, &optlen);

  assert(!result);
  assert(optlen == sizeof(optval));

  return !optval;
}

void ruli_sock_set_address(struct sockaddr_in *sa, int *sa_len, 
			   struct in_addr *addr, int port)
{
  sa->sin_family = AF_INET;
  sa->sin_port   = htons(port);
  sa->sin_addr   = *addr;
  memset((char *) sa->sin_zero, 0, sizeof(sa->sin_zero));

  if (sa_len)
    *sa_len = sizeof(struct sockaddr_in);
}

int ruli_sock_connect(int sd, struct in_addr *remote_addr, int remote_port)
{
  struct sockaddr_in sa;
  int                sa_len;
  int                result;

  ruli_sock_set_address(&sa, &sa_len, remote_addr, remote_port);

  result = connect(sd, (struct sockaddr *) &sa, sa_len);

#ifdef RULI_RES_DEBUG
  fprintf(stderr, "DEBUG: connect(): result=%d errno=%d\n", result, errno);
#endif

  if (!result)
    return RULI_SOCK_OK;
  
  if (errno == EINPROGRESS)
    return RULI_SOCK_WOULD_BLOCK;

  return RULI_SOCK_CONNECT_FAIL;
}

int ruli_sock_sendto(int sd, struct in_addr *rem_addr, int rem_port,
		     const ruli_uint8_t *buf, int msg_len)
{
  struct sockaddr_in sa;
  int                sa_len;
  int                wr;

  ruli_sock_set_address(&sa, &sa_len, rem_addr, rem_port);

#ifdef RULI_RES_DEBUG
  fprintf(stderr, 
	  "DEBUG: ruli_sock_sendto(): fd=%d len=%d dst=%s:%d", 
	  sd, msg_len, inet_ntoa(sa.sin_addr), rem_port);
#ifdef RULI_SOCK_DUMP_DEBUG
  dump_buf(stderr, buf, msg_len);
#endif
  fprintf(stderr, "\n");
#endif

  wr = sendto(sd, buf, msg_len, 0, (struct sockaddr *) &sa, sa_len);
  if (wr != msg_len) {

    assert(wr == -1);

    if (errno == EWOULDBLOCK)
      return RULI_SOCK_WOULD_BLOCK;

    if (errno == EAGAIN)
      return RULI_SOCK_WOULD_BLOCK;

    return RULI_SOCK_SEND_FAIL;
  }

  return RULI_SOCK_OK;
}

int ruli_sock_send(int sd, const ruli_uint8_t *buf, int msg_len)
{
  int wr;

#ifdef RULI_RES_DEBUG
  fprintf(stderr, "DEBUG: ruli_sock_send(): len=%d", msg_len);
#ifdef RULI_SOCK_DUMP_DEBUG
  dump_buf(stderr, buf, msg_len);
#endif
  fprintf(stderr, "\n");
#endif

  wr = send(sd, buf, msg_len, 0);
  if (wr != msg_len) {

    assert(wr == -1);

    if (errno == EWOULDBLOCK)
      return RULI_SOCK_WOULD_BLOCK;

    if (errno == EAGAIN)
      return RULI_SOCK_WOULD_BLOCK;

    return RULI_SOCK_SEND_FAIL;
  }

  return RULI_SOCK_OK;
}

int ruli_sock_recvfrom(int sd, ruli_uint8_t *buf, int buf_size, int *msg_len, 
		       struct sockaddr_in *sa, socklen_t *sa_len)
{
  int rd;

  assert(buf_size > 0);

  rd = recvfrom(sd, buf, buf_size, 0,  (struct sockaddr *) sa, sa_len);
  if (rd == -1) {
    if (errno == EWOULDBLOCK)
      return RULI_SOCK_WOULD_BLOCK;

    if (errno == EAGAIN)
      return RULI_SOCK_WOULD_BLOCK;

    return RULI_SOCK_RECV_FAIL;
  }

  assert(rd >= 0);
  assert(rd <= buf_size);

  if (msg_len)
    *msg_len = rd;

#ifdef RULI_RES_DEBUG
  fprintf(stderr, 
	  "DEBUG: ruli_sock_recvfrom(): recv_len=%d buf_size=%d", 
	  rd, buf_size);
#ifdef RULI_SOCK_DUMP_DEBUG
  dump_buf(stderr, buf, rd);
#endif
  fprintf(stderr, "\n");
#endif

  return RULI_SOCK_OK;
}

int ruli_sock_recv(int sd, ruli_uint8_t *buf, int buf_size, int *msg_len)
{
  int rd;

  assert(buf_size > 0);

  rd = recv(sd, buf, buf_size, 0);
  if (rd == -1) {
    if (errno == EWOULDBLOCK)
      return RULI_SOCK_WOULD_BLOCK;

    if (errno == EAGAIN)
      return RULI_SOCK_WOULD_BLOCK;

    return RULI_SOCK_RECV_FAIL;
  }

  assert(rd >= 0);
  assert(rd <= buf_size);

  if (msg_len)
    *msg_len = rd;

#ifdef RULI_RES_DEBUG
  fprintf(stderr, 
	  "DEBUG: ruli_sock_recv(): recv_len=%d buf_size=%d", 
	  rd, buf_size);
#ifdef RULI_SOCK_DUMP_DEBUG
  dump_buf(stderr, buf, rd);
#endif
  fprintf(stderr, "\n");
#endif

  if (!rd)
    return RULI_SOCK_CLOSED;

  return RULI_SOCK_OK;
}


