// Copyright 2012-2016 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Hook into libgraphviz for rendering dataflow graphs for MIR.

use rustc::mir::repr::{BasicBlock, Mir};

use dot;
use dot::IntoCow;

use std::fs::File;
use std::io;
use std::io::prelude::*;

use super::MirBorrowckCtxt;
use bitslice::bits_to_string;
use super::gather_moves::MoveOut;

struct Graph<'c, 'b:'c, 'a:'b, 'tcx:'a> { mbcx: &'c MirBorrowckCtxt<'b, 'a, 'tcx>,
                                          context: &'b str }

pub fn print_borrowck_graph_to(mbcx: &MirBorrowckCtxt,
                               context: &str,
                               path: &str) -> io::Result<()> {
    let g = Graph { mbcx: mbcx, context: context };
    let mut v = Vec::new();
    dot::render(&g, &mut v)?;
    println!("print_borrowck_graph_to path: {} context: {} node_id: {}",
             path, context, mbcx.node_id);
    File::create(path).and_then(|mut f| f.write_all(&v))
}

pub type Node = BasicBlock;

#[derive(Copy, Clone, PartialEq, Eq, Debug)]
pub struct Edge { source: BasicBlock, index: usize }

fn outgoing(mir: &Mir, bb: BasicBlock) -> Vec<Edge> {
    let succ_len = mir.basic_block_data(bb).terminator().successors().len();
    (0..succ_len).map(|index| Edge { source: bb, index: index}).collect()
}

impl<'c, 'b:'c, 'a:'b, 'tcx:'a> dot::Labeller<'c> for Graph<'c,'b,'a,'tcx> {
    type Node = Node;
    type Edge = Edge;
    fn graph_id(&self) -> dot::Id {
        dot::Id::new(format!("graph_for_node_{}_{}",
                             self.mbcx.node_id,
                             self.context))
            .unwrap()
    }

    fn node_id(&self, n: &Node) -> dot::Id {
        dot::Id::new(format!("bb_{}", n.index()))
            .unwrap()
    }

    fn node_label(&self, n: &Node) -> dot::LabelText {
        // A standard MIR label, as generated by write_node_label, is
        // presented in a single column in a table.
        //
        // The code below does a bunch of formatting work to format a
        // node (i.e. MIR basic-block) label with extra
        // dataflow-enriched information.  In particular, the goal is
        // to add extra columns that present the three dataflow
        // bitvectors, and the data those bitvectors represent.
        //
        // It presents it in the following format (where I am
        // presenting the table rendering via ASCII art, one line per
        // row of the table, and a chunk size of 3 rather than 5):
        //
        // ------  -----------------------  ------------  --------------------
        //                    [e1, e3, e4]
        //             [e8, e9] "= ENTRY:"  <ENTRY-BITS>
        // ------  -----------------------  ------------  --------------------
        // Left
        // Most
        // Column
        // Is
        // Just
        // Normal
        // Series
        // Of
        // MIR
        // Stmts
        // ------  -----------------------  ------------  --------------------
        //           [g1, g4, g5] "= GEN:"  <GEN-BITS>
        // ------  -----------------------  ------------  --------------------
        //                         "KILL:"  <KILL-BITS>   "=" [k1, k3, k8]
        //                                                [k9]
        // ------  -----------------------  ------------  --------------------
        //
        // (In addition, the added dataflow is rendered with a colored
        // background just so it will stand out compared to the
        // statements.)
        let mut v = Vec::new();
        let i = n.index();
        let chunk_size = 5;
        const BG_FLOWCONTENT: &'static str = r#"bgcolor="pink""#;
        const ALIGN_RIGHT: &'static str = r#"align="right""#;
        const FACE_MONOSPACE: &'static str = r#"FACE="Courier""#;
        fn chunked_present_left<W:io::Write>(w: &mut W,
                                             interpreted: &[&MoveOut],
                                             chunk_size: usize)
                                             -> io::Result<()>
        {
            // This function may emit a sequence of <tr>'s, but it
            // always finishes with an (unfinished)
            // <tr><td></td><td>
            //
            // Thus, after being called, one should finish both the
            // pending <td> as well as the <tr> itself.
            let mut seen_one = false;
            for c in interpreted.chunks(chunk_size) {
                if seen_one {
                    // if not the first row, finish off the previous row
                    write!(w, "</td><td></td><td></td></tr>")?;
                }
                write!(w, "<tr><td></td><td {bg} {align}>{objs:?}",
                       bg = BG_FLOWCONTENT,
                       align = ALIGN_RIGHT,
                       objs = c)?;
                seen_one = true;
            }
            if !seen_one {
                write!(w, "<tr><td></td><td {bg} {align}>[]",
                       bg = BG_FLOWCONTENT,
                       align = ALIGN_RIGHT)?;
            }
            Ok(())
        }
        ::rustc_mir::graphviz::write_node_label(
            *n, self.mbcx.mir, &mut v, 4,
            |w| {
                let flow = &self.mbcx.flow_state;
                let entry = flow.interpret_set(flow.sets.on_entry_set_for(i));
                chunked_present_left(w, &entry[..], chunk_size)?;
                write!(w, "= ENTRY:</td><td {bg}><FONT {face}>{entrybits:?}</FONT></td>\
                                        <td></td></tr>",
                       bg = BG_FLOWCONTENT,
                       face = FACE_MONOSPACE,
                       entrybits=bits_to_string(flow.sets.on_entry_set_for(i),
                                                flow.sets.bytes_per_block()))
            },
            |w| {
                let flow = &self.mbcx.flow_state;
                let gen = flow.interpret_set( flow.sets.gen_set_for(i));
                let kill = flow.interpret_set(flow.sets.kill_set_for(i));
                chunked_present_left(w, &gen[..], chunk_size)?;
                write!(w, " = GEN:</td><td {bg}><FONT {face}>{genbits:?}</FONT></td>\
                                       <td></td></tr>",
                       bg = BG_FLOWCONTENT,
                       face = FACE_MONOSPACE,
                       genbits=bits_to_string( flow.sets.gen_set_for(i),
                                               flow.sets.bytes_per_block()))?;
                write!(w, "<tr><td></td><td {bg} {align}>KILL:</td>\
                                        <td {bg}><FONT {face}>{killbits:?}</FONT></td>",
                       bg = BG_FLOWCONTENT,
                       align = ALIGN_RIGHT,
                       face = FACE_MONOSPACE,
                       killbits=bits_to_string(flow.sets.kill_set_for(i),
                                               flow.sets.bytes_per_block()))?;

                // (chunked_present_right)
                let mut seen_one = false;
                for k in kill.chunks(chunk_size) {
                    if !seen_one {
                        // continuation of row; this is fourth <td>
                        write!(w, "<td {bg}>= {kill:?}</td></tr>",
                               bg = BG_FLOWCONTENT,
                               kill=k)?;
                    } else {
                        // new row, with indent of three <td>'s
                        write!(w, "<tr><td></td><td></td><td></td><td {bg}>{kill:?}</td></tr>",
                               bg = BG_FLOWCONTENT,
                               kill=k)?;
                    }
                    seen_one = true;
                }
                if !seen_one {
                    write!(w, "<td {bg}>= []</td></tr>",
                           bg = BG_FLOWCONTENT)?;
                }

                Ok(())
            })
            .unwrap();
        dot::LabelText::html(String::from_utf8(v).unwrap())
    }

    fn node_shape(&self, _n: &Node) -> Option<dot::LabelText> {
        Some(dot::LabelText::label("none"))
    }
}

impl<'c, 'b:'c, 'a:'b, 'tcx:'a> dot::GraphWalk<'c> for Graph<'c,'b,'a,'tcx> {
    type Node = Node;
    type Edge = Edge;
    fn nodes(&self) -> dot::Nodes<Node> {
        self.mbcx.mir.all_basic_blocks().into_cow()
    }

    fn edges(&self) -> dot::Edges<Edge> {
        let mir = self.mbcx.mir;
        let blocks = self.mbcx.mir.all_basic_blocks();
        // base initial capacity on assumption every block has at
        // least one outgoing edge (Which should be true for all
        // blocks but one, the exit-block).
        let mut edges = Vec::with_capacity(blocks.len());
        for bb in blocks {
            let outgoing = outgoing(mir, bb);
            edges.extend(outgoing.into_iter());
        }
        edges.into_cow()
    }

    fn source(&self, edge: &Edge) -> Node {
        edge.source
    }

    fn target(&self, edge: &Edge) -> Node {
        let mir = self.mbcx.mir;
        mir.basic_block_data(edge.source).terminator().successors()[edge.index]
    }
}
