/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file CWirelessMsgHandler.cxx
    \author Romain BONDUE
    \date 17/02/2006 */
#include <sstream>

#include "CWirelessMsgHandler.h"
#include "SuWProtocolCodes.h"
#include "WE17Driver.h"
#include "ErrorsCode.h"



namespace
{
    template<typename T>
    T Extract (const std::string& Str) throw (nsErrors::CException)
    {
        std::istringstream Is (Str);
        T Value;
        Is >> Value;
        if (Is || Is.eof()) return Value;
        throw nsErrors::CException ("Invalid data read (from helper).",
                                    nsErrors::InvalidDataFromHelper);

    } // Extract()


    nsWireless::CEncryptionD ExtractEncryptionD (const std::string& Str)
                                                throw (nsErrors::CException)
    {
        int Encrypt, // Can't be EncryptType_e because of the extractor.
            Auth; // Ditto.
        std::string Key;
        std::istringstream Is (Str);
        Is >> Encrypt >> Auth;
        Is.ignore(); // The separator.
        getline (Is, Key);
        Is >> Encrypt >> Auth;
        Is.ignore(); // The separator.
        getline (Is, Key);
        if (Is || Is.eof())
            try
            {
                return nsWireless::CEncryptionD
                                    (Key.empty() ? nsWireless::CHexaKey()
                                                 : nsWireless::CHexaKey (Key),
                                     nsWireless::AuthType_e (Auth),
                                     nsWireless::EncryptType_e (Encrypt));
            }
            catch (const nsWireless::CHexaKey::CBadFormatExc&) {}
        throw nsErrors::CException ("Invalid data read (from helper).",
                                    nsErrors::InvalidDataFromHelper);

    } // ExtractEncryptionD()

} // anonymous namespace



nsRoot::CMsg nsRoot::CWirelessMsgHandler::operator () (const CMsg& Msg)
                                throw (nsErrors::CException, std::bad_alloc)
{
    try
    {
        if (Msg.GetCode() == nsWireless::DeviceNameC)
            SetDriver (Msg.GetText());
        else
        {
            if (!m_pDriver.get())
                throw nsErrors::CException ("No driver set.",
                                            nsErrors::NoDriverSet);
            switch (Msg.GetCode())
            {
              case nsWireless::SSIDC :
                m_pDriver->SetSSID (Msg.GetText());
              break;

              case nsWireless::FrequencyC :
                m_pDriver->SetFrequency (Extract<double> (Msg.GetText()));
              break;

              case nsWireless::ChannelC :
                m_pDriver->SetChannel (Extract<unsigned> (Msg.GetText()));
              break;
            
              case nsWireless::GEncryptionC :
              {
                  const nsWireless::CEncryptionD Descriptor
                                                (m_pDriver->GetEncryption());
                  std::ostringstream Os;
                  Os << Descriptor.GetEncrypt() << ' ' << Descriptor.GetAuth()
                     << ' ' << Descriptor.GetKey().GetStr();
                  return CMsg (Os.str());
              }
              //break;

              case nsWireless::SEncryptionC :
                m_pDriver->SetEncryption (ExtractEncryptionD (Msg.GetText()));
              break;

              case nsWireless::ModeC :
                m_pDriver->SetMode (nsWireless::Mode_e (Extract<int>
                                                            (Msg.GetText())));
              break;

              case nsWireless::TxRateC :
                m_pDriver->SetTxRate (Extract<int> (Msg.GetText()));
              break;

              case nsWireless::ApMacAddrC :
                m_pDriver->SetAPMacAddr (nsWireless::CMacAddress
                                                            (Msg.GetText()));
              break;

              case nsWireless::ScanC :
                m_pDriver->Scan();
              break;

              case nsWireless::CommitC :
                m_pDriver->Commit();
              break;

              default :
                throw nsErrors::CException ("Unknown command.",
                                            nsErrors::InvalidCommand);
            }
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        return CMsg (Exc.GetMsg(), Exc.GetCode());
    }
    return CMsg();

} // operator() ()


void nsRoot::CWirelessMsgHandler::SetDriver (const std::string& DeviceName)
                                throw (std::bad_alloc, nsErrors::CException)
{
    SetDriver (new nsWireless::CWE17Driver (DeviceName));

} // SetDriver()
