/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file RTModules.cxx
    \author Romain BONDUE
    \date 21/08/2005 */
#include <sstream>

#include "RTModules.h"
#include "GtkGUI.h"
#include "RTDrivers.h"
#include "RTDriversRoot.h"
#include "ErrorsCode.h"



namespace
{
    ::GtkWidget* LabelWidget (::GtkWidget* pWidget, const char* Label) throw()
    {
        ::GtkBox* const pHBox (GTK_BOX (::gtk_hbox_new (false, 2)));
        ::gtk_box_pack_start_defaults (pHBox, ::gtk_label_new (Label));
        ::gtk_box_pack_start_defaults (pHBox, pWidget);

        return GTK_WIDGET (pHBox);
        
    } // LabelWidget()


    const unsigned TabAuthRT2400Size (2);
    const nsWireless::AuthType_e TabAuthRT2400 [TabAuthRT2400Size] =
                                        {nsWireless::Open, nsWireless::Shared};

    const unsigned TabAuthRT25x0Size (3);
    const nsWireless::AuthType_e TabAuthRT25x0 [TabAuthRT25x0Size] =
                    {nsWireless::Open, nsWireless::Shared, nsWireless::WPAPSK};

    const unsigned TabAuthAdHocRT25x0Size (3);
    const nsWireless::AuthType_e TabAuthAdHocRT25x0 [TabAuthAdHocRT25x0Size] =
                {nsWireless::Open, nsWireless::Shared, nsWireless::WPANONE};

    const unsigned TabNoneEncryptSize (2);
    const nsWireless::EncryptType_e TabNoneEncrypt [TabNoneEncryptSize] =
                                        {nsWireless::None, nsWireless::WEP};

    const unsigned TabSharedEncryptSize (1);
    const nsWireless::EncryptType_e TabSharedEncrypt [TabSharedEncryptSize] =
                                                            {nsWireless::WEP};

    const unsigned TabWPAPSKRT2500EncryptSize (2);
    const nsWireless::EncryptType_e TabWPAPSKRT2500Encrypt
            [TabWPAPSKRT2500EncryptSize] = {nsWireless::TKIP, nsWireless::AES};

    const unsigned TabWPAPSKRT2570EncryptSize (1);
    const nsWireless::EncryptType_e TabWPAPSKRT2570Encrypt
                            [TabWPAPSKRT2570EncryptSize] = {nsWireless::TKIP};

        // FIXME Are they all really supported by rt2400 ?
    const unsigned TabModeRT2xx0Size (3);
    const nsWireless::Mode_e TabModeRT2xx0 [TabModeRT2xx0Size] =
                {nsWireless::Managed, nsWireless::AdHoc, nsWireless::Monitor};

} // anonymous namespace



const char* const nsCore::CRT2400Module::NewProfileDefaultName ("New profile");


const char* const nsCore::CRT2500Module::NewProfileDefaultName ("New profile");


const char* const nsCore::CRT2570Module::NewProfileDefaultName ("New profile");



nsGUI::CRT25x0ConfigDialog::CRT25x0ConfigDialog
                            (const std::vector<nsWireless::CFreq>& FreqVec,
                             const nsWireless::EncryptType_e* TabWPAEncrypt,
                             unsigned TabWPAEncryptSize) throw()
    : CSharedConfigDialog (TabModeRT2xx0, TabModeRT2xx0Size, FreqVec,
                           TabAuthRT25x0, TabAuthRT25x0Size,
                           TabNoneEncrypt, TabNoneEncryptSize,
                           TabSharedEncrypt, TabSharedEncryptSize,
                           TabWPAEncrypt, TabWPAEncryptSize)
{
    ::g_signal_connect (G_OBJECT (GetModeBox().operator ::GtkWidget*()),
                        "changed", G_CALLBACK (ModeChangedCallBack), this);

} // CRT25x0ConfigDialog


void nsGUI::CRT25x0ConfigDialog::ModeChangedCallBack (::GtkToggleButton*,
                                          CRT25x0ConfigDialog* This) throw()
{
    if (This->GetModeBox().GetActive() == nsWireless::AdHoc)
    {
        This->GetAuthBox().SetElem (TabAuthAdHocRT25x0,
                                    TabAuthAdHocRT25x0Size);
        This->GetAuthBox().SetActive (TabAuthAdHocRT25x0 [0]);
    }
    else
    {
        This->GetAuthBox().SetElem (TabAuthRT25x0, TabAuthRT25x0Size);
        This->GetAuthBox().SetActive (TabAuthRT25x0 [0]);
    }

} // ModeChangedCallBack()


nsCore::CRT2400Module::CRT2400Module (const std::string& DeviceName,
                                      const std::string& Path)
                                 throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (new nsWireless::CRT2400Driver (DeviceName)),
      m_ProfileEditor (new nsGUI::CSharedConfigDialog
                                      (TabModeRT2xx0, TabModeRT2xx0Size,
                                       m_pDriver->GetSupportedFreq(),
                                       TabAuthRT2400, TabAuthRT2400Size,
                                       TabNoneEncrypt, TabNoneEncryptSize,
                                       TabSharedEncrypt, TabSharedEncryptSize,
                                       0, 0)),
      m_ProfilesFile (Path)
{
    m_ProfilesFile.Extract();

} // CRT2400Module()


void nsCore::CRT2400Module::BecomeRoot () throw (nsErrors::CSystemExc,
                                                 std::bad_alloc)
{
    try
    {
        m_pDriver.reset (new nsWireless::CRT2400DriverRoot
                                                (m_pDriver->GetDeviceName()));
    }
    catch (const nsErrors::CException& Exc)
    {
        throw nsErrors::CSystemExc (Exc.GetMsg(), Exc.GetCode());
    }

} // BecomeRoot()


void nsCore::CRT2400Module::Connect (const nsWireless::CCell& Cell)
                                   throw (nsErrors::CSystemExc, std::bad_alloc)
{
    if (Cell.GetMode() == nsWireless::AdHoc);
    {
        if (Cell.GetChannel())
            m_pDriver->SetChannel (Cell.GetChannel());
        else throw nsErrors::CSystemExc ("Can't connect to an ad-hoc network"
                                         " this way, please use a profile.",
                                         nsErrors::RT2400AdHocLimitation);
    }
    m_pDriver->SetMode (nsWireless::GetMatchingMode (Cell.GetMode()));
    m_pDriver->SetSSID (Cell.GetSSID());
    m_pDriver->SetEncryption (Cell.GetEncryptionD());

} // Connect()


nsWireless::IDriver& nsCore::CRT2400Module::GetDriver () throw()
{
    return *m_pDriver;

} // GetDriver()


const nsWireless::IDriver& nsCore::CRT2400Module::GetDriver () const throw()
{
    return *m_pDriver;

} // GetDriver()


void nsCore::CRT2400Module::ApplyProfile (unsigned Pos)
                                                throw (nsErrors::CSystemExc)
{
    m_pDriver->SetChannel (m_ProfilesFile [Pos].GetChannel());
    m_pDriver->SetMode (m_ProfilesFile [Pos].GetMode());
    m_pDriver->SetEncryption (m_ProfilesFile [Pos].GetEncryptionD());
    m_pDriver->SetSSID (m_ProfilesFile [Pos].GetSSID());  

} // ApplyProfile()


nsUserData::CProfile* nsCore::CRT2400Module::EditNewProfile () throw()
{
    if (!m_ProfileEditor.Edit (m_ProfilesFile.back()))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return &m_ProfilesFile.back();

} // EditNewProfile()


nsCore::CRT2400Module::~CRT2400Module () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // ~CRT2400Module()


nsCore::CRT2500Module::~CRT2500Module () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // ~CRT2500Module()


nsCore::CRT2500Module::CRT2500Module (const std::string& DeviceName,
                                      const std::string& Path,
                                      nsGUI::CGtkGUI* pUI)
                                  throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (new nsWireless::CRT2500Driver (DeviceName)),
      m_ProfileEditor (new nsGUI::CRT25x0ConfigDialog
                      (m_pDriver->GetSupportedFreq(),
                       TabWPAPSKRT2500Encrypt, TabWPAPSKRT2500EncryptSize)),
      m_ProfilesFile (Path),
      m_pTable (GTK_TABLE (::gtk_table_new (5, 2, true))), m_pUI (pUI),
      m_CurrentActiveBoxItem (0), m_IsSkipped (false)
{
    m_ProfilesFile.Extract();
        // Build the private tab :
    ::gtk_container_set_border_width (GTK_CONTAINER (m_pTable), 6);
    ::gtk_table_set_col_spacings (m_pTable, 12);
    ::gtk_table_set_row_spacings (m_pTable, 8);

    const unsigned NbMode (2);
    const char* ModeNames [NbMode] = {"Mixed", "B Only"};
    CreateCBox (ModeNames, NbMode, m_pDriver->GetWirelessMode(),
                "Wireless Mode", 0, &ModeCallBack);

    const unsigned NbProtectionMode (3);
    const char* ProtectionModeNames [NbProtectionMode] = {"Auto", "On", "Off"};
    CreateCBox (ProtectionModeNames, NbProtectionMode,
                m_pDriver->GetBGProtection(), "B/G Protection", 1,
                &BGProtectionCallBack);

    const unsigned NbTxPreamble (3);
    const char* TxPreambleNames [NbTxPreamble] = {"Long", "Short", "Auto"};
    CreateCBox (TxPreambleNames, NbTxPreamble, m_pDriver->GetTxPreamble(),
                "Tx Preamble", 2, &TxPreambleCallBack);

    CreateCheckButton ("Tx burst", 3, 0, m_pDriver->GetTxBurst(),
                       &TxBurstCallBack);
    CreateCheckButton ("Turbo rate", 3, 1, m_pDriver->GetTurboRate(),
                                 &TurboRateCallBack);
    CreateCheckButton ("OFDM in ad-hoc mode", 4, 0,
                       m_pDriver->GetAdHocOFDM(), &AdHocOFDMCallBack);

    if (m_pDriver->IsRfmontxSupported())
        CreateCheckButton ("Injection in monitor mode", 4, 1,
                           m_pDriver->GetRfmontx(), &RfmontxCallBack);

} // CRT2500Module()


nsWireless::IDriver& nsCore::CRT2500Module::GetDriver () throw()
{
    return *m_pDriver;

} // GetDriver()


const nsWireless::IDriver& nsCore::CRT2500Module::GetDriver () const throw()
{
    return *m_pDriver;

} // GetDriver()


void nsCore::CRT2500Module::CreateCBox
        (const char* TabName [], unsigned TabSize, int CurrentActive,
         const char* Label, unsigned Row,
         void (*CallBack) (::GtkComboBox*, CRT2500Module*)) throw()
{

    ::GtkComboBox* const pCBox (GTK_COMBO_BOX (::gtk_combo_box_new_text()));
    for (unsigned i (0) ; i < TabSize ; ++i)
        ::gtk_combo_box_append_text (pCBox, TabName [i]);
    ::gtk_combo_box_set_active (pCBox, CurrentActive);
    ::g_signal_connect (G_OBJECT (pCBox), "changed", G_CALLBACK (CallBack),
                        this);
    ::g_signal_connect (G_OBJECT (pCBox), "set-focus-child",
                        G_CALLBACK (CBoxFocusCallBack), this);
    ::GtkWidget* const pLabel (::gtk_label_new (Label));
    ::gtk_table_attach_defaults (m_pTable, pLabel, 0, 1, Row, Row + 1);
    ::gtk_table_attach (m_pTable, GTK_WIDGET (pCBox), 1, 2, Row, Row + 1,
                        GTK_FILL, GTK_SHRINK, 0, 0);

} // CreateCBox()


void nsCore::CRT2500Module::CreateCheckButton (
         const char* Label, unsigned Row, unsigned Column, bool CurrentState,
         void (*CallBack) (::GtkToggleButton*, CRT2500Module*)) throw()
{
    ::GtkWidget* const pButton (::gtk_check_button_new_with_label (Label));
    ::gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pButton), CurrentState);
    ::g_signal_connect (G_OBJECT (pButton), "toggled", G_CALLBACK (CallBack),
                        this);
    ::gtk_table_attach_defaults (m_pTable, pButton, Column, Column + 1, Row,
                                 Row + 1);

} // CreateCheckButton()


nsUserData::CProfile* nsCore::CRT2500Module::EditNewProfile () throw()
{
    if (!m_ProfileEditor.Edit (m_ProfilesFile.back()))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return &m_ProfilesFile.back();

} // EditNewProfile()


void nsCore::CRT2500Module::Connect (const nsWireless::CCell& Cell)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    m_pDriver->SetMode (nsWireless::GetMatchingMode (Cell.GetMode()));
    if (Cell.GetMode() == nsWireless::AdHoc);
        m_pDriver->SetChannel (Cell.GetChannel());
    m_pDriver->SetSSID (Cell.GetSSID());
    m_pDriver->SetAPMacAddr (Cell.GetAPMacAddress());
    m_pDriver->SetEncryption (Cell.GetEncryptionD());

} // Connect()


void nsCore::CRT2500Module::ApplyProfile (unsigned Pos)
                                                throw (nsErrors::CSystemExc)
{
    if (m_ProfilesFile [Pos].GetMode() == nsWireless::AdHoc)
        m_pDriver->SetChannel (m_ProfilesFile [Pos].GetChannel());
    m_pDriver->SetMode (m_ProfilesFile [Pos].GetMode());
    m_pDriver->SetSSID (m_ProfilesFile [Pos].GetSSID());
    m_pDriver->SetEncryption (m_ProfilesFile [Pos].GetEncryptionD());

} // ApplyProfile()


void nsCore::CRT2500Module::SetDriver (nsWireless::CRT2500Driver* pDriver)
                                                                        throw()
{
    m_pDriver.reset (pDriver);

} // SetDriver()


void nsCore::CRT2500Module::BecomeRoot () throw (nsErrors::CSystemExc,
                                                 std::bad_alloc)
{
    try
    {
        SetDriver (new nsWireless::CRT2500DriverRoot
                                                (m_pDriver->GetDeviceName()));
    }
    catch (const nsErrors::CException& Exc)
    {
        throw nsErrors::CSystemExc (Exc.GetMsg(), Exc.GetCode());
    }

} // BecomeRoot()


void nsCore::CRT2500Module::TxBurstCallBack (::GtkToggleButton* pButton,
                                             CRT2500Module* This) throw()
{
    This->CommonCButtonCallBack (&nsWireless::CRT2500Driver::SetTxBurst,
                                 pButton);

} // TxBurstCallBack()


void nsCore::CRT2500Module::TurboRateCallBack (::GtkToggleButton* pButton,
                                               CRT2500Module* This) throw()
{
    This->CommonCButtonCallBack (&nsWireless::CRT2500Driver::SetTurboRate,
                                 pButton);

} // TurboRateCallBack()


void nsCore::CRT2500Module::AdHocOFDMCallBack (::GtkToggleButton* pButton,
                                               CRT2500Module* This) throw()
{
    This->CommonCButtonCallBack (&nsWireless::CRT2500Driver::SetAdHocOFDM,
                                 pButton);

} // AdHocOFDMCallBack()


void nsCore::CRT2500Module::RfmontxCallBack (::GtkToggleButton* pButton,
                                             CRT2500Module* This) throw()
{
    This->CommonCButtonCallBack (&nsWireless::CRT2500Driver::SetRfmontx,
                                 pButton);

} // RfmontxCallBack()


void nsCore::CRT2500Module::ModeCallBack (::GtkComboBox* pCBox,
                                          CRT2500Module* This) throw()
{
    This->CommonCBoxCallBack (reinterpret_cast<DriverCBoxMember_t>
                                (&nsWireless::CRT2500Driver::SetWirelessMode),
                              pCBox);

} // ModeCallBack()


void nsCore::CRT2500Module::BGProtectionCallBack (::GtkComboBox* pCBox,
                                                 CRT2500Module* This) throw()
{
    This->CommonCBoxCallBack (reinterpret_cast<DriverCBoxMember_t>
                                (&nsWireless::CRT2500Driver::SetBGProtection),
                              pCBox);

} // BGProtectionCallBack()


void nsCore::CRT2500Module::TxPreambleCallBack (::GtkComboBox* pCBox,
                                                CRT2500Module* This) throw()
{
    This->CommonCBoxCallBack (reinterpret_cast<DriverCBoxMember_t>
                                (&nsWireless::CRT2500Driver::SetTxPreamble),
                              pCBox);

} // TxPreambleCallBack()


void nsCore::CRT2500Module::CommonCBoxCallBack (DriverCBoxMember_t Member,
                                                ::GtkComboBox* pCBox) throw()
{
    try
    {
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
            {
                const unsigned CurrentActive
                                        (::gtk_combo_box_get_active (pCBox));
                (m_pDriver.get()->*Member) (CurrentActive);
                m_CurrentActiveBoxItem = CurrentActive;
            }
            else ResumePreviousState (pCBox);
        }
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
        ResumePreviousState (pCBox);
        m_pUI->ProcessError (Exc);
    }

} // CommonCBoxCallBack()


void nsCore::CRT2500Module::CommonCButtonCallBack
            (DriverButtonMember_t Member, ::GtkToggleButton* pButton) throw()
{
    try
    {
        const bool NewState (::gtk_toggle_button_get_active (pButton));
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
                (m_pDriver.get()->*Member) (NewState);
            else
            {
                m_IsSkipped = true;
                ::gtk_toggle_button_set_active (pButton, !NewState);
                m_IsSkipped = false;
            }
        }
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
        m_pUI->ProcessError (Exc);
    }

} // CommonCButtonCallBack()


void nsCore::CRT2500Module::CBoxFocusCallBack (::GtkComboBox* pCBox, bool,
                                        nsCore::CRT2500Module* This) throw()
{
    This->m_CurrentActiveBoxItem = ::gtk_combo_box_get_active (pCBox);

} // CBoxFocusCallBack()


nsCore::CRT2570Module::CRT2570Module (const std::string& DeviceName,
                                      const std::string& Path,
                                      nsGUI::CGtkGUI* pUI)
                                throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (new nsWireless::CRT2570Driver (DeviceName)),
      m_ProfileEditor (new nsGUI::CRT25x0ConfigDialog
                      (m_pDriver->GetSupportedFreq(),
                       TabWPAPSKRT2570Encrypt, TabWPAPSKRT2570EncryptSize)),
      m_ProfilesFile (Path), m_pUI (pUI),
      m_pBox (GTK_BOX (::gtk_vbox_new (true, 2))), m_CurrentActiveBoxItem (0),
      m_IsSkipped (false)
{
    m_ProfilesFile.Extract();
    BuildPrivateTab();

} // CRT2570Module()


void nsCore::CRT2570Module::BecomeRoot () throw (nsErrors::CSystemExc,
                                                 std::bad_alloc)
{
    try
    {
        m_pDriver.reset (new nsWireless::CRT2570DriverRoot
                                                (m_pDriver->GetDeviceName()));
    }
    catch (const nsErrors::CException& Exc)
    {
        throw nsErrors::CSystemExc (Exc.GetMsg(), Exc.GetCode());
    }

} // BecomeRoot()


nsWireless::IDriver& nsCore::CRT2570Module::GetDriver () throw()
{
    return *m_pDriver;

} // GetDriver()


const nsWireless::IDriver& nsCore::CRT2570Module::GetDriver () const throw()
{
    return *m_pDriver;

} // GetDriver()


void nsCore::CRT2570Module::Connect (const nsWireless::CCell& Cell)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    m_pDriver->SetMode (nsWireless::GetMatchingMode (Cell.GetMode()));
    m_pDriver->SetChannel (Cell.GetChannel());
    m_pDriver->SetSSID (Cell.GetSSID());
    m_pDriver->SetAPMacAddr (Cell.GetAPMacAddress());
    m_pDriver->SetEncryption (Cell.GetEncryptionD());
    // ? FIXME ? Does the SSID need to be set again? (WPA)

} // Connect()


void nsCore::CRT2570Module::ApplyProfile (unsigned Pos)
                                                throw (nsErrors::CSystemExc)
{
    if (m_ProfilesFile [Pos].GetMode() == nsWireless::AdHoc)
        m_pDriver->SetChannel (m_ProfilesFile [Pos].GetChannel());
    m_pDriver->SetMode (m_ProfilesFile [Pos].GetMode());
    m_pDriver->SetSSID (m_ProfilesFile [Pos].GetSSID());
    m_pDriver->SetEncryption (m_ProfilesFile [Pos].GetEncryptionD());
    // ? FIXME ? Does the SSID need to be set again? (WPA)

} // ApplyProfile()


nsCore::CRT2570Module::~CRT2570Module () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // ~CRT2570Module()


nsUserData::CProfile* nsCore::CRT2570Module::EditNewProfile () throw()
{
    if (!m_ProfileEditor.Edit (m_ProfilesFile.back()))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return &m_ProfilesFile.back();

} // EditNewProfile()


void nsCore::CRT2570Module::CBoxFocusCallBack (::GtkComboBox* pCBox, bool,
                                        nsCore::CRT2570Module* This) throw()
{
    This->m_CurrentActiveBoxItem = ::gtk_combo_box_get_active (pCBox);

} // CBoxFocusCallBack()


void nsCore::CRT2570Module::CommonCButtonCallBack
            (DriverButtonMember_t pMember, ::GtkToggleButton* pButton) throw()
{
    try
    {
        const bool NewState (::gtk_toggle_button_get_active (pButton));
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
                (m_pDriver.get()->*pMember) (NewState);
            else
            {
                m_IsSkipped = true;
                ::gtk_toggle_button_set_active (pButton, !NewState);
                m_IsSkipped = false;
            }
        }
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
        m_pUI->ProcessError (Exc);
    }

} // CommonCButtonCallBack()


void nsCore::CRT2570Module::RfmontxCallBack (::GtkToggleButton* pButton,
                                             CRT2570Module* This) throw()
{
    This->CommonCButtonCallBack (&nsWireless::CRT2570Driver::SetRfmontx,
                                 pButton);

} // RfmontxCallBack()


void nsCore::CRT2570Module::PrismHeaderCallBack (::GtkComboBox* pCBox,
                                                 CRT2570Module* This) throw()
{
    This->CommonCBoxCallBack (reinterpret_cast<DriverBoxMember_t>
                                (&nsWireless::CRT2570Driver::SetPrismHeader),
                              pCBox);

} // PrismHeaderCallBack()


void nsCore::CRT2570Module::AdhocModeCallBack (::GtkComboBox* pCBox,
                                               CRT2570Module* This) throw()
{
    This->CommonCBoxCallBack (reinterpret_cast<DriverBoxMember_t>
                                (&nsWireless::CRT2570Driver::SetAdHocMode),
                              pCBox);

} // AdhocModeCallBack()


void nsCore::CRT2570Module::CommonCBoxCallBack (DriverBoxMember_t pMember,
                                                ::GtkComboBox* pCBox) throw()
{
    try
    {
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
            {
                const int CurrentActive (::gtk_combo_box_get_active (pCBox));
                (m_pDriver.get()->*pMember) (CurrentActive);
                m_CurrentActiveBoxItem = CurrentActive;
            }
            else ResumePreviousState (pCBox);
        }
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
        ResumePreviousState (pCBox);
        m_pUI->ProcessError (Exc);
    }

} // AdhocModeCallBack()


void nsCore::CRT2570Module::BuildPrivateTab () throw()
{
    ::GtkComboBox* pCBox (GTK_COMBO_BOX (::gtk_combo_box_new_text()));
    ::gtk_combo_box_append_text (pCBox, "Standard 11Mpbs");
    ::gtk_combo_box_append_text (pCBox, "Auto");
    ::gtk_combo_box_append_text (pCBox, "54Mbps only");
    ::gtk_combo_box_set_active (pCBox, 0);
    ::g_signal_connect (G_OBJECT (pCBox), "changed",
                        G_CALLBACK (AdhocModeCallBack), this);
    ::g_signal_connect (G_OBJECT (pCBox), "set-focus-child",
                        G_CALLBACK (CBoxFocusCallBack), this);
    ::GtkBox* pHbox (GTK_BOX (::gtk_hbox_new (true, 2)));
    ::gtk_box_pack_start_defaults (pHbox, ::gtk_label_new ("Ad-hoc mode"));
    ::gtk_box_pack_start_defaults (pHbox, GTK_WIDGET (pCBox));
    ::gtk_box_pack_start_defaults (m_pBox, GTK_WIDGET (pHbox));

    if (m_pDriver->IsRfmontxSupported())
    {
        ::GtkWidget* const pButton (::gtk_toggle_button_new_with_label
                                                ("Injection in monitor mode"));
        ::g_signal_connect (G_OBJECT (pButton), "toggled",
                            G_CALLBACK (RfmontxCallBack), this);
        ::gtk_box_pack_start_defaults (m_pBox, pButton);
    }
    if (m_pDriver->IsPrismHeaderSupported())
    {
        pCBox = GTK_COMBO_BOX (::gtk_combo_box_new_text());
        ::gtk_combo_box_append_text (pCBox, "Default (auto)");
        ::gtk_combo_box_append_text (pCBox, "Force");
        ::gtk_combo_box_append_text (pCBox, "Force no");
        ::gtk_combo_box_set_active (pCBox, 0);
        ::g_signal_connect (G_OBJECT (pCBox), "changed",
                            G_CALLBACK (PrismHeaderCallBack), this);
        ::g_signal_connect (G_OBJECT (pCBox), "set-focus-child",
                            G_CALLBACK (CBoxFocusCallBack), this);
        pHbox = GTK_BOX (::gtk_hbox_new (true, 2));
        ::gtk_box_pack_start_defaults (pHbox, ::gtk_label_new
                                                            ("Prism header"));
        ::gtk_box_pack_start_defaults (pHbox, GTK_WIDGET (pCBox));
        ::gtk_box_pack_start_defaults (m_pBox, GTK_WIDGET (pHbox));
    }

} // BuildPrivateTab()
