/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file RTDrivers.cxx
    \author Romain BONDUE
    \date 30/08/2005 */
#include <cstring>
#include <cstdio>
#include <sstream>
#ifndef NDEBUG
#include <iostream>
#endif // NDEBUG

extern "C"{
#include <sys/ioctl.h> // SIOCDEVPRIVATE
}

#include "RTDrivers.h"
#include "ErrorsCode.h"



namespace
{
    class CScanResult
    {
      protected :
        struct SSSid
        {
            unsigned Length;
            unsigned char Text [32];

        }; // SSSid

        struct SConfigFH
        {
            unsigned m1;
            unsigned m2;
            unsigned m3;
            unsigned m4;

        }; // SConfigFH

        struct SConfig
        {
            unsigned m1;
            unsigned m2;
            unsigned m3;
            unsigned Frequency;
            SConfigFH m5;

        }; // SConfig

        struct SEncryptionD
        {
            unsigned char Id;
            unsigned char Length;
            unsigned char Data [32];

        }; // SEncryptionD

        struct SEncryptionOffset
        {
            unsigned char m1 [8];
            unsigned short m2;
            unsigned short m3;

        }; // SEncryptionOffset


      private :
        static const unsigned NbRates = 16;

        unsigned m_Length;
        char m_MacAddress [6];
        unsigned char m_Reserved [2];
        SSSid m_SSID;
        unsigned m_Privacy;
        int m_SignalStrength;
        int m_NetworkType;
        SConfig m_Config;
        int m_NetworkInfrastructureType;
        unsigned char m_SupportedRates [NbRates];
        unsigned m_IELength;
        unsigned char m_pIEs [sizeof (SEncryptionOffset) +
                                                  sizeof (SEncryptionD) * 4];


      public :
        CScanResult () throw() {memset (this, 0, sizeof (CScanResult));}

        unsigned GetLength () const throw() {return m_Length;}

        nsWireless::CMacAddress GetAPMacAddress () const throw (std::bad_alloc)
        {
            return nsWireless::CMacAddress (m_MacAddress);

        } // GetAPMacAddress()

        nsWireless::Mode_e GetMode () const throw()
        {
            switch (m_NetworkInfrastructureType)
            {
              case 0 : return nsWireless::AdHoc;
              case 1 : return nsWireless::Managed;
              default : return nsWireless::Auto;
            }

        } // GetMode()

            // Most things here are magic, ask Ralink...
        nsWireless::CEncryptionD GetEncryptionD () const throw()
        {
            nsWireless::CEncryptionD Descriptor;
            if (m_Privacy)
            {       // Check if it's WPA :
                const SEncryptionD* const pDescriptor
                                (reinterpret_cast<const SEncryptionD* const>
                                        (m_pIEs + sizeof (SEncryptionOffset)));
#ifndef NDEBUG
                std::cerr << std::hex << "Ralink magic encryption function :\n"
                             "\tpDescriptor->Id : "
                          << unsigned (pDescriptor->Id)
                          << "\n\tpDescriptor->Length : "
                          << unsigned (pDescriptor->Length)
                          << "\n\tpDescriptor->Data : "
                          << *reinterpret_cast<const unsigned*>
                                                          (pDescriptor->Data)
                          << "\n\tpDescriptor->Data [9] : "
                          << unsigned (pDescriptor->Data [9]) << std::dec
                          << std::endl;
#endif // NDEBUG
                if (pDescriptor->Id == 221 && pDescriptor->Length >= 16 &&
                    *reinterpret_cast<const unsigned*> (pDescriptor->Data)
                                                            == 0x01F25000)
                {
                    switch (pDescriptor->Data [9])
                    {
                      case 2 :
                        Descriptor.SetEncrypt (nsWireless::TKIP);
                      break;

                      case 3 : // AES-WRAP
                      case 4 : // AES-CCMP
                        Descriptor.SetEncrypt (nsWireless::AES);
                      //break;
                    }
                    if (m_NetworkInfrastructureType)
                        Descriptor.SetAuth (nsWireless::WPAPSK);
                    else
                        Descriptor.SetAuth (nsWireless::WPANONE);
                }
                else Descriptor.SetEncrypt (nsWireless::WEP);
                    /* FIXME We don't know if it's "shared" or not.
                     * Dig more in Ralink stuffs? (It seems that the driver
                     * just not reports it : get_site_survey (rt61/73) does
                     * not have that piece of information too.)
                     * RaConfig on Windows neither, *I think*. */
            }
            return Descriptor;

        } // GetEncryptionD()

        std::string GetSSID () const throw()
        {
            return std::string (reinterpret_cast<const char*> (m_SSID.Text), 0,
                                m_SSID.Length);
        
        } // GetSSID()

        double GetFrequency () const throw()
        {
            return m_Config.Frequency / 1000000.0;

        } // GetFrequency()

        nsWireless::CQuality GetQuality () const throw()
        {
            return nsWireless::CQuality (0, m_SignalStrength);

        } // GetQuality()

        unsigned GetTxRate () const throw()
        {
            return 0;

        } // GetTxRate()

    }; // CScanResult


    template<unsigned MaxSize>
    class CScanResultsTab
    {
      public :
        CScanResultsTab () throw() : m_Size (MaxSize)
        {
            memset (m_Data, 0, sizeof (CScanResult) * MaxSize);

        } // CScanResulTab()

        const CScanResult* Get () const throw()
        {
            return m_Data;

        } // Get()

        unsigned Size () const throw() {return m_Size;}


      private :
        unsigned m_Size;
        CScanResult m_Data [MaxSize];

    }; // CScanResultsTab


    struct SPrivateConfig
    {
        SPrivateConfig () throw() {memset (this, 0, sizeof (SPrivateConfig));}

        unsigned EnableTxBurst;
        unsigned EnableTurboRate;
            // Ralink : 0-AUTO, 1-always ON, 2-always OFF.
        unsigned UseBGProtection;
            // Ralink : 0-no use, 1-use 9-use short slot time when applicable.
        unsigned UseShortSlotTime;
        unsigned UseOfdmRatesIn11gAdhoc;

        unsigned RadioOn;
        unsigned m2;
        unsigned m3;

    }; // SPrivateConfig


    enum DriverWirelessMode_e {BGMixed, B, A, ABGMixed};

    enum DriverEncrypt_e {WEP, TKIP = 4, AES = 6};

    enum DriverAuth_e {Open, Shared, WPA = 3, WPAPSK, WPANone};


    nsWireless::CEncryptionD ToEncryptionDRt73 (const char* Enc,
                                                const char* Auth) throw()
    {
        nsWireless::AuthType_e AuthType;
        if (!strcmp (Auth, "WPA-PSK"))
            AuthType = nsWireless::WPAPSK;
        else if (!strcmp (Auth, "WPA2-PSK"))
            AuthType = nsWireless::WPA2PSK;
        else
            AuthType = nsWireless::GetAuthFromName (Auth);
        return nsWireless::CEncryptionD (nsWireless::CHexaKey(), AuthType,
                                         nsWireless::GetEncryptFromName (Enc));

    } // ToEncryptionDRt73()


    nsWireless::CEncryptionD ToEncryptionDRt61 (const char* Enc,
                                                const char* Auth) throw()
    {
            /* FIXME Are all these choices correct ? The most secure scheme is
                     reported each time. */
        nsWireless::AuthType_e AuthType;
        if (!strcmp (Auth, "WPA-PSK") || !strcmp (Auth, "WPA/WPA-PSK"))
            AuthType = nsWireless::WPAPSK;
        else if (!strcmp (Auth, "WPA2-PSK") || !strcmp (Auth, "WPA(2)-PSK") ||
                 !strcmp (Auth, "WPA2/WPA2-PSK") ||
                 !strcmp (Auth, "WPA(2)/WPA(2)-PSK"))
            AuthType = nsWireless::WPA2PSK;
        else
            AuthType = nsWireless::GetAuthFromName (Auth);
        nsWireless::EncryptType_e EncType;
        if (!strcmp (Enc, "TKIP, AES"))
            EncType = nsWireless::AES;
        else
            EncType = nsWireless::GetEncryptFromName (Enc);
        return nsWireless::CEncryptionD (nsWireless::CHexaKey(), AuthType,
                                         EncType);

    } // ToEncryptionDRt61()


    inline nsWireless::Mode_e ToMode (const char* Mode) throw()
    {
        if (!strcmp (Mode, "Adhoc"))
            return nsWireless::AdHoc;
        return nsWireless::Managed;

    } // ToMode()


    nsWireless::CQuality ToQuality (int Rssi) throw()
    {
        return nsWireless::CQuality (0, Rssi);

    } // ToQuality()



        // 8 8 36 20 12 12 12
        // Channel RSSI SSID Bssid WepStatus AuthMode NetworkType
    nsWireless::CCell ToCellRt73 (const char* Line) throw (nsErrors::CException)
    {       // C style :
        unsigned Channel;
        int Rssi;
        if (sscanf (Line, "%u%d", &Channel, &Rssi) != 2)
            throw nsErrors::CException ("Data parsing error (1).",
                                        nsErrors::DataParsingError1);
        Line += 17;
        int End (35);
        while (End >= 0 && Line [End] == ' ')
            --End;
        const std::string Ssid (Line, 0, End + 1);
        Line += 36;
        char Bssid [21];
        char Enc [13];
        char Auth [13];
        char NetworkType [13];
        if (sscanf (Line, "%s%s%s%s", Bssid, Enc, Auth, NetworkType) != 4)
            throw nsErrors::CException ("Data parsing error (2).",
                                        nsErrors::DataParsingError2);
#ifndef NDEBUG
        std::cerr << '\t' << Channel << ' ' << Rssi << " #" << Ssid << "# "
                  << Bssid << ' ' << Enc << ' ' << Auth << ' ' << NetworkType
                  << std::endl;
#endif // NDEBUG
        return nsWireless::CCell (nsWireless::CMacAddress (std::string (Bssid)),
                                  ToMode (NetworkType), Ssid,
                                  ToEncryptionDRt73 (Enc, Auth), Channel,
                                  ToQuality (Rssi), 0);

    } // ToCellRt73()


        // 8 8 36 20 12 20 12
        // Channel RSSI SSID Bssid WepStatus AuthMode NetworkType
    nsWireless::CCell ToCellRt61 (const char* Line) throw (nsErrors::CException)
    {       // C style :
        unsigned Channel;
        int Rssi;
        if (sscanf (Line, "%u%d", &Channel, &Rssi) != 2)
            throw nsErrors::CException ("Data parsing error (1).",
                                        nsErrors::DataParsingError1);
        Line += 17;
#ifndef NDEBUG
        std::cerr << "Line : " << Line << std::endl;
#endif // NDEBUG
        int End (35);
        while (End >= 0 && Line [End] == ' ')
            --End;
        const std::string Ssid (Line, 0, End + 1);
        Line += 36;
        char Bssid [21];
        char Enc [13];
        char Auth [21];
        char NetworkType [13];
        if (sscanf (Line, "%s%s%s%s", Bssid, Enc, Auth, NetworkType) != 4)
            throw nsErrors::CException ("Data parsing error (2).",
                                        nsErrors::DataParsingError2);
#ifndef NDEBUG
        std::cerr << '\t' << Channel << ' ' << Rssi << " #" << Ssid << "# "
                  << Bssid << ' ' << Enc << ' ' << Auth << ' ' << NetworkType
                  << std::endl;
#endif // NDEBUG
        return nsWireless::CCell (nsWireless::CMacAddress (std::string (Bssid)),
                                  ToMode (NetworkType), Ssid,
                                  ToEncryptionDRt61 (Enc, Auth), Channel,
                                  ToQuality (Rssi), 0);

    } // ToCellRt61()

} // anonymous namespace



void nsWireless::CRTDriver::GetScanResult (int IoctlCode, u16 ListOid,
                                           std::vector<CCell>& CellVec) const
                                throw (nsErrors::CException, std::bad_alloc)
{
#ifndef NDEBUG
    std::cerr << "CRTDriver::GetScanResult()\n";
#endif // NDEBUG
    CScanResultsTab<64> Results;
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (&Results);
    m_Data.data.length = sizeof (Results);
    m_Data.data.flags = ListOid;
    Ioctl (IoctlCode, "Can't get scan results through special ioctl.");
#ifndef NDEBUG
    std::cerr << "CRTDriver::GetScanResult() Results.size() : "
              << Results.Size() << std::endl;
    std::cerr << "\tRaw data :\n";
    const unsigned char* p (reinterpret_cast<const unsigned char*> (&Results));
    for (unsigned i (0) ; i < m_Data.data.length ; ++i)
        std::cerr << unsigned (p [i]) << ' ';
    std::cerr << std::endl;
#endif // NDEBUG

    const CScanResult* pResult (Results.Get());
    for (unsigned i (0) ; i < Results.Size() ; ++i)
    {
#ifndef NDEBUG
    std::cerr << "CRTDriver::GetScanResult() Results : " << i << std::endl;
#endif // NDEBUG
        unsigned Channel (GetMatchingFreq (0,
                                       pResult->GetFrequency()).GetChannel());
        CellVec.push_back (CCell (pResult->GetAPMacAddress(),
                                  pResult->GetMode(), pResult->GetSSID(),
                                  pResult->GetEncryptionD(), Channel,
                                  pResult->GetQuality(),
                                  pResult->GetTxRate()));
        pResult = reinterpret_cast<const CScanResult*>
                                    (reinterpret_cast<const char*> (pResult) +
                                                        pResult->GetLength());
    }

} // GetScanResult()


nsWireless::CRT2400Driver::CRT2400Driver (const std::string& DeviceName)
                                                throw (nsErrors::CException)
    : CRTDriver (DeviceName), m_PrivateIoctl (SIOCDEVPRIVATE) {}


void nsWireless::CRT2400Driver::Scan () throw (nsErrors::CSystemExc)
{
    m_Data.data.pointer = 0;
    m_Data.data.length = 0;
    m_Data.data.flags = 8002U; // RTLINUX_SET_OID_802_11_BSSID_LIST_SCAN
    Ioctl (m_PrivateIoctl, "Can't trigger scan through special ioctl.");

} // Scan()


void nsWireless::CRTAdvancedDriver::GetScanResult (u16 MagicNumber,
std::vector<CCell>& CellVec) const throw (nsErrors::CException, std::bad_alloc)
{
#ifndef NDEBUG
    std::cerr << "CRTAdvancedDriver::GetScanResult() Standard scan"
              << std::endl;
#endif // NDEBUG
    CWE17Driver::GetScanResult (CellVec);
#ifndef NDEBUG
    std::cerr << "CRTAdvancedDriver::GetScanResult() Standard scan success, "
              << CellVec.size() << " cells." << std::endl;
#endif // NDEBUG
    try
    {
        std::vector<CCell> RalinkCellVec;
#ifndef NDEBUG
    std::cerr << "CRTAdvancedDriver::GetScanResult() Ralink scan"
              << std::endl;
#endif // NDEBUG
        CRTDriver::GetScanResult (m_PrivateIoctl, MagicNumber, RalinkCellVec);
#ifndef NDEBUG
    std::cerr << "CRTAdvancedDriver::GetScanResult() Ralink scan success, "
              << RalinkCellVec.size() << " cells." << std::endl;
#endif // NDEBUG
        for (unsigned i (0) ; i < RalinkCellVec.size() && i < CellVec.size() ;
                                                                            ++i)
            if (RalinkCellVec [i].GetEncryptionD().GetAuth() == WPAPSK ||
                RalinkCellVec [i].GetEncryptionD().GetAuth() == WPANONE)
                    // Both results seem to be always in the same order.
                CellVec [i].GetEncryptionD() =
                                            RalinkCellVec [i].GetEncryptionD();
    }
    catch (const nsErrors::CException& Exc)
    {
#ifndef NDEBUG
        std::cerr << "CRTAdvancedDriver::GetScanResult() Ralink scan failed : "
                  << Exc.GetMsg() << ' ' << Exc.GetCode() << std::endl;
#endif // NDEBUG
    } // We have some results anyway.

} // GetScanResult()


nsWireless::CEncryptionD nsWireless::CRTAdvancedDriver::GetEncryption
                                (u16 EncryptMagicNumber, u16 AuthMagicNumber)
                                    throw (nsErrors::CSystemExc, std::bad_alloc)
{
#ifndef NDEBUG
    std::cerr << "CRTAdvancedDriver::GetEncryption()" << std::endl;
#endif // NDEBUG
    CEncryptionD Descriptor (CWE17Driver::GetEncryption());
    if (Descriptor.GetEncrypt() != None)
    {
            // FIXME is this code really working?
        DriverEncrypt_e Encrypt (::WEP);
        m_Data.data.pointer = reinterpret_cast< ::caddr_t> (Encrypt);
        m_Data.data.length = sizeof (Encrypt);
        m_Data.data.flags = EncryptMagicNumber;
        Ioctl (m_PrivateIoctl, "Can't get cipher through special ioctl.");
        DriverAuth_e Auth (::Open);
        m_Data.data.pointer = reinterpret_cast< ::caddr_t> (Auth);
        m_Data.data.length = sizeof (Auth);
        m_Data.data.flags = AuthMagicNumber;
        Ioctl (m_PrivateIoctl,
               "Can't get authentication type through special ioctl.");
        switch (Encrypt)
        {
          case ::TKIP : Descriptor.SetEncrypt (nsWireless::TKIP);
          break;

          case ::AES : Descriptor.SetEncrypt (nsWireless::AES);
          //break;

          default : ; // To avoid warnings.
        }
        switch (Auth)
        {
          case ::WPANone :
            Descriptor.SetAuth (nsWireless::WPANONE);
          break;

          case ::WPAPSK :
            Descriptor.SetAuth (nsWireless::WPAPSK);
          //break;

          default : ; // To avoid warnings.
        }
    }
    return Descriptor;

} // GetEncryption()


void nsWireless::CRTAdvancedDriver::SetRfmontx (bool B)
                                                    throw (nsErrors::CSystemExc)
{
    char Buffer (B ? 1 : 0);
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (&Buffer);
    m_Data.data.length = sizeof (Buffer);
    m_Data.data.flags = 0;
    Ioctl (m_RfmontxIoctl, "Can't set injection mode.");
    m_Flags |= Rfmontx;

} // SetRfmontx()


nsWireless::CRT2500Driver::CRT2500Driver (const std::string& DeviceName)
                                    throw (nsErrors::CException, std::bad_alloc)
        /* RT_PRIV_IOCTL, RT_OID_802_11_STA_CONFIG, RT_OID_802_11_PREAMBLE and
           RT_OID_PHY_MODE */
    : CRTAdvancedDriver (DeviceName, SIOCIWFIRSTPRIV + 1, 0x0217U, 0x0201U,
                         0x0212U, new IWPrivHandler)
{                                   // Backward compatibilty check.
    if (GetRfmontxIoctl() != InvalidIoctl && GetRfmontxIoctl() & 1)
    {
#ifndef NDEBUG
        std::cerr << "CRT2500Driver::CRT2500Driver() GetRfmonTx backward"
                     " compatibilty check." << std::endl;
#endif // NDEBUG
        char Buffer ('0');
        m_Data.data.pointer = reinterpret_cast< ::caddr_t> (&Buffer);
        m_Data.data.length = 0; // sizeof (Buffer) => EINVAL
        m_Data.data.flags = 0;
        Ioctl (GetRfmontxIoctl(), "Can't get injection mode through private"
                                  " ioctl.");
        SetFlag (Rfmontx, Buffer == '1');
#ifndef NDEBUG
        std::cerr << "CRT2500Driver::CRT2500Driver() GetRfmonTx backward"
                     " compatibilty check : " << (Buffer == '1') << std::endl;
#endif // NDEBUG
    }

} // CRT2500Driver()


void nsWireless::CRT2500Driver::SetEncryption (const CEncryptionD& Descriptor)
                                    throw (nsErrors::CSystemExc, std::bad_alloc)
{
    if (Descriptor.GetAuth() == Open || Descriptor.GetAuth() == Shared)
    {
#ifndef NDEBUG
        std::cerr << "CRT2500Driver::SetEncryption(), using standard ioctl."
                  << std::endl;
#endif // NDEBUG
        CWE17Driver::SetEncryption (Descriptor);
    }
    else
    {
        const char* const AuthName
                                (GetAuthName (Descriptor.GetAuth()).c_str());
#ifndef NDEBUG
        std::cerr << "CRT2500Driver::SetEncryption(), using private ioctls.\n\t"
                     "AuthName : " << AuthName << "\n\tEncryptName : "
                  << GetEncryptName (Descriptor.GetEncrypt()) << "\n\tKey : "
                  << std::string (Descriptor.GetKey().Get(), 0,
                                     Descriptor.GetKey().Size()) << std::endl;
#endif // NDEBUG
        SetIoctl ("AuthMode", AuthName, "Can't set authentication mode : ");
        SetIoctl ("EncrypType", GetEncryptName (Descriptor.GetEncrypt()),
                  "Can't set encryption type : ");
        SetIoctl (AuthName, std::string (Descriptor.GetKey().Get(), 0,
                                         Descriptor.GetKey().Size()),
                  "Can't set key : ");
    }

} // SetEncryption()


nsWireless::CRTAdvancedDriver::CRTAdvancedDriver (const std::string& DeviceName,
            int PrivateIoctl, u16 StaConfigMagicNumber, u16 PreambleMagicNumber,
            u16 PhyModeMagicNumber, IWPrivHandler* pHandler)
                                    throw (nsErrors::CException, std::bad_alloc)
    : CRTDriver (DeviceName), m_PrivateIoctl (PrivateIoctl),
      m_SetIoctl (InvalidIoctl), m_RfmontxIoctl (InvalidIoctl), m_Flags (0),
      m_BGProtection (AutoProtection), m_TxPreamble (AutoPreamble)
{
    try
    {
        const unsigned TabArgsSize (16);
        ::iw_priv_args TabArgs [TabArgsSize];
        for (unsigned i (GetPrivateIoctls (TabArgs, TabArgsSize)) ; i ; )
            if (!strcmp (TabArgs [--i].name, "set"))
                m_SetIoctl = TabArgs [i].cmd;
            else if (!strcmp (TabArgs [i].name, "rfmontx"))
                m_RfmontxIoctl = TabArgs [i].cmd;
            else if (!strcmp (TabArgs [i].name, "get_rfmontx"))
            {
                int* const pResult (reinterpret_cast<int*> (m_Data.name));
                *pResult = 0;
                Ioctl (TabArgs [i].cmd, "Can't get rfmontx status.");
                if (*pResult) m_Flags |= Rfmontx;
            }
            else
                (*pHandler) (TabArgs [i]);
    }
    catch (...)
    {
        delete pHandler;
        throw;
    }
    delete pHandler;

    if (m_SetIoctl == InvalidIoctl)
        throw nsErrors::CException ("Can't find \"set\" private ioctl.",
                                    nsErrors::RTAdvancedSetIoctlNotFound);
        // Get STA config :
    SPrivateConfig PrivateConfig;
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (&PrivateConfig);
    m_Data.data.length = sizeof (PrivateConfig);
    m_Data.data.flags = StaConfigMagicNumber;
    Ioctl (m_PrivateIoctl, "Can't get STA config through special ioctl.");
    if (PrivateConfig.EnableTxBurst) m_Flags |= TxBurst;
    if (PrivateConfig.EnableTurboRate) m_Flags |= TurboRate;
    m_BGProtection = BGProtection_e (PrivateConfig.UseBGProtection);
    if (PrivateConfig.UseOfdmRatesIn11gAdhoc) m_Flags |= AdHocOFDM;

        // Get TxPreamble :
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (&m_TxPreamble);
    m_Data.data.length = sizeof (m_TxPreamble);
    m_Data.data.flags = PreambleMagicNumber;
    Ioctl (m_PrivateIoctl, "Can't get Tx Preamble through special ioctl.");

        // Get wireless mode :
    DriverWirelessMode_e Mode;
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (&Mode);
    m_Data.data.length = sizeof (Mode);
    m_Data.data.flags = PhyModeMagicNumber;
    Ioctl (m_PrivateIoctl, "Cant get wireless mode through special ioctl.");
    if (Mode == B) m_Flags |= IsB_Only;

} // CRTAdvancedDriver()


void nsWireless::CRTAdvancedDriver::SetIoctl (const char* Command,
                                              const std::string& Parameter,
                                              const std::string& ErrorMsg) const
                                                throw (nsErrors::CSystemExc)
{
    const unsigned CstBufferSize (256); // Should be enough.
    char Buffer [CstBufferSize + 2]; // '=' and '\0'.
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (Buffer);
    m_Data.data.length = 0;
    m_Data.data.flags = 0;
 
    char* BufferIter (Buffer);
    while (m_Data.data.length < CstBufferSize && *Command)
    {
        *BufferIter++ = *Command++;
        ++m_Data.data.length;
    }
    *BufferIter = '=';
    ++m_Data.data.length;
    std::string::const_iterator ParameterIter (Parameter.begin());
    while (m_Data.data.length < CstBufferSize &&
           ParameterIter != Parameter.end())
    {
        *++BufferIter = *ParameterIter++;
        ++m_Data.data.length;
    }
    *++BufferIter = '\0';
    ++m_Data.data.length;

#ifndef NDEBUG
    std::cerr << "CRTAdvancedDriver::SetIoctl() m_Data.data.pointer : "
              << reinterpret_cast<const char*> (m_Data.data.pointer)
              << "\n\tm_Data.data.length : " << m_Data.data.length << std::endl;
#endif // NDEBUG
    Ioctl (m_SetIoctl, ErrorMsg + Parameter);

} // SetIoctl()


void nsWireless::CRTAdvancedDriver::SetIoctl (const char* Command, int Value,
              const std::string& ErrorMsg) const throw (nsErrors::CSystemExc)
{
    std::ostringstream Os;
    Os << Value;
    SetIoctl (Command, Os.str().c_str(), ErrorMsg);

} // SetIoctl()


nsWireless::CRT2570Driver::CRT2570Driver (const std::string& DeviceName)
                                                throw (nsErrors::CException)
    : CRTDriver (DeviceName), m_AuthIoctl (InvalidIoctl),
      m_EncIoctl (InvalidIoctl), m_KeyIoctl (InvalidIoctl),
      m_AdHocModeIoctl (InvalidIoctl), m_PrismHeaderIoctl (InvalidIoctl),
      m_RTPrivIoctl (InvalidIoctl), m_Flags (0), m_PrismMode (Default),
      m_AdHocMode (Std11)
{
    const unsigned TabArgsSize (16);
    ::iw_priv_args TabArgs [TabArgsSize];

    for (unsigned i (GetPrivateIoctls (TabArgs, TabArgsSize)) ; i ; )
        if (!strcmp (TabArgs [--i].name, "auth"))
            m_AuthIoctl = TabArgs [i].cmd;
        else if (!strcmp (TabArgs [i].name, "enc"))
            m_EncIoctl = TabArgs [i].cmd;
        else if (!strcmp (TabArgs [i].name, "wpapsk"))
            m_KeyIoctl = TabArgs [i].cmd;
        else if (!strcmp (TabArgs [i].name, "adhocmode"))
            m_AdHocModeIoctl = TabArgs [i].cmd;
        else if (!strcmp (TabArgs [i].name, "rfmontx"))
            m_RfmontxIoctl = TabArgs [i].cmd;
        else if (!strcmp (TabArgs [i].name, "forceprismheader"))
            m_PrismHeaderIoctl = TabArgs [i].cmd;
        else if (!strcmp (TabArgs [i].name, "get_rtpriv"))
            m_RTPrivIoctl = TabArgs [i].cmd;
        else if (!strcmp (TabArgs [i].name, "get_rfmontx") &&
                    GetIntIoctl (TabArgs [i].cmd, "Can't get rfmontx status."))
            m_Flags |= RfMontx;
        else if (!strcmp (TabArgs [i].name, "get_prismheader"))
            m_PrismMode = PrismMode_e (GetIntIoctl (TabArgs [i].cmd,
                                       "Can't get prism headers mode."));
        else if (!strcmp (TabArgs [i].name, "get_adhocmode"))
            m_AdHocMode = AdHocMode_e (GetIntIoctl (TabArgs [i].cmd,
                                       "Can't get adhoc mode."));

    if (m_EncIoctl == InvalidIoctl)
        throw nsErrors::CException ("Can't find \"enc\" private ioctl.",
                                    nsErrors::RT2570EncIoctlNotFound);
    if (m_AuthIoctl == InvalidIoctl)
        throw nsErrors::CException ("Can't find \"auth\" private ioctl.",
                                    nsErrors::RT2570AuthIoctlNotFound);
    if (m_KeyIoctl == InvalidIoctl)
        throw nsErrors::CException ("Can't find \"key\" private ioctl.",
                                    nsErrors::RT2570KeyIoctlNotFound);

} // CRT2570Driver()


void nsWireless::CRT2570Driver::GetScanResult (std::vector<CCell>& CellVec)
                            const throw (nsErrors::CException, std::bad_alloc)
{
#ifndef NDEBUG
    std::cerr << "CRT2570Driver::GetScanResult() Standard scan"
              << std::endl;
#endif // NDEBUG
    CWE17Driver::GetScanResult (CellVec);
#ifndef NDEBUG
    std::cerr << "CRT2570Driver::GetScanResult() Standard scan success, "
              << CellVec.size() << " cells." << std::endl;
#endif // NDEBUG
    if (m_RTPrivIoctl != InvalidIoctl) // Backward compatibilty check.
        try
        {
            std::vector<CCell> RalinkCellVec;
#ifndef NDEBUG
        std::cerr << "CRT2570Driver::GetScanResult() Ralink scan"
                  << std::endl;
#endif // NDEBUG
                                            // OID_802_11_BSSID_LIST
            CRTDriver::GetScanResult (m_RTPrivIoctl, 0x0609U, RalinkCellVec);
#ifndef NDEBUG
        std::cerr << "CRT2570Driver::GetScanResult() Ralink scan success, "
                  << RalinkCellVec.size() << " cells." << std::endl;
#endif // NDEBUG
            for (unsigned i (0) ; i < RalinkCellVec.size() &&
                                                        i < CellVec.size(); ++i)
                if (RalinkCellVec [i].GetEncryptionD().GetAuth() == WPAPSK ||
                    RalinkCellVec [i].GetEncryptionD().GetAuth() == WPANONE)
                        // Both results seem to be always in the same order.
                    CellVec [i].GetEncryptionD() =
                                            RalinkCellVec [i].GetEncryptionD();
        }
        catch (const nsErrors::CException& Exc)
        {
#ifndef NDEBUG
            std::cerr << "CRT2570Driver::GetScanResult() Ralink scan failed : "
                      << Exc.GetMsg() << ' ' << Exc.GetCode() << std::endl;
#endif // NDEBUG
        } // We have some results anyway.

} // GetScanResult()


void nsWireless::CRT2570Driver::SetEncryption (const CEncryptionD& Descriptor)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    if (Descriptor.GetAuth() == Open || Descriptor.GetAuth() == Shared)
    {
#ifndef NDEBUG
        std::cerr << "CRT2570Driver::SetEncryption(), using standard ioctl."
                  << std::endl;
#endif // NDEBUG
        CWE17Driver::SetEncryption (Descriptor);
    }
    else
    {
        /* TODO Check if there's an order that works better, this is the
                same order as rt2500 currently. */
#ifndef NDEBUG
        std::cerr << "CRT2570Driver::SetEncryption(), using private ioctls.\n\t"
                     "AuthName : " << GetAuthName (Descriptor.GetAuth())
                  << "\n\tEncryptName : "
                  << GetEncryptName (Descriptor.GetEncrypt()) << "\n\tKey : "
                  << std::string (Descriptor.GetKey().Get(), 0,
                                     Descriptor.GetKey().Size()) << std::endl;
#endif // NDEBUG
        int Auth (3); // WPAPSK
        if (Descriptor.GetAuth() == WPA2PSK)
            Auth = 4;
        else if (Descriptor.GetAuth() == WPANONE)
            Auth = 5;
        PrivateIoctl (m_AuthIoctl, Auth, "Can't set wpa authentication.");
        PrivateIoctl (m_EncIoctl, Descriptor.GetEncrypt() == TKIP ? 3 : 4,
                      "Can't set wpa encryption.");
        PrivateIoctl (m_KeyIoctl, std::string (Descriptor.GetKey().Get(), 0,
                                               Descriptor.GetKey().Size()),
                      "Can't set wpa key.");
    }

} // SetEncryption()


void nsWireless::CRT2570Driver::PrivateIoctl (int IoctlCode,
                        const std::string& Value, const std::string& ErrorMsg)
                                                throw (nsErrors::CSystemExc)
{
    m_Data.data.pointer = reinterpret_cast< ::caddr_t>
                                        (const_cast<char*> (Value.c_str()));
    m_Data.data.length = Value.size();
    m_Data.data.flags = 0;
    Ioctl (IoctlCode, ErrorMsg);

} // PrivateIoctl()


void nsWireless::CRT2570Driver::GetSupportedRates (std::vector<int>& RatesVec)
                                                                const throw()
{
    CWE17Driver::GetSupportedRates (RatesVec);
    if (RatesVec.empty())
    {   // The driver doesn't report values but supports rate changing :
        RatesVec.push_back (54000);
        RatesVec.push_back (48000);
        RatesVec.push_back (36000);
        RatesVec.push_back (24000);
        RatesVec.push_back (18000);
        RatesVec.push_back (11000);
        RatesVec.push_back (2000);
        RatesVec.push_back (1000);
    }

} // GetSupportedRates()


int nsWireless::CRT2570Driver::GetIntIoctl
            (int Code, const std::string& ErrorMsg) throw (nsErrors::CSystemExc)
{
    int* const pResult (reinterpret_cast<int*> (m_Data.name));
    *pResult = 0;
    Ioctl (Code, ErrorMsg);
    return *pResult;

} // GetIntIoctl()


nsWireless::CRT61Driver::CRT61Driver (const std::string& DeviceName)
                                  throw (nsErrors::CException, std::bad_alloc)
        /* RT_PRIV_IOCTL, RT_OID_802_11_STA_CONFIG, RT_OID_802_11_PREAMBLE and
           RT_OID_PHY_MODE */
    : CRTAdvancedDriver (DeviceName, SIOCIWFIRSTPRIV + 1, 0x050DU, 0x050FU,
                         0x050CU, new IWPrivHandlerRT61 (m_GetSiteSurveyIoctl)),
      m_LastScanTime (0)
{

} // CRT61Driver()


void nsWireless::CRT61Driver::UpdateScanTime () throw()
{
    const time_t NewTime (time (0));
    if (NewTime != -1) // Ignore errors.
        m_LastScanTime = NewTime;

} // UpdateScanTime()


void nsWireless::CRT61Driver::GetScanResult (std::vector<CCell>& CellVec)
                            const throw (nsErrors::CException, std::bad_alloc)
{                           // OID_802_11_BSSID_LIST
    //CRTAdvancedDriver::GetScanResult (0x0609U, CellVec);
    const time_t CurrentTime (time (0));
    if (CurrentTime != -1 && CurrentTime - 4 < m_LastScanTime)
    {
#ifndef NDEBUG
        std::cerr << "CRT61Driver::GetScanResult() Simulate EAGAIN.\n";
#endif // NDEBUG
        throw nsErrors::CException ("Simulate EAGAIN.", EAGAIN);
    }
    m_Data.data.length = 8192;
    char Buffer [m_Data.data.length];
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (Buffer);
    m_Data.data.flags = 0;
    Ioctl (m_GetSiteSurveyIoctl, "Can't get scan result through private"
                                 " ioctl.");
    const char* const End (Buffer + m_Data.data.length - 1);
#ifndef NDEBUG
    std::cerr << "CRT61Driver::GetScanResult() raw data : "
              << std::string (Buffer, 0, m_Data.data.length) << "\nSize : "
              << m_Data.data.length << std::endl;
#endif // NDEBUG
    const size_t LineSize ((m_Data.data.length - 1) % 109 ? 117 : 109);
    for (const char* Iter (Buffer + LineSize) ; Iter < End ; Iter += LineSize)
        CellVec.push_back (ToCellRt61 (Iter));
#ifndef NDEBUG
    std::cerr << "Raw data successfully processed.\n";
#endif // NDEBUG

} // GetScanResult()


void nsWireless::CRT61Driver::SetEncryption (const CEncryptionD& Descriptor)
                                    throw (nsErrors::CSystemExc, std::bad_alloc)
{
    if (Descriptor.GetAuth() == Open || Descriptor.GetAuth() == Shared)
    {
#ifndef NDEBUG
        std::cerr << "CRT61Driver::SetEncryption(), using standard ioctl."
                  << std::endl;
#endif // NDEBUG
        CWE17Driver::SetEncryption (Descriptor);
    }
    else
    {
#ifndef NDEBUG
        std::cerr << "CRT61Driver::SetEncryption(), using private ioctls.\n\t"
                     "AuthName : " << GetAuthName (Descriptor.GetAuth())
                  << "\n\tEncryptName : "
                  << GetEncryptName (Descriptor.GetEncrypt()) << "\n\tKey : "
                  << std::string (Descriptor.GetKey().Get(), 0,
                                     Descriptor.GetKey().Size()) << std::endl;
#endif // NDEBUG
        SetIoctl ("AuthMode", GetAuthName (Descriptor.GetAuth()),
                  "Can't set authentication mode : ");
        SetIoctl ("EncrypType", GetEncryptName (Descriptor.GetEncrypt()),
                  "Can't set encryption type : ");
        SetIoctl ("WPAPSK", std::string (Descriptor.GetKey().Get(), 0,
                                         Descriptor.GetKey().Size()),
                  "Can't set key : ");
    }

} // SetEncryption()


void nsWireless::CRT61Driver::IWPrivHandlerRT61::operator ()
                        (const ::iw_priv_args& Arg) throw (nsErrors::CException)
{
    if (!strcmp (Arg.name, "get_site_survey"))
        m_GetSiteSurveyIoctl = Arg.cmd;

} // operator()()


    // FIXME Some code duplicated from RT61.
void nsWireless::CRT73Driver::GetScanResult (std::vector<CCell>& CellVec) const
                                    throw (nsErrors::CException, std::bad_alloc)
{                           // OID_802_11_BSSID_LIST
    //CRTAdvancedDriver::GetScanResult (0x0609U, CellVec);
    const time_t CurrentTime (time (0));
    if (CurrentTime != -1 && CurrentTime - 4 < GetLastScanTime())
    {
#ifndef NDEBUG
        std::cerr << "CRT73Driver::GetScanResult() Simulate EAGAIN.\n";
#endif // NDEBUG
        throw nsErrors::CException ("Simulate EAGAIN.", EAGAIN);
    }
    m_Data.data.length = 8192;
    char Buffer [m_Data.data.length];
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (Buffer);
    m_Data.data.flags = 0;
    Ioctl (GetSiteSurveyIoctl(), "Can't get scan result through private"
                                 " ioctl.");
    const char* const End (Buffer + m_Data.data.length - 1);
#ifndef NDEBUG
    std::cerr << "CRT73Driver::GetScanResult() raw data : "
              << std::string (Buffer, 0, m_Data.data.length) << "\nSize : "
              << m_Data.data.length << std::endl;
#endif // NDEBUG
    const size_t LineSize (109);
    for (const char* Iter (Buffer + LineSize) ; Iter < End ; Iter += LineSize)
        CellVec.push_back (ToCellRt73 (Iter));
#ifndef NDEBUG
    std::cerr << "Raw data successfully processed.\n";
#endif // NDEBUG

} // GetScanResult()


void nsWireless::CRT73Driver::SetRfmontx (bool B) throw (nsErrors::CSystemExc)
{
    char Buffer (B ? '1' : '0');
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (&Buffer);
    m_Data.data.length = 2;
    m_Data.data.flags = 0;
    Ioctl (GetRfmontxIoctl(), "Can't set injection mode.");
    SetFlag (Rfmontx, B);

} // SetRfmontx()
