/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file SystemTools.cxx
    \author Romain BONDUE
    \date 20/12/2005 */
#include <csignal>
#include <cstring> // std::memset()
#include <exception>
#include <cerrno>
#ifndef NDEBUG
#include <iostream>
#include <cstdio>
#endif // NDEBUG

extern "C"{
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/wait.h> // ::wait()
#include <unistd.h>
}

#include "SystemTools.h"



namespace
{
    inline void Sigaction (int Signum, const struct sigaction& NewSettings,
                           struct sigaction* OldSettings = 0)
                                                throw (nsErrors::CSystemExc)
    {
        if (::sigaction (Signum, &NewSettings, OldSettings))
            throw nsErrors::CSystemExc ("::sigaction() failed.");

    } // Sigaction()


    void ZombieCleaner (int) throw() {::wait (0);}


        // Set up things when RutilT starts.
    struct SInitializer
    {
      public :
        const int Sd;


        SInitializer () throw (nsErrors::CSystemExc)
            : Sd (::socket (PF_INET, SOCK_DGRAM, 0))
        {
            if (Sd == -1)
                throw nsErrors::CSystemExc ("Can't create socket.");
            IgnoreSignals();

        } // SInitializer()


        ~SInitializer () throw() {::close (Sd);}


        void IgnoreSignals () throw (nsErrors::CSystemExc)
        {
            struct sigaction Settings;
            std::memset (&Settings, 0, sizeof (struct sigaction));
            Settings.sa_handler = SIG_IGN;
            Sigaction (SIGPIPE, Settings);

            Settings.sa_flags = SA_NOCLDSTOP;
            Settings.sa_handler = ZombieCleaner;
            Sigaction (SIGCHLD, Settings);

        } // IgnoreSignals()


        void RestoreSignals () throw (nsErrors::CSystemExc)
        {
            struct sigaction Settings;
            std::memset (&Settings, 0, sizeof (struct sigaction));
            Settings.sa_handler = SIG_DFL;
            Sigaction (SIGPIPE, Settings);
            Sigaction (SIGCHLD, Settings);
                /* !! UGLY !! They are modified in GtkGUI.cxx
                   They have not been modified here because this code is linked
                   with the helper, and it's not linked with gtk. */
            Sigaction (SIGINT, Settings);
            Sigaction (SIGTERM, Settings);

        } // RestoreSignals()

    } Data;


    inline void Close (int FD) throw (nsErrors::CSystemExc)
    {
        if (::close (FD))
            throw nsErrors::CSystemExc ("Can't close file descriptor.");

    } // Close()


    void InitSockaddrUn (const std::string& addr_name, ::sockaddr_un& addr)
                                                                        throw()
    {
        std::memset(&addr, 0, sizeof (addr));
        const size_t UNIX_PATH_MAX (108U);
        addr.sun_family = AF_UNIX;
        addr_name.copy (addr.sun_path + 1, UNIX_PATH_MAX - 2);
        addr.sun_path [addr_name.size() > UNIX_PATH_MAX - 1 ? UNIX_PATH_MAX
                                                    : addr_name.size()] = '\0';

    } // InitSockaddrUn()


    ::ucred* GetCredential (int Sd, unsigned char* Buffer, unsigned BufferSize)
                                                    throw (nsErrors::CSystemExc)
    {
        std::memset (Buffer, 0, BufferSize);
        ::msghdr Msg;
        std::memset (&Msg, 0, sizeof (Msg));
        Msg.msg_control = Buffer;
        Msg.msg_controllen = BufferSize;
        ::cmsghdr* const pCMsgHeader (CMSG_FIRSTHDR (&Msg));
        ::ucred* const pUCred (reinterpret_cast< ::ucred*> (CMSG_DATA
                                                                (pCMsgHeader)));
        pCMsgHeader->cmsg_len = CMSG_LEN (sizeof (::ucred));
        pCMsgHeader->cmsg_level = SOCK_STREAM;
        pCMsgHeader->cmsg_type = SCM_CREDENTIALS;
        Msg.msg_controllen = pCMsgHeader->cmsg_len;
        ::iovec DataVec;
        std::memset (&DataVec, 0, sizeof (DataVec));
        char MsgData;
        DataVec.iov_base = &MsgData;
        DataVec.iov_len = sizeof (MsgData);
        Msg.msg_iov = &DataVec;
        Msg.msg_iovlen = 1;
        int BoolValue (1);
        if (::setsockopt (Sd, SOL_SOCKET, SO_PASSCRED, &BoolValue,
                                                            sizeof (BoolValue)))
            throw nsErrors::CSystemExc ("Can't enable credential reception on"
                                        " socket.");
        const int NbRead (::recvmsg (Sd, &Msg, MSG_WAITALL));
        BoolValue = 0;
        if (::setsockopt (Sd, SOL_SOCKET, SO_PASSCRED, &BoolValue,
                                                            sizeof (BoolValue)))
            throw nsErrors::CSystemExc ("Can't enable credential reception on"
                                        " socket.");
        if (NbRead == -1)
            throw nsErrors::CSystemExc ("Can't get credential information.");
        if (size_t (NbRead) < sizeof (MsgData))
            throw nsErrors::CSystemExc ("Can't get credential on socket.");
        return pUCred;

    } // GetCredential()

} // anonymous namespace



void nsSystem::Ioctl (int Request, void* pData, const std::string& ErrorMsg)
                                                throw (nsErrors::CSystemExc)
{
    if (::ioctl (Data.Sd, Request, pData) < 0)
    {
#ifndef NDEBUG
        perror ("::ioctl() failed");
        std::cerr << "\tRequest : " << std::hex << Request << std::dec << ' '
                  << ErrorMsg << std::endl;
#endif // NDEBUG
        throw nsErrors::CSystemExc (ErrorMsg);
    }

} // Ioctl()


nsSystem::CLocalSocket::CLocalSocket () throw (nsErrors::CSystemExc)
    : m_Sd (::socket (PF_UNIX, SOCK_STREAM, 0))
{
    if (m_Sd == -1)
        throw nsErrors::CSystemExc ("Can't create socket.");

} // CLocalSocket()


void nsSystem::CLocalSocket::Close () throw (nsErrors::CSystemExc)
{
    if (m_Sd != InvalidSocket);
    {
        ::Close (m_Sd);
        m_Sd = InvalidSocket;
    }

} // Close()


void nsSystem::CLocalSocket::Write (const char* Data, size_t Size)
                                                    throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "Writing " << Size << " byte(s) in socket." << std::endl;
#endif // NDEBUG
    if (::write (m_Sd, Data, Size) != ssize_t (Size))
        throw nsErrors::CSystemExc ("Can't write in socket or short write.");

} // Write()


size_t nsSystem::CLocalSocket::Read (char* Buffer, size_t Size)
                                                    throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "Trying to read up to " << Size << " bytes in socket."
              << std::endl;
#endif // NDEBUG
    const ssize_t ByteRead (::read (m_Sd, Buffer, Size));
    if (ByteRead < 0)
    {
        if (errno == ECONNRESET)
            return 0;
        throw nsErrors::CSystemExc ("Can't read in socket.");
    }
#ifndef NDEBUG
    std::cerr << "\tRead " << ByteRead << " byte(s)" << std::endl;
#endif // NDEBUG
    return ByteRead;

} // Read()


void nsSystem::CLocalSocket::Bind (const std::string& addr_name)
                                                    throw (nsErrors::CSystemExc)
{
    ::sockaddr_un addr;
    InitSockaddrUn (addr_name, addr);
    if (::bind (m_Sd, reinterpret_cast< ::sockaddr*> (&addr), sizeof (addr)))
        throw nsErrors::CSystemExc ("Can't bind socket.");

} // Bind()


void nsSystem::CLocalSocket::Connect (const std::string& addr_name)
                                                    throw (nsErrors::CSystemExc)
{
    ::sockaddr_un addr;
    InitSockaddrUn (addr_name, addr);
    if (::connect (m_Sd, reinterpret_cast< ::sockaddr*> (&addr), sizeof (addr)))
        throw nsErrors::CSystemExc ("Can't connect socket.");

} // Connect()


void nsSystem::CLocalSocket::Listen (unsigned Backlog)
                                                    throw (nsErrors::CSystemExc)
{
    if (::listen (m_Sd, Backlog))
        throw nsErrors::CSystemExc ("Can't listen on socket.");

} // Listen()


nsSystem::CLocalSocket* nsSystem::CLocalSocket::Accept ()
                                    throw (nsErrors::CSystemExc, std::bad_alloc)
{
    const int NewSd (::accept (m_Sd, 0, 0));
    if (NewSd == -1)
        throw nsErrors::CSystemExc ("Can't accept connection on socket.");
    return new CLocalSocket (NewSd);

} // Accept()


void nsSystem::CLocalSocket::SendCredential () throw (nsErrors::CSystemExc)
{
        // Linux specific.
    const unsigned BufferSize (256);
    unsigned char Buffer [BufferSize];
    std::memset (Buffer, 0, BufferSize);
    ::msghdr Msg;
    std::memset (&Msg, 0, sizeof (Msg));
    Msg.msg_control = Buffer;
    Msg.msg_controllen = BufferSize;
    ::cmsghdr* const pCMsgHeader (CMSG_FIRSTHDR (&Msg));
    ::ucred* const pUCred (reinterpret_cast< ::ucred*> (CMSG_DATA
                                                                (pCMsgHeader)));
    pCMsgHeader->cmsg_len = CMSG_LEN (sizeof (::ucred));
    pCMsgHeader->cmsg_level = SOCK_STREAM;
    pCMsgHeader->cmsg_type = SCM_CREDENTIALS;
    pUCred->pid = ::getpid();
    const ::uid_t Euid (::geteuid());
    pUCred->uid = Euid ? ::getuid() : Euid;
    pUCred->gid = ::getgid();
    Msg.msg_controllen = pCMsgHeader->cmsg_len;
    ::iovec DataVec;
    std::memset (&DataVec, 0, sizeof (DataVec));
    char MsgData (65);
    DataVec.iov_base = &MsgData;
    DataVec.iov_len = sizeof (MsgData);
    Msg.msg_iov = &DataVec;
    Msg.msg_iovlen = 1;
    const ssize_t NbSent (::sendmsg (m_Sd, &Msg, 0));
    if (NbSent == -1)
        throw nsErrors::CSystemExc ("Can't send credential on socket.");
    if (size_t (NbSent) != DataVec.iov_len)
        throw nsErrors::CSystemExc ("Can't send credential on socket (Short"
                                    " write).");

} // SendCredential()


bool nsSystem::CLocalSocket::CheckCredentialUID (::uid_t uid)
                                                    throw (nsErrors::CSystemExc)
{
        // Linux specific.
    const unsigned BufferSize (256);
    unsigned char Buffer [BufferSize];
    return GetCredential (m_Sd, Buffer, BufferSize)->uid == uid;

} // CheckCredentialUID()


bool nsSystem::CLocalSocket::CheckCredentialPID (::pid_t pid)
                                                    throw (nsErrors::CSystemExc)
{
        // Linux specific.
    const unsigned BufferSize (256);
    unsigned char Buffer [BufferSize];
    return GetCredential (m_Sd, Buffer, BufferSize)->pid == pid;

} // CheckCredentialPID()


bool nsSystem::Fork () throw (nsErrors::CSystemExc)
{
    const ::pid_t Pid (::fork());
    if (Pid == -1)
        throw nsErrors::CSystemExc ("Can't create sub-process.");
    return Pid;

} // Fork()


void nsSystem::Exec (const char* Argv []) throw (nsErrors::CSystemExc)
{
    Data.RestoreSignals();
    ::execv (Argv [0], const_cast<char**> (Argv));
    throw nsErrors::CSystemExc (std::string ("Can't execute file : ")
                                                            + Argv [0] + '.');

} // Exec()
