/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file WE17Driver.h
    \author Romain BONDUE
    \date 05/07/2005 */
#ifndef __WE17DRIVER_H__
#define __WE17DRIVER_H__

#include <vector>

extern "C"{
    // Compilation problems if socket.h is not included (or after wireless.h).
#include <sys/socket.h>
    // Those 3 headers were included by wireless.h until WE21.
#include <linux/types.h>		/* for "caddr_t" et al		*/
#include <linux/socket.h>		/* for "struct sockaddr" et al	*/
#include <linux/if.h>			/* for IFNAMSIZ and co... */
#include <linux/wireless.h>
}
#if WIRELESS_EXT < 16
#error Invalid wireless extension version : require at least version 16.
#endif

#include "IDriver.h"



namespace nsWireless
{
    class CWE17Driver : public IDriver
    {
      public :
        CWE17Driver (const std::string&) throw (nsErrors::CSystemExc);
        ~CWE17Driver () throw();

        std::string GetDeviceName () const throw();

        std::string GetSSID () const throw (nsErrors::CSystemExc);
        void SetSSID (const std::string&) throw (nsErrors::CSystemExc);
 
        double GetFrequency () const throw (nsErrors::CException);
        void SetFrequency (double) throw (nsErrors::CSystemExc);

        unsigned GetChannel () const throw (nsErrors::CException);
        void SetChannel (unsigned) throw (nsErrors::CSystemExc);

        CEncryptionD GetEncryption () throw (nsErrors::CSystemExc,
                                             std::bad_alloc);
        void SetEncryption (const CEncryptionD&) throw (nsErrors::CSystemExc,
                                                        std::bad_alloc);

        Mode_e GetMode () const throw (nsErrors::CSystemExc);
        void SetMode (Mode_e) throw (nsErrors::CSystemExc);

        CQuality GetQuality () const throw (nsErrors::CSystemExc);

        int GetTxRate () const throw (nsErrors::CSystemExc);
        void SetTxRate (int) throw (nsErrors::CSystemExc);

        CMacAddress GetAPMacAddr () const throw (nsErrors::CSystemExc);
        void SetAPMacAddr (const CMacAddress&) throw (std::bad_alloc,
                                                      nsErrors::CSystemExc);

            /* Don't use Scan() too often :
             *      - The driver may not be ready (EAGAIN).
             *      - "The scanning operation disable normal network traffic"
             *        (Jean Tourrilhes). It has also some others side effects,
             *        see the comments in iwlib.c . */
        void Scan () throw (nsErrors::CSystemExc);
        void GetScanResult (std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);

        unsigned GetMaxSSIDSize () const throw();
        unsigned GetMaxKeySize () const throw();

        const std::vector<CFreq>& GetSupportedFreq () const throw();
        void GetSupportedRates (std::vector<int>&) const
                                                throw (nsErrors::CSystemExc);

        const std::string& GetProtoName () const throw();
        void Commit () throw (nsErrors::CSystemExc);


      protected :
        void Ioctl (int, const std::string&) const
                                                throw (nsErrors::CSystemExc);
        void AddSupportedFreq (unsigned, double) throw();
        unsigned GetPrivateIoctls (::iw_priv_args*, unsigned)
                                                throw (nsErrors::CSystemExc);
        CFreq GetMatchingFreq (unsigned, double = 0) const throw();

        mutable ::iwreq_data& m_Data;


      private :
        /** \return A negative value for a channel, a positive for a
         *          frequency. */
        double GetDriverFreqValue () const throw (nsErrors::CSystemExc);
        unsigned FillCellVec (const ::iw_event*, std::vector<CCell>&) const
                                                  throw (nsErrors::CException);

        std::string m_ProtoName;
        mutable ::iwreq m_Req;
        std::vector<CFreq> m_SupportedFreqVec;

    }; // CWE17Driver

} // namespace nsWireless

#include "private/WE17Driver.hxx"

#endif // __WE17DRIVER_H__
