;;;; -*- mode: lisp -*-
;;;;
;;;; $Id: base64.lisp,v 1.3 2005/02/07 17:45:41 scaekenberghe Exp $
;;;;
;;;; S-SYSDEPS is an abtraction layer over platform dependent functionality
;;;;
;;;; Copyright (C) 2004-2005 Sven Van Caekenberghe, Beta Nine BVBA.
;;;;
;;;; You are granted the rights to distribute and use this software
;;;; as governed by the terms of the Lisp Lesser General Public License
;;;; (http://opensource.franz.com/preamble.html), also known as the LLGPL.

(in-package :s-sysdeps)

;; managing processes

(defun current-process ()
  "Return the object representing the current process"
  #+lispworks mp:*current-process* 
  #+openmcl ccl:*current-process*)

(defun kill-process (process)
  "Kill the process represented by the object process"
  #+lispworks (mp:process-kill process)
  #+openmcl (ccl:process-kill process))

(defun run-process (name function &rest arguments)
  "Create and run a new process with name, executing function on arguments"
  #+lispworks (apply #'mp:process-run-function name '(:priority 3) function arguments)
  #+openmcl (apply #'ccl:process-run-function name function arguments))

(defun all-processes ()
  "Return a list of all processes currently running"
  #+lispworks (mp:list-all-processes)
  #+openmcl (ccl:all-processes))

;; implementing a standard TCP/IP server

(defun start-standard-server (&key port name connection-handler)
  "Start a server process with name, listening on port, delegating to connection-handler with stream as argument"
  #+lispworks (comm:start-up-server
               :function #'(lambda (socket-handle)
                             (let ((client-stream (make-instance 'comm:socket-stream
                                                                 ;; maybe specify a read timeout...
                                                                 :socket socket-handle
                                                                 :direction :io
                                                                 :element-type 'base-char)))
                               (funcall connection-handler client-stream)))
               :service port
               :announce t
               :wait t
               :process-name name)
  #+openmcl (ccl:process-run-function
             name
             #'(lambda ()
                 (let ((server-socket (ccl:make-socket :connect :passive
                                                       :local-port port
                                                       :reuse-address t)))
                   (unwind-protect
                       (loop 
                        (let ((client-stream (ccl:accept-connection server-socket)))
                          (funcall connection-handler client-stream))) 
                     (close server-socket))))))

;; opening a client TCP/IP socket stream

(defun open-socket-stream (host port)
  "Create and open a bidirectional client TCP/IP socket stream to host:port"
  #+openmcl (make-socket :remote-host host :remote-port port)
  #+lispworks (comm:open-tcp-stream host port)
  #+clisp (socket:socket-connect port host)
  #+cmu (sys:make-fd-stream (ext:connect-to-inet-socket host port) 
                            :input t :output t :buffering :none)
  #+sbcl (let ((socket (make-instance 'sb-bsd-sockets:inet-socket 
                                      :type :stream :protocol :tcp)))
           (sb-bsd-sockets:socket-connect socket 
                                          (car 
                                           (sb-bsd-sockets:host-ent-addresses 
                                            (sb-bsd-sockets:get-host-by-name host))) 
                                          port)
           (sb-bsd-sockets:socket-make-stream socket 
                                              :element-type 'character 
                                              :input t :output t :buffering :none)))

;; accessing socket stream properties

(defun get-socket-stream-property (socket-stream property)
  "Get the value of a socket stream property, one of :remote-host :remote-port :local-host :local-port"
  #+lispworks (ecase property
                ((:remote-host :remote-port) (multiple-value-bind (address port)
                                                 (comm:socket-stream-peer-address socket-stream)
                                               (if (eql property :remote-host)
                                                   (comm:ip-address-string address)
                                                 port)))
                ((:local-host :local-port) (multiple-value-bind (address port)
                                               (comm:socket-stream-peer-address socket-stream)
                                             (if (eql property :local-host)
                                                 (comm:ip-address-string address)
                                               port)))))

;; working with process locks

(defun make-process-lock (name)
  "Create a named process lock object"
  #+lispworks (mp:make-lock :name name)
  #+openmcl (ccl:make-lock name))

(defmacro with-process-lock ((lock) &body body)
  "Execute body wih the process lock grabbed, wait otherwise"
  ;; maybe it is safer to always use a timeout: 
  ;; `(mp:with-lock (,lock (format nil "Waiting for ~s" (lock-name ,lock)) 5) ,@body)
  ;; if the lock cannot be claimed in 5s, nil is returned: test it and throw a condition ?
  #+lispworks `(mp:with-lock (,lock) ,@body)
  #+openmcl `(ccl:with-lock-grabbed (,lock) ,@body))

;;;; eof
