/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 * 
 * $Id: rigdef-panel.cpp,v 1.11 2004/10/26 19:30:38 jeremy_laine Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "rigdef-panel.h"
#include "sailreader-xml.h"
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qgroupbox.h>
#include <qtabwidget.h>
#include <qpushbutton.h>



/** The constructor for a CVector3dWidget.
 *
 * @param parent the parent widget
 */
CVector3dWidget::CVector3dWidget(QWidget *parent)
        : QWidget(parent)
{
    for (int i = 0; i < 3; i++)
    {
        txtBox.push_back(new QLineEdit(this, QString::number(i)));
    }

    QLabel *lblStaticX = new QLabel(this, "lblStaticX");
    lblStaticX->setText("x : ");
    QLabel *lblStaticY = new QLabel(this, "lblStaticY");
    lblStaticY->setText("y : ");
    QLabel *lblStaticZ = new QLabel(this, "lblStaticZ");
    lblStaticZ->setText("z : ");

    QGridLayout* layout = new QGridLayout( this );
    layout->addWidget(lblStaticX, 0, 0);
    layout->addWidget(txtBox[0], 0, 1);
    layout->addWidget(lblStaticY, 0, 2);
    layout->addWidget(txtBox[1], 0, 3);
    layout->addWidget(lblStaticZ, 0, 4);
    layout->addWidget(txtBox[2], 0, 5);

}


/** Returns the vector corresponding to the user's input.
 */
CVector3d CVector3dWidget::getVector()
{
    CVector3d v;
    for (int i = 0; i < 3; i++)
        v[i] = txtBox[i]->text().toDouble();
    return v;
}


/** Set the input boxes from a given vector.
 *
 * @param v the new vector to display
 */
void CVector3dWidget::setVector(const CVector3d &v)
{
    for (unsigned int i = 0; i < 3; i++)
        txtBox[i]->setText(QString::number(v.getcoord(i)));
}


/** The constructor.
 *
 * @param parent the parent widget
 */
CRigSailWidget::CRigSailWidget(QWidget *parent)
        : QWidget(parent)
{
    grpInfo = new QGroupBox( 0, Qt::Vertical, this, "grpInfo" );
    QGridLayout* iLayout = new QGridLayout( grpInfo->layout() );

    lblFileStatic = new QLabel( grpInfo, "lblFilenameStatic" );
    iLayout->addWidget( lblFileStatic, 0, 0 );
    lblFile = new QLabel( grpInfo, "lblFilename" );
    iLayout->addWidget( lblFile, 0, 1 );
    btnReload = new QPushButton(grpInfo, "btnReload");
    iLayout->addWidget( btnReload, 0, 2);
    lblNameStatic = new QLabel( grpInfo, "lblNameStatic" );
    iLayout->addWidget( lblNameStatic, 1, 0 );
    txtName = new QLineEdit( grpInfo, "txtName" );
    iLayout->addWidget( txtName, 1, 1 );

    btnUpdate = new QPushButton(this, "btnUpdate");
    btnRemove = new QPushButton(this, "btnRemove");

    QGridLayout* layout = new QGridLayout(this);
    layout->addMultiCellWidget(grpInfo, 0, 0, 0, 2);
    wdgOrigin = new CVector3dWidget(this);
    layout->addMultiCellWidget(wdgOrigin, 1, 1, 0, 2);
    layout->addItem( new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding ), 2, 0 );
    layout->addWidget(btnRemove, 3, 0);
    layout->addItem( new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum ), 3, 1 );
    layout->addWidget(btnUpdate, 3, 2);

    languageChange();

    /* connect signals */
    connect( btnUpdate, SIGNAL( clicked() ), this, SLOT(slotUpdate()) );
    connect( btnReload, SIGNAL( clicked() ), this, SLOT(slotReload()) );
    connect( btnRemove, SIGNAL( clicked() ), this, SLOT(slotRemove()) );
}


/**
 * Sets the strings of the subwidgets using the current language.
 */
void CRigSailWidget::languageChange()
{
    grpInfo->setTitle( tr( "Sail information" ) );
    lblFileStatic->setText( tr("file") );
    lblNameStatic->setText( tr("name") );
    btnReload->setText( tr("Reload") );
    btnRemove->setText( tr("Remove") );
    btnUpdate->setText( tr("Update") );
}


/**
 * The rigsail changed, update widgets.
 *
 * @param newsail The new value of the rigsail
 */
void CRigSailWidget::setRigSail(const CRigSail& newsail)
{
    rigsail = newsail;
    lblFile->setText( rigsail.filename.data() );
    txtName->setText( rigsail.name.data() );
    wdgOrigin->setVector( rigsail.origin );
}


/**
 * The "reload" button was pressed, fire signalUpdate.
 */
void CRigSailWidget::slotReload()
{
    try
    {
        switch (rigsail.type)
        {
        case SAILDEF:
            (CSail&)rigsail = CSailWorker(CSailDefXmlReader("saildef").read(rigsail.filename.data())).makeSail();
            break;
        case SAIL3D:
            (CSail&)rigsail = CSailXmlReader("3Dsail").read(rigsail.filename.data());
            break;
        }
        signalUpdate(rigsail);
    }
    catch (CException e)
    {
        QMessageBox::information(0,tr("error"), tr("There was an error reading from the selected file."));
    }
}


/**
 * The "remove" button was pressed, fire signalRemove.
 */
void CRigSailWidget::slotRemove()
{
    signalRemove();
}


/**
 * The "update" button was pressed, fire signalUpdate.
 */
void CRigSailWidget::slotUpdate()
{
    rigsail.name = txtName->text().latin1();
    rigsail.origin = wdgOrigin->getVector();
    signalUpdate(rigsail);
}



/**
 * The constructor.
 *
 * @param parent the parent widget
 */
CRigDefPanel::CRigDefPanel(QWidget *parent)
        : QWidget(parent)
{
    QGridLayout *layout = new QGridLayout( this, 1, 1, 11, 6 );
    tabs = new QTabWidget(this);
    layout->addWidget( tabs, 0, 0);
}


/**
 * Sets the strings of the subwidgets using the current language.
 */
void CRigDefPanel::languageChange()
{
    for ( unsigned int i = 0; i < sailwidget.size(); i++)
        sailwidget[i]->languageChange();
}


/**
 * We were passed a new rig definition, update the widgets.
 *
 * @param newdef The new rig definition
 */
void CRigDefPanel::setRigDef(const CRigDef& newdef)
{
    unsigned int i;
    for ( i = 0; i < sailwidget.size(); i++)
    {
        tabs->removePage((QWidget *)sailwidget[i]);
        delete sailwidget[i];
    }

    rigdef = newdef;
    sailwidget.resize(rigdef.rigsail.size());

    for ( i = 0; i < rigdef.rigsail.size(); i++)
    {
        sailwidget[i] = new CRigSailWidget(tabs);
        sailwidget[i]->setRigSail(rigdef.rigsail[i]);

        connect(sailwidget[i], SIGNAL(signalRemove()), this, SLOT(slotRemove()));
        connect(sailwidget[i], SIGNAL(signalUpdate(const CRigSail&)), this, SLOT(slotUpdate(const CRigSail&)));

        tabs->addTab(sailwidget[i], rigdef.rigsail[i].name.data());
        ;
    }

    if (sailwidget.size() > 0)
        tabs->showPage(sailwidget[0]);
}



/**
 * The user requested the removal of the current rigsail.
 */
void CRigDefPanel::slotRemove()
{
    QWidget *current = tabs->currentPage();

    for (unsigned int i = 0; i < sailwidget.size(); i++)
    {
        if ((QWidget*)sailwidget[i] == current)
        {
            tabs->removePage(current);
            delete sailwidget[i];
            sailwidget.erase(sailwidget.begin()+i);
            rigdef.rigsail.erase(rigdef.rigsail.begin()+i);
            signalUpdate(rigdef);
            break;
        }
    }

}


/**
 * The user updated the current rigsail.
 *
 * @param newsail The new rigsail
 */
void CRigDefPanel::slotUpdate(const CRigSail& newsail)
{
    QWidget *current = tabs->currentPage();

    for (unsigned int i = 0; i < sailwidget.size(); i++)
    {
        if ((QWidget*)sailwidget[i] == current)
        {
            rigdef.rigsail[i] = newsail;
            tabs->changeTab(current, rigdef.rigsail[i].name.data());
            signalUpdate(rigdef);
            break;
        }
    }

}

