/* 
 * need to #define SH_USE_KERN
 *
 */
#define SH_SYSCALL_CODE

#include "config.h"

#ifdef HOST_IS_I86LINUX
#define SH_IDT_TABLE
#endif

#include <stdio.h>
#include <stdlib.h>

#if defined(SH_USE_KERN) 

void usage(int flag)
{
  printf("\n");
  printf("Usage: kern_head [-v | --verbose]\n");
  printf("       kern_head [-h | --help]\n");
  printf("\n");
  printf("       You need superuser privileges to use this program,\n");
  printf("       because only the superuser can read from /dev/kmem.\n");
  printf("\n");
  exit(flag);
}

#if defined(HOST_IS_LINUX)

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>
#include <sys/utsname.h>

/* number of system calls */
#define SH_MAXCALLS 512

#include "kern_head.h"

static int verbose = 0;

typedef struct _smap_entry {
#ifdef SH_SYSCALL_CODE
  unsigned int  code[2];  /* 8 bytes */
#endif
  unsigned long addr;
  char          name[64];
} smap_entry;

union {
  unsigned long addr_sys_call_table;
  unsigned char str_sys_call_table[sizeof(unsigned long)];
} sh_sys_call;

#define SYS_CODE_SIZE 1024

unsigned long addr_system_call;
unsigned char system_call_code[SYS_CODE_SIZE];

static int read_kcode (unsigned long addr, unsigned char * buf, int len);

#ifdef SH_IDT_TABLE

#include <asm/segment.h>

/* number of idt entries  */
#define SH_MAXIDT   256

unsigned char idt_table[SH_MAXIDT * 8];

char * sh_strseg(unsigned short segment)
{
  switch (segment) {
  case __KERNEL_CS:
    return "KERNEL_CS";
  case __KERNEL_DS:
    return "KERNEL_DS";
  case __USER_CS:
    return "USER_CS";
  case __USER_DS:
    return "USER_DS";
  default:
    return "unknown";
  }
}

int get_idt ()
{
  int            i, j;
  unsigned char  buf[6];
  unsigned short idt_size;
  unsigned long  idt_addr;

  int            curr_keep = 0;
  unsigned short idt_offset_lo, idt_offset_hi, idt_selector;
  unsigned char  idt_reserved, idt_flag;
  int            dpl;
  unsigned long  idt_iaddr;


  /* get address and size of Interrupt Descriptor Table
   */
  __asm__ volatile ("sidt %0": "=m" (buf));
  idt_size = *((unsigned short *) &buf[0]);
  idt_addr = *((unsigned long *)  &buf[2]);
  idt_size = (idt_size + 1)/8;

  if (idt_size > SH_MAXIDT)
    {
      fprintf(stderr, "get_idt: IDT size <%d> exceeds maximum <%d>\n", 
	      (int) idt_size, SH_MAXIDT);
      return -1;
    }

  if (0 != read_kcode (idt_addr, idt_table, idt_size*8))
    {
      fprintf(stderr, "get_idt: could not read interrupt descriptor table\n");
      return -1;
    }

  printf("/* found interrupt descriptor table */\n\n");
  printf("static unsigned char idt_table[2048] = {\n");

  for (j = 0; j < idt_size; ++j)
    {
      i = j * 8;

      idt_offset_lo = *((unsigned short *) &idt_table[i]);
      idt_selector  = *((unsigned short *) &idt_table[i+2]);
      idt_reserved  = (unsigned char) idt_table[i+4];
      idt_flag      = (unsigned char) idt_table[i+5];
      idt_offset_hi = *((unsigned short *) &idt_table[i+6]);

      idt_iaddr = (unsigned long)(idt_offset_hi << 16) + idt_offset_lo;

      if (idt_iaddr != 0)
	{
	  if (idt_flag & 64) 
	    dpl = 3;
	  else
	    dpl = 0;
	  if (idt_flag & 1)
	    {
	      if (dpl == 3)
		{
		  if (verbose > 0)
		    fprintf(stderr, "%04i  0x%-14.8lx %-12s %3i  %s\n", 
			    j, idt_iaddr, sh_strseg(idt_selector), 
			    dpl, "System");
		  printf("/* %04i  0x%-14.8lx %-12s %3i  %-10s */\n", 
			 j, idt_iaddr, sh_strseg(idt_selector), 
			 dpl, "System");
		}
	      else
		{
		  if (verbose > 0)
		    fprintf(stderr, "%04i  0x%-14.8lx %-12s %3i  %s\n", 
			    j, idt_iaddr, sh_strseg(idt_selector), 
			    dpl, "Trap");
		  printf("/* %04i  0x%-14.8lx %-12s %3i  %-10s */\n", 
			 j, idt_iaddr, sh_strseg(idt_selector), 
			 dpl, "Trap");
		}
	    }
	  else
	    {
	      if (verbose > 0)
		fprintf(stderr, "%04i  0x%-14.8lx %-12s %3i  %s\n", 
			j, idt_iaddr, sh_strseg(idt_selector), 
			dpl, "Interrupt");
	      printf("/* %04i  0x%-14.8lx %-12s %3i  %-10s */\n", 
		     j, idt_iaddr, sh_strseg(idt_selector), 
		     dpl, "Interrupt");
	    }

	  ++curr_keep;
	} else {
	  memset(&idt_table[i], '\0', 8);
	}

      printf(" 0x%-2.2hhx, 0x%-2.2hhx, 0x%-2.2hhx, 0x%-2.2hhx,"\
	     " 0x%-2.2hhx, 0x%-2.2hhx, 0x%-2.2hhx, 0x%-2.2hhx,\n",
	     idt_table[i],   idt_table[i+1], 
	     idt_table[i+2], idt_table[i+3], 
	     idt_table[i+4], idt_table[i+5], 
	     idt_table[i+6], idt_table[i+7]);

    }

  printf("};\n\n");
  printf("#define IDT_TABLE_NUM  %d\n\n", curr_keep);

  return 0;
}
/* ifdef SH_IDT_TABLE */
#endif

int fill_smap(smap_entry * sh_smap, int num)
{
  FILE * fp;
  char buf[512], addr[16], name[128];
  int  i, j, count = 0, maxcall = 0;

  fp = fopen (SYSTEMMAP, "r");

  if (!fp)
    {
      perror("fill_smap: fopen");
      fprintf(stderr, "fill_smap: error opening <%s>\n", SYSTEMMAP);
      return -1;
    }

  /* initialize
   */
  sh_sys_call.addr_sys_call_table = 0L;

  while (fgets(buf, 512, fp) != NULL)
    {
      
      if ( ( (buf[9] == 'D') || (buf[9] == 'd') ) && 
	   0 == strncmp("sys_call_table", &buf[11], 14))
	{
	  printf("/* found sys_call_table */\n");
	  /* --- copy symbol address ---
	   */
	  addr[0] = '0'; addr[1] = 'x'; addr[2] = '\0';
	  strncat(&addr[2], buf, 8);
	  addr[10] = '\0';
	  sh_sys_call.addr_sys_call_table = strtoul(addr, NULL, 0);
	  if (sh_sys_call.addr_sys_call_table == ULONG_MAX)
	    {
	      perror("fill_smap");
	      return -1;
	    }
	  else
	    {
	      printf("#define SH_SYS_CALL_TABLE %s\n", addr);
	    }
	}

      if (buf[9] != 'T')
	continue;

      if (0 == strncmp("system_call", &buf[11], 11))
	{
	  printf("/* found system_call */\n");
	  /* --- copy symbol address ---
	   */
	  addr[0] = '0'; addr[1] = 'x'; addr[2] = '\0';
	  strncat(&addr[2], buf, 8);
	  addr[10] = '\0';
	  addr_system_call = strtoul(addr, NULL, 0);
	  if (addr_system_call == ULONG_MAX)
	    {
	      perror("fill_smap");
	      return -1;
	    }
	}


      if ( (buf[11]!='s' || buf[12]!='y' || buf[13]!='s' || buf[14]!='_') &&
	   (buf[11]!='o' || buf[12]!='l' || buf[13]!='d' || buf[14]!='_'))
	continue;

      for (i = 0; i < num; ++i)
	{
	  for (j = 0; j < 128; ++j)
	    {
	      if (buf[11+j] == '\n' || buf[11+j] == '\0')
		{
		  name[j] = '\0';
		  break;
		}
	      name[j] = buf[11+j];
	    }


	  if (0 == strcmp(name, sh_smap[i].name)) 
	    {
	      
	      /* --- copy symbol address ---
	       */
	      addr[0] = '0'; addr[1] = 'x'; addr[2] = '\0';
	      strncat(&addr[2], buf, 8);
	      addr[10] = '\0';
	      sh_smap[i].addr = strtoul(addr, NULL, 0);
	      if (sh_smap[i].addr == ULONG_MAX)
		{
		  perror("fill_smap");
		  return -1;
		}
	      ++count;
	      if (i > maxcall) maxcall = i;
	      /* printf("maxcall = %d\n", maxcall); */
	      /* break; */
      	    }
	}
    }
  fclose(fp);
  if ((count > 0) && (maxcall > 0))
    return maxcall+1;
  else
    return count;
}

int read_kcode (unsigned long addr, unsigned char * buf, int len)
{
  int fd;

  if (addr == 0UL)
    {
      perror("read_kcode: invalid input");
      return -1;
    }

  fd = open ("/dev/kmem", O_RDONLY);
  if (fd < 0)
    {
      perror("read_kcode: open");
      return -1;
    }
  if (lseek(fd, addr, SEEK_SET) == (off_t) (-1))
    {
      perror("read_kcode: lseek");
      return -1;
    }
  if (read(fd, buf, len) < 0)
    {
      perror("read_kcode: read");
      return -1;
    }
  return 0;
}


int get_dispatch (int * qq)
{
  int i;

  if (addr_system_call == 0L || sh_sys_call.addr_sys_call_table == 0L)
    {
      fprintf(stderr, "get_dispatch: invalid data\n");
      return -1;
    }

  if (0 != read_kcode (addr_system_call, system_call_code, SYS_CODE_SIZE))
    {
      fprintf(stderr, "get_dispatch: could not read system_call code\n");
      return -1;
    }

  for (i = 0; i < (SYS_CODE_SIZE - 4); ++i)
    {
      if (system_call_code[i]   == sh_sys_call.str_sys_call_table[0] &&
	  system_call_code[i+1] == sh_sys_call.str_sys_call_table[1] &&
	  system_call_code[i+2] == sh_sys_call.str_sys_call_table[2] &&
	  system_call_code[i+3] == sh_sys_call.str_sys_call_table[3])
	{
	  /*
	    fprintf(stderr, "INFO: get_dispatch: found sys_call_table in "\
		    "system_call code at %d\n", i);
	  */
	  *qq = i;
	  return 0;
	}
    }
  fprintf(stderr, 
	  "get_dispatch: did not find sys_call_table in system_call code\n");
  return -1;
}
  
int check_smap (smap_entry * sh_smap, int count)
{
  int fd;
  int i, j = 0;
  unsigned long kaddr = sh_sys_call.addr_sys_call_table;
  unsigned long kmem_call_table[512];

  if (kaddr == 0L)
    {
      fprintf(stderr, "check_smap: addr_sys_call_table == 0L\n");
      return -1;
    }

  fd = open ("/dev/kmem", O_RDONLY);
  if (fd < 0)
    {
      perror("check_smap: open");
      return -1;
    }

  if(lseek(fd, (off_t)kaddr, SEEK_SET) == -1)
    {
      perror("check_smap: lseek");
      return -1;
    }

  if (sizeof(kmem_call_table) !=
      read(fd, &kmem_call_table, sizeof(kmem_call_table)))
    {
      perror("check_smap: read");
      return -1;
    }

  close(fd);

  for (i = 0; i < count; ++i)
    {
      if (sh_smap[i].name == NULL || sh_smap[i].addr == 0UL)
        break;

      if (verbose > 0)
        fprintf(stderr, "(kernel) %#lx   %#lx (map)  [%s]\n",
                kmem_call_table[i],
                sh_smap[i].addr,
                sh_smap[i].name);

      if (sh_smap[i].addr != kmem_call_table[i])
        {
          fprintf(stderr,
                  "WARNING: (kernel) %#lx != %#lx (map)  [%s]\n",
                  kmem_call_table[i],
                  sh_smap[i].addr,
                  sh_smap[i].name);
	  ++j;
        }
    }

  if (j > 0)
    {
      fprintf(stderr, "For %d syscalls, there is a mismatch between the \n",
	      j);
      fprintf(stderr, "address in the System.map %s and the kernel address.\n",
	      SYSTEMMAP);
      fprintf(stderr, "Either your System.map doesn't match your kernel or\n");
      fprintf(stderr, "your kernel syscall table has been modified (by\n");
      fprintf(stderr, "a legitimate kernel module or a rootkit).\n");
    }

  return 0;
}

#ifdef SH_SYSCALL_CODE
int read_code (smap_entry * sh_smap, int count)
{
  int fd;
  int i;

  fd = open ("/dev/kmem", O_RDONLY);
  if (fd < 0)
    {
      perror("read_code: open");
      return -1;
    }

  for (i = 0; i < count; ++i) 
    {
      if (sh_smap[i].name == NULL || sh_smap[i].addr == 0UL)
	break;

      lseek(fd, sh_smap[i].addr, SEEK_SET);
      read(fd, &(sh_smap[i].code[0]), 2 * sizeof(unsigned int));
    }

  return 0;
}
#endif


int main(int argc, char * argv[])
{
  int i, j, count, maxcall, qq;
  int which = 4;
  smap_entry sh_smap[SH_MAXCALLS];
  struct utsname utbuf;


  if (argc > 1)
    {
      if (strcmp(argv[1], "-h") == 0 ||  strcmp(argv[1], "--help") == 0)
	usage(EXIT_SUCCESS);
      else if (strcmp(argv[1], "-v") == 0 ||
	       strcmp(argv[1], "--verbose") == 0)
	verbose = 1;
    }

  if (0 != uname(&utbuf))
    {
      perror("kern_head: uname");
      exit (EXIT_FAILURE);
    }

  if      (strncmp(utbuf.release, "2.2", 3) == 0)
    which = 2;
  else if (strncmp(utbuf.release, "2.4", 3) == 0)
    which = 4;
  else if (strncmp(utbuf.release, "2.6", 3) == 0)
    which = 4;
  else
    {
      fprintf(stderr, "kern_head: kernel %s not supported\n", utbuf.release);
      exit (EXIT_FAILURE);
    }
  
  if (utbuf.machine[0] != 'i' || utbuf.machine[2] != '8' || 
      utbuf.machine[3] != '6')
    {
      fprintf(stderr, "kern_head: machine %s not supported\n", utbuf.machine);
      exit (EXIT_FAILURE);
    }

#if defined(SH_SYSCALL_CODE) || defined(SH_IDT_TABLE) 
  if (0 != getuid())
    {
      fprintf(stderr, "\n");
      
      fprintf(stderr, "NOTE:  kern_head: must run as 'root' (need to read from /dev/kmem)\n");
      fprintf(stderr, "       If you get this message, then proceed as follows:\n");
      fprintf(stderr, "       $ su\n");
      fprintf(stderr, "       $ ./kern_head > sh_ks.h\n");
      fprintf(stderr, "       $ exit\n");
      fprintf(stderr, "       $ make\n\n");
      exit (EXIT_FAILURE);
    }
#endif
      
  printf("#ifndef SH_KERN_CALLS_H\n");
  printf("#define SH_KERN_CALLS_H\n\n");

  printf("\n/* Kernel %s, machine %s -- use table %s */\n\n", 
	 utbuf.release, utbuf.machine,
	 (which == 2) ? "callz_2p2" : "callz_2p4");
      

  /* initialite the system call table 
   */
  for (i = 0; i < SH_MAXCALLS; ++i)
    {
      if (which == 2)
	{
	  if (callz_2p2[i] == NULL)
	    break;
	  strcpy(sh_smap[i].name, callz_2p2[i]);
	}
      else
	{
	  if (callz_2p4[i] == NULL)
	    break;
	  strcpy(sh_smap[i].name, callz_2p4[i]);
	}
      sh_smap[i].addr    = 0UL;
    }

  count = i;

  /* get the actual number of the highest syscalls and use no more
   */
  maxcall = fill_smap(sh_smap, count);
  if ( maxcall < 0)
    {
      printf("#endif\n");
      fprintf(stderr, "kern_head: fill_smap failed\n");
      exit (EXIT_FAILURE);
    }

  for (i = 0; i < maxcall; ++i)
    {
      if (sh_smap[i].addr == 0UL)
        {
          if (verbose > 0)
            fprintf(stdout, "** unknown syscall **: [%s]\n", sh_smap[i].name);
          strcpy(sh_smap[i].name, sh_smap[0].name);
          sh_smap[i].addr = sh_smap[0].addr;
        }
    }

  if ( check_smap(sh_smap, maxcall) < 0)
    {
      printf("#endif\n");
      fprintf(stderr, "kern_head: check_smap failed\n");
      exit (EXIT_FAILURE);
    }

#ifdef SH_SYSCALL_CODE
  if ( read_code (sh_smap, maxcall) < 0)
    {
      printf("#endif\n");
      fprintf(stderr, "kern_head: read_code failed\n");
      exit (EXIT_FAILURE);
    }
#endif

#ifdef SH_IDT_TABLE
  if ( get_idt() < 0)
    {
      printf("#endif\n");
      fprintf(stderr, "kern_head: get_idt failed\n");
      exit (EXIT_FAILURE);
    }
#endif

  if ( get_dispatch (&qq) < 0)
    {
      printf("#endif\n");
      fprintf(stderr, "kern_head: get_dispatch failed\n");
      exit (EXIT_FAILURE);
    }

  if (qq <= 252)
    printf("#define SYS_CALL_LOC  %d\n", qq);
  else
    {
      printf("#endif\n");
      fprintf(stderr, "kern_head: SYS_CALL_LOC (%d) too large\n", qq);
      exit(EXIT_FAILURE);
    }
  printf("static unsigned long system_call_addr = %#lx;\n", addr_system_call);
  printf("static unsigned char system_call_code[256] = {\n");

  for (j = 0; j < 31; ++j) 
    {
      i = j * 8;
      printf(" %#hhx, %#hhx, %#hhx, %#hhx,"\
	     " %#hhx, %#hhx, %#hhx, %#hhx,\n",
	     system_call_code[i],   system_call_code[i+1], 
	     system_call_code[i+2], system_call_code[i+3], 
	     system_call_code[i+4], system_call_code[i+5], 
	     system_call_code[i+6], system_call_code[i+7]);
    }
  i = 248;
  printf(" %#hhx, %#hhx, %#hhx, %#hhx,"\
	 " %#hhx, %#hhx, %#hhx, %#hhx };\n\n\n",
	 system_call_code[i],   system_call_code[i+1], 
	 system_call_code[i+2], system_call_code[i+3], 
	 system_call_code[i+4], system_call_code[i+5], 
	 system_call_code[i+6], system_call_code[i+7]);

  printf("#define SH_MAXCALLS %d\n\n", maxcall);

  printf("typedef struct _sh_syscall_t {\n");
#ifdef SH_SYSCALL_CODE
  printf("  unsigned int  code[2];  /* 8 bytes */\n");
#endif
  printf("  unsigned long addr;\n");
  printf("  char *        name;\n");
  printf("} sh_syscall_t;\n\n");

  printf("static sh_syscall_t sh_syscalls[] = {\n");

  for (i = 0; i < maxcall; ++i) 
    {
#ifdef SH_SYSCALL_CODE
      printf(" /* %03d */   { { %#x, %#x }, %#lx, N_(%c%s%c) },\n", 
	     i, sh_smap[i].code[0], sh_smap[i].code[1], 
	     sh_smap[i].addr, '"', sh_smap[i].name, '"');
#else
      printf(" /* %03d */   { %#lx, N_(%c%s%c) },\n", 
	     i, sh_smap[i].addr, '"', sh_smap[i].name, '"');
#endif
    }
#ifdef SH_SYSCALL_CODE
  printf(" /* eof */   { { 0x00000000, 0x00000000 }, 0x00000000,  NULL }\n");
#else
  printf(" /* eof */   { 0x00000000,  NULL }\n");
#endif
  printf("};\n\n");

  printf("#endif\n");

  exit (EXIT_SUCCESS);
}

/* if defined(HOST_IS_LINUX) */
#endif

/************************************************************
 *
 *
 *  FreeBSD Implementation
 *
 ************************************************************/

#if defined(HOST_IS_FREEBSD) || defined(__OpenBSD__)

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <err.h>
#include <kvm.h>
#include <fcntl.h>
#include <nlist.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/utsname.h>

#ifdef __FreeBSD__
#include <sys/sysent.h>
#endif

#include <sys/syscall.h>

#ifndef  SYS_MAXSYSCALL
#define  SYS_MAXSYSCALL	364
#endif

/* number of system calls */
#define SH_MAXCALLS 512
#include "kern_head.h"
static int verbose = 0;

#ifdef __OpenBSD__
struct proc;
struct sysent {
	short sy_narg;
	short sy_argsize;
	int   (*sy_call)(struct proc *, void *, register_t *);
};
#endif

typedef struct _smap_entry {
  unsigned int  code[2];  /* 8 bytes */
  unsigned long addr;
  char          name[64];
} smap_entry;

union {
  unsigned long addr_sys_call_table;
  unsigned char str_sys_call_table[sizeof(unsigned long)];
} sh_sys_call;

struct nlist sys_list[SYS_MAXSYSCALL+1];

struct  nlist   list[2];


int main(int argc, char * argv[])
{
  int i, count;
  smap_entry sh_smap[SYS_MAXSYSCALL];
  struct utsname utbuf;
  char errbuf[_POSIX2_LINE_MAX];

  struct sysent  sy;
  unsigned long offset = 0L;
  kvm_t *kd;

  list[0].n_name = "_sysent";
  list[1].n_name = NULL;

  if (argc > 1)
    {
      if (strcmp(argv[1], "-h") == 0 ||  strcmp(argv[1], "--help") == 0)
	usage(EXIT_SUCCESS);
      else if (strcmp(argv[1], "-v") == 0 ||
	       strcmp(argv[1], "--verbose") == 0)
	verbose = 1;
    }

  if (0 != uname(&utbuf))
    {
      perror("kern_head: uname");
      exit (EXIT_FAILURE);
    }

  if (0 != getuid())
    {
      fprintf(stderr, "\n");
      fprintf(stderr, "NOTE:  kern_head: must run as 'root'");
      fprintf(stderr, "(need to read from /dev/kmem)\n");
      fprintf(stderr, "       If you get this message, then proceed");
      fprintf(stderr, "as follows:\n");
      fprintf(stderr, "       $ su\n");
      fprintf(stderr, "       $ ./kern_head > sh_ks.h\n");
      fprintf(stderr, "       $ exit\n");
      fprintf(stderr, "       $ make\n\n");
      exit (EXIT_FAILURE);
    }

  kd = kvm_openfiles(NULL, NULL, NULL, O_RDONLY, errbuf);
  if (!kd)
    {
      fprintf(stderr, "check_sysent: kvm_openfiles: %s\n", errbuf);
      exit(EXIT_FAILURE);
    }

  i = kvm_nlist(kd, list);
  if (i == -1)
    {
      fprintf(stderr, "check_sysent: kvm_nlist: %s\n", kvm_geterr(kd));
      exit(EXIT_FAILURE);
    }

  printf("\n/* Kernel %s, machine %s -- use table %s */\n\n", 
	 utbuf.release, utbuf.machine, "callz_fbsd");
      
  i = 0;
  while (callz_fbsd[i] != NULL)
    {
      sys_list[i].n_name = callz_fbsd[i];
      /* fprintf(stderr, "sys_list[%d] = %s\n", i, sys_list[i].n_name); */
      ++i;
    }
  count = i;
  sys_list[i].n_name = NULL;
   
  i = kvm_nlist(kd, sys_list);
  if (i == -1)
    {
      fprintf(stderr, "check_sysent: kvm_nlist: %s\n", kvm_geterr(kd));
      /* exit(EXIT_FAILURE); */
    }
  else if (i != 0 && verbose != 0)
     {
	fprintf(stderr, "check_sysent: kvm_nlist: %d out of %d invalid.\n",
		i, count);
	fprintf(stderr, "              Probably callz_fbsd in kern_head.c\n");
	fprintf(stderr, "              is not for your kernel version.\n");
	fprintf(stderr, "              (No reason to worry, kcheck will "\
                                       "work anyway)\n\n");
     }

  for (i = 0; i < SYS_MAXSYSCALL; i++) 
    {
       if (NULL == sys_list[i].n_name)
	 break;
       if (!sys_list[i].n_value && 0 != strcmp(sys_list[i].n_name, "_nosys")
	   && verbose != 0)
	{
	  fprintf(stderr,"check_sysent: not found: slot %03d [%s]\n", 
		  i, sys_list[i].n_name);
	  /* exit(EXIT_FAILURE); */
	}
      offset = list[0].n_value + (i*sizeof(struct sysent));
      if (kvm_read(kd, offset, &sy, sizeof(struct sysent)) < 0)
	{
	  fprintf(stderr,"check_sysent: kvm_read: %s\n", kvm_geterr(kd));
	  exit(EXIT_FAILURE);
	}

      if (verbose > 0)
	fprintf(stderr, "(sysent) %#lx   %#lx (syscall)  %03d [%s]\n",
		(unsigned long) sys_list[i].n_value,
		(unsigned long) sy.sy_call,
		i, sys_list[i].n_name);

      if((unsigned long)sy.sy_call != sys_list[i].n_value && 
	 sys_list[i].n_value != 0 &&
	 0 != strcmp(sys_list[i].n_name, "_nosys"))  
	{
          fprintf(stderr,
                  "WARNING: (sysent) %#lx != %#lx (syscall)  %03d [%s]\n",
		  (unsigned long) sys_list[i].n_value,
		  (unsigned long) sy.sy_call,
		  i, sys_list[i].n_name);
	}
      sh_smap[i].addr = (unsigned long) sy.sy_call;
      strncpy(sh_smap[i].name, sys_list[i].n_name, 64);
      if(kvm_read(kd, (unsigned int) sy.sy_call, &(sh_smap[i].code[0]), 
		  2 * sizeof(unsigned int)) < 0)
	{
	  fprintf(stderr,"check_sysent: kvm_read: %s\n", kvm_geterr(kd));
	  exit(EXIT_FAILURE);
	}
    }

  if(kvm_close(kd) < 0) 
    {
      fprintf(stderr,"check_sysent: kvm_nlist: %s\n", kvm_geterr(kd));
      exit(EXIT_FAILURE);
    }
 
  printf("#ifndef SH_KERN_CALLS_H\n");
  printf("#define SH_KERN_CALLS_H\n\n");

  printf("#define SH_MAXCALLS %d\n\n", count);

  printf("typedef struct _sh_syscall_t {\n");
  printf("  unsigned int  code[2];  /* 8 bytes */\n");
  printf("  unsigned long addr;\n");
  printf("  char *        name;\n");
  printf("} sh_syscall_t;\n\n");

  printf("static sh_syscall_t sh_syscalls[] = {\n");
  for (i = 0; i < count; ++i) {
    printf(" /* %03d */ {{ 0x%-8.8x, 0x%-8.8x }, 0x%-8.8lx, N_(%c%s%c) },\n", 
	   i, sh_smap[i].code[0], sh_smap[i].code[1], 
	   sh_smap[i].addr, '"', sh_smap[i].name, '"');
  }
  printf(" /* eof */   { { 0x00000000, 0x00000000 }, 0x00000000,  NULL }\n");
  printf("};\n\n");
  printf("#endif\n");
  return 0;
}
/* if defined(HOST_IS_FREEBSD) */
#endif

/* #if defined(SH_USE_KERN) */
#else

#include <stdio.h>
#include <stdlib.h>

int main()
{
  printf("#ifndef SH_KERN_CALLS_H\n");
  printf("#define SH_KERN_CALLS_H\n\n");

  printf("/* Dummy header. */\n\n");

  printf("typedef struct _sh_syscall_t {\n");
  printf("  unsigned long addr;\n");
  printf("  char *        name;\n");
  printf("} sh_syscall_t;\n\n");

  printf("#endif\n");

  return (EXIT_SUCCESS);
}

/* #ifdef SH_USE_KERN */
#endif
