/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2003 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h> 
#include <errno.h>

#include <unistd.h>
#include <fcntl.h>

#include <sys/socket.h>
#include <sys/un.h>

#include <signal.h>

#if !defined(AF_FILE)
#define AF_FILE AF_UNIX
#endif

#define SH_MAXMSG 128

static int    sock     = -1;
static char * sockname = NULL;

#ifdef SH_STEALTH
char * globber(char * string);
#define _(string) globber(string) 
#define N_(string) string
#else
#define _(string)  string 
#define N_(string) string
#endif

#ifdef SH_STEALTH
#ifndef SH_MAX_GLOBS
#define SH_MAX_GLOBS 32
#endif
char * globber(char * str)
{
  register int i, j;
  static int  count = -1;
  static char glob[SH_MAX_GLOBS][128];

  ++count; if (count > (SH_MAX_GLOBS-1) ) count = 0;
  j = strlen(str);
  if (j > 127) j = 127;

  for (i = 0; i < j; ++i)
    {
      if (str[i] != '\n' && str[i] != '\t') 
	glob[count][i] = str[i] ^ XOR_CODE;
      else
	glob[count][i] = str[i];
    }
  glob[count][j] = '\0';
  return glob[count];
}
#endif

#define CLIENT _("yulectl")


int 
make_named_socket (char * sockname)
{
  int sock;

#if 0
  struct sockaddr_un name;
  size_t size;
#else
  sockname = NULL;
#endif

  /* Create the socket. */
  
  sock = socket (PF_UNIX, SOCK_STREAM, 0);
  if (sock < 0)
    {
      perror (_("socket"));
      return -1;
    }

#if 0
  /* Bind a name to the socket. */
  name.sun_family = AF_FILE;
  strcpy (name.sun_path, sockname);

  /* The size of the address is
     the offset of the start of the filename,
     plus its length,
     plus one for the terminating null byte. */
  size = (offsetof (struct sockaddr_un, sun_path)
          + strlen (name.sun_path) + 1);

  if (bind (sock, (struct sockaddr *) &name, size) < 0)
    {
      perror (_("bind"));
      return -1;
    }
#endif

  return sock;
}

void
termination_handler (int signum)
{
  /* Clean up. */
  if (signum != 0)
    {
      fprintf(stderr, _("Terminated on signal %d\n"), signum);
    }
#if 0
  if (sockname != NULL) unlink (sockname);
#endif
  if (sock   >= 0 ) close  (sock);

  return;
}


int send_to_server (char * serversock, char * message)
{
  struct sockaddr_un name;
  /* size_t size; */
  int size;
  int nbytes;

  /* Initialize the server socket address. 
   */
  name.sun_family = AF_UNIX;
  strncpy (name.sun_path, serversock, sizeof(name.sun_path) - 1);
  size = (offsetof (struct sockaddr_un, sun_path)
          + strlen (name.sun_path) + 1);

  nbytes = connect(sock, (struct sockaddr *) & name, size);
  if (nbytes < 0)
    {
      perror (_("connect"));
      return -1;
    }

  /* Send the datagram. 
  nbytes = sendto (sock, message, strlen (message) + 1, 0,
                   (struct sockaddr *) & name, size);
   */
  nbytes = send (sock, message, strlen (message) + 1, 0);

  if (nbytes < 0)
    {
      perror (_("send"));
      return -1;
    }
  return 0;
}

static int getline_from_server (int sock, char * buf, int size)
{
  int nbytes = 0;
  int status = 0;
  char * p   = buf;

  do {
    status = read (sock, p, 1);
    if (status <= 0)
      {
	buf[nbytes] = '\0';
	return ((status == 0) ? nbytes : status);
      }
    else if (*p == '\0')
      {
	return nbytes;
      }
    ++nbytes; ++p;
  } while (nbytes < size);
  buf[size-1] = '\0';
  return 0;
}

int recv_from_server (char * message)
{
  int nbytes = 0;
  char recvmsg[SH_MAXMSG];
  int  num = 0;

  if (message[0] == 'L' && message[1] == 'I' &&
      message[2] == 'S' && message[3] == 'T')
    {
      do {
	/*
	nbytes = recvfrom (sock, recvmsg, SH_MAXMSG, 0, NULL, 0);
	*/
	nbytes = getline_from_server (sock, recvmsg, SH_MAXMSG);
	if (nbytes < 0)
	  {
	    if (errno == EAGAIN)
	      {
		return 0;
	      }
	    else
	      {
		perror (_("recv"));
		return -1;
	      }
	  }
	else if (nbytes == 0)
	  {
	    return 0;
	  }
	if (recvmsg[0] == 'E' && recvmsg[1] == 'N' && recvmsg[2] == 'D')
	  {
	    return 0;
	  }
	++num;
	fprintf (stdout, _("%03d: %s\n"), num, recvmsg);
      } while (nbytes >= 0);
    }
  else
    {
      /*
      nbytes = recvfrom (sock, recvmsg, SH_MAXMSG, 0, NULL, 0);
      */
      nbytes = recv (sock, recvmsg, SH_MAXMSG, 0);
      if (nbytes < 0)
	{
	  perror (_("recv"));
	  return -1;
	}
    }

  /* Print a diagnostic message. */
  if (0 != strcmp (message, recvmsg))
    {
      fprintf (stderr, _("ERROR: Bounced message != original message.\n"));
      return -1;
    }
  else
    {
      fprintf (stdout, _("# Message received by server.\n"));
    }

  return 0;
}

void usage(char * name)
{
  printf(_("\nUsage : %s [-s server_socket] -c command client_hostname\n\n"), 
	 name);

  printf(_("Purpose : send commands to the server via a socket,\n"));
  printf(_("          in particular commands that the server would\n"));
  printf(_("          transfer to the client <client_hostname> when\n"));
  printf(_("          this client connects to deliver a message.\n\n"));

  printf(_("Commands: RELOAD <reload configuration>\n"));
  printf(_("          STOP   <terminate>\n"));
  printf(_("          CANCEL <cancel previous command>\n"));
  printf(_("          LIST   <list queued commands>\n"));
  printf(_("                 (requires a dummy client_hostname)\n\n"));
  return;
}

int
main (int argc, char * argv[])
{

  char   message[SH_MAXMSG] = "";
  char   clientcd[256];
  char   serversock[256];
  int    status, size;
  int    num = 1;
  int    flag = 0;
  
#ifdef HAVE_VSNPRINTF
  status = snprintf(serversock, 256, _("%s/%s.sock"), 
		    DEFAULT_PIDDIR, SH_INSTALL_NAME);
#else
  if ((strlen(DEFAULT_PIDDIR) + strlen(SH_INSTALL_NAME) + 1) >= 256)
    {
      status = -1;
    }
  else
    {
      status = sprintf (serversock, 256, _("%s/%s.sock"), 
			DEFAULT_PIDDIR, SH_INSTALL_NAME);
    }
#endif

  if ((status < 0) || (status > 255))
    {
      fprintf(stderr, _("Path too long (maximum 255): %s/%s.sock\n"), 
	      DEFAULT_PIDDIR, SH_INSTALL_NAME);
      return (EXIT_FAILURE);
    }

  while (argc > 2 && argv[num][0] == '-')
    {
      switch (argv[num][1]) 
	{
	  case 'h':
	    usage(argv[0]);
	    return (EXIT_SUCCESS);

	  case 's':
	    --argc; ++num;
	    if (argv[num] == NULL || argv[num][0] == '\0') {
	      usage(argv[0]);
	      fprintf(stderr, _("ERROR: -s: argument missing\n"));
	      return (EXIT_FAILURE);
	    } else {
	      if (strlen(argv[num]) > 255) 
		{
		  fprintf(stderr, _("ERROR: Path too long: %s\n"), argv[num]);
		  return (EXIT_FAILURE);
		}
	      strncpy (serversock, argv[num], 256);
	      serversock[255] = '\0';
	    }
	    break;

	  case 'c':
	    --argc; ++num;
	    if (argv[num] == NULL || argv[num][0] == '\0') {
	      usage(argv[0]);
	      fprintf(stderr, _("ERROR: -c: argument missing\n"));
	      return (EXIT_FAILURE);
	    } else {
	      if (strlen(argv[num]) >= SH_MAXMSG) 
		{
		  fprintf(stderr, _("ERROR: Command too long: %s\n"), 
			  argv[num]);
		  return (EXIT_FAILURE);
		}
	      strncpy (message, argv[num], SH_MAXMSG);
	      message[SH_MAXMSG-1] = '\0';
	      strncat(message, ":", SH_MAXMSG-strlen(message)-1);
	      flag = 1;
	    }
	    break;

	  default:
	    usage(argv[0]);
	    fprintf(stderr, _("ERROR: unknown option -%c\n"), argv[num][1]);
	    return (EXIT_FAILURE);
	}
      --argc; ++num;
    }

  if (flag == 0)
    {
      usage(argv[0]);
      return (EXIT_FAILURE);
    }
  if (strlen(argv[num]) > (SH_MAXMSG - strlen(message) -1)) 
    {
      fprintf(stderr, _("ERROR: Hostname too long: %s\n"), argv[num]);
      return (EXIT_FAILURE);
    }
  strncat (message, argv[num], SH_MAXMSG -strlen(message) - 1);

  if (NULL == getcwd(clientcd, 256))
    {
      perror(_("getcwd"));
      return (EXIT_FAILURE);
    }
  size = strlen(clientcd) + 1 + strlen(CLIENT) + 6;
  sockname = malloc (size);
  if (!sockname)
    {
      perror(_("main: malloc"));
      return (EXIT_FAILURE);
    }
  sprintf(sockname, _("%s/%s.sock"), clientcd, CLIENT);


  /* Make the socket.
   */
  sock = make_named_socket (sockname);
  if (sock < 0)
    {
      return (EXIT_FAILURE);
    }

  /* Set up termination handler.
   */
  signal (SIGINT,  termination_handler);
  signal (SIGHUP,  termination_handler);
  signal (SIGTERM, termination_handler);
  signal (SIGQUIT, termination_handler);

  /* Send the datagram. 
   */
  status = send_to_server (serversock, message);
  if (status < 0)
    {
      fprintf(stderr, _("Error sending command to server\n"));
      (void) termination_handler(0);
      return (EXIT_FAILURE);
    }

  /* Wait for a reply. 
   */
  if (message[0] == 'L' && message[1] == 'I' &&
      message[2] == 'S' && message[3] == 'T')
    {
      fprintf(stdout, _("# Waiting for listing.\n"));
    }
  else
    {
      fprintf(stdout, _("# Waiting for confirmation.\n"));
    }
  status = recv_from_server (message);

  if (status < 0)
    {
      fprintf(stderr, _("Error receiving data from server\n"));
      (void) termination_handler(0);
      return (EXIT_FAILURE);
    }

  /* Clean up. */
  (void) termination_handler(0);
  return (EXIT_SUCCESS);
}

