# Samizdat session management
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'zlib'
require 'samizdat/engine'

# two different #request definitions: for FCGI and for everything else
#
begin
  raise LoadError if defined?(MOD_RUBY)
  require 'fcgi'
  def request
    FCGI.each_cgi do |cgi|
      catch :finish do
        Session.new(cgi).response {|s| yield s }
      end
    end
  end
rescue LoadError
  def request
    Session.new(CGI.new).response {|s| yield s }
  end
end

# session management and CGI parameter handling
#
# todo: refactor deployment functions into this, move actual Session out into a
# cacheable object
#
class Session < SimpleDelegator
  # wrapper for CGI#cookies: add cookie name prefix, return first value
  #
  def cookie(name)
    @cgi.cookies[@cookie_prefix + name][0]
  end

  # create cookie and add it to the HTTP response
  #
  # cookie name prefix is configured via config.yaml; default expiry timeout
  # is #forever as defined in samizdat.rb
  #
  def set_cookie(name, value=nil, expires=forever)
    options['cookie'] = [] if nil == options['cookie']
    options['cookie'] << CGI::Cookie.new({
      'name' => @cookie_prefix + name,
      'value' => value,
      'expires' => Time.now + expires
    })
  end

  # translate location to a real file name
  #
  def filename(location)
    if defined?(MOD_RUBY)
      Apache.request.lookup_uri(location).filename
    else
      @cgi.env_table['DOCUMENT_ROOT'] + location
    end
  end

  # set language
  #
  def language=(lang)
    lang = config['locale']['languages'][0] if
      lang.nil? or not config['locale']['languages'].include?(lang)
    lang.untaint
    if defined? GetText
      samizdat_bindtextdomain(lang, config['locale']['path'].untaint)
    end
    @language = lang
  end

  # current language
  attr_reader :language

  # set default CGI options (set charset to UTF-8)
  #
  # set id and refresh session if there's a valid session cookie,
  # set credentials to guest otherwise
  #
  def initialize(cgi)
    @cgi = cgi

    # hack to get through to CGI variables
    class << @cgi
      public :env_table
    end
    RequestSingleton.instance.reset
    RequestSingleton.instance.env = @cgi.env_table

    @options = {'charset' => 'utf-8', 'cookie' => []}

    DRb.start_service("druby://localhost:0")
    # fixme: localhost -> config.drb_local

    @cookie_prefix = config['site']['cookie_prefix'] + '_'
    @login_timeout = config['timeout']['login']
    @last_timeout = config['timeout']['last']

    config_lang = config['locale']['languages']
    @accept_language = []   # [[lang, q]...]
    accept = @cgi.env_table['HTTP_ACCEPT_LANGUAGE'] and
      accept.scan(/([^ ,;]+)(?:;q=([^ ,;]+))?/).collect {|l, q|
        [l, (q ? q.to_f : 1.0)]
      }.sort_by {|l, q| -q }.each {|l, q|
        @accept_language.push l if config_lang.include? l
      }
    # lang cookie overrides Accept-Language
    lang = cookie('lang') and config_lang.include? lang and
      @accept_language.unshift lang
    # use first language from config by default
    @accept_language.size == 0 and @accept_language = [config_lang[0]]
    # set interface language
    self.language = @accept_language[0]

    # construct @base
    proto = @cgi.env_table['HTTPS'] ? 'https' : 'http'
    port = @cgi.env_table['SERVER_PORT'].to_i
    port = (port == {'http' => 80, 'https' => 443}[proto]) ? '': ':' + port.to_s
    host = RequestSingleton.instance.host.to_s
    @base = proto + '://' + host + port + uri_prefix + '/'

    @template = Template.new(self)

    # check session
    set_guest_name
    @session = cookie('session')
    if @session and @session != ''
      db.transaction do |db|
        @id, @login, @full_name, @email, login_time = db.select_one 'SELECT id, login, full_name, email, login_time FROM Member WHERE session = ?', @session
        if @id
          if login_time and login_time.to_time < Time.now - @login_timeout
            set_guest_name
            close   # stale session
          else
            # uncomment to regenerate session on each access:
            #@session = generate_session
            #db.do "UPDATE Member SET last_time = current_timestamp, session = ?
            #WHERE id = ?", @session, @id
            set_cookie('session', @session, @last_timeout)
            @moderator = ('yes' == cookie('moderate') and
              config['access']['moderators'].include? @login)
          end
        end
      end
    end
    super @cgi
  end

  # HTTP response options
  attr_reader :options

  # base URI of the site
  attr_reader :base

  # Template object aware of this session's options
  attr_reader :template

  # list of languages in user's order of preference
  attr_reader :accept_language

  attr_reader :id, :login, :full_name, :email

  # true if moderator priviledges are enabled
  attr_reader :moderator

  # set _@login_ and _@full_name_ to 'guest'
  #
  def set_guest_name
    @login = 'guest'
    @full_name = _(@login)
  end

  # return role name for the current user
  # ('moderator', 'member', or 'guest')
  #
  def role
    [ _('moderator'), _('member'), _('guest') ]   # rgettext hack
    if moderator then 'moderator'
    elsif id then 'member'
    else 'guest'
    end
  end

  # check if _action_ is allowed for current user
  #
  def access(action)
    config['access'][action][role()]
  end

  # check if user wants to see hidden messages
  #
  def showhidden?
    'yes' == cookie('showhidden')
  end

  # open new session on login
  #
  # redirect to front page on success
  #
  def open(login, passwd)
    db.transaction do |db|
      @id, = db.select_one 'SELECT id FROM Member m WHERE login = ?
      AND passwd = ?', login, digest(passwd)
      if @id
        @session = generate_session
        db.do "UPDATE Member
                  SET login_time = current_timestamp,
                      last_time = current_timestamp,
                      session = ?
                WHERE id = ?", @session, @id
        db.commit
        set_cookie('session', @session, @last_timeout)
        redirect(base)
      else
        if config.email_enabled?
          # check if member's email isn't confirmed yet
          id, p = db.select_one 'SELECT id, passwd FROM Member
            WHERE login = ?', login
          raise AccountBlockedError if id and p.nil?
        end
        response() do
          template.page(_('Login Failed'),
            '<p>'+_('Wrong login name or password. Try again.')+'</p>')
        end
      end
    end
  end

  # erase session from database
  #
  def close
    db.do "UPDATE Member SET session=NULL WHERE id = ?", @id
    db.commit
    @id = nil
    set_cookie('session', nil, 0)
  end

  # load member preferences by login if not already loaded
  #
  def prefs(login=@login)
    return @prefs unless @prefs.nil?
    prefs, = db.select_one 'SELECT prefs FROM Member WHERE login = ?', login
    @prefs = yaml_hash(prefs)
  end

  # save updated member preferences to database
  #
  # _confirm_ is an optional confirmation hash
  #
  def save_prefs(login=@login, confirm=nil)
    @prefs or raise RuntimeError, 'No preferences to save'
    db.do 'UPDATE Member SET prefs = ?, confirm = ? WHERE login = ?',
      YAML.dump(@prefs), confirm, login
  end

  # return list of values of CGI parameters, tranform empty values to nils
  #
  # unlike CGI#params, Session#params takes array of parameter names
  #
  def params(keys)
    keys.collect do |key|
      value = self[key]
      raise UserError, _('Input size exceeds content size limit') if
        value.methods.include? :size and
        value.size > config['limit']['content']
      case value
      when StringIO, Tempfile
        value = value.read
      end
      (value =~ /[^\s]/)? value : nil
    end
  end

  # always imitate CGI#[] from Ruby 1.8
  #
  def [](key)
    @cgi.params[key][0]
  end

  # plant a fake CGI parameter
  #
  def []=(key, value)
    @cgi.params[key] = [value]
  end

  # print header and optionally content, then clean-up and exit
  #
  # generate error page on RuntumeError exceptions
  #
  def response(options={})
    @options.update(options)
    if block_given?
      page =
      begin
        yield self
      rescue AuthError
        @options['status'] = 'AUTH_REQUIRED'
        template.page(_('Access Denied'), %{<p>#{$!}.</p>})
      rescue AccountBlockedError
        template.page(_('Account Is Blocked'),
'<p>'+_('Your account is blocked until the email address you have specified is confirmed. Confirmation message with instructions was sent to that address.')+'</p>')
      rescue UserError
        template.page(_('User Error'),
%{<p>#{$!}.</p><p>}+_("Press 'Back' button of your browser to return.")+'</p>')
      rescue ResourceNotFoundError
        @options['status'] = 'NOT_FOUND'
        referer = ' ('+_('looks like it was')+%{ <a href="#{@cgi.referer}">#{@cgi.referer}</a>)} if @cgi.referer
        template.page(_('Resource Not Found'),
'<p>'+_('The resource you requested was not found on this site. Please report this error back to the site you came from')+referer.to_s+'.</p>')
      rescue RuntimeError
        template.page(_('Runtime Error'),
'<p>'+_('Runtime error has occured:')+%{ #{CGI.escapeHTML($!.to_s)}.</p>
<pre>#{CGI.escapeHTML(caller.join("\n"))}</pre>
<p>}+_('Please report this error to the site administrator.')+'</p>')
      end
      page = compress(page)   # gzip and etag
      @cgi.out(@options) { page }
    else
      print @cgi.header(@options)
    end
    DRb.stop_service
    if defined? FCGI
      throw :finish
    else
      exit
    end
  end

  def redirect(location=referer)
    location = base if location.nil?
    response({'status' => 'REDIRECT', 'location' => location})
  end

private

  # time-salted session hash
  #
  def generate_session
    digest(@id.to_s + Time.now.to_s)
  end

  # see #compress
  #
  MIN_GZ_SIZE = 1024

  # do gzip compression and check ETag when supported by client
  #
  def compress(body)
    if body.length > MIN_GZ_SIZE and @cgi.env_table.has_key?('HTTP_ACCEPT_ENCODING')
      enc =
        case @cgi.env_table['HTTP_ACCEPT_ENCODING']
        when /x-gzip/ then 'x-gzip'
        when /gzip/   then 'gzip'
        end
      unless enc.nil?
        io = ''   # primitive StringIO replacement
        class << io
          def write(str)
            self << str
          end
        end
        gz = Zlib::GzipWriter.new(io)
        gz.write(body)
        gz.close
        body = io
        @options['Content-Encoding'] = enc
        @options['Vary'] = 'Accept-Encoding'
      end
    end

    # check ETag
    if @cgi.env_table.has_key?('HTTP_IF_NONE_MATCH')
      etag = '"' + digest(body) + '"'
      @options['ETag'] = etag
      catch(:ETagFound) do
        @cgi.env_table['HTTP_IF_NONE_MATCH'].each(',') do |e|
          if etag == e.strip.delete(',')
            @options['status'] = 'NOT_MODIFIED'
            body = ''   # don't send body
            throw :ETagFound
          end
        end
      end
    end

    body
  end
end
