   /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@t-online.de)

   This file was initially copied from the hp3300 backend.

   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RTS88xx chipset,
    such as HP4400C, HP4470C, HP3500C, 3530C, and HP ScanJet 3570C.
    Parts of this source were inspired by other backends.

		History:

		Version 0.18  21.11.04 13.alpha,
				- source sorted,
				- now only SANEI_USB_SUPPORT for a better overview(xfermodules removed)
				- read and verify the MainBoardID 
		Version 0.17p 02.11.04 12.alpha, source sorted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17p 02.11.04 12.alpha, source sourted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
		Version 0.17  09.03.04 9. alpha, HP3500 included
		Version 0.16  06.02.04 8. alpha, wait counting on LCD
		Version 0.15a 29.01.04 7. alpha, CCD switch moved to config file
		Version 0.15  11.01.04 6. alpha, a second CCD implemented
		Version 0.13a 21.11.04 4. alpha, an little fix included
		Version 0.12  22.10.03 third alpha, Backend name changed to HP_RTS88xx
		Version 0.11  30.08.03 second alpha
		Version 0.10  19.07.03 first alpha
*/

/*
		enable DEDUG output:
		use a nomal shell (konsole) and type
		export SANE_DEBUG_HP_RTS88XX=32
		xsane
*/

#ifndef BACKEND_NAME
#define BACKEND_NAME hp_rts88xx
#endif
#ifndef BUILD
#define BUILD 18
#endif

/*#define STANDALONE*/
/*#define ENABLE_VI8920*/

/* definitions for debug */
#define DEBUG								/* debug output */
/*#define DEBUG_HP3500			* debug switch to test the 3500 code under hp4470c */
/*#define STOP_HP3500_1*/

/*#define DEBUG_SCAN					* debug output about getting the immage datas */
/*#define DEBUG_L							* don't wait for lamp warm up */
/*#define USBDEBUG					* debug output about the USB in/out */
/*#define DEBUG_FILE				* additional write into a pnm file (not allowed as driver)*/

#include <stdlib.h>					/* malloc, free */
#include <string.h>					/* memcpy */
#include <stdio.h>
#include "../include/sane/sane.h"
#include "../include/sane/config.h"
#include "../include/sane/sanei.h"
#include "../include/sane/sanei_config.h"
#include "../include/sane/saneopts.h"

#ifdef STANDALONE
	#define V_MAJOR 1
	#define V_MINOR 15
	#define PATH_MAX 128
	#define DBG          fprintf
	#define DBG_MSG      stdout
	#define DBG_USB      stdout
	#define DBG_OPT      stdout
	#define DBG_SCAN     stdout
	#define DBG_ASSERT   stdout
	#define DBG_ERR      stdout
#else
#include "../include/sane/sanei_backend.h"
	#define DBG_ASSERT  1
	#define DBG_ERR     2
	#define DBG_MSG     16
	#define DBG_SCAN    32
	#define DBG_OPT     64
	#define DBG_USB     128
#endif

#define HP_RTS_CONFIG_FILE "hp_rts88xx.conf"

#include "hp_rts88xx.h"
#include "hp_rts_44x0c.h"
#include "hp_rts_35x0c.h"

/* (source) includes for data transfer methods */
#include "hp_rts_xfer.c"
/* (source) includes for special scanner code */
#include "hp_rts_44x0c.c"
#include "hp_rts_35x0c.c"


/****************************************************************************/
SANE_Bool _InitScan(TScanParams *pParams, THWParams *pHWParams, TDataPipe *pDataPipe)
/****************************************************************************/
{
	SANE_Int	iHandle;
	SANE_Bool result;
	iHandle = pHWParams->iXferHandle;

	result = SANE_FALSE;
	/* check validity of scanparameters */
	switch (pParams->iDpi) {
		case 150:
		case 200:
		case 300:
		case 600:
		break;
	default:
		DBG(DBG_ERR, "InitScan: Invalid dpi (%d)\n", pParams->iDpi);
		return SANE_FALSE;
	}
	switch (pParams->iLpi) {
		case 150:
		case 200:
		case 300:
		case 600:
		break;
	default:
		DBG(DBG_ERR, "InitScan: Invalid lpi (%d)\n", pParams->iLpi);
		return SANE_FALSE;
	}
	/*   *** Done checking scan parameters validity ***  */
	switch (pHWParams->ScannerModel) {
		case eHp3500c: case eHp3530c:
#ifdef ENABLE_VI8920
	  case eVi8920:
#endif
			result = Hp35x0c_init_scan(pHWParams,pParams,pDataPipe);
				if (result)
					DBG(DBG_ERR, "InitScan: OK\n");
				else
					DBG(DBG_ERR, "InitScan: ERROR from Hp35x0_init_scan\n");
			break;
		case eHp4400c: case eHp4470c:
			if (Hp44x0_cal_scanner(pHWParams,pParams) == SANE_TRUE){
				result = Hp44x0_init_scan(pHWParams,pParams,pDataPipe);
				if (result)
					DBG(DBG_ERR, "InitScan: OK\n");
				else
					DBG(DBG_ERR, "InitScan: ERROR from Hp44x0_init_scan\n");
			}
			break;
		case eUnknownModel:
		default:
			DBG(DBG_ERR, "InitScan: ERROR: unknown model! (%d)\n",
					(SANE_Int)pHWParams->ScannerModel);
			result = SANE_FALSE;
			break;
  }
	return result;
}

/****************************************************************************
	_FinishScan
	=================
	Finishes the scan. Makes the scanner head move back to the home position.
****************************************************************************/
SANE_Bool _FinishScan( SANE_Handle h )
{
	TScanner *s;
	THWParams *pHWParams;
	TScanParams *pParams;
	SANE_Int iHandle;
	SANE_Word size;

	s = (TScanner *)h;

	iHandle		= s->HWParams.iXferHandle;
	pHWParams	= &s->HWParams;
	pParams		= &s->ScanParams;

	DBG(DBG_MSG,"Finish scan:\n");
  /* check, if the scanner moving */
	if (Hp_rts_is_moving(iHandle)){
		DBG(DBG_MSG,"Finish scan, scanner is moving, stop it!!!\n");
		Hp_rts_stop_moving(iHandle);
	};
	usleep(100);
	
	/* clean the Scanner Buffer */
	do{
		/*if ( !*/(Hp_rts_data_ready(iHandle,&size));/*) return SANE_FALSE;*/
		if ( size > 0)
			if ( !(Hp_rts_read_data(iHandle,size,s->DataPipe.pabXferBuf))) return SANE_FALSE;
	} while (size != 0);

	DBG(DBG_MSG,"Finish scan, park to home\n");
	switch (pHWParams->ScannerModel) {
		case eHp3500c: case eHp3530c: case eHp3570c:
#ifdef ENABLE_VI8920
	  case eVi8920:
#endif
			Hp35x0c_rewind(pHWParams);
			break;
		case eHp4400c: case eHp4470c:
			Hp44x0_park_to_home(pHWParams,pParams);
			break;
		default:
			DBG(DBG_ERR, "FinishScan: ERROR: unknown model! (%d) %s\n",
					(SANE_Int)pHWParams->ScannerModel, ScannerModels[(SANE_Int)pHWParams->ScannerModel].pszName);
		return SANE_FALSE;
  }
	return SANE_TRUE;
}


/****************************************************************************/
static SANE_Int
_max_string_size (const SANE_String_Const strings[])
/****************************************************************************/
{
	SANE_Int size, max_size = 0;
	SANE_Int i;

	for (i = 0; strings[i]; ++i) {
		size = strlen (strings[i]) + 1;
		if (size > max_size)
			max_size = size;
	}
	return max_size;
}


/****************************************************************************/
SANE_Status _Init_Interface(THWParams *pHWParams, TScanParams *pParams)
/****************************************************************************/
{

	DBG(DBG_SCAN, "Init_Interface....\n");
	/* autodetect some hardware properties */
	if (Hp_rts_ProbeRegisters(pHWParams) != SANE_STATUS_GOOD) {
		DBG(DBG_ERR, "Init_Interface: Hp_rts_ProbeRegisters failed!\n");
		return SANE_STATUS_IO_ERROR;
	}
	/* default HW params */
	pHWParams->iTopLeftX      = 0;
	pHWParams->iTopLeftY      = 3;
	pHWParams->iReversedHead  = SANE_FALSE;
	DBG(DBG_MSG, "Init_Interface: Use scanner (%d) %s\n",
 		(SANE_Int)pHWParams->ScannerModel,ScannerModels[pHWParams->ScannerModel].pszName);
	switch (pHWParams->ScannerModel) {

		case eHp3500c: case eHp3530c:
#ifdef ENABLE_VI8920
		case eVi8920:
#endif
			pHWParams->iTopLeftX    = 7;
#ifndef DEBUG_HP3500
			Hp35x0c_init_power_on(pHWParams);
#else
			if ( !(Hp44x0_Wakeup(pHWParams, pParams))) return SANE_STATUS_IO_ERROR;
			if ( !(Hp44x0_cal_scanner(pHWParams, pParams))) return SANE_STATUS_IO_ERROR;
#endif
			break;
		case eHp4400c: case eHp4470c:
			pHWParams->iTopLeftX    = 7;
			if ( !(Hp44x0_Wakeup(pHWParams, pParams))) return SANE_STATUS_IO_ERROR;
			if ( !(Hp44x0_cal_scanner(pHWParams, pParams))) return SANE_STATUS_IO_ERROR;
			if ( !(Hp44x0_park_to_home(pHWParams, pParams))) return SANE_STATUS_IO_ERROR;
			break;
		case eUnknownModel:
		default:
			DBG(DBG_ERR, "Init_Interface: ERROR: internal error! (%d)\n",
		 		(SANE_Int)pHWParams->ScannerModel);
			return SANE_STATUS_UNSUPPORTED;
	}
	return SANE_STATUS_GOOD;
}


/****************************************************************************/
void _Close_Interface(THWParams *pHWParams)
/****************************************************************************/
{
	DBG(DBG_MSG, "Close_Interface: ..\n");
	switch (pHWParams->ScannerModel) {
		case eHp3500c: case eHp3530c:
#ifdef ENABLE_VI8920
	  case eVi8920:
#endif
			Hp35x0c_turn_off_lamp(pHWParams->iXferHandle);
			Hp35x0c_set_powersave_mode(pHWParams->iXferHandle,1);
			break;
		case eHp4400c: case eHp4470c:
			/* turn of scanner lamp */
			Hp44x0_SetLamp(pHWParams->iXferHandle, SANE_FALSE);
			Hp44x0_Down(pHWParams->iXferHandle);
			break;
		case eUnknownModel:
		default:
			DBG(DBG_ERR, "Close_Interface: ERROR: internal error! (%d)\n", (SANE_Int)pHWParams->ScannerModel);
			return;
	}
	Hp_rts_XferExit(pHWParams->iXferHandle);
	pHWParams->iXferHandle = 0;
}

/****************************************************************************/
static void SANE_InitOptions(TScanner *s)
/****************************************************************************/
{
	SANE_Int i, Model;
	SANE_Option_Descriptor *pDesc;
	TOptionValue *pVal;

	Model = s->HWParams.ScannerModel;
	DBG(DBG_SCAN, "SANE_InitOptions: Use model # %d\n", Model);
	/* set a neutral gamma */
	if( s->aGammaTableR == NULL )   /* Not yet allocated */
	{
		s->aGammaTableR = malloc( NUM_GAMMA_ENTRIES * sizeof( SANE_Int ) );
		s->aGammaTableG = malloc( NUM_GAMMA_ENTRIES * sizeof( SANE_Int ) );
		s->aGammaTableB = malloc( NUM_GAMMA_ENTRIES * sizeof( SANE_Int ) );

		for (i = 0; i < NUM_GAMMA_ENTRIES; i++) {
			s->aGammaTableR[i] = i;
			s->aGammaTableG[i] = i;
			s->aGammaTableB[i] = i;
		}
	}

	if ( (Model==eHp4400c) || (Model==eHp4470c) ){
		DBG(DBG_MSG,"SANE_InitOptions: use HP4x00 mode\n");
/*		mode_list[0]="B/W";*/
		mode_list[0]="Gray";
		mode_list[1]=NULL;
		if (OPT_COLOR){
			mode_list[1]="Color";
			mode_list[2]=NULL;
		}
	}else
	{ /* hp35x0 */
		DBG(DBG_MSG,"SANE_InitOptions: use HP35x0 mode\n");
		/*	mode_list[0]="B/W";*/
		mode_list[0]="Gray"; /**/
		mode_list[1]=NULL;
		if (OPT_COLOR){
			mode_list[1]="Color";
			mode_list[2]=NULL;
		}
	}

	ccd_list[0]="CCDType 0";
	ccd_list[1]="CCDType 1";
	ccd_list[2]=NULL;

	scan_option_list[0]="bed light";
	scan_option_list[1]="XPA";
	scan_option_list[2]=NULL;

	for (i = optCount; i < optLast; i++) {

		pDesc = &s->aOptions[i];
		pVal = &s->aValues[i];

		/* defaults */
		pDesc->name   = "";
		pDesc->title  = "";
		pDesc->desc   = "";
		pDesc->type   = SANE_TYPE_INT;
		pDesc->unit   = SANE_UNIT_NONE;
		pDesc->size   = sizeof(SANE_Word);
		pDesc->constraint_type = SANE_CONSTRAINT_NONE;
		pDesc->cap    = 0;

		switch (i) {

			case optCount:
				pDesc->title  = SANE_TITLE_NUM_OPTIONS;
				pDesc->desc   = SANE_DESC_NUM_OPTIONS;
				pDesc->cap    = SANE_CAP_SOFT_DETECT;
				pVal->w       = (SANE_Word)optLast;
			break;

			case optGroupGeometry:
				pDesc->title  = "Geometry";
				pDesc->type   = SANE_TYPE_GROUP;
				pDesc->size   = 0;
			break;

			case optTLX:
				pDesc->name   = SANE_NAME_SCAN_TL_X;
				pDesc->title  = SANE_TITLE_SCAN_TL_X;
				pDesc->desc   = SANE_DESC_SCAN_TL_X;
				pDesc->unit   = SANE_UNIT_MM;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pDesc->constraint.range = &rangeXmm;
				pVal->w       = rangeXmm.min;
			break;

			case optTLY:
				pDesc->name   = SANE_NAME_SCAN_TL_Y;
				pDesc->title  = SANE_TITLE_SCAN_TL_Y;
				pDesc->desc   = SANE_DESC_SCAN_TL_Y;
				pDesc->unit   = SANE_UNIT_MM;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeYmm;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->w       = rangeYmm.min;
			break;

			case optBRX:
				pDesc->name   = SANE_NAME_SCAN_BR_X;
				pDesc->title  = SANE_TITLE_SCAN_BR_X;
				pDesc->desc   = SANE_DESC_SCAN_BR_X;
				pDesc->unit   = SANE_UNIT_MM;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeXmm;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->w       = rangeXmm.max;
			break;

			case optBRY:
				pDesc->name   = SANE_NAME_SCAN_BR_Y;
				pDesc->title  = SANE_TITLE_SCAN_BR_Y;
				pDesc->desc   = SANE_DESC_SCAN_BR_Y;
				pDesc->unit   = SANE_UNIT_MM;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeYmm;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->w       = rangeYmm.max;
			break;

			case optDPI:
				pDesc->name   = SANE_NAME_SCAN_RESOLUTION;
				pDesc->title  = SANE_TITLE_SCAN_RESOLUTION;
				pDesc->desc   = SANE_DESC_SCAN_RESOLUTION;
				pDesc->unit   = SANE_UNIT_DPI;
				pDesc->constraint_type  = SANE_CONSTRAINT_WORD_LIST;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				if ((Model==eHp4400c) || (Model==eHp4470c) ){
					if (OPT_T_RES) {
						DBG(DBG_MSG,"SANE_InitOptions: use HP4x00 TEST mode \n");
						pDesc->constraint.word_list = Hp44x0c_setResolutions_t;
						pVal->w       = Hp44x0c_setResolutions_t[1];
					} else {
						DBG(DBG_MSG,"SANE_InitOptions: use HP4x00 normal mode \n");
						pDesc->constraint.word_list = Hp44x0c_setResolutions;
						pVal->w       = Hp44x0c_setResolutions[1];
					}
				}else{ /* it's a HP35xx */
					if (OPT_T_RES) {
						DBG(DBG_MSG,"SANE_InitOptions: use HP35x0 TEST mode \n");
						pDesc->constraint.word_list = Hp35x0c_setResolutions_t;
						pVal->w       = Hp35x0c_setResolutions_t[1];
					} else {
						DBG(DBG_MSG,"SANE_InitOptions: use HP35x0 normal mode \n");
						pDesc->constraint.word_list = Hp35x0c_setResolutions;
						pVal->w       = Hp35x0c_setResolutions[1];
				}
				}
			break;

			case optGroupImage:
				pDesc->title  = SANE_I18N("Image");
				pDesc->type   = SANE_TYPE_GROUP;
				pDesc->size   = 0;
			break;

			case optGammaTableRed:
				pDesc->name   = SANE_NAME_GAMMA_VECTOR_R;
				pDesc->title  = SANE_TITLE_GAMMA_VECTOR_R;
				pDesc->desc   = SANE_DESC_GAMMA_VECTOR_R;
				pDesc->size   = NUM_GAMMA_ENTRIES * sizeof( SANE_Int );
				pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeGammaTable;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->wa      = s->aGammaTableR;
			break;

			case optGammaTableGreen:
				pDesc->name   = SANE_NAME_GAMMA_VECTOR_G;
				pDesc->title  = SANE_TITLE_GAMMA_VECTOR_G;
				pDesc->desc   = SANE_DESC_GAMMA_VECTOR_G;
				pDesc->size   = NUM_GAMMA_ENTRIES * sizeof( SANE_Int );
				pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeGammaTable;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->wa      = s->aGammaTableG;
			break;

			case optGammaTableBlue:
				pDesc->name   = SANE_NAME_GAMMA_VECTOR_B;
				pDesc->title  = SANE_TITLE_GAMMA_VECTOR_B;
				pDesc->desc   = SANE_DESC_GAMMA_VECTOR_B;
				pDesc->size   = NUM_GAMMA_ENTRIES * sizeof( SANE_Int );
				pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeGammaTable;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->wa      = s->aGammaTableB;
			break;

			case optGroupMisc:
				pDesc->title = SANE_I18N ("Miscellaneous");
				pDesc->type  = SANE_TYPE_GROUP;
				pDesc->size  = 0;
			break;

			case optBrightness:
				pDesc->name   = SANE_NAME_BRIGHTNESS;
				pDesc->title  = SANE_TITLE_BRIGHTNESS;
				pDesc->desc   = SANE_DESC_BRIGHTNESS;
				pDesc->type   = SANE_TYPE_INT;
				pDesc->size   = sizeof(SANE_Int);
				pDesc->unit   = SANE_UNIT_NONE;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeBrightness;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->w       = 10;
			break;

			case OPT_MODE:
				pDesc->name   = SANE_NAME_SCAN_MODE;
				pDesc->title  = SANE_TITLE_SCAN_MODE;
				pDesc->desc   = SANE_DESC_SCAN_MODE;
				pDesc->type   = SANE_TYPE_STRING;
				pDesc->constraint_type  = SANE_CONSTRAINT_STRING_LIST;
				pDesc->size   = _max_string_size (mode_list);
				pDesc->constraint.string_list = mode_list;
				pVal->s       = strdup (mode_list[0]);
				pDesc->cap    = ~SANE_CAP_INACTIVE;
			break;

			case optXPDA:
				pDesc->name   = SANE_NAME_SCAN_SOURCE;
				pDesc->title  = SANE_TITLE_SCAN_SOURCE;
				pDesc->desc   = SANE_I18N("Switches the scaner source mode (normal/xPDA).");
				pDesc->type   = SANE_TYPE_STRING;
				pDesc->constraint_type  = SANE_CONSTRAINT_STRING_LIST;
				pDesc->size   = _max_string_size (scan_option_list);
				pDesc->constraint.string_list = scan_option_list;
				pVal->s       = strdup (scan_option_list[0]);
				if (Model == eHp4470c)
					pDesc->cap    = ~SANE_CAP_INACTIVE;
				else
					pDesc->cap    = SANE_CAP_INACTIVE;
				s->ScanParams.optXPA = SANE_FALSE;
			break;

			case optCCD:
				pDesc->name   = SANE_NAME_SCAN_SOURCE;
				pDesc->title  = "CCD_Type";
				pDesc->desc   = SANE_I18N("Switches the scaner CCD mode (0/1).");
				pDesc->type   = SANE_TYPE_STRING;
				pDesc->constraint_type  = SANE_CONSTRAINT_STRING_LIST;
				pDesc->size   = _max_string_size (ccd_list);
				pDesc->constraint.string_list = ccd_list;
				pVal->s       = strdup (ccd_list[0]);
				if ((Model==eHp4400c) || (Model==eHp4470c) ){
					if (OPT_CCD){
						pDesc->cap    = ~SANE_CAP_INACTIVE;
						s->ScanParams.oCCD_Type = 0;
					} else
						pDesc->cap    = SANE_CAP_INACTIVE;
				}else
					pDesc->cap    = SANE_CAP_INACTIVE;
			break;

			case optGainR:
				pDesc->name   = "optGainR";
				pDesc->title  = "optGainR";
				pDesc->desc   = SANE_I18N("optGainR");
				pDesc->type   = SANE_TYPE_INT;
				pDesc->size   = sizeof(SANE_Int);
				pDesc->unit   = SANE_UNIT_NONE;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeBrightness;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->w       = 0x0a;
			break;
			case optGainG:
				pDesc->name   = "optGainG";
				pDesc->title  = "optGainG";
				pDesc->desc   = SANE_I18N("optGainG");
				pDesc->type   = SANE_TYPE_INT;
				pDesc->size   = sizeof(SANE_Int);
				pDesc->unit   = SANE_UNIT_NONE;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeBrightness;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->w       = 0x0a;
			break;
			case optGainB:
				pDesc->name   = "optGainB";
				pDesc->title  = "optGainB";
				pDesc->desc   = SANE_I18N("optGainB");
				pDesc->type   = SANE_TYPE_INT;
				pDesc->size   = sizeof(SANE_Int);
				pDesc->unit   = SANE_UNIT_NONE;
				pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
				pDesc->constraint.range = &rangeBrightness;
				pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
				pVal->w       = 0x0a;
			break;

			default:
				DBG(DBG_ERR, "InitOptions: Uninitialised option %d\n", i);
			break;
		}
	}
}

/****************************************************************************/
SANE_Status
sane_init(SANE_Int *piVersion, SANE_Auth_Callback pfnAuth)
/****************************************************************************/
{
	SANE_Char line[PATH_MAX];
	SANE_Char *word;
	SANE_String_Const cp;
	SANE_Int linenumber;
	FILE *conf_fp;
	TScanner *s;

/* prevent compiler from complaing about unused parameters */
	pfnAuth = pfnAuth;
	s = malloc (sizeof (TScanner));
	if (!s)
	{
		DBG (DBG_MSG,"sane_init : malloc failed\n");
		return SANE_STATUS_NO_MEM;
	}

#ifndef STANDALONE
	DBG_INIT();
	DBG (DBG_MSG, "SANE HP_RTS88XX USB backend version %d.%d build %d from %s\n", V_MAJOR,
		V_MINOR, BUILD, PACKAGE_STRING);
#else
	DBG (DBG_MSG, "SANE HP_RTS88XX USB backend version %d.%d build %d from \n", V_MAJOR,
		V_MINOR, BUILD);
#endif

	iNumSaneDev = 0;
	_pFirstSaneDev = 0;

	/* initialise scanner object */
	DBG(DBG_OPT, "Init_Interface: call Hp_rts_XferInit\n");
	if ( Hp_rts_XferInit() != SANE_TRUE) {
		DBG(DBG_ERR, "Init_Interface: Hp_rts_XferInit failed\n");
		/* free scanner object memory */
		free ((void *) s);
		return SANE_STATUS_IO_ERROR;
	}
	/* free scanner object memory */
	free ((void *) s);
	DBG(DBG_OPT, "sane_init: call sanei_config_open\n");
	conf_fp = sanei_config_open (HP_RTS_CONFIG_FILE);
	if (conf_fp) {
		linenumber = 0;
		DBG (DBG_OPT, "sane_init: reading config file '%s'\n",HP_RTS_CONFIG_FILE);
		while (sanei_config_read (line, sizeof (line), conf_fp)) {
			word = 0;
			linenumber++;
			cp = sanei_config_get_string (line, &word);
			/* Discards white lines and comments*/
			if (!word || cp == line) {
				if (word)
					free (word);
				continue;
			}

			if (word[0] == '#') {
					free (word);
					continue;
			}
			if (strcmp (word, "option") == 0) {
				free (word);
				word = 0;
				cp = sanei_config_get_string (cp, &word);
				if (!word) {
					DBG (DBG_ERR,
						"sane_init: config file line %d: missing quotation mark?\n",
						linenumber);
					continue;
				}
				if (strcmp (word, "CCD0") == 0) {
					DBG(DBG_OPT,"sane_init: Find option '%s'\n",line);
					free (word);
					word = 0;
					OPT_CCD = SANE_FALSE;
					s->ScanParams.oCCD_Type = 0;
					continue;
				}
				if (strcmp (word, "CCD1") == 0) {
					DBG(DBG_OPT,"sane_init: Find option '%s'\n",line);
					free (word);
					word = 0;
					OPT_CCD = SANE_FALSE;
					s->ScanParams.oCCD_Type = 1;
					continue;
				}
				if (strcmp (word, "ENABLE_COLOR") == 0) {
					DBG(DBG_OPT,"sane_init: Find option '%s'\n",line);
					free (word);
					word = 0;
					OPT_COLOR = SANE_TRUE;
					continue;
				}
				if (strcmp (word, "ENABLE_T_RES") == 0) {
					DBG(DBG_OPT,"sane_init: Find option '%s'\n",line);
					free (word);
					word = 0;
					OPT_T_RES = SANE_TRUE;
					continue;
				}
			}
		}  /*while*/
		fclose(conf_fp);
	}
	else {
		DBG (DBG_ERR, "sane_init: Unable to read config file \"%s\": %s\n",
		HP_RTS_CONFIG_FILE,strerror(errno));
		DBG (DBG_OPT, "sane_init: Using default built-in values\n");
	}
	if (piVersion != NULL) {
		*piVersion = SANE_VERSION_CODE (V_MAJOR, V_MINOR, BUILD);
	}
	return SANE_STATUS_GOOD;
}


/****************************************************************************/
void
sane_exit(void)
/****************************************************************************/
{
	TDevListEntry *pDev, *pNext;

	DBG(DBG_MSG, "sane_exit\n");

	/* free device list memory */
	if (_pSaneDevList)
	{
		for (pDev = _pFirstSaneDev; pDev; pDev = pNext)
		{
			pNext = pDev->pNext;
			free ((void *) pDev->dev.name);
			free (pDev);
		}
		_pFirstSaneDev = 0;
		free (_pSaneDevList);
		_pSaneDevList = 0;
	}
}

/****************************************************************************/
SANE_Status
sane_get_devices (const SANE_Device *** device_list, SANE_Bool local_only)
/****************************************************************************/
{
	TDevListEntry *pDev;
	SANE_Int i;
#if 0
	DBG (DBG_MSG, "sane_get_devices\n");
#endif
	local_only = local_only;

	if (_pSaneDevList)
	{
		free (_pSaneDevList);
	}

	_pSaneDevList = malloc (sizeof (*_pSaneDevList) * (iNumSaneDev + 1));
	if (!_pSaneDevList)
	{
		DBG (DBG_MSG, "sane_get_devices : no mem\n");
		return SANE_STATUS_NO_MEM;
	}
	i = 0;
	for (pDev = _pFirstSaneDev; pDev; pDev = pDev->pNext)
	{
		_pSaneDevList[i++] = &pDev->dev;
		DBG (DBG_MSG, "sane_get_devices: name %s\n",pDev->dev.name);
	}
	_pSaneDevList[i++] = 0;	/* last entry is 0 */

	 *device_list = _pSaneDevList;

	return SANE_STATUS_GOOD;
}

/****************************************************************************/
SANE_Status
sane_open(SANE_String_Const name, SANE_Handle *h)
/****************************************************************************/
{
	TScanner *s;

	DBG(DBG_MSG, "sane_open: %s\n", name);

	s = malloc (sizeof (TScanner));
	if (!s)
	{
		DBG (DBG_MSG,"sane_open : malloc failed\n");
		return SANE_STATUS_NO_MEM;
	}
	*h = s;

	s->HWParams.iXferHandle = Hp_rts_XferOpen (name,(EScannerModel *) &s->HWParams.ScannerModel);
#if 0
	DBG (DBG_MSG, "sane_open: s->HWParams.ScannerModel = %d\n",s->HWParams.ScannerModel);
#endif
	if (s->HWParams.iXferHandle < 0)
	{
		DBG (DBG_ERR, "sane_open: Hp_rts_XferOpen failed for '%s'\n", name);
		return SANE_STATUS_DEVICE_BUSY;
	}

	/* initialise scanner object */
#if 1 /* only for test, normaly 1 */
	s->HWParams.lamp_warm = SANE_FALSE;
#else
	s->HWParams.lamp_warm = SANE_TRUE; /*only for test */
#endif
	s->fScanning = SANE_FALSE;
  s->fCanceled = SANE_FALSE;

	s->aGammaTableR = NULL;
	s->aGammaTableG = NULL;
	s->aGammaTableB = NULL;

	if (_Init_Interface(&s->HWParams,&s->ScanParams) == SANE_STATUS_GOOD) {
		SANE_InitOptions((TScanner *)s);
	}
	else{
		DBG (DBG_ERR, "sane_open: _Init_Interface failed\n");
		s->HWParams.iXferHandle = -1;
		return SANE_STATUS_IO_ERROR;
	}
	return SANE_STATUS_GOOD;
}


/****************************************************************************/
void
sane_close(SANE_Handle h)
/****************************************************************************/
{
	TScanner *s;

	DBG(DBG_MSG, "sane_close\n");
	s = (TScanner *)h;
	/* close scanner */
	_Close_Interface(&s->HWParams);
	#ifdef DEBUG_FILE
	if ( s->ScanParams.DebugOpen ){
		fclose(s->ScanParams.FD_r);
		s->ScanParams.DebugOpen = SANE_FALSE;
	}
	#endif
	if ( s->aGammaTableR != NULL ){
		free(s->aGammaTableR);
		s->aGammaTableR = NULL;}
	if ( s->aGammaTableG != NULL ){
		free(s->aGammaTableG);
		s->aGammaTableG = NULL;}
	if ( s->aGammaTableB != NULL ){
		free(s->aGammaTableB);
		s->aGammaTableB = NULL;}

	/* free scanner object memory */
	free ((void *) s);
}


/****************************************************************************/
const SANE_Option_Descriptor *
sane_get_option_descriptor(SANE_Handle h, SANE_Int n)
/****************************************************************************/
{
	TScanner *s;

	DBG(DBG_OPT, "sane_get_option_descriptor %d\n", n);
	if ((n < optCount) || (n >= optLast)) {
		return NULL;
	}
	s = (TScanner *)h;
	return &s->aOptions[n];
}


/****************************************************************************/
SANE_Status
sane_control_option(SANE_Handle h, SANE_Int option, SANE_Action Action, void *pVal,
										SANE_Int *pInfo)
/****************************************************************************/
{
	TScanner    *s;
	SANE_Int    info;
	SANE_Status status;

	DBG(DBG_OPT,"sane_control_option: option %d, action %d\n", option, Action);
	s = (TScanner *)h;
	info = 0;

	if (option < 0 || option >= optLast)
		return SANE_STATUS_INVAL;

	if (pInfo != NULL)
		*pInfo = 0;

	/*************************************************/
	switch (Action) {
	/*************************************************/
		case SANE_ACTION_GET_VALUE:
			switch (option) {

				/* Get options of type SANE_Word */
				case optCount: case optDPI:
				case optTLX: case optTLY: case optBRX: case optBRY:
				case optBrightness: case optGainR: case optGainG: case optGainB:
					DBG(DBG_OPT,"sane_control_option: SANE_ACTION_GET_VALUE %d = %d\n", option,
							(SANE_Int)s->aValues[option].w);
					*(SANE_Word*)pVal = s->aValues[option].w;
				break;

				/* Get options of type SANE_Word array */
				case optGammaTableRed: case optGammaTableGreen: case optGammaTableBlue:
					DBG(DBG_OPT, "Reading gamma table\n");
					memcpy (pVal, s->aValues[option].wa, s->aOptions[option].size);
				break;

				/* Get options of type SANE_TYPE STRING */
				case OPT_MODE: case optXPDA: case optCCD:
					DBG(DBG_OPT,"sane_control_option: SANE_ACTION_GET_VALUE %s\n",s->aValues[option].s);
					status = sanei_constrain_value (s->aOptions + option, s->aValues[option].s,
										(SANE_Int *)info);
					strcpy ((char *)pVal, s->aValues[option].s);
					return SANE_STATUS_GOOD;
				break;

				default: ;
					DBG(DBG_ERR,"SANE_ACTION_GET_VALUE: Invalid option (%d)\n", option);
				break;
			}
		break;
		case SANE_ACTION_SET_VALUE:
			/* no changes while in mid-pass! */
			if (s->fScanning){
				DBG(DBG_ERR,"sane_control_option: SANE_ACTION_SET_VALUE not allowed during scan\n");
				return SANE_STATUS_DEVICE_BUSY;
			}
			status = sanei_constrain_value (s->aOptions + option, pVal, pInfo);
			if (status != SANE_STATUS_GOOD)
				return status;
			switch (option) {

				/* Set options of type SANE_Word */
				case optCount:
					return SANE_STATUS_INVAL;
				break;

				case optDPI: case optTLX: case optTLY: case optBRX: case optBRY:
				case optBrightness: case optGainR: case optGainG: case optGainB:
					DBG(DBG_OPT,"sane_control_option: SANE_ACTION_SET_VALUE %d to %d\n",
						option,*(SANE_Word *) pVal);
					info |= SANE_INFO_RELOAD_PARAMS;
					s->aValues[option].w = *(SANE_Word *) pVal;
				break;

				/* Get options of type SANE_Word array */
				case optGammaTableRed: case optGammaTableGreen: case optGammaTableBlue:
					DBG(DBG_OPT, "Writing gamma table\n");
					memcpy (s->aValues[option].wa, pVal, s->aOptions[option].size);
				break;

				/* Set options of type SANE_TYPE_STRING */
				case OPT_MODE: case optXPDA: case optCCD:
					DBG(DBG_OPT,"sane_control_option: SANE_ACTION_SET_String %d to %s\n",
							option,(char *)pVal);
					if (s->aValues[option].s)
						free (s->aValues[option].s);
					s->aValues[option].s = strdup (pVal);
					if (info)
						info |= SANE_INFO_RELOAD_OPTIONS | SANE_INFO_RELOAD_PARAMS;
				break;

				default: ;
					DBG(DBG_ERR,"SANE_ACTION_SET_VALUE: Invalid option (%d)\n", option);
					return SANE_STATUS_INVAL;
			} /* switch option */
			if (pInfo != NULL) {
				*pInfo = info;
			}
			break; /*case SANE_ACTION_SET_VALUE*/
		case SANE_ACTION_SET_AUTO:
			return SANE_STATUS_UNSUPPORTED;
		default: ;
			DBG(DBG_ERR,"sane_control_option: Invalid action (%d)\n", Action);
			return SANE_STATUS_INVAL;
	}
	return SANE_STATUS_GOOD;
}


/****************************************************************************/
SANE_Status
sane_get_parameters(SANE_Handle h, SANE_Parameters *p)
/****************************************************************************/
{
	TScanner *s;
	SANE_String val;

	s = (TScanner *)h;
	val = s->aValues[OPT_MODE].s;

	/* first do some checks */
	if (s->aValues[optTLX].w >= s->aValues[optBRX].w) {
		DBG(DBG_ERR,"sane_get_parameters: TLX should be smaller than BRX\n");
		s->aValues[optTLX].w = s->aValues[optBRX].w; /* new 17.06.03 JoHu */
		/* return SANE_STATUS_INVAL; / proper error code? */
	}
	if (s->aValues[optTLY].w >= s->aValues[optBRY].w) {
		DBG(DBG_ERR,"sane_get_parameters: TLY should be smaller than BRY\n");
		s->aValues[optTLY].w = s->aValues[optBRY].w; /* new 17.06.03 JoHu */
		/* return SANE_STATUS_INVAL; / proper error code? */
	}

	/* return the data */
	p->last_frame = SANE_TRUE;
	s->ScanParams.brightness = s->aValues[optBrightness].w;
	s->ScanParams.GainR = s->aValues[optGainR].w;
	s->ScanParams.GainG = s->aValues[optGainG].w;
	s->ScanParams.GainB = s->aValues[optGainB].w;
	p->depth = 8;

	if (!strcmp (s->aValues[optXPDA].s, "XPA")){
		s->ScanParams.optXPA = SANE_TRUE;
		DBG(DBG_MSG,"sane_get_parameters: set XPA to TRUE\n");
	}else{
		s->ScanParams.optXPA = SANE_FALSE;
		DBG(DBG_MSG,"sane_get_parameters: set XPA to FALSE\n");
	}

	if (s->aOptions[optCCD].cap != SANE_CAP_INACTIVE){
		if (!strcmp (s->aValues[optCCD].s, "CCDType 0")){
			s->ScanParams.oCCD_Type = 0;
			DBG(DBG_MSG,"sane_get_parameters: set CCD to 0\n");
		}
		if (!strcmp (s->aValues[optCCD].s, "CCDType 1")){
			s->ScanParams.oCCD_Type = 1;
			DBG(DBG_MSG,"sane_get_parameters: set CCD to 1\n");
		}
	}

	if (!strcmp (val, "Color")){
		s->ScanParams.mode = COLOR;
		DBG(DBG_MSG,"sane_get_parameters: switch to color 24 bit!\n");
		p->format = SANE_FRAME_RGB;
	}else{
		if (!strcmp (val, "B/W")){
			DBG(DBG_MSG,"sane_get_parameters:  switch to B/W 8 bit\n");
			s->ScanParams.mode = BLACK_WHITE;
			p->format = SANE_FRAME_GRAY;
			p->depth = 1;
		} else {
			DBG(DBG_MSG,"sane_get_parameters:  switch to gray 8 bit\n");
			s->ScanParams.mode = GRAY;
			p->format = SANE_FRAME_GRAY;
		}
	}
	p->lines = MM_TO_PIXEL(s->aValues[optBRY].w - s->aValues[optTLY].w,
													s->aValues[optDPI].w);
	p->pixels_per_line = MM_TO_PIXEL(s->aValues[optBRX].w - s->aValues[optTLX].w,
													s->aValues[optDPI].w);
	p->bytes_per_line = p->pixels_per_line;
	/* fill in the scanparams using the option values */
	s->ScanParams.iLinesLeft = p->lines;
	s->ScanParams.iDpi = s->aValues[optDPI].w;
	s->ScanParams.iLpi = s->aValues[optDPI].w;
	s->ScanParams.iWidth = p->pixels_per_line;
	s->ScanParams.iLenght = p->lines;
	s->ScanParams.fCalib = SANE_FALSE;

	switch (s->ScanParams.iDpi) {
		case 150:
			s->ScanParams.iY =
				MM_TO_PIXEL(s->aValues[optTLY].w + s->HWParams.iTopLeftY, 150);
			s->ScanParams.iX =
				MM_TO_PIXEL(s->aValues[optTLX].w + s->HWParams.iTopLeftX, 150);
		break;
		case 300:
			s->ScanParams.iY =
				MM_TO_PIXEL(s->aValues[optTLY].w + s->HWParams.iTopLeftY, 300);
			s->ScanParams.iX =
				MM_TO_PIXEL(s->aValues[optTLX].w + s->HWParams.iTopLeftX, 300);
		break;
		case 600:
			s->ScanParams.iY =
				MM_TO_PIXEL(s->aValues[optTLY].w + s->HWParams.iTopLeftY, 600);
			s->ScanParams.iX =
				MM_TO_PIXEL(s->aValues[optTLX].w + (s->HWParams.iTopLeftX / 2), 600);
		break;
		default:
		DBG(DBG_MSG,"sane_get_parameters: invalid resolution !");
		return SANE_STATUS_INVAL;
	}
	if (s->ScanParams.mode == COLOR)
		p->bytes_per_line = p->bytes_per_line * 3;

	/* init data pipe */
	s->DataPipe.iBytesLeft = 0;
	s->DataPipe.iScanned   = 0;
	s->DataPipe.iLinesLeft = p->lines;
	s->DataPipe.iLinesInCircBuf = 1;
	DBG(DBG_MSG,
		"sane_get_parameters: lpi %3d; dpi %3d; p->lines %3d; p->pixels_per_line %3d; p->bytes_per_line %3d; Mode %x\n",
		s->ScanParams.iLpi,s->ScanParams.iDpi,p->lines,
		p->pixels_per_line,p->bytes_per_line,
		s->ScanParams.mode);
	return SANE_STATUS_GOOD;
}


/****************************************************************************/
SANE_Status
sane_start(SANE_Handle h)
/****************************************************************************/
{
	TScanner             *s;
	SANE_Parameters      par;
	SANE_Int             pabLineBufSize;
	SANE_Status          State;

	DBG(DBG_MSG,"sane_start\n");
	s = (TScanner *)h;

	if (sane_get_parameters(h, &par) != SANE_STATUS_GOOD) {
		DBG(DBG_MSG," sane_get_parameters,invalid scan parameters\n");
		return SANE_STATUS_INVAL;
	}
	#ifdef DEBUG_FILE
		s->ScanParams.DebugOpen = SANE_FALSE;
		s->ScanParams.DebugSeek0 = SANE_FALSE;
	#endif

	DBG(DBG_MSG,"sane_start: optDPI=%d optTLY=%dmm, optTLX=%dmm, optBRX=%dmm, optBRY=%dmm.\n",
		s->aValues[optDPI].w,
		s->aValues[optTLY].w,
		s->aValues[optTLX].w,
		s->aValues[optBRX].w,
		s->aValues[optBRY].w);

	DBG(DBG_MSG,"sane_start: iDpi=%d iLpi=%d iY=%dpix. iX=%dpix. iWidth=%dpix iLenght=%dpix. mode=0x%x brightness=0x%x\n",
		s->ScanParams.iDpi,      /* horizontal resolution */
		s->ScanParams.iLpi,       /* vertical resolution */
		s->ScanParams.iY,         /* in HW coordsane_readinates */
		s->ScanParams.iX,         /* in HW coordinates */
		s->ScanParams.iWidth,     /* pixels per line*/
		s->ScanParams.iLenght,    /* Pixel lines */
		s->ScanParams.mode,
		s->ScanParams.brightness);

	/* hack */
	pabLineBufSize = par.bytes_per_line * 3;
	DBG(DBG_MSG,"sane_start: pabLineBufSize=%d\n",pabLineBufSize);
	s->DataPipe.pabLineBuf = (SANE_Byte *)malloc(pabLineBufSize);

	s->DataPipe.iZeroLines = 0;

	/* copy gamma table */
	Hp_rts_WriteGammaCalibTable (s->HWParams.iXferHandle, s->aGammaTableR,
			s->aGammaTableG, s->aGammaTableB);

	/* prepare and start the actual scan */
	State = Hp_rts_CircBufferInit( h, &s->DataPipe, par.bytes_per_line,
						s->ScanParams.iWidth, s->ScanParams.mode);
	if ( State == SANE_STATUS_GOOD ){
		State = _InitScan(&s->ScanParams, &s->HWParams, &s->DataPipe);
		if ( !State ) {
/*		if ( State != SANE_STATUS_GOOD ) {*/
			DBG(DBG_MSG,"sane_start: InitScan, invalid scan parameters\n");
			return SANE_STATUS_IO_ERROR;
		}
		s->fScanning = SANE_TRUE;
		return SANE_STATUS_GOOD;
	}
	s->fScanning = SANE_FALSE;
	return State;
}


/****************************************************************************/
SANE_Status
sane_read(SANE_Handle h, SANE_Byte *buf, SANE_Int maxlen, SANE_Int *len)
/* buf have the lenght maxlen; len returns the lenght of the retuned bytes  */
/****************************************************************************/
{
	TScanner  *s;
	TDataPipe *p;

	s = (TScanner *)h;

	DBG(DBG_SCAN,"sane_read: starts here\n");
	/* sane_read only allowed after sane_start */
	if (!s->fScanning) {
		DBG(DBG_ERR," sane_read: sane_read only allowed after sane_start\n");
		return SANE_STATUS_EOF;
	}

	p = &s->DataPipe;

	/* anything left to read? */
	if ((s->ScanParams.iLinesLeft == 0) && (p->iBytesLeft == 0)) {
		DBG(DBG_SCAN," sane_read: end of scan\n");
		_FinishScan(s);
		*len = 0;
		s->fScanning = SANE_FALSE;
		Hp_rts_CircBufferExit( h );
		if (p->pabLineBuf != NULL){
			free(p->pabLineBuf);
			p->pabLineBuf = NULL;
		}
		#ifdef DEBUG_FILE
		if ( s->ScanParams.DebugOpen ){
			fclose(s->ScanParams.FD_r);
			s->ScanParams.DebugOpen = SANE_FALSE;
		}
		#endif
		return SANE_STATUS_EOF;
	}

	/* time to read the next line? */
	*len = 0;
	if (p->iBytesLeft == 0) {
		if (Hp_rts_CircBufferGetLine(h, p, p->pabLineBuf,s->ScanParams.mode)){
			p->iBytesLeft = p->iBytesPerLine;
			s->ScanParams.iLinesLeft--;
			DBG(DBG_SCAN," sane_read: p->iBytesLeft=%3d s->ScanParams.iLinesLeft=%3d\n",
				p->iBytesLeft,s->ScanParams.iLinesLeft);
		}else
		{ /* no data read from the pipe ! */
			s->DataPipe.iZeroLines = s->DataPipe.iZeroLines +1;
			DBG(DBG_SCAN," sane_read: no bytes read from CircBuffer\n");
			if (s->DataPipe.iZeroLines <10){
				*len = 0;
				return SANE_STATUS_GOOD;
			}else /* time over */
			{
				*len = 0;
				return SANE_STATUS_IO_ERROR;
			}
		}
		/* copy (part of) a line */
		*len = MIN(maxlen, p->iBytesLeft);
		memcpy(buf, &p->pabLineBuf[p->iBytesPerLine - p->iBytesLeft], *len);
		p->iBytesLeft -= *len;
	}

	DBG(DBG_SCAN," sane_read: read %d bytes; maxlen = %d\n", *len,maxlen);
	return SANE_STATUS_GOOD;
}


/****************************************************************************/
void
sane_cancel(SANE_Handle h)
/****************************************************************************/
{
	TScanner *s;

	DBG(DBG_MSG,"sane_cancel\n");
	s = (TScanner *)h;

	/* Make sure the scanner head returns home */
	_FinishScan(s);
	Hp_rts_CircBufferExit( h );
	if (s->DataPipe.pabLineBuf != NULL){
		free(s->DataPipe.pabLineBuf);
		s->DataPipe.pabLineBuf = NULL;
	}
	#ifdef DEBUG_FILE
	if ( s->ScanParams.DebugOpen ){
		fclose(s->ScanParams.FD_r);
		s->ScanParams.DebugOpen = SANE_FALSE;
	}
	#endif
	s->fCanceled = SANE_TRUE;
	s->fScanning = SANE_FALSE;
}


/****************************************************************************/
SANE_Status
sane_set_io_mode(SANE_Handle h, SANE_Bool m)
/****************************************************************************/
{
	DBG(DBG_MSG, "sane_set_io_mode: %s\n", m ? "non-blocking" : "blocking");
	/* prevent compiler from complaining about unused parameters */
	h = h;

	if (m) {
		return SANE_STATUS_UNSUPPORTED;
	}
	return SANE_STATUS_GOOD;
}


/****************************************************************************/
SANE_Status
sane_get_select_fd(SANE_Handle h, SANE_Int *fd)
/****************************************************************************/
{
	DBG(DBG_MSG, "sane_select_fd\n");
	/* prevent compiler from complaining about unused parameters */
	h = h;
	fd = fd;
	return SANE_STATUS_UNSUPPORTED;
}


#ifdef STANDALONE
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/

typedef void (*rts8801_callback)(	void	*param,
					unsigned bytes,
					void	*data);

/****************************************************************************/
void
main_UsageError()
/****************************************************************************/
{
	fprintf(stderr,	"SANE HP_RTS88XX  x y width height resolution filename\n"
			"	x, y, width, height: in pixels\n"
			"	resolution: 1200/600/300/200/150/75\n");
	exit(1);
}


/****************************************************************************/
int
main (int argc, char *argv[])
/****************************************************************************/
{
	TScanner *s;
	/*SANE_String_Const name;*/
	SANE_Int piVersion;
	unsigned	x, y, w, h;
	SANE_Word	resolution;
	char const *	filename;
	FILE		*fp;
	int		result = 1;
	int		multiplier;
	/*int	depth;
	SANE_Int i;*/
	SANE_Auth_Callback pfnAuth;
	const SANE_Device **device_list;
	SANE_Bool local;
	SANE_Handle device;
	SANE_Status status;
	SANE_Int num_dev_options;
	SANE_Int info;
	SANE_Byte *buf;
	/*SANE_Int maxlen;*/
	SANE_Int len;

	DBG (DBG_MSG, "SANE HP_RTS88XX USB test version\n");

	if (argc != 7)
		main_UsageError();

	x = atoi(argv[1]);
	y = atoi(argv[2]);
	w = atoi(argv[3]);
	h = atoi(argv[4]);
	resolution = atoi(argv[5]);
	filename = argv[6];

	switch (resolution)
	{
	case 1200:
	case 600:
	case 400:
	case 300:
	case 200:
	case 150:
	case 100:
	case 75:
	case 50:
	case 25:
		break;

	default:
		fprintf(stderr, "Invalid resolution: %s\n", argv[5]);
		return 1;
	}

	multiplier = 1200 / resolution;
	if ((y + h) * multiplier > 16800)
	{
		fprintf(stderr, "Scan parameters take the image past the bottom of the scanner\n");
		return 1;
	}
	if ((x + w) * multiplier > 10451)
	{
		fprintf(stderr, "Scan parameters take the image past the right hand side of the scanner\n");
		return 1;
	}
	if (!x || !y || !w || !h)
	{
		fprintf(stderr, "None of the co-ordinates may be zero\n");
		return 1;
	}
	if (y * multiplier < 600)
	{
		fprintf(stderr, "Warning: at this resolution 'y' values under %d are in the pre-scan area\n",
				599 / multiplier + 1);
	}
	if (x * multiplier < 251)
	{
		fprintf(stderr, "Warning: at this resolution, 'x' values under %d are left of the scan area\n",
			250 / multiplier + 1);
	}

  s = malloc (sizeof (TScanner));
  if (!s)
    {
      DBG (DBG_MSG,"main : malloc for s failed\n");
      return SANE_STATUS_NO_MEM;
    }
  buf = malloc (5000);
  if (!buf)
    {
      DBG (DBG_MSG,"main : malloc buf failed\n");
      return SANE_STATUS_NO_MEM;
    }

	if (sane_init(&piVersion,pfnAuth) != SANE_STATUS_GOOD)
	{
		fprintf(stderr, "Could not initialise scanner\n");
		/* free scanner object memory */
		free ((void *) s);
		exit(1);
	}
	local = SANE_TRUE;
	if (sane_get_devices (&device_list, local) != SANE_STATUS_GOOD)
	{
		fprintf(stderr, "Could not sane_get_devices\n");
		/* free scanner object memory */
		free ((void *) s);
		exit(1);
	}

	status = sane_open ("libusb:001:002", &device);
	if (status != SANE_STATUS_GOOD)
	{
		fprintf(stderr, "Could not sane_open\n");
		/* free scanner object memory */
		free ((void *) s);
		exit(1);
	}

	/* Get the number of options. */
	status = sane_control_option (device, 0, SANE_ACTION_GET_VALUE, &num_dev_options, 0);
	if (status != SANE_STATUS_GOOD)
	{
		fprintf(stderr, "Could not read sane_control_option\n");
		/* free scanner object memory */
		free ((void *) s);
		exit(1);
	}

/*SANE_Word:
	case optDPI: case optTLX: case optTLY: case optBRX: case optBRY:
	case optBrightness: case optGainR: case optGainG: case optGainB:
SANE_TYPE_STRING
	case OPT_MODE: case optXPDA: case optCCD:
*/
	status = sane_control_option (device,optDPI, SANE_ACTION_SET_VALUE,
								  &resolution, &info);
	status = sane_control_option (device,optTLX, SANE_ACTION_SET_VALUE,
								  &x, &info);
	status = sane_control_option (device,optBRX, SANE_ACTION_SET_VALUE,
								  &w, &info);
	status = sane_control_option (device,optTLY, SANE_ACTION_SET_VALUE,
								  &y, &info);
	status = sane_control_option (device,optBRY, SANE_ACTION_SET_VALUE,
								  &h, &info);
#if 0
	free ((void *) s);
	exit(1);
#endif
	s = (TScanner *)device;

	/* TScanParams */
	s->ScanParams.brightness = 10;
	s->ScanParams.mode = 1;         /* B/W = 0; grayscale = 1 color = 2 */
	s->ScanParams.optXPA = 0;       /* 0=normal; 1 = XPA */
	s->ScanParams.oCCD_Type = 0;    /* 0/1 */

	fprintf(stdout, "Will open file %s\n",filename);
	fp = fopen(filename, "w");

	if (fp)
	{
		fprintf(fp, "P6\n%d %d\n255\n", w, h);

		switch (s->HWParams.ScannerModel)
		{
			case eHp3500c: case eHp3530c: case eHp3570c:
				if (Hp35x0c_init_scan(&s->HWParams, &s->ScanParams, &s->DataPipe))
					;
				else result = 1;
				break;

			case eHp4400c: case eHp4470c:
				if (Hp44x0_init_scan(&s->HWParams, &s->ScanParams, &s->DataPipe))
				{
					if (Hp44x0_start_scan(&s->HWParams, &s->ScanParams, &s->DataPipe))
					{
						s->fScanning = SANE_TRUE;
					}
				} /* if */

				break;
			default:
				fprintf(stderr, "Wrong scanner model\n");
				result = 1;
		}
	}
	else
	{
		perror(filename);
	}

	if (s->fScanning == SANE_TRUE)
	{
		while ( s->fScanning == SANE_FALSE )
		{
			status = sane_read(device, (SANE_Byte *)buf, 5000, &len);
/*			status = sane_read(device, &buf, 5000, &len);*/
			/* buf have the lenght maxlen; len returns the lenght of the retuned bytes  */
		}
	}
	/* free scanner object memory */
	free ((void *) s);
	return result;
}

#endif


