// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __ps_h__
#define __ps_h__

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#if defined( __alpha) || defined(__mips64) || defined(__sparcv9) || defined(__M64)
#define BIT32 int
#else
#define BIT32 long
#endif

// Filter Base Class

class Filter {
protected:
  unsigned char buf[256];
  unsigned char* ptr;

  virtual void cflush() =0;
  virtual void eflush(ostream&) =0;

public:
  Filter();

  virtual void in(unsigned char) =0;
  virtual void out(ostream&) =0;
  void flush(ostream&);

  friend Filter& operator<<(Filter& f, unsigned char c) {f.in(c); return f;}
  friend ostream& operator<<(ostream& s, Filter& f) {f.out(s); return s;}
};

// Compress Base Class

class Compress : public virtual Filter {
public:
  virtual void in(unsigned char) =0;
  virtual void cflush() {}
};

class NoCompress : public virtual Filter, public Compress {
public:
  void in(unsigned char);
};

class RLE : public virtual Filter, public Compress {
private:
  int state;
  unsigned char current;
  unsigned char rle[128];
  int num;

  void dumpNonRepeat();
  void dumpRepeat();

public:
  RLE();

  void in(unsigned char);
  void cflush();
};

// Encode Base Class

#define LINELIMIT 80

class Encode : public virtual Filter {
protected:
  int lineCount;

public:
  Encode();

  virtual void out(ostream&) =0;
  virtual void eflush(ostream&) =0;
};

class AsciiHex : public virtual Filter, public Encode {
public:
  void out(ostream&);
  void eflush(ostream&);
};

class Ascii85 : public virtual Filter, public Encode {
private:
  int index;
  int byteswap;
  union {
    unsigned char b[4];
    unsigned BIT32 c;
  } buf85;

  void dump(ostream&);
  BIT32 swap(unsigned BIT32* ptr);

public:
  Ascii85();

  void out(ostream&);
  void eflush(ostream&);
};

// PS Filters

class NoCompressAsciiHex : public virtual Filter, public NoCompress, public AsciiHex {};
class NoCompressAscii85 : public virtual Filter, public NoCompress, public Ascii85 {};
class RLEAsciiHex : public virtual Filter, public RLE, public AsciiHex {};
class RLEAscii85 : public virtual Filter, public RLE, public Ascii85 {};

#endif

