// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "fitsimage.h"

#include "NaN.h"

void FitsImage::initHist()
{
  // make sure we have rows and cols
  {
    FitsHead* head = fits_->head();
    if (head) {
      FitsTableHDU* hdu = (FitsTableHDU*)(head->hdu());
      if (!hdu->width() || !hdu->rows())
	return;
    }
  }

  // make sure we have cols to bin on
  if (!fits_->pBinX() || !fits_->pBinY()) {
    FitsHead* head = fits_->head();
    if (head) {
      FitsTableHDU* hdu = (FitsTableHDU*)head->hdu();

      // try for X and Y
      FitsColumn* x = hdu->find("X");
      FitsColumn* y = hdu->find("Y");

      // next, try for ra and dec
      if (!x)
	x = hdu->find("RA");
      if (!y)
	y = hdu->find("DEC");

      // last chance, try first and second column
      if (!x)
	x = hdu->find(0);
      if (!y)
	y = hdu->find(1);

      if (x) {
	char* str = trim(x->ttype());
	fits_->setpBinX(str);
	delete [] str;
      }
      if (y) {
	char* str = trim(y->ttype());
	fits_->setpBinY(str);
	delete [] str;
      }
    }
  }

  if (!fits_->pBinZ()) {
    FitsHead* head = fits_->head();
    if (head) {
      FitsTableHDU* hdu = (FitsTableHDU*)head->hdu();

      // try for TIME
      FitsColumn* z = hdu->find("TIME");

      // last chance, try third column
      if (!z)
	z = hdu->find(2);

      if (z) {
	char* str = trim(z->ttype());
	fits_->setpBinZ(str);
	delete [] str;
      }
    }
  }

  // update default values from parent
  binFunction_ = parent->binFunction_;
  binBufferSize_ = parent->binBufferSize_;
  binDepth_ = parent->binDepth_;
  binFactor_ = parent->binFactor_;

  nextHist(getHistCenter());
}

int FitsImage::hasBinCol(const char* str)
{
  if (fits_) {
    FitsHead* head = fits_->head();
    if (head) {
      FitsTableHDU* hdu = (FitsTableHDU*)head->hdu();
      return hdu->find(str) ? 1 : 0;
    }
  }
  return 0;
}

void FitsImage::setBinCursor(const Vector& v)
{
  histCursor = v * refToPhysical;
}

void FitsImage::setBinFactor(const Vector& b) {
  Vector bb = b;
  binFactor_[0] *= bb[0] <= 0 ? 1 : bb[0];
  binFactor_[1] *= bb[1] <= 0 ? 1 : bb[1];
}

void FitsImage::setBinToFactor(const Vector& b) {
  Vector bb = b;
  binFactor_[0] = bb[0] <= 0 ? 1 : bb[0];
  binFactor_[1] = bb[1] <= 0 ? 1 : bb[1];
}

Matrix FitsImage::updateHistCenter()
{
  return hist_ ? nextHist(getHistCenter()) : Matrix();
}

Matrix FitsImage::updateHistCursor()
{
  return hist_ ? nextHist(histCursor) : Matrix();
}

Matrix FitsImage::updateHist(const Vector& v)
{
  // v is in bin (physical) coords
  return hist_ ? nextHist(v) : Matrix();
}

Matrix FitsImage::nextHist(const Vector& c)
{
  // cursor, c is in bin (physical) coords
  // remember where we are pointing
  histCursor = c;

  // Vector s = getHistDim()/binFactor_;
  Vector d = getHistDim();
  Vector s;
  s[0] = d[0]/binFactor_[0];
  s[1] = d[1]/binFactor_[1];

  // make sure that we have a width/height of at least 1
  if (s[0]<1)
    s[0]=1;
  if (s[1]<1)
    s[1]=1;

  int width = (int)(s[0]<binBufferSize_? s[0] : binBufferSize_);
  int height= (int)(s[1]<binBufferSize_? s[1] : binBufferSize_);
  int depth = binDepth_;

  Vector center = Vector(width, height)/2;

  if (DebugBin) {
    cerr << "width height: " << width << ' ' << height << endl;
    cerr << "center: " << center << endl;
    cerr << "center.ceil(): " << center.ceil() << endl;
  }

  if (binFactor_[0]<1 || binFactor_[1]<1) {
    actualHistCursor = histCursor;

    if (DebugBin)
      cerr << "histCursor: " << histCursor << endl;
  }
  else {
    // force to a bin boundary, then translate to center of bin cell
    //    actualHistCursor = ((histCursor/binFactor_).floor() * binFactor_) +
    //      Vector(.5,.5);
    actualHistCursor[0] = (floor(histCursor[0]/binFactor_[0]) * binFactor_[0])
      +  .5;
    actualHistCursor[1] = (floor(histCursor[1]/binFactor_[1]) * binFactor_[1])
      +  .5;

    // now, figure out any offset due to mod(lowerleft,binFactor_)
    FitsTableHDU* hdu = (FitsTableHDU*)(fits_->head())->hdu();
    Vector xd = hdu->dimension(fits_->pBinX());
    Vector yd = hdu->dimension(fits_->pBinY());
    Vector ll(xd[0],yd[0]);

    //    Vector a = ((ll/binFactor_).floor() * binFactor_) + Vector(.5,.5);
    Vector a;
    a[0] = (floor(ll[0]/binFactor_[0]) * binFactor_[0]) + .5;
    a[1] = (floor(ll[1]/binFactor_[1]) * binFactor_[1]) + .5;
    Vector offset = a-ll;
    actualHistCursor -= offset;

    if (DebugBin) {
      cerr << "histCursor: " << histCursor << endl;
      cerr << "actualHistCursor: " << actualHistCursor << endl;
      cerr << "ll: " << ll << endl;
      cerr << "offset: " << offset << endl;
    }
  }

  // new physicalToData
  Matrix mm = 
    Translate(-actualHistCursor) * 
    Scale(1./binFactor_[0],1./binFactor_[1]) *
    Translate(center.ceil());

  if (DebugBin)
    cerr << "mm: " << mm << endl << endl;

  if (hist_)
    delete hist_;

  hist_ = new FitsHist(fits_, width, height, depth, mm,
		       binFunction_, binFactor_);

  // reset LTMV keywords
  keyLTMV = 0;

  if (!hist_->isValid()) {
    reset();
    return Matrix();
  }

  // reset scanmode
  parent->resetScanMode();
  load();

  // return matrix
  return refToPhysical * mm * dataToRef;
}

Vector FitsImage::getHistColMinMax(const char* col)
{
  return fits_ ? fits_->getColMinMax(col) : Vector();
}

Vector FitsImage::getHistColDim(const char* col)
{
  return fits_ ? fits_->getColDim(col) : Vector();
}

Vector FitsImage::getHistDim()
{
  if (!isBinTable())
    return Vector();

  // assumes we aready have our columns
  FitsTableHDU* hdu = (FitsTableHDU*)(fits_->head())->hdu();
  Vector xd = hdu->dimension(fits_->pBinX());
  Vector yd = hdu->dimension(fits_->pBinY());

  // if DBL_MAX, we will get nan
  Vector r(xd[1]-xd[0],yd[1]-yd[0]);

  if (isnand(r[0]) || isnand(r[1]))
    return Vector(DBL_MAX,DBL_MAX);
  else
    return r;
}

Vector FitsImage::getHistCenter()
{
  // return bin (physical) coords
  if (!isBinTable())
    return Vector();

  // assumes we aready have our columns
  FitsTableHDU* hdu = (FitsTableHDU*)(fits_->head())->hdu();
  Vector xd = hdu->dimension(fits_->pBinX());
  Vector yd = hdu->dimension(fits_->pBinY());

  // if DBL_MAX, we will get nan
  Vector r = Vector(xd[1]-xd[0],yd[1]-yd[0])/2 + Vector(xd[0],yd[0]);

  if (isnand(r[0]) || isnand(r[1]))
    return Vector();
  else
    return r;
}

const char* FitsImage::getHistFilter()
{
  return fits_ ? fits_->pFilter() : NULL;
}

const char* FitsImage::getHistX()
{
  return fits_ ? fits_->pBinX() : NULL;
}

const char* FitsImage::getHistY()
{
  return fits_ ? fits_->pBinY() : NULL;
}

const char* FitsImage::getHistZ()
{
  return fits_ ? fits_->pBinZ() : NULL;
}

char* FitsImage::getHistList()
{
  if (!isBinTable())
    return NULL;

  FitsHead* head = fits_->head();
  return ((FitsTableHDU*)head->hdu())->list();
}



