// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "ruler.h"
#include "fitsimage.h"

Ruler::Ruler(const Ruler& a) : BaseLine(a)
{
  p3 = a.p3;
  coordSystem = a.coordSystem;
  skyFrame = a.skyFrame;
  distSystem = a.distSystem;
  distDist = a.distDist;
  dist = a.dist;
}

Ruler::Ruler(Base* p, const Vector& ptr1, const Vector& ptr2,
	     Coord::CoordSystem sys, Coord::SkyFrame sky, 
	     Coord::CoordSystem distsys, Coord::SkyDist distfor,
	     const char* clr, int* dsh,
	     int wth, const char* fnt, const char* txt, 
	     unsigned short prop, const char* cmt,
	     const List<Tag>& tg, const List<CallBack>& cb) 
  : BaseLine(p, ptr1, ptr2, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  coordSystem = sys;
  skyFrame = sky;
  distSystem = distsys;
  distDist = distfor;
  dist = 0;

  strcpy(type_,"ruler");
  handle = new Vector[2];
  numHandle = 2;

  updateBBox();
}

void Ruler::renderX(Drawable drawable, Coord::InternalSystem sys, RenderMode mode)
{
  GC lgc = renderXGC(mode);

  Vector aa = parent->mapFromRef(p1,sys);
  Vector bb = parent->mapFromRef(p2,sys);
  Vector cc = parent->mapFromRef(p3,sys);
  Vector dd = modifyArrow(p2,p1,sys);
  Vector ee = modifyArrow(p1,p2,sys);

  // line
  XDrawLine(display, drawable, lgc, dd[0], dd[1], ee[0], ee[1]);
  renderXArrow(drawable, p2, p1, sys, lgc);
  renderXArrow(drawable, p1, p2, sys, lgc);

  // axes
  renderXLineDash(lgc);
  XDrawLine(display, drawable, lgc, aa[0], aa[1], cc[0], cc[1]);
  XDrawLine(display, drawable, lgc, bb[0], bb[1], cc[0], cc[1]);

  // dist
  ostringstream str;
  distToStr(str);

  if (tkfont_) {
    XSetFont(display, lgc, Tk_FontId(tkfont_));

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(tkfont_, &metrics);
    char* buf = dupstr(str.str().c_str());
    int width = Tk_TextWidth(tkfont_, buf, strlen(buf));

    Vector tt = ((bb-aa)/2+aa) * Translate(-width/2.,-metrics.descent);
    Tk_DrawChars(display, drawable, lgc, tkfont_, buf, strlen(buf), 
		 tt[0], tt[1]);
    if (buf)
      delete [] buf;
  }
}

GC Ruler::renderXGC(RenderMode mode)
{
  // set width, color, dash
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color); 
    renderXLineNoDash(gc);
    return gc;

  case XOR:
    renderXLineDash(gcxor);
    return gcxor;
  }
}

void Ruler::renderPS(int mode)
{
  renderPSGC(mode);

  Vector aa = parent->mapFromRef(p1,Coord::CANVAS);
  Vector bb = parent->mapFromRef(p2,Coord::CANVAS);
  Vector cc = parent->mapFromRef(p3,Coord::CANVAS);
  Vector dd = modifyArrow(p2,p1,Coord::CANVAS);
  Vector ee = modifyArrow(p1,p2,Coord::CANVAS);

  // line
  {
    ostringstream str;
    str << "newpath " 
	<< dd.TkCanvasPs(parent->canvas) << "moveto"
	<< ee.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    renderPSArrow(p2,p1,Coord::CANVAS);
    renderPSArrow(p1,p2,Coord::CANVAS);
  }

  // axes
  renderPSLineDash();
  {
    ostringstream str;
    str << "newpath " 
	<< aa.TkCanvasPs(parent->canvas) << "moveto"
	<< cc.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl;
    str << "newpath " 
	<< bb.TkCanvasPs(parent->canvas) << "moveto"
	<< cc.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl;
    str << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }


  // dist
  if (psfont_) {
    ostringstream vstr;
    ostringstream str;

    const char* ff = Tk_NameOfFont(psfont_);
    str << '/' << psFontName(ff)
    	<< " findfont " << int(psFontSize(ff)*parent->getDisplayRatio())
	<< " scalefont setfont" << endl;

    distToStr(vstr);
    char* buf = dupstr(vstr.str().c_str());
    Vector tt = ((bb-aa)/2 + aa).TkCanvasPs(parent->canvas);
    str << "gsave" << endl
	<< "newpath " << endl
	<< tt << " moveto" << endl
	<< '(' << psQuote(buf) << ')' << endl
	<< "dup true charpath pathbbox " << endl
	<< "closepath " << endl
	<< "3 -1 roll sub 2.5 div " << endl
	<< "3 1 roll sub 2 div exch " << endl
	<< tt << " moveto rmoveto show " << endl
	<< "grestore" << endl;

    str << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    if (buf)
      delete buf;
  }
}

void Ruler::renderPSGC(int mode)
{
  renderPSColor(mode, parent->getXColor(colorName));
  renderPSLineNoDash();
}

#ifdef _MACOSX
void Ruler::renderMACOSX()
{
  renderMACOSXGC();

  Vector aa = parent->mapFromRef(p1,Coord::CANVAS);
  Vector bb = parent->mapFromRef(p2,Coord::CANVAS);
  Vector cc = parent->mapFromRef(p3,Coord::CANVAS);
  Vector dd = modifyArrow(p2,p1,Coord::CANVAS);
  Vector ee = modifyArrow(p1,p2,Coord::CANVAS);

  // line
  macosxDrawLine(dd,ee);
  renderMACOSXArrow(p2,p1,Coord::CANVAS);
  renderMACOSXArrow(p1,p2,Coord::CANVAS);

  // axes
  renderMACOSXLineDash();
  macosxDrawLine(aa,cc);
  macosxDrawLine(bb,cc);

  // dist
  {
    ostringstream vstr;
    distToStr(vstr);

    if (psfont_) {
      Tcl_DString psdstr;
      Tcl_DStringInit(&psdstr);
      int psSize = Tk_PostscriptFontName(psfont_, &psdstr);
      macosxFont(Tcl_DStringValue(&psdstr), psSize);
      Tcl_DStringFree(&psdstr);

      Tk_FontMetrics metrics;
      Tk_GetFontMetrics(psfont_, &metrics);
      char* buf = dupstr(vstr.str().c_str());
      int width = Tk_TextWidth(psfont_, vbuf, strlen(buf));

      Vector tt = ((bb-aa)/2 + aa) * Translate(-width/2., -metrics.descent);
      macosxDrawText(tt, 0, buf);
      if (buf)
	delete buf;
    }
  }
}

void Ruler::renderMACOSXGC()
{
  macosxColor(parent->getXColor(colorName));
  renderMACOSXLineNoDash();
}
#endif

// WIN32
#ifdef _WIN32
void Ruler::renderWIN32()
{
  renderWIN32GC();

  Vector aa = parent->mapFromRef(p1,Coord::CANVAS);
  Vector bb = parent->mapFromRef(p2,Coord::CANVAS);
  Vector cc = parent->mapFromRef(p3,Coord::CANVAS);
  Vector dd = modifyArrow(p2,p1,Coord::CANVAS);
  Vector ee = modifyArrow(p1,p2,Coord::CANVAS);

  // line
  win32DrawLine(dd,ee);
  renderWIN32Arrow(p2,p1,Coord::CANVAS);
  renderWIN32Arrow(p1,p2,Coord::CANVAS);

  // axes
  renderWIN32LineDash();
  win32DrawLine(aa,cc);
  win32DrawLine(bb,cc);

  // dist
  {
    ostringstream vstr;
    distToStr(vstr);

    if (tkfont_) {
      win32Font(tkfont_);

      Tk_FontMetrics metrics;
      Tk_GetFontMetrics(tkfont_, &metrics);
      char* buf = dupstr(vstr.str().c_str());
      int width = Tk_TextWidth(tkfont_, buf, strlen(buf));

      Vector tt = ((bb-aa)/2 + aa) * Translate(-width/2., -metrics.descent);
      win32DrawText(tt, 0, buf);
      if (buf)
	delete buf;
    }
  }
}

void Ruler::renderWIN32GC()
{
  win32Color(parent->getXColor(colorName));
  renderWIN32LineNoDash();
}
#endif

// Support

void Ruler::updateHandles()
{
  center = (p2-p1)/2 + p1;
  angle  = (p2-p1).angle();
  if (parent->isIIS())
    angle = -angle;

  // calc p3, dist
  FitsImage* ptr = parent->findFits(coordSystem,center);
  Vector a = ptr->mapFromRef(p1,coordSystem,skyFrame);
  Vector b = ptr->mapFromRef(p2,coordSystem,skyFrame);
  p3 = ptr->mapToRef(Vector(b[0],a[1]),coordSystem,skyFrame);
  dist = ptr->mapDistFromRef(p2, p1, distSystem, distDist);

  // generate handles in canvas coords
  handle[0] = parent->mapFromRef(p1,Coord::CANVAS);
  handle[1] = parent->mapFromRef(p2,Coord::CANVAS);
}

void Ruler::calcAllBBox()
{
  // P3
  bbox.bound(parent->mapFromRef(p3,Coord::CANVAS));

  // make room for text
  if (tkfont_) {
    Vector v = (p2-p1)/2 + p1;

    ostringstream str;
    distToStr(str);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(tkfont_, &metrics);
    char* buf = dupstr(str.str().c_str());
    int width = Tk_TextWidth(tkfont_, buf, strlen(buf));

    Vector ll = parent->mapFromRef(v,Coord::CANVAS) *
      Translate(-width/2.,-metrics.descent);
    Vector ur = parent->mapFromRef(v,Coord::CANVAS) *
      Translate(width/2.,metrics.ascent);
    
    bbox.bound(ll);
    bbox.bound(ur);

    if (buf)
      delete [] buf;
  }

  Marker::calcAllBBox();
}

void Ruler::updateCoords(const Matrix& mx)
{
  p3*=mx;
  BaseLine::updateCoords(mx);
}

int Ruler::isOn(const Vector& v, const Vector& v1, const Vector& v2)
{
  // v : canvas coords
  // v1: ref coords
  // v2: ref coords

  // do this in canvas coords, not ref coords

  Vector l1 = parent->mapFromRef(v1,Coord::CANVAS);
  Vector l2 = parent->mapFromRef(v2,Coord::CANVAS);
  double a = (l2-l1).angle();

  Matrix m = Translate(-l1) * Rotate(a);
  Vector end = l2*m;
  Vector vv = v*m;
  return (vv[0]>0 && vv[0]<end[0] && vv[1]>-3 && vv[1]<3);
}

int Ruler::isIn(const Vector& vv)
{
  // test to see if point is on edge, if so, considered inside

  if (isOn(vv,p1,p2) || isOn(vv,p1,p3) || isOn(vv,p2,p3))
    return 1;

  /*
    v[0]-- x value of point being tested
    v[1]-- y value of point being tested

    This algorithm is from "An Introduction to Ray Tracing", Academic Press,
    1989, edited by Andrew Glassner, pg 53
    -- a point lies in a polygon if a line is extended from the point to 
    infinite
    in any direction and the number of intersections with the polygon is odd.
    This is valid for both concave and convex polygons.
    Points on a vertex are considered inside.
    Points on a edge are considered inside.
  */

  // analysis in ref coords
  Vector v = parent->mapToRef(vv,Coord::CANVAS);

  int crossings = 0;   // number of crossings
  int sign;

  // for all edges

  for (int i=0; i<3; i++) {
    Vector v1, v2;
    switch (i) {
    case 0:
      v1 = p1-v;
      v2 = p2-v;
      sign = ((v1[1])>=0) ? 1 : -1; // init sign
      break;
    case 1:
      v1 = p2-v;
      v2 = p3-v;
      break;
    case 2:
      v1 = p3-v;
      v2 = p1-v;
      break;
    }

    int nextSign = (v2[1]>=0) ? 1 : -1; // sign holder for p2

    if (sign != nextSign) {
      if (v1[0]>0 && v2[0]>0)
	crossings++;
      else if (v1[0]>0 || v2[0]>0) {
	if (v1[0]-(v1[1]*(v2[0]-v1[0])/(v2[1]-v1[1])) > 0)
	  crossings++;
      }
      sign = nextSign;
    }
  }

  return fmod(float(crossings),float(2)) ? 1 : 0; // if odd, point is inside
}

void Ruler::setCoordSystem(Coord::CoordSystem sys, Coord::SkyFrame sky, 
			   Coord::CoordSystem dsys, Coord::SkyDist dist)
{
  coordSystem = sys;
  skyFrame = sky;
  distSystem = dsys;
  distDist = dist;
  updateBBox();
}

void Ruler::distToStr(ostringstream& str)
{
  switch (distSystem) {
  case Coord::IMAGE:
    str << dist << " img";
    break;
  case Coord::PHYSICAL:
    str << dist << " phy";
    break;
  case Coord::AMPLIFIER:
    str << dist << " amp";
    break;
  case Coord::DETECTOR:
    str << dist << " det";
    break;
  default:
    if (parent->findFits()->hasWCSCel(distSystem))
      switch (distDist) {
      case Coord::DEGREE:
	str << dist << " deg";
	break;
      case Coord::ARCMIN:
	str << dist << '\'';
	break;
      case Coord::ARCSEC:
	str << dist << '"';
	break;
      }
    else
      str << dist << " lin";
  }
}

// list

void Ruler::list(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		 Coord::SkyFormat format, int conj, int strip)
{
  if (!strip) {
    FitsImage* ptr = parent->findFits(sys,center);
    listPre(str, sys, sky, ptr, strip, 1);

    switch (sys) {
    case Coord::IMAGE:
    case Coord::PHYSICAL:
    case Coord::DETECTOR:
    case Coord::AMPLIFIER:
      {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	str << type_ << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	    << v2[0] << ',' << v2[1] << ')';
      }
      break;
    default:
      if (ptr->hasWCS(sys)) {
	if (ptr->hasWCSCel(sys)) {
	  switch (format) {
	  case Coord::DEGREES:
	    {
	      Vector v1 = ptr->mapFromRef(p1,sys,sky);
	      Vector v2 = ptr->mapFromRef(p2,sys,sky);
	      str << type_ << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		  << ',' << v2[0] << ',' << v2[1] << ')';
	    }
	    break;
	  case Coord::SEXAGESIMAL:
	    {
	      char buf[64];
	      char ra1[16], ra2[16];
	      char dec1[16], dec2[16];
	      {
		ptr->mapFromRef(p1,sys,sky,format,buf,64);
		string x(buf);
		istringstream wcs(x);
		wcs >> ra1 >> dec1;
	      }
	      {
		ptr->mapFromRef(p2,sys,sky,format,buf,64);
		string x(buf);
		istringstream wcs(x);
		wcs >> ra2 >> dec2;
	      }
	      str << type_ << '(' << ra1 << ',' << dec1 << ',' 
		  << ra2 << ',' << dec2 << ')';
	    }
	    break;
	  }
	}
	else {
	  Vector v1 = ptr->mapFromRef(p1,sys);
	  Vector v2 = ptr->mapFromRef(p2,sys);
	  str << type_ << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	      << ',' << v2[0] << ',' << v2[1] << ')';
	}
      }
    }

    if (conj)
      str << " ||";

    str << " ruler=";
    coord.listCoordSystem(str, coordSystem, skyFrame, 1, ptr->hasWCSCel(coordSystem));
    str << ' ';
    coord.listDistSystem(str, distSystem, distDist, ptr->hasWCSCel(distSystem));
    listProperties(str, 0);
  }
}

void Ruler::listXML(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky, 
		    Coord::SkyFormat format)
{
  FitsImage* ptr = parent->findFits(sys,center);
  Vector vv[2];
  vv[0] = p1;
  vv[1] = p2;

  ostringstream sysstr;
  coord.listCoordSystem(sysstr,coordSystem,skyFrame,0,ptr->hasWCSCel(coordSystem));
  ostringstream diststr;
  coord.listDistSystem(diststr,distSystem,distDist,ptr->hasWCSCel(distSystem));

  XMLRowInit();
  XMLRow(XMLSHAPE,type_);

  XMLRowPoint(ptr,sys,sky,format,vv,2);
  XMLRow(XMLPARAM,(char*)(sysstr.str().c_str()));
  XMLRow(XMLPARAM2,(char*)(diststr.str().c_str()));

  XMLRowProps(ptr,sys);
  XMLRowEnd(str);
}

