#include <glib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include "saryer.h"
#include "getopt.h"

#ifdef SUFARY211
#  include "sufary211.h"
#else
#  include "sufary.h"
#endif

typedef long	(*BenchmarkFunc)(const gchar *file_name, 
				 const gchar *pattern);

static void	benchmark 	(const gchar *target, 
			  	 BenchmarkFunc benchmark_func,
			  	 const gchar *file_name,
			  	 const gchar *pattern);
static long	sary		(const gchar *file_name, 
				 const gchar *pattern);
static long	sufary		(const gchar *file_name, 
				 const gchar *pattern);
static void 	parse_options 	(int argc, char **argv);
static void 	show_usage	(void);

static gint iterations = 1;
static gboolean run_sary    = TRUE;
static gboolean run_sufary  = TRUE;

int 
main (int argc, char **argv)
{
    gchar *file_name, *pattern;

    parse_options(argc, argv);
    if (optind  >= argc) {
	show_usage();
	exit(EXIT_FAILURE);
    }

    pattern   = argv[optind];
    file_name = argv[optind + 1];

    if (run_sary) {
	benchmark("Sary", sary, file_name, pattern);
    }
    if (run_sufary) {
#ifdef SUFARY211
	benchmark("SUFARY 2.1.1", sufary, file_name, pattern);
#else
	benchmark("SUFARY", sufary, file_name, pattern);
#endif
    }

    return 0;
}

static void
benchmark (const gchar *target, 
	   BenchmarkFunc benchmark_func,
	   const gchar *file_name,
	   const gchar *pattern)
{
    long hitnum;
    double elapsed;

    clock_t start = clock();
    hitnum = benchmark_func(file_name, pattern);
    elapsed = (clock() - start) / CLOCKS_PER_SEC;

    g_print("%-15s: %.2f (%ld hit)\n", target, elapsed, hitnum);
}


static long
sary (const gchar *file_name, const gchar *pattern)
{
    Saryer *saryer;
    gint i;
    gboolean status = FALSE;
    long hitnum;

    saryer = saryer_new(file_name);
    if (saryer == NULL) {
        g_printerr("error: %s(.ary): %s\n", file_name, g_strerror(errno));
        exit(EXIT_FAILURE);
    }

    for (i = 0; i < iterations; i++) {
	status = saryer_search(saryer, pattern);
    }

    if (status == TRUE) {
	hitnum = saryer_count_occurrences(saryer);
    } else {
	hitnum = 0;
    }

    saryer_destroy(saryer);
    return hitnum;
}

#ifdef SUFARY211

static long
sufary (const gchar *file_name, const gchar *pattern)
{
    SUFARY *ary;
    eresult result = FAIL;
    gint i;
    long hitnum;

    ary = sa_openfiles(file_name, NULL);
    if (ary == NULL) {
        g_printerr("error: %s(.ary): %s\n", file_name, g_strerror(errno));
	exit(EXIT_FAILURE);
    }

    for (i = 0; i < iterations; i++) {
	sa_reset(ary);
	result = sa_search(ary, pattern, strlen(pattern), 0);
    }

    if (result == CONT) {
	hitnum = ary->right - ary->left + 1;
    } else {
	hitnum = 0;
    }

    sa_closefiles(ary);
    return hitnum;
}

#else

static long
sufary (const gchar *file_name, const gchar *pattern)
{
    SUFARY *ary;
    SUF_RESULT result;
    gint i;
    long hitnum;

    ary = sa_open(file_name, NULL);
    if (ary == NULL) {
        g_printerr("error: %s(.ary): %s\n", file_name, g_strerror(errno));
	exit(EXIT_FAILURE);
    }

    for (i = 0; i < iterations; i++) {
	result = sa_find(ary, 0, ary->arraysize - 1, 
			     pattern, strlen(pattern), 0);
    }

    if (result.stat == SUCCESS) {
	hitnum = result.right - result.left + 1;
    } else {
	hitnum = 0;
    }

    sa_close(ary);
    return hitnum;
}

#endif


static const char *short_options = "n:osS";
static struct option long_options[] = {
    { "iteration",		required_argument,	NULL, 'n' },
    { "sary",			no_argument,		NULL, 's' },
    { "sufary",			no_argument,		NULL, 'S' },
    { NULL, 0, NULL, 0 }
};

static void
parse_options (int argc, char **argv)
{
    while (1) {
        int ch = getopt_long(argc, argv, short_options, long_options, NULL);
        if (ch == EOF) {
	    break;
	}
	switch (ch) {
	case 'n':
	    iterations = atoi(optarg);
            break;
	case 's':
	    run_sary   = 1;
	    run_sufary = 0;
            break;
	case 'S':
	    run_sary   = 0;
	    run_sufary = 1;
            break;
	}
    }
}

static void
show_usage (void)
{
    g_print("Usage: vs-sufary [-n NUM] <pattern> <file>\n");
}

