/*
 * scamper_do_neighbourdisc
 *
 * $Id: scamper_do_neighbourdisc.c,v 1.11 2010/05/11 20:22:09 mjl Exp $
 *
 * Copyright (C) 2009-2010 Matthew Luckie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef lint
static const char rcsid[] =
  "$Id: scamper_do_neighbourdisc.c,v 1.11 2010/05/11 20:22:09 mjl Exp $";
#endif

#include <sys/types.h>

#if defined(_MSC_VER)
typedef unsigned __int8 uint8_t;
typedef unsigned __int16 uint16_t;
typedef unsigned __int32 uint32_t;
typedef __int16 int16_t;
#define __func__ __FUNCTION__
#endif

#ifdef _WIN32
#include <winsock2.h>
#include <ws2tcpip.h>
#endif

#include <sys/types.h>

#ifndef _WIN32
#include <sys/time.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#include <netinet/if_ether.h>
#include <unistd.h>
#endif

#if defined(__linux__)
#define __FAVOR_BSD
#elif !defined(_WIN32)
#include <net/if_dl.h>
#include <net/if_types.h>
#include <net/route.h>
#endif

#if defined(__APPLE__)
#include <stdint.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <assert.h>

#if defined(DMALLOC)
#include <dmalloc.h>
#endif

#include "scamper.h"
#include "scamper_addr.h"
#include "scamper_list.h"
#include "scamper_neighbourdisc.h"
#include "scamper_fds.h"
#include "scamper_dl.h"
#include "scamper_probe.h"
#include "scamper_task.h"
#include "scamper_options.h"
#include "scamper_if.h"
#include "scamper_getsrc.h"
#include "scamper_queue.h"
#include "scamper_file.h"
#include "scamper_debug.h"
#include "scamper_target.h"
#include "scamper_do_neighbourdisc.h"
#include "mjl_list.h"
#include "utils.h"

#if !defined(ETHERTYPE_IPV6)
#define ETHERTYPE_IPV6 0x86DD
#endif

#if !defined(ETHERTYPE_IP)
#define ETHERTYPE_IP 0x0800
#endif

#if !defined(ETHERTYPE_ARP)
#define ETHERTYPE_ARP 0x0806
#endif

#if !defined(ND_NEIGHBOR_SOLICIT)
#define ND_NEIGHBOR_SOLICIT 135
#endif

#ifdef _WIN32
struct ip6_hdr
{
  uint32_t        ip6_vfc_flow;
  uint16_t        ip6_plen;
  uint8_t         ip6_nxt;
  uint8_t         ip6_hlim;
  struct in6_addr ip6_src;
  struct in6_addr ip6_dst;
};
struct icmp6_hdr
{
  uint8_t  icmp6_type;
  uint8_t  icmp6_code;
  uint16_t icmp6_cksum;
  uint32_t icmp6_data32;
};
#endif

static scamper_task_funcs_t nd_funcs;
extern scamper_addrcache_t *addrcache;
static uint8_t *pktbuf;
static size_t pktbuf_len;

typedef struct nd_state
{
  scamper_fd_t   *fd;
  int             ifindex;
  int             replyc;
  scamper_task_t *task0;
  dlist_t        *cbs;
} nd_state_t;

typedef struct ndcb
{
  void         *param;
  void        (*cb)(void *, scamper_addr_t *);
  dlist_node_t *node;
} ndcb_t;

#define ND_OPT_FIRSTRESPONSE     1
#define ND_OPT_IFNAME            2
#define ND_OPT_REPLYC            3
#define ND_OPT_ATTEMPTS          4
#define ND_OPT_ALLATTEMPTS       5
#define ND_OPT_WAIT              6

static const scamper_option_in_t opts[] = {
  {'F', NULL, ND_OPT_FIRSTRESPONSE, SCAMPER_OPTION_TYPE_NULL},
  {'i', NULL, ND_OPT_IFNAME,        SCAMPER_OPTION_TYPE_STR},
  {'o', NULL, ND_OPT_REPLYC,        SCAMPER_OPTION_TYPE_NUM},
  {'q', NULL, ND_OPT_ATTEMPTS,      SCAMPER_OPTION_TYPE_NUM},
  {'Q', NULL, ND_OPT_ALLATTEMPTS,   SCAMPER_OPTION_TYPE_NULL},
  {'w', NULL, ND_OPT_WAIT,          SCAMPER_OPTION_TYPE_NUM},
};
static const int opts_cnt = SCAMPER_OPTION_COUNT(opts);

const char *scamper_do_neighbourdisc_usage(void)
{
  return "neighbourdisc [-FQ] [-i if] [-o replyc] [-q attempts] [-w wait]\n";
}

static void do_nd_cbs(nd_state_t *state, scamper_addr_t *mac)
{
  ndcb_t *ndcb;

  if(state != NULL && state->cbs != NULL)
    {
      while((ndcb = dlist_head_pop(state->cbs)) != NULL)
	{
	  ndcb->cb(ndcb->param, mac);
	  free(ndcb);
	}
      dlist_free(state->cbs); state->cbs = NULL;
    }

  return;
}

static void nd_handleerror(scamper_task_t *task, int error)
{
  do_nd_cbs(task->state, NULL);
  scamper_queue_done(task->queue, 0);
  return;
}

static void nd_done(scamper_task_t *task)
{
  scamper_neighbourdisc_t *nd = task->data;
  nd_state_t *state = task->state;
  do_nd_cbs(state, nd != NULL ? nd->dst_mac : NULL);
  scamper_queue_done(task->queue, 0);
  return;
}

static void nd_state_free(nd_state_t *state)
{
  if(state->fd != NULL) scamper_fd_free(state->fd);
  free(state);
  return;
}

static int nd_state_alloc(scamper_task_t *task)
{
  scamper_neighbourdisc_t *nd = task->data;
  scamper_dl_t *dl;
  nd_state_t *state;
  uint8_t src[6];
  int i;

  assert(nd != NULL);

  gettimeofday_wrap(&nd->start);

  if((state = malloc_zero(sizeof(nd_state_t))) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc state");
      goto err;
    }

  if(scamper_if_getifindex(nd->ifname, &state->ifindex) != 0)
    {
      printerror(errno, strerror, __func__,
		 "could not get ifindex for %s", nd->ifname);
      goto err;
    }

  if((nd->src_ip = scamper_getsrc(nd->dst_ip, state->ifindex)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not get src ip");
      goto err;
    }

  if(scamper_if_getmac(state->ifindex, src) != 0)
    {
      printerror(errno, strerror, __func__, "could not get src mac");
      goto err;
    }

  if((nd->src_mac = scamper_addrcache_get_ethernet(addrcache, src)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not get src mac");
      goto err;
    }

  if((state->fd = scamper_fd_dl(state->ifindex)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not get fd");
      goto err;
    }

  if((dl = scamper_fd_write_state(state->fd)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not get dl");
      goto err;
    }

  if((i = scamper_dl_tx_type(dl)) != SCAMPER_DL_TX_ETHERNET)
    {
      scamper_debug(__func__, "dl type %d not ethernet", i);
      goto err;
    }

  task->state = state;
  return 0;

 err:
  if(state != NULL) nd_state_free(state);
  return -1;
}

static void do_nd_handle_timeout(scamper_task_t *task)
{
  scamper_neighbourdisc_t *nd = task->data;
  nd_state_t *state = task->state;

  if(((nd->flags & SCAMPER_NEIGHBOURDISC_FLAG_ALLATTEMPTS) == 0 &&
      nd->dst_mac != NULL) || nd->probec == nd->attempts ||
     (state->replyc >= nd->replyc && nd->replyc != 0))
    {
      nd_done(task);
      return;
    }

  return;
}

static void do_nd_probe_arp(scamper_task_t *task)
{
  scamper_neighbourdisc_t *nd = task->data;
  uint16_t u16;
  size_t off = 0;

  /*
   * standard 14 byte ethernet header followed by 28 byte arp request.
   *
   * 6 bytes: broadcast ethernet mac address
   * 6 bytes: src mac address
   * 2 bytes: ethernet type
   *
   * 2 bytes: ethernet address space: 0x0001
   * 2 bytes: protocol address space: 0x0800
   * 1 byte:  the length of an ethernet mac address: 6
   * 1 byte:  the length of an ip address: 4
   * 2 bytes: request packet: 0x0001
   * 6 bytes: src mac address
   * 4 bytes: src IP address
   * 6 bytes: dst mac address: all zeros in request
   * 4 bytes: dst IP address
   */
  memset(pktbuf, 0xff, 6); off = 6;
  memcpy(pktbuf+off, nd->src_mac->addr, 6); off += 6;
  u16 = htons(ETHERTYPE_ARP); memcpy(pktbuf+off, &u16, 2); off += 2;
  u16 = htons(0x0001); memcpy(pktbuf+off, &u16, 2); off += 2;
  u16 = htons(ETHERTYPE_IP); memcpy(pktbuf+off, &u16, 2); off += 2;
  pktbuf[off++] = 6;
  pktbuf[off++] = 4;
  u16 = htons(0x0001); memcpy(pktbuf+off, &u16, 2); off += 2;
  memcpy(pktbuf+off, nd->src_mac->addr, 6); off += 6;
  memcpy(pktbuf+off, nd->src_ip->addr, 4); off += 4;
  memset(pktbuf+off, 0, 6); off += 6;
  memcpy(pktbuf+off, nd->dst_ip->addr, 4); off += 4;

  return;
}

static void do_nd_probe_nsol(scamper_task_t *task)
{
  scamper_neighbourdisc_t *nd = task->data;
  struct ip6_hdr *ip6;
  struct icmp6_hdr *icmp6;
  uint16_t u16, *w;
  uint8_t ip6_dst[16];
  uint8_t sol[4];
  size_t off = 0;
  int i, sum = 0;

  /* figure out the lower 4 bytes of the solicited multicast address */
  memcpy(sol, ((uint8_t *)nd->dst_ip->addr)+12, 4);
  sol[0] = 0xff;

  /* figure out the destination IPv6 address of this message */
  ip6_dst[0] = 0xff;
  ip6_dst[1] = 0x02;
  memset(ip6_dst+2, 0, 9);
  ip6_dst[11] = 0x01;
  memcpy(ip6_dst+12, sol, 4);

  /* ethernet header: 14 bytes */
  pktbuf[off++] = 0x33;
  pktbuf[off++] = 0x33;
  memcpy(pktbuf+off, sol, 4); off += 4;
  memcpy(pktbuf+off, nd->src_mac->addr, 6); off += 6;
  u16 = htons(ETHERTYPE_IPV6); memcpy(pktbuf+off, &u16, 2); off += 2;

  /* IPv6 header: 40 bytes */
  ip6 = (struct ip6_hdr *)(pktbuf+off); off += sizeof(struct ip6_hdr);
  memset(ip6, 0, sizeof(struct ip6_hdr));
#ifndef _WIN32
  ip6->ip6_vfc  = 0x60;
#else
  ip6->ip6_vfc_flow = htonl(0x60000000);
#endif
  ip6->ip6_plen = htons(32);
  ip6->ip6_nxt  = IPPROTO_ICMPV6;
  ip6->ip6_hlim = 255;
  memcpy(&ip6->ip6_src, nd->src_ip->addr, 16);
  memcpy(&ip6->ip6_dst, ip6_dst, 16);

  /* ICMP6 neighbour discovery: 32 bytes */
  icmp6 = (struct icmp6_hdr *)(pktbuf+off); off += sizeof(struct icmp6_hdr);
  icmp6->icmp6_type = ND_NEIGHBOR_SOLICIT;
  icmp6->icmp6_code = 0;
#ifndef _WIN32
  icmp6->icmp6_data32[0] = 0;
#else
  icmp6->icmp6_data32 = 0;
#endif
  memcpy(pktbuf+off, nd->dst_ip->addr, 16); off += 16;
  pktbuf[off++] = 0x01;
  pktbuf[off++] = 0x01;
  memcpy(pktbuf+off, nd->src_mac->addr, 6); off += 6;
  icmp6->icmp6_cksum = 0;

  /* build up the ICMP6 checksum, which includes a psuedo header */
  w = (uint16_t *)&ip6->ip6_src;
  sum += *w++; sum += *w++; sum += *w++; sum += *w++;
  sum += *w++; sum += *w++; sum += *w++; sum += *w++;
  w = (uint16_t *)&ip6->ip6_dst;
  sum += *w++; sum += *w++; sum += *w++; sum += *w++;
  sum += *w++; sum += *w++; sum += *w++; sum += *w++;
  sum += ip6->ip6_plen;
  sum += htons(IPPROTO_ICMPV6);
  w = (uint16_t *)icmp6;
  for(i = ntohs(ip6->ip6_plen); i > 1; i -= 2)
    sum += *w++;
  if(i != 0)
    sum += ((uint8_t *)w)[0];
  sum  = (sum >> 16) + (sum & 0xffff);
  sum += (sum >> 16);
  if((u16 = ~sum) == 0)
    u16 = 0xffff;
  icmp6->icmp6_cksum = u16;

  return;
}

static void do_nd_handle_dl(scamper_task_t *task, scamper_dl_rec_t *dl)
{
  scamper_neighbourdisc_t *nd = task->data;
  nd_state_t *state = task->state;
  scamper_neighbourdisc_probe_t *probe;
  scamper_neighbourdisc_reply_t *reply;
  uint16_t opt_off;
  uint8_t *opt;
  uint8_t *mac = NULL;

#ifdef HAVE_SCAMPER_DEBUG
  char a[64], b[64];
#endif

  if(nd->probec == 0)
    return;
  probe = nd->probes[nd->probec-1];

  if(SCAMPER_DL_IS_ARP(dl))
    {
      if(nd->method != SCAMPER_NEIGHBOURDISC_METHOD_ARP ||
	 SCAMPER_DL_IS_ARP_OP_REPLY(dl) == 0 ||
	 SCAMPER_DL_IS_ARP_HRD_ETHERNET(dl) == 0 ||
	 SCAMPER_DL_IS_ARP_PRO_IPV4(dl) == 0)
	{
	  return;
	}

      mac = dl->dl_arp_sha;
    }
  else if(SCAMPER_DL_IS_ICMPV6(dl))
    {
      if(nd->method != SCAMPER_NEIGHBOURDISC_METHOD_ND_NSOL ||
	 SCAMPER_DL_IS_ICMP6_ND_NADV(dl) == 0)
	{
	  return;
	}

      if((reply = scamper_neighbourdisc_reply_alloc()) == NULL)
	{
	  printerror(errno, strerror, __func__, "could not alloc reply");
	  goto err;
	}
      timeval_cpy(&reply->rx, &dl->dl_tv);      

      /*
       * loop through the attached options, trying to find the
       * destination link-address option
       */
      opt = dl->dl_icmp6_nd_opts;
      opt_off = 0;

      while(opt_off + 8 <= dl->dl_icmp6_nd_opts_len)
	{
	  if(opt[0]==2 && opt[1]==1 && dl->dl_icmp6_nd_opts_len-opt_off >= 8)
	    {
	      mac = opt+2;
	      break;
	    }
	  if(opt[1] == 0)
	    return;
	  opt_off += (opt[1] * 8);
	  opt     += (opt[1] * 8);
	}
    }

  if(mac == NULL)
    return;

  if((reply = scamper_neighbourdisc_reply_alloc()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not alloc reply");
      goto err;
    }
  timeval_cpy(&reply->rx, &dl->dl_tv);
  reply->mac = scamper_addrcache_get_ethernet(addrcache, mac);
  if(reply->mac == NULL)
    {
      printerror(errno, strerror, __func__, "could not get reply->mac");
      goto err;
    }

  scamper_debug(__func__, "%s is-at %s",
		scamper_addr_tostr(nd->dst_ip, a, sizeof(a)),
		scamper_addr_tostr(reply->mac, b, sizeof(b)));
  do_nd_cbs(state, reply->mac);

  if(scamper_neighbourdisc_reply_add(probe, reply) != 0)
    {
      printerror(errno, strerror, __func__, "could not add reply");
      goto err;
    }

  if(nd->dst_mac == NULL)
    nd->dst_mac = scamper_addr_use(reply->mac);

  if(probe->rxc == 1)
    {
      state->replyc++;
      if((nd->flags & SCAMPER_NEIGHBOURDISC_FLAG_FIRSTRESPONSE) != 0)
	nd_done(task);
    }

  return;

 err:
  return;
}

static void do_nd_probe(scamper_task_t *task)
{
  scamper_neighbourdisc_probe_t *probe = NULL;
  scamper_neighbourdisc_t *nd = task->data;
  nd_state_t   *state = task->state;
  scamper_dl_t *dl;
  size_t len;
  char ip[64], mac[32];

  if(state == NULL)
    {
      if(nd_state_alloc(task) != 0)
	goto err;
      state = task->state;
    }

  /* determine the length of the packet to transmit */
  if(nd->method == SCAMPER_NEIGHBOURDISC_METHOD_ARP)
    len = 42;
  else if(nd->method == SCAMPER_NEIGHBOURDISC_METHOD_ND_NSOL)
    len = 86;
  else goto err;

  /* make sure the pktbuf is at least that size */
  if(pktbuf_len < len)
    {
      if(realloc_wrap((void **)&pktbuf, len) != 0)
	{
	  printerror(errno, strerror, __func__, "could not realloc");
	  goto err;
	}
      pktbuf_len = len;
    }

  /* form the probe to send */
  if(nd->method == SCAMPER_NEIGHBOURDISC_METHOD_ARP)
    do_nd_probe_arp(task);
  else if(nd->method == SCAMPER_NEIGHBOURDISC_METHOD_ND_NSOL)
    do_nd_probe_nsol(task);
  else goto err;

  /* allocate a probe record to store tx time and associated replies */
  if((probe = scamper_neighbourdisc_probe_alloc()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not alloc probe");
      goto err;
    }

  /* send the probe.  record the time it is sent */
  dl = scamper_fd_write_state(state->fd);
  gettimeofday_wrap(&probe->tx);
  if(scamper_dl_tx(dl, pktbuf, len) == -1)
    {
      goto err;
    }

  scamper_addr_tostr(nd->dst_ip, ip, sizeof(ip));
  scamper_addr_tostr(nd->src_mac, mac, sizeof(mac));
  scamper_debug(__func__, "who-has %s tell %s", ip, mac);

  if(scamper_neighbourdisc_probe_add(nd, probe) != 0)
    {
      printerror(errno, strerror, __func__, "could not add probe");
      goto err;
    }

  scamper_queue_wait(task->queue, nd->wait);
  return;

 err:
  nd_handleerror(task, errno);
  return;
}

static void do_nd_write(scamper_file_t *sf, scamper_task_t *task)
{
  scamper_file_write_neighbourdisc(sf, task->data);
  return;
}

static void do_nd_free(scamper_task_t *task)
{
  scamper_neighbourdisc_t *nd = task->data;
  nd_state_t *state = task->state;
  scamper_task_t *task0 = NULL;
  scamper_addr_t *dst = NULL;

  if(nd != NULL && state != NULL && state->task0 != NULL)
    {
      dst = scamper_addr_use(nd->dst_ip); assert(dst != NULL);
      task0 = state->task0;
    }

  if(nd != NULL)
    scamper_neighbourdisc_free(nd);

  if(state != NULL)
    nd_state_free(state);

  if(task0 != NULL)
    {
      scamper_target_attach(task0, dst);
      scamper_addr_free(dst);
    }

  return;
}

static int nd_arg_param_validate(int optid, char *param, long *out)
{
  long tmp;

  switch(optid)
    {
    case ND_OPT_IFNAME:
    case ND_OPT_ALLATTEMPTS:
      return 0;

    case ND_OPT_ATTEMPTS:
      if(string_tolong(param, &tmp) != 0 || tmp < 1 || tmp > 65535)
       return -1;
      break;

    case ND_OPT_REPLYC:
      if(string_tolong(param, &tmp) != 0 || tmp < 0 || tmp > 65535)
	return -1;
      break;

    case ND_OPT_WAIT:
      if(string_tolong(param, &tmp) != 0 || tmp < 100 || tmp > 65535)
	return -1;
      break;

    default:
      return -1;
    }

  if(out != NULL)
    *out = tmp;

  return 0;
}

int scamper_do_neighbourdisc_arg_validate(int argc, char *argv[], int *stop)
{
  return scamper_options_validate(opts, opts_cnt, argc, argv, stop,
				  nd_arg_param_validate);
}

scamper_task_t *scamper_do_neighbourdisc_alloctask(void *data,
						   scamper_list_t *list,
						   scamper_cycle_t *cycle)
{
  scamper_neighbourdisc_t *nd = (scamper_neighbourdisc_t *)data;
  scamper_task_t *task;

  if((task = scamper_task_alloc(data, &nd_funcs)) == NULL)
    goto err;

  /* now, associate the list and cycle with the ping */
  nd->list  = scamper_list_use(list);
  nd->cycle = scamper_cycle_use(cycle);

  return task;

 err:
  return NULL;
}

void *scamper_do_neighbourdisc_alloc(char *str)
{
  scamper_neighbourdisc_t *nd = NULL;
  scamper_option_out_t *opts_out = NULL, *opt;
  char    *ifname   = NULL;
  uint16_t attempts = 1;
  uint16_t replyc   = 0;
  uint16_t wait     = 1000;
  uint8_t  flags    = 0;
  char    *dst      = NULL;
  long     tmp      = 0;

  /* try and parse the string passed in */
  if(scamper_options_parse(str, opts, opts_cnt, &opts_out, &dst) != 0)
    {
      scamper_debug(__func__, "could not parse command");
      goto err;
    }

  if(dst == NULL)
    goto err;

  for(opt = opts_out; opt != NULL; opt = opt->next)
    {
      if(opt->type != SCAMPER_OPTION_TYPE_NULL &&
	 nd_arg_param_validate(opt->id, opt->str, &tmp) != 0)
	{
	  scamper_debug(__func__, "validation of optid %d failed", opt->id);
	  goto err;
	}

      switch(opt->id)
	{
	case ND_OPT_FIRSTRESPONSE:
	  flags |= SCAMPER_NEIGHBOURDISC_FLAG_FIRSTRESPONSE;
	  break;

	case ND_OPT_IFNAME:
	  ifname = opt->str;
	  break;

	case ND_OPT_ATTEMPTS:
	  attempts = (uint16_t)tmp;
	  break;

	case ND_OPT_REPLYC:
	  replyc = (uint16_t)tmp;
	  break;

	case ND_OPT_ALLATTEMPTS:
	  flags |= SCAMPER_NEIGHBOURDISC_FLAG_ALLATTEMPTS;
	  break;

	case ND_OPT_WAIT:
	  wait = (uint16_t)tmp;
	  break;

	default:
	  scamper_debug(__func__, "unhandled option %d", opt->id);
	  goto err;
	}
    }

  scamper_options_free(opts_out); opts_out = NULL;

  if(ifname == NULL)
    goto err;

  if((nd = scamper_neighbourdisc_alloc()) == NULL)
    goto err;

  if((nd->dst_ip = scamper_addrcache_resolve(addrcache,AF_UNSPEC,dst)) == NULL)
    goto err;

  if(nd->dst_ip->type == SCAMPER_ADDR_TYPE_IPV4)
    nd->method = SCAMPER_NEIGHBOURDISC_METHOD_ARP;
  else if(nd->dst_ip->type == SCAMPER_ADDR_TYPE_IPV6)
    nd->method = SCAMPER_NEIGHBOURDISC_METHOD_ND_NSOL;
  else
    goto err;

  if(scamper_neighbourdisc_ifname_set(nd, ifname) != 0)
    goto err;

  /*
   * if we only want the first response, then we can't want more than
   * one reply
   */
  if((flags & SCAMPER_NEIGHBOURDISC_FLAG_FIRSTRESPONSE) != 0 &&
     (replyc > 1 || (flags & SCAMPER_NEIGHBOURDISC_FLAG_ALLATTEMPTS) != 0))
    {
      scamper_debug(__func__, "invalid combination of arguments");
      goto err;
    }

  /*
   * if we have asked for all attempts to be sent, but we have limited the
   * number of replies we want to less than the number of probes we will send,
   * then these arguments are conflicting
   */
  if(replyc < attempts && replyc != 0 &&
     (flags & SCAMPER_NEIGHBOURDISC_FLAG_ALLATTEMPTS) != 0)
    {
      scamper_debug(__func__, "invalid combination of arguments");
      goto err;
    }

  nd->flags    = flags;
  nd->attempts = attempts;
  nd->replyc   = replyc;
  nd->wait     = wait;

  return nd;

 err:
  if(nd != NULL) scamper_neighbourdisc_free(nd);
  if(opts_out != NULL) scamper_options_free(opts_out);
  return NULL;
}

int scamper_do_neighbourdisc_do(int ifindex, scamper_addr_t *dst, void *param,
				void (*cb)(void *param, scamper_addr_t *dst))
{
  scamper_neighbourdisc_t *nd = NULL;
  scamper_task_t *task = NULL;
  scamper_task_t *task0;
  nd_state_t *state;
  uint8_t method;
  ndcb_t *ndcb = NULL;
  char ifname[64];

  if(dst->type == SCAMPER_ADDR_TYPE_IPV4)
    method = SCAMPER_NEIGHBOURDISC_METHOD_ARP;
  else if(dst->type == SCAMPER_ADDR_TYPE_IPV6)
    method = SCAMPER_NEIGHBOURDISC_METHOD_ND_NSOL;
  else
    {
      scamper_debug(__func__, "unhandled dst->type %d", dst->type);
      goto err;
    }

  /*
   * see if anything else is using this address as their target.  if so,
   * and it is a neighbour discovery process, then piggy back on that.
   */
  task0 = scamper_target_find(dst);
  if(task0 != NULL && task0->funcs == &nd_funcs)
    {
      state = task0->state;
      if(state->cbs == NULL && (state->cbs = dlist_alloc()) == NULL)
	{
	  printerror(errno, strerror, __func__, "could not alloc state->cbs");
	  goto err;
	}
      if((ndcb = malloc_zero(sizeof(ndcb_t))) == NULL)
	{
	  printerror(errno, strerror, __func__, "could not alloc ndcb");
	  goto err;
	}
      ndcb->cb = cb;
      ndcb->param = param;
      if((ndcb->node = dlist_tail_push(state->cbs, ndcb)) == NULL)
	{
	  printerror(errno, strerror, __func__, "could not add ndcb");
	  goto err;
	}

      return 0;
    }

  if(scamper_if_getifname(ifname, sizeof(ifname), ifindex) != 0)
    {
      /* called function prints error */
      goto err;
    }

  if((nd = scamper_neighbourdisc_alloc()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not alloc nd");
      goto err;
    }
  if(scamper_neighbourdisc_ifname_set(nd, ifname) != 0)
    {
      printerror(errno, strerror, __func__, "could not set ifname");
      goto err;
    }

  nd->method    = method;
  nd->dst_ip    = scamper_addr_use(dst);
  nd->flags    |= SCAMPER_NEIGHBOURDISC_FLAG_FIRSTRESPONSE;
  nd->attempts  = 2;
  nd->wait      = 250;
  nd->replyc    = 1;

  if((task = scamper_task_alloc(nd, &nd_funcs)) == NULL)
    {
      /* called function prints error */
      goto err;
    }

  if(nd_state_alloc(task) != 0)
    {
      /* called function prints error */
      goto err;
    }
  state = task->state;

  if(state->cbs == NULL && (state->cbs = dlist_alloc()) == NULL)
    {
      printerror(errno, strerror, __func__, "could not alloc state->cbs");
      goto err;
    }
  if((ndcb = malloc_zero(sizeof(ndcb_t))) == NULL)
    {
      printerror(errno, strerror, __func__, "could not alloc ndcb");
      goto err;
    }
  ndcb->cb = cb;
  ndcb->param = param;
  if((ndcb->node = dlist_tail_push(state->cbs, ndcb)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not add ndcb");
      goto err;
    }

  if(task0 != NULL)
    {
      state->task0 = task0;
      if(scamper_target_detach(task0, dst) != 0)
	{
	  /* called function prints error */
	  goto err;
	}
    }

  if((task->targetset = scamper_targetset_alloc(task)) == NULL)
    {
      /* called function prints error */
      goto err;
    }

  if(scamper_queue_probe_head(task->queue) != 0)
    {
      /* called function prints error */
      goto err;
    }

  return 0;

 err:
  return -1;
}

int scamper_do_neighbourdisc_dstaddr(void *data, void *param,
				     int (*each)(struct scamper_addr *,void *))
{
  scamper_neighbourdisc_t *nd = (scamper_neighbourdisc_t *)data;
  return each(nd->dst_ip, param);
}

void scamper_do_neighbourdisc_free(void *data)
{
  scamper_neighbourdisc_free((scamper_neighbourdisc_t *)data);
  return;
}

void scamper_do_neighbourdisc_cleanup()
{
  if(pktbuf != NULL)
    {
      free(pktbuf);
      pktbuf = NULL;
    }

  return;
}

int scamper_do_neighbourdisc_init()
{
  nd_funcs.probe          = do_nd_probe;
  nd_funcs.handle_timeout = do_nd_handle_timeout;
  nd_funcs.write          = do_nd_write;
  nd_funcs.task_free      = do_nd_free;
  nd_funcs.task_addrs     = scamper_do_neighbourdisc_dstaddr;
  nd_funcs.handle_dl      = do_nd_handle_dl;

  return 0;
}
