/*
 * scamper_probe.c
 *
 * $Id: scamper_probe.c,v 1.50 2010/04/07 01:16:29 mjl Exp $
 *
 * Copyright (C) 2005-2010 The University of Waikato
 * Author: Matthew Luckie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef lint
static const char rcsid[] =
  "$Id: scamper_probe.c,v 1.50 2010/04/07 01:16:29 mjl Exp $";
#endif

#include <sys/types.h>

#if defined(_MSC_VER)
typedef unsigned __int8 uint8_t;
typedef unsigned __int16 uint16_t;
typedef unsigned __int32 uint32_t;
typedef __int16 int16_t;
#define __func__ __FUNCTION__
#endif

#ifdef _WIN32
#include <winsock2.h>
#include <ws2tcpip.h>
#endif

#ifndef _WIN32
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#if defined(__APPLE__)
#include <stdint.h>
#endif

#if defined(DMALLOC)
#include <dmalloc.h>
#endif

#include <assert.h>

#include "scamper.h"
#include "scamper_addr.h"
#include "scamper_fds.h"
#include "scamper_dl.h"
#include "scamper_probe.h"
#include "scamper_udp4.h"
#include "scamper_udp6.h"
#include "scamper_icmp_resp.h"
#include "scamper_icmp4.h"
#include "scamper_icmp6.h"
#include "scamper_tcp4.h"
#include "scamper_tcp6.h"
#include "scamper_dl.h"
#include "scamper_debug.h"
#include "utils.h"

#ifndef TH_FIN
#define TH_FIN 0x01
#endif

#ifndef TH_SYN
#define TH_SYN 0x02
#endif

#ifndef TH_RST
#define TH_RST 0x04
#endif

#ifndef	TH_PUSH
#define TH_PUSH 0x08
#endif

#ifndef TH_ACK
#define TH_ACK 0x10
#endif

#ifndef	TH_URG
#define TH_URG 0x20
#endif

#ifndef TH_ECE
#define TH_ECE 0x40
#endif

#ifndef TH_CWR
#define TH_CWR 0x80
#endif

#ifndef ICMP_UNREACH
#define ICMP_UNREACH 3
#endif

#ifndef ICMP_ECHO
#define ICMP_ECHO 8
#endif

#ifndef ICMP_UNREACH_NEEDFRAG
#define ICMP_UNREACH_NEEDFRAG 4
#endif

#ifndef ICMP6_DST_UNREACH
#define ICMP6_DST_UNREACH 1
#endif

#ifndef ICMP6_PACKET_TOO_BIG
#define ICMP6_PACKET_TOO_BIG 2
#endif

#ifndef ICMP6_ECHO_REQUEST
#define ICMP6_ECHO_REQUEST 128
#endif

/*
 * this pad macro determines the number of extra bytes we have to allocate
 * so that the next element (the IP header) of the buffer is aligned
 * appropriately after the datalink header.
 */
#define PAD(s) ((s > 0) ? (1 + ((s - 1) | (sizeof(long) - 1)) - s) : 0)

static uint8_t *pktbuf = NULL;
static size_t   pktbuf_len = 0;

#ifdef HAVE_SCAMPER_DEBUG
static char *tcp_flags(char *buf, size_t len, scamper_probe_t *probe)
{
  uint8_t flags = probe->pr_tcp_flags;
  uint8_t flag;
  size_t off;
  int i;

  buf[0] = '\0';
  if(probe->pr_len != 0)
    {
      flags &= ~(TH_ACK);
    }

  off = 0;
  for(i=0; i<8 && flags != 0; i++)
    {
      flag = 1 << i;

      if((flags & flag) == 0)
	continue;
      flags &= ~flag;

      switch(flag)
	{
	case TH_SYN:
	  string_concat(buf, len, &off, " syn");
	  break;

	case TH_RST:
	  string_concat(buf, len, &off, " rst");
	  break;

	case TH_FIN:
	  string_concat(buf, len, &off, " fin");
	  break;

	case TH_ACK:
	  string_concat(buf, len, &off, " ack");
	  break;

	case TH_PUSH:
	  string_concat(buf, len, &off, " psh");
	  break;

	case TH_URG:
	  string_concat(buf, len, &off, " urg");
	  break;

	case TH_ECE:
	  string_concat(buf, len, &off, " ece");
	  break;

	case TH_CWR:
	  string_concat(buf, len, &off, " cwr");
	  break;
	}
    }

  return buf;
}

static char *tcp_pos(char *buf, size_t len, scamper_probe_t *probe)
{
  size_t off = 0;

  string_concat(buf, len, &off, "%u", probe->pr_tcp_seq);
  if(probe->pr_tcp_flags & TH_ACK)
    string_concat(buf, len, &off, ":%u", probe->pr_tcp_ack);

  return buf;
}
#endif

/*
 * scamper_probe_send
 *
 * this meta-function is responsible for
 *  1. sending a probe
 *  2. handling any error condition incurred when sending the probe
 *  3. recording details of the probe with the trace's state
 */
int scamper_probe(scamper_probe_t *probe)
{
  int    (*send_func)(scamper_probe_t *);
  int    (*build_func)(scamper_probe_t *, uint8_t *, size_t *);
  size_t   pad, len;
  uint8_t *buf;

#ifdef HAVE_SCAMPER_DEBUG
  char tcp[16];
  char pos[32];
  char addr[128];
  char icmp[16];
  scamper_addr_tostr(probe->pr_ip_dst, addr, sizeof(addr));
#endif

  send_func = NULL;
  build_func = NULL;

  probe->pr_errno = 0;

  /* determine which function scamper should use to build or send the probe */
  if(probe->pr_ip_dst->type == SCAMPER_ADDR_TYPE_IPV4)
    {
      switch(probe->pr_ip_proto)
	{
	case IPPROTO_UDP:
	  send_func = scamper_udp4_probe;
	  build_func = scamper_udp4_build;
	  scamper_debug(__func__, "udp %s, ttl %d, %d:%d, len %d",
			addr, probe->pr_ip_ttl,
			probe->pr_udp_sport, probe->pr_udp_dport,
			probe->pr_len + 28);
	  break;

	case IPPROTO_TCP:
	  build_func = scamper_tcp4_build;
	  scamper_debug(__func__,
			"tcp %s, ttl %d, %d:%d%s, ipid %04x, %s, len %d",
			addr, probe->pr_ip_ttl,
			probe->pr_tcp_sport, probe->pr_tcp_dport,
			tcp_flags(tcp, sizeof(tcp), probe),
			probe->pr_ip_id, tcp_pos(pos, sizeof(pos), probe),
			probe->pr_len + 20 + scamper_tcp4_hlen(probe));
	  break;

	case IPPROTO_ICMP:
	  send_func = scamper_icmp4_probe;
	  build_func = scamper_icmp4_build;
	  if(probe->pr_icmp_type == ICMP_ECHO)
	    {
	      if(probe->pr_icmp_sum != 0)
		{
		  snprintf(icmp, sizeof(icmp), ", sum %04x",
			   ntohs(probe->pr_icmp_sum));
		}
	      else icmp[0] = '\0';
	      scamper_debug(__func__, "icmp %s echo, ttl %d%s, seq %d, len %d",
			    addr, probe->pr_ip_ttl, icmp, probe->pr_icmp_seq,
			    probe->pr_len + 28);
	    }
	  else if(probe->pr_icmp_type == ICMP_UNREACH)
	    {
	      if(probe->pr_icmp_code == ICMP_UNREACH_NEEDFRAG)
		snprintf(icmp,sizeof(icmp),"ptb %d", probe->pr_icmp_mtu);
	      else
		snprintf(icmp,sizeof(icmp),"unreach %d", probe->pr_icmp_code);
	      scamper_debug(__func__, "icmp %s %s, len %d",
			    addr, icmp, probe->pr_len + 28);
	    }
	  else
	    {
	      scamper_debug(__func__, "icmp %s type %d, code %d, len %d",
			    addr, probe->pr_icmp_type, probe->pr_icmp_code,
			    probe->pr_len + 28);
	    }
	  break;
	}
    }
  else if(probe->pr_ip_dst->type == SCAMPER_ADDR_TYPE_IPV6)
    {
      switch(probe->pr_ip_proto)
	{
	case IPPROTO_UDP:
	  send_func = scamper_udp6_probe;
	  build_func = scamper_udp6_build;
	  scamper_debug(__func__, "udp %s, ttl %d, %d:%d, len %d",
			addr, probe->pr_ip_ttl,
			probe->pr_udp_sport, probe->pr_udp_dport,
			probe->pr_len + 48);
	  break;

	case IPPROTO_TCP:
	  build_func = scamper_tcp6_build;
	  scamper_debug(__func__, "tcp %s ttl %d, %d:%d%s, %s, len %d",
			addr, probe->pr_ip_ttl,
			probe->pr_tcp_sport, probe->pr_tcp_dport,
			tcp_flags(tcp, sizeof(tcp), probe),
			tcp_pos(pos, sizeof(pos), probe),
			probe->pr_len + 40 + scamper_tcp6_hlen(probe));
	  break;

	case IPPROTO_ICMPV6:
	  send_func = scamper_icmp6_probe;
	  build_func = scamper_icmp6_build;
	  if(probe->pr_icmp_type == ICMP6_ECHO_REQUEST)
	    {
	      if(probe->pr_icmp_sum != 0)
		{
		  snprintf(icmp, sizeof(icmp), ", sum %04x",
			   ntohs(probe->pr_icmp_sum));
		}
	      else icmp[0] = '\0';
	      scamper_debug(__func__, "icmp %s echo, ttl %d%s, seq %d, len %d",
			    addr, probe->pr_ip_ttl, icmp, probe->pr_icmp_seq,
			    probe->pr_len + 48);
	    }
	  else if(probe->pr_icmp_type == ICMP6_PACKET_TOO_BIG)
	    {
	      scamper_debug(__func__, "icmp %s ptb %d, len %d",
			    addr, probe->pr_icmp_mtu, probe->pr_len + 48);
	    }
	  else if(probe->pr_icmp_type == ICMP6_DST_UNREACH)
	    {
	      scamper_debug(__func__, "icmp %s unreach %d, len %d",
			    addr, probe->pr_icmp_code, probe->pr_len + 48);
	    }
	  else 
	    {
	      scamper_debug(__func__, "icmp %s type %d, code %d, len %d",
			    addr, probe->pr_icmp_type, probe->pr_icmp_code,
			    probe->pr_len + 48);
	    }
	  break;
	}
    }

  /* if we're not using the datalink to send the packet, then send it now */
  if(probe->pr_dl == NULL)
    {
      if(send_func != NULL)
	{
	  return send_func(probe);
	}

      probe->pr_errno = EINVAL;
      return -1;
    }

  /* if the header type is not known (we cannot build it) then bail */
  if(build_func == NULL)
    {
      probe->pr_errno = EINVAL;
      return -1;
    }

  /*
   * calculate the number of pad bytes to put at the front of the packet
   * buffer so that the IP layer is properly aligned for the architecture
   */
  pad = PAD(probe->pr_dl_len);

  /* determine a suitable value for the length parameter */
  if(pad + probe->pr_dl_len >= pktbuf_len)
    {
      len = 0;
    }
  else
    {
      len = pktbuf_len - pad - probe->pr_dl_len;
    }

  /*
   * try building the probe.  if it returns -1, then hopefully the len field
   * will supply a clue as to what it should be
   */
  if(build_func(probe, pktbuf + pad + probe->pr_dl_len, &len) == -1)
    {
      assert(pktbuf_len < pad + probe->pr_dl_len + len);

      /* reallocate the packet buffer */
      len += pad + probe->pr_dl_len;
      if((buf = realloc(pktbuf, len)) == NULL)
	{
	  probe->pr_errno = errno;
	  printerror(errno, strerror, __func__, "could not realloc");
	  return -1;
	}
      pktbuf     = buf;
      pktbuf_len = len;

      len = pktbuf_len - pad - probe->pr_dl_len;
      if(build_func(probe, pktbuf + pad + probe->pr_dl_len, &len) == -1)
	{
	  probe->pr_errno = EINVAL;
	  return -1;
	}
    }

  /* add the datalink header size back to the length field */
  len += probe->pr_dl_len;

  /* pre-pend the datalink header, if there is one */
  if(probe->pr_dl_len > 0)
    {
      memcpy(pktbuf+pad, probe->pr_dl_buf, probe->pr_dl_len);
    }

  gettimeofday_wrap(&probe->pr_tx);
  if(scamper_dl_tx(probe->pr_dl, pktbuf+pad, len) == -1)
    {
      probe->pr_errno = errno;
      return -1;
    }

  probe->pr_tx_raw    = pktbuf + pad + probe->pr_dl_len;
  probe->pr_tx_rawlen = len - probe->pr_dl_len;

  return 0;
}

void scamper_probe_cleanup()
{
  if(pktbuf != NULL)
    {
      free(pktbuf);
      pktbuf = NULL;
    }
  
  pktbuf_len = 0;
  return;
}
