; Scheme 9 from Empty Space
; Test Suite
; By Nils M Holm, 2007, 2008, 2009

;  This is a comment

(define testfile "__testfile__")

(if (file-exists? testfile)
    (wrong (string-append "Please delete the file \""
                          testfile
                          "\" before running this test.")))

(define Errors 0)

(define (void) (if #f #f))

(define (seq)
  (let ((n 1))
    (lambda ()
      (let ((x n))
        (set! n (+ 1 n))
        x))))

(define (fail expr result expected)
  (display "test failed: ")
  (write expr)
  (newline)
  (display "got result:  ")
  (write result)
  (newline)
  (display "expected:    ")
  (write expected)
  (newline)
  (set! Errors (+ 1 Errors)))

(define (test3 expr result expected)
;  (write expr) (display " => ") (write result) (newline)
  (if (not (equal? result expected))
      (fail expr result expected)))

(define-macro (test form result)
  `(test3 ',form ,form ,result))

; --- syntax ---

; symbols

(test 'x 'x)
(test 'mississippi 'mississippi)
(test 'MIssissiPPi 'mississippi)
(test '!$%&*+-./^_ '!$%&*+-./^_)

; booleans

(test #t #t)
(test #f #f)

; characters

(test #\x #\x)
(test #\C #\C)
(test #\( #\()
(test #\) #\))
(test #\; #\;)
(test #\space #\space)
(test #\newline #\newline)

; strings

(test "test" "test")
(test "TeSt" "TeSt")
(test "TEST" "TEST")
(test "hello, world!" "hello, world!")
(test "\"hello, world!\"" "\"hello, world!\"")
(test "a\\/b" "a\\/b")
(test "(((;)))" "(((;)))")

; pairs -- erm, well

(test '() '())
(test '(a b c) '(a b c))
(test '(a (b) c) '(a (b) c))
(test '(((((x))))) '(((((x))))))
(test '((caar . cdar) . (cadr . cddr)) '((caar . cdar) . (cadr . cddr)))

; vectors

(test '#() '#())
(test '#(a b c) '#(a b c))
(test '#(a (b) c) '#(a (b) c))
(test '#(((((x))))) '#(((((x))))))
(test '#((caar cadar) (caadr  cadadr)) '#((caar cadar) (caadr  cadadr)))
(test '#(#(a b c) #(d e f)) '#(#(a b c) #(d e f)))
(test '#(#(#(#(#(x))))) '#(#(#(#(#(x))))))

; numbers -- integers

(test 0 0)
(test 1 1)
(test 1234567 1234567)
(test -0 0)
(test -1 -1)
(test -1234567 -1234567)
(test 123456789012345678901234567890 123456789012345678901234567890)
(test -123456789012345678901234567890 -123456789012345678901234567890)

(test #b10101010100101010101   698709)
(test #b+10101010100101010101 +698709)
(test #b-10101010100101010101 -698709)
(test #d1234567890987654321   1234567890987654321)
(test #d+1234567890987654321 +1234567890987654321)
(test #d-1234567890987654321 -1234567890987654321)
(test #o123456707654321   5744369817809)
(test #o+123456707654321 +5744369817809)
(test #o-123456707654321 -5744369817809)
(test #x123456789abcdef0fedcba98765432   94522879700260683132212139638805554)
(test #x+123456789abcdef0fedcba98765432 +94522879700260683132212139638805554)
(test #x-123456789abcdef0fedcba98765432 -94522879700260683132212139638805554)

; numbers -- reals

(test 0.0 0.0)
(test -0.0 -0.0)
(test 1.0 1.0)
(test -1.0 -1.0)
(test 12345.0 12345.0)
(test -12345.0 -12345.0)
(test 1.2345 1.2345)
(test -1.2345 -1.2345)
(test 0.12345 0.12345)
(test -0.12345 -0.12345)
(test -0.00012345 -0.00012345)
(test 0.1 0.1)
(test 0.01 0.01)
(test 0.001 0.001)
(test 0.0000000000001 0.0000000000001)
(test 1e0 1.0)
(test 1e-0 1.0)
(test 1e1 10.0)
(test 1e2 100.0)
(test 1e5 100000.0)
(test 1e10 10000000000.0)
(test 1e-1 0.1)
(test 1e-2 0.01)
(test 1e-5 0.00001)
(test 1e-10 0.0000000001)
(test 123.456e0 123.456)
(test 123.456e1 1234.56)
(test 123.456e2 12345.6)
(test 123.456e3 123456.0)
(test 123.456e4 1234560.0)
(test 123.456e5 12345600.0)
(test 123.456e10 1234560000000.0)
(test -123.456e0 -123.456)
(test -123.456e1 -1234.56)
(test -123.456e2 -12345.6)
(test -123.456e3 -123456.0)
(test -123.456e4 -1234560.0)
(test -123.456e5 -12345600.0)
(test -123.456e10 -1234560000000.0)
(test 123.456e-1 12.3456)
(test 123.456e-2 1.23456)
(test 123.456e-3 0.123456)
(test 123.456e-4 0.0123456)
(test 123.456e-5 0.00123456)
(test 123.456e-10 0.0000000123456)
(test -123.456e-1 -12.3456)
(test -123.456e-2 -1.23456)
(test -123.456e-3 -0.123456)
(test -123.456e-4 -0.0123456)
(test -123.456e-5 -0.00123456)
(test -123.456e-10 -0.0000000123456)
(test +123.45e+678 123.45e678)
(test -123.45e+678 -123.45e678)
(test +123.45e-678 123.45e-678)
(test -123.45e-678 -123.45e-678)
(test 1. 1.0)
(test .1 0.1)
(test 1.e1 10.0)
(test .1e1 1.0)
(test 1000e0 1e3)
(test 100e1 1e3)
(test 10e2 1e3)
(test 1e3 1e3)
(test .1e4 1e3)
(test .01e5 1e3)
(test .001e6 1e3)
(test 12345678.901D10 1.2345678901e+17)
(test 12345678.901E10 1.2345678901e+17)
(test 12345678.901F10 1.2345678901e+17)
(test 12345678.901L10 1.2345678901e+17)
(test 12345678.901S10 1.2345678901e+17)

; and
(test (and) #t)
(test (and #f) #f)
(test (and #f #f) #f)
(test (and #f #t) #f)
(test (and #t #f) #f)
(test (and #t #t) #t)
(test (and 1 2 3) 3)
(test (and #f 2 3) #f)
(test (and 1 #f 3) #f)
(test (and 1 2 #f) #f)
(test (and 'foo) 'foo)
(test (and #t) #t)
(test (and 1) 1)
(test (and #\x) #\x)
(test (and "x") "x")
(test (and '(x)) '(x))
(test (and '()) '())
(test (and '#(x)) '#(x))
(test (and (lambda (x) x) #t) #t)

; begin
(test (begin 1) 1)
(test (begin 1 "2") "2")
(test (begin 1 "2" #\3) #\3)
(test (let ((x (seq)) (y 0))
         (begin (set! y (- y (x)))
                (set! y (- y (x)))
                (set! y (- y (x))))
                y)
       -6)

; cond
;(test (cond) (void))
(test (cond (#t 1)) 1)
(test (cond (1 1)) 1)
(test (cond ('x 1)) 1)
(test (cond (#\x 1)) 1)
(test (cond ("x" 1)) 1)
(test (cond ('(a b c) 1)) 1)
(test (cond ('() 1)) 1)
(test (cond (#(1 2 3) 1)) 1)
(test (cond (#f 1)) (void))
(test (cond (#f 1) (#t 2)) 2)
(test (cond (#f 1) (else 2)) 2)
(test (cond (else 2)) 2)
(test (cond (#t 1 2 3)) 3)
(test (cond (else 1 2 3)) 3)
(test (cond (#f (#f))) (void))
(test (cond (#f)) (void))
(test (cond (#f) (#t)) #t)
(test (cond (1 => list)) '(1))
(test (cond (#f => list) (#t => list)) '(#t))

; define
(define x 'foo)
(test (let () (define x 1) x) 1)
(test ((lambda () (define x 0) x)) 0)
(test (begin ((lambda () (define x 0) x)) x) 'foo)
(test (begin (let () (define x 0) x) x) 'foo)
(test (begin (let () (define x 0) x)) 0)
(test (let () (letrec () (define x 0) x) x) 'foo)
(test (let () (letrec () (define x 0) x)) 0)
(test (let () (define (f) 1) (f)) 1)
(test (let () (define (f x) x) (f 1)) 1)
(test (let () (define (f x y) x) (f 1 2)) 1)
(test (let () (define (f x y) y) (f 1 2)) 2)
(test (let () (define (f . x) x) (f)) '())
(test (let () (define (f . x) x) (f 1)) '(1))
(test (let () (define (f . x) x) (f 1 2)) '(1 2))
(test (let () (define (f x . y) y) (f 1 2)) '(2))
(test (let () (define f (lambda () 1)) (f)) 1)
(test (let () (define f (lambda (x) x)) (f 1)) 1)
(test (let () (define f (lambda (x y) x)) (f 1 2)) 1)
(test (let () (define f (lambda (x y) y)) (f 1 2)) 2)
(test (let () (define f (lambda x x)) (f)) '())
(test (let () (define f (lambda x x)) (f 1)) '(1))
(test (let () (define f (lambda x x)) (f 1 2)) '(1 2))
(test (let () (define f (lambda (x . y) y)) (f 1 2)) '(2))
(test ((lambda ()
          (define (e x) (or (zero? x) (o (- x 1))))
          (define (o x) (if (zero? x) #f (e (- x 1))))
          (list (o 5) (e 5))))
      '(#t #f))

; if
(test (if #f #f) (void))
(test (if #t 1) 1)
(test (if 1 1) 1)
(test (if 'a 1) 1)
(test (if #\a 1) 1)
(test (if "a" 1) 1)
(test (if '(1 2 3) 1) 1)
(test (if '() 1) 1)
(test (if '#(1 2 3) 1) 1)
(test (if #t 1 2) 1)
(test (if #f 1 2) 2)
(test (if #f (#f)) (void))

; lambda
(test ((lambda () '())) '())
(test ((lambda (x) x) 1) 1)
(test ((lambda (x y z) (list x y z)) 1 2 3) '(1 2 3))

(test (((lambda (x) (lambda (y) (cons x y))) 1) 2) '(1 . 2))

(test ((lambda (a . b) a) 'foo) 'foo)
(test ((lambda (a . b) b) 'foo) '())
(test ((lambda (a . b) b) 'foo 'bar) '(bar))
(test ((lambda (a . b) b) 'foo 'bar 'baz) '(bar baz))

(test ((lambda (a b . c) a) 'foo 'bar) 'foo)
(test ((lambda (a b . c) b) 'foo 'bar) 'bar)
(test ((lambda (a b . c) c) 'foo 'bar) '())
(test ((lambda (a b . c) c) 'foo 'bar 'baz) '(baz))

(test ((lambda a a)) '())
(test ((lambda a a) 'foo) '(foo))
(test ((lambda a a) 'foo 'bar) '(foo bar))
(test ((lambda a a) 'foo 'bar 'baz) '(foo bar baz))

(test ((lambda (x) ((lambda () x))) 1) 1)

(test ((lambda () 1 2 3)) 3)

(test ((lambda (x) ((lambda () (set! x 1))) x) 0) 1)

; let
(test (let () 1) 1)
(test (let () 1 2 3) 3)
(test (let ((x 1)) x) 1)
(test (let ((x 1) (y 2) (z 3)) (list x y z)) '(1 2 3))

(test (let ((x 0))
         (let ((x 1)
               (y (* x 1)))
           y))
       0)
(test (let ((x 0))
         (let ((x 1))
           (let ((y (* x 1)))
             y)))
       1)

; letrec
(test (letrec () 1) 1)
(test (letrec () 1 2 3) 3)
(test (letrec ((x 1)) x) 1)
(test (letrec ((x 1) (y 2) (z 3)) (list x y z)) '(1 2 3))

(test (letrec
         ((even-p
            (lambda (x)
              (or (null? x) (odd-p (cdr x)))))
          (odd-p
            (lambda (x)
              (if (null? x) #f (even-p (cdr x))))))
          (list (odd-p '(i i i i i))
                (even-p '(i i i i i))))
      '(#t #f))

(test (let* () 1) 1)
(test (let* () 1 2 3) 3)
(test (let* ((x 'first)) x) 'first)
(test (let* ((x 'first) (y 'second) (z 'third)) (list x y z))
      '(first second third))
(test (let* ((x 0))
         (let* ((x 1)
                (y (* x 5)))
           y))
       5)
(test (let* ((x 3)
              (y (cons 2 x))
              (z (cons 1 y)))
         z)
      '(1 2 . 3))
(test (let* ((x 3)
              (x (cons 2 x))
              (x (cons 1 x)))
         x)
      '(1 2 . 3))

; or
(test (or) #f)
(test (or #f) #f)
(test (or #f #f) #f)
(test (or #f #t) #t)
(test (or #t #f) #t)
(test (or #t #t) #t)
(test (or 1 2 3) 1)
(test (or #f 2 3) 2)
(test (or 1 #f 3) 1)
(test (or #f #f 3) 3)
(test (or 'foo) 'foo)
(test (or #t) #t)
(test (or 1) 1)
(test (or #\x) #\x)
(test (or "x") "x")
(test (or '(x)) '(x))
(test (or '()) '())
(test (or '#(x)) '#(x))

; quote
(test (quote foo) 'foo)
(test (quote quote) 'quote)
(test (quote #t) #t)
(test (quote 1) 1)
(test (quote #\X) #\X)
(test (quote "abc") "abc")
(test (quote ()) '())
(test (quote (1 2 3)) '(1 2 3))
(test (quote #(1 2 3)) '#(1 2 3))
(test (quote (lambda (x) x)) '(lambda (x) x))
(test '1 '1)
(test ''1 ''1)
(test '''1 '''1)
(test '#f #f)
(test '1 1)
(test '#\b #\b)
(test '"abc" "abc")

; --- setters ---

(define x 0)
(test (begin (set! x 1) x) 1)
(test (begin ((lambda (x) (set! x 0)) 'void) x) 1)
(test (begin (let ((x 'void)) (set! x 0)) x) 1)
(test (begin (let* ((x 'void)) (set! x 0)) x) 1)
(test (begin (letrec ((x 'void)) (set! x 0)) x) 1)
(test (begin (set! x 2) x) 2)

(define p (cons 1 2))
(test (begin (set-car! p 'a) p) '(a . 2))
(test (begin (set-cdr! p 'b) p) '(a . b))

; --- type predicates ---

(test (boolean? #f) #t)
(test (boolean? #\c) #f)
(test (boolean? 1) #f)
(test (boolean? 0.1) #f)
(test (boolean? '(pair)) #f)
(test (boolean? (lambda () #f)) #f)
(test (boolean? "string") #f)
(test (boolean? 'symbol) #f)
(test (boolean? '#(vector)) #f)
(test (boolean? (current-input-port)) #f)
(test (boolean? (current-output-port)) #f)

(test (char? #f) #f)
(test (char? #\c) #t)
(test (char? 1) #f)
(test (char? 0.1) #f)
(test (char? '(pair)) #f)
(test (char? (lambda () #f)) #f)
(test (char? "string") #f)
(test (char? 'symbol) #f)
(test (char? '#(vector)) #f)
(test (char? (current-input-port)) #f)
(test (char? (current-output-port)) #f)

(test (input-port? #f) #f)
(test (input-port? #\c) #f)
(test (input-port? 1) #f)
(test (input-port? 0.1) #f)
(test (input-port? '(pair)) #f)
(test (input-port? (lambda () #f)) #f)
(test (input-port? "string") #f)
(test (input-port? 'symbol) #f)
(test (input-port? '#(vector)) #f)
(test (input-port? (current-input-port)) #t)
(test (input-port? (current-output-port)) #f)

(test (integer? #f) #f)
(test (integer? #\c) #f)
(test (integer? 1) #t)
(test (integer? 1.0) #t)
(test (integer? 0.1) #f)
(test (integer? '(pair)) #f)
(test (integer? (lambda () #f)) #f)
(test (integer? "string") #f)
(test (integer? 'symbol) #f)
(test (integer? '#(vector)) #f)
(test (integer? (current-input-port)) #f)
(test (integer? (current-output-port)) #f)

(test (number? #f) #f)
(test (number? #\c) #f)
(test (number? 1) #t)
(test (number? 0.1) #t)
(test (number? '(pair)) #f)
(test (number? (lambda () #f)) #f)
(test (number? "string") #f)
(test (number? 'symbol) #f)
(test (number? '#(vector)) #f)
(test (number? (current-input-port)) #f)
(test (number? (current-output-port)) #f)

(test (output-port? #f) #f)
(test (output-port? #\c) #f)
(test (output-port? 1) #f)
(test (output-port? 0.1) #f)
(test (output-port? '(pair)) #f)
(test (output-port? (lambda () #f)) #f)
(test (output-port? "string") #f)
(test (output-port? 'symbol) #f)
(test (output-port? '#(vector)) #f)
(test (output-port? (current-input-port)) #f)
(test (output-port? (current-output-port)) #t)

(test (pair? #f) #f)
(test (pair? #\c) #f)
(test (pair? 1) #f)
(test (pair? 0.1) #f)
(test (pair? '(pair)) #t)
(test (pair? (lambda () #f)) #f)
(test (pair? "string") #f)
(test (pair? 'symbol) #f)
(test (pair? '#(vector)) #f)
(test (pair? (current-input-port)) #f)
(test (pair? (current-output-port)) #f)

(test (port? #f) #f)
(test (port? #\c) #f)
(test (port? 1) #f)
(test (port? 0.1) #f)
(test (port? '(pair)) #f)
(test (port? (lambda () #f)) #f)
(test (port? "string") #f)
(test (port? 'symbol) #f)
(test (port? '#(vector)) #f)
(test (port? (current-input-port)) #t)
(test (port? (current-output-port)) #t)

(test (procedure? #f) #f)
(test (procedure? #\c) #f)
(test (procedure? 1) #f)
(test (procedure? 0.1) #f)
(test (procedure? '(procedure)) #f)
(test (procedure? (lambda () #f)) #t)
(test (procedure? "string") #f)
(test (procedure? 'symbol) #f)
(test (procedure? '#(vector)) #f)
(test (procedure? (current-input-port)) #f)
(test (procedure? (current-output-port)) #f)

(test (string? #f) #f)
(test (string? #\c) #f)
(test (string? 1) #f)
(test (string? 0.1) #f)
(test (string? '(pair)) #f)
(test (string? (lambda () #f)) #f)
(test (string? "string") #t)
(test (string? 'symbol) #f)
(test (string? '#(vector)) #f)
(test (string? (current-input-port)) #f)
(test (string? (current-output-port)) #f)

(test (symbol? #f) #f)
(test (symbol? #\c) #f)
(test (symbol? 1) #f)
(test (symbol? 0.1) #f)
(test (symbol? '(pair)) #f)
(test (symbol? (lambda () #f)) #f)
(test (symbol? "string") #f)
(test (symbol? 'symbol) #t)
(test (symbol? '#(vector)) #f)
(test (symbol? (current-input-port)) #f)
(test (symbol? (current-output-port)) #f)

(test (vector? #f) #f)
(test (vector? #\c) #f)
(test (vector? 1) #f)
(test (vector? 0.1) #f)
(test (vector? '(pair)) #f)
(test (vector? (lambda () #f)) #f)
(test (vector? "string") #f)
(test (vector? 'symbol) #f)
(test (vector? '#(vector)) #t)
(test (vector? (current-input-port)) #f)
(test (vector? (current-output-port)) #f)

; --- conversion procedures ---

(test (char->integer #\A) 65)
(test (char->integer #\z) 122)
(test (char->integer #\newline) 10)
(test (char->integer #\space) 32)

(test (integer->char 65) #\A)
(test (integer->char 122) #\z)
(test (integer->char 10) #\newline)
(test (integer->char 32) #\space)
(test (integer->char 65.0) #\A)

(test (list->string '(#\S #\t #\r #\i #\n #\g)) "String")
(test (list->string '()) "")

(test (list->vector '(#t foo 1 #\c "s" (1 2 3) #(u v)))
      '#(#t foo 1 #\c "s" (1 2 3) #(u v)))
(test (list->vector '()) '#())

(test (string->list "String") '(#\S #\t #\r #\i #\n #\g))
(test (string->list "") '())

(test (string->symbol "foo") 'foo)
(test (string->symbol "string->symbol") 'string->symbol)

(test (symbol->string 'foo) "foo")
(test (symbol->string 'symbol->string) "symbol->string")
(test (symbol->string (string->symbol "miSSissiPPi")) "miSSissiPPi")

(test (eq? (string->symbol "foo") 'foo) #t)

(test (vector->list '#(#t foo 1 #\c "s" (1 2 3) #(u v)))
      '(#t foo 1 #\c "s" (1 2 3) #(u v)))
(test (vector->list '#()) '())

; --- more control ---

(test (apply (lambda () 1) '()) 1)
(test (apply car '((a . b))) 'a)
(test (apply cdr '((a . b))) 'b)
(test (apply cons '(1 2)) '(1 . 2))
(test (apply list '(1 2 3)) '(1 2 3))
(test (apply list 1 '(2 3)) '(1 2 3))
(test (apply list 1 2 '(3)) '(1 2 3))
(test (apply list 1 2 3 '()) '(1 2 3))

(test (case 'a ((a b) 'first) ((c d) 'second)) 'first)
(test (case 'b ((a b) 'first) ((c d) 'second)) 'first)
(test (case 'c ((a b) 'first) ((c d) 'second)) 'second)
(test (case 'd ((a b) 'first) ((c d) 'second)) 'second)
(test (case 'x ((a b) 'first) ((c d) 'second)) (void))
(test (case 'x ((a b) 'first) (else 'default)) 'default)
(test (case 'd ((a) 'a) ((b) 'b) ((c) 'c) (else 'default)) 'default)
(test (case 'c ((a) 'a) ((b) 'b) ((c) 'c) (else 'default)) 'c)
(test (case 'b ((a) 'a) ((b) 'b) ((c) 'c) (else 'default)) 'b)
(test (case 'a ((a) 'a) ((b) 'b) ((c) 'c) (else 'default)) 'a)
(test (case 'x ((a) 'a) ((b) 'b) ((c) 'c) (else 'default)) 'default)
(test (case 'x ((b) 'b) ((c) 'c) (else 'default)) 'default)
(test (case 'x ((c) 'c) (else 'default)) 'default)
(test (case 'x (else 'default)) 'default)
(test (case 'x (else 1 2 3)) 3)
(test (case 'x ((y) #f)) (void))

(test (do () (#t 123)) 123)
(test (do () (#t)) (void))
(test (do ((i 1 (+ 1 i))) ((= i 10) i) i) 10)
(test (do ((i 1 (+ 1 i)) (j 17)) ((= i 10) j) i) 17)
(test (do ((i 1 (+ 1 i)) (j 2 (+ 2 j))) ((= i 10) j) i) 20)
(test (do ((i 1 (+ 1 i)) (j 2 (+ 2 j))) ((= i 10) (* i j)) i) 200)
(test (let ((j 1)) (do ((i 0 (+ 1 i))) ((= i 10) j) (set! j (+ j 3)))) 31)
(test (do ((i 1 (+ 1 i)) (j 0)) ((= i 10) j) (set! j 1)) 1)
(test (do ((i 1 (+ 1 i)) (j 0)) ((= i 10) j) 1 2 3 (set! j 1)) 1)

(test (let ((a (list (list 'a) (list 'b) (list 'c))))
         (for-each (lambda (x) (set-car! x 'x)) a)
         a)
      '((x) (x) (x)))
(test (let ((a (list (list 'a) (list 'b) (list 'c))))
         (for-each (lambda (x y) (set-car! x y)) a '(x y z))
         a)
      '((x) (y) (z)))

(define s (seq))
(s)
(define x (delay (s)))
(test (list (force x) (force x) (force x)) '(2 2 2))

(test (map - '(1 2 3)) '(-1 -2 -3))
(test (map cons '(1 2 3) '(a b c))
      '((1 . a) (2 . b) (3 . c)))
(test (map list '(1 2 3) '(a b c) '(#\x #\y #\z))
      '((1 a #\x) (2 b #\y) (3 c #\z)))

; --- quasiquotation ---

(define x 'foo)
(test `x 'x)
(test `,x 'foo)
(test `(1 2 3) '(1 2 3))
(test `(y ,x z) '(y foo z))
(test `(1 2 3 ,(list 4 5)) '(1 2 3 (4 5)))
(test `(1 2 3 ,@(list 4 5)) '(1 2 3 4 5))
(test `#(y ,x z) '#(y foo z))
(test `#(1 2 3 ,(list 4 5)) '#(1 2 3 (4 5)))
(test `#(1 2 3 ,@(list 4 5)) '#(1 2 3 4 5))
(test `(a b c (,x y z)) '(a b c (foo y z)))
(test `(a b c (,x ,@(list 'y 'z))) '(a b c (foo y z)))

; --- lists ---

(test (append '() '(a b c)) '(a b c))
(test (append '(a b c) '()) '(a b c))
(test (append '() '()) '())
(test (append) '())
(test (append '(a b)) '(a b))
(test (append '(a b) '(c d)) '(a b c d))
(test (append '(a b) '(c d) '(e f)) '(a b c d e f))
(test (append '(a b) 'c) '(a b . c))
(test (append '(a) 'b) '(a . b))
(test (append 'a) 'a)

(test (assoc 'c '((a . a) (b . b))) #f)
(test (assoc 'b '((a . a) (b . b))) '(b . b))
(test (assoc 'a '((a . a) (b . b))) '(a . a))
(test (assoc 'x '()) #f)
(test (assoc '(x) '(((x) . x))) '((x) . x))
(test (assoc "x" '(("x" . x))) '("x" . x))
(test (assoc 1 '((1 . x))) '(1 . x))
(test (assoc #\x '((#\x . x))) '(#\x . x))

(test (assv 'c '((a . a) (b . b))) #f)
(test (assv 'b '((a . a) (b . b))) '(b . b))
(test (assv 'a '((a . a) (b . b))) '(a . a))
(test (assv 'x '()) #f)
(test (assv '(x) '(((x) . x))) #f)
(test (assv "x" '(("x" . x))) #f)
(test (assv 1 '((1 . x))) '(1 . x))
(test (assv #\x '((#\x . x))) '(#\x . x))

(test (assq 'c '((a . a) (b . b))) #f)
(test (assq 'b '((a . a) (b . b))) '(b . b))
(test (assq 'a '((a . a) (b . b))) '(a . a))
(test (assq 'x '()) #f)
(test (assq '(x) '(((x) . x))) #f)
(test (assq "x" '(("x" . x))) #f)

(define tree '((((1 . 2) . (3 . 4)) . ((5 . 6) . (7 . 8)))
              .
              (((9 . 10) . (11 . 12)) . ((13 . 14) . (15 . 16)))))
(test (caar tree) '((1 . 2) . (3 . 4)))
(test (cadr tree) '((9 . 10) . (11 . 12)))
(test (cdar tree) '((5 . 6) . (7 . 8)))
(test (cddr tree) '((13 . 14) . (15 . 16)))
(test (caaar tree) '(1 . 2))
(test (caadr tree) '(9 . 10))
(test (cadar tree) '(5 . 6))
(test (caddr tree) '(13 . 14))
(test (cdaar tree) '(3 . 4))
(test (cdadr tree) '(11 . 12))
(test (cddar tree) '(7 . 8))
(test (cdddr tree) '(15 . 16))
(test (caaaar tree) 1)
(test (caaadr tree) 9)
(test (caadar tree) 5)
(test (caaddr tree) 13)
(test (cadaar tree) 3)
(test (cadadr tree) 11)
(test (caddar tree) 7)
(test (cadddr tree) 15)
(test (cdaaar tree) 2)
(test (cdaadr tree) 10)
(test (cdadar tree) 6)
(test (cdaddr tree) 14)
(test (cddaar tree) 4)
(test (cddadr tree) 12)
(test (cdddar tree) 8)
(test (cddddr tree) 16)

(test (car '(1 1)) 1)
(test (car '(1 . 2)) 1)
(test (cdr '(1 2)) '(2))
(test (cdr '(1 . 2)) 2)
(test (cons 1 2) '(1 . 2))
(test (cons 1 '(2)) '(1 2))
(test (cons 1 (cons 2 '())) '(1 2))

(test (length '()) 0)
(test (length '(1)) 1)
(test (length '(1 2 3)) 3)

(test (list) '())
(test (list '()) '(()))
(test (list 'x) '(x))
(test (list (list 'x)) '((x)))
(test (list 'a 'b) '(a b))
(test (list 'a 'b 'c) '(a b c))
(test (list 'a 'b 'c 'd) '(a b c d))
(test (list 'a 'b 'c 'd 'e) '(a b c d e))

(test (list-ref '(1 2 3) 0) 1)
(test (list-ref '(1 2 3) 1) 2)
(test (list-ref '(1 2 3) 2) 3)

(test (list-tail '(1 2 3) 0) '(1 2 3))
(test (list-tail '(1 2 3) 1) '(2 3))
(test (list-tail '(1 2 3) 2) '(3))
(test (list-tail '(1 2 3) 3) '())

(test (list? #f) #f)
(test (list? #\c) #f)
(test (list? 1) #f)
(test (list? '(pair)) #t)
(test (list? (lambda () #f)) #f)
(test (list? "string") #f)
(test (list? 'symbol) #f)
(test (list? '#(vector)) #f)
(test (list? (current-input-port)) #f)
(test (list? (current-output-port)) #f)
(test (list? '()) #t)
(test (list? '(1)) #t)
(test (list? '(1 . ())) #t)
(test (list? '(1 2 3)) #t)
(test (list? '(1 . 2)) #f)
(test (list? '(1 2 . 3)) #f)
(let ((cyclic (list 1 2 3)))
  (set-cdr! (cddr cyclic) cyclic)
  (if (list? cyclic)
      (fail '(list? 'cyclic) #t)
      (test (list? 'cyclic) #f)))

(test (member 'c '(a b)) #f)
(test (member 'b '(a b)) '(b))
(test (member 'a '(a b)) '(a b))
(test (member 'x '()) #f)
(test (member '(x) '((x))) '((x)))
(test (member "x" '("x")) '("x"))
(test (member 1 '(1)) '(1))
(test (member #\x '(#\x)) '(#\x))

(test (memv 'c '(a b)) #f)
(test (memv 'b '(a b)) '(b))
(test (memv 'a '(a b)) '(a b))
(test (memv 'x '()) #f)
(test (memv '(x) '((x))) #f)
(test (memv "x" '("x")) #f)
(test (memv 1 '(1)) '(1))
(test (memv #\x '(#\x)) '(#\x))

(test (memq 'c '(a b)) #f)
(test (memq 'b '(a b)) '(b))
(test (memq 'a '(a b)) '(a b))
(test (memq 'x '()) #f)
(test (memq '(x) '((x))) #f)
(test (memq "x" '("x")) #f)

(test (null? #f) #f)
(test (null? #\c) #f)
(test (null? 1) #f)
(test (null? '(pair)) #f)
(test (null? (lambda () #f)) #f)
(test (null? "string") #f)
(test (null? 'symbol) #f)
(test (null? '#(vector)) #f)
(test (null? (current-input-port)) #f)
(test (null? (current-output-port)) #f)
(test (null? '()) #t)

(test (reverse '(1)) '(1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse '()) '())

; --- arithmetics ---

(test (+  1234567890  9876543210)  11111111100)
(test (+  1234567890 -9876543210)  -8641975320)
(test (+ -1234567890  9876543210)   8641975320)
(test (+ -1234567890 -9876543210) -11111111100)
(test (+  9876543210  1234567890)  11111111100)
(test (+  9876543210 -1234567890)   8641975320)
(test (+ -9876543210  1234567890)  -8641975320)
(test (+ -9876543210 -1234567890) -11111111100)
(test (+ 1234567890 0) 1234567890)
(test (+ 0 1234567890) 1234567890)
(test (+ 1 2 3 4 5 6 7 8 9 10) 55)
(test (+ 1) 1)
(test (+) 0)

(test (+  0.0) 0.0)
(test (+  1.0) 1.0)
(test (+ -1.0) -1.0)
(test (+ 0.0 1234567890123.4) 1234567890123.4)
(test (+ 1234567890123.4 0.0) 1234567890123.4)
(test (+  123.45  123.45) 246.9)
(test (+  123.45 -123.45) 0.0)
(test (+ -123.45  123.45) 0.0)
(test (+ -123.45 -123.45) -246.9)
(test (+  1e10  12345.67) 1.000001234567e10)
(test (+  1e10 -12345.67) 9.99998765433e9)
(test (+ -1e10  12345.67) -9.99998765433e9)
(test (+ -1e10 -12345.67) -1.000001234567e10)
(test (+  1e-10  12345.67) 12345.6700000001)
(test (+  1e-10 -12345.67) -12345.6699999999)
(test (+ -1e-10  12345.67) 12345.6699999999)
(test (+ -1e-10 -12345.67) -12345.6700000001)
(test (+  12345.67  1e10) 1.000001234567e10)
(test (+  12345.67 -1e10) -9.99998765433e9)
(test (+ -12345.67  1e10) 9.99998765433e9)
(test (+ -12345.67 -1e10) -1.000001234567e10)
(test (+  12345.67  1e-10) 12345.6700000001)
(test (+  12345.67 -1e-10) 12345.6699999999)
(test (+ -12345.67  1e-10) -12345.6699999999)
(test (+ -12345.67 -1e-10) -12345.6700000001)
(test (+ 999999999.9 1) 1000000000.9)
(test (+ 1 999999999.9) 1000000000.9)
(test (+ 1000000000.9 -1) 999999999.9)
(test (+ -1 1000000000.9) 999999999.9)
(test (+ 12345.67  1234567) 1246912.67)
(test (+ 12345.67 123456.7) 135802.37)
(test (+ 12345.67 12345.67) 24691.34)
(test (+ 12345.67 1234.567) 13580.237)
(test (+ 12345.67 123.4567) 12469.1267)
(test (+ 12345.67 12.34567) 12358.01567)
(test (+ 12345.67 1.234567) 12346.904567)
(test (+ 12345.67 .1234567) 12345.7934567)
(test (+  1234567 12345.67) 1246912.67)
(test (+ 123456.7 12345.67) 135802.37)
(test (+ 12345.67 12345.67) 24691.34)
(test (+ 1234.567 12345.67) 13580.237)
(test (+ 123.4567 12345.67) 12469.1267)
(test (+ 12.34567 12345.67) 12358.01567)
(test (+ 1.234567 12345.67) 12346.904567)
(test (+ .1234567 12345.67) 12345.7934567)
(test (+ 1.1 2.2 3.3 4.4 5.5) 16.5)
(test (exact? (+   1.0   1.0)) #t)
(test (exact? (+ #i1.0   1.0)) #f)
(test (exact? (+   1.0 #i1.0)) #f)
(test (exact? (+ #i1.0 #i1.0)) #f)

(test (-  1234567890  9876543210)  -8641975320)
(test (-  1234567890 -9876543210)  11111111100)
(test (- -1234567890  9876543210) -11111111100)
(test (- -1234567890 -9876543210)   8641975320)
(test (-  9876543210  1234567890)   8641975320)
(test (-  9876543210 -1234567890)  11111111100)
(test (- -9876543210  1234567890) -11111111100)
(test (- -9876543210 -1234567890)  -8641975320)
(test (- 1234567890 0) 1234567890)
(test (- 0 1234567890) -1234567890)
(test (- 1 2 3 4 5 6 7 8 9 10) -53)
(test (- 1234567890) -1234567890)
(test (- 0) 0)

(test (-  0.0) 0.0)
(test (-  1.0) -1.0)
(test (- -1.0) 1.0)
(test (- 0.0 1234567890123.4) -1234567890123.4)
(test (- 1234567890123.4 0.0) 1234567890123.4)
(test (-  123.45  123.45) 0.0)
(test (-  123.45 -123.45) 246.9)
(test (- -123.45  123.45) -246.9)
(test (- -123.45 -123.45) 0.0)
(test (-  1e10  12345.67) 9.99998765433e9)
(test (-  1e10 -12345.67) 1.000001234567e10)
(test (- -1e10  12345.67) -1.000001234567e10)
(test (- -1e10 -12345.67) -9.99998765433e9)
(test (-  1e-10  12345.67) -12345.6699999999)
(test (-  1e-10 -12345.67) 12345.6700000001)
(test (- -1e-10  12345.67) -12345.6700000001)
(test (- -1e-10 -12345.67) 12345.6699999999)
(test (-  12345.67  1e10) -9.99998765433e9)
(test (-  12345.67 -1e10) 1.000001234567e10)
(test (- -12345.67  1e10) -1.000001234567e10)
(test (- -12345.67 -1e10) 9.99998765433e9)
(test (-  12345.67  1e-10) 12345.6699999999)
(test (-  12345.67 -1e-10) 12345.6700000001)
(test (- -12345.67  1e-10) -12345.6700000001)
(test (- -12345.67 -1e-10) -12345.6699999999)
(test (- 999999999.9 -1) 1000000000.9)
(test (- -1 999999999.9) -1000000000.9)
(test (- 1000000000.9 1) 999999999.9)
(test (- 1 1000000000.9) -999999999.9)
(test (- 12345.67  1234567) -1222221.33)
(test (- 12345.67 123456.7) -111111.03)
(test (- 12345.67 12345.67) 0.0)
(test (- 12345.67 1234.567) 11111.103)
(test (- 12345.67 123.4567) 12222.2133)
(test (- 12345.67 12.34567) 12333.32433)
(test (- 12345.67 1.234567) 12344.435433)
(test (- 12345.67 .1234567) 12345.5465433)
(test (-  1234567 12345.67) 1222221.33)
(test (- 123456.7 12345.67) 111111.03)
(test (- 12345.67 12345.67) 0.0)
(test (- 1234.567 12345.67) -11111.103)
(test (- 123.4567 12345.67) -12222.2133)
(test (- 12.34567 12345.67) -12333.32433)
(test (- 1.234567 12345.67) -12344.435433)
(test (- .1234567 12345.67) -12345.5465433)
(test (- 1.1 2.2 3.3 4.4 5.5) -14.3)
(test (exact? (-   2.0   1.0)) #t)
(test (exact? (- #i2.0   1.0)) #f)
(test (exact? (-   2.0 #i1.0)) #f)
(test (exact? (- #i2.0 #i1.0)) #f)

(test (*  1234567  7654321)  9449772114007)
(test (*  1234567 -7654321) -9449772114007)
(test (* -1234567  7654321) -9449772114007)
(test (* -1234567 -7654321)  9449772114007)
(test (*  7654321  1234567)  9449772114007)
(test (*  7654321 -1234567) -9449772114007)
(test (* -7654321  1234567) -9449772114007)
(test (* -7654321 -1234567)  9449772114007)
(test (* 1234567 1) 1234567)
(test (* 1 1234567) 1234567)
(test (* 1234567 0) 0)
(test (* 0 1234567) 0)
(test (* 1 2 3 4 5 6 7 8 9 10) 3628800)
(test (* 1 2 3 4 5 6 7 8 9) 362880)
(test (* 2) 2)
(test (*) 1)

(test (* 0.0  0.0)  0.0)
(test (* 0.0  0.1)  0.0)
(test (* 0.0  1.0)  0.0)
(test (* 0.0 -0.0)  0.0)
(test (* 0.0 -0.1)  0.0)
(test (* 0.0 -1.0)  0.0)
(test (* 0.1  0.0)  0.0)
(test (* 0.1  0.1)  0.01)
(test (* 0.1  1.0)  0.1)
(test (* 0.1 -0.0)  0.0)
(test (* 0.1 -0.1) -0.01)
(test (* 0.1 -1.0) -0.1)
(test (* 1.0  0.0)  0.0)
(test (* 1.0  0.1)  0.1)
(test (* 1.0  1.0)  1.0)
(test (* 1.0 -0.0)  0.0)
(test (* 1.0 -0.1) -0.1)
(test (* 1.0 -1.0) -1.0)
(test (*  123.45  123.45) 15239.9025)
(test (*  123.45 -123.45) -15239.9025)
(test (* -123.45  123.45) -15239.9025)
(test (* -123.45 -123.45) 15239.9025)
(test (*  123.45e+100  123.45e+100) 1.52399025e204)
(test (*  123.45e+100 -123.45e+100) -1.52399025e204)
(test (* -123.45e+100  123.45e+100) -1.52399025e204)
(test (* -123.45e+100 -123.45e+100) 1.52399025e204)
(test (*  123.45e-100  123.45e-100) 1.52399025e-196)
(test (*  123.45e-100 -123.45e-100) -1.52399025e-196)
(test (* -123.45e-100  123.45e-100) -1.52399025e-196)
(test (* -123.45e-100 -123.45e-100) 1.52399025e-196)
(test (* 12345.67 .1234567) 1524.155677489)
(test (* 12345.67 1.234567) 15241.55677489)
(test (* 12345.67 12.34567) 152415.5677489)
(test (* 12345.67 123.4567) 1524155.677489)
(test (* 12345.67 1234.567) 15241556.77489)
(test (* 12345.67 12345.67) 152415567.7489)
(test (* 12345.67 123456.7) 1.524155677489e9)
(test (* 12345.67 1234567.) 1.524155677489e10)
(test (* -.1234567 12345.67) -1524.155677489)
(test (* -1.234567 12345.67) -15241.55677489)
(test (* -12.34567 12345.67) -152415.5677489)
(test (* -123.4567 12345.67) -1524155.677489)
(test (* -1234.567 12345.67) -15241556.77489)
(test (* -12345.67 12345.67) -152415567.7489)
(test (* -123456.7 12345.67) -1.524155677489e9)
(test (* -1234567. 12345.67) -1.524155677489e10)
(test (* 1.0   2   3   4   5   6   7   8   9) 362880.0)
(test (* 1.1 2.2 3.3 4.4 5.5 6.6 7.7 8.8 9.9) 855652.05811008)

(test (/ 0.0  0.1) 0.0)
(test (/ 0.0  1.0) 0.0)
(test (/ 0.0 -0.1) 0.0)
(test (/ 0.0 -1.0) 0.0)
(test (/ 0.1  0.1) 1.0)
(test (/ 0.1  1.0) 0.1)
(test (/ 0.1 -0.1) -1.0)
(test (/ 0.1 -1.0) -0.1)
(test (/ 1.0  0.1) 10.0)
(test (/ 1.0  1.0) 1.0)
(test (/ 1.0 -0.1) -10.0)
(test (/ 1.0 -1.0) -1.0)
(test (/  12345.  123.45) 100.0)
(test (/  12345. -123.45) -100.0)
(test (/ -12345.  123.45) -100.0)
(test (/ -12345. -123.45) 100.0)
(test (/  152337.3  123.4) 1234.5)
(test (/  152337.3 -123.4) -1234.5)
(test (/ -152337.3  123.4) -1234.5)
(test (/ -152337.3 -123.4) 1234.5)
(test (/  1.52399025e+204  123.45e+100) 123.45e+100)
(test (/ -1.52399025e+204 -123.45e+100) 123.45e+100)
(test (/ -1.52399025e+204  123.45e+100) -123.45e+100)
(test (/  1.52399025e+204 -123.45e+100) -123.45e+100)
(test (/  1.52399025e-196  123.45e-100) 123.45e-100)
(test (/ -1.52399025e-196 -123.45e-100) 123.45e-100)
(test (/ -1.52399025e-196  123.45e-100) -123.45e-100)
(test (/  1.52399025e-196 -123.45e-100) -123.45e-100)
(test (/ 12345.67 .1234567) 100000.0)
(test (/ 12345.67 1.234567) 10000.0)
(test (/ 12345.67 12.34567) 1000.0)
(test (/ 12345.67 123.4567) 100.0)
(test (/ 12345.67 1234.567) 10.0)
(test (/ 12345.67 12345.67) 1.0)
(test (/ 12345.67 123456.7) 0.1)
(test (/ 12345.67 1234567.) 0.01)
(test (/ -.1234567 12345.67) -0.00001)
(test (/ -1.234567 12345.67) -0.0001)
(test (/ -12.34567 12345.67) -0.001)
(test (/ -123.4567 12345.67) -0.01)
(test (/ -1234.567 12345.67) -0.1)
(test (/ -12345.67 12345.67) -1.0)
(test (/ -123456.7 12345.67) -10.0)
(test (/ -1234567. 12345.67) -100.0)
(test (/ 1.0) 1.0)
(test (/ 2.0) 0.5)
(test (/ 5.0) 0.2)
(test (/ 128.0 64 32 16 8 4 2 1) 6.103515625e-5)

(test (<  1234567890  9876543210) #t)
(test (<  1234567890 -9876543210) #f)
(test (< -1234567890  9876543210) #t)
(test (< -1234567890 -9876543210) #f)
(test (<  9876543210  1234567890) #f)
(test (<  9876543210 -1234567890) #f)
(test (< -9876543210  1234567890) #t)
(test (< -9876543210 -1234567890) #t)
(test (< -1234567890 -1234567890) #f)
(test (<  1234567890  1234567890) #f)
(test (< 1234567890 0) #f)
(test (< 0 1234567890) #t)
(test (< 1 2 3 4 5 6 7 8 9 10) #t)
(test (< 1 2 3 4 5 6 7 8 9 9) #f)

(test (<  0.0  0.0) #f)
(test (< -0.0  0.0) #f)
(test (<  0.0 -0.0) #f)
(test (< -0.0 -0.0) #f)
(test (<  1.0  1.0) #f)
(test (< -1.0  1.0) #t)
(test (<  1.0 -1.0) #f)
(test (< -1.0 -1.0) #f)
(test (<  0.1  0.1) #f)
(test (< -0.1  0.1) #t)
(test (<  0.1 -0.1) #f)
(test (< -0.1 -0.1) #f)
(test (<  123.45e+67  123.45e+67) #f)
(test (<  123.45e+67  123.45e-67) #f)
(test (<  123.45e-67  123.45e+67) #t)
(test (<  123.45e-67  123.45e-67) #f)
(test (<  123.45e+67 -123.45e+67) #f)
(test (<  123.45e+67 -123.45e-67) #f)
(test (<  123.45e-67 -123.45e+67) #f)
(test (<  123.45e-67 -123.45e-67) #f)
(test (< -123.45e+67  123.45e+67) #t)
(test (< -123.45e+67  123.45e-67) #t)
(test (< -123.45e-67  123.45e+67) #t)
(test (< -123.45e-67  123.45e-67) #t)
(test (< -123.45e+67 -123.45e+67) #f)
(test (< -123.45e+67 -123.45e-67) #t)
(test (< -123.45e-67 -123.45e+67) #f)
(test (< -123.45e-67 -123.45e-67) #f)
(test (<  0.0 0.0) #f)
(test (<  1.0 0.0) #f)
(test (< -1.0 0.0) #t)
(test (<  0.1 0.0) #f)
(test (< -0.1 0.0) #t)
(test (<  1e+0 0.0) #f)
(test (< -1e+0 0.0) #t)
(test (<  1e-0 0.0) #f)
(test (< -1e-0 0.0) #t)
(test (<  1e+100 0.0) #f)
(test (< -1e+100 0.0) #t)
(test (<  1e-100 0.0) #f)
(test (< -1e-100 0.0) #t)
(test (<  1e+10000 0.0) #f)
(test (< -1e+10000 0.0) #t)
(test (<  1e-10000 0.0) #f)
(test (< -1e-10000 0.0) #t)
(test (< 0.0  0.0) #f)
(test (< 0.0  1.0) #t)
(test (< 0.0 -1.0) #f)
(test (< 0.0  0.1) #t)
(test (< 0.0 -0.1) #f)
(test (< 0.0  1e+0) #t)
(test (< 0.0 -1e+0) #f)
(test (< 0.0  1e-0) #t)
(test (< 0.0 -1e-0) #f)
(test (< 0.0  1e+100) #t)
(test (< 0.0 -1e+100) #f)
(test (< 0.0  1e-100) #t)
(test (< 0.0 -1e-100) #f)
(test (< 0.0  1e+10000) #t)
(test (< 0.0 -1e+10000) #f)
(test (< 0.0  1e-10000) #t)
(test (< 0.0 -1e-10000) #f)
(test (<  1.0e14  1.0e14) #f)
(test (<  1.0e14 -1.0e14) #f)
(test (< -1.0e14  1.0e14) #t)
(test (< -1.0e14 -1.0e14) #f)
(test (<  1.0e14  1.1e14) #t)
(test (<  1.0e14 -1.1e14) #f)
(test (< -1.0e14  1.1e14) #t)
(test (< -1.0e14 -1.1e14) #f)
(test (<  1.0e-14  1.0e-14) #f)
(test (<  1.0e-14 -1.0e-14) #f)
(test (< -1.0e-14  1.0e-14) #t)
(test (< -1.0e-14 -1.0e-14) #f)
(test (<  1.0e-14  1.1e-14) #t)
(test (<  1.0e-14 -1.1e-14) #f)
(test (< -1.0e-14  1.1e-14) #t)
(test (< -1.0e-14 -1.1e-14) #f)
(test (<  1234567890123.4  1234567890123) #f)
(test (<  1234567890123.4 -1234567890123) #f)
(test (< -1234567890123.4  1234567890123) #t)
(test (< -1234567890123.4 -1234567890123) #t)
(test (<  1234567890123  1234567890123.4) #t)
(test (<  1234567890123 -1234567890123.4) #f)
(test (< -1234567890123  1234567890123.4) #t)
(test (< -1234567890123 -1234567890123.4) #f)
(test (< 1e-5 1e-4 1e-3 0.01 0.1 1.0 10.0) #t)
(test (< 1e-5 1e-4 1e-3 0.01 0.1 1.0  1.0) #f)
(test (< 1.0 1.0 1.0 1.0 1.0 1.0) #f)

(test (<=  1234567890  9876543210) #t)
(test (<=  1234567890 -9876543210) #f)
(test (<= -1234567890  9876543210) #t)
(test (<= -1234567890 -9876543210) #f)
(test (<=  9876543210  1234567890) #f)
(test (<=  9876543210 -1234567890) #f)
(test (<= -9876543210  1234567890) #t)
(test (<= -9876543210 -1234567890) #t)
(test (<= -1234567890 -1234567890) #t)
(test (<=  1234567890  1234567890) #t)
(test (<= 1234567890 0) #f)
(test (<= 0 1234567890) #t)
(test (<= 1 2 3 4 5 6 7 8 9 10) #t)
(test (<= 1 2 3 4 5 6 7 8 9 9) #t)

(test (<=  0.0  0.0) #t)
(test (<= -0.0  0.0) #t)
(test (<=  0.0 -0.0) #t)
(test (<= -0.0 -0.0) #t)
(test (<=  1.0  1.0) #t)
(test (<= -1.0  1.0) #t)
(test (<=  1.0 -1.0) #f)
(test (<= -1.0 -1.0) #t)
(test (<=  0.1  0.1) #t)
(test (<= -0.1  0.1) #t)
(test (<=  0.1 -0.1) #f)
(test (<= -0.1 -0.1) #t)
(test (<=  123.45e+67  123.45e+67) #t)
(test (<=  123.45e+67  123.45e-67) #f)
(test (<=  123.45e-67  123.45e+67) #t)
(test (<=  123.45e-67  123.45e-67) #t)
(test (<=  123.45e+67 -123.45e+67) #f)
(test (<=  123.45e+67 -123.45e-67) #f)
(test (<=  123.45e-67 -123.45e+67) #f)
(test (<=  123.45e-67 -123.45e-67) #f)
(test (<= -123.45e+67  123.45e+67) #t)
(test (<= -123.45e+67  123.45e-67) #t)
(test (<= -123.45e-67  123.45e+67) #t)
(test (<= -123.45e-67  123.45e-67) #t)
(test (<= -123.45e+67 -123.45e+67) #t)
(test (<= -123.45e+67 -123.45e-67) #t)
(test (<= -123.45e-67 -123.45e+67) #f)
(test (<= -123.45e-67 -123.45e-67) #t)
(test (<=  0.0 0.0) #t)
(test (<=  1.0 0.0) #f)
(test (<= -1.0 0.0) #t)
(test (<=  0.1 0.0) #f)
(test (<= -0.1 0.0) #t)
(test (<=  1e+0 0.0) #f)
(test (<= -1e+0 0.0) #t)
(test (<=  1e-0 0.0) #f)
(test (<= -1e-0 0.0) #t)
(test (<=  1e+100 0.0) #f)
(test (<= -1e+100 0.0) #t)
(test (<=  1e-100 0.0) #f)
(test (<= -1e-100 0.0) #t)
(test (<=  1e+10000 0.0) #f)
(test (<= -1e+10000 0.0) #t)
(test (<=  1e-10000 0.0) #f)
(test (<= -1e-10000 0.0) #t)
(test (<= 0.0  0.0) #t)
(test (<= 0.0  1.0) #t)
(test (<= 0.0 -1.0) #f)
(test (<= 0.0  0.1) #t)
(test (<= 0.0 -0.1) #f)
(test (<= 0.0  1e+0) #t)
(test (<= 0.0 -1e+0) #f)
(test (<= 0.0  1e-0) #t)
(test (<= 0.0 -1e-0) #f)
(test (<= 0.0  1e+100) #t)
(test (<= 0.0 -1e+100) #f)
(test (<= 0.0  1e-100) #t)
(test (<= 0.0 -1e-100) #f)
(test (<= 0.0  1e+10000) #t)
(test (<= 0.0 -1e+10000) #f)
(test (<= 0.0  1e-10000) #t)
(test (<= 0.0 -1e-10000) #f)
(test (<=  1.0e14  1.0e14) #t)
(test (<=  1.0e14 -1.0e14) #f)
(test (<= -1.0e14  1.0e14) #t)
(test (<= -1.0e14 -1.0e14) #t)
(test (<=  1.0e14  1.1e14) #t)
(test (<=  1.0e14 -1.1e14) #f)
(test (<= -1.0e14  1.1e14) #t)
(test (<= -1.0e14 -1.1e14) #f)
(test (<=  1.0e-14  1.0e-14) #t)
(test (<=  1.0e-14 -1.0e-14) #f)
(test (<= -1.0e-14  1.0e-14) #t)
(test (<= -1.0e-14 -1.0e-14) #t)
(test (<=  1.0e-14  1.1e-14) #t)
(test (<=  1.0e-14 -1.1e-14) #f)
(test (<= -1.0e-14  1.1e-14) #t)
(test (<= -1.0e-14 -1.1e-14) #f)
(test (<=  1234567890123.4  1234567890123) #f)
(test (<=  1234567890123.4 -1234567890123) #f)
(test (<= -1234567890123.4  1234567890123) #t)
(test (<= -1234567890123.4 -1234567890123) #t)
(test (<=  1234567890123  1234567890123.4) #t)
(test (<=  1234567890123 -1234567890123.4) #f)
(test (<= -1234567890123  1234567890123.4) #t)
(test (<= -1234567890123 -1234567890123.4) #f)
(test (<= 1e-5 1e-4 1e-3 0.01 0.1 1.0 10.0) #t)
(test (<= 1e-5 1e-4 1e-3 0.01 0.1 1.0  1.0) #t)
(test (<= 1.0 1.0 1.0 1.0 1.0 1.0) #t)

(test (=  1234567890  9876543210) #f)
(test (=  1234567890 -9876543210) #f)
(test (= -1234567890  9876543210) #f)
(test (= -1234567890 -9876543210) #f)
(test (=  9876543210  1234567890) #f)
(test (=  9876543210 -1234567890) #f)
(test (= -9876543210  1234567890) #f)
(test (= -9876543210 -1234567890) #f)
(test (= -1234567890  1234567890) #f)
(test (=  1234567890 -1234567890) #f)
(test (=  1234567890  1234567890) #t)
(test (= -1234567890 -1234567890) #t)
(test (= 0 0) #t)
(test (= 0 1234567890) #f)
(test (= 1234567890 0) #f)
(test (= 1 1 1 1 1 1 1 1 1 1) #t)
(test (= 1 1 1 1 1 1 1 1 1 0) #f)

(test (=  0.0  0.0) #t)
(test (=  0.0 -0.0) #t)
(test (= -0.0  0.0) #t)
(test (= -0.0 -0.0) #t)
(test (=  1.0  1.0) #t)
(test (=  1.0 -1.0) #f)
(test (= -1.0  1.0) #f)
(test (= -1.0 -1.0) #t)
(test (=  0.1  0.1) #t)
(test (=  0.1 -0.1) #f)
(test (= -0.1  0.1) #f)
(test (= -0.1 -0.1) #t)
(test (= 1.0   1) #t)
(test (=   1 1.0) #t)
(test (=  123.456e3 123456) #t)
(test (=  123.456e4 1234560) #t)
(test (=  123.456e5 12345600) #t)
(test (=  123.456e10 1234560000000) #t)
(test (= -123.456e3  -123456) #t)
(test (= -123.456e4  -1234560) #t)
(test (= -123.456e5  -12345600) #t)
(test (= -123.456e10 -1234560000000) #t)
(test (=  1.2345678901234  12345678901234.) #f)
(test (=  1.2345678901234  1234567890123.4) #f)
(test (=  1.2345678901234  123456789012.34) #f)
(test (=  1.2345678901234  12345678901.234) #f)
(test (=  1.2345678901234  1234567890.1234) #f)
(test (=  1.2345678901234  123456789.01234) #f)
(test (=  1.2345678901234  12345678.901234) #f)
(test (=  1.2345678901234  1234567.8901234) #f)
(test (=  1.2345678901234  123456.78901234) #f)
(test (=  1.2345678901234  12345.678901234) #f)
(test (=  1.2345678901234  1234.5678901234) #f)
(test (=  1.2345678901234  123.45678901234) #f)
(test (=  1.2345678901234  12.345678901234) #f)
(test (=  1.2345678901234  1.2345678901234) #t)
(test (= -1.2345678901234 1.2345678901234) #f)
(test (=  1.2345678901234 -1.2345678901234) #f)
(test (=  1.2345678901234  1.2345678901233) #f)
(test (=  1.2345678901234  1.2345678901235) #f)
(test (= 1e50 100000000000000000000000000000000000000000000000000) #t)
(test (= 100000000000000000000000000000000000000000000000000 1e50) #t)
(test (= 12345678901234.0 12345678901234) #t)
(test (= 12345678901234 12345678901234.0) #t)
(test (= -12345678901234.0 -12345678901234) #t)
(test (= -12345678901234 -12345678901234.0) #t)
(test (= 1.0 1.0 1.0 1.0 1.0 1.0 1.0) #t)
(test (= 1.0 1.0 1.0 1.0 1.0 1.0  .1) #f)

(test (>  1234567890  9876543210) #f)
(test (>  1234567890 -9876543210) #t)
(test (> -1234567890  9876543210) #f)
(test (> -1234567890 -9876543210) #t)
(test (>  9876543210  1234567890) #t)
(test (>  9876543210 -1234567890) #t)
(test (> -9876543210  1234567890) #f)
(test (> -9876543210 -1234567890) #f)
(test (> -1234567890 -1234567890) #f)
(test (>  1234567890  1234567890) #f)
(test (> 1234567890 0) #t)
(test (> 0 1234567890) #f)
(test (> 9 8 7 6 5 4 3 2 1 0) #t)
(test (> 9 8 7 6 5 4 3 2 1 1) #f)

(test (>  0.0  0.0) #f)
(test (> -0.0  0.0) #f)
(test (>  0.0 -0.0) #f)
(test (> -0.0 -0.0) #f)
(test (>  1.0  1.0) #f)
(test (> -1.0  1.0) #f)
(test (>  1.0 -1.0) #t)
(test (> -1.0 -1.0) #f)
(test (>  0.1  0.1) #f)
(test (> -0.1  0.1) #f)
(test (>  0.1 -0.1) #t)
(test (> -0.1 -0.1) #f)
(test (>  123.45e+67  123.45e+67) #f)
(test (>  123.45e+67  123.45e-67) #t)
(test (>  123.45e-67  123.45e+67) #f)
(test (>  123.45e-67  123.45e-67) #f)
(test (>  123.45e+67 -123.45e+67) #t)
(test (>  123.45e+67 -123.45e-67) #t)
(test (>  123.45e-67 -123.45e+67) #t)
(test (>  123.45e-67 -123.45e-67) #t)
(test (> -123.45e+67  123.45e+67) #f)
(test (> -123.45e+67  123.45e-67) #f)
(test (> -123.45e-67  123.45e+67) #f)
(test (> -123.45e-67  123.45e-67) #f)
(test (> -123.45e+67 -123.45e+67) #f)
(test (> -123.45e+67 -123.45e-67) #f)
(test (> -123.45e-67 -123.45e+67) #t)
(test (> -123.45e-67 -123.45e-67) #f)
(test (>  0.0 0.0) #f)
(test (>  1.0 0.0) #t)
(test (> -1.0 0.0) #f)
(test (>  0.1 0.0) #t)
(test (> -0.1 0.0) #f)
(test (>  1e+0 0.0) #t)
(test (> -1e+0 0.0) #f)
(test (>  1e-0 0.0) #t)
(test (> -1e-0 0.0) #f)
(test (>  1e+100 0.0) #t)
(test (> -1e+100 0.0) #f)
(test (>  1e-100 0.0) #t)
(test (> -1e-100 0.0) #f)
(test (>  1e+10000 0.0) #t)
(test (> -1e+10000 0.0) #f)
(test (>  1e-10000 0.0) #t)
(test (> -1e-10000 0.0) #f)
(test (> 0.0  0.0) #f)
(test (> 0.0  1.0) #f)
(test (> 0.0 -1.0) #t)
(test (> 0.0  0.1) #f)
(test (> 0.0 -0.1) #t)
(test (> 0.0  1e+0) #f)
(test (> 0.0 -1e+0) #t)
(test (> 0.0  1e-0) #f)
(test (> 0.0 -1e-0) #t)
(test (> 0.0  1e+100) #f)
(test (> 0.0 -1e+100) #t)
(test (> 0.0  1e-100) #f)
(test (> 0.0 -1e-100) #t)
(test (> 0.0  1e+10000) #f)
(test (> 0.0 -1e+10000) #t)
(test (> 0.0  1e-10000) #f)
(test (> 0.0 -1e-10000) #t)
(test (>  1.0e14  1.0e14) #f)
(test (>  1.0e14 -1.0e14) #t)
(test (> -1.0e14  1.0e14) #f)
(test (> -1.0e14 -1.0e14) #f)
(test (>  1.0e14  1.1e14) #f)
(test (>  1.0e14 -1.1e14) #t)
(test (> -1.0e14  1.1e14) #f)
(test (> -1.0e14 -1.1e14) #t)
(test (>  1.0e-14  1.0e-14) #f)
(test (>  1.0e-14 -1.0e-14) #t)
(test (> -1.0e-14  1.0e-14) #f)
(test (> -1.0e-14 -1.0e-14) #f)
(test (>  1.0e-14  1.1e-14) #f)
(test (>  1.0e-14 -1.1e-14) #t)
(test (> -1.0e-14  1.1e-14) #f)
(test (> -1.0e-14 -1.1e-14) #t)
(test (>  1234567890123.4  1234567890123) #t)
(test (>  1234567890123.4 -1234567890123) #t)
(test (> -1234567890123.4  1234567890123) #f)
(test (> -1234567890123.4 -1234567890123) #f)
(test (>  1234567890123  1234567890123.4) #f)
(test (>  1234567890123 -1234567890123.4) #t)
(test (> -1234567890123  1234567890123.4) #f)
(test (> -1234567890123 -1234567890123.4) #t)
(test (> 10.0 1.0 0.1 0.01 1e-3 1e-4 1e-5) #t)
(test (> 10.0 1.0 0.1 0.01 1e-3 1e-4 1e-4) #f)
(test (> 1.0 1.0 1.0 1.0 1.0 1.0) #f)

(test (>=  1234567890  9876543210) #f)
(test (>=  1234567890 -9876543210) #t)
(test (>= -1234567890  9876543210) #f)
(test (>= -1234567890 -9876543210) #t)
(test (>=  9876543210  1234567890) #t)
(test (>=  9876543210 -1234567890) #t)
(test (>= -9876543210  1234567890) #f)
(test (>= -9876543210 -1234567890) #f)
(test (>= -1234567890 -1234567890) #t)
(test (>=  1234567890  1234567890) #t)
(test (>= 1234567890 0) #t)
(test (>= 0 1234567890) #f)
(test (>= 9 8 7 6 5 4 3 2 1 0) #t)
(test (>= 9 8 7 6 5 4 3 2 1 1) #t)

(test (>=  0.0  0.0) #t)
(test (>= -0.0  0.0) #t)
(test (>=  0.0 -0.0) #t)
(test (>= -0.0 -0.0) #t)
(test (>=  1.0  1.0) #t)
(test (>= -1.0  1.0) #f)
(test (>=  1.0 -1.0) #t)
(test (>= -1.0 -1.0) #t)
(test (>=  0.1  0.1) #t)
(test (>= -0.1  0.1) #f)
(test (>=  0.1 -0.1) #t)
(test (>= -0.1 -0.1) #t)
(test (>=  123.45e+67  123.45e+67) #t)
(test (>=  123.45e+67  123.45e-67) #t)
(test (>=  123.45e-67  123.45e+67) #f)
(test (>=  123.45e-67  123.45e-67) #t)
(test (>=  123.45e+67 -123.45e+67) #t)
(test (>=  123.45e+67 -123.45e-67) #t)
(test (>=  123.45e-67 -123.45e+67) #t)
(test (>=  123.45e-67 -123.45e-67) #t)
(test (>= -123.45e+67  123.45e+67) #f)
(test (>= -123.45e+67  123.45e-67) #f)
(test (>= -123.45e-67  123.45e+67) #f)
(test (>= -123.45e-67  123.45e-67) #f)
(test (>= -123.45e+67 -123.45e+67) #t)
(test (>= -123.45e+67 -123.45e-67) #f)
(test (>= -123.45e-67 -123.45e+67) #t)
(test (>= -123.45e-67 -123.45e-67) #t)
(test (>=  0.0 0.0) #t)
(test (>=  1.0 0.0) #t)
(test (>= -1.0 0.0) #f)
(test (>=  0.1 0.0) #t)
(test (>= -0.1 0.0) #f)
(test (>=  1e+0 0.0) #t)
(test (>= -1e+0 0.0) #f)
(test (>=  1e-0 0.0) #t)
(test (>= -1e-0 0.0) #f)
(test (>=  1e+100 0.0) #t)
(test (>= -1e+100 0.0) #f)
(test (>=  1e-100 0.0) #t)
(test (>= -1e-100 0.0) #f)
(test (>=  1e+10000 0.0) #t)
(test (>= -1e+10000 0.0) #f)
(test (>=  1e-10000 0.0) #t)
(test (>= -1e-10000 0.0) #f)
(test (>= 0.0  0.0) #t)
(test (>= 0.0  1.0) #f)
(test (>= 0.0 -1.0) #t)
(test (>= 0.0  0.1) #f)
(test (>= 0.0 -0.1) #t)
(test (>= 0.0  1e+0) #f)
(test (>= 0.0 -1e+0) #t)
(test (>= 0.0  1e-0) #f)
(test (>= 0.0 -1e-0) #t)
(test (>= 0.0  1e+100) #f)
(test (>= 0.0 -1e+100) #t)
(test (>= 0.0  1e-100) #f)
(test (>= 0.0 -1e-100) #t)
(test (>= 0.0  1e+10000) #f)
(test (>= 0.0 -1e+10000) #t)
(test (>= 0.0  1e-10000) #f)
(test (>= 0.0 -1e-10000) #t)
(test (>=  1.0e14  1.0e14) #t)
(test (>=  1.0e14 -1.0e14) #t)
(test (>= -1.0e14  1.0e14) #f)
(test (>= -1.0e14 -1.0e14) #t)
(test (>=  1.0e14  1.1e14) #f)
(test (>=  1.0e14 -1.1e14) #t)
(test (>= -1.0e14  1.1e14) #f)
(test (>= -1.0e14 -1.1e14) #t)
(test (>=  1.0e-14  1.0e-14) #t)
(test (>=  1.0e-14 -1.0e-14) #t)
(test (>= -1.0e-14  1.0e-14) #f)
(test (>= -1.0e-14 -1.0e-14) #t)
(test (>=  1.0e-14  1.1e-14) #f)
(test (>=  1.0e-14 -1.1e-14) #t)
(test (>= -1.0e-14  1.1e-14) #f)
(test (>= -1.0e-14 -1.1e-14) #t)
(test (>=  1234567890123.4  1234567890123) #t)
(test (>=  1234567890123.4 -1234567890123) #t)
(test (>= -1234567890123.4  1234567890123) #f)
(test (>= -1234567890123.4 -1234567890123) #f)
(test (>=  1234567890123  1234567890123.4) #f)
(test (>=  1234567890123 -1234567890123.4) #t)
(test (>= -1234567890123  1234567890123.4) #f)
(test (>= -1234567890123 -1234567890123.4) #t)
(test (>= 10.0 1.0 0.1 0.01 1e-3 1e-4 1e-5) #t)
(test (>= 10.0 1.0 0.1 0.01 1e-3 1e-4 1e-4) #t)
(test (>= 1.0 1.0 1.0 1.0 1.0 1.0) #t)

(test (abs 1234567890) 1234567890)
(test (abs -1234567890) 1234567890)
(test (abs 0) 0)
(test (abs 1.234567890) 1.23456789)
(test (abs 1.234567890) 1.23456789)
(test (abs 0.0) 0.0)
(test (abs -0.0) 0.0)

(test (ceiling  0.0) 0.0)
(test (ceiling  1.0) 1.0)
(test (ceiling -1.0) -1.0)
(test (ceiling  1.1) 2.0)
(test (ceiling  1.4) 2.0)
(test (ceiling  1.5) 2.0)
(test (ceiling  1.9) 2.0)
(test (ceiling -1.1) -1.0)
(test (ceiling -1.4) -1.0)
(test (ceiling -1.5) -1.0)
(test (ceiling -1.9) -1.0)

(define pi 3.14159265358979323846264338327950288419716939937510)
(define pi/4  (/ pi 4))
(define pi/2  (/ pi 2))
(define 3pi/4 (+ pi/2 pi/4))
(define 3pi/2 (+ pi pi/2))
(define 5pi/4 (+ pi pi/4))
(define 7pi/4 (+ pi 3pi/4))
(define 2pi   (+ pi pi))

(test (cos   0.0) 1.0)
(test (cos  pi/2) 0.0)
(test (cos  pi  ) -1.0)
(test (cos 3pi/2) 0.0)
(test (cos 2pi  ) 1.0)
(test (<  0.7071067810 (cos  pi/4)  0.7071067819) #t)
(test (< -0.7071067819 (cos 3pi/4) -0.7071067810) #t)
(test (< -0.7071067819 (cos 5pi/4) -0.7071067810) #t)
(test (<  0.7071067810 (cos 7pi/4)  0.7071067819) #t)
(test (<  0.1699671420 (cos   1.4)  0.1699671430) #t)
(test (< -0.9422223409 (cos   2.8) -0.9422223400) #t)
(test (< -0.4902608219 (cos   4.2) -0.4902608210) #t)
(test (<  0.7755658780 (cos   5.6)  0.7755658789) #t)

(test (even? -1) #f)
(test (even? 0) #t)
(test (even? 1) #f)
(test (even? 2) #t)
(test (even? 1234567890) #t)
(test (even? 1234567891) #f)

(test (exact->inexact  #e0.0) #i0)
(test (exact->inexact  #i0.0) #i0)
(test (exact->inexact #e-0.0) #i0)
(test (exact->inexact #i-0.0) #i0)
(test (exact->inexact  #e1.0) #i1)
(test (exact->inexact  #i1.0) #i1)
(test (exact->inexact #e-1.0) #i-1)
(test (exact->inexact #i-1.0) #i-1)
(test (exact->inexact  #e1.1) #i1.1)
(test (exact->inexact  #i1.1) #i1.1)
(test (exact->inexact #e-1.1) #i-1.1)
(test (exact->inexact #i-1.1) #i-1.1)
(test (exact? (exact->inexact  0)) #f)
(test (exact? (exact->inexact  1)) #f)
(test (exact? (exact->inexact -1)) #f)
(test (exact? (exact->inexact  12345678901234567890)) #f)
(test (exact? (exact->inexact -12345678901234567890)) #f)
(test (exact? (exact->inexact  0.0)) #f)
(test (exact? (exact->inexact -0.0)) #f)
(test (exact? (exact->inexact  0.1)) #f)
(test (exact? (exact->inexact -0.1)) #f)
(test (exact? (exact->inexact  1.0)) #f)
(test (exact? (exact->inexact -1.0)) #f)
(test (exact? (exact->inexact  1234567890.1234)) #f)
(test (exact? (exact->inexact -1234567890.1234)) #f)
(test (exact? (exact->inexact  0.1234567890123)) #f)
(test (exact? (exact->inexact -0.1234567890123)) #f)
(test (exact? (exact->inexact  1.2345e+100)) #f)
(test (exact? (exact->inexact  1.2345e-100)) #f)
(test (exact? (exact->inexact -1.2345e+100)) #f)
(test (exact? (exact->inexact -1.2345e-100)) #f)

(test (exact?  0) #t)
(test (exact?  1) #t)
(test (exact? -1) #t)
(test (exact?  12345678901234567890) #t)
(test (exact? -12345678901234567890) #t)
(test (exact?  0.0) #t)
(test (exact? -0.0) #t)
(test (exact?  0.1) #t)
(test (exact? -0.1) #t)
(test (exact?  1.0) #t)
(test (exact? -1.0) #t)
(test (exact?  1234567890.1234) #t)
(test (exact? -1234567890.1234) #t)
(test (exact?  0.1234567890123) #t)
(test (exact? -0.1234567890123) #t)
(test (exact?  1.2345e+100) #t)
(test (exact?  1.2345e-100) #t)
(test (exact? -1.2345e+100) #t)
(test (exact? -1.2345e-100) #t)
(test (exact?  #i0) #f)
(test (exact?  #i1) #f)
(test (exact? #i-1) #f)
(test (exact?  #i12345678901234567890) #f)
(test (exact? #i-12345678901234567890) #f)
(test (exact?  #i0.0) #f)
(test (exact? #i-0.0) #f)
(test (exact?  #i0.1) #f)
(test (exact? #i-0.1) #f)
(test (exact?  #i1.0) #f)
(test (exact? #i-1.0) #f)
(test (exact?  #i1234567890.1234) #f)
(test (exact? #i-1234567890.1234) #f)
(test (exact?  #i0.1234567890123) #f)
(test (exact? #i-0.1234567890123) #f)
(test (exact?  #i1.2345e+100) #f)
(test (exact?  #i1.2345e-100) #f)
(test (exact? #i-1.2345e+100) #f)
(test (exact? #i-1.2345e-100) #f)

(test (expt 0 2) 0)
(test (expt 2 0) 1)
(test (expt 2 1) 2)
(test (expt 2 2) 4)
(test (expt 2 3) 8)
(test (expt -2 3) -8)
(test (expt -2 4) 16)
(test (expt 2 100) 1267650600228229401496703205376)
(test (expt 2.0 2.0) 4.0)

(test (expt 0. 2) 0)
(test (expt 2.0 0) 1.0)
(test (expt 2.0 1) 2.0)
(test (expt 2.0 2) 4.0)
(test (expt 2.0 3) 8.0)
(test (expt -2.0 3) -8.0)
(test (expt -2.0 4) 16.0)
(test (expt 2.0 40) 1.099511627776e+12)

(test (expt 1 -1) 1.0)
(test (expt  2 -1) 0.5)
(test (expt -2  -1) -0.5)
(test (expt  2  -1) 0.5)
(test (expt  2  -2) 0.25)
(test (expt -2  -2) 0.25)
(test (expt  2  -3) 0.125)
(test (expt -2  -3) -0.125)
(test (expt  2 -10) 0.0009765625)
(test (expt -2 -10) 0.0009765625)
(test (expt 10   -10) 1.0e-10)
(test (expt 10  -100) 1.0e-100)
(test (expt 10 -1000) 1.0e-1000)

(test (floor  0.0) 0.0)
(test (floor  1.0) 1.0)
(test (floor -1.0) -1.0)
(test (floor  1.1) 1.0)
(test (floor  1.4) 1.0)
(test (floor  1.5) 1.0)
(test (floor  1.9) 1.0)
(test (floor -1.1) -2.0)
(test (floor -1.4) -2.0)
(test (floor -1.5) -2.0)
(test (floor -1.9) -2.0)

(test (gcd) 0)
(test (gcd 17) 17)
(test (gcd 18 12) 6)
(test (gcd 289 85 34) 17)

(test (inexact->exact  #e0.0) 0)
(test (inexact->exact  #i0.0) 0)
(test (inexact->exact #e-0.0) 0)
(test (inexact->exact #i-0.0) 0)
(test (inexact->exact  #e1.0) 1)
(test (inexact->exact  #i1.0) 1)
(test (inexact->exact #e-1.0) -1)
(test (inexact->exact #i-1.0) -1)
(test (inexact->exact  #e1.1) 1.1)
(test (inexact->exact  #i1.1) 1.1)
(test (inexact->exact #e-1.1) -1.1)
(test (inexact->exact #i-1.1) -1.1)
(test (exact? (inexact->exact  #i0)) #t)
(test (exact? (inexact->exact  #i1)) #t)
(test (exact? (inexact->exact #i-1)) #t)
(test (exact? (inexact->exact  #i12345678901234567890)) #t)
(test (exact? (inexact->exact #i-12345678901234567890)) #t)
(test (exact? (inexact->exact  #i0.0)) #t)
(test (exact? (inexact->exact #i-0.0)) #t)
(test (exact? (inexact->exact  #i0.1)) #t)
(test (exact? (inexact->exact #i-0.1)) #t)
(test (exact? (inexact->exact  #i1.0)) #t)
(test (exact? (inexact->exact #i-1.0)) #t)
(test (exact? (inexact->exact  #i1234567890.1234)) #t)
(test (exact? (inexact->exact #i-1234567890.1234)) #t)
(test (exact? (inexact->exact  #i0.1234567890123)) #t)
(test (exact? (inexact->exact #i-0.1234567890123)) #t)
(test (exact? (inexact->exact  #i1.2345e+100)) #t)
(test (exact? (inexact->exact  #i1.2345e-100)) #t)
(test (exact? (inexact->exact #i-1.2345e+100)) #t)
(test (exact? (inexact->exact #i-1.2345e-100)) #t)

(test (exp 0) 1)
(test (< 1.6487212700 (exp 0.5) 1.6487212709) #t)
(test (< 2.7182818280 (exp 1.0) 2.7182818289) #t)
(test (< 7.3890560980 (exp 2.0) 7.3890560990) #t)
(test (< 20.085536920 (exp 3.0) 20.085536929) #t)

(test (inexact?  0) #f)
(test (inexact?  1) #f)
(test (inexact? -1) #f)
(test (inexact?  12345678901234567890) #f)
(test (inexact? -12345678901234567890) #f)
(test (inexact?  0.0) #f)
(test (inexact? -0.0) #f)
(test (inexact?  0.1) #f)
(test (inexact? -0.1) #f)
(test (inexact?  1.0) #f)
(test (inexact? -1.0) #f)
(test (inexact?  1234567890.1234) #f)
(test (inexact? -1234567890.1234) #f)
(test (inexact?  0.1234567890123) #f)
(test (inexact? -0.1234567890123) #f)
(test (inexact?  1.2345e+100) #f)
(test (inexact?  1.2345e-100) #f)
(test (inexact? -1.2345e+100) #f)
(test (inexact? -1.2345e-100) #f)
(test (inexact?  #i0) #t)
(test (inexact?  #i1) #t)
(test (inexact? #i-1) #t)
(test (inexact?  #i12345678901234567890) #t)
(test (inexact? #i-12345678901234567890) #t)
(test (inexact?  #i0.0) #t)
(test (inexact? #i-0.0) #t)
(test (inexact?  #i0.1) #t)
(test (inexact? #i-0.1) #t)
(test (inexact?  #i1.0) #t)
(test (inexact? #i-1.0) #t)
(test (inexact?  #i1234567890.1234) #t)
(test (inexact? #i-1234567890.1234) #t)
(test (inexact?  #i0.1234567890123) #t)
(test (inexact? #i-0.1234567890123) #t)
(test (inexact?  #i1.2345e+100) #t)
(test (inexact?  #i1.2345e-100) #t)
(test (inexact? #i-1.2345e+100) #t)
(test (inexact? #i-1.2345e-100) #t)

(test (lcm) 1)
(test (lcm 17) 17)
(test (lcm 12 18) 36)
(test (lcm 5 12 18) 180)

(test (log 1) 0.0)
(test (< -2.3025850930 (log 0.1) -2.3025850920) #t)
(test (<  0.6931471800 (log 2.0)  0.6931471809) #t)
(test (<  1.0986122880 (log 3.0)  1.0986122889) #t)

(test (min 1) 1)
(test (min 2 1 3) 1)
(test (min 2 1 -2 -1 3) -2)
(test (min 2.0 1 -2 -1 3) -2)
(test (exact? (min 2.0 1 -2 -1 3)) #t)
(test (min #i2.0 1 -2 -1 3) -2.0)
(test (inexact? (min #i2.0 1 -2 -1 3)) #t)

(test (max 1) 1)
(test (max 2 3 1) 3)
(test (max 2 -2 5 -1 3) 5)
(test (max 2 -2 5 -1 3.0) 5)
(test (exact? (max 2 -2 5 -1 3.0)) #t)
(test (max 2 -2 5 -1 #i3.0) 5.0)
(test (inexact? (max 2 -2 5 -1 #i3.0)) #t)

(test (modulo  1234567890  12345)  6165)
(test (modulo  1234567890 -12345) -6180)
(test (modulo -1234567890  12345)  6180)
(test (modulo -1234567890 -12345) -6165)
(test (modulo  12345  1234567890)  12345)
(test (modulo  12345 -1234567890) -1234555545)
(test (modulo -12345  1234567890)  1234555545)
(test (modulo -12345 -1234567890) -12345)
(test (modulo  12345  12345) 0)
(test (modulo  12345 -12345) 0)
(test (modulo -12345  12345) 0)
(test (modulo -12345 -12345) 0)

(test (negative? -1) #t)
(test (negative?  0) #f)
(test (negative?  1) #f)
(test (negative? -1.0) #t)
(test (negative? -0.1) #t)
(test (negative?  0.0) #f)
(test (negative? -0.0) #f)
(test (negative?  0.1) #f)
(test (negative?  1.0) #f)
(test (negative? -1e+100) #t)
(test (negative?  1e+100) #f)
(test (negative? -1e-100) #t)
(test (negative?  1e-100) #f)

(test (not #f) #t)
(test (not #\c) #f)
(test (not 1) #f)
(test (not '(pair)) #f)
(test (not (lambda () #f)) #f)
(test (not "string") #f)
(test (not 'symbol) #f)
(test (not '#(vector)) #f)
(test (not (current-input-port)) #f)
(test (not (current-output-port)) #f)

(test (odd? -1) #t)
(test (odd? 0) #f)
(test (odd? 1) #t)
(test (odd? 2) #f)
(test (odd? 1234567890) #f)
(test (odd? 1234567891) #t)

(test (positive? -1) #f)
(test (positive?  0) #f)
(test (positive?  1) #t)
(test (positive? -1.0) #f)
(test (positive? -0.1) #f)
(test (positive?  0.0) #f)
(test (positive? -0.0) #f)
(test (positive?  0.1) #t)
(test (positive?  1.0) #t)
(test (positive? -1e+100) #f)
(test (positive?  1e+100) #t)
(test (positive? -1e-100) #f)
(test (positive?  1e-100) #t)

(test (quotient  1234567890  12345)  100005)
(test (quotient  1234567890 -12345) -100005)
(test (quotient -1234567890  12345) -100005)
(test (quotient -1234567890 -12345)  100005)
(test (quotient  12345  1234567890)  0)
(test (quotient  12345 -1234567890)  0)
(test (quotient -12345  1234567890)  0)
(test (quotient -12345 -1234567890)  0)
(test (quotient  12345  12345)  1)
(test (quotient  12345 -12345) -1)
(test (quotient -12345  12345) -1)
(test (quotient -12345 -12345)  1)

(test (remainder  1234567890  12345)  6165)
(test (remainder  1234567890 -12345)  6165)
(test (remainder -1234567890  12345) -6165)
(test (remainder -1234567890 -12345) -6165)
(test (remainder  12345  1234567890)  12345)
(test (remainder  12345 -1234567890)  12345)
(test (remainder -12345  1234567890) -12345)
(test (remainder -12345 -1234567890) -12345)
(test (remainder  12345  12345) 0)
(test (remainder  12345 -12345) 0)
(test (remainder -12345  12345) 0)
(test (remainder -12345 -12345) 0)

(test (round  0.0) 0.0)
(test (round  1.0) 1.0)
(test (round -1.0) -1.0)
(test (round  1.1) 1.0)
(test (round  1.4) 1.0)
(test (round  1.5) 2.0)
(test (round  1.9) 2.0)
(test (round -1.1) -1.0)
(test (round -1.4) -1.0)
(test (round -1.5) -2.0)
(test (round -1.9) -2.0)

(test (sin    0.0) 0.0)
(test (sin   pi/2) 1.0)
(test (sin   pi  ) 0.0)
(test (sin  3pi/2) -1.0)
(test (sin    2pi) 0.0)
(test (<  0.7071067810 (sin  pi/4)  0.7071067819) #t)
(test (<  0.7071067810 (sin 3pi/4)  0.7071067819) #t)
(test (< -0.7071067819 (sin 5pi/4) -0.7071067810) #t)
(test (< -0.7071067819 (sin 7pi/4) -0.7071067810) #t)
(test (<  0.9854497290 (sin   1.4)  0.9854497300) #t)
(test (<  0.3349881500 (sin   2.8)  0.3349881509) #t)
(test (< -0.8715757729 (sin   4.2) -0.8715757720) #t)
(test (< -0.6312666379 (sin   5.6) -0.6312666370) #t)

(test (sqrt 0) 0)
(test (sqrt 1) 1)
(test (sqrt 144) 12)
(test (sqrt 15241578750190521) 123456789)
(test (sqrt 15241578750190521.0) 123456789.0)
(test (< 1.4142135620   (sqrt 2) 1.4142135629) #t)
(test (< 11.090536500 (sqrt 123) 11.090536509) #t)
(test (sqrt 15239.9025) 123.45)
(test (sqrt 1e20) 1e10)

(test (tan    0.0) 0.0)
(test (tan   pi/4) 1.0)
(test (tan  3pi/4) -1.0)
(test (tan  5pi/4) 1.0)
(test (tan  7pi/4) -1.0)
(test (tan  2pi  ) 0.0)
(test (<  5.7978837150 (tan 1.4)  5.7978837159) #t)
(test (< -0.3555298319 (tan 2.8) -0.3555298310) #t)
(test (<  1.7777797740 (tan 4.2)  1.7777797749) #t)
(test (< -0.8139432839 (tan 5.6) -0.8139432830) #t)

(test (truncate  0.0) 0.0)
(test (truncate  1.0) 1.0)
(test (truncate -1.0) -1.0)
(test (truncate  1.1) 1.0)
(test (truncate  1.4) 1.0)
(test (truncate  1.5) 1.0)
(test (truncate  1.9) 1.0)
(test (truncate -1.1) -1.0)
(test (truncate -1.4) -1.0)
(test (truncate -1.5) -1.0)
(test (truncate -1.9) -1.0)

(test (zero? -1) #f)
(test (zero?  0) #t)
(test (zero?  1) #f)
(test (zero? -1.0) #f)
(test (zero? -0.1) #f)
(test (zero?  0.0) #t)
(test (zero? -0.0) #t)
(test (zero?  0.1) #f)
(test (zero?  1.0) #f)
(test (zero?  1e+100) #f)
(test (zero?  1e-100) #f)
(test (zero? -1e+100) #f)
(test (zero? -1e-100) #f)

; --- equivalence ---

(test (eq? 'x 'x) #t)
(test (eq? eq? eq?) #t)
(test (eq? '() '()) #t)
(test (eq? 'x 'y) #f)
(test (eq? 'x '(x . y)) #f)
(test ((lambda (x) (eq? x x)) '(x . y)) #t)
(test (eq? #t #t) #t)
(test (eq? #f #f) #t)
(test (eq? (list 'pair) (list 'pair)) #f)
(test (eq? (lambda () #f) (lambda () #f)) #f)
(test (eq? "string" "string") #f)
(test (eq? 'symbol 'symbol) #t)
(test (eq? (vector 'vector) (vector 'vector)) #f)

(test (eqv? #f #f) #t)
(test (eqv? #\c #\c) #t)
(test (eqv? 1 1) #t)
(test (eqv? (list 'pair) (list 'pair)) #f)
(test (eqv? (lambda () #f) (lambda () #f)) #f)
(test (eqv? "string" "string") #f)
(test (eqv? 'symbol 'symbol) #t)
(test (eqv? (vector 'vector) (vector 'vector)) #f)

(test (equal? #f #f) #t)
(test (equal? #\c #\c) #t)
(test (equal? 1 1) #t)
(test (equal? '(pair) '(pair)) #t)
(test (equal? (lambda () #f) (lambda () #f)) #f)
(test (equal? "string" "string") #t)
(test (equal? 'symbol 'symbol) #t)
(test (equal? '#(vector) #(vector)) #t)
(test (equal? tree tree) #t)

; --- chars ---

(test (char-alphabetic? #\a) #t)
(test (char-alphabetic? #\A) #t)
(test (char-alphabetic? #\z) #t)
(test (char-alphabetic? #\Z) #t)
(test (char-alphabetic? #\@) #f)
(test (char-alphabetic? #\[) #f)
(test (char-alphabetic? #\`) #f)
(test (char-alphabetic? #\{) #f)

(test (char-ci<? #\+ #\+) #f)
(test (char-ci<? #\+ #\-) #t)
(test (char-ci<? #\A #\A) #f)
(test (char-ci<? #\A #\a) #f)
(test (char-ci<? #\a #\A) #f)
(test (char-ci<? #\a #\a) #f)
(test (char-ci<? #\A #\Z) #t)
(test (char-ci<? #\A #\z) #t)
(test (char-ci<? #\a #\Z) #t)
(test (char-ci<? #\a #\z) #t)
(test (char-ci<? #\Z #\A) #f)
(test (char-ci<? #\Z #\a) #f)
(test (char-ci<? #\z #\A) #f)
(test (char-ci<? #\z #\a) #f)
(test (char-ci<? #\a #\b #\c) #t)
(test (char-ci<? #\a #\b #\b) #f)
(test (char-ci<? #\b #\b #\a) #f)
(test (char-ci<? #\c #\b #\a) #f)

(test (char-ci<=? #\+ #\+) #t)
(test (char-ci<=? #\+ #\-) #t)
(test (char-ci<=? #\A #\A) #t)
(test (char-ci<=? #\A #\a) #t)
(test (char-ci<=? #\a #\A) #t)
(test (char-ci<=? #\a #\a) #t)
(test (char-ci<=? #\A #\Z) #t)
(test (char-ci<=? #\A #\z) #t)
(test (char-ci<=? #\a #\Z) #t)
(test (char-ci<=? #\a #\z) #t)
(test (char-ci<=? #\Z #\A) #f)
(test (char-ci<=? #\Z #\a) #f)
(test (char-ci<=? #\z #\A) #f)
(test (char-ci<=? #\z #\a) #f)
(test (char-ci<=? #\a #\b #\c) #t)
(test (char-ci<=? #\a #\b #\b) #t)
(test (char-ci<=? #\b #\b #\a) #f)
(test (char-ci<=? #\c #\b #\a) #f)

(test (char-ci=? #\+ #\+) #t)
(test (char-ci=? #\+ #\-) #f)
(test (char-ci=? #\A #\A) #t)
(test (char-ci=? #\A #\a) #t)
(test (char-ci=? #\a #\A) #t)
(test (char-ci=? #\a #\a) #t)
(test (char-ci=? #\A #\Z) #f)
(test (char-ci=? #\A #\z) #f)
(test (char-ci=? #\a #\Z) #f)
(test (char-ci=? #\a #\z) #f)
(test (char-ci=? #\a #\A #\a) #t)
(test (char-ci=? #\a #\A #\b) #f)

(test (char-ci>? #\+ #\+) #f)
(test (char-ci>? #\+ #\-) #f)
(test (char-ci>? #\A #\A) #f)
(test (char-ci>? #\A #\a) #f)
(test (char-ci>? #\a #\A) #f)
(test (char-ci>? #\a #\a) #f)
(test (char-ci>? #\A #\Z) #f)
(test (char-ci>? #\A #\z) #f)
(test (char-ci>? #\a #\Z) #f)
(test (char-ci>? #\a #\z) #f)
(test (char-ci>? #\Z #\A) #t)
(test (char-ci>? #\Z #\a) #t)
(test (char-ci>? #\z #\A) #t)
(test (char-ci>? #\z #\a) #t)
(test (char-ci>? #\a #\b #\c) #f)
(test (char-ci>? #\a #\b #\b) #f)
(test (char-ci>? #\b #\b #\a) #f)
(test (char-ci>? #\c #\b #\a) #t)

(test (char-ci>=? #\+ #\+) #t)
(test (char-ci>=? #\+ #\-) #f)
(test (char-ci>=? #\A #\A) #t)
(test (char-ci>=? #\A #\a) #t)
(test (char-ci>=? #\a #\A) #t)
(test (char-ci>=? #\a #\a) #t)
(test (char-ci>=? #\A #\Z) #f)
(test (char-ci>=? #\A #\z) #f)
(test (char-ci>=? #\a #\Z) #f)
(test (char-ci>=? #\a #\z) #f)
(test (char-ci>=? #\Z #\A) #t)
(test (char-ci>=? #\Z #\a) #t)
(test (char-ci>=? #\z #\A) #t)
(test (char-ci>=? #\z #\a) #t)
(test (char-ci>=? #\a #\b #\c) #f)
(test (char-ci>=? #\a #\b #\b) #f)
(test (char-ci>=? #\b #\b #\a) #t)
(test (char-ci>=? #\c #\b #\a) #t)

(test (char-downcase #\a) #\a)
(test (char-downcase #\A) #\a)
(test (char-downcase #\z) #\z)
(test (char-downcase #\Z) #\z)
(test (char-downcase #\@) #\@)
(test (char-downcase #\[) #\[)
(test (char-downcase #\`) #\`)
(test (char-downcase #\{) #\{)

(test (char-lower-case? #\a) #t)
(test (char-lower-case? #\A) #f)
(test (char-lower-case? #\z) #t)
(test (char-lower-case? #\Z) #f)
(test (char-lower-case? #\@) #f)
(test (char-lower-case? #\[) #f)
(test (char-lower-case? #\`) #f)
(test (char-lower-case? #\{) #f)

(test (char-numeric? #\0) #t)
(test (char-numeric? #\9) #t)
(test (char-numeric? #\/) #f)
(test (char-numeric? #\:) #f)

(test (char-upcase #\a) #\A)
(test (char-upcase #\A) #\A)
(test (char-upcase #\z) #\Z)
(test (char-upcase #\Z) #\Z)
(test (char-upcase #\@) #\@)
(test (char-upcase #\[) #\[)
(test (char-upcase #\`) #\`)
(test (char-upcase #\{) #\{)

(test (char-upper-case? #\a) #f)
(test (char-upper-case? #\A) #t)
(test (char-upper-case? #\z) #f)
(test (char-upper-case? #\Z) #t)
(test (char-upper-case? #\@) #f)
(test (char-upper-case? #\[) #f)
(test (char-upper-case? #\`) #f)
(test (char-upper-case? #\{) #f)

(test (char-whitespace? #\0) #f)
(test (char-whitespace? #\9) #f)
(test (char-whitespace? #\a) #f)
(test (char-whitespace? #\z) #f)
(test (char-whitespace? #\ ) #t)
(test (char-whitespace? #\space) #t)
(test (char-whitespace? #\newline) #t)
(test (char-whitespace? (integer->char 9)) #t)
(test (char-whitespace? (integer->char 10)) #t)
(test (char-whitespace? (integer->char 12)) #t)
(test (char-whitespace? (integer->char 13)) #t)

(test (char<? #\+ #\+) #f)
(test (char<? #\+ #\-) #t)
(test (char<? #\A #\A) #f)
(test (char<? #\A #\a) #t)
(test (char<? #\a #\A) #f)
(test (char<? #\a #\a) #f)
(test (char<? #\A #\Z) #t)
(test (char<? #\A #\z) #t)
(test (char<? #\a #\Z) #f)
(test (char<? #\a #\z) #t)
(test (char<? #\Z #\A) #f)
(test (char<? #\Z #\a) #t)
(test (char<? #\z #\A) #f)
(test (char<? #\z #\a) #f)
(test (char<? #\a #\b #\c) #t)
(test (char<? #\a #\a #\b) #f)
(test (char<? #\c #\c #\b) #f)
(test (char<? #\c #\b #\a) #f)

(test (char<=? #\+ #\+) #t)
(test (char<=? #\+ #\-) #t)
(test (char<=? #\A #\A) #t)
(test (char<=? #\A #\a) #t)
(test (char<=? #\a #\A) #f)
(test (char<=? #\a #\a) #t)
(test (char<=? #\A #\Z) #t)
(test (char<=? #\A #\z) #t)
(test (char<=? #\a #\Z) #f)
(test (char<=? #\a #\z) #t)
(test (char<=? #\Z #\A) #f)
(test (char<=? #\Z #\a) #t)
(test (char<=? #\z #\A) #f)
(test (char<=? #\z #\a) #f)
(test (char<=? #\a #\b #\c) #t)
(test (char<=? #\a #\a #\b) #t)
(test (char<=? #\c #\c #\b) #f)
(test (char<=? #\c #\b #\a) #f)

(test (char=? #\+ #\+) #t)
(test (char=? #\+ #\-) #f)
(test (char=? #\A #\A) #t)
(test (char=? #\A #\a) #f)
(test (char=? #\a #\A) #f)
(test (char=? #\a #\a) #t)
(test (char=? #\A #\Z) #f)
(test (char=? #\A #\z) #f)
(test (char=? #\a #\Z) #f)
(test (char=? #\a #\z) #f)
(test (char=? #\Z #\A) #f)
(test (char=? #\Z #\a) #f)
(test (char=? #\z #\A) #f)
(test (char=? #\z #\a) #f)
(test (char=? #\a #\a #\a) #t)
(test (char=? #\a #\a #\b #\a) #f)

(test (char>? #\+ #\+) #f)
(test (char>? #\+ #\-) #f)
(test (char>? #\A #\A) #f)
(test (char>? #\A #\a) #f)
(test (char>? #\a #\A) #t)
(test (char>? #\a #\a) #f)
(test (char>? #\A #\Z) #f)
(test (char>? #\A #\z) #f)
(test (char>? #\a #\Z) #t)
(test (char>? #\a #\z) #f)
(test (char>? #\Z #\A) #t)
(test (char>? #\Z #\a) #f)
(test (char>? #\z #\A) #t)
(test (char>? #\z #\a) #t)
(test (char>? #\a #\b #\c) #f)
(test (char>? #\a #\a #\b) #f)
(test (char>? #\c #\c #\b) #f)
(test (char>? #\c #\b #\a) #t)

(test (char>=? #\+ #\+) #t)
(test (char>=? #\+ #\-) #f)
(test (char>=? #\A #\A) #t)
(test (char>=? #\A #\a) #f)
(test (char>=? #\a #\A) #t)
(test (char>=? #\a #\a) #t)
(test (char>=? #\A #\Z) #f)
(test (char>=? #\A #\z) #f)
(test (char>=? #\a #\Z) #t)
(test (char>=? #\a #\z) #f)
(test (char>=? #\Z #\A) #t)
(test (char>=? #\Z #\a) #f)
(test (char>=? #\z #\A) #t)
(test (char>=? #\z #\a) #t)
(test (char>=? #\a #\b #\c) #f)
(test (char>=? #\a #\a #\b) #f)
(test (char>=? #\c #\c #\b) #t)
(test (char>=? #\c #\b #\a) #t)

; --- strings ---

(define (string-downcase s)
  (apply string (map char-downcase (string->list s))))

(test (make-string 0) "")
(test (make-string 1) " ")
(test (make-string 3 #\x) "xxx")
(test (make-string 2.0 #\x) "xx")

(test (number->string 0) "0")
(test (number->string 123) "123")
(test (number->string 165 2) "10100101")
(test (number->string 375 8) "567")
(test (number->string 789 10) "789")
(test (string-downcase (number->string 11259375 16)) "abcdef")
(test (number->string +165 2) "10100101")
(test (number->string +375 8) "567")
(test (number->string +789 10) "789")
(test (string-downcase (number->string +11259375 16)) "abcdef")
(test (number->string -165 2) "-10100101")
(test (number->string -375 8) "-567")
(test (number->string -789 10) "-789")
(test (string-downcase (number->string -11259375 16)) "-abcdef")
(test (number->string 1.0) "1")
(test (number->string 123.0) "123")
(test (number->string 123.45) "123.45")
(test (number->string 3.1415926535) "3.1415926535")
(test (number->string 123456789.5) "123456789.5")
(test (number->string 1234567890.1) "1.2345678901e+9")
(test (number->string 12345.67e100) "1.234567e+104")
(test (number->string 1.23450) "1.2345")
(test (number->string 0.12345) "0.12345")
(test (number->string 0.012345) "0.012345")
(test (number->string 0.0012345) "0.0012345")
(test (number->string 0.00012345) "0.00012345")
(test (number->string 0.000012345) "1.2345e-5")
(test (number->string 12345e-100) "1.2345e-96")
(test (number->string -1.0) "-1")
(test (number->string -123.0) "-123")
(test (number->string -123.45) "-123.45")
(test (number->string -3.1415926535) "-3.1415926535")
(test (number->string -123456789.5) "-123456789.5")
(test (number->string -1234567890.1) "-1.2345678901e+9")
(test (number->string -12345.67e100) "-1.234567e+104")
(test (number->string -1.23450) "-1.2345")
(test (number->string -0.12345) "-0.12345")
(test (number->string -0.012345) "-0.012345")
(test (number->string -0.0012345) "-0.0012345")
(test (number->string -0.00012345) "-0.00012345")
(test (number->string -0.000012345) "-1.2345e-5")
(test (number->string -12345e-100) "-1.2345e-96")

(test (string) "")
(test (string #\x) "x")
(test (string #\a #\b #\c) "abc")

(test (string->number "") #f)
(test (string->number "+") #f)
(test (string->number "-") #f)
(test (string->number "0") 0)
(test (string->number "123") 123)
(test (string->number "10100101" 2) 165)
(test (string->number "567" 8) 375)
(test (string->number "789" 10) 789)
(test (string->number "abcdef" 16) 11259375)
(test (string->number "+1010" 2) 10)
(test (string->number "+123" 8) 83)
(test (string->number "+123" 10) 123)
(test (string->number "+123" 16) 291)
(test (string->number "-1010" 2) -10)
(test (string->number "-123" 8) -83)
(test (string->number "-123" 10) -123)
(test (string->number "-123" 16) -291)
(test (string->number "02" 2) #f)
(test (string->number "08" 8) #f)
(test (string->number "0a" 10) #f)
(test (string->number "0g" 16) #f)
(test (string->number " 1") #f)
(test (string->number "1 ") #f)
(test (string->number "+1 ") #f)
(test (string->number "-1 ") #f)
(test (string->number "0.0") 0.0)
(test (string->number "-0.0") -0.0)
(test (string->number "1.0") 1.0)
(test (string->number "-1.0") -1.0)
(test (string->number "12345.0") 12345.0)
(test (string->number "-12345.0") -12345.0)
(test (string->number "1.2345") 1.2345)
(test (string->number "-1.2345") -1.2345)
(test (string->number "0.12345") 0.12345)
(test (string->number "-0.12345") -0.12345)
(test (string->number "-0.00012345") -0.00012345)
(test (string->number "0.1") 0.1)
(test (string->number "0.01") 0.01)
(test (string->number "0.001") 0.001)
(test (string->number "0.0000000000001") 0.0000000000001)
(test (string->number "1e0") 1.0)
(test (string->number "1e-0") 1.0)
(test (string->number "1e1") 10.0)
(test (string->number "1e2") 100.0)
(test (string->number "1e5") 100000.0)
(test (string->number "1e10") 10000000000.0)
(test (string->number "1e-1") 0.1)
(test (string->number "1e-2") 0.01)
(test (string->number "1e-5") 0.00001)
(test (string->number "1e-10") 0.0000000001)
(test (string->number "123.456e0") 123.456)
(test (string->number "123.456e1") 1234.56)
(test (string->number "123.456e2") 12345.6)
(test (string->number "123.456e3") 123456.0)
(test (string->number "123.456e4") 1234560.0)
(test (string->number "123.456e5") 12345600.0)
(test (string->number "123.456e10") 1234560000000.0)
(test (string->number "-123.456e0") -123.456)
(test (string->number "-123.456e1") -1234.56)
(test (string->number "-123.456e2") -12345.6)
(test (string->number "-123.456e3") -123456.0)
(test (string->number "-123.456e4") -1234560.0)
(test (string->number "-123.456e5") -12345600.0)
(test (string->number "-123.456e10") -1234560000000.0)
(test (string->number "123.456e-1") 12.3456)
(test (string->number "123.456e-2") 1.23456)
(test (string->number "123.456e-3") 0.123456)
(test (string->number "123.456e-4") 0.0123456)
(test (string->number "123.456e-5") 0.00123456)
(test (string->number "123.456e-10") 0.0000000123456)
(test (string->number "-123.456e-1") -12.3456)
(test (string->number "-123.456e-2") -1.23456)
(test (string->number "-123.456e-3") -0.123456)
(test (string->number "-123.456e-4") -0.0123456)
(test (string->number "-123.456e-5") -0.00123456)
(test (string->number "-123.456e-10") -0.0000000123456)
(test (string->number "+123.45e+678") 123.45e678)
(test (string->number "-123.45e+678") -123.45e678)
(test (string->number "+123.45e-678") 123.45e-678)
(test (string->number "-123.45e-678") -123.45e-678)
(test (string->number "1.") 1.0)
(test (string->number ".1") 0.1)
(test (string->number "1.e1") 10.0)
(test (string->number ".1e1") 1.0)
(test (string->number "1000e0") 1e3)
(test (string->number "100e1") 1e3)
(test (string->number "10e2") 1e3)
(test (string->number "1e3") 1e3)
(test (string->number ".1e4") 1e3)
(test (string->number ".01e5") 1e3)
(test (string->number ".001e6") 1e3)
(test (string->number "12345678.901d10") 1.2345678901e+17)
(test (string->number "12345678.901e10") 1.2345678901e+17)
(test (string->number "12345678.901f10") 1.2345678901e+17)
(test (string->number "12345678.901l10") 1.2345678901e+17)
(test (string->number "12345678.901s10") 1.2345678901e+17)
(test (string->number "12345678.901D10") 1.2345678901e+17)
(test (string->number "12345678.901E10") 1.2345678901e+17)
(test (string->number "12345678.901F10") 1.2345678901e+17)
(test (string->number "12345678.901L10") 1.2345678901e+17)
(test (string->number "12345678.901S10") 1.2345678901e+17)
(test (string->number "1 ") #f)
(test (string->number "1.1 ") #f)
(test (string->number "1.1e1 ") #f)
(test (string->number "1e") #f)
(test (string->number "1e+") #f)
(test (string->number "1e-") #f)
(test (string->number "1.e") #f)
(test (string->number "e1") #f)
(test (string->number "+e1") #f)
(test (string->number "-e1") #f)
(test (string->number ".e1") #f)
(test (string->number "+.e1") #f)
(test (string->number "-.e1") #f)
(test (string->number ".") #f)
(test (string->number "#b100") 4)
(test (string->number "#o100") 64)
(test (string->number "#d100") 100)
(test (string->number "#x100") 256)
(test (string->number "#e100") 100.0)
(test (string->number "#i100") 100.0)
(test (inexact? (string->number "#i100")) #t)

(test (string-append "" "") "")
(test (string-append "abc" "") "abc")
(test (string-append "" "def") "def")
(test (string-append "abc" "def") "abcdef")
(test (string-append "abc" "def" "xyz") "abcdefxyz")

(test (string-ci<? "test" "test") #f)
(test (string-ci<? "test" "tesa") #f)
(test (string-ci<? "test" "tesz") #t)
(test (string-ci<? "TEST" "tesa") #f)
(test (string-ci<? "TEST" "tesz") #t)
(test (string-ci<? "test" "TESA") #f)
(test (string-ci<? "test" "TESZ") #t)
(test (string-ci<? "TEST" "TESA") #f)
(test (string-ci<? "TEST" "TESZ") #t)
(test (string-ci<? "test" "tes") #f)
(test (string-ci<? "test" "test0") #t)
(test (string-ci<? "test0" "test") #f)
(test (string-ci<? "ab" "cd" "ef") #t)
(test (string-ci<? "ab" "ab" "cd") #f)
(test (string-ci<? "cd" "cd" "ab") #f)
(test (string-ci<? "ef" "cd" "ab") #f)

(test (string-ci<=? "test" "test") #t)
(test (string-ci<=? "test" "tesa") #f)
(test (string-ci<=? "test" "tesz") #t)
(test (string-ci<=? "TEST" "tesa") #f)
(test (string-ci<=? "TEST" "tesz") #t)
(test (string-ci<=? "test" "TESA") #f)
(test (string-ci<=? "test" "TESZ") #t)
(test (string-ci<=? "TEST" "TESA") #f)
(test (string-ci<=? "TEST" "TESZ") #t)
(test (string-ci<=? "test" "tes") #f)
(test (string-ci<=? "test" "test0") #t)
(test (string-ci<=? "test0" "test") #f)
(test (string-ci<=? "ab" "cd" "ef") #t)
(test (string-ci<=? "ab" "ab" "cd") #t)
(test (string-ci<=? "cd" "cd" "ab") #f)
(test (string-ci<=? "ef" "cd" "ab") #f)

(test (string-ci=? "abc" "abc") #t)
(test (string-ci=? "abC" "abc") #t)
(test (string-ci=? "aBc" "abc") #t)
(test (string-ci=? "aBC" "abc") #t)
(test (string-ci=? "Abc" "abc") #t)
(test (string-ci=? "AbC" "abc") #t)
(test (string-ci=? "ABc" "abc") #t)
(test (string-ci=? "ABC" "abc") #t)
(test (string-ci=? "aBc" "AbC") #t)
(test (string-ci=? "abc" "abd") #f)
(test (string-ci=? "abc" "abcd") #f)
(test (string-ci=? "abcd" "abc") #f)
(test (string-ci=? "abc" "abc" "abc") #t)
(test (string-ci=? "abc" "abc" "cba") #f)

(test (string-ci>? "test" "test") #f)
(test (string-ci>? "test" "tesa") #t)
(test (string-ci>? "test" "tesz") #f)
(test (string-ci>? "TEST" "tesa") #t)
(test (string-ci>? "TEST" "tesz") #f)
(test (string-ci>? "test" "TESA") #t)
(test (string-ci>? "test" "TESZ") #f)
(test (string-ci>? "TEST" "TESA") #t)
(test (string-ci>? "TEST" "TESZ") #f)
(test (string-ci>? "test" "tes") #t)
(test (string-ci>? "test" "test0") #f)
(test (string-ci>? "test0" "test") #t)
(test (string-ci>? "ab" "cd" "ef") #f)
(test (string-ci>? "ab" "ab" "cd") #f)
(test (string-ci>? "cd" "cd" "ab") #f)
(test (string-ci>? "ef" "cd" "ab") #t)

(test (string-ci>=? "test" "test") #t)
(test (string-ci>=? "test" "tesa") #t)
(test (string-ci>=? "test" "tesz") #f)
(test (string-ci>=? "TEST" "tesa") #t)
(test (string-ci>=? "TEST" "tesz") #f)
(test (string-ci>=? "test" "TESA") #t)
(test (string-ci>=? "test" "TESZ") #f)
(test (string-ci>=? "TEST" "TESA") #t)
(test (string-ci>=? "TEST" "TESZ") #f)
(test (string-ci>=? "test" "tes") #t)
(test (string-ci>=? "test" "test0") #f)
(test (string-ci>=? "test0" "test") #t)
(test (string-ci>=? "ab" "cd" "ef") #f)
(test (string-ci>=? "ab" "ab" "cd") #f)
(test (string-ci>=? "cd" "cd" "ab") #t)
(test (string-ci>=? "ef" "cd" "ab") #t)

(test (string-copy "") "")
(test (string-copy "abcdef") "abcdef")
(test (begin (let ((s "abc"))
                (let ((s2 (string-copy s)))
                  (string-set! s2 1 #\x)
                  s)))
      "abc")

(test (let ((s (make-string 1))) (string-fill! s #\x) s) "x")
(test (let ((s (make-string 3))) (string-fill! s #\z) s) "zzz")

(test (string-length "") 0)
(test (string-length "a") 1)
(test (string-length "ab") 2)
(test (string-length "abc") 3)
(test (string-length "Hello, World!") 13)

(test (string-ref "abc" 0) #\a)
(test (string-ref "abc" 1) #\b)
(test (string-ref "abc" 2) #\c)
(test (string-ref "abc" 1.0) #\b)

(define s (string #\1 #\2 #\3))
(test (begin (string-set! s 0 #\a) s) "a23")
(test (begin (string-set! s 2 #\c) s) "a2c")
(test (begin (string-set! s 1 #\b) s) "abc")
(test (begin (string-set! s 1.0 #\space) s) "a c")

(test (string<? "test" "test") #f)
(test (string<? "test" "tesa") #f)
(test (string<? "test" "tesz") #t)
(test (string<? "TEST" "tesa") #t)
(test (string<? "TEST" "tesz") #t)
(test (string<? "test" "TESA") #f)
(test (string<? "test" "TESZ") #f)
(test (string<? "TEST" "TESA") #f)
(test (string<? "TEST" "TESZ") #t)
(test (string<? "test" "tes") #f)
(test (string<? "test" "test0") #t)
(test (string<? "test0" "test") #f)
(test (string<? "ab" "cd" "ef") #t)
(test (string<? "ab" "ab" "cd") #f)
(test (string<? "cd" "cd" "ab") #f)
(test (string<? "ef" "cd" "ab") #f)

(test (string<=? "test" "test") #t)
(test (string<=? "test" "tesa") #f)
(test (string<=? "test" "tesz") #t)
(test (string<=? "TEST" "tesa") #t)
(test (string<=? "TEST" "tesz") #t)
(test (string<=? "test" "TESA") #f)
(test (string<=? "test" "TESZ") #f)
(test (string<=? "TEST" "TESA") #f)
(test (string<=? "TEST" "TESZ") #t)
(test (string<=? "test" "tes") #f)
(test (string<=? "test" "test0") #t)
(test (string<=? "test0" "test") #f)
(test (string<=? "ab" "cd" "ef") #t)
(test (string<=? "ab" "ab" "cd") #t)
(test (string<=? "cd" "cd" "ab") #f)
(test (string<=? "ef" "cd" "ab") #f)

(test (string=? "abc" "abc") #t)
(test (string=? "aBc" "abc") #f)
(test (string=? "abc" "abd") #f)
(test (string=? "abc" "abcd") #f)
(test (string=? "abcd" "abc") #f)
(test (string=? "abc" "abc" "abc") #t)
(test (string=? "abc" "abc" "cba") #f)

(test (string>? "test" "test") #f)
(test (string>? "test" "tesa") #t)
(test (string>? "test" "tesz") #f)
(test (string>? "TEST" "tesa") #f)
(test (string>? "TEST" "tesz") #f)
(test (string>? "test" "TESA") #t)
(test (string>? "test" "TESZ") #t)
(test (string>? "TEST" "TESA") #t)
(test (string>? "TEST" "TESZ") #f)
(test (string>? "test" "tes") #t)
(test (string>? "test" "test0") #f)
(test (string>? "test0" "test") #t)
(test (string>? "ab" "cd" "ef") #f)
(test (string>? "ab" "ab" "cd") #f)
(test (string>? "cd" "cd" "ab") #f)
(test (string>? "ef" "cd" "ab") #t)

(test (string>=? "test" "test") #t)
(test (string>=? "test" "tesa") #t)
(test (string>=? "test" "tesz") #f)
(test (string>=? "TEST" "tesa") #f)
(test (string>=? "TEST" "tesz") #f)
(test (string>=? "test" "TESA") #t)
(test (string>=? "test" "TESZ") #t)
(test (string>=? "TEST" "TESA") #t)
(test (string>=? "TEST" "TESZ") #f)
(test (string>=? "test" "tes") #t)
(test (string>=? "test" "test0") #f)
(test (string>=? "test0" "test") #t)
(test (string>=? "ab" "cd" "ef") #f)
(test (string>=? "ab" "ab" "cd") #f)
(test (string>=? "cd" "cd" "ab") #t)
(test (string>=? "ef" "cd" "ab") #t)

(test (substring "" 0 0) "")
(test (substring "abc" 0 0) "")
(test (substring "abc" 0 1) "a")
(test (substring "abc" 0 2) "ab")
(test (substring "abc" 0 3) "abc")
(test (substring "abc" 1 1) "")
(test (substring "abc" 1 2) "b")
(test (substring "abc" 1 3) "bc")
(test (substring "abc" 2 2) "")
(test (substring "abc" 2 3) "c")
(test (substring "abc" 3 3) "")
(test (substring "abc" 0.0 3.0) "abc")

; --- vectors ---

(test (make-vector 0) #())
(test (make-vector 1) #(#f))
(test (make-vector 3 'x) #(x x x))
(test (make-vector 2.0 'x) #(x x))

(test (vector) '#())
(test (vector 'x) '#(x))
(test (vector 1 2 3) '#(1 2 3))
(test (vector (vector 'x)) '#(#(x)))

(test (let ((v (vector))) (vector-fill! v 'x) v) '#())
(test (let ((v (vector 1 2 3))) (vector-fill! v 'z) v) '#(z z z))

(test (vector-length #()) 0)
(test (vector-length #(a)) 1)
(test (vector-length #(a b)) 2)
(test (vector-length #(a b c)) 3)
(test (vector-length #(1 2 3 #(4 5 6) 7 8 9)) 7)

(test (vector-ref #(a b c) 0) 'a)
(test (vector-ref #(a b c) 1) 'b)
(test (vector-ref #(a b c) 2) 'c)
(test (vector-ref #(a b c) 2.0) 'c)

(define v (vector 1 2 3))
(test (begin (vector-set! v 0 'a) v) '#(a 2 3))
(test (begin (vector-set! v 2 'c) v) '#(a 2 c))
(test (begin (vector-set! v 1 'b) v) '#(a b c))
(test (begin (vector-set! v 1.0 'x) v) '#(a x c))

; --- I/O ---

(if (file-exists? testfile) (delete-file testfile))

(test (call-with-output-file testfile
         (lambda (out)
           (write '(this is a test) out)
           (close-output-port out)
           (call-with-input-file testfile read)))
      '(this is a test))

(delete-file testfile)

(test (let ((out (open-output-file testfile)))
         (write '(this is a test) out)
         (close-output-port out)
         (let ((in (open-input-file testfile)))
           (let ((x (read in)))
             (close-input-port in)
             x)))
      '(this is a test))

(delete-file testfile)

(test (let ((out (open-output-file testfile)))
         (display "Hello-World" out)
         (close-output-port out)
         (let ((in (open-input-file testfile)))
           (let ((x (read in)))
             (close-input-port in)
             x)))
      'hello-world)

(delete-file testfile)

(test (begin (with-output-to-file testfile
               (lambda () (write '(this is a test))))
             (with-input-from-file testfile read))
      '(this is a test))

(define (visibility-check x)
  (delete-file testfile)
  (let ((out (open-output-file testfile)))
    (write x out)
    (display #\space out)
    (display x out)
    (display #\space out)
    (write 'the-end out)
    (close-output-port out)
    (let ((in (open-input-file testfile)))
      (let ((vis (read in)))
        (let ((invis (read in)))
          (close-input-port in)
          (list vis invis))))))

(test (visibility-check #f) '(#f #f))
(test (visibility-check 1) '(1 1))
(test (visibility-check 12345678901234567890)
                        '(12345678901234567890 12345678901234567890))
(test (visibility-check -12345678901234567890)
                        '(-12345678901234567890 -12345678901234567890))
(test (visibility-check 1.0) '(1.0 1.0))
(test (visibility-check 12345.6789e+10) '(12345.6789e+10 12345.6789e+10))
(test (visibility-check -12345.6789e+10) '(-12345.6789e+10 -12345.6789e+10))
(test (visibility-check 12345.6789e-10) '(12345.6789e-10 12345.6789e-10))
(test (visibility-check -12345.6789e-10) '(-12345.6789e-10 -12345.6789e-10))
(test (visibility-check #\A) '(#\A a))
(test (visibility-check "x y") '("x y" x))
(test (visibility-check 'foo) '(foo foo))
(test (visibility-check '(1 2 3)) '((1 2 3) (1 2 3)))
(test (visibility-check '#(1 2 3)) '(#(1 2 3) #(1 2 3)))
(test (visibility-check " ") '(" " the-end))
(test (visibility-check #\space) '(#\space the-end))
(test (visibility-check #\newline) '(#\newline the-end))

(delete-file testfile)

(test (begin (with-output-to-file testfile newline)
               (with-input-from-file testfile read-char))
      #\newline)

(delete-file testfile)

(test (begin (call-with-output-file testfile
               (lambda (out) (newline out)
                             (close-output-port out)))
             (call-with-input-file testfile read-char))
      #\newline)

(delete-file testfile)

(test (begin (close-output-port (open-output-file testfile))
             (let* ((in (open-input-file testfile))
                    (e (read in)))
               (close-input-port in)
               (eof-object? e)))
      #t)

(delete-file testfile)

(define foo 'bar)
(test (let ((out (open-output-file testfile)))
         (write '(define foo 'baz) out)
         (close-output-port out)
         (load testfile)
         foo)
      'baz)

(define (with-range lo hi fn)
  (if (< hi lo)
      '()
      (cons (fn lo) (with-range (+ 1 lo) hi fn))))

(delete-file testfile)

(test (call-with-output-file testfile
        (lambda (out)
          (with-range 32 126
            (lambda (x) (write-char (integer->char x) out)
                        (integer->char x)))))
      (with-range 32 126 integer->char))

(define (while-not-eof input fn)
  (let ((c (fn input)))
    (if (eof-object? c)
        '()
        (cons c (while-not-eof input fn)))))

(test (let ((in (open-input-file testfile)))
         (while-not-eof in read-char))
      (with-range 32 126 integer->char))

(test (let ((in (open-input-file testfile)))
         (let ((c (peek-char in)))
           (cons c (while-not-eof in read-char))))
       (cons #\space (with-range 32 126 integer->char)))

; does GC close unused files?
; Set NFILES to a number that is greater than MAX_PORTS in s9.h
(let ((NFILES 100))
  (test (letrec
          ((open
             (lambda (n)
               (open-input-file testfile)
               (if (< n 1) #t (open (- n 1))))))
          (open NFILES))
        #t))

; --- syntax expansion ---

(define-syntax keyword
  (syntax-rules ()
    ((_) '())))
(test (keyword) '())

(define-syntax iff
  (syntax-rules (then else)
    ((_ p then t)
      (and p t))
    ((_ p then t else f)
      (cond (p t) (else f)))))
(test (iff #t then 'foo) 'foo)
(test (iff #f then 'foo) #f)
(test (iff #t then 'foo else 'bar) 'foo)
(test (iff #f then 'foo else 'bar) 'bar)
(test (iff #f then (wrong "wrong")) #f)
(test (iff #t then 'foo else (wrong "wrong")) 'foo)
(test (iff #f then (wrong "wrong") else 'bar) 'bar)

(define-syntax foo-syntax
  (syntax-rules ()
    ((_ x) x)
    ((_ x y ...) (cons x (foo-syntax y ...)))))
(test (foo-syntax 1 2 3 4 5) '(1 2 3 4 . 5))

(define-syntax bar-syntax
  (syntax-rules ()
    ((_) '())
    ((_ x ...) (list '(x x) ...))))
(test (bar-syntax) '())
(test (bar-syntax x) '((x x)))
(test (bar-syntax x y) '((x x) (y y)))
(test (bar-syntax x y z) '((x x) (y y) (z z)))

(define-syntax rev-syntax
  (syntax-rules ()
    ((_ () r) r)
    ((_ (a . d) r) (rev-syntax d (a . r)))
    ((_ x) (rev-syntax x ()))))
(test (rev-syntax ()) '())
(test (rev-syntax (2 1 cons)) '(1 . 2))
(test (rev-syntax ('bar 'foo #t if)) 'foo)

(define-syntax ell
  (syntax-rules ()
    ((_ ((a b) ...) c ...)
       (list '((b a) ...) c ...))))
(test (ell ()) '(()))
(test (ell () 0) '(() 0))
(test (ell ((1 2)) 3) '(((2 1)) 3))
(test (ell ((1 2) (3 4) (5 6)) 7) '(((2 1) (4 3) (6 5)) 7))
(test (ell ((1 2)) 3 4 5) '(((2 1)) 3 4 5))

(define-syntax false
  (syntax-rules ()
    ((_ x y ...)
       (if x (list y ...) (if #f #f)))))

(test (false #t 1 2 3) '(1 2 3))
(test (false #f 1 2 3) (void))

(define-syntax fluid-let
  (syntax-rules ()
    ((_ () expr . exprs)
      (begin expr . exprs))
    ((_ ((v1 a1) (v2 a2) ...) expr . exprs)
      (let ((outer-v v1))
        (set! v1 a1)
        (fluid-let ((v2 a2) ...)
          (let ((r (begin expr . exprs)))
            (set! v1 outer-v)
            r))))))

(test (let ((x 0) (y 1) (z 2))
        (fluid-let ((x #f) (y #f))
          x y)
        (list x y z))
      '(0 1 2))

(cond ((zero? Errors)
        (display "Everything fine!"))
      (else
        (display Errors)
        (if (> Errors 1)
            (display " errors.")
            (display " error."))))
(display #\newline)

(if (file-exists? testfile) (delete-file testfile))
