# coding=UTF8
#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2008 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""
Unit tests for schooltool.gradebook.generations.evolve1
"""

from datetime import datetime
import unittest
import doctest

from zope.annotation.interfaces import IAnnotatable, IAnnotations
from zope.app.generations.utility import getRootFolder
from zope.component import provideAdapter
from zope.dublincore.annotatableadapter import DCkey, ZDCAnnotatableAdapter
from zope.dublincore.interfaces import IWriteZopeDublinCore, IZopeDublinCore

from schooltool.generations.tests import ContextStub
from schooltool.intervention.generations.tests import provideAdapters
from schooltool.intervention.generations.evolve1 import evolve
from schooltool.intervention.intervention import InterventionMessage
from schooltool.intervention.intervention import InterventionGoal


def doctest_evolve1():
    """Evolution to generation 1.

    First, we'll set up the app object:

        >>> provideAdapters()
        >>> provideAdapter(ZDCAnnotatableAdapter, adapts=(IAnnotatable,),
        ...                provides=IWriteZopeDublinCore)
        >>> context = ContextStub()
        >>> app = getRootFolder(context)

    We'll set up our test with data that will be effected by running the
    evolve script:

        >>> root = app[u'schooltool.interventions'] = {}
        >>> year = root['2009'] = {}
        >>> student = year['jdoe'] = {}

        >>> messages = student['messages'] = {}
        >>> message1 = messages['1'] = InterventionMessage('', '', '')
        >>> message1.created = None
        >>> dublin = IZopeDublinCore(message1)
        >>> dublin.created = datetime(2009, 1, 1, 1, 1, 1)

        >>> goals = student['goals'] = {}
        >>> goal1 = goals['1'] = InterventionGoal('', '', '', '', '', '', '', 
        ...                                       '')
        >>> goal1.created = None
        >>> dublin = IZopeDublinCore(goal1)
        >>> dublin.created = datetime(2009, 2, 2, 2, 2, 2)
        >>> dublin.creators = (u'sb.person.jdoe',)

    Finally, we'll run the evolve script, testing the effected values before and
    after:

        >>> message1.created is None
        True
        >>> DCkey in IAnnotations(message1)
        True
        >>> goal1.creator is None
        True
        >>> goal1.created is None
        True
        >>> DCkey in IAnnotations(goal1)
        True

        >>> evolve(context)

        >>> message1.created
        datetime.datetime(2009, 1, 1, 1, 1, 1, tzinfo=tzinfo(0))
        >>> DCkey in IAnnotations(message1)
        False
        >>> goal1.creator
        u'jdoe'
        >>> goal1.created
        datetime.datetime(2009, 2, 2, 2, 2, 2, tzinfo=tzinfo(0))
        >>> DCkey in IAnnotations(goal1)
        False

    What if the intervention container doesn't exist yet in the
    application:

        >>> context = ContextStub()
        >>> app = getRootFolder(context)
        >>> evolve(context)

    """


def test_suite():
    return unittest.TestSuite([
        doctest.DocTestSuite(optionflags=doctest.ELLIPSIS
                                         | doctest.NORMALIZE_WHITESPACE
                                         | doctest.REPORT_NDIFF
                                         | doctest.REPORT_ONLY_FIRST_FAILURE),
        ])

if __name__ == '__main__':
    unittest.main(defaultTest='test_suite')

