#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""
Student Intervention widgets.
"""
from zope.browser.interfaces import IAdding 
from zope.formlib.interfaces import MissingInputError
from zope.formlib.interfaces import IBrowserWidget, IInputWidget
from zope.app.form.browser.submit import Update
from zope.browserpage import ViewPageTemplateFile
from zope.component import getUtility, queryUtility
from zope.interface import implements
from zope.security.proxy import removeSecurityProxy

from schooltool.contact.interfaces import IContactable, IContact, IUniqueFormKey
from schooltool.contact.browser.relationship import get_relationship_title
from schooltool.course.interfaces import ISectionContainer
from schooltool.email.interfaces import IEmailUtility
from schooltool.group.interfaces import IGroupContainer
from schooltool.person.interfaces import IPerson
from schooltool.schoolyear.interfaces import ISchoolYear
from schooltool.term.interfaces import IDateManager

from schooltool.intervention import InterventionGettext as _
from schooltool.intervention import intervention


class WidgetBase(object):
    """Base Class for all Widget Classes"""

    implements(IBrowserWidget, IInputWidget)

    template = None
    _prefix = 'field.'
    _error = ''

    # See zope.formlib.interfaces.IWidget
    name = None
    label = property(lambda self: self.context.title)
    hint = property(lambda self: self.context.description)
    visible = True

    # See zope.formlib.interfaces.IInputWidget
    required = property(lambda self: self.context.required)

    def __init__(self, field, request):
        self.context = field
        self.request = request

    def isAddView(self):
        return IAdding.providedBy(self.context.context)

    def setRenderedValue(self, value):
        """See zope.formlib.interfaces.IWidget"""
        pass

    def setPrefix(self, prefix):
        """See zope.formlib.interfaces.IWidget"""
        self._prefix = prefix

    def applyChanges(self, content):
        """See zope.formlib.interfaces.IInputWidget"""
        field = self.context
        new_value = self.getInputValue()
        old_value = field.query(content, self)
        if new_value == old_value:
            return False
        field.set(content, new_value)
        return True

    def hasInput(self):
        """See zope.formlib.interfaces.IInputWidget"""
        return True

    def hasValidInput(self):
        """See zope.formlib.interfaces.IInputWidget"""
        return True

    def getInputValue(self):
        """See zope.formlib.interfaces.IInputWidget"""
        return None

    def error(self):
        """See zope.formlib.interfaces.IBrowserWidget"""
        return self._error

    def __call__(self):
        """See zope.formlib.interfaces.IBrowserWidget"""
        return self.template()


class PersonListWidget(WidgetBase):
    """Person List Widget"""

    template = ViewPageTemplateFile('templates/person_list.pt')

    def __init__(self, field, request):
        super(PersonListWidget, self).__init__(field, request)

        # set student from field, view type
        if self.isAddView():
            self.student = IPerson(field.context.context)
        else:
            self.student = IPerson(field.context)
        self.student = removeSecurityProxy(self.student)

        # establish email capability and error message
        utility = getUtility(IEmailUtility)
        self.enabled = utility.enabled()
        self.message = ""
        if not self.enabled:
            self.message = _('Email server is not enabled')
        else:
            sender = IContact(self.request.principal._person)
            if not sender.email:
                self.message = _('User has no email address')
                self.enabled = False

        # build responsible, notified and changed lists of contacts
        term = queryUtility(IDateManager).current_term
        advisors = [IContact(removeSecurityProxy(advisor))
                    for advisor in self.student.advisors]
        groups = IGroupContainer(ISchoolYear(term))
        administrators = [IContact(member)
                          for member in groups['administrators'].members]
        responsible = advisors + [contact for contact in administrators
                                  if contact not in advisors]
        for section in ISectionContainer(term).values():
            if self.student in section.members:
                for teacher in section.instructors:
                    if IContact(teacher) not in responsible:
                        responsible.append(IContact(teacher))
        notified = [IContact(self.student)]
        notified += [contact for contact in IContactable(self.student).contacts]
        if self.isAddView():
            changed = []
        else:
            changed = [contact
                       for contact in field.context.at_one_time_responsible
                       if contact not in responsible + notified]

        # set choices attribute with complete list of contact choices
        self.choices = responsible + notified + changed

        # build value list according to view type, request
        valueList = self.getRequestValue()
        if not self.isAddView() and not Update in self.request:
            valueList = getattr(field.context, field.getName())

        # build final view attributes: responsible, notified and changed
        self.responsible = []
        self.notified = []
        self.changed = []
        for choice in self.choices:
            checked = ''
            if choice in valueList:
                checked = 'checked'

            value = intervention.contactName(choice)
            if choice in advisors:
                value = '%s (%s)' % (value, _('advisor'))
                if self.isAddView() and not Update in self.request:
                    checked = 'checked'

            if choice in IContactable(self.student).contacts:
                title = get_relationship_title(self.student, choice)
                value = '%s (%s)' % (value, title.lower())

            person = {
                'name': IUniqueFormKey(choice),
                'checked': checked,
                'value': value,
                'hasEmail': bool(removeSecurityProxy(choice).email),
                }

            if choice in notified:
                self.notified.append(person)
            elif choice in changed:
                self.changed.append(person)
            else:
                self.responsible.append(person)

    def canEmail(self, person):
        return self.enabled and person['hasEmail']

    def getRequestValue(self):
        contacts = []
        for contact in self.choices:
            key = IUniqueFormKey(contact)
            if key in self.request:
                contacts.append(contact)
        return contacts

    def setPrefix(self, prefix):
        """See zope.formlib.interfaces.IWidget"""
        self._prefix = prefix
        self.name = prefix + self.context.username

    def getInputValue(self):
        """See zope.formlib.interfaces.IInputWidget"""
        contacts = self.getRequestValue()
        if not contacts and self.enabled:
            self._error = MissingInputError.__doc__
            raise MissingInputError('', '')
        return contacts

    def hasValidInput(self):
        """See zope.formlib.interfaces.IInputWidget"""
        return bool(self.getInputValue())


class GoalMetWidget(WidgetBase):
    """Goal Met Widget"""

    template = ViewPageTemplateFile('templates/goal_met.pt')
    label = property(lambda self: '')

    def __init__(self, field, request):
        super(GoalMetWidget, self).__init__(field, request)
        
        if self.isAddView():
            value = False
        else:
            value = getattr(field.context, field.getName())

        if Update in self.request:
            value = self.getInputValue()

        self.goal_not_met = not value and 'checked' or None
        self.goal_met = value and 'checked' or None

    def getInputValue(self):
        """See zope.formlib.interfaces.IInputWidget"""
        return self.request['goal_met'] == 'Yes'
