#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2003 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""
The schooltool relationships.

$Id: relationship.py 2366 2004-12-30 14:42:46Z gintas $
"""

import sets
from persistent import Persistent
from zope.interface import implements, classProvides, moduleProvides
from zope.app.traversing.api import getPath
from schooltool.db import PersistentPairKeysDictWithNames
from schooltool.db import MaybePersistentKeysSet
from schooltool.interfaces import IRemovableLink, IRelatable, IQueryLinks
from schooltool.interfaces import ILinkSet, ILink, IPlaceholder
from schooltool.interfaces import IRelationshipSchemaFactory
from schooltool.interfaces import IRelationshipSchema
from schooltool.interfaces import IRelationshipEvent
from schooltool.interfaces import IRelationshipAddedEvent
from schooltool.interfaces import IRelationshipRemovedEvent
from schooltool.interfaces import IRelationshipValencies
from schooltool.interfaces import IFaceted, ISchemaInvocation
from schooltool.interfaces import IModuleSetup, IValency
from schooltool.interfaces import IUnlinkHook
from schooltool.interfaces import IURIObject
from schooltool.component import registerRelationship
from schooltool import component
from schooltool.event import EventMixin

moduleProvides(IModuleSetup)

__metaclass__ = type


class Link(Persistent):
    """A side (view) of a relationship belonging to one of the two
    ends of a relationship.

    An object of this class is in an invalid state until it is passed
    to a Relationship's constructor.
    """

    implements(IRemovableLink)

    __name__ = __parent__ = None

    def __init__(self, source, role):
        if not IURIObject.providedBy(role):
            raise TypeError("Role must be a URIObject (got %r)" % (role,))
        if not IRelatable.providedBy(source):
            raise TypeError("Parent must be IRelatable (got %r)" % (source, ))
        self.source = source
        self.role = role
        self.callbacks = MaybePersistentKeysSet()
        # self.relationship is set when this link becomes part of a
        # Relationship

    def _getTarget(self):
        return self.relationship.traverse(self).source

    target = property(_getTarget)

    def _getTitle(self):
        # XXX This is an ad-hoc bogosity (who said a link's target has a
        #     title?) that will need to be rethought later.
        return self.target.title

    title = property(_getTitle)

    def _getReltype(self):
        return self.relationship.reltype

    reltype = property(_getReltype)

    def unlink(self):
        self.__parent__.remove(self)
        otherlink = self.relationship.traverse(self)
        self.target.__links__.remove(otherlink)
        event = RelationshipRemovedEvent((self, otherlink))
        event.dispatch(self.target)
        event.dispatch(otherlink.target)
        self._notifyCallbacks()
        otherlink._notifyCallbacks()

    def _notifyCallbacks(self):
        for callback in self.callbacks:
            if IUnlinkHook.providedBy(callback):
                callback.notifyUnlinked(self)
            else:
                callback(self)
        self.callbacks.clear()

    def registerUnlinkCallback(self, callback):
        if IUnlinkHook.providedBy(callback) or callable(callback):
            # this also has the nice side effect of notifying persistence that
            # self has changed
            self.callbacks.add(callback)
        else:
            raise TypeError("Callback must provide IUnlinkHook or be"
                            " callable. Got %r." % (callback, ))


class _LinkRelationship(Persistent):
    """The central part of a relationship.

    This an internal API for links.  Basically, it holds references to
    two links and its name.
    """

    def __init__(self, reltype, a, b):
        self.reltype = reltype
        self.a = a
        self.b = b
        for obj in a, b:
            obj.relationship = self
            relatable = obj.source
            assert IRelatable.providedBy(relatable)
            relatable.__links__.add(obj)

    def traverse(self, link):
        """Returns the link that is at the other end to the link passed in."""
        # Note that this will not work if link is proxied.
        if link is self.a:
            return self.b
        elif link is self.b:
            return self.a
        else:
            raise ValueError("Not one of my links: %r" % (link, ))


def relate(reltype, (a, role_of_a), (b, role_of_b)):
    """Sets up a relationship between two IRelatables with
    Link-_LinkRelationship-Link structure.

    Returns links attached to objects a and b respectively.
    """

    link_a = Link(a, role_of_b)
    link_b = Link(b, role_of_a)
    _LinkRelationship(reltype, link_a, link_b)
    return link_a, link_b


class RelationshipSchema:
    classProvides(IRelationshipSchemaFactory)
    implements(IRelationshipSchema)

    def __init__(self, reltype, **roles):
        self.type = reltype
        if len(roles) != 2:
            raise TypeError("A relationship must have exactly two ends.")

        self.roles = roles

    def __call__(self, **parties):
        if len(self.roles) != len(parties):
            raise TypeError("Wrong number of parties to this relationship."
                            " Need %s, got %r" % (len(self.roles), parties))
        L, N = [], []
        for name, uri in self.roles.items():
            party = parties.pop(name, None)
            if party is None:
                raise TypeError("This relationship needs a %s party."
                                " Got %r" % (name, parties))
            L.append((party, uri))
            N.append(name)
        links = component.relate(self.type, L[0], L[1])
        return {N[1]: links[0], N[0]: links[1]}


class RelationshipEvent(EventMixin):

    implements(IRelationshipEvent)

    def __init__(self, links):
        EventMixin.__init__(self)
        self.links = links

    def __str__(self):
        event = self.__class__.__name__
        s = ["%s" % event]
        reltype = self.links[0].reltype
        if reltype is not None:
            s.append("reltype=%r" % reltype.uri)
        title = self.links[0].title
        if title:
            s.append("title=%r" % title)
        for link in self.links:
            try:
                path = getPath(link.target)
            except TypeError:
                path = str(link.target)
            s.append("link=%r, %r" % (link.role.uri, path))
        return "\n    ".join(s) + '\n'

    def __unicode__(self):
        # XXX This is very similar to __str__.
        event = self.__class__.__name__
        s = [u"%s" % event]
        reltype = self.links[0].reltype
        if reltype is not None:
            s.append(u"reltype='%s'" % reltype.uri)
        title = self.links[0].title
        if title:
            s.append(u"title='%s'" % title.replace("'", "''"))
        for link in self.links:
            try:
                path = getPath(link.target)
            except TypeError:
                path = str(link.target)
            s.append(u"link='%s', '%s'" % (link.role.uri, path))
        return u"\n    ".join(s) + u'\n'


class RelationshipAddedEvent(RelationshipEvent):
    implements(IRelationshipAddedEvent)


class RelationshipRemovedEvent(RelationshipEvent):
    implements(IRelationshipRemovedEvent)


def defaultRelate(reltype, (a, role_of_a), (b, role_of_b)):
    """See IRelationshipFactory"""
    links = relate(reltype, (a, role_of_a), (b, role_of_b))
    event = RelationshipAddedEvent(links)
    event.dispatch(a)
    event.dispatch(b)
    return links


class LinkSet:
    """Set of links."""
    # Note: add and addPlaceholder methods are type-checked because we care
    #       a lot about the type of objects in this set.

    implements(ILinkSet)

    __name__ = 'relationships'
    __parent__ = None

    def __init__(self, parent=None):
        self.__parent__ = parent
        self._data = PersistentPairKeysDictWithNames()

    def add(self, link):
        """Add a link to the set.

        If an equivalent link (with the same reltype, role and target)
        already exists in the set, raises a ValueError.
        """
        if not ILink.providedBy(link):
            raise TypeError('link must provide ILink', link)

        key = (link.target, (link.reltype, link.role))
        value = self._data.get(key)
        if value is None:
            self._data[key] = link
        elif IPlaceholder.providedBy(value):
            self._data[key] = link
            value.replacedBy(link)
        else:
            assert ILink.providedBy(value)
            raise ValueError('duplicate link', link)
        link.__parent__ = self

    def _removeLink(self, link):
        key = (link.target, (link.reltype, link.role))
        try:
            value = self._data[key]
        except KeyError:
            raise ValueError('link not in set', link)

        if value is link:
            del self._data[key]
        else:
            raise ValueError('link not in set', link)
        link.__parent__ = None

    def _removePlaceholder(self, placeholder):
        for key, value in self._data.iteritems():
            if value is placeholder:
                del self._data[key]
                break
        else:
            raise ValueError('placeholder not in set', placeholder)

    def remove(self, link_or_placeholder):
        """Remove a link from the set.

        If an equivalent link does not exist in the set, raises a ValueError.
        """
        if ILink.providedBy(link_or_placeholder):
            self._removeLink(link_or_placeholder)
        elif IPlaceholder.providedBy(link_or_placeholder):
            self._removePlaceholder(link_or_placeholder)
        else:
            raise TypeError('remove must be called with a link or a'
                            ' placeholder. Got %r' % (link_or_placeholder,))

    def __iter__(self):
        for value in self._data.itervalues():
            if ILink.providedBy(value):
                yield value

    def addPlaceholder(self, for_link, placeholder):
        """Add a placeholder to the set to fill the place of the given link.
        """
        if (ILink.providedBy(for_link) and
            IPlaceholder.providedBy(placeholder)):
            key = (for_link.target, (for_link.reltype, for_link.role))
            if key in self._data:
                raise ValueError(
                    'Tried to add placeholder as duplicate for link',
                    for_link)
            self._data[key] = placeholder
        else:
            raise TypeError('for_link must be an ILink and placeholder must'
                            ' by an IPlaceholder', for_link, placeholder)

    def iterPlaceholders(self):
        """Returns an iterator over the placeholders in the set."""
        for value in self._data.itervalues():
            if IPlaceholder.providedBy(value):
                yield value

    def getLink(self, name):
        return self._data.valueForName(name)


class RelatableMixin(Persistent):

    implements(IRelatable, IQueryLinks)

    __name__ = __parent__ = None

    def __init__(self):
        self.__links__ = LinkSet(self)

    def listLinks(self, role=None):
        """See IQueryLinks"""
        if role is None:
            return list(self.__links__)
        else:
            return [link for link in self.__links__ if link.role == role]

    def getLink(self, name):
        return self.__links__.getLink(name)


class RelationshipValenciesMixin(RelatableMixin):

    implements(IRelationshipValencies)

    valencies = ()

    def __init__(self):
        RelatableMixin.__init__(self)

    def _valency2invocation(self, valency):
        schema = valency.schema
        this = valency.keyword
        keywords = list(schema.roles.keys())
        if this not in keywords:
            raise ValueError("Incorrect key %r in valency %r used." %
                             (this, valency))
        keywords.remove(this)
        other = keywords[0]
        return {(schema.type, schema.roles[this]):
                SchemaInvocation(schema, this, other)}

    def getValencies(self):
        result = {}
        valencies = self.valencies
        if type(valencies) != type(()):
            valencies = (valencies,)
        for valency in valencies:
            result.update(self._valency2invocation(valency))
        if IFaceted.providedBy(self):
            all_facet_valencies = sets.Set()
            conflict = sets.Set()
            for facet in component.FacetManager(self).iterFacets():
                if (IRelationshipValencies.providedBy(facet)
                    and facet.active):
                    valencies = facet.getValencies()
                    facet_valencies = sets.Set(valencies.keys())
                    conflict |= all_facet_valencies & facet_valencies
                    all_facet_valencies |= facet_valencies
                    result.update(valencies)
            if conflict:
                raise TypeError("Conflicting facet valencies: %r" % conflict)
        return result


class SchemaInvocation:

    implements(ISchemaInvocation)

    def __init__(self, schema, this, other):
        self.schema = schema
        self.this = this
        self.other = other


class Valency:

    implements(IValency)

    def __init__(self, schema, keyword):
        self.schema = schema
        self.keyword = keyword


def setUp():
    # Register the default relationship handler.
    registerRelationship(None, defaultRelate)
