# bitmaps.tcl:
# Chess piece bitmaps used by Scid.

# The piece images used here were generated from freeware chess
# fonts available in TrueType format at the En Passant website
# "Chess Fonts" page, http://www.enpassant.dk/chess/fonteng.htm

# The authors of the original TrueType fonts are:
#   Alpha: Eric Bentzen.
#   Leipzig, Merida: Armando H. Marroquin.


set boardStyles {}


##########
# Size 20 is only used in Material search window, not for boards.
# It has two special extra images, wm20 and bm20, which contain a
# bishop and knight, for indicating a minor piece.
#

image create photo wm20 -data {
R0lGODlhFAAUAMIAAH9/f7+/vz8/PwAAAP///////////////yH5BAEKAAcALAAAAAAUABQA
AANweLp8cCG02cSwNB8RSACctgBAR3iDqJDd5wlidBLCHGb1CQzzx+yPDYAWyJ1gixpSwOLM
CClFESSRup7RImF42zxP0Vpg0EE6SGjSCqxTKYxHN4RJ6sYETHxVNa3vM2gDQyBoGkNbhIdq
FHRBZyAaCQA7}

image create photo bm20 -data {
R0lGODlhFAAUAMIAAH9/f7+/vwAAAD8/P////////////////yH5BAEKAAcALAAAAAAUABQA
AANneLp8cCG02YQYgupj+5CbEgyYAAQCuJHlNYZo1wHDo7VyOjSAebQxS69R25UCvVlmMXIp
TrmhSGgB4S5LzoVQegK+YJtWwLWEFjnzGVL7ftYQMoGQggerZ3CrLealoxomXxJIX1kNCQA7
}

image create photo p20 -data {
R0lGODlh8AAUAMIAAH9/fz8/P7+/vwAAAP///////////////yH5BAEKAAcALAAAAADwABQA
AAP+eLrc/jDKSau9OOvNu/8VAIBkJAhl2ohqe5xuCgTBGL/oQaMwJew30KwWhAkGA0Vv8gvk
LALRUxJ9BTSAk40qFXWzkKZWCmQit5Uo2GdDW4ZuiFQ9koZ33mxZAjhjBidJFDNIRBgBhRQD
Q4t9NH0NP3o1BEgEYjNTDix/SIITfQOIcROIooOFpouekV6PlQMEQ2qaK6QSsZUholGit5GA
BJeAuMIixgDCnwrJAbKLsMPNsiY0VxeeyxGhnoZm2cTb4OMrP88C0X3NVWF+2CLaELnCUTRm
6CfDtQuUwv7G1xb8iHUkmSV1lZy0GpErSZR9DbJVUOULCUQl3VRdPDL+rtsKRM8MxuqDjlcr
FBIflkomK+CdLP8I2Ivg5NIOmxIe5RnygOSzhDKlLGqg01LCGjMhUHQpj1AhmfEYmHIy8JSJ
jlZXAHIUDWRBojWcFnK1zZk/bw9oBLt09lcuMcpA7eS0CU8WVyIeMTBHD9ARdMjkjPt14BhF
QEkddDuhSm7MqIUrrgJ0ZhSDvJIr+7o14x9dB3z9zTtCE3A+nHxiUpNXsFKgXj+mHPN3pKa/
z5cH48LqJJwDVWoT8enYDis4W9k4cjzjliWk0p5ZBn5QcKdvOardXqqXN1nJVjFpxMTNgJw4
4zypj3V6kRlxecYl7g0+mZtewcLQ/vYMjTb+U6lh5fXfJtmVNcpOj/xnGzL/kHaeO/AZ1xtN
AaY3nHk9dZOHKnH0th43M1D4T2KXzebEbKKVFcoMhDEz1y8cvUjIPo3AU2MmNI0zYGEU2eiJ
a3JUqF9PFT6nnnd5GHMdRrScQMeSC3Q23oCdxXaEapAdMI+Sisy1I0YyQslMgOi48iU34AzY
yxlQJTfUA1hRoJMXYmJkHESOLIXIl1v+A5mAMgE2IkS9qLUGdDH9gIt0fprAaHQRxHeHeIfV
eEc2CuV0Z6TrNVYcVrdEodp0ZY36WVVsPrPYb/HxmVFykfrYyJfLddTeCx15MZ8ovJlEVHx1
zoNillrWICgh2zxpeluLfbZVnllK9pefNiiaSopPWLrVD0BZoqnbboOhBexxEQF7bXxuGfdg
rlTEJxt9YDh1n0Dj7rOmjhtVmmmG6E2ArlRpapGmYsDa6+2qlwYcxAWHyrHwwxAX1h47EVds
8cUYZ6xxBwkAADs=
}

image create photo e20 -height 20 -width 20
set x 0
foreach p {wp wn wb wr wq wk bp bn bb br bq bk} {
  image create photo ${p}20 -width 20 -height 20
  ${p}20 copy p20 -from $x 0 [expr $x + 19] 19
  incr x 20
}


####################
# Alpha:

lappend boardStyles Alpha

set pieceImageData(Alpha,25) {
R0lGODlhLAEZAMIAAL+/vz8/PwAAAH9/f////////////////yH5BAEKAAcALAAAAAAsARkA
AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq4QAHSBELD0QFcuLN+rzUsC3yZAmP0WA6EiqQEM
gq+jI9gRWKVL5YGZcUI7SW038IQNjBYnUz14haOSM3crzzyt4kvYA4gJdGg4e2xuSXARd0KJ
dlZUQ2GBGGcABACOGH2TegMEf0kETAKgeQ1yinUafzpWkXqNeE0xAYcTd56NNrgUi0uwqWCc
BKNjwcJFrXGzlbMWn7igjcMUAcwKfcgVqhu2vhd9r6zYENwCtIjg6N0SMXCW4hPaGp/Ccziz
lsXHFtTGwtQV8/oJrBehHJpZ8cbJYIevnDt2AMGRaiELWiMn+9K9Q5L+Dt1EBqwW+Nm10GEQ
k9TKASxGME45gf3+1SJCzxiZbJRgCrNkjg4cUcz6dFqgpgG/neyWKfO3cYtEV3iSvJrVlJvV
bB1f9TS04OIBS1esETy6rFwbajn1TePUJw0Znf14ShBqTJTNng6ohdkr8wERVU4q2XjbRgFQ
BnRjUkp8d1qjptb8ZDWI1xq4Tlor6N2bpO+UsGCpSI7SCDERP6d5nmZVpHLkxhMypR1o813A
aPQ+IiHDGZXfM1mE/bHLhUygNjCFLo2rO7QV15ZTpksE/BxuzOoQ8eZ8E0JIp6Vf+RiJxAsu
5ebxYIxAlukuF3BpQgMi0G7dlTpbEn0yY+n+k2AYxVBYV/GdAVdCRmlEQRCSgQNWGwiCdBk6
EXU0kRf9ZTXYFwSiU4SB4PwVgRp6bQZdUjAtZ0kcs9knTS35GKMfA2RY0Y+LeLSCHEz/6bTe
AyQ+xhUQWdUYxFDWRfMUSRZO0OBkO5SSXlRBkNOGblsshmUWBgkZY4REyZcidEvABUpsVJlJ
lTkBxeViTFjasiUd04XGCZiWTUYmeNPFBmVmpUA4IXYXdUefcBfIoNIBy93o5CycXWPBm446
OZuP2DRaF6ZYlTbpc1k0CKGTMnAX5YKTOfZnpi5c9pdWbaEJ20oKtFfpOtt1BplhZuKJRIw8
kmKJTpR2QqZzi9L+yuh0ZTzK3a4HQMkklMICJZ5Ujxk7TVqxenOkj74G18+MDxSL6IKCSUYG
Jb7ShS0uwVRD6lRdaNhsMtzNGW2qqLY5SlgUjSKjjJzIO6KYrXl76bjhRtvbvQs+LBVW6xrj
hCx+VswcRBQgC2oak00Mz8MNdyUxxAUJNhVyG2VyRoidHYuwP3syujDDu/x3Ermm5fosdDsK
xi2WL9HzbSUle+yVZiGj/MC7UdlTJckGP63uy6wQLRkntsQwzcy1ZWOm0ePwVeMZPhd0s5uk
5tQj0kTbx1NdDT+JDmRQ1lg2JDDvVdCfAEOw2pF3EEE0doOLEq6BY1ftQGBjy2VUpCua3sOd
d2PfB8TaSE+T+UbkdASd0hIlK1JH+97tHeDhmiXxik6e3NlMsvs2js8ka2HKXirhk+/juD/8
ju8nm+7A7hIb+jTguoXeJAPO9/48Yqy3O7XExpcSPF8fsSH76LW/0YD3e0Dfmznkfz9i+AO2
wL5r6T8M/vvjPxxopI+zPySQ9K8ffs1YCKAAB0jAAhrwgAhMoAIXGIEEAAA7
}

set pieceImageData(Alpha,30) {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}

set pieceImageData(Alpha,35) {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}

set pieceImageData(Alpha,40) {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=
}

set pieceImageData(Alpha,45) {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}

set pieceImageData(Alpha,50) {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}

set pieceImageData(Alpha,55) {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==
}

set pieceImageData(Alpha,60) {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}

set pieceImageData(Alpha,65) {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}

set pieceImageData(Alpha,70) {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}


####################
# Leipzig:

lappend boardStyles Leipzig

set pieceImageData(Leipzig,25) {
R0lGODlhLAEZAMIAAH9/f7+/vz8/P////wAAAP///////////yH5BAEKAAcALAAAAAAsARkA
AAP+eLrc/jDKSau9OOvNu/8gBYQkFgBAoKDjcaZZW3LybD+verBKHt+ZgAAg0AErrGEtQhwq
hoKVwHlBRY+VAGFrxEoABGIYuW1tCasyRkC4er+AAaoD8xB9IitLrp9fBANuGkSENGdLGHV0
BCeMZG0oh2JjF2eLahsofBsBQoIaQkldTHEDpqeofhFaOqw9jlUsiBZgmEGeHkJtbKMQtWfA
wLMMri6wxbRQwxWWmUqUQYCmRZ++SrLYVEyBC2gr3BMyKUXiGc2DZ9omwLh5fNgpmxLn3gfn
TAuE2ss4cYoXQkzFYcPPQcA9Q3oZLAMvEqwIStK0iAhB15lGYFIAK5L+5SIodqHWleHycFWw
k1wmEJT4BJpBNhd/aUyn0EGRAUKioTKlhSOzAHKCypsQquEdCBSpZHxyhccCLQxrAYLJwqPK
dEFOkvNJAdiAdAV7JNx5gmsDTEopTUkjCGopXVDaAJWTUgKbuNXmvSBbV8KJQP4AqQoH9GbA
E9tagFnhh5I/BrVQhZIjjactpMHyRoAE8yIyk5Aohwn7BAXcF5rtOfa21J7iLZDDnK5MTXbr
ioFOAarpQNROnLDDDQyMk/QBaaWEpm4iRcXjQsfVlfkN5Xctv1+98m6QEWqwTvWYbBki2uW8
Mnd/MVm7w9HS1lNqbMnODvjG7Crn4kzNvXL+qlO3+VKKPwIZp8yBgz3QhgtE7CDHcTAsdlYY
PM01nkBzXWdXZrSgFIZnBf1CAH3mQYRhCgNOsGBP7Y3AS3sOzMeFdyDJGF5vyWFIwRS/mcKI
hTc+MKBQCW6mnEBfEebPOIGUhRNi3FH4VULdNSFQibGdtF1LHmoBxlRdnUGicSN9aGY4FHJx
F1TAIQIGVEKANyMgI/ETj2QF8dijk1cWNCJdpZhlV5zdUUMUioywMaWXJ/TiJV2AZUTEVwOV
1BtKeWo13jjzFTlhmX2pNE5gZRFVZmdUhfrKfF81UoYWPqrKXY/8xKkfclwAuOAX+vHEH3eF
AXVYWEOACgl6mPH+8c5o5Gj0q4iH+LULWKx2NhCaWv1xx5ph/KoaWFYEg9lojMhEqVTC7eRp
PpP62q1sqARnVyRWrIHkvd7u0SMfqU3hCUo8svcFpgPTSBCVh4AzDzZBnvdusfKS0itZDWLW
RGfj3bVrfqfk5Fccwrrh3ymyAssClBYgqExYd+5b3Dzk7StGyhxKO5LIJ/mz5SSQ2CtHoymm
W5wsAPITmsxsVHLmjtsucNNvlz0wxYi7YDBiKX/G0TB3VeXsFATlQsuQFkrfA1GiOKM0FFIo
NmK1mW5FbI2HYOnlUDCmbY2jbluy5fTEJGPJwGhVrVEUsx5Lywt5d7RhaZSTTLWp4Jf+skNG
2mqv6/TQCldA5ZKlmro46I5v9+VF9n1IWrt8Z/B0j1H3R3EW16hrxXZxToGqv4/HJlO4qj/y
net0Zz1wzrQzK8yXuFPDs+6f9VYbqMqIMPI0xu0AOCorGfSUPop0QorLBapIODxyK3gTNtPo
nSXBNBcvEA7fp2eE+AKSv7aCRwMIU5jXkMX/iHK91nVFf/upgQBlACYpyEJ6RsmG0fTnPgWM
KIBJSJKpite3b3HwQfkQRjfkJSI3FQ9vC3PVIdymInLgTRc72p6vYojA+YklPpwix2ms4C+u
1VBz9ngdnrzVP+4J7GzF8xbGTigDi2ipCB4SlCYQCEQzYYp0iEvcCBEjGImucBEeg2uBl+hF
OFhICDIYNIo6hEQ349SCYZQb3Akr6ME5ykeMeBsejKI0x9jxEX4iaCMSvsgyQp6MGDhil/ca
8AJDdrAH/wBFJPPwRUo6EpFcUyQjDWJI0wBkkiYA5RtGScpSmvKUqCxlAgAAOw==
}

set pieceImageData(Leipzig,30) {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}

set pieceImageData(Leipzig,35) {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}

set pieceImageData(Leipzig,40) {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}

set pieceImageData(Leipzig,45) {
R0lGODlhHAItAMIAAH9/fwAAAL+/vz8/P////////////////yH5BAEKAAcALAAAAAAcAi0A
AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987/9AGiBAFARJgkEAoBgO
QkMikRGVMhdS4nWUDDyPo6oRHOpendBsgKreHtql7pf8cQKUbvrmPhQM+SB2SmwAhUsMXneH
JAECBANzentwkhxiVXmTWWxZbpQjWZGVGEsEpkOjewAES3cEhYGlqAtDpqyeq38lAbZrLX6F
wZkoalOpGZiYsZu0n2+du8wswMLDJ6Wnvi13MwJEA6ZSYx5/r8W84cW5i6CO3i5RwcYs5y/c
Mt5e6Y3L5/7+1jqogVevBTZzK8ahieFNCYF8ATGUoybMVrVCjnRZgsX+IB8/KgDGhfGScZDB
Ygmbifol5aGWNP9iOkPGcYFHkQcwmuijZMC7FpDcQVqha4BGF0rcFfGGc9IqW1CjSs0WUQKk
O6KOKrhqtEQto+C0rVASiughslVLkD3XVIOdi3Dh4uGwNiu7rVa8Fgk5L+FAs5B4oZ1WBCqk
VyvdHrQY96nFu6R6NdD6RjKJWuXEqvDYF0WUtS/9qnvkJZbjqagfp4Xwt1me1iLimdOc4o+h
tpcFZ4SGouHUw5AVyxwe3AI6Vg2kJLecgWJI1+CMinWOW5M4s+pCn+CsJnpxiYmExbw4dwPs
Z5rPW6CukqlJBexjH/8uwrcjxKS1c9G3z9b+VdoagKZFY1VIkZgFXL3WGVcHWgVQTgNG0YQ/
DSIzUxz+ZENfB9wZVYVPnW0gIHHddbCWgprVFeCDf1jhS4FlgeCNY7Nsxx9U93VBEglJ9cKK
YUyB0GNqUzV0QTAiRYGTch3pB0x1B4wYipQ6UihjQTZSeJ8TUK7YyT6NJCXkP/JYaQGSnCzZ
GUQd6RQBaB4aSOWJihg4ZlSlmZDPfamJId9pvBz3kIdXNuJcXI4UceYS7ymg3xs1TjhdKx85
IKCHMIJjCpzBXPqBlBVyOJymXXR55GisZDFjoSERaYsfilZQYHKQMdnMpNc9cKJUdQllBa92
dmCHFeXx2Iirpnj+FCqC6MhzSyGnDMWqIRFaQa2pWDzV1QJkLZdenk08VRMn30ilI43f8Jks
aKtBACNvxhJ3X08hjPhUG+B64MdUwfyGraPwRtmZrQBHUuAwYjD21J5QFfYKtBpuKAFnZcZa
n6FoEaksCNi0koshyUbqgaBMvXoscxUYFU6SFtfCDpu0iDtMKKkN6dJpeErTnEz/rlcumKjS
+8GXUmECoHlq5KcPSheAxnKlysCn38EPGLhpG+bg6YiA9xIsURKa+udTu8aBUwWyHpH9QMnP
AtrL0ciEHQ6eeJL9R6MQ1uTyFtK9dlvV4hCp8ik4RzXrHlWG4gcjRDwWk6aU7fxr0Qb+wg1e
0phfGMHht24RtcC15kqrLBQKyoqUpId4Adsm95zysU8iO1uliikcaL/piGD6yS7h2ZwikTCl
zd5YBMltIm1x5mpDiiCrs+RsxesxcdrSbqGBGBF+ouWkJF1Xqs9T4IfQUs/zuegCG5X8rzKF
/W7SVXDgG7CuO6huzcDma6FUgfLP/akVC6Ah1IaIfN1uQreY0E/wwr0sOM9qrgqfz2JSvzeR
aAlquI/+Loc18FUuNhZRWAhPwYFvFHARUYvf8RqoBQ8qZTgvbJwKV5SaDYqIdxoznUsspgEc
/kiHEruAoW7UE/8M0RImdFQhvuCKbVHKJo9ygAMNcyOlOYb+YQ3zWtPaV68LBippNxMIqs5B
wAi0YoewC5whkAiuYEVNGjAzYwsnNzuAOI4XMwTP/XAUxInZzFXQ8gkVbWgcnPWPj2V8QC1+
1DC6saKCB/DGI66gPn6ojBtMuUcSHjmBtfDxKnd0DCQckkVCTuB96tAX6uyYiMnhrXsdJCNM
vtiSzFlvgotYixFUtLFIok9Xc5zjKhLVnYeQZnb5WZau5DYVU4rPYbJTGQ5vQhdmHvIxkHSA
JBnpo0ZyUkR8MUITn0C6KIWFCZIUmSJb8pgdmo19MuRd2m44KlF5kVKwYqfxvES06BWKlo2z
ZTZ1CSETDuREvtSiA4zGC2KGrA3+Gmxoqj5HisIZ7n8QwKLs3APEKC4qZ7vDqHEEMD/ShC0p
C6xdJhuKwWFiMFG3SSAFJheFlb2zcUnATAtnV0ISye+ejTsMu8KhzHV2h1rqEYgrboLGranz
VOI4GL6sFTA5TjFV4+POLfLxNglWYDH8EqmlfIijqOTjj1dTXSGX18dFCTAZkSNFDNNaJk0F
bp8S+Ey0pPmKfOKxX1jpWlul6FN6/iMsqtpXh7IhRuqFgV+tOoVFRgZUf2ALRj+yDUBvlxkP
DjY5FoXKU5/p0qLJ7UN7nF02XSE4sX6Vj3xiXUp76KFVUmiJYlXjEh9WMd6uIkwxbOxwNnCT
9/n1IO/+S9YtF0Ucxjnwe0zrKUAuEhMhqqqSIMMcrLKaVAQxk39FnQxL99iTTEEzlALZkjCU
kk1FWtORvUikpCIESKQO9qxzE8woMbFf9+yjvbYN1vWImR2akbSOjQuHfNHzjTr9khzF8Kxn
PyveAaXWXEglG8U22UoY3S1QiWtvH1KzqtqxVrQea5XHUNNL60xPwGXYHR99Jyyyyo4qGICI
0hpM3WKSpb0FlUkia4q69bbNjoipsUWNtB/vWS26k7hxDdVGsaP2uBOqGpoOlRveBsi2dQ04
TAQVKsQlu9atGU5zXOUq5WZqgGKhxXCBgAywx/10qjwE3XgYq1QHx2knFlH+L40mO7ILb/TM
CR3QjYGR5Q9oVLnLLZtylSKKbY5ZrYUkY6TTW9IiXo2kiAZtmy+KNEOjDdM55plwjwXjRO/Z
JZSVy6ZLCC2mMnWNI6ORfdsAMYT00MaLRnWOy4tbn+irw6TBCeuIVNUjgUWQYAPEfubmzW7O
mgKDw+3jIHaPDJDyYQH2UC+67MdzGNuepfixl0nkGAo3Q4SEQ0ItHShQD/SksqO8IWPC3Wtn
pjow3xDTsbUgcCiaupTuDhdDSULuU6JDxt68tsOLjODKJbIhFEdlYOgMRcs6eil55s6rZSqc
3YREDrkJKL1XLvG87hmVHn0mwIsBFgoV/FNb9nf+BaqkbGA3My9Iw5QwusUIUJ+UiEeEsBda
aOmVzWYw9Nw1ujrxmRGssuHaJGiTLhi2hEOqT6GWAKMGRRKyC1I6dfgG03e3m6V7nSyXii3X
kjim1DJZRmM9OEgTvknBLdjL9rqygdq701q3UEn4yU9Ph2kvtuxQBOGuz1jvGV/iYhAgXs/6
lpEEXsJjw/Cpi1Zae7qlUDCeZo+vO4l1fkO9733IYMXRtnAee0Cm64Y1b1+6yOdtMlUDdV2s
pwkCnCEFu5hMwa/s7VeUe5h/BuW43/Xj8qn6HGLdKt+NJtCFE54MpyH7Ny5WBoQsE3QvQe8O
5jjxWz2myjKGjciPhdD+D04R8cMSe3EZkfkrS2e0spj13jVqd8Q9wpBRDIIbBXgmxEYgjVFe
SHNGyIJUlMUnDGgyYccA6xcjyQdUj3FKwWCAg+cuH6iAtXdjdvCAgwNIa0FZWkB/nEMXPmcu
mYcI4Bd+ZCYHxdYUKpMEmldbOzIByvcPb1ZFUnZzqVY9KVYNP9MH9lZZf4cFQfh5R5M4y4eB
7EeFPyh21iJ9ZFQxiIZfvHUpuKNullcY1MBrrUJNP1WDqGGEAcKGi/ZKAlM4C4EXYkMIlHNu
rLFlAihs6xRnRLJmFOAEiEF/KmN/qaZ8/ReFhfFUthUJeIBBBjN9WsiIXuVyLEUixCRfsrH+
fje1YN/WZjM4YgJoWgu1dLiVhnCSESS1ig8RWE7gTAlWilk0hCUIdi23NlpwiyimJCYShH8H
c/h2ihnIamg0KrLIh6M2WhOjcheEVXfGhdOVZxVVinJYZrRYNNYAd8UWMgvzY4QzL+EBdzOl
jKIYasqCO4ajbbmYUUSDi9R4f5XFgpYYc9zoYBDVE8dlenWyLBHCgAx4jRMgchcEZASpiaIi
dADZKe0oXgq5kPk3DA0hEjgoHk+AB0OXhZGEiLQCkQzoh5tzT094K/jHkM0GPYoIf/VYHBPZ
EahzkVrikhKzkmQmKyIpH170hMLoWMdniQhzIOMDLwWiPhAwe3tKSJOBwIUjuBNKOS5KB4wq
WY8/mVH46DmmhxtGCThIWQdNOZIi+D5LeQxiOZZkWZZmeZZomZZquZZs2ZZu+ZZwGZdyOZd0
WZcYkAAAOw==
}

set pieceImageData(Leipzig,50) {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}

set pieceImageData(Leipzig,55) {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}

set pieceImageData(Leipzig,60) {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}

set pieceImageData(Leipzig,65) {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}

set pieceImageData(Leipzig,70) {
R0lGODlhSANGAMIAAH9/f7+/vz8/PwAAAP///////////////yH5BAEKAAcALAAAAABIA0YA
AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
n9CodEqtWq/YbBQA0HpB3HD4IS4HyGVu7vz1qdvwTPrdmHPZ9fk6vhYM/gMCeHwyAYACC4Z/
gi+AjgMPj45dDpJ/NwCOg4QwAX6Hm5wuioGJoI2WkZYDlA2rmJqiMaSPobIpigB+lIt+iC6v
lautDME0hgJcfra3J7SxzSzPvI6/LcbFw6qSNs8DzNEkiwEEBJ6X4SqZAOXorOV+qOztkMLz
5euq5QSZ9TR/+wiUSpdiFTqCJ/o5UsANWCphlogtwCbDkjWEJPoFpCf+EWOIdfQOGJqXSd6+
g9nu8WOlDx/Kiipf/rAjRojCiB5L3LzkDVVDV9ogPrJBcSZNOj92bVzZMWnTIn90sTwQtZ9J
jmmiBsyULOuAfVZrvDvpL4hBQEI+BR2SjMkjagsdSppjsGuZojDw9jgrk4fWjWGH/AFnM+6B
m4RL/BV4VuVOSWD7vljm8qnfRWLUCuFb1uy3JTcXIPbJuTRnG2oBWd5xOohSwFMFD1YSWoHC
1SYW83XMN3JnE0gbPLvoIPi/QN/KIZOsI/VcqLOT1D6sOq/p62dXGDdVTcL2iqxICVj5m4fG
jVF/xOY68MinAJ66suJqXeXS+/jzBzaRmnj+ok/+LZBaYiwA5Ic5gTBWmEG4/dOKQgGm9Yh8
h1iH3YVDodAfBOe058CANwACX4KkyCaIcp8AIeIBy4xTRD/j+fYVAb5cld+NOLpU3keBKBNb
AwBCwJWPBKoAEI2LhLSZQSrO1l+RODw2YYUO2XXUlUelmNBzDwRJhiRQoqCJcoYB4U10PiDH
SoKpNYhakhvlAs9n14yV45337SdOjEoKBUGCJ0XokI7zMKfDmXSmqdpaSfWGJJorLIbnpDqK
+ZCf23Qnlmo6GrqGXjmwORiCb/lACjn3nTOeIlXO4ZxpVvq4Iwh2KpjpnzHNWiepgZDjKQ6I
elaam7OUhqpmddr+h2MYk+o5gl5lZvPTDIeYMyJaCwaUnqkAjUQWl5+OeqMvSBL7AYboYpsb
nwJFGO1E7P6qQQBcvNXRYIC2+Fu9kwAQZkaM7oAoIPyoZa4018VYXYE9SpluVVLp6oFFf6oL
1LQi0LuTcUkqcu2+G/s7A7n71NiDIjGiOid5p9xwqoy1UiYvCAM/vMq/GLy2S8WS6fwjza9O
iEe1JavbYV04gxC0h4dyto94SZtQs0W8tjC1zbVoCO7FO756MAVHr8IIQ+FpEljYFkU9ca3O
ukzwfQLwuZOgLry31Xy1IlluC85ZSWlmE+IS+AO0RDC4CA7jRBVftl339blM8nA121T+1rC0
tm89+vgGS2N9uNSfM1A4z4sAbBpckSe+9QqfLLVtuMndibbaYMAZWTv2IYNM3bZTmp+XHWj8
HYRCTgKBWrYI/2/nYl+NNmd0k8D85ihMPk5AyNOQOOa1oMy753V9IPxTxH+58IcrOqB8zmLL
CgpnaFNIMcJ5k0c7CS//HXqVS2WCe//fmFkI7HYjMeDIex7whi2QVRxW7GwbTRmOBZhXmmTA
R3Wwot9Z7jcC61VDZeKhlkG4Vw3GSEx6/eLC38KgqeCBqUsWY8BNnnI+0e3veHNR2dMcJoZh
pOpV0RvgkVzHNBoQ0HfD4aAcxHW3/y1FPkq8wCdYiC4qnpD+ApRBUIAUYot15KOBr8vGsRIV
AURZcCtBI6K9AkIvg0RRAhQs4sjSpRIAvdECqiNhvyDFutj5LlUCjAAQzUdG0RgPjD8jW9Ym
oEAcOUeHj+odIBeptR45J2JB1A4T/xhJPhoJULBxYpzWRL0N0EJ+e5xLxDypAVAiiZBN8WIi
WVTDicQkiDcb1yQAyKnfgUqIFYTFw9h1KtJYQo+OYEwp5zWjZSlrKYZQIl5uQ8hZGixTlTNc
LfoHydtAU0RxgiSvYmi60xEFUON7pi+hsQKPwWxGkDnJF3kHSZQ9cZMIWqYt70a4d1GnXh0Z
XSVyNQEpifObktLRiQ6oOOBgR5/+kMMMs64DwvS1MyLMAosV8TXLFMQNcAvtD5kaBtAQ4EWg
MjykDcmpyAxFYGNP8xqqFIEfcT2PNyrFn728YtEYnGcnf3wkC85TmYwCDo0dFdMzo/nNg8aD
Vs1sV8UemBJbpVRLleDTSCbwqoP+sH5cmdQgQXehOy5Rkhjc5kgLuaV43mhC86xTRFjGqcSZ
VBLuquXi3jUdrukVSI94WmPoUdMZMU9lNxRfMlbCLuyxRBdmtYA9TShJPAlVHa70n36i6sUX
gOQ+UdUWfuLKgSFJhYZfKeKRZMLR4lEoMXj1nSvh0dgcXaqDi01rFyBbiDEJtqy/ZatOjomj
uSTVoSP+XAVdS/WBx6C2dH79jUvr8EIJZG6czRsiegLoNHr4M2P49M2bYldCTnYykx74bKdW
G1htfccZ9UNQTUcrXA10FYavBEqfFKC7WSqQAnPxXViLFlSM6bQ0491kHC1RUTmWM5nFfctx
SzAwdgBxuez0wH3RR06M0SKC1bXuLjvJGWV5cCw78ehsN2IyGRxReOZdaxgdqhLhHahdBsQc
RHNW23IBsMcO5sAZGJjSkmZjv7Q0MmB7dQEw/VG5AjbwCBYc5D5WY6MYavCOX+oo70ZkwiVY
Gjsa6TD0SpaWfZlhdJc8YwXYsckjFpsb2VgXKKe4rV4d6ZbZ50ck5nknkS3+Mn1FScQ9VyAX
5KCXWhZbsEPMo431TWA2G8eOl7BXv6RFc6SxGVo8zTXK031wiV2MtWTkck6bbm6XKRuwdrIr
o+boh780WsdAh5C60cIYcxmQPTjXNbbm6Jz7JlS09t6ZwpW9Z6rDHN5JaVfZ0N2SOjUbX1IZ
mgJDujK/UtimNgNTteOhKk2Vachw7+u72oRwqlTIyxEDZtZ5kjLNsgwDKg9WbJgzswYwiMzV
acfEsKa1ONvCn0nv9XXeeFCoqcNSEf+6vcXGTmPjeWxxrBg/WGXBETlZbRkvu7QXXwmh7/bx
Z82oH5AE0CjHEsgLwIhpAKFP4+AZm1G9ZDlgji7+fsTTbnDS2eewWfhdMTQZ8Hn5uqQK9MxH
+C031mcsj/R4jHTRttpN+nDOnUioP5yBzMH059jpZnudg9w8r/za/Ww2pVDs1BYiu9MuGbl4
88Kn+D477kg2OZIuYnM6ffaLNGWVKcbTcoa/TY0JfQy0L74x5GII7eoDH3tNDfG9V2/VrTEm
hmmeOaEPPdoHj07WRTKtZ+y7hIAmFTzIwbxVjflIQGU12hO6rMJnIH8x/mx8AW2pg1JbnQ0v
yD12/0zNrgQFmWD9et7xxb9PRSnzRIYWmeluzF2LXoiF+B1IdPflBtpmF5X8eFDOajt5zFJM
P/pt5RpzBkup0uhWtUr+G2/4hV2SOwYHW3uNndCzPM1+jUYwzxBZ4zYp53d5aqc/5KBe2/VX
QlR88LRU3mYkdfdMjNdMTwU6MbIMlEYqHcgUZLOAUyF9SBI1SKccjOESBERxvtV98iZpD6N0
/CV5WnEexDUnkKdpN1h+uyYX7QcZZwJ/wSc+1fUYS7cwXPJfrZRKMyKCQ+SEATZ8o4IqqkF2
wwV3EZaDuBdjKcgYXmWFDtVjMxRvJad3WoRQA0deV8QB4jIbn7UzWbQ7A7Yz4yZ4p6du46Rn
LDdicdM6eahHaJdWjYGANEgPUOhuVWd1UKZ+lOSDqwJsXdVJBfFCWTdIWXcmIEc13LMyDjP+
Iq/3bAxCYwUGeVsYY0pBMmTYUxa3bmFwUEOIfL0CICq0bfPBDxByDjn4T3rmh7mgXVHRhj0i
Lpnma0x0E2RRMEdXKL4ydXhYcSMgiKFIVmW1NImWL5rIEegHfj61KjMFUI7hRTPlK7n4fkHD
Qxbxfh3AdIWiFex4TCChEPjAiLaHbR23WZC3cTdidr7QIWbXgwCjTvk4gSogHqvEhFfWa/DV
TM8gI2MGhVIIdKyER+6GDK/RfsnYDk60DDeGagKZXjZzj4+3eceCKttTNDkIjdHYCQCpUTki
MoRIg/DTAQ6zEk5oYe+gXO9Yg8nRajyykhiXc7e3GD4ZX89jduX+h4AFtobMljvXE1P4JHOW
AnaUlzCrpEPzqHWHJxCVxowiJ3tdyXLjp1Wep4hVJI1REUfKR2xBd2rk8ZLoMkdhgDwciU4p
VGVQBZOZx4bHJH2yxhQ8+BU5mRwHEixt9WRAiQFElUU1lTtecyeXVXBGSUQyOAG712dk0naY
NZV8YSVUJhX6Vk33EDdd6T+GsF7sQJrv4C120iaw+DCBuEtT84gBBhYUlCsyaG8hZiFRuJu/
RJbDUAY7dBebKX+Zc2M6KYSBBRICmCDoWJiydZhwNg+oCDcSFTRTWXs5pVM9tlmfiTjVhoVY
OIzeeY7whicaI2ZK41vJ5y3JtwsFmQz+7sl8gKmQbneFdFRwcEJB8KmJ2LdeJ0hVb4ch3ckj
+IaTKel4jlaeSESLDshngeUtrYMMPCQz8ukrePVJPhl0J6CTWOifkyBT2Jl/3qloZ2FBV0mE
wDdf8TaZe3VGXPhuRKZhRNMuOIcu7wFK9ZkbD7Oh+UlHihlrqwaVjsdu0LRKx8AgJMacTjeJ
RPqiccIvItCYH4WbeBWYOcpsHQpadqmXr9ePvdKKhhRwIfqKEQUxwqmKdEedyjJgLDagGNB9
TrpdHVQtBZmgWNI8keSmDnoheupw5gY96zNz+1kaP4ehOXKAIjRXZCY2kWKZcSqn4CVhVYol
5phYB7adyqb+dNPpS4ICpwE5liFAggW0pQPZIyRpMCQBIY+GcpqUoVy4O3OKUTGmMTxplmF3
Ap0TNxdKOtgBqZfnL5kRK1JVA55Aa/cAaeXZiiXDorr4qPHWp5Enq+ZFqw2lSSHnWGV4VqNU
R8zxGs+Znf+ID3MgTzbwcv0CRAw6Ra3qrPF2barKrvhwTS6Akmg6nsD2glSaleTWmuZUA02q
UeKkMXmSizPXkjSRIwBKoD0CrwEInenZKzsHPDwqlrFTTF+CqREGrlOmJgi2KTqScpIkJ1aF
lAxrkljqqrkHehoncZ8UTwNYPHTkWLk4RWZwpudEdXNZsWpJdSd6aM82bLDyrx3+yTm0x64w
4lOOhlRDO2/cR5dB5k4c5497smiPtmgceBzypKXxRhYM0z9AWxdiOrLjObB1Crb/io0wAI0E
S3pqtWhl9JF3k4t3Ry8AxKwO94P7F2CgKgKYka8lCqVXSLZUZhdaurbVVKtYSjQIlHaRmR/r
N6fSGR6RsVAOC7lZ6zoQGBkMs4D02hjriFypCj3DQhJLi6CIi0K95LYcgjU4dXldYZqJdlpH
mkxSsnnIqZTpiJe9uQF9aYuSx26yVizB+Bbv0QI3IQgSKxwJeCeP653Lpau7FDdiMXwWKIYY
iLt66Tgeyjg6mjCneiG9ZSzGyymqm3ZwO3eOJzdRQZL+Isp+8wmEc7W3nzcs25sd3eu9nduz
G5BsZAGtF/A8goKPhim/2Qu+WHuGK5eGVIi9RJtLIkty1Rg5AboIc7ZyFgEiMZAwozBWMOQ5
9qG/PBOZ8cHAJtcxkOhkqbGyi0h7RGW/E7yZOvQ8S1p0jvpz/vu/8trBjZux8edCVZGPEWO4
OOQvqLpRZyoI4mlxfbsY3jq5vkHEPZy7ENuw1cdDEUdqGrzB7eu3n8a1Ghic40omQtwl3Gis
3whr8EEjhnuQQglWdiJrYZONSJw20GSd/hLFWBqZR7sDAhy1BHx6O1wyYyw6XFyiLNpL7cAn
BtMpihxVMVrC2PqbJYqtMZD+rze8sb/rqwCjmYZcuUBTj/ozNugXE6HZS96EdxyJfjpknTN1
OcFVb/yrbJesWNc6wBkWpVlKX51AtYG8boumdCijMjT7GJfELxRLwhXAxh48Gt8joH0Akx+c
rRO0oy8QNoBzIRslypcqzBxaDV5nOyUysdxcF62nMqzKN7FMnaRaqrVsyw3aAVPUuBFDw1P8
qEczy7w2Cb9bvoqxmWYQa2e6u5icLvhsSrobzWhXyPSsu5Yqo4sCPqtU0Fh5lhCdwhqEsiQn
0f+bzpzkNW1lpz2ksZO4UAzbIdnI0MhcAe+H0VsxeuFXasTK0Aj90Y5jIbFyB5SSTu5z0pOa
Jdz+S9MoLcR9LDsqOwrLa14uLTUMCld2aw8li76Jq20KWqRfu84c8MtxWj4q7DkarX8MPXGe
LFlLfWVNDRQY66QZh7pv0cuQ1tAP29MFVNVW7dC9jFBd/bY1HKf39796oH8bmHM2NgbBY8yU
WtiuUhX4ubAvGtEqgI9yrW0rd9f5TIOSbb4MvV0Ho9NuknCHFpeCLWT8wtJ/JGsJ/UGvuiEq
ZtqcFD9DxdGOWbrjWdc5LbWEozouWTwzNWGqo80uh9J5GapAF6f8iIBaJWyJQ9LFi1l4Ocj5
i8061tmNQRhJfbhCw7s7ZQcldhR/vNEzKq7XfNMsdsuxGtw6TVLKGlz+ERmpZy1W6X1gso1E
Iv0fDRPD1FQxmMNl6wujTIaYvv3TuAzKqw3b9sXEpmqenvnE7Ky7TX1iMUgWuCHD1RpdzAGN
vE2P/S3B/41OzS25kZSDbRKvAkq67/ywzrq4gpPXJduRUJMfwSwcTYyo/DVFLM7PFv7UhdXP
AD7aZJq9f/3eCPzIao2Xlc22Fw6Mq8HgvK0qNPKnmwDh4dPbRd68a6MVDM5g43TSOcunR3fi
oo2dzOra7JqwK5VM47PKdaEcY4RvYDxruVlQnpriKQ2zOW6AIy6T1fLmCPsVOWyrePnlUa7P
SVXlZ26LWR6DUG7jWmqf8ogh3nWSdOrBHZ7+AkTlrAJu3bnMsElMdaeFrk6pVlDrDSQNwKZF
cCqN5/AKwmw25wVE44k71IdKwUUtasut3H9O27bRFYXstOw7TMpgLpOO6Hn3sB+DNdci2QaT
65jRvne13klpqMAOGw0yHFZE0jtXfV9lprbuHabOrqie6hIlwlg211c9m/rj1nfp2y1b6/GN
fxIFjTK2wNnNoI9D2oZd73QR54KE0kOO7E/OpJje7WD07HULYC7i4pgRsO8RzEX6ZgZfeL8O
7ABPyDBZ4dWDrjnm3V9L8TiO7vyq7vV6MRXu5L0wGCijn6FwNHfo8fheRvz+QVpM2cy64Q/V
8Spv4XQjiGwwOavXoa78rfIrLwFt7ZplLToyrxpNXfT9aroeb/PaHpNEPoj57rBIT83VM/W3
Xc1Wf/RF7ug+/ytCWkZlNggQ/l75/DVTny7zWrbtM8gFpfbkibQXzvVdT4+ZFPQuL98YVQFi
7uY+n/aDS/Zp6/aTOgv2XvhXjz+Gn/h9zUimdAdA7/jzMi+KP/lsjwZMAPhYT/lYIoOarfmG
fWiNf/jCAfm3J/meX/iYUPnPqPo50fqu//qwH/uyP/u0X/u2f/u4n/u6v/u83/u+//vAH/zC
P/zEX/zGf/y8nwAAOw==
}


####################
# Merida:

lappend boardStyles Merida

set pieceImageData(Merida,25) {
R0lGODlhLAEZAMIAAL+/v39/fz8/PwAAAP///////////////yH5BAEKAAcALAAAAAAsARkA
AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqmJAAGTCq8heS1etqwT8zt+P1kIwECx0mcHHBqMB
N8xLTsYL+JARoYJo3D0pyk5AMLZeikaAoNshsytls1sxKEvU4x3BXH9TWnVfcAOEZhaEaUUf
hGEWAYV0hJF1d4iTl1KPWEkEnY0VAwRqoWSbGAIEfhM8BIwBe4x7hgujAGqtW3kHpLZFgrSa
vxOMkmeIrqbGnxSPxM6uDgCEvcVck9TCB4DJh524Fy5rr1Vr2Q+aGKze6949DaE9jzSN0las
yxD4GM7phcjlGtCBe0Zw1iR7A+bRotRMnwOHjl6RkbKGHYBOeFigigH+rlSVjzwmNqjlY0Ej
d9I4TrDGoaHBSkUaxhAl0oK0aeBesWPnjsFNKoZOmvlJgWXAhLBeQpipc2cqpUFQdVL5h4zT
TmMEIdElkJI2qFoOzAzL7ZwzqA64EHE2s+aEm4zMzfAIkpxWKl26DgVraKyhsg3GsHMbIVSR
q57MVVy35oK4uo0jcNVR5aucl/V41AnlSZPmbA2JoWVQJxVBpAmLElQVYfFg1kfypkZXL1e0
OpoftWrlWd4wpxCB/VjXFCNsWlJ3T5VLB/gqhkI2HnCB0OAt4qXlDRZ2OviMhKE3I6L5llir
uGfqPnpeXawidAKRi2LVO7uLVNydr7y4kzP+Y2F+BTLfGfplsd4rShi2yx67mDKTRRvxt44a
EnSnGkHnFYKfBGslt9Zo75xWnhLN7CIJJDgxEMNa09wSA1yBHCeWhMatUpw3LeykSz7r+PcN
KAW2hsceLtIXw0s+EqdOfxyu9txqoaGy4wPOZFgMKCIOE9M0rhSy1kNWTlMFjIT8GARSGKWW
RXLY0ehNcA/ugSOIpAXJYzlquEFGnmY255Q0br4JkzPCkEkMi7H02QCiodFpUpYVHlMEIx0y
AiaiviT0DCre6dbZfsX1Bhxa121HIJPDuIDGlqv24sCS7CTZDojhgYgpMs8gSGp3zJlIEBjd
/RqYqlUadqgsa97+mFVhpYoUp5KVgKRmN6hKEAdiISmFJka53YejjKNs+kum0Lh3FrJZlIbV
tFgKay0ykJX7Ti8e7UZOKBQy+ydEUtmiUqk80SphWDjw+taRH+E2JlWLcqFeSGhoaeFSa6yh
pqH2rUJGH47KtxPDZyLWoyAV5dDtcJEtFei3EOBx5Hcrw5JqwhQhBnJae/45hoxs2rxSsIqc
udlQp43jncIhSQF0NhsHO2VaQFvCocipGBiZGjHLLBlddrQbrI0Pg2Rt2GJHCrSBxWCs4XqS
PUwtYp0+C+FoWSGWUdtU9/SbyGy3PAalgL1DddywzkpYYKU4le27UXfstNZ+s7jnIVH6a3wv
IuTQ7ZHkmY+dt6Ny67iMyS5gndCLH31x8sO4tU06SDlYa0vYszMONOjBSjl66qZjTbrqvbGu
N+Jjzrn41sVzezwEhevoqKyiMyAYVj8EU4U3hyNNdt9Qp069ZjJi/vCXkkXtlcRFQz6d+bk+
VAjZli41zsN7hD+/euq/yjVkz2//d8MxCYkRHvO3lJHGf1XgV86cZ8CHNG9WwQlP0bwGJSIs
ykm3ClqI2Mcv84WPfWAjW0QQOBKfYCEKMzgHAiszNojRBQ6aydXwVLi9EfqvhMBwwiaAt0Ja
3euFzIjhWWa4giIa8YhITKISl8jEJjrxiVCM4gYSAAA7
}

set pieceImageData(Merida,30) {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}

set pieceImageData(Merida,35) {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}

set pieceImageData(Merida,40) {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}

set pieceImageData(Merida,45) {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}

set pieceImageData(Merida,50) {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}

set pieceImageData(Merida,55) {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}

set pieceImageData(Merida,60) {
R0lGODlh0AI8AMIAAH9/fz8/PwAAAL+/v////////////////yH5BAEKAAcALAAAAADQAjwA
AAP+eLrc/jDKSau9OOvNu/9gKI5kaZ5oqq5s675wLM90bd94ru987//AoHBILBqPyKRyyWw6
n9CodEqtWpcDAGBw7Xq/quwWfA0EuM8BmmwKCN6CADDO1pjXTXW9Daf//D4AbnEAMXBnEGZC
A314LH19DZBwkpMCJAODhQ0Ab1qeDp1xji9vQIJvAZulqaQKikGMcK4plpcMtpWTmJqhk6sL
oogzpj6DfXIvoqkOg8k+smZvtCe5uJa6kCOyh5UDBASyDcfTxLc95MzKyM3q0JPUJtYL8wr1
H9zu9tgM6fG153a4OQNO2LpP5V7FAWVMwLdwgB4JAAeuGL2JFC3uo0j+QOOHaeHcAHvDUWMn
guKUIdQC7Ea6bi6WzVpArqWOl88k7rq282JPEJZG8lMgk5JKUSx1nORYUYBNFJ0AhCwnS01C
HiQzBlyRtWmlklu7dtxaQZAqBksLOr32MCVRpxTd4DH79ENROHVj3O2Tt8ReLvmu7rin4p7h
oRfoog3KU1swS3PN9OWwdy0OuUwJnIwZJ+40kWpVaRmM0asLsbZKj7UENgOkZ2k1W37rFOli
qU3X+PO7Uss/Fy/ZtaicaZLoyTAIA0zNvLlHCq8f/1oMCdhfmpB+Z2DpyXcOsa1Pq5YJtjqO
k1pEsio/iT2k8BYGpp8NsnhLcjkPnPzmFj3+qvwjPHeDc6w4ZwtyLNyFoAgGNogYBS+twlg/
wknnmH7GoSCgDeBp9QJmU6XCEXr4aZcCfsmpltmKLDK1IQRduRUYgAdU9cBu+zw0loZkDdjc
h5YER0iJNqBYoINIYjDPjA/Y2E4jPl1Ywos0gMiRWwnCxQ1oFEXVJV815GMigyq2aGZmVDqA
WkDFzaamQxAIg8eaPKLzI2fINPXLl3iFCeWRSDaoJGJt5iUYdcPs8xMJacoQm2cLgsANRFIx
oqNsuPE5ygwDDZTimaCu2ChbofnSI4ZuXmPTQOBgOYJtO1TGEAuB6XggU8skCoOR4rHk66/A
BvvrqFHCRF2cs1b+suqDIsAqUGee0XhCNHCZ6WVmy4wpAqsEeCpeoA5iEBicnBxK21Oy2DSu
thWs+2yQLqwrW2pb4DrTrhmKF+q+Hl7gbrnkAoxusgr8uw08PIDIiLQnDMRIpvxeeS+tIkbL
LgfgJinunwLfqA92qdJmLj6pzBfwZfm2IKSBBFHEzcUcjHsIzBpUHLGZg2Qg5gPOUhgRyE/J
RHME0Zg89DbHlKfrCaLcfOYgR1sQTWZQUwyS00y93MHPIj8QyZshH+AGQJfmfEObtixtgqwO
tgxO1SvInJ3VtQlrd7AT10zjMl4bpctTY/NYNrEnzjziMQyH4InLXGLd7civ2swU3GH+OOP4
2614QDCqIRNC1jI2iULZr74EQHXIwKqwcnTLtfJ4oBBTznSDUbe7OrhqDxp6n9mYijrhnJBe
rumToy58w4u3aBCPLuN1OUTAZwC1mdGoLHnEzoBg9m0/RxXVOAvZ9DEFaP+CxrUjWpZFarl/
cDvr1TiPOe7NR4/B+xWqjtfddhvbwfYWyo9M7sO7aySuSSsbA4YqBZLv1UhW7euAKC7FonQx
D3rVel71+OAQUG1wBWxrUKQksLzHoKJcnwDGwhwItI3tjzuHSCG2alMTX3EsBCGckDySN6+M
ZeqDJMAf/FQgp4gVCocfkwnAkpUPAoaNaHyxYQy9hBRMoWL+igiBHMY6GCpXBQgjfHmeZ04W
AodRjXhX8lbcMpY3D2TrNizE0Dci4j03cSNS0aCgpuCAJmRALI1arFnGeITGHmZMR7IrY2r0
lLIw8DBUE2xW3gZIHX1QcgF39JfrVnQX6M2xOnoMUe3eUkhQqccvnTnJ1SrIxTP5DYllasqK
LJilP54pjnZZSE68pJGs9HIicTzGgqZ2JrRdakuhLB4ZKcPGEVqgig9xjumEVMhXgoBtjJxO
lo7Dv/k4k4T+K0pjfllAsZVTAsSk3jG+YQYMZrBFiWSmLa31RA9gxlKQfKfyNre1VrooltAb
jj5vWc8NOOWUC3zcGnx5joVh6jH+qjhV30qJs0cKg18AlGAzOSgVB0lFVjrq2UeYk81FQWWj
X5xVURZ6ocCsQokWsNkV5Te/j37joq0qitsyKs+I4VJS0PLpQFlkzQ7IAlu9sWUgJTnPfX5T
AtOwiJceVrBqdQINoGHhZyQKsJsNAiwUdWVBn4nSg/nQkK8pjwhkdpyCmGxuICyr4r7GOYb8
Kx0SousEmja/oIwRrHtMa1OeikKhEjYCq8SoP+k5VrL+sTKPbew1h+pURgHTD9cqxrUcmBUH
sopYj9wXlx61L9Hl0odTYuNNmRNSfmpgL02FbZZQm1J13EVkvNNmOmJarduhcbRBvR1uTOtG
yrLop/b+DGrEOmSmom6gcTlNKgWRi8qmHleyF3BDbhL60M2uxbuFOOqOKhDaLjI0mYydLG23
xUYwMqea9vOYcVji1pmCKa7rVSSUFFRXUPB3XLzFDctadd6+knReQAUoUZdpVOaa0rgriqcg
revgh8bVupzErmNlI4drLewAdALR90BDVQrw1WkiQd/N4tsAuTlnlFVVrTttwREJu6Y5JTUP
fsF1WJ5F5y5yyEWEzHnfvXZFhJj7BPGE2C/3KbeiB3QhelskXn6JdAMq/mdlL2xYElwrxRmM
qmrE/CWEqGVBFS7mQRfLLxaDL1BR1mQzc7wntWg4AiGk8zlnl1/1RhEytpD+WRb3DKPSOIg/
a0Z0g5rsgXTC080eC6uanxwq527HuBWmrpchPMMe+wI36cpgVEiihcc1btRc1LQ3HGc5rHlx
Ay5OTRDbqwr8wZerZD1wczxdFrn6mS9BSVuwtQkdLrZtfsRDEoFxLWc2L5vBzFRwBcmB4qWa
OM3NhfamMdxpTHRFmJ/1RAxnIYj12UwuAXZcFbFGXJIdOnLtDVSNrW3k994OxuDss7vjjSQT
JU+IGIGmR+2MNAdn7otTzpowsZ3hOxea2w0XKMTTx+uvPPu9xjnELBhtZEl7MLHYczg63xdB
g8Y7h2G0M6/5zWxY8ji1LBcUWX+L5CqiPLgiJ9r+V0W1KUkSLwvUuy/DMwPEaCc8a66tbpct
2zKhpcN05fMcTds9gao4buizpLfUSI7vm08IST8c32tPLlB9jzTmBpJzW5A8Fjs3B41Owsc6
WXlwJzNQwcKUDtYsLT2Q75NrJ514mWEOmS755kosWTteyHE/v1da2lQWO6wB3nMnb9xAL0XS
rfWLOzy93LJoJ1B2rxZ1v7b9de+tX5yvPYqPax2q7k1mLxbj8Wy3vNeEeGvur1jxDVs55/mu
zuVKD3jYO1usTiu6Rg8JlI2jXoe4NZBaXU7ZS5b98/AOPftu3Mr31uv0uhcs9FY/AU0ItfgX
IIlDiS728uZT2zoTYsn+18Zpe/UeomKcZakRKjXnI54lJOcrWfd6EkB5jWRy0PJefaNsfAJj
bZJMeQQD/tFN3oRKr0FfNwOAxNZ/pFcbU5ZmYlAab2R5R8d+6EdeARdKakQd6pZ0BiUaxzQf
MRBrL5Za1iUGd4NeVLd1izNoLLd/kmcHLEd+C/hR2+c7DpI+wFdomhEsxTGD7vd3+LYPqpB/
2MJTtnM9yKdYJ6h2jhN3Y/dHjIA5NJIuLbiEn0ZP90c+LDeF4gVDzVRqY3FYwcFNpaWBB9hg
MTc0e2E6+YNJZ6Vy2bdrL+BookWAbAh5l+OC5Sc5UsQyiTdvXciBJRh58IdOT6Z+gGeI53f+
fwwHaS5UexV1iX3HPt20OkToY4RghV0yewmGdjQjM3pShnMmiGQic3RDgXiDiI2YKb4ifwJY
KsWVCtonDWhIIaIYKivYdy0DGugGNpXYcL33MPX1K5gzhRDAiSFHis2GgV+YRbVDWlbIf5aH
dqnoM84zgtkAO3wyjSVjNzbmbaGHb9QSevGYGNq3gY2GdaDCiyBzCBBBI9NzOSVUjiyzRpRW
bdzIW9G4L6B4EawIH684j//jR/7zjzzGJ+eYCMcnMRsZirp4N/fIfUFyisxxiyXJP6szAtD1
PJtRkc7zjMiYf3wzV1iEh3wnSfl4f1kWDrqoR6o2KBVEgQvGkvn+OIkFGGhPUos9RCaHGFfJ
eIg8uT8NiSu8h5KHp24z9UUROV6NRmQhwy1iVJOnBXFXtmk7GTk4GYfk2E84GG8w+JDFEnqS
ApbLcnKagkSPWB3csYZpaIVBiQE96ZLHeBGCV1pyyV2sGJhSo0AzCZg5mX58CSwmtW2LWZgW
MpkUeCA6GRSbyZnUp33T6JgYyZRkqTmDlAITGJLC0pbLN5R3wyKuiWUOhoPCIpuYqZiQ6Zfg
E5WImZhZEIDY+Jc0mZv7IIcuuYPLl5XfiAq953Wwoz8/mJchEJzSRJp8JpqW1YTQ6UcddXu4
x51sJBpz6JRd6ZUmYHWXmZgM4FAkIZP+tXSZU+mbUslUXcmYKHiU7OkANDhwbpeenfGeCxma
MTeV+vmc+pmb/Nhm4CmEh/l+DYqDtGB1mghtEqqX8qmWWcOa06WgzmabwpJMiZmg+/lm0/mf
/pJ42TgNRwWG7amik8VNrAkss0mbN3l0tlkTKLl730c9vyglqHme6CmP53maeMYcP9pBAsej
SCpIJAqkghSC/FZqctmiPgg79eKPYKOfKNCf/tmUrHcgeLhaWvKOV1pkkplyxQmlN0afylil
Wvg8I8lbQlqiJlSkkSkMhyelrxFNfBQifMkfRjMoT6pXGhBrushav5Y2rekcFfekJ4V2vpin
hdoc2lJna2r+qDoThWcYjpzKboxIp+dppzmCp2nSkkgVO3EgltxJUES4oFfXoEioClXpMoPG
a3jVo/zCp0FIqFwaP5JKnRFggK9BrH84AfM1o43KpvioGjmKpMGoFbgqFs9KL9HaFB8JPrUa
dNmKrIroNEbaN17FonGabTcmpH2UXG5jhcXRrVT4oKByRZyXj+6KWKE3qY0Sb1bRXiRZqclV
G5JakFsEsAUarrQJrwTFmzUCqxB6bcEhgHyKBndUL3upcc7EsE4DaVFEkzClh06xrfrXRjZ6
lArrpV9qsEsUKBLra8VWqRpbqPtGop6qn8PZX7/aiHUTQrpCfFGEEOfospEZpj2zq4sPS6DX
SmVrGaq4d5Q1252pibMVaxx4wLMvJLD56bKnRbIEqp02mY9d6q/kM6EA6B1NQhdky58XA7SJ
eaZYio3WGbCg57XTArZVJ7bpcbYtZrbYiUlpq7bN4rS/sLf4YF9TqrB7cLh5wKESmLQpORqI
+7iQSzSKexSE27iGG7mYm7mau7mc27me+7mgG7qiO7qkW7qme7qom7qqu7qs27qu+7qwG7uy
O7u0W7u2e7sfkAAAOw==
}

set pieceImageData(Merida,65) {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}

set pieceImageData(Merida,70) {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}

######################################################################

# setPieceData:
#   Given a piece font name and size, resets the images for
#   pieces of that size.
#
proc setPieceData {font size} {
  image create photo e$size -height $size -width $size
  image create photo tempPieces -data $::pieceImageData($font,$size)
  set x 0
  foreach p {wp wn wb wr wq wk bp bn bb br bq bk} {
    image create photo $p$size -width $size -height $size
    $p$size copy tempPieces -from $x 0 [expr $x + $size - 1] [expr $size - 1]
    incr x $size
  }
  image delete tempPieces
}

# setPieceFont:
#   Given a piece font name, resets all piece images in all
#   available board sizes to that font.
#
proc setPieceFont {font} {
  #set start [clock clicks -milli]
  foreach size $::boardSizes { setPieceData $font $size }
  #set end [clock clicks -milli]
  #puts "Font: $font. Time: [expr $end - $start] ms."
}

# Ensure the board style is valid:
if {[lsearch -exact $boardStyles $boardStyle] == -1} {
  set boardStyle [lindex $boardStyles 0]
}

# Set up the board style:
setPieceFont $boardStyle
