###
### main.tcl: Routines for creating and updating the main window.
###


############################################################
# Keyboard move entry:
#   Handles letters, digits and BackSpace/Delete keys.
#   Note that king- and queen-side castling moves are denoted
#   "OK" and "OQ" respectively.
#   The letters n, r, q, k, o and l are promoted to uppercase
#   automatically. A "b" can match to a b-pawn or Bishop move,
#   so in some rare cases, a capital B may be needed for the
#   Bishop move to distinguish it from the pawn move.

set moveEntry(Text) ""
set moveEntry(List) {}

# Bind Alt+letter key to nothing, to stop Alt+letter from
# matching the move entry bindings, so Alt+letter ONLY invokes
# the menus:
foreach key {a b c d e f g h i j k l m n o p q r s t u v w x y z} {
  bind . <Alt-$key> {
    # nothing
  }
}

proc moveEntry_Clear {} {
  global moveEntry
  set moveEntry(Text) ""
  set moveEntry(List) {}
}

proc moveEntry_Complete {} {
  global moveEntry
  set len [llength $moveEntry(List)]
  if {$len > 0} {
    if {$moveEntry(AutoExpand)} {
      # Play a bell sound to let the user know the move was accepted already,
      # but only if move announcement is off?
      # bell
    }
    set move [lindex $moveEntry(List) 0]
    if {$move == "OK"} { set move "O-O" }
    if {$move == "OQ"} { set move "O-O-O" }
    set action "replace"
    if {![sc_pos isAt vend]} { set action [confirmReplaceMove] }
    if {$action == "replace"} {
      sc_move addSan $move
    } elseif {$action == "var"} {
      sc_var create
      sc_move addSan $move
    }
    moveEntry_Clear
    updateBoard -pgn -animate
    ::utils::sound::AnnounceNewMove $move
    if {$action == "replace"} { ::tree::doTraining }
  }
}

proc moveEntry_Backspace {} {
  global moveEntry
  set moveEntry(Text) [string range $moveEntry(Text) 0 \
                         [expr {[string length $moveEntry(Text)] - 2}]]
  set moveEntry(List) [sc_pos matchMoves $moveEntry(Text) $moveEntry(Coord)]
  updateStatusBar
}

proc moveEntry_Char {ch} {
  global moveEntry
  set oldMoveText $moveEntry(Text)
  set oldMoveList $moveEntry(List)
  append moveEntry(Text) $ch
  set moveEntry(List) [sc_pos matchMoves $moveEntry(Text) $moveEntry(Coord)]
  set len [llength $moveEntry(List)]
  if {$len == 0} {
    # No matching moves, so do not accept this character as input:
    set moveEntry(Text) $oldMoveText
    set moveEntry(List) $oldMoveList
  } elseif {$len == 1} {
    # Exactly one matching move, so make it if AutoExpand is on,
    # or if it equals the move entered. Note the comparison is
    # case insensitive to allow for 'b' to match both pawn and
    # Bishop moves.
    set move [string tolower [lindex $moveEntry(List) 0]]

    if {$moveEntry(AutoExpand) > 0  ||
        ![string compare [string tolower $moveEntry(Text)] $move]} {
      moveEntry_Complete
    }
  } elseif {$len == 2} {
    # Check for the special case where the user has entered a b-pawn
    # capture that clashes with a Bishop move (e.g. bxc4 and Bxc4):
    set first [string tolower [lindex $moveEntry(List) 0]]
    set second [string tolower [lindex $moveEntry(List) 1]]
    if {[string equal $first $second]} {
        set moveEntry(List) [list $moveEntry(Text)]
        moveEntry_Complete
    }
  }
  updateStatusBar
}


# preMoveCommand: called before making a move to store text in the comment
#   editor window and EPD windows.
proc preMoveCommand {} {
  resetAnalysis
  ::commenteditor::storeComment
  storeEpdTexts
}

sc_info preMoveCmd preMoveCommand


# updateTitle:
#   Updates the main Scid window title.
#
proc updateTitle {} {
  set title "Scid - "
  set fname [sc_base filename]
  set fname [file tail $fname]
  append title "$fname ($::tr(game) "
  append title "[::utils::thousands [sc_game number]] / "
  append title "[::utils::thousands [sc_base numGames]])"
  wm title . $title
}

# updateStatusBar:
#   Updates the main Scid window status bar.
#
proc updateStatusBar {} {
  global statusBar moveEntry
  ::windows::switcher::Refresh
  ::maint::Refresh
  set statusBar "  "

  if {$moveEntry(Text) != ""} {
    append statusBar "Enter move: \[" $moveEntry(Text) "\]  "
    foreach thisMove $moveEntry(List) {
      append statusBar $thisMove " "
    }
    return
  }

  # Check if translations have not been set up yet:
  if {! [info exists ::tr(Database)]} { return }

  # Show "%%" if base is read-only, "XX" if game altered, "--" otherwise:
  if {[sc_base isReadOnly]} {
    append statusBar "%%"
  } elseif {[sc_game altered]} {
    append statusBar "XX"
  } else {
    append statusBar "--"
  }

  set current [sc_base current]
  append statusBar "  $::tr(Database)"
  if {$current != [sc_info clipbase]} {
    append statusBar " $current"
  }
  append statusBar ": "
  set fname [sc_base filename]
  set fname [file tail $fname]
  if {$fname == ""} { set fname "<none>" }
  append statusBar $fname

  # Show filter count:
  append statusBar "   $::tr(Filter)"
  append statusBar ": [filterText]"
}


proc toggleRotateBoard {} {
  ::board::flip .board
}

proc toggleCoords {} {
  global boardCoords
  set boardCoords [expr {1 - $boardCoords} ]
  ::board::coords .board
}

frame .button.space3 -width 15
button .button.flip -image tb_flip -takefocus 0 \
  -command "::board::flip .board"

button .button.coords -image tb_coords -takefocus 0 -command toggleCoords
bind . <KeyPress-0> toggleCoords

button .button.stm -image tb_stm -takefocus 0 -command toggleSTM

proc toggleSTM {} {
  global boardSTM
  set boardSTM [expr {1 - $boardSTM} ]
  ::board::stm .board
}

image create photo autoplay_off -data {
R0lGODdhFAAUAKEAANnZ2QAAAFFR+wAAACwAAAAAFAAUAAACMYSPqbvBb4JLsto7D94StowI
IgOG4ugd55oC6+u98iPXSz0r+Enjcf7jtVyoofGoKAAAOw==
}

image create photo autoplay_on -data {
R0lGODdhFAAUAKEAAP//4AAAAPoTQAAAACwAAAAAFAAUAAACMYSPqbvBb4JLsto7D94StowI
IgOG4ugd55oC6+u98iPXSz0r+Enjcf7jtVyoofGoKAAAOw==
}

button .button.autoplay -image autoplay_off -command toggleAutoplay
button .button.trial -image tb_trial -command {setTrialMode toggle}

foreach i {start back forward end intoVar exitVar addVar autoplay \
             flip coords stm trial} {
  .button.$i configure -relief flat -border 1 -highlightthickness 0 \
    -anchor n -takefocus 0
  bind .button.$i <Any-Enter> "+.button.$i configure -relief groove"
  bind .button.$i <Any-Leave> "+.button.$i configure -relief flat; statusBarRestore %W; break"
}

pack .button.start .button.back .button.forward .button.end \
  .button.space .button.intoVar .button.exitVar .button.addVar .button.space2 \
  .button.autoplay .button.trial .button.space3 .button.flip .button.coords \
  .button.stm -side left -pady 1 -padx 0 -ipadx 0 -pady 0 -ipady 0


############################################################
### The board:

::board::new .board $boardSize
#.board.bd configure -relief solid -border 2
::board::showMarks .board 1
if {$boardCoords} {
  ::board::coords .board
}
if {$boardSTM} {
  ::board::stm .board
}

# .gameInfo is the game information widget:
#
autoscrollframe .gameInfoFrame text .gameInfo
.gameInfo configure -width 20 -height 6 -fg black -bg white -wrap none \
  -state disabled -cursor top_left_arrow -setgrid 1
::htext::init .gameInfo

# Right-mouse button menu for gameInfo frame:
menu .gameInfo.menu -tearoff 0

.gameInfo.menu add checkbutton -label GInfoHideNext \
  -variable gameInfo(hideNextMove) -offvalue 0 -onvalue 1 -command updateBoard

.gameInfo.menu add checkbutton -label GInfoMaterial \
  -variable gameInfo(showMaterial) -offvalue 0 -onvalue 1 -command updateBoard

.gameInfo.menu add checkbutton -label GInfoFEN \
  -variable gameInfo(showFEN) -offvalue 0 -onvalue 1 -command updateBoard

.gameInfo.menu add checkbutton -label GInfoMarks \
  -variable gameInfo(showMarks) -offvalue 0 -onvalue 1 -command updateBoard

.gameInfo.menu add checkbutton -label GInfoWrap \
  -variable gameInfo(wrap) -offvalue 0 -onvalue 1 -command updateBoard

.gameInfo.menu add checkbutton -label GInfoFullComment \
  -variable gameInfo(fullComment) -offvalue 0 -onvalue 1 -command updateBoard

.gameInfo.menu add checkbutton -label GInfoPhotos \
  -variable gameInfo(photos) -offvalue 0 -onvalue 1 \
  -command {updatePlayerPhotos -force}

.gameInfo.menu add separator

.gameInfo.menu add radiobutton -label GInfoTBNothing \
  -variable gameInfo(showTB) -value 0 -command updateBoard

.gameInfo.menu add radiobutton -label GInfoTBResult \
  -variable gameInfo(showTB) -value 1 -command updateBoard

.gameInfo.menu add radiobutton -label GInfoTBAll \
  -variable gameInfo(showTB) -value 2 -command updateBoard

.gameInfo.menu add separator

.gameInfo.menu add command -label GInfoDelete -command {
  catch {sc_game flag delete [sc_game number] invert}
  updateBoard
  ::windows::gamelist::Refresh
}

.gameInfo.menu add cascade -label GInfoMark -menu .gameInfo.menu.mark
menu .gameInfo.menu.mark
foreach flag $maintFlaglist {
  .gameInfo.menu.mark add command -label "" -command "
    catch {sc_game flag $flag \[sc_game number\] invert}
    updateBoard
    ::windows::gamelist::Refresh
  "
}

bind .gameInfo <ButtonPress-3> "tk_popup .gameInfo.menu %X %Y"
bind . <F9> "tk_popup .gameInfo.menu %X %Y"


# setBoard:
#   Resets the squares of the board according to the board string
#   "boardStr" and the piece bitmap size "psize".
#
proc setBoard {board boardStr psize {rotated 0}} {
  for {set i 0} { $i < 64 } { incr i } {
    if {$rotated > 0} {
      set piece [string index $boardStr [expr {63 - $i}]]
    } else {
      set piece [ string index $boardStr $i ]
    }
    $board.$i configure -image $::board::letterToPiece($piece)$psize
  }
}

# updateVarMenus:
#   Updates the menus for moving into or deleting an existing variation.
#   Calls sc_var list and sc_var count to get the list of variations.
#
proc updateVarMenus {} {
  set varList [sc_var list]
  set numVars [sc_var count]
  .button.intoVar.menu delete 0 end
  .menu.edit.del delete 0 end
  .menu.edit.first delete 0 end
  .menu.edit.main delete 0 end
  for {set i 0} {$i < $numVars} {incr i} {
    set move [lindex $varList $i]
    set state normal
    if {$move == ""} {
      set move "($::tr(empty))"
      set state disabled
    }
    set str "[expr {$i + 1}]: $move"
    set commandStr "sc_var moveInto $i; updateBoard"
    if {$i < 9} {
      .button.intoVar.menu add command -label $str -command $commandStr \
        -underline 0
    } else {
      .button.intoVar.menu add command -label $str -command $commandStr
    }
    set commandStr "sc_var delete $i; updateBoard -pgn"
    .menu.edit.del add command -label $str -command $commandStr
    set commandStr "sc_var first $i; updateBoard -pgn"
    .menu.edit.first add command -label $str -command $commandStr
    set commandStr "sc_var promote $i; updateBoard -pgn"
    .menu.edit.main add command -label $str -command $commandStr \
      -state $state
  }
}

# V and Z key bindings: move into/out of a variation.
#
bind . <KeyPress-v> {
  if {[sc_var count] == 1} {
    # There is only one variation, so enter it
    sc_var moveInto 0
    updateBoard
  } else {
    # Present a menu of the possible variations
    focus .
    update idletasks
    .button.intoVar.menu post [winfo pointerx .] [winfo pointery .]
  }
}
bind . <KeyPress-z> {.button.exitVar invoke}

# editMyPlayerNames
#   Present the dialog box for editing the list of player
#   names from whose perspective the board should be shown
#   whenever a game is loaded.
#
proc editMyPlayerNames {} {
  global myPlayerNames
  set w .editMyPlayerNames
  if {[winfo exists $w]} { return }
  toplevel $w
  wm title $w "Scid: [tr OptionsBoardNames]"
  pack [frame $w.b] -side bottom -fill x

  autoscrollframe $w.desc text $w.desc.text \
    -foreground black -background gray90 \
    -width 50 -height 8 -wrap word -cursor top_left_arrow
  $w.desc.text insert end [string trim $::tr(MyPlayerNamesDescription)]
  $w.desc.text configure -state disabled
  pack $w.desc -side top -fill x
  autoscrollframe $w.f text $w.f.text \
    -background white -width 50 -height 10 -wrap none
  foreach name $myPlayerNames {
    $w.f.text insert end "\n\"$name\""
  }
  pack $w.f -side top -fill both -expand yes
  button $w.b.white -text $::tr(White) -command {
      .editMyPlayerNames.f.text insert end "\"[sc_game info white]\"\n"
  }
  button $w.b.black -text $::tr(Black) -command {
      .editMyPlayerNames.f.text insert end "\"[sc_game info black]\"\n"
  }
  button $w.b.help -text $::tr(Help) \
    -command {helpWindow Options MyPlayerNames}
  button $w.b.ok -text OK -command editMyPlayerNamesOK
  button $w.b.cancel -text $::tr(Cancel) -command "grab release $w; destroy $w"
  pack $w.b.cancel $w.b.ok -side right -padx 5 -pady 5
  pack $w.b.white $w.b.black $w.b.help -side left -padx 5 -pady 5
  grab $w
}

proc editMyPlayerNamesOK {} {
  global myPlayerNames
  set w .editMyPlayerNames
  set text [string trim [$w.f.text get 1.0 end]]
  set myPlayerNames {}
  foreach name [split $text "\n"] {
      set name [string trim $name]
      if {[string match "\"*\"" $name]} {
        set name [string trim $name "\""]
      }
      if {$name != ""} { lappend myPlayerNames $name }
  }
  grab release $w
  destroy $w
}

# flipBoardForPlayerNames
#   Check if either player in the current game has a name that matches
#   a pattern in the specified list and if so, flip the board if
#   necessary to show from that players perspective.
#
proc flipBoardForPlayerNames {namelist {board .board}} {
  set white [sc_game info white]
  set black [sc_game info black]
  foreach pattern $namelist {
    if {[string match $pattern $white]} {
      ::board::flip $board 0
      return
    }
    if {[string match $pattern $black]} {
      ::board::flip $board 1
      return
    }
  }
}

# updateBoard:
#    Updates the main board. Also updates the navigation buttons, disabling
#    those that have no effect at this point in the game.
#    Also ensure all menu settings are up to date.
#    If a parameter "-pgn" is specified, the PGN text is also regenerated.
#    If a parameter "-animate" is specified, board changes are animated.
#
proc updateBoard {args} {
  global boardSize gameInfo
  set pgnNeedsUpdate 0
  set animate 0
  foreach arg $args {
   if {! [string compare $arg "-pgn"]} { set pgnNeedsUpdate 1 }
   if {! [string compare $arg "-animate"]} { set animate 1 }
  }

  # Remove marked squares informations.
  # (This must be done _before_ updating the board!)
  ::board::mark::clear .board

  ::board::resize .board $boardSize
  ::board::update .board [sc_pos board] $animate

  # Draw arrows and marks, color squares:

  foreach {cmd discard} [::board::mark::getEmbeddedCmds [sc_pos getComment]] {
      set type   [lindex $cmd 0]
      set square [::board::sq [lindex $cmd 1]]
      set color  [lindex $cmd end]
      if {[llength $cmd] < 4} { set cmd [linsert $cmd 2 ""] }
      set dest   [expr {[string match {[a-h][1-8]} [lindex $cmd 2]] \
                  ? [::board::sq [lindex $cmd 2]] : [lindex $cmd 2]}]
      # add mark to board
      eval ::board::mark::add .board $type $square $dest $color
  }

  # Update the status of each navigation button:
  if {[sc_pos isAt start]} {
    .button.start configure -state disabled
  } else { .button.start configure -state normal }
  if {[sc_pos isAt end]} {
    .button.end configure -state disabled
  } else { .button.end configure -state normal }
  if {[sc_pos isAt vstart]} {
    .button.back configure -state disabled
  } else { .button.back configure -state normal }
  if {[sc_pos isAt vend]} {
    .button.forward configure -state disabled
  } else { .button.forward configure -state normal }
  # Cannot add a variation to an empty line:
  if {[sc_pos isAt vstart]  &&  [sc_pos isAt vend]} {
    .menu.edit entryconfig [tr EditAdd] -state disabled
    .button.addVar configure -state disabled
    bind . <Control-a> {}
  } else {
    .menu.edit entryconfig [tr EditAdd] -state normal
    .button.addVar configure -state normal
    bind . <Control-a> {sc_var create; updateBoard -pgn}
  }
  if {[sc_var count] == 0} {
    .button.intoVar configure -state disabled
    .menu.edit entryconfig [tr EditDelete] -state disabled
    .menu.edit entryconfig [tr EditFirst] -state disabled
    .menu.edit entryconfig [tr EditMain] -state disabled
  } else {
    .button.intoVar configure -state normal
    .menu.edit entryconfig [tr EditDelete] -state normal
    .menu.edit entryconfig [tr EditFirst] -state normal
    .menu.edit entryconfig [tr EditMain] -state normal
  }
  updateVarMenus
  if {[sc_var level] == 0} {
    .button.exitVar configure -state disabled
  } else {
    .button.exitVar configure -state normal
  }

  if {![sc_base inUse]  ||  $::trialMode  ||  [sc_base isReadOnly]} {
    .tb.save configure -state disabled
  } else {
    .tb.save configure -state normal
  }
  .gameInfo configure -state normal
  .gameInfo delete 0.0 end
  ::htext::display .gameInfo [sc_game info -hide $gameInfo(hideNextMove) \
                                -material $gameInfo(showMaterial) \
                                -cfull $gameInfo(fullComment) \
                                -fen $gameInfo(showFEN) -tb $gameInfo(showTB)]
  if {$gameInfo(wrap)} {
    .gameInfo configure -wrap word
    .gameInfo tag configure wrap -lmargin2 10
    .gameInfo tag add wrap 1.0 end
  } else {
    .gameInfo configure -wrap none
  }
  .gameInfo configure -state disabled
  updatePlayerPhotos
  updateEpdWins
  if {[winfo exists .analysisWin1]} { updateAnalysis 1 }
  if {[winfo exists .analysisWin2]} { updateAnalysis 2 }
  if {[winfo exists .treeWin]} { ::tree::refresh }
  if {[winfo exists .commentWin]} { ::commenteditor::Refresh }
  if {[::tb::isopen]} { ::tb::results }
  updateMenuStates
  moveEntry_Clear
  updateStatusBar
  if {[winfo exists .twinchecker]} { updateTwinChecker }
  if {[winfo exists .pgnWin]} { ::pgn::Refresh $pgnNeedsUpdate }
  if {[winfo exists .noveltyWin]} { updateNoveltyWin }
}

# Set up player photos:

image create photo photoW
image create photo photoB
label .photoW -background white -image photoW -anchor ne
label .photoB -background white -image photoB -anchor ne

proc readPhotoFile {fname} {
  set oldcount [array size ::photo]
  if {! [file readable $fname]} { return }
  catch {source $fname}
  set newcount [expr {[array size ::photo] - $oldcount}]
  if {$newcount > 0} {
    ::splash::add "Found $newcount player photos in [file tail $fname]"
  }
}

proc photo {player data} {
  set ::photo($player) $data
}

array set photo {}

# Read all Scid photo (*.spf) files in the Scid data/user/config directories:
foreach photofile [glob -nocomplain -directory $scidDataDir "*.spf"] {
  readPhotoFile $photofile
}
foreach photofile [glob -nocomplain -directory $scidUserDir "*.spf"] {
  readPhotoFile $photofile
}
foreach photofile [glob -nocomplain -directory $scidConfigDir "*.spf"] {
  readPhotoFile $photofile
}

# Read players.img for compatibility with older versions:
readPhotoFile [file join $scidUserDir players.img]

set photo(oldWhite) {}
set photo(oldBlack) {}

# updatePlayerPhotos
#   Updates the player photos in the game information area
#   for the two players of the current game.
#
proc updatePlayerPhotos {{force ""}} {
  global photo
  if {$force == "-force"} {
    # Force update even if it seems unnecessary. This is done
    # when the user selects to show or hide the photos.
    set photo(oldWhite) {}
    set photo(oldBlack) {}
    place forget .photoW
    place forget .photoB
  }
  if {! $::gameInfo(photos)} { return }
  set white [sc_game info white]
  set black [sc_game info black]
  if {$black != $photo(oldBlack)} {
    set photo(oldBlack) $black
    place forget .photoB
    if {[info exists ::photo($black)]} {
      image create photo photoB -data $::photo($black)
      .photoB configure -image photoB -anchor ne
      place .photoB -in .gameInfo -x -1 -relx 1.0 \
        -rely 0.12 -relheight 0.88 -anchor ne
    }
  }
  if {$white != $photo(oldWhite)} {
    set photo(oldWhite) $white
    place forget .photoW
    if {[info exists ::photo($white)]} {
      image create photo photoW -data $::photo($white)
      .photoW configure -image photoW -anchor ne
      place .photoW -in .gameInfo -x -82 -relx 1.0 \
        -rely 0.12 -relheight 0.88 -anchor ne
    }
  }
}

#########################################################
### Chess move input

# Globals for mouse-based move input:

set selectedSq -1
set currentSq -1
set bestSq -1

set EMPTY 0
set KING 1
set QUEEN 2
set ROOK 3
set BISHOP 4
set KNIGHT 5
set PAWN 6

# getPromoPiece:
#     Called whenever a pawn promotion move is made, to query the user
#     for the promotion piece. I'd prefer this to show the piece bitmaps
#     rather than english names, should change it sometime...
#
proc getPromoPiece {} {
  global EMPTY QUEEN ROOK BISHOP KNIGHT
  if {[catch {tk_dialog .dialog "Select Promotion Piece" \
                "Select the promotion piece:" \
                "" 0 "Queen" "Rook" "Bishop" "Knight"} answer]} {
    return 2
  }
  return [expr {$answer + 2}]
}

# confirmReplaceMove:
#   Asks the user what to do when adding a move when a move already
#   exists.
#   Returns a string value:
#      "replace" to replace the move, truncating the game.
#      "var" to add the move as a new variation.
#      "cancel" to do nothing.
#
proc confirmReplaceMove {} {
  global askToReplaceMoves trialMode
  if {! $askToReplaceMoves} { return "replace" }
  if {$trialMode} { return "replace" }

  option add *Dialog.msg.wrapLength 4i interactive
  catch {tk_dialog .dialog "Scid: $::tr(ReplaceMove)?" \
           $::tr(ReplaceMoveMessage) "" 0 \
           $::tr(ReplaceMove) $::tr(AddNewVar) [tr EditTrial] \
           $::tr(Cancel)} answer
  option add *Dialog.msg.wrapLength 3i interactive
  if {$answer == 0} { return "replace" }
  if {$answer == 1} { return "var" }
  if {$answer == 2} { setTrialMode 1; return "replace" }
  return "cancel"
}

proc addNullMove {} {
    addMove null null
}

# addMove:
#   Adds the move indicated by sq1 and sq2 if it is legal. If the move
#   is a promotion, getPromoPiece will be called to get the promotion
#   piece from the user.
#   If the optional parameter is "-animate", the move will be animated.
#
proc addMove { sq1 sq2 {animate ""}} {
  global EMPTY
  set nullmove 0
  if {$sq1 == "null"  &&  $sq2 == "null"} { set nullmove 1 }
  if {!$nullmove  &&  [sc_pos isLegal $sq1 $sq2] == 0} {
    # Illegal move, but if it is King takes king then treat it as
    # entering a null move:
    set board [sc_pos board]
    set k1 [string tolower [string index $board $sq1]]
    set k2 [string tolower [string index $board $sq2]]
    if {$k1 == "k"  &&  $k2 == "k"} { set nullmove 1 } else { return }
  }
  set promo $EMPTY
  if {[sc_pos isPromotion $sq1 $sq2] == 1} { set promo [getPromoPiece] }
  set action "replace"
  if {![sc_pos isAt vend]} {
    set action [confirmReplaceMove]
  }
  if {$action == "replace"} {
    # nothing
  } elseif {$action == "var"} {
    sc_var create
  } else {
    # Do not add the move at all:
    return
  }
  if {$nullmove} {
    sc_move addSan null
  } else {
    # if {[winfo exists .commentWin]} { .commentWin.cf.text delete 0.0 end }
    sc_move add $sq1 $sq2 $promo
    set san [sc_game info previous]
    after idle [list ::utils::sound::AnnounceNewMove $san]
  }
  moveEntry_Clear
  updateBoard -pgn $animate
  ::tree::doTraining
}

# addSanMove
#   Like addMove above, but takes the move in SAN notation instead of
#   a pair of squares.
#
proc addSanMove {san {animate ""} {noTraining ""}} {
  set action "replace"
  if {![sc_pos isAt vend]} {
    set action [confirmReplaceMove]
  }
  if {$action == "replace"} {
    # nothing
  } elseif {$action == "var"} {
    sc_var create
  } else {
    # Do not add the move at all:
    return
  }
  # if {[winfo exists .commentWin]} { .commentWin.cf.text delete 0.0 end }
  sc_move addSan $san
  moveEntry_Clear
  updateBoard -pgn $animate
  ::utils::sound::AnnounceNewMove $san
  if {$noTraining != "-notraining"} {
    ::tree::doTraining
  }
}

# enterSquare:
#   Called when the mouse pointer enters a board square.
#   Finds the best matching square for a move (if there is a
#   legal move to or from this square), and colors the squares
#   to indicate the suggested move.
#
proc enterSquare { square } {
  global highcolor currentSq bestSq bestcolor selectedSq suggestMoves
  set currentSq $square
  if {$selectedSq == -1} {
    ::board::recolor .board
    set bestSq -1
    if {$suggestMoves} {
      set bestSq [sc_pos bestSquare $square]
    }
    if {[expr {$bestSq != -1}]} {
      ::board::colorSquare .board $square $bestcolor
      ::board::colorSquare .board $bestSq $bestcolor
    }
  }
}

# leaveSquare:
#    Called when the mouse pointer leaves a board square.
#    Recolors squares to normal (lite/dark) color.
#
proc leaveSquare { square } {
  global currentSq selectedSq bestSq
  set currentSq -1
  if {$square != $selectedSq} {
    ::board::colorSquare .board $square
  }
  if {$bestSq != -1} {
    ::board::colorSquare .board $bestSq
  }
}

# pressSquare:
#    Called when the left mouse button is pressed on a square. Sets
#    that square to be the selected square.
#
proc pressSquare { square } {
  global selectedSq highcolor
  if {$selectedSq == -1} {
    set selectedSq $square
    ::board::recolor .board
    ::board::colorSquare .board $square $highcolor
    # Drag this piece if it is the same color as the side to move:
    set c [string index [sc_pos side] 0]  ;# will be "w" or "b"
    set p [string index [::board::piece .board $square] 0] ;# "w", "b" or "e"
    if {$c == $p} {
      ::board::setDragSquare .board $square
    }
  } else {
    ::board::setDragSquare .board -1
    ::board::colorSquare .board $selectedSq
    ::board::colorSquare .board $square
    if {$square != $selectedSq} {
      addMove $square $selectedSq -animate
    }
    set selectedSq -1
    enterSquare $square
  }
}

# pressSquare2:
#   Called when the middle mouse button is pressed on a square. This
#   makes the suggested best move.
#
proc pressSquare2 { square } {
  global selectedSq bestSq
  ::board::colorSquare .board $bestSq
  ::board::colorSquare .board $square
  addMove $square $bestSq -animate
  enterSquare $square
}

# releaseSquare:
#   Called when the left mouse button is released over a square.
#   If the square is different to that the button was pressed on, it
#   is a dragged move; otherwise it is just selecting this square as
#   part of a move.
#
proc releaseSquare { w x y } {
  global selectedSq bestSq currentSq

  ::board::setDragSquare $w -1
  set square [::board::getSquare $w $x $y]
  if {$square < 0} {
    set selectedSq -1
    ::board::recolor $w
    return
  }

  if {$square == $selectedSq} {
    if {$::suggestMoves} {
      # User pressed and released on same square, so make the
      # suggested move if there is one:
      set selectedSq -1
      ::board::colorSquare $w $bestSq
      ::board::colorSquare $w $square
      addMove $square $bestSq -animate
      enterSquare $square
    } else {
      # Current square is the square user pressed the button on,
      # so we do nothing.
    }
  } else {
    # User has dragged to another square, so try to add this as a move:
    addMove $square $selectedSq
    ::board::colorSquare $w $selectedSq
    set selectedSq -1
    ::board::colorSquare $w $square
  }
}


# backSquare:
#    Handles the retracting of a move (when the right mouse button is
#    clicked on a square). Recolors squares to normal color also.
#    If the move is the last in the game or variation, is is removed
#    by truncating the game after retracting the move.
#
proc backSquare {} {
  global selectedSq bestSq
  set lastMoveInLine 0
  if {[sc_pos isAt vend]} {
    set lastMoveInLine 1
  }
  sc_move back
  if {$lastMoveInLine} {
    sc_game truncate
  }
  ::board::colorSquare .board $selectedSq
  ::board::colorSquare .board $bestSq
  set selectedSq -1
  set bestSq -1
  updateBoard -pgn -animate
  ::utils::sound::AnnounceBack
}


##
## Auto-playing of moves:
##
set autoAnnotate 1
set autoplayMode 0

set tempdelay 0
trace variable tempdelay w {::utils::validate::Regexp {^[0-9]*\.?[0-9]*$}}

proc setAutoplayDelay {{askAnnotate 0}} {
  global autoplayDelay tempdelay autoplayResult
  set autoplayResult -1
  set tempdelay [expr {$autoplayDelay / 1000.0}]
  set w .apdialog
  toplevel $w
  wm title $w "Scid"
  wm resizable $w 0 0
  label $w.label -text $::tr(AnnotateTime:)
  pack $w.label -side top -pady 5 -padx 5
  entry $w.entry -background white -width 10 -textvariable tempdelay
  pack $w.entry -side top -pady 5
  bind $w.entry <Escape> { .apdialog.buttons.cancel invoke }
  bind $w.entry <Return> { .apdialog.buttons.ok invoke }

  if {$askAnnotate} {
    addHorizontalRule $w
    label $w.avlabel -text $::tr(AnnotateWhich:)
    radiobutton $w.all -text $::tr(AnnotateAll) \
      -variable annotateMoves -value all -anchor w
    radiobutton $w.white -text $::tr(AnnotateWhite) \
      -variable annotateMoves -value white -anchor w
    radiobutton $w.black -text $::tr(AnnotateBlack) \
      -variable annotateMoves -value black -anchor w
    radiobutton $w.notbest -text $::tr(AnnotateNotBest) \
      -variable annotateMoves -value notbest -anchor w
    pack $w.avlabel -side top
    pack $w.all $w.white $w.black $w.notbest -side top -fill x
  }

  addHorizontalRule $w

  set b [frame $w.buttons]
  pack $b -side top -fill x
  button $b.cancel -text $::tr(Cancel) -command {
    focus .
    grab release .apdialog
    destroy .apdialog
    focus .
    set autoplayResult 0
  }
  button $b.ok -text "OK" -command {
    grab release .apdialog
    if {$tempdelay < 0.1} { set tempdelay 0.1 }
    set autoplayDelay [expr {int($tempdelay * 1000)}]
    focus .
    grab release .apdialog
    destroy .apdialog
    focus .
    set autoplayResult 1
  }
  pack $b.cancel $b.ok -side right -padx 5 -pady 5
  focus $w.entry
  update idletasks
  grab $w
  tkwait variable autoplayResult
  return $autoplayResult
}


proc toggleAutoplay {{askForDelay 0}} {
  global autoplayMode annotateMode
  if {$autoplayMode == 0} {
    if {$askForDelay} {
      if {! [setAutoplayDelay 1]} {
        set annotateMode 0
        return
      }
    }
    set autoplayMode 1
    set annotateMode 1
    .button.autoplay configure -image autoplay_on -relief sunken
    autoplay
  } else {
    cancelAutoplay
  }
}

proc autoplay {} {
  global autoplayDelay autoplayMode autoAnnotate
  if {$autoplayMode == 0} { return }
  if {[sc_pos isAt vend]} {
    cancelAutoplay
    return
  }
  if {$autoAnnotate} {
    if {![sc_pos isAt start]} { addAnalysisVariation }
  }
  ::move::Forward
  after $autoplayDelay autoplay
}

proc cancelAutoplay {} {
  global autoplayMode annotateMode
  set autoplayMode 0
  set annotateMode 0
  after cancel autoplay
  .button.autoplay configure -image autoplay_off -relief flat
}

bind . <Control-z> {toggleAutoplay; break}
bind . <Escape> cancelAutoplay

set trialMode 0

proc setTrialMode {mode} {
  global trialMode
  if {$mode == "toggle"} {
    set mode [expr {1 - $trialMode}]
  }
  if {$mode == $trialMode} { return }
  if {$mode == "update"} { set mode $trialMode }

  if {$mode == 1} {
    set trialMode 1
    sc_game push copy
    .button.trial configure -image tb_trial_on
  } else {
    set trialMode 0
    sc_game pop
    .button.trial configure -image tb_trial
  }
  updateBoard -pgn
}


